/*
 *  pcap-compatible 802.11 packet sniffer
 *
 *  Copyright (C) 2004  Christophe Devine
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <netpacket/packet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#include "pcap.h"

#ifndef ETH_P_ALL
#define ETH_P_ALL 3
#endif

#ifndef ARPHRD_IEEE80211
#define ARPHRD_IEEE80211 801
#endif

#ifndef ARPHRD_IEEE80211_PRISM
#define ARPHRD_IEEE80211_PRISM 802
#endif

#define AMOD(x,y) ((x) % (y) < 0 ? ((x) % (y)) + (y) : (x) % (y))

char usage[] =

"\n"
"  airodump 1.0 - (C) 2004 Christophe Devine\n"
"\n"
"  usage: %s [options] <interface> <pcap filename>\n"
"\n"
"      -b <bssid> : only write packets that belong to this BSSID\n"
"      -w         : only write packets with  potentially weak IV\n"
"      -e         : only write WEP encrypted data packets\n"
"\n";

unsigned char buffer[65536];

int do_exit = 0;

void sighandler( int signum )
{
    if( signum == SIGINT || signum == SIGTERM )
        do_exit = 1;
}

int main( int argc, char *argv[] )
{
    int i, n, fromlen;
    long int cnt1, cnt2;
    int bssid[6], outfile;
    int onlywep, onlyweak;
    int arptype, raw_sock;
    unsigned char *h80211;

    struct ifreq ifr;
    struct packet_mreq mr;
    struct sockaddr_ll sll;
    struct sockaddr_ll from;
    struct pcap_file_header pfh;
    struct pcap_pkthdr pkh;
    struct timeval tv;
    time_t tm_prev;

    fd_set rfds;

    /* check the arguments */

    if( argc < 3 )
    {
    usage:
        printf( usage, argv[0] );
        return( 1 );
    }

    bssid[0] = -1;
    onlywep = onlyweak = 0;

    while( 1 )
    {
        int option = getopt( argc, argv, "b:we" );

        if( option < 0 ) break;

        switch( option )
        {
            case 'b':

                if( sscanf( optarg, "%x:%x:%x:%x:%x:%x",
                            &bssid[0], &bssid[1], &bssid[2],
                            &bssid[3], &bssid[4], &bssid[5] ) != 6 )
                    goto usage;

                for( i = 0; i < 6; i++ )
                    if( bssid[i] < 0 || bssid[i] > 255 )
                        goto usage;

                break;

            case 'e' : onlywep  = 1; break;
            case 'w' : onlyweak = 1; break;

            default : goto usage;
        }
    }

    if( argc - optind != 2 )
        goto usage;

    /* attempt to open or create the output file */

    optind++;

    n = sizeof( struct pcap_file_header );

    if( ( outfile = open( argv[optind], O_LARGEFILE | O_RDWR ) ) < 0 )
    {
        if( ( outfile = open( argv[optind], O_LARGEFILE | O_RDWR |
                                            O_CREAT, 0644 ) ) < 0 )
        {
            perror( "open" );
            return( 1 );
        }

        /* newly created file, so append the pcap header */

        pfh.magic           = TCPDUMP_MAGIC;
        pfh.version_major   = PCAP_VERSION_MAJOR;
        pfh.version_minor   = PCAP_VERSION_MINOR;
        pfh.thiszone        = 0;
        pfh.sigfigs         = 0;
        pfh.snaplen         = 65535;
        pfh.linktype        = LINKTYPE_IEEE802_11;

        write( outfile, &pfh, n );
    }
    else
    {
        /* existing file, check the pcap header contents */

        if( read( outfile, &pfh, n ) != n )
        {
            perror( "read" );
            return( 1 );
        }

        if( pfh.magic != TCPDUMP_MAGIC )
        {
            fprintf( stderr, "wrong magic from pcap file header\n"
                             "(got 0x%08X, expected 0x%08X)\n",
                             pfh.magic, TCPDUMP_MAGIC );
            return( 1 );
        }

        if( pfh.linktype != LINKTYPE_IEEE802_11 )
        {
            fprintf( stderr, "wrong linktype from pcap file header\n"
                             "(got %d, expected LINKTYPE_IEEE802_11)\n",
                     pfh.linktype );
            return( 1 );
        }

        if( lseek64( outfile, 0, SEEK_END ) < 0 )
        {
            perror( "lseek64(SEEK_END)" );
            return( 1 );
        }
    }

    optind--;

    /* create the raw socket */

    if( ( raw_sock = socket( PF_PACKET, SOCK_RAW,
                             htons( ETH_P_ALL ) ) ) < 0 )
    {
        perror( "socket(PF_PACKET)" );
        return( 1 );
    }

    /* find the interface index */

    memset( &ifr, 0, sizeof( ifr ) );
    strncpy( ifr.ifr_name, argv[optind], sizeof( ifr.ifr_name ) - 1 );

    if( ioctl( raw_sock, SIOCGIFINDEX, &ifr ) < 0 )
    {
        perror( "ioctl(SIOCGIFINDEX)" );
        return( 1 );
    }

    memset( &mr, 0, sizeof( mr ) );
    mr.mr_ifindex = ifr.ifr_ifindex;
    mr.mr_type    = PACKET_MR_PROMISC;

    /* bind the raw socket to the interface */

    memset( &sll, 0, sizeof( sll ) );
    sll.sll_family   = AF_PACKET;
    sll.sll_ifindex  = ifr.ifr_ifindex;
    sll.sll_protocol = htons( ETH_P_ALL );

    if( bind( raw_sock, (struct sockaddr *) &sll,
              sizeof( sll ) ) < 0 )
    {
        perror( "bind(ETH_P_ALL)" );
        return( 1 );
    }

    /* lookup the hardware type */

    if( ioctl( raw_sock, SIOCGIFHWADDR, &ifr ) < 0 )
    {
        perror( "ioctl(SIOCGIFHWADDR)" );
        return( 1 );
    }

    arptype = ifr.ifr_hwaddr.sa_family;

    if( arptype != ARPHRD_IEEE80211 &&
        arptype != ARPHRD_IEEE80211_PRISM )
    {
        fprintf( stderr, "unsupported hardware link type %d\n",
                 ifr.ifr_hwaddr.sa_family );
        fprintf( stderr, "expected ARPHRD_IEEE80211 or "
                 "ARPHRD_IEEE80211_PRISM\n" );
        return( 1 );
    }

    /* enable promiscuous mode */

    if( setsockopt( raw_sock, SOL_PACKET, PACKET_ADD_MEMBERSHIP,
                    &mr, sizeof( mr ) ) < 0 )
    {
        perror( "setsockopt(PACKET_MR_PROMISC)" );
        return( 1 );
    }

    signal( SIGINT, sighandler );

    tm_prev = time( NULL );

    cnt1 = cnt2 = 0;

    while( 1 )
    {
        if( do_exit ) break;

        /* wait for incoming packets */

        FD_ZERO( &rfds );
        FD_SET( raw_sock, &rfds );

        tv.tv_sec  = 1;
        tv.tv_usec = 0;

        if( select( raw_sock + 1, &rfds, NULL, NULL, &tv ) < 0 )
        {
            if( errno == EINTR ) continue;
            perror( "\nselect" );
            return( 1 );
        }
 
        if( time( NULL ) - tm_prev >= 1 )
        {
            tm_prev = time( NULL );
            printf( "\r\33[1KSeen %ld packets, wrote %ld packets\r",
                    cnt1, cnt2 );
            fflush( stdout );
        }

        if( ! FD_ISSET( raw_sock, &rfds ) ) continue;

        /* at least one packet available for reading */

        fromlen = sizeof( from );

        if( ( n = recvfrom( raw_sock, buffer, sizeof( buffer ),
              0, (struct sockaddr *) &from, &fromlen ) ) < 0 )
        {
            if( errno == EINTR ) continue;
            perror( "\nrecvfrom" );
            return( 1 );
        }

        if( from.sll_family != AF_PACKET )
        {
            fprintf( stderr, "\nfrom.sll_family == %d, should be 17 "
                     "(AF_PACKET)\n", from.sll_family );
            return( 1 );
        }

        cnt1++;

        /* skip the prism header if present */

        pkh.len = n;

        h80211 = buffer;

        if( arptype == ARPHRD_IEEE80211_PRISM )
        {
            n = *(int *)( h80211 + 4 );
            h80211 += n; pkh.len -= n;

            if( pkh.len <= 0 )
            {
                fprintf( stderr, "\nFIXME: wrong prism header size ?\n" );
                continue;
            }
        }

        if( bssid[0] != -1 )
        {
            /* only log packets that match our bssid */

            if( h80211[16] != bssid[0] || h80211[17] != bssid[1] ||
                h80211[18] != bssid[2] || h80211[19] != bssid[3] ||
                h80211[20] != bssid[4] || h80211[21] != bssid[5] )
                continue;
        }

        if( onlywep || onlyweak )
        {
            /* only log encrypted data packets */

            if( pkh.len < 40 ) continue;
            if( ( h80211[0] & 0x0C ) != 0x08 ) continue;
            if( ( h80211[1] & 0x40 ) != 0x40 ) continue;
        }

        if( onlyweak )
        {
            /* check if this IV is potentially weak */

            int x, y, z, a, b, B;

            x = h80211[24];
            y = h80211[25];
            z = h80211[26];

            a = ( x + y ) & 0xFF;
            b = AMOD( (x + y) - z, 256 );

            for( B = 0; B < 13; B++ )
            {
                if((((0 <= a && a < B) ||
                 (a == B && b == (B + 1) * 2)) &&
                 (B % 2 ? a != (B + 1) / 2 : 1)) ||
                 (a == B + 1 && (B == 0 ? b == (B + 1) * 2 : 1)) ||
                 (x == B + 3 && y == 255) ||
                 (B != 0 && !(B % 2) ? (x == 1 && y == (B / 2) + 1) ||
                 (x == (B / 2) + 2 && y == 255 - x) : 0))
                    break;
            }

            if( B == 13 ) continue;
        }

        /* append the packet at the end of the output file */

        if( gettimeofday( &pkh.ts, NULL ) < 0 )
        {
            perror( "\ngettimeofday" );
            return( 1 );
        }

        pkh.caplen = pkh.len;

        if( write( outfile, &pkh, sizeof( pkh ) ) != sizeof( pkh ) ||
            write( outfile, h80211, pkh.len     ) != (int) pkh.len )
        {
            perror( "\nwrite" );
            return( 1 );
        }

        cnt2++;
    }

    printf( "\r\33[1KSeen %ld packets, wrote %ld packets\n",
            cnt1, cnt2 );

    return( 0 );
}
