/*
 *  802.11 to Ethernet pcap translator
 *
 *  Copyright (C) 2004  Christophe Devine
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <time.h>

#include "pcap.h"
#include "safe_io.h"

#define SWAP(x,y) { unsigned char tmp = x; x = y; y = tmp; }

char usage[] =

"\n"
"  802ether 2.0 - (C) 2004 Christophe Devine\n"
"\n"
"  usage: %s <pcap infile> <pcap outfile> [wep key]\n"
"\n";

unsigned char buffer[65536];

int main( int argc, char *argv[] )
{
    char *s;
    unsigned char K[16];
    unsigned char S[256];
    unsigned char *h80211;
    unsigned char arphdr[12];
    unsigned char wepkey[13];

    struct pcap_file_header pfh_in;
    struct pcap_file_header pfh_out;
    struct pcap_pkthdr pkh;

    int weplen, fd_in, fd_out;
    int n, i, j, p1, p2;
    long int cnt1, cnt2;
    time_t tm_prev;

    /* check the arguments */

    if( argc != 3 && argc != 4 )
    {
    usage:
        printf( usage, argv[0] );
        return( 1 );
    }

    weplen = 0;

    if( argc == 4 )
    {
        s = argv[3];

        while( sscanf( s, "%x", &n ) == 1 )
        {
            if( n < 0 || n > 255 )
                goto usage;

            wepkey[weplen++] = n;

            if( weplen >= 13 ) break;

            if( ! ( s = strchr( s, ':' ) ) )
                break;

            s++;
        }

        if( weplen != 5 && weplen != 13 )
            goto usage;
    }

    /* open the input file and check the pcap header */

    fd_in = open( argv[1], O_RDONLY | O_LARGEFILE );

    if( fd_in < 0 )
    {
        perror( "open(pcap fd_in)" );
        return( 1 );
    }

    n = sizeof( struct pcap_file_header );

    if( safe_read( fd_in, &pfh_in, n ) != n )
    {
        perror( "read(pcap header)" );
        return( 1 );
    }

    if( pfh_in.magic != TCPDUMP_MAGIC )
    {
        fprintf( stderr, "wrong magic from pcap file header\n"
                         "(got 0x%08X, expected 0x%08X)\n",
                         pfh_in.magic, TCPDUMP_MAGIC );
        return( 1 );
    }

    if( pfh_in.linktype != LINKTYPE_IEEE802_11 &&
        pfh_in.linktype != LINKTYPE_PRISM_HEADER )
    {
        fprintf( stderr, "unsupported linktype %d from pcap fd_in\n",
                 pfh_in.linktype );
        return( 1 );
    }

    /* attempt to open or create the output file */

    n = sizeof( struct pcap_file_header );

    if( ( fd_out = open( argv[2], O_LARGEFILE | O_RDWR ) ) < 0 )
    {
        if( ( fd_out = open( argv[2], O_LARGEFILE | O_RDWR |
                                       O_CREAT, 0644 ) ) < 0 )
        {
            perror( "creat(pcap fd_out)" );
            return( 1 );
        }

        /* newly created file, so append the pcap header */

        pfh_out.magic           = TCPDUMP_MAGIC;
        pfh_out.version_major   = PCAP_VERSION_MAJOR;
        pfh_out.version_minor   = PCAP_VERSION_MINOR;
        pfh_out.thiszone        = 0;
        pfh_out.sigfigs         = 0;
        pfh_out.snaplen         = 65535;
        pfh_out.linktype        = LINKTYPE_ETHERNET;

        if( safe_write( fd_out, &pfh_out, n ) != n )
        {
            perror( "write(pcap header)" );
            return( 1 );
        }
    }
    else
    {
        /* existing file, check the pcap header contents */

        if( safe_read( fd_out, &pfh_out, n ) != n )
        {
            perror( "read(pcap header)" );
            return( 1 );
        }

        if( pfh_out.magic != TCPDUMP_MAGIC )
        {
            fprintf( stderr, "wrong magic from pcap file header\n"
                             "(got 0x%08X, expected 0x%08X)\n",
                             pfh_out.magic, TCPDUMP_MAGIC );
            return( 1 );
        }

        if( pfh_out.linktype != LINKTYPE_ETHERNET )
        {
            fprintf( stderr, "wrong linktype from pcap file header\n"
                             "(got %d, expected LINKTYPE_ETHERNET)\n",
                     pfh_out.linktype );
            return( 1 );
        }

        if( lseek64( fd_out, 0, SEEK_END ) < 0 )
        {
            perror( "lseek64(SEEK_END)" );
            return( 1 );
        }
    }

    memcpy( K + 3, wepkey, weplen );

    tm_prev = time( NULL );

    cnt1 = cnt2 = 0;

    while( 1 )
    {
        /* every 1s, update the display */

        if( time( NULL ) - tm_prev >= 1 )
        {
            tm_prev = time( NULL );
            printf( "\r\33[2KRead %ld packets, wrote "
                    "%ld packets\r", cnt1, cnt2 );
            fflush( stdout );
        }

        /* read one packet */

        n = sizeof( pkh );

        if( safe_read( fd_in, &pkh, n ) != n )
            break;

        n = pkh.caplen;

        if( safe_read( fd_in, buffer, n ) != n )
            break;

        cnt1++;

        h80211 = buffer;

        if( pfh_in.linktype == LINKTYPE_PRISM_HEADER )
        {
            /* remove the prism header if necessary */

            n = *(int *)( h80211 + 4 );

            if( n < 8 || n >= (int) pkh.caplen )
                continue;

            h80211 += n; pkh.caplen -= n;
        }

        if( pkh.caplen < 40 ) continue;

        /* only consider data packets */

        if( ( h80211[0] & 0x0C ) != 0x08 ) continue;

        /* if the packet in encrypted, decrypt it */

        if( ( h80211[1] & 0x40 ) == 0x40 )
        {
            if( weplen == 0 ) continue;

            memcpy( K, h80211 + 24, 3 );

            for( i = 0; i < 256; i++ )
                S[i] = i;

            for( i = j = 0; i < 256; i++ )
            {
                j = ( j + S[i] + K[i & (2 + weplen)]) & 0xFF;
                SWAP( S[i], S[j] );
            }

            p1 = p2 = 0;

            for( i = 1, j = 0; i <= (int) pkh.caplen - 32; i++ )
            {
                p1 = ( p1 + 1     ) & 0xFF;
                p2 = ( p2 + S[p1] ) & 0xFF;
                SWAP( S[p1], S[p2] );
                h80211[28 + i - 1] ^= S[(S[p1] + S[p2]) & 0xFF];
            }

            /* remove the WEP IV/KeyID + padding/FCS */

            pkh.len    -= 4 + ( h80211[27] & 0x3F ) + 4;
            pkh.caplen -= 4 + ( h80211[27] & 0x3F ) + 4;

            memcpy( h80211 + 24, h80211 + 28, pkh.caplen - 24 );
        }

        /* remove the 802.11 & LLC header */

        pkh.len    -= 24 + 6;
        pkh.caplen -= 24 + 6;

        /* also create the Ethernet link layer (MAC dst+src) */

        pkh.len    += 12;
        pkh.caplen += 12;

        switch( h80211[1] & 3 )
        {
            case  0:    /* To DS = 0, From DS = 0: DA, SA, BSSID */

                memcpy( arphdr + 0, h80211 +  4, 6 );
                memcpy( arphdr + 6, h80211 + 10, 6 );
                break;

            case  1:    /* To DS = 1, From DS = 0: BSSID, SA, DA */

                memcpy( arphdr + 0, h80211 + 16, 6 );
                memcpy( arphdr + 6, h80211 + 10, 6 );
                break;

            case  2:    /* To DS = 0, From DS = 1: DA, BSSID, SA */

                memcpy( arphdr + 0, h80211 +  4, 6 );
                memcpy( arphdr + 6, h80211 + 16, 6 );
                break;

            default:    /* To DS = 1, From DS = 1: RA, TA, DA, SA */

                memcpy( arphdr + 0, h80211 + 16, 6 );
                memcpy( arphdr + 6, h80211 + 24, 6 );
                break;
        }

        memcpy( h80211 + 18, arphdr, 12 );

        n = sizeof( pkh );

        if( safe_write( fd_out, &pkh, n ) != n )
        {
            perror( "\nwrite(packet header)" );
            return( 1 );
        }

        n = pkh.caplen;

        if( safe_write( fd_out, h80211 + 18, n ) != n )
        {
            perror( "\nwrite(packet data)" );
            return( 1 );
        }

        cnt2++;
    }

    printf( "\r\33[2KRead %ld packets, wrote "
            "%ld packets.\n", cnt1, cnt2 );

    return( 0 );
}
