/*
 *  pcap-compatible 802.11 packet sniffer
 *
 *  Copyright (C) 2004  Christophe Devine
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <netpacket/packet.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#include "pcap.h"
#include "safe_io.h"

#ifndef ETH_P_ALL
#define ETH_P_ALL 3
#endif

#ifndef ARPHRD_IEEE80211
#define ARPHRD_IEEE80211 801
#endif

#ifndef ARPHRD_IEEE80211_PRISM
#define ARPHRD_IEEE80211_PRISM 802
#endif

#define ARP_REQUEST_HEADER              \
    "\xAA\xAA\x03\x00\x00\x00\x08"      \
    "\x06\x00\x01\x08\x00\x06\x04"

#define REFRESH_TIMEOUT 250000

char usage[] =

"\n"
"  airodump 2.0 - (C) 2004 Christophe Devine\n"
"\n"
"  usage: %s <wifi interface> <output filename> [mac filter]\n"
"\n";

struct AP_info
{
    struct AP_info *prev;
    struct AP_info *next;
    unsigned char bssid[6];
    unsigned char essid[33];
    unsigned char lanip[4];
    unsigned char *uniq;
    unsigned long nb_pkt;
    unsigned long nb_ivs;
    time_t tinit, tlast;
    int channel;
    int is_wep;
    int power;
};

unsigned char buffer[65536];

int do_exit = 0;

void sighandler( int signum )
{
    if( signum == SIGINT || signum == SIGTERM )
        do_exit = 1;
}

int main( int argc, char *argv[] )
{
    FILE *f_csv;

    int i, n;
    int macfilter;
    int power, arptype;
    int fd_raw, fd_pcap;
    unsigned long iv;
    long time_slept;

    char *s;
    unsigned char *p;
    unsigned char *h80211;
    unsigned char bssid[6];
    unsigned char maddr[6];

    struct AP_info *ap_1st;
    struct AP_info *ap_cur;
    struct AP_info *ap_prv;
    struct AP_info *ap_end;

    struct ifreq ifr;
    struct packet_mreq mr;
    struct sockaddr_ll sll;
    struct pcap_file_header pfh;
    struct pcap_pkthdr pkh;
    struct winsize ws;
    struct timeval tv;

    fd_set rfds;

    /* create the raw socket */

    if( ( fd_raw = socket( PF_PACKET, SOCK_RAW,
                             htons( ETH_P_ALL ) ) ) < 0 )
    {
        perror( "socket(PF_PACKET)" );
        return( 1 );
    }

    /* drop privileges */

    setuid( getuid() );

    /* check the arguments */

    if( argc < 3 || argc > 4 )
    {
    usage:

        printf( usage, argv[0] );
        return( 1 );
    }

    macfilter = 0;

    if( argc == 4 )
    {
        i = 0;
        s = argv[3];

        while( sscanf( s, "%x", &n ) == 1 )
        {
            if( n < 0 || n > 255 )
                goto usage;

            maddr[i] = n; i++;

            s = strchr( s, ':' );

            if( i == 6 || ! s ) break;

            s++;
        }

        if( i != 6 ) goto usage;

        macfilter = 1;
    }

    /* attempt to open or create the output file */

    sprintf( buffer, "%s.pcap", argv[2] );

    n = sizeof( struct pcap_file_header );

    if( ( fd_pcap = open( buffer, O_LARGEFILE | O_RDWR ) ) < 0 )
    {
        if( ( fd_pcap = open( buffer, O_LARGEFILE | O_RDWR |
                                      O_CREAT, 0644 ) ) < 0 )
        {
            perror( "creat" );
            return( 1 );
        }

        /* newly created file, so append the pcap header */

        pfh.magic           = TCPDUMP_MAGIC;
        pfh.version_major   = PCAP_VERSION_MAJOR;
        pfh.version_minor   = PCAP_VERSION_MINOR;
        pfh.thiszone        = 0;
        pfh.sigfigs         = 0;
        pfh.snaplen         = 65535;
        pfh.linktype        = LINKTYPE_IEEE802_11;

        if( safe_write( fd_pcap, &pfh, n ) != n )
        {
            perror( "write(pcap header)" );
            return( 1 );
        }
    }
    else
    {
        /* existing file, check the pcap header contents */

        if( safe_read( fd_pcap, &pfh, n ) != n )
        {
            perror( "read (pcap header)" );
            return( 1 );
        }

        if( pfh.magic != TCPDUMP_MAGIC )
        {
            fprintf( stderr, "wrong magic from pcap file header\n"
                             "(got 0x%08X, expected 0x%08X)\n",
                             pfh.magic, TCPDUMP_MAGIC );
            return( 1 );
        }

        if( pfh.linktype != LINKTYPE_IEEE802_11 )
        {
            fprintf( stderr, "wrong linktype from pcap file header\n"
                             "(got %d, expected LINKTYPE_IEEE802_11)\n",
                     pfh.linktype );
            return( 1 );
        }

        if( lseek64( fd_pcap, 0, SEEK_END ) < 0 )
        {
            perror( "lseek64(SEEK_END)" );
            return( 1 );
        }
    }

    /* find the interface index */

    memset( &ifr, 0, sizeof( ifr ) );
    strncpy( ifr.ifr_name, argv[optind], sizeof( ifr.ifr_name ) - 1 );

    if( ioctl( fd_raw, SIOCGIFINDEX, &ifr ) < 0 )
    {
        perror( "ioctl(SIOCGIFINDEX)" );
        return( 1 );
    }

    /* bind the raw socket to the interface */

    memset( &sll, 0, sizeof( sll ) );
    sll.sll_family   = AF_PACKET;
    sll.sll_ifindex  = ifr.ifr_ifindex;
    sll.sll_protocol = htons( ETH_P_ALL );

    if( bind( fd_raw, (struct sockaddr *) &sll,
              sizeof( sll ) ) < 0 )
    {
        perror( "bind(ETH_P_ALL)" );
        return( 1 );
    }

    /* lookup the hardware type */

    if( ioctl( fd_raw, SIOCGIFHWADDR, &ifr ) < 0 )
    {
        perror( "ioctl(SIOCGIFHWADDR)" );
        return( 1 );
    }

    arptype = ifr.ifr_hwaddr.sa_family;

    if( arptype != ARPHRD_IEEE80211 &&
        arptype != ARPHRD_IEEE80211_PRISM )
    {
        fprintf( stderr, "unsupported hardware link type %d\n",
                 ifr.ifr_hwaddr.sa_family );
        fprintf( stderr, "expected ARPHRD_IEEE80211 or "
                 "ARPHRD_IEEE80211_PRISM\n" );
        return( 1 );
    }

    /* enable promiscuous mode */

    memset( &mr, 0, sizeof( mr ) );
    mr.mr_ifindex = sll.sll_ifindex;
    mr.mr_type    = PACKET_MR_PROMISC;

    if( setsockopt( fd_raw, SOL_PACKET, PACKET_ADD_MEMBERSHIP,
                    &mr, sizeof( mr ) ) < 0 )
    {
        perror( "setsockopt(PACKET_MR_PROMISC)" );
        return( 1 );
    }

    signal( SIGINT, sighandler );

    power = -1;

    ap_1st = NULL;
    ap_prv = NULL;
    ap_end = NULL;

    time_slept = 0;

    printf( "\33[2J" );
    fflush( stdout );

    while( 1 )
    {
        if( do_exit ) break;

        /* wait for incoming packets */

        FD_ZERO( &rfds );
        FD_SET( fd_raw, &rfds );

        tv.tv_sec  = 0;
        tv.tv_usec = REFRESH_TIMEOUT;

        if( select( fd_raw + 1, &rfds, NULL, NULL, &tv ) < 0 )
        {
            if( errno == EINTR ) continue;
            perror( "select" );
            return( 1 );
        }

        time_slept += REFRESH_TIMEOUT - tv.tv_usec;

        if( time_slept > REFRESH_TIMEOUT )
        {
            /* display the list of access points we have */

            time_slept = 0;

            if( ioctl( 0, TIOCGWINSZ, &ws ) < 0 )
            {
                ws.ws_row = 25;
                ws.ws_col = 80;
            }

            n = ws.ws_col - 60;
            if( n > 32 ) n = 32;
            if( n <  0 ) n =  0;

            printf( "\33[2;1H  BSSID              CH W PWR"
                    "  Packets   LAN IP / # IVs   ESSID\n\n" );

            ap_cur = ap_end;

            i = 0;

            while( ap_cur != NULL )
            {
                if( time( NULL ) - ap_cur->tlast > 120 )
                {
                    ap_cur = ap_cur->next;
                    continue;
                }

                if( i++ > ws.ws_row - 6 )
                    break;

                printf( "  %02X:%02X:%02X:%02X:%02X:%02X",
                        ap_cur->bssid[0], ap_cur->bssid[1],
                        ap_cur->bssid[2], ap_cur->bssid[3],
                        ap_cur->bssid[4], ap_cur->bssid[5] );

                printf( "  %2d %c %3d %8ld  ",
                        ap_cur->channel, ap_cur->is_wep,
                        ap_cur->power,   ap_cur->nb_pkt );

                if( ap_cur->is_wep == 'Y' )
                {
                    printf( "%15ld", ap_cur->nb_ivs ); 
                }
                else
                {
                    buffer[0] = '\0';

                    if( *(unsigned long *)( ap_cur->lanip ) )
                        sprintf( buffer, "%3d.%3d.%3d.%3d",
                                    ap_cur->lanip[0], ap_cur->lanip[1],
                                    ap_cur->lanip[2], ap_cur->lanip[3] );

                    printf( "%15s", buffer );
                }

                sprintf( buffer, "%-32s", ap_cur->essid );
                buffer[n] = '\0';
                printf( "   %s\33[K\n", buffer );

                ap_cur = ap_cur->prev;
            }

            printf( "\33[J\n" );

            continue;
        }

        /* one packet available for reading */

        memset( buffer, 0, 64 );

        if( ( n = read( fd_raw, buffer, sizeof( buffer ) ) ) < 0 )
        {
            perror( "read" );
            return( 1 );
        }

        /* skip the prism header if present */

        pkh.caplen = n;

        h80211 = buffer;

        if( arptype == ARPHRD_IEEE80211_PRISM )
        {
            power = *(int *)( h80211 + 0x5C );

            n = *(int *)( h80211 + 4 );

            if( n < 8 || n >= (int) pkh.caplen )
                continue;

            h80211 += n; pkh.caplen -= n;
        }

        pkh.len = pkh.caplen;

        ap_cur = NULL;

        if( pkh.caplen < 24 )
            goto write_packet;

        /* skip control frames */

        if( ( h80211[0] & 0x0C ) == 0x04 )
            goto write_packet;

        /* skip management frames without an ESSID */

        if( ( h80211[0] & 0x0C ) == 0x00 )
            if( ( h80211[0] & 0xF0 ) != 0x00 &&
                ( h80211[0] & 0xF0 ) != 0x50 &&
                ( h80211[0] & 0xF0 ) != 0x80 )
                goto write_packet;

        /* update our chained list of access points */

        switch( h80211[1] & 3 )
        {
            case  0: i = 16; break;      /* DA, SA, BSSID  */
            case  1: i =  4; break;      /* BSSID, SA, DA  */
            case  2: i = 10; break;      /* DA, BSSID, SA  */
            default: i =  4; break;      /* RA, TA, DA, SA */
        }

        memcpy( bssid, h80211 + i, 6 );

        ap_cur = ap_1st;

        while( ap_cur != NULL )
        {
            if( ! memcmp( ap_cur->bssid, bssid, 6 ) )
                break;

            ap_prv = ap_cur;
            ap_cur = ap_cur->next;
        }

        if( ap_cur == NULL )
        {
            /* looks like a new access point, add it */

            if( ! ( ap_cur = (struct AP_info *) malloc(
                             sizeof( struct AP_info ) ) ) )
            {
                perror( "malloc" );
                return( 1 );
            }

            if( ap_1st == NULL )
                ap_1st = ap_cur;
            else
                ap_prv->next = ap_cur;

            memset( ap_cur, 0, sizeof( struct AP_info ) );
            memcpy( ap_cur->bssid, bssid, 6 );

            ap_cur->prev = ap_prv;

            if( ! ( ap_cur->uniq = (unsigned char *) malloc(
                                           256 * 256 * 256 ) ) )
            {
                perror( "malloc" );
                return( 1 );
            }

            ap_cur->tinit   = time( NULL );
            ap_cur->tlast   = time( NULL );
            ap_cur->power   = power;
            ap_cur->is_wep  = '?';
            ap_cur->channel = -1;

            ap_end = ap_cur;
        }

        if( time( NULL ) - ap_cur->tlast >= 1 )
        {
            ap_cur->tlast = time( NULL );
            ap_cur->power = power;
        }

        ap_cur->nb_pkt++;

        /* packet parsing: Beacon or Probe Response */

        if( h80211[0] == 0x80 ||
            h80211[0] == 0x50 )
        {
            ap_cur->is_wep = ( h80211[34] & 0x10 ) ? 'Y' : 'N';

            p = h80211 + 36;

            while( p < h80211 + pkh.caplen )
            {
                if( p + 2 + p[1] > h80211 + pkh.caplen )
                    break;

                if( p[0] == 0x00 && p[2] != '\0' )
                {
                    n = ( p[1] > 32 ) ? 32 : p[1];

                    memset( ap_cur->essid, 0, 33 );
                    memcpy( ap_cur->essid, p + 2, n );

                    for( i = 0; i < n; i++ )
                        if( ap_cur->essid[i] < ' ' )
                            ap_cur->essid[i] = '.';
                }

                if( p[0] == 0x03 )
                    ap_cur->channel = (int) p[2];

                p += 2 + p[1];
            }
        }

        /* packet parsing: Association Request */

        if( h80211[0] == 0x00 )
        {
            p = h80211 + 28;

            while( p < h80211 + pkh.caplen )
            {
                if( p + 2 + p[1] > h80211 + pkh.caplen )
                    break;

                if( p[0] == 0x00 && p[2] != '\0' )
                {
                    n = ( p[1] > 32 ) ? 32 : p[1];

                    memset( ap_cur->essid, 0, 33 );
                    memcpy( ap_cur->essid, p + 2, n );

                    for( i = 0; i < n; i++ )
                        if( ap_cur->essid[i] < ' ' )
                            ap_cur->essid[i] = '.';
                }

                p += 2 + p[1];
            }
        }

        /* packet parsing: non-encrypted data */

        if( ( h80211[0] & 0x0C ) == 0x08 &&
            ( h80211[1] & 0x40 ) == 0x00 )
        {
            if( memcmp( h80211 + 24, ARP_REQUEST_HEADER, 14 ) == 0 )
                memcpy( ap_cur->lanip, h80211 + 46, 4 );
        }

        /* packet parsing: WEP encrypted data */

        if( ( h80211[0] & 0x0C ) == 0x08 &&
            ( h80211[1] & 0x40 ) == 0x40 )
        {
            iv = h80211[24] + ( h80211[25] <<  8 )
                            + ( h80211[26] << 16 );

            ap_cur->is_wep = 'Y';

            if( ap_cur->uniq[iv] == 0xF0 ||
                ap_cur->uniq[iv] == 0xF1 )
                continue;

            ap_cur->uniq[iv] = 0xF0;
            ap_cur->nb_ivs++;
        }

        /* append the packet at the end of the output file */

    write_packet:

        if( macfilter == 1 )
        {
            /* MAC address filtering enabled */

            if( memcmp( h80211 +  4, maddr, 6 ) &&
                memcmp( h80211 + 10, maddr, 6 ) &&
                memcmp( h80211 + 16, maddr, 6 ) &&
                memcmp( h80211 + 24, maddr, 6 ) )
                continue;
        }

        if( ( h80211[0] & 0x0C ) == 0x08 &&
            ( h80211[1] & 0x40 ) == 0x40 )
        {
            /* check that we haven't already seen this IV */

            if( ap_cur == NULL )
                continue;

            iv = h80211[24] + ( h80211[25] <<  8 )
                            + ( h80211[26] << 16 );

            if( ap_cur->uniq[iv] == 0xF1 )
                continue;

            ap_cur->uniq[iv] = 0xF1;
        }

        if( gettimeofday( &pkh.ts, NULL ) < 0 )
        {
            perror( "gettimeofday" );
            return( 1 );
        }

        n = sizeof( pkh );

        if( safe_write( fd_pcap, &pkh, n ) != n )
        {
            perror( "write(packet header)" );
            return( 1 );
        }

        n = pkh.caplen;

        if( safe_write( fd_pcap, h80211, n ) != n )
        {
            perror( "write(packet data)" );
            return( 1 );
        }

    }

    /* save the list of access points in CSV format */

    sprintf( buffer, "%s.csv", argv[2] );

    if( ( f_csv = fopen( buffer, "w+" ) ) != NULL )
    {
        ap_cur = ap_1st;

        while( ap_cur != NULL )
        {
            fprintf( f_csv, "%10ld; %10ld; ",
                     ap_cur->tinit, ap_cur->tlast );

            fprintf( f_csv, "%02X:%02X:%02X:%02X:%02X:%02X; ",
                     ap_cur->bssid[0], ap_cur->bssid[1],
                     ap_cur->bssid[2], ap_cur->bssid[3],
                     ap_cur->bssid[4], ap_cur->bssid[5] );

            fprintf( f_csv, "%2d; %c; %3d; ",
                     ap_cur->channel,
                     ap_cur->is_wep,
                     ap_cur->power );

            fprintf( f_csv, "%8ld; %8ld; ",
                     ap_cur->nb_pkt, ap_cur->nb_ivs );

            fprintf( f_csv, "%3d.%3d.%3d.%3d; ",
                     ap_cur->lanip[0], ap_cur->lanip[1],
                     ap_cur->lanip[2], ap_cur->lanip[2] );

            fprintf( f_csv, "%-32s\r\n", ap_cur->essid );

            ap_cur = ap_cur->next;
        }
    }

    return( 0 );
}
