/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include "RC4.h"
#include "crc-32.h"

struct Packet {
  int len;
  unsigned char *buf; // includes IV
  Packet *next;
};

class ChecksumVerifier : public Verifier {
protected:
  Packet *packets;
public:
  ChecksumVerifier() {packets=NULL;}
  ~ChecksumVerifier() {
    Packet *cur=packets,*next;
    while(cur) {
      next=cur->next;
      delete[] cur->buf;
      delete cur;
      cur=next;
    }
  }
  void addSample(unsigned char *buf,int len) {
    Packet *p=new Packet;
    p->len=len;
    p->buf=new unsigned char[len];
    memcpy(p->buf,buf,len);
    p->next=packets;
    packets=p;
  }
  virtual int checkKey(unsigned char *k,int klen) {
    unsigned char key[16],*buf;
    RC4 *rc;
    int i;
    Packet *cur=packets;
    memcpy(key+3,k,klen);
    while(cur) {
      rc=new RC4();
      buf=new unsigned char[cur->len-4];
      memcpy(key,cur->buf,3);
      rc->keyWith(key,klen+3);
      memcpy(buf,cur->buf+4,cur->len-4);
      for(i=0;i<cur->len-4;i++) {
	buf[i] ^= rc->step();
      }
      if (doFCS(buf,cur->len-4)!=0xdebb20e3) {
	delete[] buf;
	delete rc;
	return(RES_FAILURE);
      }
      cur=cur->next;
    }
    return(RES_SUCCESS);
  }
};
  
int main(int argc,char *argv[]) {
  int r,i,fd,len,keySize=5,breadth=1;
  int ksamples=0,csamples[13];
  unsigned char buf[4000];
  RC4Crack *rc;
  ChecksumVerifier *cv;
  RC4::setupIdentity();
  memset(csamples,0,sizeof(int)*13);
  while(1) {
    r=getopt(argc,argv,"b:l:");
    if (r<0) break;
    switch(r) {
    case 'b':
      breadth=atoi(optarg);
      if (breadth<=0 || breadth>5) breadth=2;
      break;
    case 'l':
      if (atoi(optarg)==40) {
	keySize=5;
      } else if (atoi(optarg)==128) {
	keySize=13;
      } else {
	fprintf(stderr,"Invalid key length, must be 40 or 128\n");
	exit(1);
      }
      break;
    case '?':
      exit(1);
      fprintf(stderr,"Usage: %s [-b breadth] [-l 40|128] <file>\n",argv[0]);
    }
  }
  if (argc-optind!=1) {
    fprintf(stderr,"Usage: %s [-b breadth] [-l 40|128] <file>\n",argv[0]);
    exit(1);
  }
  fd=open(argv[optind],O_RDONLY);
  if (fd<0) {
    perror("Unable to open file");
    exit(1);
  }
  cv=new ChecksumVerifier();
  rc=new RC4Crack(cv,keySize);
  printf("Reading packets\n");
  while(1) {
    r=read(fd,&len,sizeof(int));
    if (r==0) break;
    if (r<0) {
      perror("Error reading packet len");
      exit(1);
    }
    if (r>2000) {
      fprintf(stderr,"Packet too long, %d\n",r);
      exit(1);
    }
    r=read(fd,buf,len);
    if (r!=len) {
      perror("Error reading packet");
      exit(1);
    }
    if (buf[0]>=3 && buf[0]<16) {
      csamples[buf[0]-3]+=rc->addSample(buf,buf[4] ^ 0xAA);
    }
    if (ksamples<10) {cv->addSample(buf,len); ksamples++;}
  }
  printf("Performing crack, keySize=%d bit, breadth=%d\n",
	 (keySize==5?40:128),breadth);
  for(i=0;i<keySize;i++) {
    printf("Key Byte %d: %d samples\n",i,csamples[i]);
  }
  printf("Check samples: %d\n\n",ksamples);
  r=rc->tryCrack(breadth,buf);
  if (r==RES_SUCCESS) {
    printf("GOT KEY!\n");
    printf("Hex = ");
    for(i=0;i<keySize;i++) {
      printf("%2.2x%c",buf[i],i==keySize-1?'\n':':');
    }
    printf("String = '");
    for(i=0;i<keySize;i++) {
      printf("%c",buf[i]>=32 && buf[i]<127?buf[i]:'.');
    }
    printf("'\n");
  } else {
    printf("FAILED! r=%d\n",r);
  }
}

