/*
 * dstumbler v1.0 [aps.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <errno.h>

#ifdef __OpenBSD__
#include <curses.h>
#else
#include <ncurses.h>
#endif

#include "dstumbler.h"
#include "screen.h"

/*
 * poll the wi driver or parse monitor mode output to find an ap
 */
void
parse_ap(const char *iface)
{
  int i, found, next = 1, num = 0;
  struct aps_s new;

  for(i = 0; i < aps_len; i++)
  {
    /* if this isn't the ap we're looking for, zero out it's quality data */
    memset((char *)aps[i]->quality, 0, sizeof(aps[i]->quality));
  }

  while(next)
  {
    memset((char *)&new, 0, sizeof(struct aps_s));

    if(monmode)
    {
      if(mon_parse_ap(iface, &new) == -1)
        return;
    }
    else if(scanmode)
    {
      if((next = wi_parse_scan_ap(iface, &new, num++)) == -1)
        return;
    }
    else if(wi_parse_ap(iface, &new) == -1)
      return;

    /* fill in cached gps coordinates */
    getgps(&new.ns, &new.ew);

    if((found = find_ap(new.ssid, new.bssid, new.chan)) == -2)
      return;
    else if(found == -1)
    {
      /* set timestamps */
      gettimeofday(&new.firstseen, NULL);
      gettimeofday(&new.lastseen, NULL);

      add_ap(&new);

      redraw_aps();

      if(audiomode)
        audio_play((!new.weped || new.weped == 2) ?
         SPKR_AP_PLAY_NOWEP : SPKR_AP_PLAY_WEP);

      if(aps_new == 0 || autosel)
        redraw_apinfo();

      /* if we're in autoselect mode, make sure we refresh stats for this ap */
      if(!nodemode && autosel)
        redraw_graph();
    }
    else
    {
      if(aps[found]->weped && (!new.weped || new.weped == 2))
        aps[found]->weped = new.weped;
      if(new.keyed)
        aps[found]->keyed = new.keyed;
      if(new.interval)
        aps[found]->interval = new.interval;
      if(new.srate)
        aps[found]->srate = new.srate;

      gettimeofday(&new.lastseen, NULL);

      /* update the found entry with quality info and timestamp and gps */
      memcpy((char *)aps[found]->quality, (char *)new.quality,
       sizeof(aps[found]->quality));
      memcpy((char *)&aps[found]->lastseen, (char *)&new.lastseen,
       sizeof(aps[found]->lastseen));
      memcpy((char *)&aps[found]->ns, (char *)&new.ns, sizeof(aps[found]->ns));
      memcpy((char *)&aps[found]->ew, (char *)&new.ew, sizeof(aps[found]->ew));

      /* write log */
      add_ap_log(aps[found], &new);

      if(found != aps_new || autosel)
        APNEW(found);

      redraw_aps();

      if(found == aps_cur || autosel)
      {
        redraw_apinfo();

        if(!nodemode)
          redraw_graph();
      }
    }

    if(!scanmode)
      break;
  }
}

/*
 * search through the aps linked list to see if the ap already exists
 */
int
find_ap(const char *ssid, const char *bssid, int chan)
{
  int i, found = -1;

  for(i = 0; i < aps_len; i++)
  {
    if(scanmode || monmode ? (memcmp(aps[i]->bssid, bssid, MACSIZE) == 0 &&
     strcmp(aps[i]->ssid, ssid) == 0 && aps[i]->chan == chan) :
     (memcmp(aps[i]->bssid, bssid, MACSIZE) == 0 ||
     /* if we can't match ssid, make sure can at least match channel */
     (strcmp(aps[i]->ssid, ssid) == 0 && aps[i]->chan == chan)))
      found = i;
  }

  if(found == -1 && strlen(ssid) == 0)
    return -2;

  return found;
}

/*
 * redraw the ap screen with all of the aps in the aps linked list
 */
#define TEMP_SSID_LEN (AP_SCR_COLS - 42)
#define SSID_FMT_LEN 7
#define MFG_FMT_LEN 7
#define MAC_STR_LEN 17
void
redraw_aps(void)
{
  int off = 0, i = 0, wrote = 0, ap_color, ap_color_bold, code;
  char temp_ssid[TEMP_SSID_LEN], ssid_fmt[SSID_FMT_LEN];
  struct aps_s *ptr;

  werase(ap_scr);
  off = (apchange || !autosel) ? aps_cur : aps_new;

  if(off < aps_win)
    aps_win = off;
  else if(off > (aps_win + AP_SCR_LINES - 1))
    aps_win = off - (AP_SCR_LINES - 1);

  for(i = aps_win; i < aps_len; i++)
  {
    ptr = aps[i];

    if(wrote++)
      waddch(ap_scr, '\n');

    if(i == aps_cur)
    {
      ap_color = AP_COLOR_SEL;
      ap_color_bold = AP_COLOR_SEL_BOLD;
    }
    else
    {
      ap_color = AP_COLOR;
      ap_color_bold = AP_COLOR_BOLD;
    }

    memset(temp_ssid, 0, TEMP_SSID_LEN - 1);
    memcpy(temp_ssid, ptr->ssid, MIN(SSIDLEN, (TEMP_SSID_LEN - 1)));
    temp_ssid[TEMP_SSID_LEN - 1] = '\0';
    snprintf(ssid_fmt, SSID_FMT_LEN - 1, "%%-%ds", TEMP_SSID_LEN - 1);

    /* figure out how many symbols our quality is to determine color code */
    if(ptr->quality[0] < 6)
      code = COLOR_CODE_LOW;
    else if(ptr->quality[0] > 5 && ptr->quality[0] < 9)
      code = COLOR_CODE_LOW_BOLD;
    else if(ptr->quality[0] > 8 && ptr->quality[0] < 12)
      code = COLOR_CODE_MED;
    else if(ptr->quality[0] > 11 && ptr->quality[0] < 15)
      code = COLOR_CODE_MED_BOLD;
    else if(ptr->quality[0] > 14 && ptr->quality[0] < 18)
      code = COLOR_CODE_HI;
    else if(ptr->quality[0] > 17)
      code = COLOR_CODE_HI_BOLD;

    wattrset(ap_scr, 0);

    /*
     * sorry for the messyness... better then using a loop
     */
    waddch(ap_scr, (((autosel && !nodemode) ? i == aps_new : i == aps_cur) ?
     APS_CUR | code : (ptr->quality[0] || ptr->quality[1] || ptr->quality[2] ?
     APS_NEW | code : ' ')));

    wattrset(ap_scr, ap_color_bold);
    wprintw(ap_scr, " [");
    wattrset(ap_scr, ap_color);
    wprintw(ap_scr, "%2d", ptr->chan);
    wattrset(ap_scr, ap_color_bold);
    wprintw(ap_scr, "] ");
    wattrset(ap_scr, ap_color);
    wprintw(ap_scr, ssid_fmt, temp_ssid);
    wattrset(ap_scr, ap_color_bold);
    wprintw(ap_scr, " (");
    wattrset(ap_scr, ap_color);

    if(resolvmfg)
    {
      char mfg[MFG_STRLEN], mfg_fmt[MFG_FMT_LEN];
      mfgresolve(mfg, ptr->bssid);
      snprintf(mfg_fmt, MFG_FMT_LEN - 1, "%%-%ds", MAC_STR_LEN);
      wprintw(ap_scr, mfg_fmt, mfg);
    }
    else
      wprintw(ap_scr, "%02x:%02x:%02x:%02x:%02x:%02x",
       ptr->bssid[0], ptr->bssid[1], ptr->bssid[2],
       ptr->bssid[3], ptr->bssid[4], ptr->bssid[5]);

    wattrset(ap_scr, ap_color_bold);
    wprintw(ap_scr, ")");
    waddch(ap_scr, defssid(ptr->ssid) ? APS_DEFSSID | AP_COLOR_DEFSSID : ' ');
    waddch(ap_scr, ptr->keyed == 1 ? APS_KEYED | AP_COLOR_KEYED :
     (ptr->keyed == 2 ? APS_OPEN | AP_COLOR_OPEN : ' '));
    waddch(ap_scr, ptr->adhoc == 1 ? APS_ADHOC | AP_COLOR_ADHOC :
     (ptr->adhoc == 2 ? APS_BSS | AP_COLOR_BSS : ' '));
    waddch(ap_scr, ptr->weped == 1 ?
     (ptr->wepid ? APS_40BIT : APS_WEP) | AP_COLOR_WEPED :
     (ptr->weped == 2 ? APS_NOWEP | AP_COLOR_NOWEP : ' '));
    wattrset(ap_scr, ap_color);
    wprintw(ap_scr, "%03d", ptr->quality[0]);
    waddch(ap_scr, ':' | ap_color_bold);
    wprintw(ap_scr, "%03d", ptr->quality[1]);
    waddch(ap_scr, ':' | ap_color_bold);
    wprintw(ap_scr, "%03d", ptr->quality[2]);

    if(wrote == AP_SCR_LINES)
      break;
  }

  redraw.ap_scr++;
}

/*
 * redraw current ap's info (in case we've polled more info from it)
 */
#define INFO_BUF_LEN (INFO_SCR_COLS * INFO_SCR_LINES)
#define SSID_BUF_LEN (INFO_SCR_COLS - 6)
#define MFG_BUF_LEN INFO_SCR_COLS
#define MON_BUF_LEN 12 
#define FIRSTSEEN_LEN 20
#define LASTSEEN_LEN 20
void
redraw_apinfo(void)
{
  static time_t then = 0, now;
  static struct tm firsttm, lasttm;

  int i, mode = 0;
  char info_buf[INFO_BUF_LEN], ssid_buf[SSID_BUF_LEN];
  char mfg_buf[MFG_BUF_LEN], mon_buf[MON_BUF_LEN], mfg[MFG_STRLEN];
  struct aps_s *ptr;

  now = time(NULL);

  ptr = aps[(autosel && apnew && !nodemode) ? aps_new : aps_cur];
  if(then != now || apnew || apchange)
  {
    if(apnew || apchange || !then)
      gmtime_r((time_t *)&ptr->firstseen.tv_sec, &firsttm);

    gmtime_r((time_t *)&ptr->lastseen.tv_sec, &lasttm);
    then = now;
  }

  if(usegps)
    snprintf(mfg_buf, MFG_BUF_LEN - 1, "GPS: %c:%.3f %c:%.3f",
     ptr->ns.dir, ptr->ns.coordinates, ptr->ew.dir, ptr->ew.coordinates);
  else
  {
    mfgresolve(mfg, ptr->bssid);
    snprintf(mfg_buf, MFG_BUF_LEN - 1, "Mfg: %s", mfg);
  }

  if((monmode || (prism2 && scanmode)) && ptr->srate)
    snprintf(mon_buf, MON_BUF_LEN - 1, "%d.%d/%d", ptr->srate / 2,
     (ptr->srate % 2) * 5, ptr->interval);
  else if(!prism2 && scanmode && ptr->interval)
    snprintf(mon_buf, MON_BUF_LEN - 1, "     %d", ptr->interval);
  else
    mon_buf[0] = '\0';

  strncpy(ssid_buf, ptr->ssid, SSID_BUF_LEN);
  ssid_buf[SSID_BUF_LEN - 1] = '\0';

  werase(info_scr);
  snprintf(info_buf, INFO_BUF_LEN - 1,
   " SSID: %s\n"
   " BSSID: %02x:%02x:%02x:%02x:%02x:%02x\n"
   " %s\n"
   " Channel: %-2d   %s\n"
   " Signal/Noise: %d/%d/%d\n"
   " First Seen: %d:%d:%d\n"
   " Last Seen:  %d:%d:%d\n",
   ptr->ssid, ptr->bssid[0], ptr->bssid[1], ptr->bssid[2], ptr->bssid[3],
   ptr->bssid[4], ptr->bssid[5], mfg_buf, ptr->chan, mon_buf, ptr->quality[0],
   ptr->quality[1], ptr->quality[2], firsttm.tm_hour, firsttm.tm_min,
   firsttm.tm_sec, lasttm.tm_hour, lasttm.tm_min, lasttm.tm_sec);

  for(i = 0; i < strlen(info_buf); i++)
  {
    if((info_buf[i] == ':' || info_buf[i] == '/') && info_buf[i + 1] == ' ')
      mode = 1;
    else if(info_buf[i] == '\n')
      mode = 0;

    if(mode && info_buf[i] != ':' && info_buf[i] != '/')
      waddch(info_scr, info_buf[i] | INFO_COLOR_BOLD);
    else
      waddch(info_scr, info_buf[i] | INFO_COLOR);
  }

  redraw.info_scr++;
}

/*
 * redraw signal to noise graph for current ap
 */
void
redraw_graph(void)
{
  static int wrote = 0;
  int i;
  struct aps_s *ptr;

  ptr = aps[autosel ? aps_new : aps_cur];

  if(apchange || (autosel && apnew))
  {
    werase(graph_scr);
    wrote = 0;

    for(i = (ptr->log_len > GRAPH_SCR_LINES ?
     ptr->log_len - GRAPH_SCR_LINES : 0); i < ptr->log_len; i++)
      print_graph(ptr->log[i]->quality, wrote++);
  }
  else
  {
    print_graph(ptr->quality, wrote++);

    if(audiomode)
      audio_graph(ptr->quality[0]);
  }

  redraw.graph_scr++;
}

/*
 * print one line on the graph using quality array
 */
void
print_graph(int *quality, int wrote)
{
  int i, s2n[2], sym[2], max = (prism2 ? GRAPH_PRISM_MAX : GRAPH_NORM_MAX);

  for(i = 0; i < 2; i++)
    s2n[i] = ((GRAPH_SCR_COLS - 12) * quality[i + 1]) / max;

  sym[0] = GRAPH_SIGNAL | GRAPH_COLOR_SIGNAL;
  sym[1] = GRAPH_NOISE | GRAPH_COLOR_NOISE;

  /* in case noise is > signal */
  if(s2n[1] > s2n[0])
  {
    SWAP(s2n[0], s2n[1]);
    SWAP(sym[0], sym[1]);
  }

  if(wrote)
    waddch(graph_scr, '\n');

  wprintw(graph_scr, "%03d", quality[0]);
  waddch(graph_scr, ':' | GRAPH_COLOR_BOLD);
  wprintw(graph_scr, "%03d", quality[1]);
  waddch(graph_scr, ':' | GRAPH_COLOR_BOLD);
  wprintw(graph_scr, "%03d ", quality[2]);

  for(i = 0; i < (GRAPH_SCR_COLS - 13); i++)
  {
    if(i < s2n[1])
      waddch(graph_scr, sym[1]);
    else if(i < s2n[0])
      waddch(graph_scr, sym[0]);
    else
      waddch(graph_scr, ' ');
  }
}

/*
 * add ap to the linked list (along with a log of the addition)
 */
#define LOG_INIT_LEN (sizeof(struct log_s *) * PREALLOC_LOG)
#define NODE_INIT_LEN (sizeof(struct node_s *) * PREALLOC_NODE)
void
add_ap(const struct aps_s *ap)
{
  int i;
  struct aps_s *ptr;

  /* reallocate aps pointer array if we're reaching the end */
  if(aps_len == aps_max)
  {
    struct aps_s **new_aps;

    if((new_aps = (struct aps_s **)realloc(aps, (aps_max + PREALLOC_APS) *
     sizeof(struct aps_s *))) == NULL)
    {
      alert("error: unable to reallocate memory: %s", strerror(errno));
      exitclean(2);
    }

    aps = new_aps;

    for(i = aps_max; i < (aps_max + PREALLOC_APS); i++)
      aps[i] = NULL;

    aps_max += PREALLOC_APS;
  }

  if((aps[aps_len] = (struct aps_s *)malloc(sizeof(struct aps_s))) == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memcpy((char *)aps[aps_len], (char *)ap, sizeof(struct aps_s));

  ptr = aps[aps_len];

  /* now pre-allocate log pointer array */
  if((ptr->log = (struct log_s **)malloc(LOG_INIT_LEN)) == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memset((char *)ptr->log, 0, LOG_INIT_LEN);
  ptr->log_max = PREALLOC_LOG;
  ptr->log_len = 0;

  add_ap_log(ptr, ap);

  /* now pre-allocate node pointer array */
  if((ptr->node = (struct node_s **)malloc(NODE_INIT_LEN)) == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memset((char *)ptr->node, 0, NODE_INIT_LEN);
  ptr->node_max = PREALLOC_NODE;
  ptr->node_len = 0;

  APNEW(aps_len);

  aps_len++;
}

/*
 * add log to the specified ap
 */
void
add_ap_log(struct aps_s *ap, const struct aps_s *info)
{
  int i;
  struct log_s *logptr;

  /* reallocate log pointer array if we're reaching the end */
  if(ap->log_len == ap->log_max)
  {
    struct log_s **new_log;

    if((new_log = (struct log_s **)realloc(ap->log,
     (ap->log_max + PREALLOC_LOG) * sizeof(struct log_s *))) == NULL)
    {
      alert("error: unable to reallocate memory: %s", strerror(errno));
      exitclean(2);
    }

    ap->log = new_log;

    for(i = ap->log_max; i < (ap->log_max + PREALLOC_LOG); i++)
      ap->log[i] = NULL;

    ap->log_max += PREALLOC_LOG;
  }

  /* make first log entry */
  if((ap->log[ap->log_len] = (struct log_s *)malloc(sizeof(struct log_s)))
   == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memset((char *)ap->log[ap->log_len], 0, sizeof(struct log_s));

  logptr = ap->log[ap->log_len];

  getgps(&logptr->ns, &logptr->ew);

  logptr->quality[0] = info->quality[0];
  logptr->quality[1] = info->quality[1];
  logptr->quality[2] = info->quality[2];

  logptr->seen.tv_sec = info->lastseen.tv_sec;
  logptr->seen.tv_usec = info->lastseen.tv_usec; 

  if(uselog)
    log_ap(log_fd, ap, logptr);

  ap->log_len++;
}
