/*
 * dstumbler v1.0 [wistat.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>

#ifdef __OpenBSD__
#include <curses.h>
#else
#include <ncurses.h>
#endif

#include "dstumbler.h"

#define wi_getval(x, y) wi_ioctl(x, y, SIOCGWAVELAN)
#define wi_setval(x, y) wi_ioctl(x, y, SIOCSWAVELAN)
#define wi_getdebugval(x, y) wi_ioctl(x, y, SIOCGPRISM2DEBUG)
#define wi_setdebugval(x, y) wi_ioctl(x, y, SIOCSPRISM2DEBUG)

/*
 * poll the wi device to determine if we're associated with an ap
 */
#define CYCLEWEP(x, y)\
{\
  if(!prism2)\
  {\
    y++;\
    y %= 2;\
  }\
\
  setcrypt(x, &y);\
}
int
wi_parse_ap(const char *iface, struct aps_s *aps)
{
  static int wep = 0;

  if(nodevice || monmode)
    return -1;

  /* first make sure we're connected to something */
  getbssid(iface, aps->bssid);
  getquality(iface, aps->quality);

  if((aps->bssid[0] == 0x44 && aps->bssid[1] == 0x44 && aps->bssid[2] == 0x44)
   || (!prism2 && aps->quality[1] == 27 && aps->quality[2] == 27) ||
   (prism2 && aps->quality[1] == 47))
  {
    CYCLEWEP(iface, wep);
    return -1;
  }

  getssid(iface, aps->ssid);
  getchan(iface, &aps->chan);

  /* manually calculate quality (s/n ratio) if we're using a prism2 */
  if(prism2)
  {
    aps->quality[2] = aps->quality[0];

    /* default noise is 27, if we get it assume it isn't accurate */
    aps->quality[0] = (aps->quality[0] == 27 &&
     aps->quality[1] < aps->quality[2] ? 0 :
     aps->quality[1] - aps->quality[2]);
  }
  /* wavelan's sometimes calculate the noise wrong... we'll compensate */
  else
    aps->quality[2] = aps->quality[1] - aps->quality[0];

  aps->weped = (!wep && !prism2 ? 2 : wep);

  CYCLEWEP(iface, wep);
  return 0;
}

/*
 * poll the wi device for aps found by it's internal ap scanning interface
 */
int
wi_parse_scan_ap(const char *iface, struct aps_s *aps, int num)
{
  static int scanbuf_len;
  static u_char scanbuf[1596];

  int i, offset;
  struct wi_req wreq;
  struct wi_scan_p2_hdr *res_h;
  struct wi_scan_res *res;

  if(nodevice || monmode || !scanmode)
    return -1;

  /*
   * use static variables along with an incrementing numerical input to
   * determine when to poll and when to return cached aps... helps keep
   * compatibility with the other scanning modes.
   */
  if(num == 0)
  {
    getscanres(iface, &wreq);

    if(prism2)
    {
      res_h = (struct wi_scan_p2_hdr *)&wreq.wi_val;

      /* if our reason is 0, it means we didn't recieve anything good */
      if(res_h->wi_reason == 0)
        return -1;

      offset = 4;
    }
    else
      offset = 0;

    memcpy((char *)scanbuf, (char *)wreq.wi_val + offset,
     (wreq.wi_len * 2) - offset);
    scanbuf_len = ((wreq.wi_len * 2) - offset) /
     (prism2 ? WI_PRISM2_RES_SIZE : WI_WAVELAN_RES_SIZE);

    setscanreq(iface);
  }

  if(num >= scanbuf_len)
    return -1;

  res = (struct wi_scan_res *)((char *)scanbuf +
   ((prism2 ? WI_PRISM2_RES_SIZE : WI_WAVELAN_RES_SIZE) * num));

  strncpy(aps->ssid, res->wi_ssid, le16toh(res->wi_ssid_len));
  aps->ssid[le16toh(res->wi_ssid_len)] = '\0';

  aps->chan = le16toh(res->wi_chan);
  aps->quality[2] = MAX(le16toh(res->wi_noise), 27);
  aps->quality[1] = MAX(le16toh(res->wi_signal), 27);
  aps->quality[0] = aps->quality[1] - aps->quality[2];

  memcpy(aps->bssid, res->wi_bssid, MACSIZE);
  aps->interval = le16toh(res->wi_interval);

  if(le16toh(res->wi_capinfo) & WI_CAPINFO_ESS)
    aps->adhoc = 2;
  if(le16toh(res->wi_capinfo) & WI_CAPINFO_IBSS)
    aps->adhoc = 1;
  if(le16toh(res->wi_capinfo) & WI_CAPINFO_PRIV)
    aps->weped = 1;
  else
    aps->weped = 2;

  /*
   * prism2 cards have supported rates in the scan response frame,
   * unfortunaltely, wavelan's do not.
   */
  if(prism2)
  {
    for(i = 0; res->wi_srates[i] != 0; i++)
      aps->srate = MAX(aps->srate, (res->wi_srates[i] & WI_VAR_SRATES_MASK));
  }

  /*
   * if we've reached the end of the list of aps, return a 0 so it won't call
   * us for more aps.. otherwise, keep on going.
   */
  return(num >= scanbuf_len - 1 ? 0 : 1);
}

/*
 * calls ioctl on a socket to the device to poll information
 */
void
wi_ioctl(const char *iface, struct wi_req *wreq, int cmd)
{
  static int s = -1;
  struct ifreq ifr;

  memset((char *)&ifr, 0, sizeof(ifr));
  strcpy(ifr.ifr_name, iface);
  ifr.ifr_data = (caddr_t)wreq;

  if(s < 0 ? (s = socket(AF_INET, SOCK_DGRAM, 0)) == -1 : 0)
  {
    alert("error: unable to create socket: %s", strerror(errno));
    exitclean(EXIT_WIERR);
  }

  if(ioctl(s, cmd, &ifr) == -1)
  {
    alert("error: unable to ioctl device socket: %s", strerror(errno));
    exitclean(EXIT_WIERR);
  }
}

/*
 * stat the card for specific info. the void * will be filled in depending
 * on the cmd. you must make sure the memory it's writing to is allocated
 * enough for the operation that's being performed.
 */
void
wi_getstat(const char *iface, int cmd, char *mem, int len)
{
  int i;
  struct wi_req wreq;

  memset((char *)&wreq, 0, sizeof(wreq));
  wreq.wi_len = WI_MAX_DATALEN;
  wreq.wi_type = cmd;

  wi_getval(iface, &wreq);

  switch(cmd)
  {
    case WI_RID_CURRENT_SSID:
    case WI_RID_OWN_SSID:
      wi_strncpyval(mem, (char *)wreq.wi_val, len);
      break;
    case WI_RID_CURRENT_BSSID:
    case WI_RID_MAC_NODE:
      memcpy(mem, (char *)wreq.wi_val, len);
      break;
    case WI_RID_CURRENT_CHAN:
    case WI_RID_PORTTYPE:
    case WI_RID_ENCRYPTION:
      *((int *)mem) = le16toh(wreq.wi_val[0]);
      break;
    case WI_RID_COMMS_QUALITY:
      for(i = 0; i < (wreq.wi_len - 1); i++)
        ((int *)mem)[i] = le16toh(wreq.wi_val[i]);
      break;
    case WI_RID_DEFLT_CRYPT_KEYS:
      memcpy(mem, (char *)&wreq, sizeof(struct wi_ltv_keys));
      break;
    case WI_RID_SCAN_RES:
      memcpy(mem, (char *)&wreq, sizeof(struct wi_req));
      break;
    case WI_RID_PRISM2:
      *((int *)mem) = le16toh(wreq.wi_val[0]);
      break;
    default:
      break;
  }
}

/*
 * set stats on the card. the void * will be parsed depending on the cmd.
 */
void
wi_setstat(const char *iface, int cmd, char *mem, int len)
{
  struct wi_req wreq;

  memset((char *)&wreq, 0, sizeof(wreq));
  wreq.wi_type = cmd;

  switch(cmd)
  {
    case WI_RID_PORTTYPE:
    case WI_RID_ENCRYPTION:
      wreq.wi_len = len;
      wreq.wi_val[0] = htole16(*((int *)mem));
      break;
    case WI_RID_OWN_SSID:
      wreq.wi_len = MIN((strlen(mem) + 1) / 2, len);
      wreq.wi_val[0] = htole16(strlen(mem));
      memcpy((char *)&wreq.wi_val[1], mem, strlen(mem));
      break;
    case WI_RID_DEFLT_CRYPT_KEYS:
      wreq.wi_len = len;
      memcpy((char *)&wreq.wi_val,
       (char *)&((struct wi_ltv_keys *)mem)->wi_keys,
       sizeof(struct wi_key) * 4);
      break;
    case WI_RID_MAC_NODE:
      wreq.wi_len = len / 2;
      memcpy((char *)&wreq.wi_val, mem, len);
      break;
    case WI_RID_SCAN_REQ:
      wreq.wi_len = (prism2 ? 3 : 1);
      wreq.wi_type = WI_RID_SCAN_REQ;

      if(prism2)
      {
        wreq.wi_val[0] = htole16(0x3FFF); /* set to scan all 14 channels */
        wreq.wi_val[1] = htole16(0x000F); /* scan on all 4 rates */
      }
      break;
    default:
      break;
  }

  wi_setval(iface, &wreq);
}

void
wi_getdebug(const char *iface, int cmd, char *mem)
{
  struct wi_req wreq;

  memset((char *)&wreq, 0, sizeof(wreq));
  wreq.wi_len = WI_MAX_DATALEN;
  wreq.wi_type = cmd;

  wi_getdebugval(iface, &wreq);

  switch(cmd)
  {
    case WI_DEBUG_MONITOR:
      *((int *)mem) = le16toh(wreq.wi_val[0]);
      break;
    default:
      break;
  }
}

/*
 * sets debug mode stats on the card.
 */
void
wi_setdebug(const char *iface, int cmd, int param0)
{
  struct wi_req wreq;

  memset((char *)&wreq, 0, sizeof(wreq));
  wreq.wi_type = cmd;

  switch(cmd)
  {
    case WI_DEBUG_STOPTEST:
    case WI_DEBUG_MONITOR:
      wreq.wi_len = 0;
      break;
    case WI_DEBUG_CHAN:
      wreq.wi_len = 1;
      wreq.wi_val[0] = htole16(param0);
      break;
    default:
      break;
  }

  wi_setdebugval(iface, &wreq);
}

/*
 * decodes a string in a wi_val
 */
void
wi_strncpyval(char *dst, const char *src, int maxlen)
{
  int i, len;

  len = le16toh(*(u_int16_t *)src);
  len = MIN(len, maxlen - 1);

  memcpy(dst, (src + sizeof(u_int16_t)), len);

  /* filter out any stray \0's */
  for(i = 0; i < len; i++)
  {
    if(dst[i] == '\0')
      dst[i] = ' ';
  }

  dst[len] = '\0';
}

/*
 *
 */
void
wi_setrandmacaddr(const char *iface)
{
  int i;
  u_char temp_macaddr[MACSIZE];

  /* set a random mac address */
  srand((u_int)(time(NULL) - (1 << 31)));

  temp_macaddr[0] = 0;
  for(i = 1; i < MACSIZE; i++)
    temp_macaddr[i] = rand() % 256;

  setmacaddr(iface, temp_macaddr);
}

/*
 * start encryption stuff (set key to bogous value so we can detect weped
 * networks)
 */
void
wi_start(const char *iface)
{
  int i, port, zero = 0;
  struct wi_ltv_keys keys;

  if(nodevice)
    return;

  /* backup port type and set to ibss/adhoc so we don't send out packetz */
  getporttype(iface, &backup.porttype);

  port = (monmode ? 3 : 1);
  setporttype(iface, &port); 

  getprism2(iface, &prism2);

  if(macreset)
  {
    /* backup the mac address if we're told to spoof */
    getmacaddr(iface, backup.macaddr);

    wi_setrandmacaddr(iface);
  }

  if(monmode)
  {
    if(!prism2)
    {
      alert("error: cannot use monitor mode on non-prism2 cards");
      exitclean(2);
    }

    /* backup if we're in monitor mode right now or not */
    getdebugmonitor(iface, &backup.monitor);

    /* set monitor mode */
    setdebugmonitor(iface);
    setdebugchan(iface, 1);

    backup.wi_started++;
    return;
  }

  if(scanmode)
  {
    setscanreq(iface);

    /*
     * some systems need the card to wait a bit before more config options are
     * set, otherwise it could lock up the system.
     */
    usleep(POLLSPEED);
  }

  /* first backup the current keys and encryption status */
  getcryptkeys(iface, &backup.keys);
  getcrypt(iface, &backup.crypt);

  memset((char *)&keys, 0, sizeof(struct wi_ltv_keys));

  /* now set our bogous info */
  for(i = 0; i < 4; i++)
  {
    keys.wi_keys[i].wi_keylen = htole16(5);
    keys.wi_keys[i].wi_keydat[0] = 0x01;
    keys.wi_keys[i].wi_keydat[1] = 0x23;
    keys.wi_keys[i].wi_keydat[2] = 0x45;
    keys.wi_keys[i].wi_keydat[3] = 0x67;
    keys.wi_keys[i].wi_keydat[4] = 0x89;
  }

  setcryptkeys(iface, &keys);
  setcrypt(iface, &zero);

  getdefaultssid(iface, backup.defaultssid);

  /* set a null ssid for the default */
  setdefaultssid(iface, "\0");

  backup.wi_started++;
}

/*
 * stop encryption stuff (restore previous values for the device)
 */
void
wi_stop(const char *iface)
{
  if(!backup.wi_started)
    return;

  setporttype(iface, &backup.porttype);

  if(macreset)
    /* reset the mac address back to what it was */
    setmacaddr(iface, backup.macaddr);

  if(monmode)
  {

    if(backup.monitor)
      setdebugmonitor(iface);
    else
      setdebugstop(iface);

    backup.wi_started = 0;
    return;
  }

  /* set the keys back to what they were */
  setcryptkeys(iface, &backup.keys);

  /* set the encryption status back */
  setcrypt(iface, &backup.crypt);

  /* set the default ssid back */
  setdefaultssid(iface, backup.defaultssid);

  backup.wi_started = 0;
}
