#include <gtk/gtk.h>

#include <pthread.h>
#include <unistd.h>

#include "gtkskan.h"
#include "iwcommon.h"
#include "wavelan_support.h"
#include "scan_data.h"
#include "network_db.h"
#include "scan_dhcp.h"
#include "skan.h"
#include "skan_config.h"

/**************************************************
 some global definitions 
 **************************************************/

extern gtkskan_config_t skan_conf;

extern GtkWidget *my_clist;
extern GtkWidget *location_entry;

/***************************************************************************
 the main skanner
 ***************************************************************************/
/* 
   set ESSID to nil to force skan for an ID
   wait a couple seconds then ask for the ESSID
   if it's set then lookup the MAC of our AP
   if it's new log it, otherwise log stats and reset/scan again

   TODO: add DHCP/NAT service detection
   TODO: maybe add GPS data
 */

void *wavelan_skanzor(void * data)
{

  int sock = socket(AF_INET, SOCK_DGRAM, 0);
  scan_data_t scan_data;
  gboolean *scanning = (gboolean*)data;
  
  while(*scanning== TRUE) {      
    /* main skanning loop - should lock this, but we'll just count on 
       one instance of it ever existing for now */

    /* check if essid is set - we might have found a network or already be 
       on one */

    gdk_threads_enter();
    if(wavelan_get_essid(sock, skan_conf.interface)) {

      gtkskan_update_statusbar("Scanning... What You Say??? We Get Signal");

      memset(&scan_data, 0, sizeof(scan_data));

      get_scan_data(sock, skan_conf.interface, &scan_data);

      {
        gps_gpgll gps_sample;
	
        if(gps_current_gll(&gps_sample)) {
          gps_merge_interface_data(&scan_data, &gps_sample);
          gps_db_save_data(&gps_sample, scan_data.ap_mac, scan_data.link_quality);
	}
      }

      if(!network_db_exists(scan_data.ap_mac)) {
//	gdk_beep();
      } 

      add_scan_data_to_clist(GTK_CLIST(my_clist), &scan_data);
      network_db_save_data(&scan_data);

      /* reset the ESSID */
      wavelan_set_essid(sock, skan_conf.interface, NULL);
      
    } else {
      gtkskan_update_statusbar("Scanning... No Signal");
    }
	gdk_threads_leave();
    /* sleep long enough to scan */
    sleep(3);
  }
  
  close(sock);
  pthread_exit(NULL);
}

void add_row_from_db(scan_data_t *row, void *user_data)
{
  add_scan_data_to_clist(GTK_CLIST(my_clist), row);
}

void initialize_interface_from_db(void)
{
  network_db_iterate(add_row_from_db, my_clist);
}

void row_to_scan_data(GtkCList *clist, int row, scan_data_t *out)
{
  gchar *col_text = NULL;

  memset(out, 0, sizeof(scan_data_t));

  /* MAC */
  gtk_clist_get_text (clist, row, 0, &col_text);
  strcpy(out->essid, col_text);

  gtk_clist_get_text (clist, row, 1, &col_text);
  strcpy(out->ap_mac, col_text);

  gtk_clist_get_text (clist, row, 2, &col_text);
  out->link_quality = atoi(col_text);

  gtk_clist_get_text (clist, row, 4, &col_text);
  out->frequency = atof(col_text);

  gtk_clist_get_text (clist, row, 5, &col_text);
  strcpy(out->location, col_text);

}

/**************************************************
 interface support 
 **************************************************/
void handle_save_location(GtkWidget *w, gpointer user_data)
{
	GtkCList *clist = GTK_CLIST(my_clist);
	GtkEntry *entry = GTK_ENTRY(location_entry);
	gchar *location_text = gtk_entry_get_text(entry);
	scan_data_t data;
	
	if (clist->focus_row == -1) {
		gtkskan_update_statusbar("Cannot save location: no AP selected");
		return;
	}
	if (location_text && strlen(location_text) > 1024) {
		gtkskan_update_statusbar("Text too long to save");
		return;
	}

    row_to_scan_data(clist, clist->focus_row, &data);
    strcpy(data.location, location_text);
    network_db_save_data(&data);
    add_scan_data_to_clist(clist, &data);
	gtkskan_update_statusbar("Location saved.");
}

void handle_clist_selection(GtkWidget *w, gint x, gint y, GdkEvent * e, gpointer user_data)
{
	GtkCList *list = GTK_CLIST(w);
	gchar * location_text = NULL;
	
	gtk_clist_get_text(list, list->focus_row, 5, &location_text);
	gtk_entry_set_text(GTK_ENTRY(location_entry), location_text);
	gtk_widget_set_sensitive(location_entry, TRUE);
}

void handle_clist_unselection(GtkWidget *w, gint x, gint y, GdkEvent *e, 
		gpointer user_data) 
{
	gtk_entry_set_text(GTK_ENTRY(location_entry), "No selection");
	gtk_widget_set_sensitive(location_entry, FALSE);
}

int find_record_or_new_row(GtkCList *clist, const char *mac)
{
  int i = 0;
  if(clist->rows == 0)
    return -1;

  /* FIXME: this will not scale to anything reasonable */
  for(i = 0; i < clist->rows; i++) {
    gchar *text;
    
    gtk_clist_get_text (clist, i, 1, &text);

    if(strcmp(text, mac) == 0) {
      return i;
    }
  }

  return -1;
}


void
add_scan_data_to_clist(GtkCList *clist, scan_data_t *data)
{
  gchar *row[] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
  int row_num = find_record_or_new_row(clist, data->ap_mac);

  char signal_buf[3], avg_signal_buf[10], freq[32], latitude[10], longitude[10];

  sprintf(signal_buf, "%d", data->link_quality);
  if(data->link_samples != 0) {
    sprintf(avg_signal_buf, "%d", (int)(data->link_total / data->link_samples));
  } else {
    memset(avg_signal_buf, 0, sizeof(avg_signal_buf));
  } 

  sprintf(freq, "%2.5f", data->frequency / (1024 * 1024 * 1024));
  sprintf(latitude, "%d.%2.3f", data->gps_coords.latitude.degrees, data->gps_coords.latitude.minutes);
  sprintf(longitude, "%d.%2.3f", data->gps_coords.longitude.degrees, data->gps_coords.longitude.minutes);

  row[0] = data->essid;
  row[1] = data->ap_mac;
  row[2] = signal_buf;
  row[3] = avg_signal_buf;
  row[4] = freq;
  row[5] = (data->location[0] == '\0' ? "Not Known" : data->location);
  row[6] = latitude;
  row[7] = longitude;

  if(row_num == -1) {
    gtk_clist_insert(clist, row_num, row);
  } else {
    gtk_clist_set_text(clist, row_num, 2, row[2]);
    gtk_clist_set_text(clist, row_num, 3, row[3]);
    gtk_clist_set_text(clist, row_num, 5, row[5]);
    gtk_clist_set_text(clist, row_num, 6, row[6]);
    gtk_clist_set_text(clist, row_num, 7, row[7]);
  }

}

void start_or_stop_scan(GtkWidget *widget, gpointer user_data) 
{
  static gboolean state = FALSE;
  gchar *status_text = NULL;
  pthread_t skannor_thread;
  
  switch(state)
    {
		case TRUE:
			state = FALSE;
			status_text = "Inactive";
			break;
    	case FALSE:
			state = TRUE;
			status_text = "Scanning...";
			break;
    }
	pthread_create(&skannor_thread, NULL, wavelan_skanzor, &state);
  
	gtkskan_scanning(state);
	gtkskan_update_statusbar((gchar*)status_text);
}

void main_window_exit(GtkWidget *widget, gpointer user_data) 
{
  gtk_main_quit();
}
