/*
 * Raw Fake AP
 *
 * Copyright (c) 2005,2006 Laurent Butti
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 */

/*
 *
 * This tool provide a basic cmdline interface to "emulate" thousands
 * of fake APs. It uses raw injection which is available on some 
 * drivers/firmwares.
 *
 * This tool is a proof-of-concept code. Use it at your own risks.
 * Do not blame me for bugs and ugly coding style!
 * 
 */

#include <stdint.h>
#include <sys/ioctl.h>
#include <netpacket/packet.h>
#include <net/if.h>
#include <netinet/ether.h>
#include <sys/time.h>
#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <time.h>
#include <pcap.h>

int handle_probe_req(const u_char *packet_data);
void close(int sock);
void usage(char *arg0);


/* Constants */
const char *author = "Laurent Butti";
const char *version = "0.2";

#define PRISM_HEADER 144

#define WLAN_HEADER_SIZE 24
#define WLAN_MGMT_FIXED_PARAM_SIZE 12
#define WLAN_TAG_PARAM_SIZE 2 + 32 + 10 + 3 + 11 + 24

#define ARPHRD_IEEE80211 801
#define ARPHRD_IEEE80211_PRISM 802

#define MAX_AP_NUMBER 10000
#define MAX_BEACON_INTERVAL 60000
#define MAX_OUI_TABLE 500
#define MAX_SSID_TABLE 500
#define MAX_PAYLOAD 512
#define MAX_SSID_LEN 32

#define RANDCRYPT 0x01
#define RANDBG 0x02
#define CHANNELHOPPING 0x04
#define TXHOPPING 0x08
#define SSIDALNUM 0x10
#define PROBERESPMODE 0x20
#define RANDMAC 0x40
#define FLAG8 0x80

/* Structures */
typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;

/* Definition of a probe request frame */
struct probe_req {
    u16 frame_control;
    u16 duration;
    u8 da[6];
    u8 sa[6];
    u8 bssid[6];
    u16 seq_ctrl;
    /* Tagged parameters */
    u8 tags[WLAN_TAG_PARAM_SIZE];
}
__attribute__ ((packed));

/* Definition of a beacon frame */
struct mgmt_beacon {
    u16 frame_control;
    u16 duration;
    u8 da[6];
    u8 sa[6];
    u8 bssid[6];
    u16 seq_ctrl;
    u64 timestamp;
    u16 beacon_int;
    u16 capab_info;
    /* Tagged parameters */
    u8 tags[WLAN_TAG_PARAM_SIZE];
}
__attribute__ ((packed));

/* Define fake aps */
struct fake_ap {
    struct mgmt_beacon *beacons;
    u8 channel[MAX_AP_NUMBER];
    u8 size[MAX_AP_NUMBER];
}
__attribute__ ((packed));

/* Define OUIs */
struct oui {
    u8 val[3];
};

/* Define SSIDs */
struct ssid {
    u8 val[MAX_SSID_LEN+1];
};

/* Global variables */
int ctrl_c = 0;
u8 channel = 1;
int isprism = 0;
int sock = 0;
int ap_number = 0;
u32 beacon_interval_ms = 100;
u32 stat_beacons, stat_bytes;
#define BROADCAST "\xff\xff\xff\xff\xff\xff"
u_char *p;
pcap_t *pcap;

/* Global structures */
struct fake_ap *fakeap;
struct oui ouis[MAX_OUI_TABLE];
struct ssid ssids[MAX_SSID_TABLE];

static void sighandler(int sig) {
    if (sig == SIGINT) {
        ctrl_c++;
        fprintf(stderr, "Info\t: User interrupt\n");
    }
}

static int
open_sock (char *ifname, int sock) {
    struct sockaddr_ll ll_addr;
    struct ifreq ifr;

    sock = socket (PF_PACKET, SOCK_RAW, ETH_P_ALL);
    if (sock == -1) {
        fprintf(stderr, "Error\t: Socket failed\n");
        return (-1);
    }

    ll_addr.sll_family = AF_PACKET;
    ll_addr.sll_protocol = 0;
    ll_addr.sll_halen = ETH_ALEN;

    strncpy(ifr.ifr_name, ifname, IFNAMSIZ);

    if (ioctl(sock, SIOCGIFINDEX, &ifr) < 0) {
        fprintf(stderr, "Error\t: ioctl(SIOCGIFINDEX) failed\n");
        return (-1);
    }

    ll_addr.sll_ifindex = ifr.ifr_ifindex;

    if (ioctl(sock, SIOCGIFHWADDR, &ifr) < 0) {
        fprintf(stderr, "Error\t: ioctl(SIOCGIFHWADDR) failed\n");
        return (-1);
    }

    memcpy(ll_addr.sll_addr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);

    if (ifr.ifr_hwaddr.sa_family != ARPHRD_IEEE80211 &&
            ifr.ifr_hwaddr.sa_family != ARPHRD_IEEE80211_PRISM) {
        fprintf( stderr, "Error\t: bad linktype\n");
        return (-1);
    }

    if (bind (sock, (struct sockaddr *)&ll_addr, sizeof(ll_addr)) == -1) {
        fprintf(stderr, "Error\t: bind failed\n");
        close(sock);
        return (-1);
    }

    return sock;
}

void pcap_callback (u_char *p, const struct pcap_pkthdr *pkthdr, const u_char *packet_data) {
    /* Check for probe request frames */
    if (packet_data[0 + isprism * PRISM_HEADER] == 0x40) {
        handle_probe_req (packet_data);
        return;
    }
    return;
}

int handle_probe_req (const u_char *packet_data) {
    int i;
    u8 mac_tmp[6];

    /* Check if probe request is broadcast directed */
    if (memcmp(packet_data + 4
               + isprism * PRISM_HEADER, BROADCAST, 6) != 0) {
        fprintf(stderr, "Info\t: Probe Request non broadcasted\n");
        return -1;
    }

    /* Retrieve source mac address */

    memcpy(mac_tmp, &packet_data[10 + isprism * PRISM_HEADER], 6);

    /* Select the fake AP regarding to the channel */
    /* Send back appropriate responses (several APs per channels) */

    for (i = 0; i < ap_number; i++) {
        if (fakeap->channel[i] == channel) {
            /* Copy MAC */
            memcpy(fakeap->beacons[i].da, mac_tmp, 6);
            /* Send probe response */
            send(sock, &(fakeap->beacons[i]), fakeap->size[i], 0);
            /* Update sequence number */
            fakeap->beacons[i].seq_ctrl = ((((fakeap->beacons[i].seq_ctrl) & 0xFFF0) >> 4) + 1) << 4;
            /* Update BSS Timestamp */
            /* FIXME: To be tuned */
            fakeap->beacons[i].timestamp += (beacon_interval_ms * 1024);

            /* Update statistics */
            stat_beacons++;
            stat_bytes += fakeap->size[i];
        }
    }
    return -1;
}

int
main (int argc, char **argv) {

    /* Options */
    char *ifname = NULL;
    char *oui_filename = NULL;
    char *ssid_filename = NULL;
    char *str_mac = NULL;
    char c;
    u8 flags;
    char err[PCAP_ERRBUF_SIZE];

    /* Construction */
    u8 ssid_length;
    u8 rand_mac[6];
    u8 mac_dst[6];

    /* General purpose */
    u8 cmd[32];
    u16 i, j, k;
    u16 oui_table_len, ssid_table_len;
    u32 stat_pps;
    float elapsed_time;
    struct timeval time_start, time_end, delta;
    struct timespec ts;

    /* Initialize default values */
    flags = 0;
    sock = 0;
    ap_number = 10;
    oui_table_len = ssid_table_len = 0;
    beacon_interval_ms = 100;
    stat_beacons = stat_bytes = stat_pps = 0;
    memset(mac_dst, 0, sizeof(mac_dst));
    memset(cmd, 0, sizeof(cmd));

    printf("\nRaw Fake AP (%s) proof-of-concept code\n"
           "Copyright (c) 2005-2006, %s\n\n"
           , version, author);

    /* Retrieve command line options */
    while ((c = getopt(argc, argv, "hi:wgctsrmb:n:o:p:q:")) != EOF) {
        switch(c) {
        case 'h':
            usage(argv[0]);
            return -1;
        case 'i':
            if (optarg) {
                ifname = optarg;
                break;
            } else {
                usage(argv[0]);
                return -1;
            }
        case 'w':
            flags |= RANDCRYPT;
            break;
        case 'g':
            flags |= RANDBG;
            break;
        case 'c':
            flags |= CHANNELHOPPING;
            break;
        case 't':
            flags |= TXHOPPING;
            break;
        case 's':
            flags |= SSIDALNUM;
            break;
        case 'r':
            flags |= PROBERESPMODE;
        case 'm':
            flags |= RANDMAC;
            break;
        case 'b':
            if (optarg) {
                beacon_interval_ms = atoi(optarg);
                break;
            } else {
                usage(argv[0]);
                return -1;
            }
        case 'n':
            if (optarg) {
                ap_number = atoi(optarg);
                break;
            } else {
                usage(argv[0]);
                return -1;
            }
        case 'o':
            if (optarg) {
                oui_filename = optarg;
                break;
            } else {
                usage(argv[0]);
                return -1;
            }
        case 'p':
            if (optarg) {
                ssid_filename = optarg;
                break;
            } else {
                usage(argv[0]);
                return -1;
            }
        case 'q':
            if (optarg) {
                str_mac = (u_char *) ether_aton(optarg);
                memcpy(mac_dst, str_mac, sizeof(mac_dst));
                break;
            } else {
                usage(argv[0]);
                return -1;
            }
        default:
            usage(argv[0]);
            return -1;
        }
    }

    if (ap_number > MAX_AP_NUMBER) {
        fprintf(stderr, "Error\t: Invalid Fake AP number value, must be below %d\n", MAX_AP_NUMBER);
        usage(argv[0]);
        return -1;
    }

    if (beacon_interval_ms > MAX_BEACON_INTERVAL) {
        fprintf(stderr, "Error\t: Invalid delay value, must be below %d ms\n", MAX_BEACON_INTERVAL);
        usage(argv[0]);
        return -1;
    }

    if (ifname == NULL) {
        fprintf(stderr, "Error\t: Ifname must be specified\n");
        usage(argv[0]);
        return (-1);
    }

    fakeap = (struct fake_ap *) malloc (sizeof (struct fake_ap));
    fakeap->beacons = (struct mgmt_beacon * ) malloc (MAX_AP_NUMBER * sizeof(struct mgmt_beacon));

    if ((fakeap == NULL) || (fakeap->beacons == NULL)) {
        fprintf(stderr, "Error\t: malloc() failed\n");
        return (-1);
    }

    /* Seed the random engine */
    srandom(time(NULL));

    /* Fill a table with valid OUIs */
    if (oui_filename != NULL) {
        FILE *ap_oui;
        char line[10];
        u16 i = 0;

        if (oui_filename == NULL)
            return (-1);

        ap_oui = fopen(oui_filename, "r");

        if (ap_oui == NULL) {
            fprintf(stderr, "Error\t: Unknown '%s' file\n", oui_filename);
            return (-1);
        }

        fprintf(stderr, "Info\t: Reading '%s' file\n", oui_filename);

        while (fgets(line, sizeof(line), ap_oui)) {
            i++;
            if ((*line != '#') && ((*line))) {
                sscanf(line, "%2x:%2x:%2x",
                       &ouis[i].val[0], &ouis[i].val[1], &ouis[i].val[2]);
            }
        }
        oui_table_len = i;
    }

    /* Fill a table with valid ESSIDs */
    if (ssid_filename != NULL) {
        FILE *ap_ssid;
        char line[MAX_SSID_LEN+2];
        u16 i = 0;

        if (ssid_filename == NULL)
            return (-1);

        ap_ssid = fopen(ssid_filename, "r");

        if (ap_ssid == NULL) {
            fprintf(stderr, "Error\t: Unknown '%s' file\n", ssid_filename);
            return (-1);
        }

        fprintf(stderr, "Info\t: Reading '%s' file\n", ssid_filename);

        while (fgets(line, sizeof(line), ap_ssid)) {
            if ((*line != '#') && ((*line))) {
                sscanf(line, "%33c", &ssids[i].val);
                if (strlen(ssids[i].val) < MAX_SSID_LEN + 1) {
                    ssids[i].val[strlen(ssids[i].val)-1] = '\0';
                } else {
                    ssids[i].val[32] = '\0';
                }
            }
            i++;
        }
        ssid_table_len = i;
    }

    /* Build all beacon or probe response frames */
    for (i = 0; i < ap_number ; i++) {
        u8 offset = 0;

        /* Create beacon or probe response frames */
        if (!(flags & PROBERESPMODE))
            fakeap->beacons[i].frame_control = 0x0080;
        else
            fakeap->beacons[i].frame_control = 0x0050;

        /* Duration field is usually set to zero */
        fakeap->beacons[i].duration = 0x0000;

        /* Try to make valid OUIs */
        memset(rand_mac, 0, sizeof(rand_mac));
        if (oui_filename != NULL) {
            u8 oui_rand = rand() % oui_table_len;
            rand_mac[0] = ouis[oui_rand].val[0];
            rand_mac[1] = ouis[oui_rand].val[1];
            rand_mac[2] = ouis[oui_rand].val[2];
        } else {
            rand_mac[0] = rand() & 0x03;
            rand_mac[1] = rand() & 0xE1;
            rand_mac[2] = rand() & 0xFE;
        }

        rand_mac[3] = rand() & 0xFE;
        rand_mac[4] = rand() & 0xFE;
        rand_mac[5] = rand() & 0xFE;
        memcpy(fakeap->beacons[i].sa, rand_mac, sizeof(rand_mac));
        memcpy(fakeap->beacons[i].bssid, rand_mac, sizeof(rand_mac));
        if (str_mac != NULL) {
            memcpy(fakeap->beacons[i].da, mac_dst, sizeof(mac_dst));
        } else
            memcpy(fakeap->beacons[i].da, BROADCAST, 6) ;

        /* Do not fragment beacon frames and elect a sequence number */
        fakeap->beacons[i].seq_ctrl = rand() & 0xFFF0;

        /* Try to make coherent timestamps (not low/high, i.e. on 32b) */
        fakeap->beacons[i].timestamp = (u64) (rand());
        fakeap->beacons[i].beacon_int = (u16) (beacon_interval_ms);
        fakeap->beacons[i].capab_info = 0x0000 | 0x0001;

        /* Fill first tagged parameter with 0 (SSID parameter set) */
        fakeap->beacons[i].tags[offset++] = 0x00;
        /* Fill SSID length */
        fakeap->beacons[i].tags[offset++] = ssid_length = (u8)(rand() % 32 + 1);

        /* Fill SSID */
        u8 randcloaked = rand() % 10;
        /* A certain percentage of SSIDs are cloaked */
        if (randcloaked > 8) {
            /* Some are cloaked with several 0x00 bytes */
            if ((rand() % 2) == 1)
                for (j = offset; j < offset + ssid_length; j++) {
                    fakeap->beacons[i].tags[j] = 0x00;
                }
            /* Some are zero length */
            else {
                ssid_length = 1;
                fakeap->beacons[i].tags[offset - 1] = ssid_length;
                fakeap->beacons[i].tags[offset] = 0x00;
            }
        } else {
            if (flags & SSIDALNUM) {
                for (j = offset; j < offset + ssid_length; j++) {
                    /* Alnum caracters are 48-57, 65-90, 97-122 */
                    u8 alnumrand = rand() % 3;
                    if (alnumrand == 0)
                        fakeap->beacons[i].tags[j] = (rand() % 9) + 48;
                    if (alnumrand == 1)
                        fakeap->beacons[i].tags[j] = (rand() % 25) + 65;
                    if (alnumrand == 2)
                        fakeap->beacons[i].tags[j] = (rand() % 25) + 97;
                }
            } else if (ssid_filename != NULL) {
                u8 ssidrand = rand() % ssid_table_len;
                k = 0;
                ssid_length = strlen(ssids[ssidrand].val);
                fakeap->beacons[i].tags[offset-1] = ssid_length;
                for (j = offset; j < offset + ssid_length; j++)
                    fakeap->beacons[i].tags[j] = ssids[ssidrand].val[k++];
            } else {
                /* Others are fully randomized */
                for (j = offset; j < offset + ssid_length; j++) {
                    fakeap->beacons[i].tags[j] = rand() % 0xFF;
                }
            }
        }
        offset += ssid_length;

        /* Fill second tagged parameter with 1 (Supported Rates) */
        /* Fill supported rates length */
        /* Fill supported rates values */
        memcpy(fakeap->beacons[i].tags + offset, "\x01\x04\x82\x84\x8B\x96", 6);

        if ((flags & RANDBG) && ((rand() % 2) == 1)) {
            memcpy(fakeap->beacons[i].tags + offset, "\x01\x08\x82\x84\x8B\x96\x24\x30\x48\x6C", 10);
        }

        offset += fakeap->beacons[i].tags[offset + 1] + 2;

        /* Fill the third tagged parameter with 3 (DS Parameter Set, Current Channel) */
        fakeap->beacons[i].tags[offset++] = 0x03;
        /* Fill the current channel length (should be set to 1) */
        fakeap->beacons[i].tags[offset++] = 0x01;
        /* Fill the current channel with garbage */
        fakeap->beacons[i].tags[offset++] = fakeap->channel[i] = rand() % 13 + 1;

        /* Randomize Open/WEP/WPA/RSN capabilities */
        if (flags & RANDCRYPT) {
            u8 crypto = rand() % 4;
            u8 auth = rand() % 2;
            u8 *wpa_tkip_tkip_eap = "\xdd\x16\x00\x50\xf2\x01\x01\x00\x00\x50\xf2\x02\x01\x00\x00\x50\xf2\x02\x01\x00\x00\x50\xf2\x01";
            u8 *wpa_tkip_tkip_psk = "\xdd\x16\x00\x50\xf2\x01\x01\x00\x00\x50\xf2\x02\x01\x00\x00\x50\xf2\x02\x01\x00\x00\x50\xf2\x02";
            u8 *rsn_ccmp_ccmp_psk = "\x30\x12\x01\x00\x00\x0f\xac\x04\x01\x00\x00\x0f\xac\x04\x01\x00\x00\x0f\xac\x02";
            u8 *rsn_ccmp_ccmp_eap = "\x30\x12\x01\x00\x00\x0f\xac\x04\x01\x00\x00\x0f\xac\x04\x01\x00\x00\x0f\xac\x01";

            if (crypto == 1) {
                fakeap->beacons[i].capab_info |= 0x0010;
            }
            if (crypto == 2) {
                fakeap->beacons[i].capab_info |= 0x0010;
                if (auth == 1) {
                    memcpy(fakeap->beacons[i].tags + offset, wpa_tkip_tkip_psk, 24);
                } else {
                    memcpy(fakeap->beacons[i].tags + offset, wpa_tkip_tkip_eap, 24);
                }
                offset += 24;
            }
            if (crypto == 3) {
                fakeap->beacons[i].capab_info |= 0x0010;
                if (auth == 1) {
                    memcpy(fakeap->beacons[i].tags + offset, rsn_ccmp_ccmp_psk, 20);
                } else {
                    memcpy(fakeap->beacons[i].tags + offset, rsn_ccmp_ccmp_eap, 20);
                }
                offset += 20;
            }
        }

        /* Fill a proprietary tagged parameter */
        memcpy(fakeap->beacons[i].tags + offset, "\xDE\x09\x52\x41\x57\x46\x41\x4B\x45\x41\x50", 11);
        offset += 11;

        /* Calculate size for convenience */
        fakeap->size[i] = WLAN_HEADER_SIZE + WLAN_MGMT_FIXED_PARAM_SIZE + offset;
    }

    /* Open raw socket */
    sock = open_sock (ifname, sock);

    if (sock == -1 ) {
        fprintf(stderr,
                "Error\t: Cannot open socket\n"
                "Info\t: Must be root with a 802.11 card with RFMON enabled\n");
        return (-1);
    }

    fprintf(stderr, "Info\t: Injecting %d fake access points via %s\n\n", ap_number, ifname);

    if (gettimeofday(&time_start, NULL) == -1)
        fprintf(stderr, "Error\t: Can not set timer\n");

    /* Main loop for injection in probe response mode */
    if (flags & PROBERESPMODE) {

        pcap = pcap_open_live(ifname, 512, 1, 1, err);

        if (pcap == NULL) {
            fprintf(stderr, "Error\t: pcap_open_live failed (%s)\n", err);
            return -1;
        }

        pcap_setnonblock(pcap, 1, err);

        /* Check FCS frame depending on chipsets (command-line) */

        /* Check datalink for prism headers or not */
        switch (pcap_datalink(pcap)) {
        case DLT_PRISM_HEADER:
            fprintf(stderr, "Info\t: PRISM datalink type\n");
            isprism = 1;
            break;
        case DLT_IEEE802_11:
            fprintf(stderr, "Info\t: IEEE802_11 datalink type\n");
            isprism = 0;
            break;
        default:
            fprintf(stderr, "Error\t: Unknown datalink type\n");
            return -1;
        }

        while(ctrl_c == 0) {
            /* Loop waiting for capture */
            /* Waiting for about ten packets */
            pcap_loop(pcap, 10, pcap_callback, (u_char *) p);

            /* Change channel */
            /* Basic technique, should use wireless ioctls instead */

            channel = channel % 13 + 1;
            snprintf(cmd, sizeof(cmd), "iwconfig %s channel %d", ifname, channel);
            system(cmd);

            /* Change tx power */
            /* Basic technique, should use wireless ioctls instead */
            if (flags & TXHOPPING) {
                snprintf(cmd, sizeof(cmd), "iwconfig %s txpower %dmW", ifname, (rand() % 0x60));
                system(cmd);
            }

            /* Catch SIGINT */
            signal(SIGINT, sighandler);
        }
    }

    /* Main loop for injection in Beacon mode */
    if (!(flags & PROBERESPMODE)) {
        while(ctrl_c == 0) {
            for (i = 0; i < ap_number ; i++) {
                clock_gettime(CLOCK_REALTIME, &ts);

                /* Try to calculate a coherent interval before sending */
                ts.tv_nsec += ( ((int)(beacon_interval_ms) * 1024000) / ((int)(ap_number)) - 1000000);
                if(ts.tv_nsec > 1000000000) {
                    ts.tv_nsec -= 1000000000;
                    ts.tv_sec += 1;
                }

                /* Change channel */
                /* Basic technique, should use wireless ioctls instead */
                if (flags & CHANNELHOPPING) {
                    snprintf(cmd, sizeof(cmd), "iwconfig %s channel %d", ifname, fakeap->channel[i]);
                    system(cmd);
                }

                /* Change tx power */
                /* Basic technique, should use wireless ioctls instead */
                if (flags & TXHOPPING) {
                    snprintf(cmd, sizeof(cmd), "iwconfig %s txpower %dmW", ifname, (rand() % 0x60));
                    system(cmd);
                }

                /* Send beacon frame */
                send(sock, &(fakeap->beacons[i]), fakeap->size[i], 0);

                /* Update timestamp */
                fakeap->beacons[i].timestamp += (beacon_interval_ms * 1024);

                /* Increment sequence number */
                fakeap->beacons[i].seq_ctrl = ((((fakeap->beacons[i].seq_ctrl) & 0xFFF0) >> 4) + 1) << 4;

                /* Do basic statistics */
                stat_beacons++;
                stat_bytes += fakeap->size[i];

                /* Catch SIGINT */
                signal(SIGINT, sighandler);

                /* Try to wait with a coherent beacon interval */
                /* But will fatally jitter: difficult to achieve */
                clock_nanosleep(CLOCK_REALTIME, TIMER_ABSTIME, &ts, NULL);
            }
        }
    }

    if (gettimeofday(&time_end, NULL) == -1)
        fprintf(stderr, "Error\t: Can not set timer\n");

    (&delta)->tv_sec = (&time_end)->tv_sec - (&time_start)->tv_sec;
    (&delta)->tv_usec = (&time_end)->tv_usec - (&time_start)->tv_usec;
    if ((&delta)->tv_usec < 0) {
        (&delta)->tv_sec--;
        (&delta)->tv_usec += 1000000;
    }

    elapsed_time = (float)(delta.tv_sec) + (float)(delta.tv_usec) / 1000000;
    stat_pps = stat_beacons / elapsed_time;

    printf("Fake AP\t: %d emulated\n"
           "Responses\t: %d injected\n"
           "PPS\t: %d packets per second\n"
           "Bytes\t: %d sent\n"
           "Time\t: %f seconds\n"
           , ap_number, stat_beacons, stat_pps, stat_bytes, elapsed_time);

    /* Close socket and free memory */
    close(sock);
    free(fakeap->beacons);
    free(fakeap);
    return 0;

}

void usage(char *arg0) {
     printf("usage: %s [-h] [-i <ifname>] [-gctsfm] [-b <int>] [-n <num>] [-o <oui_file>] [-p <ssid_file>] [-q <mac>]\n"
           "\t-h\t\t: display this message\n"
           "\t-i <ifname>\t: select interface for raw injection\n"
           "\t-w\t\t: randomize Open/WEP/WPA/RSN access points (default: no)\n"
           "\t-g\t\t: randomize b/g cards (default: no)\n"
           "\t-c\t\t: do channel hopping (default: no)\n"
           "\t-t\t\t: do tx power hopping (default: no)\n"
           "\t-s\t\t: send valid alnum characters in ESSID (default: no)\n"
           "\t-r\t\t: send probe response frames (default: no)\n"
           "\t-m\t\t: randomize MAC addresses (default: no, use valid OUIs)\n"
           "\t-b <int>\t: beacon interval in 1.024 ms multiple (default: 100)\n"
           "\t-n <num>\t: number of fake APs to emulate (default: 10)\n"
           "\t-o <oui_file>\t: oui file (default: no)\n"
           "\t-p <ssid_file>\t: essid file (default: no)\n"
           "\t-q <mac>\t: destination mac address xx:..:xx (default: broadcast)\n"
           , arg0);
}
