/**
 * @file linux.c
 *
 * Linux versions of the APIs required by the discovery core.
 *
 * wicrawl - A modular and thorough wi-fi scanner
 * http://midnightresearch.com/projects/wicrawl - for details
 *
 * Original Code: jspence, Focus
 * Contributors:
 * $Id: linux.c,v 1.20 2006/09/12 04:47:55 jspence Exp $
 *
 * Copyright (C) 2005-2006 Midnight Research Laboratories
 *
 * THIS SOFTWARE IS PROVIDED "AS IS". NO WARRANTY IS ASSUMED.
 * NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED. IT CAN BURN
 * YOUR HARD DISK, ERASE ALL YOUR DATA AND BREAK DOWN YOUR
 * MICROWAVE OVEN. YOU ARE ADVISED.
 *
 * wicrawl is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.  For details see doc/LICENSE.
 *
*/

#include "wicrawl.h"

#include <err.h>

/**
 * Use the SIOCGIWNAME ioctl on a specific interface to see if it
 * supports the Linux wireless API.
 *
 * @param iface [in] A pointer to a C string with the name of the
 * interface to check.
 *
 * @return 1 if the interface supports the Linux wireless API, 0
 * otherwise.
 */
int isWireless(const char * iface)
{
  int s;
  int rc;
  int ret;
  struct iwreq req;

  s = socket(AF_INET, SOCK_DGRAM, 0);
  if(s == -1)
    err(EXIT_FAILURE, "socket() failed to create an ioctl socket");

  memcpy(req.ifr_ifrn.ifrn_name, iface,IFNAMSIZ-1);

  rc = ioctl(s, SIOCGIWNAME, &req);
  if(rc == -1)
    ret = 0;
  else
    ret = 1;

  close(s);

  return ret;
}

/**
 * See if this is an aironet card.  They use a similar naming scheme
 * but a different config API, so we have to explicitly test for them
 * when we see an interface name that could be them.
 *
 * @param iface [in] A character string with the name of the interface
 * to check for aironetness.
 *
 * @return If the interface is an Aironet card, you get a file pointer
 * to the Status file in /proc for the driver.  If not, you get NULL.
 * You must close the returned file pointer with fclose(3) when you
 * are finished screwing with it.
 */
static FILE * isAironet(const char * iface) {
  FILE * fp;
  int count;
  char fnbuf[64];

  count = snprintf(fnbuf, sizeof(fnbuf) - 1, "/proc/driver/aironet/%s/Status", iface);
  if(count < 0)
    return 0;

  fnbuf[count] = '\0';

  fp = fopen(fnbuf, "rw");

  return fp;
}

/**
 * Enable an interface by using the SIOCGIFFLAGS ioctl on it to set
 * the IFF_UP bit.
 *
 * @param iface [in] A pointer to a C string with the name of the
 * interface to enable.
 */
static void enableInterface(char * iface)
{
  int s;
  int rc;
  struct ifreq req;

  s = socket(AF_INET, SOCK_DGRAM, 0);
  if(s == -1) {
    warn("socket() failed to create an ioctl socket in enableInterface");
    return;
  }

  memset(&req, 0, sizeof(req));
  memcpy(req.ifr_ifrn.ifrn_name, iface, IFNAMSIZ - 1);
  rc = ioctl(s, SIOCGIFFLAGS, &req);
  if(rc == -1) {
    warn("ioctl(SIOCGIFFLAGS) failed in enableInterface()");
    goto out;
  }

  req.ifr_ifru.ifru_flags |= IFF_UP;

  rc = ioctl(s, SIOCSIFFLAGS, &req);
  if(rc == -1) {
    warn("ioctl(SIOCSIFFLAGS) failed in enableInterface()");
    goto out;
  }

 out:
  close(s);
}

// 1 == associated, 0 == not so much.
int isAssociated(char * iface)
{
  int s;
  int rc;
  int ret;
  struct iwreq req;
  bssid_t zero_bssid = { 0, 0, 0, 0, 0, 0 };

  s = socket(AF_INET, SOCK_DGRAM, 0);
  if(s == -1) {
    warn("socket() failed in isAssociated()");
    return 0;
  }
   
  memset(&req, 0, sizeof(req));
  memcpy(req.ifr_name, iface, strlen(iface));
  rc = ioctl(s, SIOCGIWAP, &req);
  if(rc == -1) {
    warn("ioctl(SIOCGIWAP) failed in isAssociated()");
    return 0;
  }

  if(!memcmp(req.u.ap_addr.sa_data, zero_bssid, 6))
    ret = 0;
  else
    ret = 1;

  return ret;
}

int setMonitorMode(char *iface, int enable)
{
  int rc;

  int s;
  struct protoent * proto;
  
  proto = getprotobyname("udp");
  if(proto == NULL) {
    warn("getprotobyname() failed in setMonitorMode()");
    return 0;
  }
 
  s = socket(AF_INET, SOCK_DGRAM, proto->p_proto);
  if(s == -1) {
    warn("socket() failed in setMonitorMode()");
    return 0;
  }

  // First, turn the interface on.
  enableInterface(iface);

  // "ath" interfaces are:
  //   Atheros (madwifi)
  
  // "eth" interfaces can be:
  //   Intel 2100 series (ipw2100)
  //   Intel 2200 series (ipw2200)
  //   Cisco Aironet cards (airo)

  // Is it an aironet card?
  if(! strncmp(iface, "eth", 3)) {
    FILE * fp;

    fp = isAironet(iface);
    if(fp != NULL) {
      // Yup, the file's there, so this interface must be an Aironet card.
      
      fprintf(fp, "Mode: %s\n", enable ? "rfmon" : "ESS");
      fclose(fp);
    }
  }

  // Both ath and eth interfaces use the linux wireless API for monitor mode.
  if(! strncmp(iface, "eth", 3) || ! strncmp(iface, "ath", 3) || ! strncmp(iface, "wlan", 4) || ! strncmp(iface, "wifi", 4)) {
    struct iwreq req;

    memset(&req, 0, sizeof(req));

    strncpy(req.ifr_ifrn.ifrn_name, iface, sizeof(req.ifr_ifrn.ifrn_name));
      
    rc = ioctl(s, SIOCGIWMODE, &req);
    if(rc == -1) {
      warn("ioctl(SIOCGWIMODE) failed on %s", iface);
      return 0;
    }
    
    if(enable) {
      req.u.mode = IW_MODE_MONITOR;
    }
    else {
      // Now, you'd think you could just turn off the monitor bit, but
      // it turns out that the atheros driver whines when no mode is set.
      // I think that's kind of funny, because no bits is defined as
      // IW_MODE_AUTO, where the driver picks the appropriate mode...
      req.u.mode = IW_MODE_INFRA;
    }
      
    rc = ioctl(s, SIOCSIWMODE, &req);
    if(rc == -1) {
      warn("ioctl(SIOCSWIMODE) failed to %s monitor mode on %s", 
           enable ? "enable" : "disable",
           iface);
      return 0;
    }

    return 1;
  }

  // "wlan" interfaces can be:
  //   Prism 2 (wlan-ng)
  else if(! strncmp(iface, "wlan", 4)) {
    // wlan-ng version
    {
#if 0
      p80211ioctl_req_t req;

      req.magic = P80211_IOCTL_MAGIC;

      // see wlanctl.c in the linux-wlan sources for an example
#endif

      int pid;
      int status;
      char * args[5];

      // for now, we just call wlanctl-ng
      pid = fork();
      if(! pid) {
        args[0] = "wlanctl-ng";
        args[1] = "lnxreq_wlansniff";
        args[2] = enable ? "enable=true" : "enable=false"; 
        args[3] = "channel=6";
        args[4] = NULL;
        execve("wlanctl-ng", args, NULL);
      }

      waitpid(pid, &status, 0);

      if(WEXITSTATUS(status)) {
        warnx("setMonitorMode() failed to enable monitor mode for %s", iface);
        return 0;
      }

      return 1;
    }

    close(s);

    return 1;
  }
  else {
    warnx("setMonitorMode() doesn't know how to enable monitoring on interface %s", iface);
    return 0;
  }
}

void * channelHopper(void * arg)
{
  int s;
  FILE * airo_fp;
  unsigned int channel;
  unsigned int lastChannel;
  chanhop_request_t * cr;
  char buf[16];
  int hopFail = 0;

  cr = (chanhop_request_t *) arg;
  
  airo_fp = isAironet(cr->iface);

  channel = 1; 

  while(cr->shutdown != 0) {
    if(airo_fp != NULL) {
      
      snprintf(buf, sizeof(buf) - 1, "Channel: %u", channel + 1);
      fprintf(airo_fp, "%s", buf);
      channel++;
      // there are supposed to be wireless-G and -A cards, but we 
      // don't know how to test for them.  Assume B for now.
      //channel %= 11;
    }

    {
      struct iwreq wrq;
      //Zero out the struct
      memset(&wrq,0,sizeof(struct iwreq));
      strncpy(wrq.ifr_name,cr->iface,IFNAMSIZ-1);
   
     

      //There is prob a way better way then doing it this way, but
      //this will get us psuedo random channel hopper.We also wanna
      //make sure we don't get the same channel twice
      while(channel == lastChannel){
        channel = 1+(int)(11.0*rand()/(RAND_MAX+1.0));
      }
      wrq.u.freq.m = channel;
      // strcpy(wrq.u.freq.m , channel , sizeof(channel));

      s = socket(AF_INET, SOCK_DGRAM, 0);
      if(s == -1){
        warn("socket() failed to create a dinky socket for ioctl()");
        return NULL;
      }
      //As it turns out Madwifi drivers sometimes needs a second chance, 
      //so we won't spit out a warning unless it truely fails.  
      //And even if it does fail, oh well.. we will just keep going
      //if it fails more then 10 times, then we will fail and return null;
      //
      if(ioctl(s, SIOCSIWFREQ, &wrq) < 0){
              
        if (ioctl(s,SIOCSIWFREQ,&wrq) < 0){
          //A Debug message , not that it matters but its here..
          printf("Tried to hop to channel %i on device %s \n",channel,cr->iface);

          hopFail++;
        }
              
      }
      if(hopFail >= 10){
        //For now we will do nothing .. there appears to be a problem
        //with setting the cards into monitor mode. We need to verify
        //that the card is infact in monitor mode before we 
        //kick off this thread
        warn("Failed 10+ times on a channel hop.");
        //      return NULL;
        hopFail = 0;
      }

      ap.channel = channel;
      
      close(s);
      lastChannel = channel;
      usleep(cr->delay_usecs);
    }
  }

  return NULL;
}

