/*IO Operations
 *Written by Brett Lomas
 *Copyright 1999
 */


/*Version 0.2*/

/*Changes
 *0.2:
 *  Minor Bug fixes, and made more robust (I hope)
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "io.h"


#define INCR 512

static int locked = 0;
static int amountread  = 0;
static int file_size = 0;
static char *comments = NULL;
//static FILE *file = NULL;
static int fd = -1;

int setFileIOpath(char *path, char *comm, int lock) {
  /*Will setup the file IO from the path to the file, and a string of character comments
   *(NULL = no comment lines)
   */

  if(path == NULL)
    return 0;

  fd = open(path, O_RDONLY);
  if(fd == -1) {
#ifdef DEBUG
    perror("Error opening the file");
#endif
    return 0;
  }

  struct stat stats;
  if(fstat(/*fileno(file)*/fd, &stats) == -1) {
#ifdef DEBUG
    perror("Error getting file size");
#endif
    //fclose(file);
    close(fd);
    //file = NULL;
    fd = -1;
    return 0;
}
  
  file_size = stats.st_size;
  if(comm !=  NULL) {
    comments = (char *)malloc(strlen(comm) + 1);
    if(comments == NULL) {
#ifdef DEBUG
      perror("Error allocating memory");
#endif
      //fclose(file);
      close(fd);
      //file = NULL;
      fd = -1;
      file_size = 0;
      return 0;
    }

    strcpy(comments, comm);
  }

  /*Added 9/5/99 for filelocking*/
  if(lock && lockf(fd, F_TLOCK, 0) == -1) {
    if(errno == EACCES || errno == EAGAIN) {
      char *err = strerror(errno);
      fprintf(stderr, "Error opening %s -> %s\n", path, err);
      free(err);
    }
    else
      perror("Error locking the file");

    return 0;
  }
  else
    locked = 1;
  
  return 1;
}

int resetFileIO() {
  /*Reset the file IO, ready for another set*/

  if(locked)
    if(lockf(/*fileno(file)*/fd, F_ULOCK, 0) == -1)
      perror("Error releasing the lock on the file");

  /*
  if(file != NULL) {
    fclose(file);
    file = NULL;
  }
  */

  if(fd != -1) {
    close(fd);
    fd = -1;
  }

  if(comments != NULL) {
    free(comments);
    comments = NULL;
  }

  amountread = 0;
  file_size = 0;
  locked = 0;

  return 1;

}

static char *readline_() {
  /*This is an interal support function to the external readline()
   *This function will return the next available line in the file, without consideration of comments
   *or empty lines etc
   *
   *If there is nothing left to read in the file, will return NULL
   */

  char *line = NULL;
  int size = 0;

  if(amountread >= file_size || fd == -1)
    return NULL; /*EOF reached or IO has not been set!!*/

  char *temp;
  do {
    temp = (char *)realloc(line, size + INCR);
    if(temp == NULL) {
#ifdef DEBUG
      perror("Error allocating memory");
#endif
      if(line != NULL)
	free(line);
      return NULL;
    }

    line = temp;

    memset(line + size, '\0', INCR);

    //    if((size += fread(line + size, INCR - 1, 1, file)) != 1 && errno) {
    if((size += read(fd, line + size, INCR - 1) == -1)) {
#ifdef DEBUG
      perror("Error reading from file");
#endif
      free(line);
      return NULL;
    }
//    size += INCR;
  }
  while(strchr(line, '\n') == NULL && (amountread + strlen(line) < (unsigned int)file_size));

    char *ptr = strchr(line, '\n');

    if(ptr != NULL)
      *ptr = '\0'; /*This terminates the string at the first occurance of a new line*/
  
  
  amountread += strlen(line) + 1;

  if(amountread < file_size)
    /*Seek to the start of the next line, as we may have read to far - but we only do this if not EOF*/
    //fseek(file, amountread, SEEK_SET);
    lseek(fd, amountread, SEEK_SET);
    
  return line;
}

static int isCommentLine(char *line) {
  /*Support function to the external function readline(), this return 1 if the line is a comment line
   *0 otherwise
   */

  if(comments == NULL)
    return 0;

  return strchr(comments, *line) != NULL;
}

char *readline() {
  /*The external readline function, which will return the next line the file which is NOT any of the following:
   *  - A comment line or;
   *  - An empty line
   */

  char *line = readline_();

  while(line != NULL && (strlen(line) <= 0 || isCommentLine(line))) {
    free(line);
    line = readline_();
  }

  return line;
}

void postion(off_t pos) {

  lseek(fd, pos, SEEK_SET);
}

int writeline(char *str) {

  int len = strlen(str);
  char *temp;

  if(str[len - 1] != '\n') {
    temp = (char *)malloc(len + 1);
    if(temp == NULL) {
#ifdef DEBUG
      perror("Error allocating memory");
#endif
      return 0;
    }

    strcpy(temp, str);
    strcat(temp, "\n");
  }
  else
    temp = str;

  if(write(fd, temp, strlen(temp)) == -1) {
#ifdef DEBUG
    perror("Error writing to the file");
#endif
      return 0;
  }

  return 1;
}

#ifdef IO_TESTING
int main(int argc, char **argv) {

  if(!setFileIOpath(argv[1], "#")) {
    fprintf(stderr, "Error setting up file IO\n");
    exit(1);
  }
  
  char *line = NULL;

  
  while((line = readline()) != NULL) {
    printf("%s\n", line);
    free(line);
  }

  printf("END OF FILE\n");

  resetFileIO();
}

#endif
