%{
/*
 *  ipfm.y - functions parsing the configuration file
 *  
 *  Robert Cheramy <tibob@via.ecp.fr>, 1999
 *
 *  Thanks to Etienne BERNARD <eb@via.ecp.fr> for his
 *  * little manual for Lex and Yacc
 *    http://www.via.ecp.fr/~eb/textes/minimanlexyacc-english.html
 *  * sample code in ippl (http://www.via.ecp.fr/~hugo/ippl)
 *  which helped me to understand Lex and Yacc
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <db.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "filter.h"
#include "init.h"
#include "data.h"

void yyerror(char *);
void parseerror(char *, int);
int yylex(void);

extern int line;

extern char *filemask;
extern char *device;
extern struct OptionsType Options;
extern struct AllLogsType * pAllLogs;
extern int yydebug;

#ifdef YYDEBUG
 yydebug = 1;
#endif
%}


%union {
  long            longval;
  char            *stringval;
  struct s_host   hostval;
  struct s_filter *filterval;
  int (*SortFunc)(const void *, const void *);
}


%token SLASH
%token LOG
%token NEWLOG
%token EOL
%token TIME
%token DEVICE
%token NOT
%token FROM
%token TO
%token NONE
%token BOTH
%token WITH
%token SORT


%token<longval>   NUMBER
%token<longval>   DATETYPE
%token<stringval> IP
%token<stringval> FILENAME
%token<stringval> STRING
%token<SortFunc>  SORTFUNC
%token<longval>   DNS

%type<hostval>   Domain
%type<longval>   Time
%type<longval>   Date
%type<longval>   Not
%type<longval>   Direction
%type<longval>   With
%type<stringval> Device
%type<filterval> Rule


%start Input
%%

Input:
          /* Empty */
        | Input Line
        ;

Line:   EOL
      | Rule EOL {
        $1->next = pAllLogs->Filter; pAllLogs->Filter = $1;
      }

      | Time EOL {
        pAllLogs->TimeInterval = $1;
	pAllLogs->NextTime = time(NULL) + pAllLogs->TimeInterval;
      }

      | FILENAME {
	free(pAllLogs->LogFile);
        pAllLogs->LogFile = $1;
      }

      | Device EOL {
        device = $1;
      }

      | SORT SORTFUNC {
	pAllLogs->Sort = 1;
	pAllLogs->SortFunc = $2;	
      }

      | DNS {
	pAllLogs->ReverseLookup = $1;
      }

      | NEWLOG EOL {
	struct AllLogsType * pNewLog;;
	pNewLog = (struct AllLogsType *) malloc (sizeof(struct AllLogsType));
	pNewLog->Next = pAllLogs;
	pNewLog->Filter = NULL;
	pNewLog->Data = NULL;
	pNewLog->TimeInterval = DEFAULT_OPTIONS_TIMEINTERVAL;
	pNewLog->NextTime = time(NULL) + pNewLog->TimeInterval;
	pNewLog->LogFile = strdup(DEFAULT_OPTIONS_LOGFILE);
	pNewLog->Sort = DEFAULT_OPTIONS_SORT;
	pNewLog->SortFunc = NULL;
	pNewLog->ReverseLookup = DEFAULT_OPTIONS_REVERSELOOKUP;

	pAllLogs = pNewLog;
      }

      | error EOL {
        parseerror("Skipping invalid line", line-1);
      }
      ;

Rule :   LOG Direction Domain With Domain {
         $$ = (struct s_filter *)malloc(sizeof(struct s_filter));
	 $$->tlog  = $2;
	 $$->thost = $3;
	 $$->olog  = $4;
	 $$->ohost = $5;
       }
       ;

Direction :   NONE {
              $$ = 0;
            }
            | FROM {
              $$ = 1;
	    }
            | TO {
	      $$ = 2;
	    }
            | BOTH {
	      $$ = 3;
	    }
            | /* Nothing */ {
	      $$ = 3;
	    }
            ;

With:   Not WITH {
	 $$ = $1;
       }
       | /* Notiong */ {
         $$ = 1;
       }
       ;

Not:   NOT {
       /* NO LOG */
       $$ = 0;
     }
     | /* Notiong -> LOG */ {
       $$ = 1;
     }
     ;

Domain:	IP SLASH IP {
	  struct in_addr inp;
	  inet_aton($1, &inp);
	  $$.ip = inp.s_addr;
	  inet_aton($3, &inp);
	  $$.mask = inp.s_addr;
	  free($1); free($3);
	}

        | IP {
          struct in_addr inp;
          inet_aton($1, &inp);
          $$.ip = inp.s_addr;
          $$.mask = 0xffffffff;
	  free($1);
        }

	| /* Nothing */ {
	  $$.ip   = 0;
	  $$.mask = 0;

	}
        ;

Time:   TIME Date {
	  $$ = $2;
      }
      ;

Date:   Date NUMBER DATETYPE {
          $$ = $1 + $2 * $3;
      }
      | NUMBER DATETYPE {
          $$ = $1 * $2;
      }
      ;

Device: DEVICE STRING {
        $$ = $2;
      }
      ;

%%

void yyerror(char *s) {
  printf("yacc error : %s\n",s);
}

void parseerror(char *s, int line) {
  printf("Parse error line %d: %s\n", line, s);
}
