/* scfio.c
 * Single char. action structured field functions using ncurses. 
 * Jim Jackson  Apr 97
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

/*
 * These functions are based on the ncfio.c functions, that action
 *  all key input to build a field until certain terminators are pressed.
 * These functions are given a key character to action on the field.
 * Either the key does something to the field or it doesn't. Either way
 * the value of the field always shows the current value.
 * 
 * These functions are useful in screen based programs that may need to 
 * schedule several things, one of which is single key input handling.
 * 
 * actfint(k,att,ip,w) action key on integer on screen -  w max digits
 * putfint(att,n,w)  display integer n on screen -  w max digits
 * actfstr(k,att,s,ip,w)  action key on a string on screen - w max characters
 * putfstr(att,s,w)  display a string on screen - w max characters
 * actfreal(k,att,rp,w,dp) action key on a real on screen - w max chars, dp dec places
 * putfreal(att,r,w,dp) display a real on screen - w max chars, dp dec places
 * actfopt(k,att,sp,ip,w) action key on one of options char *sp[] set *ip to selected
 * putfopt(att,sp,i,w) display char *sp[i] in field of width w
 * 
 * actfield(k,SCFp)    action key on field defined by SCField *SCFp
 * putfield(SCFp)    display field defined by SCField *SCFp
 * putrevfield(SCFp)    display field defined by SCField *SCFp in reversed att
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <signal.h>
#include <ctype.h>
#include <time.h>
#include "scfio.h"        /* local header file */

/* reverse_attr(a)  calculate the reverse attribute of attribute a
 *    if its not a color pair then simply toggle A_REVERSE on a
 *    if it is a color pair then make it COLOR_PAIRS-pair_number
 */

reverse_attr(a)
int a;
{
   int r;

   if (PAIR_NUMBER(a)) r=(a&(~A_COLOR))+(COLOR_PAIR(COLOR_PAIRS-PAIR_NUMBER(a)));
   else r=((a+A_REVERSE)&A_REVERSE)+(a&~A_REVERSE);
   return(r);
}

/* actfint(k,att,ip,w)   action key k on input field of width w, attribute att 
 *           digits build up integer value,
 *           DEL or backspace erases least sig digit,
 *           KEY_UP or + / KEY_DOWN or -   inc's/dec's the integer,
 *           ip points to where to store the integer.
 */

actfint(k,att,ip,w)
int k,att,*ip,w;
{
   int c,y,x;
   
   getyx(stdscr,y,x);
   if (w>=COLS) w=COLS-1;
   
   if (k=='\b' || k==KEY_BACKSPACE || k==127 || k==KEY_DC) {
      *ip/=10;
   } else if (isdigit(k)) {
      *ip=(*ip)*10+k-'0';
   } else if (k==KEY_UP || k=='+') {
      (*ip)++;
   } else if ((k==KEY_DOWN || k=='-') && *ip) {
      (*ip)--;
   } else {
      return(k);
   }
   mvputfint(y,x,att,*ip,w);
   move(y,x+w-1);
   refresh();
   return(-1);
}


/* putfint(att,n,w)   display integer n in a field
 *           of width w using attribute att.
 */

putfint(att,n,w)
int att,n,w;
{
   int c,i,j;
   chtype f[COLS];
   
   if (w>=COLS) w=COLS-1;
   for (c='0', i=w, j=n; i; i--) {
      f[i]=j%10+att+c; j/=10;
      if (!j) c=' ';
   }
   addchnstr(f+1,w);
}

/* actfstr(k,att,s,ip,w)  Action key k on an input field of width w using 
 *           attribute att.
 *           Input chars to string s at position s[*ip]
 *           DELETE deletes char over cursor,
 *           <- or ^H deletes char to left of cursor,
 *           Left and right arrows move cursor,
 *           HOME and END move cursor to start, end of string,
 */

actfstr(k,att,s,ip,w)
int k,att;
char *s;
int *ip,w;
{
   int i,l,y,x;
   char *q,*p;
   
   getyx(stdscr,y,x);
   
   if (w>=COLS-x) w=COLS-x;

   for (i=0; i<w && s[i]; i++) { }   /* make sure string is w chars wide */
   for ( ; i<w; i++) { s[i]=' '; }
   s[i]=0;
   if (*ip>=w) *ip=w-1;
   if (*ip<0) *ip=0;
   
   if (k=='\b' || k==KEY_BACKSPACE || k==127) {
      if (*ip) {
	 for ( q=s+(--(*ip)) ; *q=*(q+1); q++) {  }
	 *q=' ';
      }
   } else if (isprint(k)) {
      for ( q=s+*ip ; *q; q++) {
	 l=*q; *q=k; k=l;
      }
      if (*ip<(w-1)) (*ip)++;
   } else if (k==KEY_DC) {
      for ( q=s+*ip ; *q=*(q+1); q++) {  }
      *q=' ';
   } else if (k==KEY_LEFT) {
      if (*ip) (*ip)--; 
   } else if (k==KEY_RIGHT) {
      if (*ip<(w-1)) (*ip)++;
   } else if (k==KEY_HOME) {
      *ip=0;
   } else if (k==KEY_END) {
      *ip=w-1;
   }
   s[w]=0;
   mvputfstr(y,x,att,s,w);
   move(y,x+*ip);
   refresh();

   return(k);
}

/* putfstr(att,s,w)   display string s in a field
 *           of width w using attribute att.
 */

putfstr(att,s,w)
int att;
char *s;
int w;
{
   int c,i;
   chtype f[COLS],*p;
   
   if (w>=COLS) w=COLS-1;
   for ( p=f, i=0 ; i<w ; i++) {
      if (*s) c=*s++; else c=' ';
      *p++=att+c;
   }
   *p=0;
   addchnstr(f,w);
}

/* actfopt(k,att,sp,ip,w)  action key k in an option field char *sp[]
 *           display with attribute att, in field of width w
 *           UP and DOWN arrows move back and forward an option.
 *           SPACE moves forward thru' options
 *           HOME and END move to first and last option
 */

actfopt(k,att,sp,ip,w)
int k,att;
char **sp;
int *ip,w;
{
   int i,l,y,x,ratt;
   char f[COLS+1],*q,*p;
   
   getyx(stdscr,y,x);
   if (w>=COLS-x) w=COLS-x;

   if (k=='\b' || k==KEY_BACKSPACE || k==127 || k==KEY_UP) {
      k=KEY_UP;
      if (*ip) (*ip)--;
      else k=KEY_END;
   }  /* no else here on purpose */
   if (k==KEY_DOWN || k==' ') {
      (*ip)++;
      if (sp[*ip]==NULL) *ip=0;
   } else if (k==KEY_HOME) {
      *ip=0;
   } else if (k==KEY_END) {
      for ( ; sp[*ip+1]!=NULL; (*ip)++) { }
   } else if (k!=KEY_UP) {
      return(k);
   }
   putfstr(att,sp[*ip],w);
   refresh();
   return(-1);
}

/* putfield(fp)   display field *fp 
 */

putfield(fp)
struct SCField *fp;
{
   int t;
   
   t=(fp->type)&SCF_type;
   if (t==SCF_integer) {
      mvputfint(fp->row,fp->col,fp->attr,*(fp->val.i),fp->W);
      move(fp->row,fp->col+fp->W-1);
   } else if (t==SCF_string) {
      mvputfstr(fp->row,fp->col,fp->attr,fp->val.s,fp->W);
      move(fp->row,fp->col+fp->adj);
   } else if (t==SCF_option) 
      mvputfstr(fp->row,fp->col,fp->attr,(fp->val.sp)[fp->adj],fp->W);
   else return(1);
   return(0);
}


/* actfield(k,fp)   action key k on field *fp 
 */

actfield(k,fp)
int k;
struct SCField *fp;
{
   int t,c;
   
   if ((fp->type)&SCF_fixed) return(-1);
   t=(fp->type)&SCF_type;
   if (t==SCF_integer) c=mvactfint(fp->row,fp->col,k,fp->attr,fp->val.i,fp->W);
   else if (t==SCF_string) 
     c=mvactfstr(fp->row,fp->col,k,fp->attr,fp->val.s,&(fp->adj),fp->W);
   else if (t==SCF_option) 
     c=mvactfopt(fp->row,fp->col,k,fp->attr,fp->val.sp,&(fp->adj),fp->W);
   else return(-1);
   return(c);
}

 
