/* siggen.c
 * Ncurses based Signal generator, 16/8 bit, 2 channels if stereo card,
 * Emulates usual functions of a lab sig. gen. generating sine, square,
 *  triangle, sawtooth, pulse, noise waveforms.
 *
 * Linux Version
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

/*
 * siggen :
 * 
 * A one second's worth buffer of samples is maintained (samplerate samples)
 * for each channel. These are mixed into a 1 second play buffer
 * that in play mode is played in a circular fashion. By keeping freq.
 * as an integer number of Hertz there is always an exact number of full
 * cycles in 1 sec, so the 1 sec play buffer start and finish match
 * seamlessly. If only mono card then the two channels are mixed into
 * one playing channel.
 * 
 * ToDo:
 *        Mono Mixing into one buffer needs to take account of gains.
 * 
 * History:
 *  20May97 V1.4   Tidied up generation, and added phase offset for
 *                 second channel with respect to first channel.
 *                 Added 'R' key to resync the two channels, in case
 *                 phase was off because of playing different frequencies
 *  13May97 V1.3   Changed IO library in order to action single key
 *                 presses on fields, so that we can generate sound
 *                 all the time and key actions take 'immediate' effect.
 *                 i.e. continuous playing. Generation changed to sampling
 *                 every F'th sample from a 1Hz set of samples to create
 *                 samples for a frequency of F Hz.
 *  15Mar97 V1.2   Pulse not working - fixed by changing default ratio and
 *                 ratio2 from 0 to -1, forces generator() to pick up def. value
 *  --Feb97 V1.1   Various screen changes - help line, some tweeking
 *                 of ncfio key handling etc.
 *  --Jan97 V1.0   2 channels - if stereo support, then each chan.
 *                 is fed to a seperate stereo output. If only mono
 *                 is possible then the two channels are mixed into the
 *                 one output.
 *                 While entering details of function required
 *                 the program ceases playing any previously generated
 *                 samples.
 */

#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <math.h>
#include "siggen.h"

#define MAXPRM 32
#define chkword(a,b) ((n>=a)&&(strncmp(p,b,n)==0))

int vflg,dflg;

int DAC;                         /* File Handle for DSP */
unsigned int samplerate;         /* Samples/sec        */
unsigned int stereo;             /* stereo mono */
unsigned int afmt;               /* format for DSP  */
int Nfragbufs;                   /* number of driver frag buffers */
int fragsize;                    /* size of driver buffer fragments */
int fragsamplesize;              /* size of fragments in samples */

    /* channel 1  ... */
char wf[32]="sine";              /* waveform type */
unsigned int freq=440;           /* signal frequency */
int ratio=-1;                    /* used in pulse, sweep etc */
int Gain=100;                    /* Amplification factor */

    /* channel 2  ... */
char wf2[32]="off";              /* waveform type */
unsigned int freq2=440;          /* signal frequency */
int ratio2=-1;                   /* used in pulse, sweep etc */
int Gain2=100;                   /* Amplification factor */
int phase=0;                     /* phase difference of chan2 from chan1 */

char *sys;

help(e)
int e;
{  fputs(VERSION,stderr);
   fputs("\nUsage: \n 1: siggen [flags] [waveform [freq [param]]]\n",stderr);
   fputs("      waveform is sine, square, triangle, sawtooth, noise, pulse\n",stderr);
   fputs("      for pulse param is Mark/Space ratio as a %.\n\n",stderr);
   fputs("Defaults: output to /dev/dsp, 22050 samples/sec, stereo, 16 bit\n",stderr);
   fputs("          samples if possible, else 8 bit and/or mono. \n\n",stderr);
   fputs("flags: -s samples    generate with samplerate of samples/sec\n",stderr);
   fputs(" -8/-16 or -b 8|16   force 8 bit or 16 bit mode.\n",stderr);
   fputs("       -1,-2         force mono or stereo (1 or 2 channels)\n",stderr);
   fputs("      -NB n          Numer of Buffers to create is n, def. is 3\n",stderr);
   return(e);
}

/* main
 *
 */
 
main(argc,argv)
int argc;
char **argv;
{
   unsigned int v[MAXPRM],maxv,i,j,k,l,m,n,N;
   FILE *f;
   char *p,bf[130];
   int c;
   unsigned int t;
   
   if ((p=strrchr(sys=*argv++,'/'))!=NULL) sys=++p;
   argc--;
   
   samplerate=SAMPLERATE; afmt=AFMT_QUERY;
   Nfragbufs=DEFAULT_BUFFERS;
   stereo=-1;
   vflg=dflg=0;
   
   while (argc && **argv=='-') {          /* all flags and options must come */
      n=strlen(p=(*argv++)+1); argc--;    /* before paramters                */
      if (chkword(1,"samplerate")) {
	 if (argc && isdigit(**argv)) { samplerate=atoi(*argv++); argc--; }
      }
      else if (chkword(2,"NB")) {
	 if (argc && isdigit(**argv)) { Nfragbufs=atoi(*argv++); argc--; }
      }
      else if (chkword(2,"16")) { afmt=AFMT_S16_LE; }
      else if (chkword(1,"bits")) {
	 i=0;
	 if (argc) { 
	    i=atoi(*argv++); argc--;
	 }
	 if (i==8) afmt=AFMT_U8;
	 else if (i==16) afmt=AFMT_S16_LE;
	 else exit(err_rpt(EINVAL,"must be '-b 8' or '-b 16'."));
      }
      else if (chkword(1,"A")) {
	 if (argc && isdigit(**argv)) { 
	    Gain=atoi(*argv++); argc--;
	 }
      }
      else {                              /* check for single char. flags    */
	 for (; *p; p++) {
	    if (*p=='h') exit(help(EFAIL));
	    else if (*p=='1') stereo=0;
	    else if (*p=='2') stereo=1;
	    else if (*p=='8') afmt=AFMT_U8;
	    else if (*p=='d') dflg=1;
	    else if (*p=='v') vflg++;
	    else {
	       *bf='-'; *(bf+1)=*p; *(bf+2)=0;
	       exit(help(err_rpt(EINVAL,bf)));
	    }
	 }
      }
   }
   
   if (argc) {
      strncpy(wf,*argv++,32); wf[31]=0; argc--;    /* waveform type */
      if (argc) {
	 freq2=freq=atoi(*argv++); argc--;
	 if (argc) { ratio=atoi(*argv++); argc--; }
	 if (argc) exit(help(err_rpt(EINVAL,"Too many parameters")));
      }
   }
   
   /* if no format specified then try 16 bit */
   i=afmt;
   n=stereo;
   if ((DAC=DACopen(DAC_FILE,"w",&samplerate,&i,&n))<0) {
      exit(err_rpt(errno,"Opening DSP for output."));
   }
   if (((afmt!=AFMT_QUERY) && (i!=afmt)) ||
       ((stereo!=-1) && (n!=stereo))) {
      exit(err_rpt(EINVAL,"Sound card doesn't support format requested."));
   }
   afmt=i; stereo=n;

   if ((fragsize=setfragsize(DAC,Nfragbufs,samplerate,afmt,stereo))<0) {
      exit(err_rpt(errno,"Problem setting appropriate fragment size."));
   }
   fragsamplesize=(fragsize>>(afmt==AFMT_S16_LE))>>(stereo);
       
   if (freq > samplerate/2) {
      fprintf(stderr,"%d Hz is more than half the sampling rate\n",freq);
      exit(err_rpt(EINVAL,"Frequency setting too great"));
   }

   if (vflg) {
      printf("%s %s bit samples being generated.\n",
	     (stereo)?"Stereo, ":"Mono, ",(afmt==AFMT_S16_LE)?"16":"8");
      printf("Samples scaled by a factor of %d/100\n",Gain);
      printf("Playing at %d samples/sec\n",samplerate);
      printf("Buffer fragment size is %d bytes (%d samples). Aprox. %d millisecs.\n",
	     fragsize, fragsamplesize,
	     1000*((fragsize>>(stereo))>>(afmt==AFMT_S16_LE))/samplerate);
      printf("\n<Press Return to Continue>\n");
      getchar();
   }

   WinGen(DAC);
   
   close(DAC);
   exit(0);
}  
