/* sigscr.c
 * Screen handling for ncurses SigGEN program
 * Jim Jackson    Dec 96
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include "scfio.h"
#include "siggen.h"
#include <sys/soundcard.h>

/* set number of millisecs to wait while checking for ESC sequences */

#define KEY_DELAY 25

/* Main screen character strings............
 */

char *hl[]={
   "     /====\\            /====\\  ======  |\\    |  ",
   "     |                 |       |       | \\   |    ",
   "     \\====\\  o  /--\\   |   __  |===    |  \\  | ",
   "          |  |  |  |   |    |  |       |   \\ |    ",
   "     \\====/  |  \\--|   \\====/  ======  |    \\| ",
   "                   |  ",
   "       Digital  \\--/  Signal Generator ", 
   " -----------------------------------------------------------------------------",
   NULL };

char *oh[]={
   "      /\\         ",
   "   --/--\\--/--   ", 
   "         \\/      ",
   NULL };

char *sig=" Jim Jackson <jj@scs.leeds.ac.uk> ";

int Lsamplerate,             /* Local copy of card params to prevent */
    LNfb,                   
    Lafmt,Lstereo;           /* playing being screwed while updating */
                             /* card characteristics.                */

char Mchan1[]="  Channel 1 ",
     Mchan2[]="  Channel 2 ",
     Mstarting[]=" Starting up .......... ",
     Mwavef[]="  Waveform: ",
     Mfreq[]= " Frequency: ",
     Mgain[]= "Hz  Gain: ",
     Mphase[]="Phase:        deg.",
     Mratio[]=" M/S Ratio: ",
     Moffs[]= "Phaseshift: ",
     Msampl[]=" Samplerate       /sec                    x       byte buffs",
     Minfo[]=" : ",
     Mtime[16]="",
     Mstereo[]="STEREO",
     Mmono[]=  " MONO ",
     M16b[]=  "16 bit",
     M8b[]=   " 8 bit",
     Mmain[]=
     "<- / -> move fields,      'R'esync channels, 'Q'uit, 'S'etup samplerate etc",
     Mcard[]=
     "'S' to action new settings               Left/Right Arrow Keys move fields.",
     Moff[]="OFF", Msin[]="SINE", Mcos[]="COSINE", Msqu[]="SQUARE",
     Mtri[]="TRIANGLE", Msaw[]="SAWTOOTH", Mpulse[]="PULSE", 
     Mnoise[]="NOISE";

char *WAVEFORMS[]={ Moff, Msin, Mcos, Msqu, Mpulse, Msaw, Mtri, Mnoise, NULL };
char *Mbits[]={ M8b, M16b, NULL };
char *Mchans[]={ Mmono, Mstereo, NULL };
   
struct SCField
 FsamP= { SCF_string, 9, 0, 0, Msampl, 80, 0 },
 FsamV= { SCF_integer, 9, 12, 0, &Lsamplerate, 6, 0 },
 Fbits= { SCF_option, 9, 24, 0, Mbits, 6, 0 },
 Fchans= { SCF_option, 9, 32, 0, Mchans, 6, 0 },    
 FNfrag= { SCF_integer, 9, 40, 0, &LNfb, 2, 0 },
 Ffrag= { SCF_integer, 9, 43, 0, &fragsize, 6, 0 },
 FinfoP= { SCF_string, 21, 0, 0, Minfo, 80, 0 },
 FinfoP2= { SCF_string, 22, 0, 0, Minfo, 80, 0 },
 Finfo= { SCF_string, 21, 3, 0, Mstarting, 75, 0 },
 Finfo2= { SCF_string, 22, 3, 0, "", 75, 0 },
 Ftime= { SCF_string, 9, 69, 0, Mtime, 10, 0 },

 Fchan1= { SCF_string,  11, 0, 0, Mchan1, 12, 0 },
 Fwav1P= { SCF_string,  12, 2, 0, Mwavef, 12, 0 },
 Fwav1V= { SCF_option,  12, 14, 0, WAVEFORMS, 8, 0 },
 Ffrq1P= { SCF_string,  12, 22, 0, Mfreq, 12, 0 },
 Ffrq1V= { SCF_integer, 12, 34, 0, &freq, 6, 0 },
 Fgn1P=  { SCF_string,  12, 40, 0, Mgain, 10, 0 },
 Fgn1V=  { SCF_integer, 12, 50, 0, &Gain, 5, 0 },
    
 Fchan2= { SCF_string,  15, 0, 0, Mchan2, 12, 0 },
 Fwav2P= { SCF_string,  16, 2, 0, Mwavef, 12, 0 },
 Fwav2V= { SCF_option,  16, 14, 0, WAVEFORMS, 8, 0 },
 Ffrq2P= { SCF_string,  16, 22, 0, Mfreq, 12, 0 },
 Ffrq2V= { SCF_integer, 16, 34, 0, &freq2, 6, 0 },
 Fgn2P=  { SCF_string,  16, 40, 0, Mgain, 10, 0 },
 Fgn2V=  { SCF_integer, 16, 50, 0, &Gain2, 5, 0 },
 FphP=   { SCF_string,  16, 58, 0, Mphase, 18, 0 },
 FphV=   { SCF_integer, 16, 65, 0, &phase, 6, 0 };

struct SCField 
 *CHhdrs[]= { &Fchan1, &Fwav1P, &Ffrq1P, &Fgn1P,
              &Fchan2, &Fwav2P, &Ffrq2P, &Fgn2P, &FphP, NULL },
 *CHvals[]= { &Fwav1V, &Ffrq1V, &Fgn1V,
              &Fwav2V, &Ffrq2V, &Fgn2V, &FphV, NULL },
 *scrP[]= { &FsamP, &FinfoP, &FinfoP2, &Finfo, &Finfo2, 
            &Ffrag, &Ftime, NULL, },
 *scrV[]= { &FsamV, &Fbits, &Fchans, &FNfrag, NULL };
    
/* Help strings for second info line.
 */

char Hwav[]="DOWN (or SPACE) & UP ARROW to select WaveForm",
     Hnum[]="Enter Digits, DEL or <- to delete, UP/DOWN arrow for Inc/Decrement",
     Hbits[]="Press <SPACE> to toggle 8 / 16 bit setting",
     Hchan[]="Press <SPACE> to toggle Mono / Stereo setting";
    

char *Hvals[]={ Hwav, Hnum, Hnum, 
                Hwav, Hnum, Hnum, Hnum, NULL };
char *Hset[]= { Hnum, Hbits, Hchan, Hnum, NULL };

/* waveform buffer array.......
 * same number or more entries as WAVEFORMS[] above....
 */

int *wavbufs[]={ NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
                  NULL,  NULL,  NULL,  NULL };   

/* WinGen()   The main interactive screen
 */

WinGen(fd)
int fd;
{
   int c,i,n,st,t;
   int f,wavef,card;
   char **hh;
   struct SCField **ff;
   
   WinGenInit();
   
   hh=Hvals; ff=CHvals;
   doinfo2(Mmain,0);
   wavef=card=f=0;
   dispfield(ff,hh,f);
   refresh();
   t=time(NULL);
 
   for (;;) {
      if ((i=time(NULL))-t) { dotime(t=i); }
      if (playsamples(fd)<0) {
	 move(COLS-1,0); endwin();
	 return(err_rpt(errno,"Problem generating output samples."));
      }
      if ((c=getch())==-1) {
         delay(25000);
	 continue;
      }
      if (c=='s' || c=='S') {  
	 if (card) {
	    /* check if card params have changed - if so close and reopen */
            Lafmt=(Fbits.adj)?AFMT_S16_LE:AFMT_U8;
	    Lstereo=Fchans.adj; 
            if ((Lsamplerate!=samplerate) || (Lafmt!=afmt) || (Lstereo!=stereo) || (LNfb!=Nfragbufs)) {
	       close(fd);
	       if ((fd=DACopen(DAC_FILE,"w",&Lsamplerate,&Lafmt,&Lstereo))<0 ||
		   (fragsize=setfragsize(fd,LNfb,Lsamplerate,Lafmt,Lstereo))<0) {
		  move(COLS-1,0); endwin();
		  return(err_rpt(errno,"Opening DSP for output."));
	       }
	       fragsamplesize=(fragsize>>(Lafmt==AFMT_S16_LE))>>(Lstereo);
	       putfield(&Ffrag);
	    }
	    dofields(scrV);
	    /* if samplerate or format has changed, re-generate waveforms */
	    if ((Lsamplerate!=samplerate) || (Lafmt!=afmt)) {
	       if (genwavbufs(WAVEFORMS,wavbufs,Lsamplerate,Lafmt)) {
	          move(COLS-1,0); endwin();
		  return(err_rpt(EINVAL,"Problem generating Base waveforms."));
	       }
	    }
	    samplerate=Lsamplerate; afmt=Lafmt; stereo=Lstereo; Nfragbufs=LNfb;
            playsamples(-1);    /* resync left and right channels */
	    dispfield(ff,hh,f);
	    card=0; f=wavef;
	    ff=CHvals; hh=Hvals;
	    dispfield(ff,hh,f);
	    doinfo2(Mmain,0);
	 } else {   /* enter card setting alteration stuff */
	    dispfield(ff,hh,f);
	    card=1; wavef=f; f=0;
	    ff=scrV; hh=Hset;
	    dispfield(ff,hh,f);
	    doinfo2(Mcard,0);
	 }
      }      

      if (c==KEY_END || c=='q' || c=='Q') {
	 move(COLS-1,0); endwin();
	 return(c);
      }
      if (c=='r' || c=='R') {
         playsamples(-1);   /* synch left and right channels */
         continue;
      }
      f=doinput(c,ff,hh,f);
   }
}

/*
 *  doinput(c,aa,H,f)  action char c on field aa[f]
 *      if field changes display old field in normal and  new
 *      field in reverse, and display relevant new help line from H[]
 *  return new field value.
 */

doinput(c,aa,H,f)
struct SCField *aa[];
char *H[];
int f;
{
   int fn;
   
   c=actfield(c,aa[f]);
   fn=-1;
   if (c==KEY_LEFT) {
      if (f) fn=f-1;
      else {
	 for (fn=f ; aa[fn+1]!=NULL; fn++) { }
      }
   } else if (c==KEY_RIGHT || c=='\r' || c=='\n' || c=='\t') {
	 fn=f+1; if (aa[fn]==NULL) fn=0;
   } else if (c==KEY_HOME) { fn=0; }
   if (fn!=-1) {
      dispfield(aa,H,f);
      f=fn;
      dispfield(aa,H,f);
   }
   refresh();
   return(f);
}

dispfield(ff,hh,f)
struct SCField *ff[];
char *hh[];
int f;
{
   doinfo(hh[f],0);
   ff[f]->attr=reverse_attr(ff[f]->attr);
   putfield(ff[f]);
}

WinGenInit()
{
   int att_h1,att_h2,att_h3,att_h4,att_h5,att_h6;
   int n,i;
   
   Lsamplerate=samplerate;         /* set local working copy of samplerate */
   Lafmt=afmt; Lstereo=stereo;
   LNfb=Nfragbufs;

   Fbits.adj=(afmt==16);
   Fchans.adj=stereo;
   
   if (genwavbufs(WAVEFORMS,wavbufs,samplerate,afmt)) {
      return(err_rpt(EINVAL,"Problem generating Base waveforms."));
   }
   n=strlen(wf);
   for (Fwav1V.adj=i=0; WAVEFORMS[i]!=NULL; i++) {
      if (strncasecmp(wf,WAVEFORMS[i],n)==0) {
	 Fwav1V.adj=i; break;
      }
   }

   tzset();              /* set timezone stuff up for dotime() function */
   
   initscr();
   cbreak(); noecho();
   nonl(); intrflush(stdscr,FALSE); keypad(stdscr,TRUE);
   nodelay(stdscr,TRUE);
   timeout(KEY_DELAY);
   
   if (has_colors()) {
      start_color();
      init_pair(1,COLOR_BLUE,COLOR_GREEN);
      init_pair(COLOR_PAIRS-1,COLOR_GREEN,COLOR_BLUE);
      init_pair(2,COLOR_CYAN,COLOR_RED);
      init_pair(COLOR_PAIRS-2,COLOR_RED,COLOR_CYAN);
      init_pair(3,COLOR_GREEN,COLOR_RED);
      init_pair(COLOR_PAIRS-3,COLOR_RED,COLOR_GREEN);
      init_pair(4,COLOR_YELLOW,COLOR_RED);
      init_pair(COLOR_PAIRS-4,COLOR_RED,COLOR_YELLOW);
      init_pair(5,COLOR_WHITE,COLOR_BLUE);
      init_pair(COLOR_PAIRS-5,COLOR_BLUE,COLOR_WHITE);
      init_pair(6,COLOR_RED,COLOR_BLUE);
      init_pair(COLOR_PAIRS-6,COLOR_BLUE,COLOR_RED);
      att_h1=COLOR_PAIR(1);
      att_h2=COLOR_PAIR(2)+A_BOLD;
      att_h3=COLOR_PAIR(3);
/*      att_h4=COLOR_PAIR(4)+A_BOLD; */
      att_h4=COLOR_PAIR(2);
      att_h5=COLOR_PAIR(5);
      att_h6=COLOR_PAIR(6);
   } else {
      att_h1=A_REVERSE; att_h5=att_h6=att_h2=att_h4=A_BOLD;
      att_h3=A_NORMAL;
   }
   
   dofatt(scrV,att_h4); dofatt(scrP,att_h3);
   Ffrag.attr=att_h4;
   dofatt(CHhdrs,att_h5);
   dofatt(CHvals,att_h6);
   
   dohdr(0,att_h1,att_h2,att_h3);
   n=8;
   for (i=0; i<n; i++) {
      mvputfstr(i+Fchan1.row-1,0,att_h5,"",80);
   }

   dofields(scrP); dofields(scrV);
   dofields(CHhdrs);
   dofields(CHvals);
   refresh();
}

/* dohdr(y,a1,a2,a3)  write header out starting at line y using attributes
 *   a1, s2, and a3
 */

dohdr(y,a1,a2,a3)
int y,a1,a2,a3;
{
   int i;

   for ( i=0; hl[i]!=NULL; i++) {
      mvputfstr(y+i,0,a1,hl[i],80);
   }
   for ( i=0; oh[i]!=NULL; i++) {
      mvputfstr(y+i+1,53,a2,oh[i],17);
   }
   mvputfstr(y+6,44,a3,sig,strlen(sig));
   mvputfstr(24,0,a1,VERSION,80);
   dotime(time(NULL));
}

/* Display string s in the Info field of the screen, and pause for w
 * seconds, if w is non-zero
 */

doinfo(s,w)
char *s;
int w;
{
   Finfo.val.s=s;
   putfield(&Finfo);
   refresh();
   if (w) sleep(w);
}

/* Display string s in the 2nd Info field of the screen, and pause for w
 * seconds, if w is non-zero
 */

doinfo2(s,w)
char *s;
int w;
{
   Finfo2.val.s=s;
   putfield(&Finfo2);
   refresh();
   if (w) sleep(w);
}

dotime(t)
int t;
{
   int y,x;

   t+=(daylight*3600);
   sprintf(Mtime,"%2d:%02d:%02d",(t/3600)%24,(t/60)%60,t%60);
   getyx(stdscr,y,x);
   putfield(&Ftime);
   move(y,x);
   refresh();
}

dofields(aa)
struct SCField *aa[];
{
   int i;
   
   for (i=0; aa[i]!=NULL; i++) { putfield(aa[i]); }
}

dofatt(aa,at)
struct SCField *aa[];
int at;
{
   int i;
   
   for (i=0; aa[i]!=NULL; i++) { aa[i]->attr=at; }
}

mergefields(aa1,aa2)
struct SCField *aa1[],*aa2[];
{
   int i,j;
   
   for (i=0; aa1[i]!=NULL; i++) { }
   for (j=0; aa2[j]!=NULL; i++,j++) { aa1[i]=aa2[j]; }
   aa1[i]=aa2[j];      /* copy the terminating null */
}

genwavbufs(W,bufs,samplerate,afmt)
char *W[];
char *bufs[];
int samplerate;
int afmt;
{
   int i,st,N;
   
   N=samplerate<<(afmt==16);
   for (i=0; W[i]!=NULL; i++) {
      if (bufs[i]!=NULL) free(bufs[i]);
      if ((bufs[i]=malloc(N))==NULL) return(ENOMEM);
      if ((st=generate(W[i],bufs[i],N,1,100,samplerate,-1,afmt))==0) 
	return(EINVAL);
   }
   return(0);
}

/* playsamples(fd)   check state of output sound buffers, if necessary
 *                generate next fragment of samples and write it away.
 * if fd < 0 then the offset pointers are resynched by setting to zero.
 */

playsamples(fd)
int fd;
{
   static int fr_size=0;
   static unsigned char *frag=NULL;
   static int of1,of2;

   short int v,*vp,*wv1,*wv2;
   unsigned char *p,c,*w1,*w2;
   int i,n,ph,tof2,t;
  
   ph=samplerate*(phase%360)/360;    /* phase diff of chan2 reduced */
                                     /* to 0-359 degrees then converted */
                                     /* to equiv offset into wave buffs */ 

   if (fd<0) { of2=0; return(of1=0); }     /* reset pointers */ 
   if (fragsize!=fr_size) {
      if (fr_size) free(frag);
      if ((frag=malloc(fragsize))==NULL) return(-1);
      fr_size=fragsize; of1=0; of2=0;
   }

   for (;;) {
      if ((i=getfreeobufs(fd))<=0) return(i);
      tof2=(of2+ph)%samplerate;  /* add phase to current generator offset */
      if (afmt==AFMT_S16_LE) {
	 wv1=(short int *)(wavbufs[Fwav1V.adj]); 
	 wv2=(short int *)(wavbufs[Fwav2V.adj]);
	 if (stereo) {
	    for (i=0, vp=(short int *)frag; i<fragsamplesize; i++) {
	       t=(50+(((int)(wv1[of1]))*Gain))/100;
	       *vp++=(short int)((t>32767)?32767:((t<-32767)?-32767:t));
	       t=(50+(((int)(wv2[tof2]))*Gain2))/100;
	       *vp++=(short int)((t>32767)?32767:((t<-32767)?-32767:t));
	       of1=(of1+freq)%samplerate;
	       tof2=(tof2+freq2)%samplerate;
	    }
	 } else {
	    for (i=0, vp=(short int *)frag; i<fragsamplesize; i++) {
	       t=(((int)(wv1[of1])*Gain)+
		  ((int)(wv2[tof2])*Gain2)+100)/200;
	       *vp++=(short int)((t>32767)?32767:((t<-32767)?-32767:t));
	       of1=(of1+freq)%samplerate;
	       tof2=(tof2+freq2)%samplerate;
	    }
	 }
      } else {  /* afmt is 8 bit */
	 w1=(unsigned char *)(wavbufs[Fwav1V.adj]); 
	 w2=(unsigned char *)(wavbufs[Fwav2V.adj]);
	 if (stereo) {
	    for (i=0, p=frag; i<fragsamplesize; i++) {
	       t=128+(50+((int)(w1[of1])-128)*Gain)/100;
	       *p++=(unsigned char)((t>255)?255:((t<0)?0:t));
	       t=128+(50+((int)(w2[tof2])-128)*Gain2)/100;
	       *p++=(unsigned char)((t>255)?255:((t<0)?0:t));
	       of1=(of1+freq)%samplerate;
	       tof2=(tof2+freq2)%samplerate;
	    }
	 } else {
	    for (i=0, p=frag; i<fragsamplesize; i++) {
	       t=128+(((int)(w1[of1])*Gain)+
		      ((int)(w2[tof2])*Gain2)-156)/200;
	       /*  -156 = -128-128+50+50 */
	       *p++=(unsigned char)((t>255)?255:((t<0)?0:t));
	       of1=(of1+freq)%samplerate;
	       tof2=(tof2+freq2)%samplerate;
	    }
	 }
      }   /* afmt==AFMTS16_LE */
      of2=(tof2+samplerate-ph)%samplerate;  /* remove phase from generator offset */
      if (write(fd,frag,fragsize)<0) return(-1);
   }
   return(0);
}

