/* sgen.c
 * makes a raw 8 bit digital sound signal containing a waveform of a particular
 * frequency and sampling rate. Output is to a file, or /dev/dsp (linux)
 * Jim Jackson     Linux Version
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

/* History...................
 * 
 * Based on my noddy DOS version created sample files to be played
 *   by other utils. This had lotsa faults, now fixed.
 * 
 * --Oct96  Original Linux version. Fixed faulty sample generation
 *          in DOS program - you have to generate samples over several
 *          cycles to ensure that a you can put samples end-to-end
 *          to fill a second, see comments in mksin(). Added stereo
 *          and antiphase stuff. Eventually worked out how to generate
 *          square/pulse signals accurately, and sawtooth. 
 *          Triangle still to do.
 * --Dec96  Added noise generator using the random function - sounds ok
 *          Need to figure what to do to get pink noise - I think!
 * 18Dec96  Started splitting up stuff into different files so that I can 
 *          write some different front ends. All the code to create the 
 *          samples is in generator.c, and some misc. stuff in misc.c
 * 29Dec96  Added amplitude factor -A N where N is a percentage.
 *          The sample is created to optimally fill the sample space when
 *          N is 100 (the default value). The samples generated are scaled
 *          by N/100, overly large values simply being 'clipped'.
 *          To create a trapezoid wave form, generate a triangle wave
 *          with N>100, depending on slope needed on waveform.
 * 02Jan97  Triangle generation sorted.
 * 19Jan97  Added cosine - sin with 90 degrees offset
 *          added generation by reading a generation file, 
 *            -c config_file
 *          each line of which defines a wave to generate and which
 *          channel (1 or 2) to mix this wave into. This can be used to 
 *          specify complex waveforms.
 */

/* Configuration file format:
 * 
 *  blank lines and lines with first non-space character being '#' are ignored
 *  Other lines should be of form.......
 * 
 *   [ch] waveform freq [amplitude [offset/ratio]]
 * 
 *   ch         is the optional channel number (1 or 2) to mix this wave into
 *              when operating in stereo.
 *   waveform   type of wave to generate
 *   freq       frequency in Hz
 *   amplitude  optional relative amplitude in %. Default is 100%
 *   offset/ratio  optional extra parameter required
 */

/* In mono only one buffer is used to hold one secs worth of samples
 *   because we only allow integers to specify the frequency, we always
 *   have an exact number of full cycles in one sec, therefore this buffer
 *   can be treated cyclically to generate the signal for any period.
 * 
 * In stereo, two buffers hold the signal for each channel - 2nd channel is
 *   either an in phase or antiphase copy of channel 1 at moment - and they 
 *   are mixed into a large double size playing buffer.
 */

#include <stdio.h>
#include <strings.h>
#include <errno.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <math.h>

#define VERSION "sgen  Ver. 1.2 (Dec 96)   Digital Signal Generator"

#define ABUF_SIZE 4096
#define DAC_FILE "/dev/dsp"
#define SAMPLERATE 22050
#define EFAIL -1

#define MAXPRM 32
#define chkword(a,b) ((n>=a)&&(strncmp(p,b,n)==0))

int aflg,fflg,vflg,dflg;

unsigned int samplerate,freq;    /* Samples/sec and signal frequency */
unsigned int ratio;              /* used in pulse, sweep etc */
unsigned int antiphase;          /* 2nd channel in antiphase to 1st channel */
unsigned int Gain;               /* Amplification factor */
unsigned int stereo;             /* stereo mono */
unsigned int afmt;               /* format for DSP  */

unsigned int abuf_size;          /* audio buffer size */
unsigned char *abuf;             /* pointer to audio buffer */

int lbuf_flag,rbuf_flag;       
unsigned char *lbuf,*rbuf;       /* waveform bufs, left and right channels */
unsigned int lbuf_size;          /* size of 1 sec, 1 chan, waveform buffer */
unsigned char *plbuf;            /* play buffer - mixed left and right bufs */
int plbuf_size;                  /* size of playback buffer */

char *sys,*outfile,*configfile;

help(e)
int e;
{  fputs(VERSION,stderr);
   fputs("\nUsage: \n 1: sgen [flags] waveform freq\n",stderr);
   fputs("      waveform is square, triangle, sawtooth, noise\n",stderr);
   fputs(" 2: sgen [flags] sin|cos freq [phase]\n",stderr);
   fputs("      sin/cos has extra phase param (def. is 0 degrees)\n",stderr);
   fputs(" 3: sgen [flags] pulse freq [Mark/Space]\n",stderr);
   fputs("      pulse has extra param Mark/Space % - def. is 10 (%)\n",stderr);
/*   fputs(" 4: sgen [flags] sweep freq [end_freq secs]\n",stderr); */
/*   fputs("      generates a swept sine wave from freq to end_freq in secs seconds\n\n",stderr); */
   fputs("Defaults: output to /dev/dsp, 22050 samples/sec, mono, 16 bit\n",stderr);
   fputs("          samples if possible, else 8 bit. \n\n",stderr);
   fputs("flags: -f,-a         force overwrite/append of/to file\n",stderr);
   fputs("       -o file       write digital sample to file ('-' is stdout)\n",stderr);
   fputs("       -s samples    generate with samplerate of samples/sec\n",stderr);
   fputs("       -v            be verbose.\n",stderr);
   fputs(" -8/-16 or -b 8|16   force 8 bit or 16 bit mode.\n",stderr);
   fputs("       -1,-2a        mono (def) or stereo in antiphase\n",stderr);
   fputs("       -A n          scale samples by n/100, def. n is 100\n",stderr);
   fputs("       -c file       configuration file contains details for generation\n",stderr);
   return(e);
}

/* main
 *
 */
 
main(argc,argv)
int argc;
char **argv;
{
   unsigned int v[MAXPRM],maxv,i,j,k,l,m,n,N;
   FILE *f;
   char *p,*wf,*fnm,fname[130],*omode;
   int fd;
   unsigned int t;
   
   if ((p=strrchr(sys=*argv++,'/'))!=NULL) sys=++p;
   argc--;
   
   configfile=outfile=NULL;
   samplerate=SAMPLERATE; afmt=AFMT_QUERY;
   antiphase=stereo=0; Gain=100;
   dflg=vflg=aflg=fflg=0;
   
   while (argc && **argv=='-') {          /* all flags and options must come */
      n=strlen(p=(*argv++)+1); argc--;    /* before paramters                */
      if (chkword(1,"samplerate")) {
	 if (argc && isdigit(**argv)) { samplerate=atoi(*argv++); argc--; }
      }
      else if (chkword(1,"output")) {     /* e.g. string option              */
	 if (argc) { outfile=*argv++; argc--; } /* output file name          */
      }
      else if (chkword(1,"config")) {     /* e.g. string option              */
	 if (argc) { configfile=*argv++; argc--; } /* output file name          */
      }
      else if (chkword(2,"16")) { afmt=AFMT_S16_LE; }
      else if (chkword(1,"bits")) {
	 i=0;
	 if (argc) { 
	    i=atoi(*argv++); argc--;
	 }
	 if (i==8) afmt=AFMT_U8;
	 else if (i==16) afmt=AFMT_S16_LE;
	 else exit(err_rpt(EINVAL,"must be '-b 8' or '-b 16'."));
      }
      else if (chkword(1,"A")) {
	 if (argc && isdigit(**argv)) { 
	    Gain=atoi(*argv++); argc--;
	 }
      }
      else if (chkword(2,"2a")) {
	 stereo=antiphase=1;
      }
      else {                              /* check for single char. flags    */
	 for (; *p; p++) {
	    if (*p=='h') exit(help(EFAIL));
	    else if (*p=='1') stereo=0;
	    else if (*p=='2') stereo=1;
	    else if (*p=='8') afmt=AFMT_U8;
	    else if (*p=='a') aflg=1;
	    else if (*p=='d') dflg=1;
	    else if (*p=='v') vflg=1;
	    else {
	       *fname='-'; *(fname+1)=*p; *(fname+2)=0;
	       exit(help(err_rpt(EINVAL,fname)));
	    }
	 }
      }
   }
   
   if (configfile == NULL) {
      if (argc<2) exit(help(err_rpt(EINVAL,"Not enough parameters")));
      wf=*argv++; argc--;    /* waveform type */
      freq=atoi(*argv++); argc--;
      ratio=-1;
      if (argc) { ratio=atoi(*argv++); argc--; }
   }
   if (argc) exit(help(err_rpt(EINVAL,"Too many parameters")));
   if (configfile!=NULL) {
      if ((f=fopen(configfile,"r"))==NULL) {
	 exit(err_rpt(errno,configfile));
      }
      fclose(f);
   }
      
   if (vflg) puts(VERSION);

   if (outfile==NULL) {           /* if no outfile then write direct to DAC */
                                  /* if no format specified then try 16 bit */
      i=(afmt==AFMT_QUERY)?AFMT_S16_LE:afmt ; 
      n=stereo;
      if ((fd=DACopen(fnm=DAC_FILE,"w",&samplerate,&i,&n))<0) {
	 if (afmt==AFMT_QUERY) {  /* if no format specified try for 8 bit.. */
	    i=AFMT_U8;
	    fd=DACopen(fnm=DAC_FILE,"w",&samplerate,&i,&n);
	 }
	 if (fd<0) exit(err_rpt(errno,"Opening DSP for output."));
      }
      afmt=i;
      if (vflg) {
	 fputs(fnm,stdout); fputs(": DAC Opened for output\n",stdout);
	 fputd(samplerate,stdout); fputs(" samples/sec\n",stdout);
	 if (i==AFMT_U8) { p="unsigned 8 bit samples, "; }
	 else if (i==AFMT_S16_LE) { p="signed 16 bit samples, little endian, "; }
	 else { p="Unknown audio format, "; }
	 fputs(p,stdout);
	 p=(stereo)?"Stereo mode (2 channels)\n":"Mono mode (1 channel)\n";
	 fputs(p,stdout);
      }
      if (i!=afmt || n!=stereo) {
	 exit(err_rpt(EINVAL,"Sound card doesn't support format requested."));
      }
   } else { 
      afmt=(afmt==AFMT_QUERY)?AFMT_S16_LE:afmt ; /* set 16 bit mode unless
						  * some format is forced */
      omode=(aflg)?"a":"w";
      if ((f=fopen(fnm=outfile,"r"))!=NULL) {
	 fclose(f);
	 if ( (!aflg) && (!fflg) ) exit(err_rpt(EEXIST,fnm));
      }
      if ((f=fopen(fnm,omode))==NULL) { exit(err_rpt(errno,fnm)); }
      if (vflg) { 
	 fputs(fnm,stdout);
	 if (*omode=='a') fputs(": Opened in append mode.\n",stdout);
	 else fputs(": Opened clean for writing.\n",stdout);
      }
      fd=fileno(f);
   }

   lbuf_size=samplerate;                  /* buf sizes if 8 bit samples */
   if (afmt==AFMT_S16_LE) lbuf_size<<=1;  /* double size if 16 bit      */
   else if (afmt!=AFMT_U8) {
      exit(err_rpt(EINVAL,"Only unsigned 8 and signed 16 bit, supported."));
   }
	
   lbuf_flag=rbuf_flag=0;
   lbuf=(unsigned char *)malloc(lbuf_size);
   rbuf=(unsigned char *)malloc(lbuf_size);
   plbuf=(unsigned char *)malloc(plbuf_size=lbuf_size<<stereo);
   if ((rbuf==NULL) || (lbuf==NULL) || (plbuf==NULL)) {
      exit(err_rpt(ENOMEM,"Attempting to get buffers"));
   }
   i=vflg; vflg=0;
   mknull(lbuf,lbuf_size,0,0,samplerate,0,afmt);
   mknull(lbuf,lbuf_size,0,0,samplerate,0,afmt);
   vflg=i;
   
   if (vflg) {
      fputs((stereo)?"Stereo, ":"Mono, ",stdout);
      fputs((afmt==AFMT_S16_LE)?"16":"8",stdout);
      fputs(" bit samples being generated.\n",stdout);
      fputs("Samples scaled by a factor of ",stdout);
      fputd(Gain,stdout); fputs("/100.\n",stdout);
      fputd(lbuf_size,stdout); 
      fputs(" byte buffer allocated per channel.\n",stdout);
      if (configfile!=NULL) {
	 fputs(configfile,stdout);
	 fputs(" is the configuration file.\n",stdout);
      }
   }

   if (configfile!=NULL) {
      doconfig(configfile);
      if (stereo) {
	 /* mix samplerate left, right samples into stereo buffer  */
	 bufmix(plbuf,lbuf,rbuf,samplerate,afmt);
      } else {
	 /* digitally mix the left and right samples into the mono buffer */
	 if (lbuf_flag && rbuf_flag) {
	    chanmix(plbuf,lbuf,rbuf,samplerate,afmt);
	 } else {
	    memcpy(plbuf,(lbuf_flag)?lbuf:rbuf,lbuf_size);
	 }
      }
   } else {
      if (freq > samplerate/2) {
	 fputd(freq,stderr); 
	 fputs(" Hz is more than half the sampling rate\n",stderr);
	 exit(err_rpt(EINVAL,"Frequency setting too great"));
      }
      if (generate(wf,lbuf,lbuf_size,freq,Gain,samplerate,ratio,afmt)==0) {
	 exit(err_rpt(ENOSYS,wf));
      }
      if (stereo) {
	 if (antiphase) {
	    do_antiphase(rbuf,lbuf,samplerate,afmt);
	 } else memcpy(rbuf,lbuf,lbuf_size);
	 bufmix(plbuf,lbuf,rbuf,samplerate,afmt);
      } else {      
	 memcpy(plbuf,lbuf,plbuf_size);
      }
   }

   if (outfile==NULL) { 
      playloop(fd,plbuf,plbuf_size);
   } else {
      if (write(fd,plbuf,plbuf_size) < 0) {
         err_rpt(errno,outfile);
      } else if (vflg) {
         fputs("Samples written to file ",stdout); fputs(outfile,stdout);
         fputs("\n",stdout);
      }
   }
   
   free(lbuf); free(rbuf); free(plbuf);
   close(fd);
   exit(0);
}  

/* doconfig(fnm)   read and action the configuration file fnm
 * 
 *  lines are of form
 * 
 *   [ch] waveform freq [amplitude [extra]]
 */

doconfig(fnm)
char *fnm;
{
   FILE *f;
   char *wf,*s,*p,c,bf[258],*tbuf,*snd;
   int fr,amp,param,*flg;
   
   if (vflg) {
      fputs(fnm,stdout); 
      fputs(" - being read as configuration file\n",stdout);
   }
   if ((tbuf=(unsigned char *)malloc(lbuf_size))==NULL) {
      return(err_rpt(ENOMEM,"Allocating temporary buffer."));
   }

   if ((f=fopen(fnm,"r"))==NULL) {
      return(err_rpt(errno,fnm));
   }
   
   for (; (p=fgets(bf,256,f))!=NULL; ) {
      for ( ; isspace(*p); p++) { }
      if (vflg) fputs(p,stdout);
      if ( *p==0 || *p=='#') continue;
      flg=&lbuf_flag; snd=lbuf; amp=100; param=-1;
      if ((*p=='1' || *p=='2') && isspace(*(p+1))) {
	 if (*p=='2') {
	    snd=rbuf; flg=&rbuf_flag;
	 }
	 for ( ++p; isspace(*p); p++) { }
      }
      for (wf=p; *p && !isspace(*p); p++) { }
      if (*p) *p++=0;
      for ( ; isspace(*p); p++) { }
      fr=(int)strtol(p,&s,10);
      p=s;
      if (*p && isspace(*p)) {
	 for ( ; isspace(*p); p++) { }
	 amp=(int)strtol(p,&s,10);
	 p=s;
	 if (*p && isspace(*p)) {
	    for ( ; isspace(*p); p++) { }
	    param=(int)strtol(p,&s,10);
	    p=s;
	 }
      }
      if ((generate(wf,tbuf,lbuf_size,fr,amp,samplerate,param,afmt))==0) {
	 err_rpt(ENOSYS,wf);
	 continue;
      }
      if (*flg) {
	 bufmix(snd,snd,tbuf,samplerate,afmt);
      } else {
	 memcpy(snd,tbuf,lbuf_size);
	 *flg=1;
      }
   }

   free(tbuf);
   fclose(f);
}

/*
 *  playloop(fd,bf,bfn)   bf is a sample buffer of size bfn bytes.
 *      playloop plays this continuously to file descr fd
 */

playloop(fd,bf,bfn)
int fd;
unsigned char *bf;
int bfn;
{
   int i,bn;
   unsigned char *p;
   
   bn=bfn/10;
   for (;;) {
      for ( p=bf,i=0; i< bfn; i+=bn) { 
	 if (write(fd,p+i,bn) < 0) return(0);
	 /* check here if need to supend output or handle keypresses etc*/
      }
      if ((i!=bfn) && (write(fd,p+i-bn,bfn-i+bn)<0)) return(0);
   }
}

/*
 * DACopen(char *fnm, char *mode, int *samples, int *fmt, int *stereo)
 * 
 *   open dspfile for read "r" or write "w", and set samples per sec
 *    as sampling rate - note we get pointer for samples so we can 
 *    return the actual samplerate set. Stereo is non zero for stereo,
 *    and fmt is the format (8 or 16 bit)
 *    and again these are pointers, so actually settings can be returned.
 *   return file descriptor or -1 on error.
 */

DACopen(fnm,mode,samples,fmt,stereo)
char *fnm;
char *mode;
int *samples, *stereo, *fmt;
{
   int fd;
   int m;
   
   if (*mode=='r') m=O_RDONLY;
   else if (*mode=='w') m=O_WRONLY;
   else {
      errno=EINVAL; return(-1);
   }
   if ((fd = open (fnm,m,0)) >= 0) {  /* params must be set in this order */
      if (ioctl(fd, SNDCTL_DSP_SETFMT, fmt)>=0) { 
	 if (ioctl(fd, SNDCTL_DSP_STEREO, stereo)>=0) { 
	    if (ioctl(fd, SNDCTL_DSP_SPEED, samples)>=0) { 
	       return(fd);
	    }
	 }
      }
   }
   return(-1);
}

