/* mix.c
 * Simple Linux Mixer controller
 * Jim Jackson  Fri 16 Jun 1995
 */
/*
mix: (version 1) Nov 1996
based on the dos sb16 mixer......
mix: (version 1) Fri 16 Jun 1995

NAME:
        mix  -  Simple Linux Mixer controller

SYNOPSIS:
        mix [-v] [-h] [-o file] [-i file] [command(s)] 

DESCRIPTION:

Controls the Mixer settings from the command line parameter(s).
The commands are detailed below, capitals showing the minimum abbreviation
allowed. Upper or lower case can be used on the command line. All Volume
settings are in range 0-100 (0 min, 100 max), but these are scaled to the
mixers actual range, hence set volume may be slightly different.

SHow              outputs the settings of the mixer. This is the default 
                   if no command line parameters are given
dev N or L,R      sets mixer device 'dev' to volume N, or to seperate
                    left and right stereo volume L,R
                  If device doesn't support stereo settings then max of L,R
                    is used.
INput dev         set the DSP input to be 'dev' or 'NOne' to turn inputs off
Verbose           makes the program output the settings after doing the
                   commands
                   
Use '-' as a filename to indicate standard input.

FLAGS:

-h           lists usage summary, which also lists the mixer devices and
              the possible input devices.
-v           verbose - outputs the results of commands. Same as Verbose above

OPTIONS:

-i file      read commands from file
-o file      output to file file

NOTES:

BUGS:

Any questions, suggestions or problems, etc. should be sent to

Jim Jackson
School of Computer Studies
University of Leeds
Leeds LS2 9JT
U.K.

Email:  jj@scs.leeds.ac.uk

 */

#include <stdio.h>
#include <strings.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <linux/soundcard.h>

#define PROGNAME "mix"

#define EFAIL -1

#define TRUE 1
#define FALSE 0
#define nl puts("")
#define option(b,a) ((n>=a)&&(strncasecmp(p,b,n)==0))

#define doshow() printvols(devmask)

#define MIXDEV "/dev/mixer"

int mixer;           /* Mixer File Descriptor */
int devmask,recmask,sources,stereodevs;
char *devname[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_LABELS;


char *outfile;
char *infile;
int vflg;
char *sys;

char *strupper(),*reads(),*index(),*sindex(),*tab(),*strncpy(),*strcpy(),
     *strcmp();
FILE *fopen();
unsigned char readmixer();

main(argc,argv)
int argc;
char **argv;

 { int i,j,n,st;
   char fname[132],*p;
   FILE *fpr,*fpw;


   sys=*argv++; argc--;
   sys=(*sys)?sys:PROGNAME;
   if ((p=strrchr(sys,'/'))!=NULL) { sys=p+1; }
        
   infile=outfile=NULL;
   vflg=FALSE;

   if ((mixer = open(MIXDEV, O_RDONLY)) < 0) {
     exit(err_rpt(errno,MIXDEV));
   }
   if (mixinit()) {
      exit(err_rpt(errno,"Getting Mixer details."));
   }
   
   while (argc && **argv=='-')
    { n=strlen(p=(*argv++)+1); argc--;
      if (option("output",1))
        { if (argc)
           { outfile=*argv++; argc--; }
        }
      else if (option("input",1))
        { if (argc)
           { infile=*argv++; argc--; }
        }
      else
	{ for (; *p; p++)
	    { if (*p=='h') exit(help());
	      else if (*p=='v') vflg=TRUE;
	      else
		{ *fname='-'; *(fname+1)=*p; *(fname+2)=0;
		  exit(err_rpt(EINVAL,fname));
		}
	    }
	}
    }

   if (outfile!=NULL)
    { if (strcmp(outfile,"-")==0) fpw=stdout;
      else if ((fpw=fopen(outfile,"w"))==NULL)
       { exit(err_rpt(errno,outfile)); }
    }
   else
    { fpw=stdout; }

   if (infile!=NULL)
    { if (strcmp(infile,"-")==0) fpr=stdin;
      else if ((fpr=fopen(infile,"r"))==NULL)
       { exit(err_rpt(errno,infile)); }
    }
   else
    { fpr=stdin; }

   if (infile!=NULL)
    { st=docmdfile(infile,fpr);  /* do any command file first */
      st+=docmds(argc,argv);
    }
   else if (argc)
      st+=docmds(argc,argv);
   else
      st+=docmdline("show");
    
   exit(st);
 }

docmdfile(fn,f)
char *fn;
FILE *f;
{
  char bf[514],*p;
  
  while ((p=fgets(bf,512,f))!=NULL) { docmdline(bf); }

}

docmdline(s)
char *s;
{
  int n,v;
  char *aa[64],c,*p,*r,*q;

  for ( ; *s && isspace(*s); s++) {} /* skip initial whitespace */
  if (*s=='#') return(0);            /* comment line - ignore   */
  for (n=0; *s; n++)
   { 
     aa[n]=s++;
     for ( ; *s && !isspace(*s); s++ ) {}
     if (isspace(*s)) *s++=0;
     for ( ; *s && isspace(*s); s++) {}
   } 
  docmds(n,aa);
}

docmds(c,v)
int c;
char **v;
{
   int i,st,n,d,dopr;
   char *p;

   for ( ; c; )
    {
      n=strlen(p=*v++); c--;
      if (option("show",2)) doshow();
      else if (option("verbose",1)) vflg=1;
      else if (option("input",2)) {
         dopr=0;
	 if ( c && ((d=isdev(p=*v,n=strlen(*v)))>=0 || option("none",2))) {
	    v++; c--; 
            i=(d>=0)?(1<<d):0;
	    if ( i && (recmask&i)==0 ) {
               fprintf(stderr,"%s is not a source device.\n",devname[d]);
	    } else {
	       if (ioctl(mixer, SOUND_MIXER_WRITE_RECSRC, &i) < 0) {
		  fprintf(stderr,"Couldn't set %s as recording source.\n",
                          (i)?(devname[d]):"nothing");
	       } else { sources=i; };
               if (vflg) dopr=1;
            }
	 } else { dopr=1; }
         if (dopr) {
            if (sources) printvols(sources);
            else printf("No source inputs active.");
	 }
      } else if (option("all",3)) {
	 if (c && isvol(*v)) {
	    p=*v++; c--;
	    doallvols(devmask,p);
	 } else doshow();
      } else if ((d=isdev(p,n))>=0) {
	 if (c && isvol(*v)) {
	    p=*v++; c--;
	 } else p="";
	 dovol(d,p);
      } else { 
	 fprintf(stderr,"Command '%s' not recognised.\n",p);
	 st=1;
      }
    }
}

isvol(s)
char *s;
{
   return(isdigit(*s) ||
	  (strncasecmp(s,"off",3)==0) ||
	  (strncasecmp(s,"full",4)==0));
}

/* dovol(d,s)   set mixer device d to volume specified by string s
 *             s is an "NN" or "LL,RR", i.e. a numeric ascii string
 *             or two num ascii strings for stereo left and right settings
 */

dovol(d,s)
int d;
char *s;
{
  int lv,rv,n,dopr;
  char buf[130];

   dopr=vflg;
   if (s!=NULL && *s) {
      if (strncasecmp(s,"off",3)==0) { lv=0; s+=3; }
      else if (strncasecmp(s,"full",4)==0) { lv=100; s+=4; }
      else { 
	 for ( lv=0; isdigit(*s); s++) { lv=lv*10+(*s-'0'); } 
      }
      if (*s == ',' || *s == '/') {
	 s++;
	 if (strcasecmp(s,"off")==0) { rv=0; }
	 else if (strcasecmp(s,"full")==0) { rv=100; }
	 else {
	    for ( rv=0; isdigit(*s); s++) { rv=rv*10+(*s-'0'); }
	 }
      } else rv=lv;
      if (isstereo(d)) n=(rv<<8)+lv;
      else n=(lv>rv)?lv:rv;
      if (ioctl(mixer, MIXER_WRITE(d), &n) < 0) {
	 sprintf(buf,"Problem setting level of mixer channel %s.",
		 devname[d]);
	 err_rpt(ENODEV,buf);
      }
   } else { dopr=1; }
   if (dopr) printvol(d);
}

printvols(devs)
int devs;
{
   int i;

   for (i=0; i<SOUND_MIXER_NRDEVICES; i++) {
      if (devs&(1<<i)) {
         printvol(i);
      }
   }
}

doallvols(devs,s)
int devs,s;
{
   int i;

   for (i=0; i<SOUND_MIXER_NRDEVICES; i++) {
      if (devs&(1<<i)) {
         dovol(i,s);
      }
   }
}

printvol(d)
int d;
{
   int n;
   char buf[130];
   
   if (devmask&(1<<d)) {
      if (ioctl(mixer, MIXER_READ(d), &n) < 0) {
	 sprintf(buf,"Problem reading current level of mixer channel %s.",
		 devname[d]);
	 err_rpt(ENODEV,buf);
      } else {
	 printf("%-6s  %3d",devname[d],n&255);
	 if (stereodevs & (1<<d)) { printf("/%-3d ",(n>>8)&255);
	 } else { printf("     "); }
	 printf("%2s ", (recmask & (1<<d))? "*":""); 
	 printf("%3s ", (sources & (1<<d))? "<-":"");
	 puts(""); /* nl */
      }
   }
}

/* mixinit()   get mixer characteristics - return 0 if ok
 *      else return non-zero errno
 */

mixinit()
{
   if ((ioctl(mixer, SOUND_MIXER_READ_DEVMASK, &devmask) >= 0) &&
       (ioctl(mixer, SOUND_MIXER_READ_RECMASK, &recmask) >= 0) &&
       (ioctl(mixer, SOUND_MIXER_READ_RECSRC, &sources) >= 0)  &&
       (ioctl(mixer, SOUND_MIXER_READ_STEREODEVS, &stereodevs) >= 0)) {
      return(0);
   }
   return(errno);
}

isstereo(d)
int d;
{
   return(stereodevs&(1<<d));
}

isdev(dev,n)
char *dev;
int n;
{
   int i;
   
   for (i=0; i<SOUND_MIXER_NRDEVICES; i++) {
      if (strncasecmp(dev,devname[i],n)==0) {
	 return((devmask&(1<<i))?i:-1);
      }
   }
   return(-1);
}
   
help()
{
int i;

printf("\nSet Linux Mixer from the command line\n\n");
printf("Commands are :-\n");
printf(" Verbose or -v  verbose mode, outputs results of actions\n");
printf(" DEV N or L,R   sets DEV to N or left to L, Right to R\n");
printf("  where DEV can be :\n   ");
for (i=0; i<SOUND_MIXER_NRDEVICES; i++) {
  if (devmask&(1<<i)) printf("%s ",devname[i]);
}
printf("\n INput DEV      sets the DSP input to be DEV, where DEV can be :\n   ");
for (i=0; i<SOUND_MIXER_NRDEVICES; i++) {
  if (recmask&(1<<i)) printf("%s ",devname[i]);
}
printf("\n\n");
printf("Usage:   %s [-h] [-v] [-i cmdfile] [-o file] [command(s)]\n\n",sys);
}

/***ERR_RPT***/
/* err_rpt(err,msg)
 */

err_rpt(err,msg)
short int err;
char *msg;

 { extern int sys_nerr;
   extern const char *const sys_errlist[];
   extern char *sys;

   if (err>sys_nerr)
     { fprintf(stderr,"%s: Err %d :-  %s\n",sys,err,msg);
     }
   else if (err) fprintf(stderr,"[%s] %s : %s\n",sys,sys_errlist[err],msg);
   return(err);
 }

/***STRUPPER***
 *  char *strupper(p)   convert all lower case chars to upper in str p
 */

char *strupper(p)
char *p;
 { char *i,d;
   d='a'-'A';
   for (i=p ; *i ; i++)
     if ((*i>='a') && (*i<='z')) *i-=d ;
   return(p);
 }

/***DELNL***/
/* char *delnl(s)   remove trailing nl from end of string s
 *                  return same pointer as given
 */

char *delnl(s)
char *s;

 { char *p;

   if (s!=NULL && *s && *(p=s+strlen(s)-1)=='\n') *p=0;
   return(s);
 }
