/* sweepgen.c
 * Ncurses based Sweep generator, 16/8 bit, 2 channels if stereo card,
 * Emulates an audio sweep signal generator
 * both the swept and sweeping frequencies can be
 *   sine, square, triangle, sawtooth, pulse, noise waveforms
 * 
 * Linux Version
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

/*
 * sweepgen :
 * 
 * 1 secs worth of 1 Hz samples for each waveform are generated, and the
 * sweeping frequency F is generated by picking out every Fth sample
 * The swept freq, f is generated by using the samples from F to 
 * increment a selection pointer out of the swept waveform buffer.
 * The selected samples are stored in a playbuff that is fed to the
 * DSP when full.
 * 
 * 
 * History:
 *  03Nov97 V1.0  Started based on siggen 1.5
 */

#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <math.h>
#include "sweepgen.h"

#define MAXPRM 32
#define chkword(a,b) ((n>=a)&&(strncmp(p,b,n)==0))

int vflg,dflg;

int DAC;                         /* File Handle for DSP */
unsigned int samplerate;         /* Samples/sec        */
unsigned int stereo;             /* stereo mono */
unsigned int afmt;               /* format for DSP  */
int Nfragbufs;                   /* number of driver frag buffers */
int fragsize;                    /* size of driver buffer fragments */
int fragsamplesize;              /* size of fragments in samples */

    /* channel 1 the sweeping signal  ... */
char wf[32]="sawtooth";          /* waveform type */
unsigned int freq=1;             /* signal frequency */
int ratio=-1;                    /* used in pulse, sweep etc */
int Gain=100;                    /* Amplification factor */

    /* channel 2  the swept signal ... */
char wf2[32]="sine";             /* waveform type */
unsigned int freqF=500;          /* signal low frequency */
unsigned int freqT=1500;         /* signal high frequency */
unsigned int freqC=1000;         /* signal centre frequency */
int frdev=500;                   /* signal deviation frequency */
int ratio2=-1;                   /* used in pulse, sweep etc */
int Gain2=100;                   /* Amplification factor */
int phase=0;                     /* phase difference of chan2 from chan1 */

char *sys;

help(e)
int e;
{  fputs(VERSION,stderr);
   fputs("\nUsage: \n 1: sweepgen [flags] [waveform [freq [param]]]\n",stderr);
   fputs("      waveforms are sine, square, triangle, sawtooth, noise, pulse\n\n",stderr);
   fputs("Defaults: output to /dev/dsp, 22050 samples/sec, stereo, 16 bit\n",stderr);
   fputs("          samples if possible, else 8 bit and/or mono. \n\n",stderr);
   fputs("flags: -s samples    generate with samplerate of samples/sec\n",stderr);
   fputs(" -8/-16 or -b 8|16   force 8 bit or 16 bit mode.\n",stderr);
   fputs("       -1    force mono - swept frequency output only\n",stderr);
   fputs("       -2    force stereo - swept freq on one chan, sweeping on other chan.\n",stderr);
   fputs("      -NB n  Numer of Buffers to create is n, def. is 3\n",stderr);
   return(e);
}

/* main
 *
 */
 
main(argc,argv)
int argc;
char **argv;
{
   unsigned int v[MAXPRM],maxv,i,j,k,l,m,n,N;
   FILE *f;
   char *p,bf[130];
   int c;
   unsigned int t;
   
   if ((p=strrchr(sys=*argv++,'/'))!=NULL) sys=++p;
   argc--;
   
   samplerate=SAMPLERATE; afmt=AFMT_QUERY;
   Nfragbufs=DEFAULT_BUFFERS;
   stereo=-1;
   vflg=dflg=0;
   
   while (argc && **argv=='-') {          /* all flags and options must come */
      n=strlen(p=(*argv++)+1); argc--;    /* before paramters                */
      if (chkword(1,"samplerate")) {
	 if (argc && isdigit(**argv)) { samplerate=atoi(*argv++); argc--; }
      }
      else if (chkword(2,"NB")) {
	 if (argc && isdigit(**argv)) { Nfragbufs=atoi(*argv++); argc--; }
      }
      else if (chkword(2,"16")) { afmt=AFMT_S16_LE; }
      else if (chkword(1,"bits")) {
	 i=0;
	 if (argc) { 
	    i=atoi(*argv++); argc--;
	 }
	 if (i==8) afmt=AFMT_U8;
	 else if (i==16) afmt=AFMT_S16_LE;
	 else exit(err_rpt(EINVAL,"must be '-b 8' or '-b 16'."));
      }
      else if (chkword(1,"A")) {
	 if (argc && isdigit(**argv)) { 
	    Gain=atoi(*argv++); argc--;
	 }
      }
      else {                              /* check for single char. flags    */
	 for (; *p; p++) {
	    if (*p=='h') exit(help(EFAIL));
	    else if (*p=='1') stereo=0;
	    else if (*p=='2') stereo=1;
	    else if (*p=='8') afmt=AFMT_U8;
	    else if (*p=='d') dflg=1;
	    else if (*p=='v') vflg++;
	    else {
	       *bf='-'; *(bf+1)=*p; *(bf+2)=0;
	       exit(help(err_rpt(EINVAL,bf)));
	    }
	 }
      }
   }
   
   if (argc) {
      strncpy(wf,*argv++,32); wf[31]=0; argc--;    /* waveform type */
      if (argc) {
	 freq=atoi(*argv++); argc--;
	 if (argc) { ratio=atoi(*argv++); argc--; }
	 if (argc) exit(help(err_rpt(EINVAL,"Too many parameters")));
      }
   }
   
   /* if no format specified then try 16 bit */
   i=afmt;
   n=stereo;
   if ((DAC=DACopen(DAC_FILE,"w",&samplerate,&i,&n))<0) {
      exit(err_rpt(errno,"Opening DSP for output."));
   }
   if (((afmt!=AFMT_QUERY) && (i!=afmt)) ||
       ((stereo!=-1) && (n!=stereo))) {
      exit(err_rpt(EINVAL,"Sound card doesn't support format requested."));
   }
   afmt=i; stereo=n;

   if ((fragsize=setfragsize(DAC,Nfragbufs,samplerate,afmt,stereo))<0) {
      exit(err_rpt(errno,"Problem setting appropriate fragment size."));
   }
   fragsamplesize=(fragsize>>(afmt==AFMT_S16_LE))>>(stereo);
       
   if (freq > samplerate/2) {
      fprintf(stderr,"%d Hz is more than half the sampling rate\n",freq);
      exit(err_rpt(EINVAL,"Frequency setting too great"));
   }

   if (vflg) {
      printf("%s %s bit samples being generated.\n",
	     (stereo)?"Stereo, ":"Mono, ",(afmt==AFMT_S16_LE)?"16":"8");
      printf("Samples scaled by a factor of %d/100\n",Gain);
      printf("Playing at %d samples/sec\n",samplerate);
      printf("Buffer fragment size is %d bytes (%d samples). Aprox. %d millisecs.\n",
	     fragsize, fragsamplesize,
	     1000*((fragsize>>(stereo))>>(afmt==AFMT_S16_LE))/samplerate);
      printf("\n<Press Return to Continue>\n");
      getchar();
   }

   WinGen(DAC);
   
   close(DAC);
   exit(0);
}  
