/* sweepscr.c
 * Screen handling for ncurses SweepGEN program
 * Jim Jackson    Nov 97
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include "scfio.h"
#include "sweepgen.h"
#include <sys/soundcard.h>

/* set number of millisecs to wait while checking for ESC sequences */

#define KEY_DELAY 25

/* Main screen character strings............
 */

char *hl[]={
   " /====\\                         /====\\ ====== |\\    |  ",
   " |                              |      |      | \\   |  ",
   " \\====\\ |    |  /-\\  /-\\  /--\\  |   __ |===   |  \\  | ",
   "      | |    | |  / |  /  |  |  |    | |      |   \\ |    ",
   " \\====/  \\/\\/   \\--  \\--  |--/  \\====/ ====== |    \\| ",
   "                          |     ",
   "           Digital Sweep  |  Generator ", 
   " -----------------------------------------------------------------------------",
   NULL };

char *oh[]={
   "      /\\         ",
   "   --/--\\--/--   ", 
   "         \\/      ",
   NULL };

char *sig=" Jim Jackson <jj@scs.leeds.ac.uk> ";

int Lsamplerate,             /* Local copy of card params to prevent */
    LNfb,                   
    Lafmt,Lstereo;           /* playing being screwed while updating */
                             /* card characteristics.                */

char Mchan1[]="  Sweep WaveForm ",
     Mchan2[]="  Swept WaveForm ",
     Mstarting[]=" Starting up .......... ",
     Mwavef[]="  Waveform: ",
     Mfreq[]= " Frequency:        Hz",
     Mfreq1[]= " Frequency From:        Hz  To:        Hz",
     Mfreq2[]= "         Centre:        Hz  +/-        Hz",
     Mratio[]=" M/S Ratio: ",
     Mgain[]= "Hz  Gain: ",
     Msampl[]=" Samplerate       /sec                    x       byte buffs",
     Minfo[]=" : ",
     Mtime[16]="",
     Mstereo[]="STEREO",
     Mmono[]=  " MONO ",
     M16b[]=  "16 bit",
     M8b[]=   " 8 bit",
     Mmain[]=
     "<- / -> move fields,                         'Q'uit, 'S'etup samplerate etc",
     Mcard[]=
     "'S' to action new settings               Left/Right Arrow Keys move fields.";

char *Mbits[]={ M8b, M16b, NULL };
char *Mchans[]={ Mmono, Mstereo, NULL };
   
struct SCField
 FsamP= { SCF_string, 9, 0, 0, Msampl, 80, 0, 0 },
 FsamV= { SCF_integer, 9, 12, 0, &Lsamplerate, 6, 0, 0 },
 Fbits= { SCF_option, 9, 24, 0, Mbits, 6, 0, 0 },
 Fchans= { SCF_option, 9, 32, 0, Mchans, 6, 0, 0 },
 FNfrag= { SCF_integer, 9, 40, 0, &LNfb, 2, 0, 0 },
 Ffrag= { SCF_integer, 9, 43, 0, &fragsize, 6, 0, 0 },
 FinfoP= { SCF_string, 21, 0, 0, Minfo, 80, 0, 0 },
 FinfoP2= { SCF_string, 22, 0, 0, Minfo, 80, 0, 0 },
 Finfo= { SCF_string, 21, 3, 0, Mstarting, 75, 0, 0 },
 Finfo2= { SCF_string, 22, 3, 0, "", 75, 0, 0 },
 Ftime= { SCF_string, 9, 69, 0, Mtime, 10, 0, 0 },

 Fchan1= { SCF_string,  11, 0, 0, Mchan1, 12, 0, 0 },
 Fwav1P= { SCF_string,  12, 2, 0, Mwavef, 12, 0, 0 },
 Fwav1V= { SCF_option,  12, 14, 0, NULL, 8, 0, 0 },
 FfreqP= { SCF_string,  12, 22, 0, Mfreq, 22, 0, 0 },
 FfreqV= { SCF_integer, 12, 34, 0, &freq, 6, 0, 0 },

 Fchan2= { SCF_string,  15, 0, 0, Mchan2, 12, 0, 0 },
 Fwav2P= { SCF_string,  16, 2, 0, Mwavef, 12, 0, 0 },
 Fwav2V= { SCF_option,  16, 14, 0, NULL, 8, 0, 0 },
 Ffrq1P= { SCF_string,  16, 22, 0, Mfreq1, 45, 0, 0 },
 Ffrq2P= { SCF_string,  17, 22, 0, Mfreq2, 45, 0, 0 },
 FfrqFV= { SCF_integer, 16, 39, 0, &freqF, 6, 0, 1 },
 FfrqTV= { SCF_integer, 16, 54, 0, &freqT, 6, 0, 2 },
 FfrqCV= { SCF_integer, 17, 39, 0, &freqC, 6, 0, 3 },
 FfdevV= { SCF_integer, 17, 54, 0, &frdev, 6, 0, 4 },
 Fgn2P=  { SCF_string,  16, 40, 0, Mgain, 10, 0, 0 },
 Fgn2V=  { SCF_integer, 16, 50, 0, &Gain2, 5, 0, 0 };

struct SCField 
 *CHhdrs[]= { &Fchan1, &Fwav1P, &FfreqP, 
              &Fchan2, &Fwav2P, &Ffrq1P, &Ffrq2P,
/*              &Fgn2P, */
              NULL },
 *CHvals[]= { &Fwav1V, &FfreqV,
              &Fwav2V, &FfrqFV, &FfrqTV, &FfrqCV, &FfdevV,
/*              &Fgn2V, */
              NULL },
 *scrP[]= { &FsamP, &FinfoP, &FinfoP2, &Finfo, &Finfo2, 
            &Ffrag, &Ftime, NULL, },
 *scrV[]= { &FsamV, &Fbits, &Fchans, &FNfrag, NULL };
    
/* Help strings for second info line.
 */

char Hwav[]="DOWN (or SPACE) & UP ARROW to select WaveForm",
     Hnum[]="Enter Digits, DEL or <- to delete, UP/DOWN arrow for Inc/Decrement",
     Hbits[]="Press <SPACE> to toggle 8 / 16 bit setting",
     Hchan[]="Press <SPACE> to toggle Mono / Stereo setting";
    

char *Hvals[]={ Hwav, Hnum, 
                Hwav, Hnum, Hnum, Hnum, Hnum, NULL };
char *Hset[]= { Hnum, Hbits, Hchan, Hnum, NULL };

/* waveform name and buffer array.......
 * same number or more entries as WAVEFORMS[] ....
 */

char **WAVEFORMS;
int **wavbufs;

/* WinGen()   The main interactive screen
 */

WinGen(fd)
int fd;
{
   int c,i,n,st,t;
   unsigned int j,k;
   int f,wavef,card;
   char **hh;
   struct SCField **ff;
   
   if (n=WinGenInit()) return(n);
   
   hh=Hvals; ff=CHvals;
   doinfo2(Mmain,0);
   wavef=card=f=0;
   dispfield(ff,hh,f);
   refresh();
   t=time(NULL);
     
   for (;;) {
      if ((i=time(NULL))-t) { dotime(t=i); }
      if (playsamples(fd)<0) {
	 move(COLS-1,0); endwin();
	 return(err_rpt(errno,"Problem generating output samples."));
      }
      if ((c=getch())==-1) {
         delay(25000);
	 continue;
      }
      if (c=='s' || c=='S') {  
	 if (card) {
	    /* check if card params have changed - if so close and reopen */
            Lafmt=(Fbits.adj)?AFMT_S16_LE:AFMT_U8;
	    Lstereo=Fchans.adj; 
            if ((Lsamplerate!=samplerate) || (Lafmt!=afmt) || (Lstereo!=stereo) || (LNfb!=Nfragbufs)) {
	       close(fd);
	       if ((fd=DACopen(DAC_FILE,"w",&Lsamplerate,&Lafmt,&Lstereo))<0 ||
		   (fragsize=setfragsize(fd,LNfb,Lsamplerate,Lafmt,Lstereo))<0) {
		  move(COLS-1,0); endwin();
		  return(err_rpt(errno,"Opening DSP for output."));
	       }
	       fragsamplesize=(fragsize>>(Lafmt==AFMT_S16_LE))>>(Lstereo);
	       putfield(&Ffrag);
	    }
	    dofields(scrV);
	    /* if samplerate or format has changed, re-generate waveforms */
	    /* we generate 16 bit samples cos we need it for sweeping freq */
	    /*  then we adjust if we are writing 8 bit samples             */
	    if ((Lsamplerate!=samplerate) || (Lafmt!=afmt)) {
	       if (genwavbufs(WAVEFORMS,wavbufs,Lsamplerate,AFMT_S16_LE)) {
	          move(COLS-1,0); endwin();
		  return(err_rpt(EINVAL,"Problem generating Base waveforms."));
	       }
	    }
	    samplerate=Lsamplerate; afmt=Lafmt; stereo=Lstereo; Nfragbufs=LNfb;
            playsamples(-1);    /* resync left and right channels */
	    dispfield(ff,hh,f);
	    card=0; f=wavef;
	    ff=CHvals; hh=Hvals;
	    dispfield(ff,hh,f);
	    doinfo2(Mmain,0);
	 } else {   /* enter card setting alteration stuff */
	    dispfield(ff,hh,f);
	    card=1; wavef=f; f=0;
	    ff=scrV; hh=Hset;
	    dispfield(ff,hh,f);
	    doinfo2(Mcard,0);
	 }
      }      

      if (c=='q' || c=='Q') {
	 move(COLS-1,0); endwin();
	 return(c);
      }
      f=doinput(c,ff,hh,f);
      if (i=(ff[f]->id)) {
	 if (i<=2) {
	    freqC=(freqF+freqT+1)/2;
	    frdev=((freqT>freqF)?freqT:freqF)-freqC;
	 } else {
	    if ((i==3) && (freqC<frdev)) freqC=frdev;
	    else if ((i==4) && (freqC<frdev)) frdev=freqC;
	    if (freqF<freqT) { 
	       freqF=freqC-frdev; freqT=freqC+frdev; 
	    } else {
	       freqT=freqC-frdev; freqF=freqC+frdev; 
	    }
	 }
	 putfield(&FfrqCV); putfield(&FfdevV);
	 putfield(&FfrqTV); putfield(&FfrqFV);
	 putfield(ff[f]);
      }
   }
}

/*
 *  doinput(c,aa,H,f)  action char c on field aa[f]
 *      if field changes display old field in normal and  new
 *      field in reverse, and display relevant new help line from H[]
 *  return new field value.
 */

doinput(c,aa,H,f)
struct SCField *aa[];
char *H[];
int f;
{
   int fn;
   
   c=actfield(c,aa[f]);
   fn=-1;
   if (c==KEY_LEFT) {
      if (f) fn=f-1;
      else {
	 for (fn=f ; aa[fn+1]!=NULL; fn++) { }
      }
   } else if (c==KEY_RIGHT || c=='\r' || c=='\n' || c=='\t') {
	 fn=f+1; if (aa[fn]==NULL) fn=0;
   } else if (c==KEY_HOME) { fn=0; }
   if (fn!=-1) {
      dispfield(aa,H,f);
      f=fn;
      dispfield(aa,H,f);
   }
   refresh();
   return(f);
}

dispfield(ff,hh,f)
struct SCField *ff[];
char *hh[];
int f;
{
   doinfo(hh[f],0);
   ff[f]->attr=reverse_attr(ff[f]->attr);
   putfield(ff[f]);
}

WinGenInit()
{
   int att_h1,att_h2,att_h3,att_h4,att_h5,att_h6;
   int n,i;
   
   Lsamplerate=samplerate;         /* set local working copy of samplerate */
   Lafmt=afmt; Lstereo=stereo;
   LNfb=Nfragbufs;

   Fbits.adj=(afmt==16);
   Fchans.adj=stereo;
   
   WAVEFORMS=(char **)getWavNames();
   Fwav1V.val.sp=WAVEFORMS;   Fwav2V.val.sp=WAVEFORMS;
   for (n=0; WAVEFORMS[n]!=NULL; n++) { }
   if ((wavbufs=(int **)malloc((n+1)*sizeof(int *)))==NULL) {
      return(err_rpt(ENOMEM,"Out of Memory."));
   }
   for (i=0; i<=n; i++) { wavbufs[i]=NULL; }
   
   if (genwavbufs(WAVEFORMS,wavbufs,samplerate,afmt)) {
      return(err_rpt(EINVAL,"Problem generating Base waveforms."));
   }
   setwaveform(&Fwav1V,wf);
   setwaveform(&Fwav2V,wf2);
   
   tzset();              /* set timezone stuff up for dotime() function */
   
   initscr();
   cbreak(); noecho();
   nonl(); intrflush(stdscr,FALSE); keypad(stdscr,TRUE);
   nodelay(stdscr,TRUE);
   timeout(KEY_DELAY);
   
   if (has_colors()) {
      start_color();
      init_pair(1,COLOR_BLUE,COLOR_GREEN);
      init_pair(COLOR_PAIRS-1,COLOR_GREEN,COLOR_BLUE);
      init_pair(2,COLOR_CYAN,COLOR_RED);
      init_pair(COLOR_PAIRS-2,COLOR_RED,COLOR_CYAN);
      init_pair(3,COLOR_GREEN,COLOR_RED);
      init_pair(COLOR_PAIRS-3,COLOR_RED,COLOR_GREEN);
      init_pair(4,COLOR_YELLOW,COLOR_RED);
      init_pair(COLOR_PAIRS-4,COLOR_RED,COLOR_YELLOW);
      init_pair(5,COLOR_WHITE,COLOR_BLUE);
      init_pair(COLOR_PAIRS-5,COLOR_BLUE,COLOR_WHITE);
      init_pair(6,COLOR_RED,COLOR_BLUE);
      init_pair(COLOR_PAIRS-6,COLOR_BLUE,COLOR_RED);
      att_h1=COLOR_PAIR(1);
      att_h2=COLOR_PAIR(2)+A_BOLD;
      att_h3=COLOR_PAIR(3);
/*      att_h4=COLOR_PAIR(4)+A_BOLD; */
      att_h4=COLOR_PAIR(2);
      att_h5=COLOR_PAIR(5);
      att_h6=COLOR_PAIR(6);
   } else {
      att_h1=A_REVERSE; att_h5=att_h6=att_h2=att_h4=A_BOLD;
      att_h3=A_NORMAL;
   }
   
   dofatt(scrV,att_h4); dofatt(scrP,att_h3);
   Ffrag.attr=att_h4;
   dofatt(CHhdrs,att_h5);
   dofatt(CHvals,att_h6);
   
   dohdr(0,att_h1,att_h2,att_h3);
   n=8;
   for (i=0; i<n; i++) {
      mvputfstr(i+Fchan1.row-1,0,att_h5,"",80);
   }

   dofields(scrP); dofields(scrV);
   dofields(CHhdrs);
   dofields(CHvals);
   refresh();
}

setwaveform(sp,wv)
struct SCField *sp;
char *wv;
{
   int i,n;
   
   n=strlen(wv);
   for (sp->adj=i=0; WAVEFORMS[i]!=NULL; i++) {
      if (strncasecmp(wv,WAVEFORMS[i],n)==0) {
	 sp->adj=i; return(0);
      }
   }
}

/* dohdr(y,a1,a2,a3)  write header out starting at line y using attributes
 *   a1, s2, and a3
 */

dohdr(y,a1,a2,a3)
int y,a1,a2,a3;
{
   int i;

   for ( i=0; hl[i]!=NULL; i++) {
      mvputfstr(y+i,0,a1,hl[i],80);
   }
   for ( i=0; oh[i]!=NULL; i++) {
      mvputfstr(y+i+1,55,a2,oh[i],17);
   }
   mvputfstr(y+6,44,a3,sig,strlen(sig));
   mvputfstr(24,0,a1,VERSION,80);
   dotime(time(NULL));
}

/* Display string s in the Info field of the screen, and pause for w
 * seconds, if w is non-zero
 */

doinfo(s,w)
char *s;
int w;
{
   Finfo.val.s=s;
   putfield(&Finfo);
   refresh();
   if (w) sleep(w);
}

/* Display string s in the 2nd Info field of the screen, and pause for w
 * seconds, if w is non-zero
 */

doinfo2(s,w)
char *s;
int w;
{
   Finfo2.val.s=s;
   putfield(&Finfo2);
   refresh();
   if (w) sleep(w);
}

dotime(t)
int t;
{
   int y,x;

   t+=(daylight*3600);
   sprintf(Mtime,"%2d:%02d:%02d",(t/3600)%24,(t/60)%60,t%60);
   getyx(stdscr,y,x);
   putfield(&Ftime);
   move(y,x);
   refresh();
}

dofields(aa)
struct SCField *aa[];
{
   int i;
   
   for (i=0; aa[i]!=NULL; i++) { putfield(aa[i]); }
}

dofatt(aa,at)
struct SCField *aa[];
int at;
{
   int i;
   
   for (i=0; aa[i]!=NULL; i++) { aa[i]->attr=at; }
}

mergefields(aa1,aa2)
struct SCField *aa1[],*aa2[];
{
   int i,j;
   
   for (i=0; aa1[i]!=NULL; i++) { }
   for (j=0; aa2[j]!=NULL; i++,j++) { aa1[i]=aa2[j]; }
   aa1[i]=aa2[j];      /* copy the terminating null */
}

genwavbufs(W,bufs,samplerate,afmt)
char *W[];
char *bufs[];
int samplerate;
int afmt;
{
   int i,st,N;
   
   N=samplerate<<(afmt==16);
   for (i=0; W[i]!=NULL; i++) {
      if (bufs[i]!=NULL) free(bufs[i]);
      if ((bufs[i]=malloc(N))==NULL) return(ENOMEM);
      if ((st=generate(W[i],bufs[i],N,1,100,samplerate,-1,afmt))==0) 
	return(EINVAL);
   }
   return(0);
}

/* playsamples(fd)   check state of output sound buffers, if necessary
 *                generate next fragment of samples and write it away.
 * if fd < 0 then the offset pointers are resynched by setting to zero.
 */

playsamples(fd)
int fd;
{
   static int fr_size=0;
   static unsigned char *frag=NULL;
   static int of1,of2;   /* of1 for the sweeping, of2 for swept frequencies */

   short int v,*vp,*wv1,*wv2;
   unsigned char *p;
   int i,n,t1,t2,dev;
  
   if (fd<0) { of2=0; return(of1=0); }     /* reset pointers */ 
   if (fragsize!=fr_size) {
      if (fr_size) free(frag);
      if ((frag=malloc(fragsize))==NULL) return(-1);
      fr_size=fragsize; of1=0; of2=0;
   }

#define TO8BIT(V) (unsigned char)((V>>8)+128)
     
   /*      (afmt==AFMT_S16_LE) */
   /* It's all done in signed 16 bit format - because we need the 
    * precision for the sweeping waveform. We adjust down to 8 bit
    * if writing 8 bit samples on output.
    */
      
   dev=(freqF<freqT)?frdev:-frdev;
   for (;;) {
      if ((i=getfreeobufs(fd))<=0) return(i);
      
      wv1=(short int *)(wavbufs[Fwav1V.adj]);  /* sweeping waveform */
      wv2=(short int *)(wavbufs[Fwav2V.adj]);  /* swept waveform */

      for (i=0, vp=(short int *)(p=frag); i<fragsamplesize; i++) {
	 t1=(int)(wv1[of1]);
	 if (stereo) {
	    if (afmt==AFMT_S16_LE) *vp++=wv1[of1]; else *p++=TO8BIT(wv1[of1]);
	 }
	 of1=(of1+freq)%samplerate;
	 if (afmt==AFMT_S16_LE) *vp++=wv2[of2]; else *p++=TO8BIT(wv2[of2]);
	 of2=((int)samplerate+of2+(int)freqC+((t1*dev)/32768))%samplerate;
      }
      if (write(fd,frag,fragsize)<0) return(-1);
   }
   return(0);
}
