//===================================================================
//
// Uptime
//
// By Mark Russinovich
// Systems Internals
// http://www.sysinternals.com
//
// Uses the performance data helper (PDH) library to query a 
// system's uptime
//
//===================================================================
#include <windows.h>
#include <stdio.h>
#include <pdh.h>
#include <pdhmsg.h>

static char	localMachine[] = ".";
char		*computerName = localMachine;


//-------------------------------------------------------------------
//
// Usage
//
// Prints correct syntax.
//
//-------------------------------------------------------------------
void Usage( char *ProgramName )
{
	printf("usage: %s [computer name]\n\n", ProgramName );
}


//-------------------------------------------------------------------
//
// PdhError
//
// Handles the PDH error case.
//
//-------------------------------------------------------------------
void PdhError( PDH_STATUS Status ) 
{
	if( Status == PDH_CSTATUS_NO_MACHINE ) {

		printf("Computer %s was not found.\n\n", computerName );
		return;
	}

	if( strcmp( computerName, "." ) )
		printf("Unable to query uptime for %s\n\n", computerName );
	else
		printf("Unable to query uptime for this system\n\n");
}


//-------------------------------------------------------------------
//
// main
//
// Grab the uptime from the computer specified.
//
//-------------------------------------------------------------------
int main( int argc, char *argv[])
{
	PDH_STATUS  status;
	HQUERY		perfQuery;
	HCOUNTER	uptimeCounter;
	char		uptimeCounterPath[1024];
	PDH_FMT_COUNTERVALUE uptimeValue;
	DWORD		days, hours, minutes, seconds;

	//
	// Print banner
	//
	printf("\nPsUptime - system uptime utility for Windows NT/2K\n");
	printf("by Mark Russinovich\n");
	printf("Sysinternals - www.sysinternals.com\n\n");

	// 
	// Check for NT
	//
	if( GetVersion() >= 0x80000000 ) {

		printf("%s requres Windows NT/2K.\n\n", argv[0]);
		return -1;
	}

	//
	// Check arguments
	//
	if( argc > 2 ) {

		Usage( argv[0] );
		return -1;
	}

	if( argc == 2 ) {
	
		computerName = argv[1];
	}

	//
	// Create a PDH query
	//
	if( PdhOpenQuery( NULL, 0, &perfQuery ) != ERROR_SUCCESS ) {

		printf("Unable to create performance query.\n\n");
		return -1;
	}

	//
	// If its not local, tell the user about the delay
	//
	if( strcmp( computerName, "." )) {
		
		printf("Querying %s...", computerName );
		fflush( stdout );
	}

	//
	// Associate the uptime counter with the query
	//
	sprintf( uptimeCounterPath, "\\\\%s\\System\\System Up Time", computerName );

	status = PdhAddCounter( perfQuery, uptimeCounterPath,
							0, &uptimeCounter );
	if( status != ERROR_SUCCESS ) {

		PdhError( status );
		return -1;
	}

	//
	// Snapshot the counter value
	//
	status = PdhCollectQueryData( perfQuery );
	if( status != ERROR_SUCCESS ) {

		PdhError( status );
		return -1;
	}

	//
	// Tell them we're done
	//
	if( strcmp( computerName, "." )) {

		printf("done.\n\n");
	}

	//
	// Get the formatted counter value
	//
	status = PdhGetFormattedCounterValue( uptimeCounter, PDH_FMT_LARGE , NULL, &uptimeValue );
	if( status != ERROR_SUCCESS ) {

		PdhError( status );
		return -1;
	}

    //
    // Close the query
    //
    PdhCloseQuery( &perfQuery );

	//
	// Now we have the system boot time, convert it to readable form
	//
	days =  (DWORD) (uptimeValue.largeValue / (3600*24));
	hours = (DWORD) (uptimeValue.largeValue % (3600*24) / 3600);
	minutes = (DWORD) (uptimeValue.largeValue % 3600) / 60;
	seconds = (DWORD) (uptimeValue.largeValue % 60);
	
	printf("%s has been up for %d day%s, %d hour%s, %d minute%s, %d second%s.\n\n",
		strcmp( computerName, "." ) ? computerName : "This computer",
		days, days != 1 ? "s" : "",
		hours, hours != 1 ? "s" : "",
		minutes, minutes != 1 ? "s" : "",
		seconds, seconds != 1 ? "s" : "" );

	return 0;
}
