//
// Reghide.c
//
// by Mark Russinovich
// http://www.sysinternals.com
//
// This program demonstrates how the Native API can be used to 
// create object names that are inaccessible from the Win32 API. While
// there are many different ways to do this, the method used here it to
// include a terminating NULL that is explicitly made part of the key name.
// There is no way to describe this with the Win32 API, which treats a NULL
// as the end of the name string and will therefore chop it. Thus, Regedit
// and Regedt32 won't be able to access this key, though it will be visible.                
//
#include <windows.h>
#include <stdio.h>
#include "reghide.h"

//
// The name of the key and value that we're going to create
//
WCHAR KeyNameBuffer[]		= L"\\Registry\\Machine\\SOFTWARE";
WCHAR NewKeyNameBuffer[]	= L"Systems Internals";
WCHAR HiddenKeyNameBuffer[] = L"Can't touch me!\0";
WCHAR HiddenValueNameBuffer[]= L"Hidden Value";

//
// Convenience output routine
//
VOID Output( char *msg, DWORD Buttons )
{
	MessageBox( NULL, msg, "RegHide", Buttons );
}


//
// Loads and finds the entry points we need in NTDLL.DLL
//
VOID LocateNTDLLEntryPoints()
{
	if( !(NtCreateKey = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"NtCreateKey" )) ) {

		printf("Could not find NtCreateKey entry point in NTDLL.DLL\n");
		exit(1);
	}
	if( !(NtDeleteKey = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"NtDeleteKey" )) ) {

		printf("Could not find NtDeleteKey entry point in NTDLL.DLL\n");
		exit(1);
	}
	if( !(NtSetValueKey = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
			"NtSetValueKey" )) ) {

		printf("Could not find NtSetValueKey entry point in NTDLL.DLL\n");
		exit(1);
	}
}


//
// Create the key and value, tell the user to try to access it, and then delete it
// after they've tried
//
int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
						LPSTR lpCmdLine, int nCmdShow )
{
	UNICODE_STRING KeyName, ValueName;
	HANDLE SoftwareKeyHandle, SysKeyHandle, HiddenKeyHandle;
	ULONG Status;
	OBJECT_ATTRIBUTES ObjectAttributes;
	ULONG Disposition;
	char input;

	//
	// Load the entry points we need
	//
	LocateNTDLLEntryPoints();

	//
	// Print banner
	//
	Output( "\n                                              Reghide\n"
			"Creates a Registry key that cannot be opened with Regedit/Regedt32\n\n"
			"                                 by Mark Russinovich\n"
			"                             http://www.sysinternals.com\n", MB_OK );

	//
	// Open the Software key
	//
	KeyName.Buffer = KeyNameBuffer;
	KeyName.Length = wcslen( KeyNameBuffer ) *sizeof(WCHAR);
	InitializeObjectAttributes( &ObjectAttributes, &KeyName, 
			OBJ_CASE_INSENSITIVE, NULL, NULL );
	Status = NtCreateKey( &SoftwareKeyHandle, KEY_ALL_ACCESS, 
					&ObjectAttributes, 0,  NULL, REG_OPTION_NON_VOLATILE,
					&Disposition );
	if( !NT_SUCCESS( Status )) {

		Output("Error: Couldn't open HKLM\\Software\n", MB_ICONERROR );
		exit(1);
	}

	//
	// Create the Systems Internals key
	//
	KeyName.Buffer = NewKeyNameBuffer;
	KeyName.Length = wcslen( NewKeyNameBuffer ) *sizeof(WCHAR);
	InitializeObjectAttributes( &ObjectAttributes, &KeyName, 
			OBJ_CASE_INSENSITIVE, SoftwareKeyHandle, NULL );
	Status = NtCreateKey( &SysKeyHandle, KEY_ALL_ACCESS, 
					&ObjectAttributes, 0, NULL, REG_OPTION_NON_VOLATILE,
					&Disposition );
	if( !NT_SUCCESS( Status )) {

		Output("Error: Couldn't create HKLM\\Software\\Systems Internals\n",
				MB_ICONERROR);
		exit(1);
	}

	//
	// Create the Hidden key
	//
	KeyName.Buffer = HiddenKeyNameBuffer;
	// length here must include terminating null
	KeyName.Length = wcslen( HiddenKeyNameBuffer ) *sizeof(WCHAR) + sizeof(WCHAR);
	InitializeObjectAttributes( &ObjectAttributes, &KeyName, 
			OBJ_CASE_INSENSITIVE, SysKeyHandle, NULL );
	Status = NtCreateKey( &HiddenKeyHandle, KEY_ALL_ACCESS, 
					&ObjectAttributes, 0, NULL, REG_OPTION_NON_VOLATILE,
					&Disposition );
	if( !NT_SUCCESS( Status )) {

		Output("Error: Couldn't create HKLM\\Software\\Systems Internals\\RegHide\n",
				MB_ICONERROR);
		exit(1);
	}

	//
	// Create the hidden value
	//
	ValueName.Buffer = HiddenValueNameBuffer;
	ValueName.Length = wcslen( HiddenValueNameBuffer ) *sizeof(WCHAR);
	Status = NtSetValueKey( HiddenKeyHandle, &ValueName, 0, REG_SZ, 
						HiddenValueNameBuffer, 
						wcslen( HiddenValueNameBuffer ) * sizeof(WCHAR) );
	if( !NT_SUCCESS( Status )) {

		Output("Error: Couldn't create our hidden value\n", MB_ICONERROR);
		NtDeleteKey( HiddenKeyHandle );
		exit(1);
	}

	//
	// Let the user try and open our key!
	//
	Output("Try and open the key \"HKLM\\SOFTWARE\\Systems Internals\\Can't touch me!\"\n"
		"with Regedit or Regedt32 (or any other Registry editor). There is a value\n"
		"in the key called \"Hidden Value\".\n\n"
		"When done trying, press any key to have the key deleted and exit.\n",
		MB_ICONINFORMATION|MB_OK );

	input = getchar();

	//
	// Cleanup the key
	//
	NtDeleteKey( HiddenKeyHandle );
	return 0;

}