//============================================================================
//
// NTCrash2
//
// Copyright (C) 1997 Mark Russinovich
//
// Discovers holes in native APIs with faulty parameter validation.
//
//============================================================================
#include <windows.h>
#include <stdio.h>
#include <process.h>
#include <stdio.h>
#include <time.h>


#define TESTING 0

#define TRYSPERCALL		1000000
#define PARAMSPERTRY	15

// progress indicator
char progress[] = { '-', '\\', '|', '/', '-', '\\', '|', '/' };

// log file handle
HANDLE	logfile;

// buffer that will go to the log file
char	logrec[4096];

//
// bad call array - pre-initialized with calls that we can't get past
//
int totbad = 0;
int maxbad = -1;
int badcalls[512];

// log the calls to a file?
BOOL	logcalls = FALSE;

// which table to exercise?
// 0 = ntoskrnl
// 1000 = win2k
// 2000 = spud
DWORD   table = 0;

// buffer 
char	buffer[1024];


//============================================================================
//
// scanlog
//
// Reads a permanetn log file to obtain system calls that are known to fail,
// and then appends the last call from crash.log, since that was the last
// executed before the last failure.
//
//============================================================================
void scanlog()
{
	FILE	*fp;
	int		badnumber, bw, len;
	char	badline[1024], nextline[1024];
	HANDLE	badfile;

	// read bad call file 
    switch( table ) {
    case 0:	fp = fopen( "ntoskrnl.log", "r" ); break;
    case 1: fp = fopen( "win32k.log", "r" ); break;
    case 2: fp = fopen( "spud.log", "r" ); break;
    }
	if( fp ) {
        switch( table ) {
        case 0:	printf("Scanning ntoskrnl.log\n" ); break;
        case 1:	printf("Scanning win2k.log\n" ); break;
        case 2:	printf("Scanning spud.log\n" ); break;
        }
		while( fgets( badline, 1024, fp ) ) {
			if( strlen( badline ) > 2 ) {
				sscanf( badline, "0x%x", &badnumber );
				badcalls[ totbad++] = badnumber;
				if( badnumber > maxbad ) maxbad = badnumber;
				printf("  Excluding 0x%x\n", badnumber );
			}
		}
		fclose( fp );
	}

	// now get the latest from the crash log
    switch( table ) {
    case 0:	fp = fopen( "crashn.log" , "r" ); break;
    case 1: fp = fopen( "crashw.log" , "r" ); break;
    case 2: fp = fopen( "crashs.log" , "r" ); break;
    }
	if( fp ) {

		// scan the file to get the last line recorded
		badline[0] = 0;
        switch( table ) {
        case 0:	printf("Scanning crashn.log\n" ); break;
        case 1:	printf("Scanning crashw.log\n" ); break;
        case 2:	printf("Scanning crashs.log\n" ); break;
        }
		while( fgets( nextline, 1024, fp )) {
			if( strlen( nextline ) > 2 )
				strcpy(badline, nextline);
		}
		fclose( fp );

		// were there any records in the crash log?
		if( badline[0] ) {

			// extract the failed call number
			sscanf( badline, "0x%x", &badnumber );
			badcalls[ totbad++] = badnumber;
			if( badnumber > maxbad ) maxbad = badnumber;
			printf("  Excluding 0x%x\n", badnumber );
			
			// add it to the bad call file
            switch( table ) {
            case 0:
				badfile = CreateFile("ntoskrnl.log", GENERIC_WRITE, 0, NULL, 
						OPEN_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
                break;
            case 1:
				badfile = CreateFile("win32k.log", GENERIC_WRITE, 0, NULL, 
						OPEN_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
                break;
            case 2:
				badfile = CreateFile("spud.log", GENERIC_WRITE, 0, NULL, 
						OPEN_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
                break;
            }

			// fix the carriage return
			len = strlen( badline );
			badline[ len-1 ] = '\r';
			badline[ len   ] = '\n';
			badline[ len+1 ] = 0;
			SetFilePointer( badfile, 0, 0, FILE_END );
			WriteFile( badfile, badline, strlen(badline), &bw, 0);
			FlushFileBuffers(badfile); // just to be safe
			CloseHandle( badfile );
		}
	}	


	// create a new log
    switch( table ) {
    case 0:
		DeleteFile("crashn.log");
		logfile = CreateFile("crashn.log", GENERIC_WRITE, 0, NULL, 
				CREATE_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
        break;
    case 1:
		DeleteFile("crashw.log");
		logfile = CreateFile("crashw.log", GENERIC_WRITE, 0, NULL, 
				CREATE_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
        break;
    case 2:
		DeleteFile("crashs.log");
		logfile = CreateFile("crashs.log", GENERIC_WRITE, 0, NULL, 
				CREATE_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
        break;
	}
	CloseHandle(logfile);
}


//============================================================================
//
// logrecord
//
// Synchronously writes an entry into the log file.
//
//============================================================================
void logrecord( int callnumber )
{
	int		br;

	// just to be safe
	memset( logrec, 0, 4096);
	sprintf( logrec, "0x%x \r\n", callnumber );
    switch( table ) {
    case 0:
#if TRUNCATE
		logfile = CreateFile("crashn.log", GENERIC_WRITE, 0, NULL, 
			CREATE_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
#else
		logfile = CreateFile("crashn.log", GENERIC_WRITE, 0, NULL, 
			OPEN_EXISTING, FILE_FLAG_WRITE_THROUGH, 0 );
		SetFilePointer( logfile, 0, 0, FILE_END );
#endif
        break;
    case 1:
#if TRUNCATE
		logfile = CreateFile("crashw.log", GENERIC_WRITE, 0, NULL, 
			CREATE_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
#else
		logfile = CreateFile("crashw.log", GENERIC_WRITE, 0, NULL, 
			OPEN_EXISTING, FILE_FLAG_WRITE_THROUGH, 0 );
		SetFilePointer( logfile, 0, 0, FILE_END );
#endif
        break;
    case 2:
#if TRUNCATE
		logfile = CreateFile("crashs.log", GENERIC_WRITE, 0, NULL, 
			CREATE_ALWAYS, FILE_FLAG_WRITE_THROUGH, 0 );
#else
		logfile = CreateFile("crashs.log", GENERIC_WRITE, 0, NULL, 
			OPEN_EXISTING, FILE_FLAG_WRITE_THROUGH, 0 );
		SetFilePointer( logfile, 0, 0, FILE_END );
#endif
        break;
	}
	WriteFile( logfile, logrec, strlen(logrec), &br, NULL);
	FlushFileBuffers(logfile); // just to be safe
	CloseHandle(logfile);
}


//============================================================================
//
// threadloop
//
// Just spin while firing random calls pointing at garbage into Win32k.sys
//
//============================================================================
void threadloop( void *param )
{
	int		loop = 0;
	int		i;
	int		callnumber;
	int		thrdnum = (int) param;
	long	stack[100];
	long	*stackptr;
	int		trys;
    int     numsyscalls;
	
	// start firing at 1+the highest bad call number we know of
	if( maxbad >= 0x1000) maxbad -= 0x1000;
    switch( table ) {
    case 0: numsyscalls = 0x220; break;
    case 1: numsyscalls = 0xF0; break;
    case 2: numsyscalls = 10; break;
    }
	for( loop = maxbad+1; loop < numsyscalls; loop++) {

		// if its win32, adjust the call number
        switch( table ) {
        case 0: callnumber = loop; break;
        case 1:	callnumber = loop + 0x1000; break;
        case 2: callnumber = loop + 0x2000; break;
        }

		if( logcalls ) {

			// save record to log file
			logrecord( callnumber );
		}

		if( loop != maxbad + 1) 
			printf("\b.\n0x%x../", callnumber);
		else
			printf("0x%x../", callnumber );

		// play with the seed
		srand((unsigned)time( NULL ));
		fflush(stdout);
	
		// go for it
		for( trys = 0 ; trys < TRYSPERCALL; trys++ ) { 

			if( !(trys %10000 )) {
				
				printf("\b%c", progress[ (trys/10000) % 8 ] );
			}

			// push params
			for( i = 0 ; i < PARAMSPERTRY; i++) {
				
				switch( rand() % 10 ) {
				case 0:
					stack[i] = 0;
					break;
				case 1:
					stack[i] = 0x80000000;
					break;
				case 2:
					stack[i] = -1;
					break;
				case 3:
					stack[i] = 0x7fffffff;
					break;
				case 4:
					stack[i] = 0xFFFFFFF0;
					break;
				case 5:
					stack[i] = (long) buffer;
					break;
				case 6:
					stack[i] = 0x70000001;
					break;
				case 7:
					stack[i] = 0xFFFFFFFC;
					break;
				case 8:
					stack[i] = 0x70000004;
					break;
				case 9:
					stack[i] = loop % 64;
					break;
				}
			}

			// spud has a stupid hack to "protect" itself
			if( table == 2 && callnumber == 0x2000 ) {
				stack[0] = 0x2000;
				OutputDebugString("*\n");
			}
				
			__try {
#if !TESTING
					// do the systrap
					stackptr = stack;
					_asm mov eax, callnumber;
					_asm mov edx, stackptr;
					_asm int 2eh;
#endif
			} __except( EXCEPTION_EXECUTE_HANDLER ) {

			} 
		}
	}
}

//============================================================================
//
// usage
//
//============================================================================
void usage( char *name )
{
	printf("usage: %s [-l] [-n]\n"
			"-l		Log to stable file\n"
			"-w		Exercise WIN32K\n", name );
	exit(1);
}


//============================================================================
//
// main
//
// If logging, saves info to a log file. If not, fire off a bunch of threads
// to crash the system real quick-like.
//
//============================================================================
void main( int argc, char *argv[])
{
	int i;
	printf("\nNTCrash2\n");
	printf("Copyright (C) 1997 Mark Russinovich\n");
	printf("http://www.ntinternals.com\n\n");

	for( i = 1; i < argc; i ++ ) {
		if( argv[i][0] != '-' )
			usage( argv[0] );
		switch( argv[i][1] ) {
		case 'l':
		case 'L':
			logcalls = TRUE;
			printf("Logging calls enabled\n" );
			break;
		case 'w':
		case 'W':
			table = 1;
			break;
        case 'S':
        case 's':
			table = 2;
            break;
		default:
			usage(argv[0]);
		}
	}

    switch( table ) {
    case 0:	printf("Exercising NTOSKRNL\n"); break;
    case 1: printf("Exercising WIN2K\n"); break;
    case 2: printf("Exercising SPUD\n"); break;
    }
	printf("\n");	
	
	// Do logging setup
	if( logcalls ) {
		scanlog();
	}

	printf("\nRunning stress test. Press Ctrl-C to quit.\n");

	threadloop( (void *) 0 );
}
