//======================================================================
// 
// Filemon.c
//
// Copyright (C) 1996-1998 Mark Russinovich and Bryce Cogswell
//
// Implements the equivalent of Windows 95 IFSMgr_FileSystemApiHook.
//
//======================================================================
#include "ntddk.h"
#include "stdarg.h"
#include "stdio.h"
#include "ioctlcmd.h"
#include "filemon.h"
#include "winioctl.h"

#include "memtrack.h"

//----------------------------------------------------------------------
//                           DEFINES
//----------------------------------------------------------------------

//
// Set this to 0 to make a processor-independent version that does
// not print out the process name
//
#define GETPROCESS 1

//
// Set this to 0 so that this driver can be built with pre-NT 5.0
// versions of the SDK
//
#define NT5_IOCTLS 1

//
// Print macro that only turns on when debugging is on
//
#if DBG
#define DbgPrint(arg) DbgPrint arg
#else
#define DbgPrint(arg) 
#endif

//
// The name of the System process, in which context we're called in our DriverEntry
//
#define SYSNAME    "System"


//----------------------------------------------------------------------
//                          TYPEDEFS
//----------------------------------------------------------------------

// 
// Directory control structure
//
typedef struct {
    ULONG Length;
    PUNICODE_STRING FileName;
    FILE_INFORMATION_CLASS FileInformationClass;
    ULONG FileIndex;
} QUERY_DIRECTORY, *PQUERY_DIRECTORY;

//
// Lock control data structure
//
typedef struct {
    PLARGE_INTEGER Length;
    ULONG Key;
    LARGE_INTEGER ByteOffset;
} LOCK_CONTROL, *PLOCK_CONTROL;

//----------------------------------------------------------------------
//                         FORWARD DEFINES
//---------------------------------------------------------------------- 

NTSTATUS FilemonDispatch( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp );
VOID     FilemonUnload( IN PDRIVER_OBJECT DriverObject );

//
// These are prototypes for Filemon's Fast I/O hooks. The originals 
// prototypes can be found in NTDDK.H
// 
BOOLEAN  FilemonFastIoCheckifPossible( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
                                       IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN BOOLEAN CheckForReadOperation,
                                       OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoRead( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
                            IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, OUT PVOID Buffer,
                            OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoWrite( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
                             IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN PVOID Buffer,
                             OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoQueryBasicInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
                                      OUT PFILE_BASIC_INFORMATION Buffer,
                                      OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoQueryStandardInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
                                         OUT PFILE_STANDARD_INFORMATION Buffer,
                                         OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoLock( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
                            IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
                            BOOLEAN FailImmediately, BOOLEAN ExclusiveLock,
                            OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoUnlockSingle( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
                                    IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
                                    OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoUnlockAll( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId,
                                 OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoUnlockAllByKey( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId, ULONG Key,
                                      OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoDeviceControl( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait,
                                     IN PVOID InputBuffer, IN ULONG InputBufferLength, 
                                     OUT PVOID OutbufBuffer, IN ULONG OutputBufferLength, IN ULONG IoControlCode,
                                     OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
VOID     FilemonFastIoAcquireFile( PFILE_OBJECT FileObject );
VOID     FilemonFastIoReleaseFile( PFILE_OBJECT FileObject );
VOID     FilemonFastIoDetachDevice( PDEVICE_OBJECT SourceDevice, PDEVICE_OBJECT TargetDevice );

//
// These are new NT 4.0 Fast I/O calls
//
BOOLEAN  FilemonFastIoQueryNetworkOpenInfo(IN PFILE_OBJECT FileObject,
                                           IN BOOLEAN Wait, OUT struct _FILE_NETWORK_OPEN_INFORMATION *Buffer,
                                           OUT struct _IO_STATUS_BLOCK *IoStatus, IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoAcquireForModWrite( IN PFILE_OBJECT FileObject,
                                          IN PLARGE_INTEGER EndingOffset, OUT struct _ERESOURCE **ResourceToRelease,
                                          IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlRead( IN PFILE_OBJECT FileObject,
                               IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                               IN ULONG LockKey, OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
                               IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlReadComplete( IN PFILE_OBJECT FileObject,
                                       IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoPrepareMdlWrite( IN PFILE_OBJECT FileObject,
                                       IN PLARGE_INTEGER FileOffset, IN ULONG Length, IN ULONG LockKey,
                                       OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
                                       IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlWriteComplete( IN PFILE_OBJECT FileObject,
                                        IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain,
                                        IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoReadCompressed( IN PFILE_OBJECT FileObject,
                                      IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                                      IN ULONG LockKey, OUT PVOID Buffer, OUT PMDL *MdlChain,
                                      OUT PIO_STATUS_BLOCK IoStatus,
                                      OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
                                      IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoWriteCompressed( IN PFILE_OBJECT FileObject,
                                       IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                                       IN ULONG LockKey, IN PVOID Buffer, OUT PMDL *MdlChain,
                                       OUT PIO_STATUS_BLOCK IoStatus,
                                       IN struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
                                       IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlReadCompleteCompressed( IN PFILE_OBJECT FileObject,
                                                 IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlWriteCompleteCompressed( IN PFILE_OBJECT FileObject,
                                                  IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain,
                                                  IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoQueryOpen( IN struct _IRP *Irp,
                                 OUT PFILE_NETWORK_OPEN_INFORMATION NetworkInformation,
                                 IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoReleaseForModWrite( IN PFILE_OBJECT FileObject,
                                          IN struct _ERESOURCE *ResourceToRelease, IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoAcquireForCcFlush( IN PFILE_OBJECT FileObject,
                                         IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoReleaseForCcFlush( IN PFILE_OBJECT FileObject,
                                         IN PDEVICE_OBJECT DeviceObject );

//----------------------------------------------------------------------
//                         GLOBALS
//---------------------------------------------------------------------- 

//
// This is Filemon's user-inteface device object. It is addressed
// by calls from the GUI including CreateFile and DeviceIoControl
//
PDEVICE_OBJECT      GUIDevice;

//
// Indicates if the GUI wants activity to be logged
//
BOOLEAN             FilterOn = FALSE;

//
// Global filter (sent to us by the GUI)
//
FILTER              FilterDef;

//
// This lock protects access to the filter array
//
KSPIN_LOCK          FilterMutex;

//
// Array of process and path filters 
//
ULONG               NumProcessFilters = 0;
PCHAR               ProcessFilters[MAXFILTERS];
ULONG               NumProcessExcludeFilters = 0;
PCHAR               ProcessExcludeFilters[MAXFILTERS];
ULONG               NumPathIncludeFilters = 0;
PCHAR               PathIncludeFilters[MAXFILTERS];
ULONG               NumPathExcludeFilters = 0;
PCHAR               PathExcludeFilters[MAXFILTERS];

//
// Which type of timing are we doing (told to us by GUI)
//
BOOLEAN             TimeIsDuration;

//
// Once a load is initiated, this flag prevents the processing of
// further IRPs. This is required because an unload can only take
// place if there are any IRP's for which an IoCompletion has
// been registered that has not actually completed.
//
BOOLEAN             UnloadInProgress = FALSE;

//
// This is the offset into a KPEB of the current process name. This is determined
// dynamically by scanning the process block belonging to the GUI for the name
// of the system process, in who's context we execute in DriverEntry
//
ULONG               ProcessNameOffset;

//
// This variable keeps track of the outstanding IRPs (ones for which
// a completion routine has been registered, but that have not yet
// passed through the completion routine), which is used in 
// the unload determination logic. The CountMutex protects data
// races on updating the count.
//
ULONG               OutstandingIRPCount = 0;
KSPIN_LOCK          CountMutex;

//
// Table of our hook devices for each drive letter. This makes it
// easy to look up the device object that was created to hook a 
// particular drive.
//
PDEVICE_OBJECT      LDriveDevices[26];

//
// This table keeps track of which drives are controlled by common
// network redirectors (unhooking one results in all of the group
// being unhooked). Each device object that is hooked is assigned
// a unique identifier. Network devices are used to represent 
// multiple logical drives, so the drive map entries for network
// drive handled by the same network device object have the
// same identifier.
//
int                 LDriveMap[26];
int                 LDriveGroup = 0;

//
// Hash table for keeping names around. This is necessary because 
// at any time the name information in the fileobjects that we
// see can be deallocated and reused. If we want to print accurate
// names, we need to keep them around ourselves. 
//
PHASH_ENTRY	    HashTable[NUMHASH];

//
// Mutex to protect hash table.
//
KSPIN_LOCK	    HashMutex;

//
// The current output buffer
//
PSTORE_BUF          Store = NULL;

//
// Each IRP is given a sequence number. This allows the return status
// of an IRP, which is obtained in the completion routine, to be 
// associated with the IRPs parameters that were extracted in the Dispatch
// routine.
//
ULONG               Sequence = 0;

//
// This mutex protects the output buffer
//
KSPIN_LOCK          StoreMutex;

//
// Filemon keeps track of the number of distinct output buffers that
// have been allocated, but not yet uploaded to the GUI, and caps
// the amount of memory (which is in non-paged pool) it takes at
// 1MB.
//
ULONG               NumStore = 0;
ULONG               MaxStore = 1000000/MAX_STORE;

//
// Lookaside lists. One for hash entries and one for pathnames
//
NPAGED_LOOKASIDE_LIST  HashLookaside;
NPAGED_LOOKASIDE_LIST  PathLookaside;

//
// These are the text representations of the classes of IRP_MJ_SET/GET_INFORMATION
// calls
//
CHAR    *FileInformation[] = {
    "",
    "FileDirectoryInformation",
    "FileFullDirectoryInformation",
    "FileBothDirectoryInformation",
    "FileBasicInformation",
    "FileStandardInformation",
    "FileInternalInformation",
    "FileEaInformation",
    "FileAccessInformation",
    "FileNameInformation",
    "FileRenameInformation",
    "FileLinkInformation",
    "FileNamesInformation",
    "FileDispositionInformation",
    "FilePositionInformation",
    "FileFullEaInformation",
    "FileModeInformation",
    "FileAlignmentInformation",
    "FileAllInformation",
    "FileAllocationInformation",
    "FileEndOfFileInformation",
    "FileAlternateNameInformation",
    "FileStreamInformation",
    "FilePipeInformation",
    "FilePipeLocalInformation",
    "FilePipeRemoteInformation",
    "FileMailslotQueryInformation",
    "FileMailslotSetInformation",
    "FileCompressionInformation",
    "FileCopyOnWriteInformation",
    "FileCompletionInformation",
    "FileMoveClusterInformation",
    "FileOleClassIdInformation",
    "FileOleStateBitsInformation",
    "FileNetworkOpenInformation",
    "FileObjectIdInformation",
    "FileOleAllInformation",
    "FileOleDirectoryInformation",
    "FileContentIndexInformation",
    "FileInheritContentIndexInformation",
    "FileOleInformation",
    "FileMaximumInformation",
};


//
// These are textual representations of the IRP_MJ_SET/GET_VOLUME_INFORMATION
// classes
//
CHAR *VolumeInformation[] = {
    "",
    "FileFsVolumeInformation",
    "FileFsLabelInformation",
    "FileFsSizeInformation",
    "FileFsDeviceInformation",
    "FileFsAttributeInformation",
    "FileFsQuotaQueryInformation",
    "FileFsQuotaSetInformation",
    "FileFsControlQueryInformation",
    "FileFsControlSetInformation",
    "FileFsMaximumInformation",
};
    
//
// This Filemon's Fast I/O dispatch table. Note that NT assumes that
// file system drivers support some Fast I/O calls, so this table must
// be present for an file system filter driver
//
FAST_IO_DISPATCH    FastIOHook = {
    sizeof(FAST_IO_DISPATCH), 
    FilemonFastIoCheckifPossible,
    FilemonFastIoRead,
    FilemonFastIoWrite,
    FilemonFastIoQueryBasicInfo,
    FilemonFastIoQueryStandardInfo,
    FilemonFastIoLock,
    FilemonFastIoUnlockSingle,
    FilemonFastIoUnlockAll,
    FilemonFastIoUnlockAllByKey,
    FilemonFastIoDeviceControl,
    FilemonFastIoAcquireFile,
    FilemonFastIoReleaseFile,
    FilemonFastIoDetachDevice,

    //
    // new for NT 4.0
    //
    FilemonFastIoQueryNetworkOpenInfo,
    FilemonFastIoAcquireForModWrite,
    FilemonFastIoMdlRead,
    FilemonFastIoMdlReadComplete,
    FilemonFastIoPrepareMdlWrite,
    FilemonFastIoMdlWriteComplete,
    FilemonFastIoReadCompressed,
    FilemonFastIoWriteCompressed,
    FilemonFastIoMdlReadCompleteCompressed,
    FilemonFastIoMdlWriteCompleteCompressed,
    FilemonFastIoQueryOpen,
    FilemonFastIoReleaseForModWrite,
    FilemonFastIoAcquireForCcFlush,
    FilemonFastIoReleaseForCcFlush
};


//----------------------------------------------------------------------
//      P A T T E R N   M A T C H I N G   R O U T I N E S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// MatchOkay
//
// Only thing left after compare is more mask. This routine makes
// sure that its a valid wild card ending so that its really a match.
//
//----------------------------------------------------------------------
BOOLEAN MatchOkay( PCHAR Pattern )
{
    //
    // If pattern isn't empty, it must be a wildcard
    //
    if( *Pattern && *Pattern != '*' ) {
 
        return FALSE;
    }

    //
    // Matched
    //
    return TRUE;
}


//----------------------------------------------------------------------
//
// MatchWithPatternCore
//
// Performs nifty wildcard comparison.
//
//----------------------------------------------------------------------
BOOLEAN MatchWithPatternCore( PCHAR Pattern, PCHAR Name )
{
    //
    // End of pattern?
    //
    if( !*Pattern ) {

        return FALSE;
    }

    //
    // If we hit a wild card, do recursion
    //
    if( *Pattern == '*' ) {

        Pattern++;

        while( *Name && *Pattern ) {

            //
            // See if this substring matches
            //
            if( *Pattern == *Name || *Name == '*' ) {

                if( MatchWithPatternCore( Pattern+1, Name+1 )) {

                    return TRUE;
                }
            }

            //
            // Try the next substring
            //
            Name++;
        }

        //
        // See if match condition was met
        //
        return MatchOkay( Pattern );
    } 

    //
    // Do straight compare until we hit a wild card
    //
    while( *Name && *Pattern != '*' ) {

        if( *Pattern == *Name ) {

            Pattern++;
            Name++;

        } else {

            return FALSE;
        }
    }

    //
    // If not done, recurse
    //
    if( *Name ) {

        return MatchWithPatternCore( Pattern, Name );
    }

    //
    // Make sure its a match
    //
    return MatchOkay( Pattern );
}


//----------------------------------------------------------------------
//
// MatchWithPattern
//
// Converts strings to upper-case before calling core comparison routine.
//
//----------------------------------------------------------------------
BOOLEAN MatchWithPattern( PCHAR Pattern, PCHAR Name )
{
    PCHAR      Upname;
    int        i;
    BOOLEAN    isMatch;

    //
    // Allocate space for up-cased version of the name
    //
    Upname = MemAllocatePool( NonPagedPool, strlen( Name ) + 1);
    if ( !Upname ) return FALSE;
    //
    // Make the name upcased
    //
    i = 0;
    while( *Name ) {
        if( *Name >= 'a' && *Name <= 'z' )
            Upname[i++] = *Name - 'a' + 'A';
        else
            Upname[i++] = *Name;
        Name++;
    }
    Upname[i] = 0;

    //
    // Now do the comparison
    //
    isMatch = MatchWithPatternCore( Pattern, Upname );
    MemFreePool( Upname );
    return isMatch;
}


//----------------------------------------------------------------------
//            B U F F E R   M A N A G E M E N T
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// FilemonFreeStore
//
// Frees all the data output buffers that we have currently allocated.
//
//----------------------------------------------------------------------
VOID FilemonFreeStore()
{
    PSTORE_BUF  prev;
    
    //
    // Just traverse the list of allocated output buffers
    //

    while( Store ) {
        prev = Store->Next;
        MemFreePool( Store );
        Store = prev;
    }
}   


//----------------------------------------------------------------------
//
// FilemonNewStore
//
// Called when the current buffer has filled up. This allocates a new
// buffer and stick it at the head (newest) entry of our buffer list.
//
//----------------------------------------------------------------------
void FilemonNewStore( void )
{
    PSTORE_BUF prev = Store, newstore;

    //
    // If we've already allocated the allowed number of buffers, just
    // reuse the current one. This will result in output records being
    // lost, but it takes ALOT of file system activity to cause this.
    //
    if( MaxStore == NumStore ) {

        Store->Len = 0;
        return; 
    }

    //
    // If the output buffer we currently are using is empty, just
    // use it.
    //
    if( !Store->Len ) 
        return;

    //
    // Allocate a new output buffer
    //
    newstore = MemAllocatePool( NonPagedPool, sizeof(*Store) );
    if( newstore ) { 

        //
        // Allocation was successful so add the buffer to the list
        // of allocated buffers and increment the buffer count.
        //
        Store   = newstore;
        Store->Len  = 0;
        Store->Next = prev;
        NumStore++;

    } else {

        //
        // The allocation failed - just reuse the current buffer
        //
        Store->Len = 0;

    }
}


//----------------------------------------------------------------------
//
// FilemonOldestStore
//
// Traverse the list of allocated buffers to find the last one, which
// will be the oldest (as we want to return the oldest data to the GUI
// first).
//
//----------------------------------------------------------------------
PSTORE_BUF FilemonOldestStore( void )
{
    PSTORE_BUF  ptr = Store, prev = NULL;

    //
    // Traverse the list
    //    
    while ( ptr->Next ) {

        ptr = (prev = ptr)->Next;
    }

    //
    // Remove the buffer from the list
    //
    if ( prev ) {

        prev->Next = NULL;    
    }
    NumStore--;
    return ptr;
}


//----------------------------------------------------------------------
//
// FilemonResetStore
//
// When a GUI instance has close communication (exited), but the driver
// can't unload due to oustdanding IRPs, all the output buffers except
// one are all deallocated so that the memory footprint is shrunk as much 
// as possible.
//
//----------------------------------------------------------------------
VOID FilemonResetStore()
{
    PSTORE_BUF  current, next;

    //
    // Traverse the list of output buffers
    //
    current = Store->Next;
    while( current ) {

        //
        // Free the buffer
        //
        next = current->Next;

        MemFreePool( current );

        current = next;
    }

    // 
    // Move the output pointer in the buffer that's being kept
    // the start of the buffer.
    // 
    Store->Len = 0;
    Store->Next = NULL;
}


//----------------------------------------------------------------------
//
// UpdateStore
//
// This "printfs" a string into an output buffer.
//
//----------------------------------------------------------------------
void UpdateStore( ULONG seq, PLARGE_INTEGER time, const CHAR * format, ... ) 
{   
    PENTRY             Entry;
    int                len;
    va_list            arg_ptr;
    KIRQL              oldirql;
    PCHAR              text;

    //
    // If no GUI is there to receive the output or if no filtering is desired, don't bother
    //     
    if( !FilterOn ) {
     
        return;
    }

    //
    // Allocate a buffer
    //
    text = MemAllocateFromNPagedLookasideList( &PathLookaside );
    if( !text ) {

        return;
    }

    //
    // Send text out as debug output  This is x86 specific.
    //      
#define A (&format)
    DbgPrint(( (char *)format, A[1], A[2], A[3], A[4], A[5], A[6] ));
    DbgPrint(( "\n" ));
#undef A

    // 
    // Lock the output buffer.
    // 
    KeAcquireSpinLock( &StoreMutex, &oldirql );

    //
    // Vsprintf to determine the length of the buffer
    //
    va_start( arg_ptr, format );
    len = vsprintf( text, format, arg_ptr );
    va_end( arg_ptr );
    
    //
    // ULONG align for Alpha
    //
    len += 4; len &=  0xFFFFFFFC; // +1 to include null terminator and +3 to allign on 32 bit

    //
    // If the current output buffer is near capacity, move to a new 
    // output buffer
    //
    if ( Store->Len + len + sizeof(ENTRY) +1 >= MAX_STORE ) {

        FilemonNewStore();
    }

    //
    // Extract the sequence number and store it
    //
    Entry = (void *)(Store->Data+Store->Len);
    Entry->seq = seq;
    Entry->time = *time;

    memcpy( Entry->text, text, len );
 
    //
    // Store the length of the string, plus 1 for the terminating
    // NULL  
    //   
    Store->Len += (Entry->text - (PCHAR) Entry ) + len;

    //
    // Release the output buffer lock
    //
    KeReleaseSpinLock( &StoreMutex, oldirql );

    //
    // Free the buffer
    //
    MemFreePool( text );
}

//----------------------------------------------------------------------
//       H A S H   T A B L E   M A N A G E M E N T
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// FilemonHashCleanup
//
// Called when we are unloading to free any memory that we have 
// in our possession.
//
//----------------------------------------------------------------------
VOID FilemonHashCleanup()
{
    PHASH_ENTRY		hashEntry, nextEntry;
    ULONG		i;

    //
    // First, free the hash table entries
    //
    for( i = 0; i < NUMHASH; i++ ) {

        hashEntry = HashTable[i];

        while( hashEntry ) {
            nextEntry = hashEntry->Next;
            MemFreePool( hashEntry->FullPathName );
            MemFreeToNPagedLookasideList( &HashLookaside, hashEntry );
            hashEntry = nextEntry;
        }

        HashTable[i] = NULL;
    }
}


//----------------------------------------------------------------------
//
// FilemonFreeHashEntry
//
// When we see a file close, we can free the string we had associated
// with the fileobject being closed since we know it won't be used
// again.
//
//----------------------------------------------------------------------
VOID FilemonFreeHashEntry( PFILE_OBJECT fileObject )
{
    PHASH_ENTRY		hashEntry, prevEntry;
    KIRQL                oldirql;

    KeAcquireSpinLock( &HashMutex, &oldirql );

    //
    // Look-up the entry
    //
    hashEntry = HashTable[ HASHOBJECT( fileObject ) ];
    prevEntry = NULL;

    while( hashEntry && hashEntry->FileObject != fileObject ) {

        prevEntry = hashEntry;
        hashEntry = hashEntry->Next;
    }
 
    //  
    // If we fall of the hash list without finding what we're looking
    // for, just return.
    //
    if( !hashEntry ) {

        KeReleaseSpinLock( &HashMutex, oldirql );
        return;
    }

    //
    // Got it! Remove it from the list
    //
    if( prevEntry ) {

        prevEntry->Next = hashEntry->Next;

    } else {

        HashTable[ HASHOBJECT( fileObject )] = hashEntry->Next;
    }

    //
    // Free the memory associated with the name of the free entry.
    //
    MemFreePool( hashEntry->FullPathName );
    MemFreeToNPagedLookasideList( &HashLookaside, hashEntry );

    KeReleaseSpinLock( &HashMutex, oldirql );
}

//----------------------------------------------------------------------
//       P A T H  A N D  P R O C E S S  N A M E  R O U T I N E S
//----------------------------------------------------------------------


//----------------------------------------------------------------------
//
// FilemonFreeFilters
//
// Fress storage we allocated for filter strings.
//
//----------------------------------------------------------------------
VOID FilemonFreeFilters()
{
    ULONG   i;
    
    for( i = 0; i < NumProcessFilters; i++ ) {

        MemFreePool( ProcessFilters[i] );
    }
    for( i = 0; i < NumProcessExcludeFilters; i++ ) {

        MemFreePool( ProcessExcludeFilters[i] );
    }
    for( i = 0; i < NumPathIncludeFilters; i++ ) {

        MemFreePool( PathIncludeFilters[i] );
    }
    for( i = 0; i < NumPathExcludeFilters; i++ ) {

        MemFreePool( PathExcludeFilters[i] );
    }
    NumProcessFilters = 0;
    NumProcessExcludeFilters = 0;
    NumPathIncludeFilters = 0;
    NumPathExcludeFilters = 0;
}

//----------------------------------------------------------------------
//
// MakeFilterArray
//
// Takes a filter string and splits into components (a component
// is seperated with a ';')
//
//----------------------------------------------------------------------
VOID MakeFilterArray( PCHAR FilterString,
                      PCHAR FilterArray[],
                      PULONG NumFilters )
{
    PCHAR filterStart;
    ULONG filterLength;

    //
    // Scan through the process filters
    //
    filterStart = FilterString;
    while( *filterStart ) {

        filterLength = 0;
        while( filterStart[filterLength] &&
               filterStart[filterLength] != ';' ) {

            filterLength++;
        }

        //
        // Ignore zero-length components
        //
        if( filterLength ) {

            FilterArray[ *NumFilters ] = 
                MemAllocatePool( NonPagedPool, filterLength + 1  );
            strncpy( FilterArray[ *NumFilters ],
                     filterStart, filterLength );
            FilterArray[ *NumFilters ][filterLength] = 0;
            (*NumFilters)++;
        }
    
        //
        // Are we done?
        //
        if( !filterStart[filterLength] ) break;

        //
        // Move to the next component (skip over ';')
        //
        filterStart += filterLength + 1;
    }
}


//----------------------------------------------------------------------
//
// FilemonUpdateFilters
//
// Takes a new filter specification and updates the filter
// arrays with them.
//
//----------------------------------------------------------------------
VOID FilemonUpdateFilters()
{
    KIRQL  oldirql;

    //
    // Free old filters (if any)
    //
    KeAcquireSpinLock( &FilterMutex, &oldirql );
    FilemonFreeFilters();

    //
    // Create new filter arrays
    //
    MakeFilterArray( FilterDef.processfilter,
                     ProcessFilters, &NumProcessFilters );
    MakeFilterArray( FilterDef.excludeprocess,
                     ProcessExcludeFilters, &NumProcessExcludeFilters );
    MakeFilterArray( FilterDef.pathfilter,
                     PathIncludeFilters, &NumPathIncludeFilters );
    MakeFilterArray( FilterDef.excludefilter,
                     PathExcludeFilters, &NumPathExcludeFilters );    

    KeReleaseSpinLock( &FilterMutex, oldirql );
}


//----------------------------------------------------------------------
//
// ApplyNameFilter
//
// If the name matches the exclusion mask, we do not log it. Else if
// it doesn't match the inclusion mask we do not log it. 
//
//----------------------------------------------------------------------
BOOLEAN ApplyNameFilter( PCHAR fullname )
{
    ULONG i;
    KIRQL oldirql;

    //
    // If no GUI or no filename return FALSE
    //
    if ( !fullname ) return FALSE;

    //   
    // If it matches the exclusion string, do not log it
    //
    KeAcquireSpinLock( &FilterMutex, &oldirql );
    for( i = 0; i < NumPathExcludeFilters; i++ ) {

        if( MatchWithPattern( PathExcludeFilters[i], fullname ) ) {

            KeReleaseSpinLock( &FilterMutex, oldirql );
            return FALSE;
        }
    }
 
    //
    // If it matches an include filter then log it
    //
    for( i = 0; i < NumPathIncludeFilters; i++ ) {

        if( MatchWithPattern( PathIncludeFilters[i], fullname )) {

            KeReleaseSpinLock( &FilterMutex, oldirql );
            return TRUE;
        }
    }

    //
    // It didn't match any include filters so don't log
    //
    KeReleaseSpinLock( &FilterMutex, oldirql );
    return FALSE;
}


//----------------------------------------------------------------------
//
// FilemonGetFullPath
//
// Takes a fileobject and filename and returns a canonical path,
// nicely formatted, in fullpathname.
//
//----------------------------------------------------------------------
VOID FilemonGetFullPath( PFILE_OBJECT fileObject, PHOOK_EXTENSION hookExt, 
                         PCHAR fullPathName )
{
    ULONG               pathLen;
    PCHAR               pathOffset;
    PFILE_OBJECT        relatedFileObject;
    PHASH_ENTRY         hashEntry, newEntry;
    ANSI_STRING         componentName;
    KIRQL               oldirql;

    //
    // Only do this if a GUI is active and filtering is on
    //
    if( !FilterOn || !hookExt || !fullPathName) {
     
        fullPathName[0] = 0;
        return;
    }

    //
    // First, lookup the object in the hash table to see if a name 
    // has already been generated for it
    //
    KeAcquireSpinLock( &HashMutex, &oldirql );

    hashEntry = HashTable[ HASHOBJECT( fileObject ) ];
    while( hashEntry && hashEntry->FileObject != fileObject )  {

        hashEntry = hashEntry->Next;
    }

    //
    // Did we find an entry?
    //
    if( hashEntry ) {

        //
        // Yes, so get the name from the entry.
        //
        strcpy( fullPathName, hashEntry->FullPathName );
        KeReleaseSpinLock( &HashMutex, oldirql );

        return;
    }

    KeReleaseSpinLock( &HashMutex, oldirql );

    //
    // We didn't find the name in the hash table, so we have to attempt
    // to construct it from the file objects.  Note that we won't always
    // be able to successfully do this, because the file system may
    // deallocate the name in the file object at its discretion. 
    //

    //
    // Is it DASD (Volume) I/O?
    //
    if( !fileObject || !fileObject->FileName.Length || fileObject->FileName.Length > 2*MAXPATHLEN ) {

        sprintf( fullPathName, "%C: DASD", hookExt->LogicalDrive );
        return;
    }
    
    //
    // Do this in an exception handling block, in case of mangled names in the
    // file object
    //
    try {

        //
        // Now, create the full path name. First, calculate the length taking into 
        // account space for seperators and the leading drive letter plus ':'
        //
        pathLen = fileObject->FileName.Length/2 + 2;

        relatedFileObject = fileObject->RelatedFileObject;
    
        //
        // Only look at related file object if this is a relative name
        //
        if( fileObject->FileName.Buffer[0] != L'\\' )  {

            while( relatedFileObject ) {
	        
                // 
                // If its too long, just stop.
                // 
    
                if( pathLen + relatedFileObject->FileName.Length/2+1 >= MAXPATHLEN ) {
 
                    break;
                }

                pathLen += relatedFileObject->FileName.Length/2+1;

                relatedFileObject = relatedFileObject->RelatedFileObject;
            }
    
        }

        //
        // Add the drive letter first at the front of the name
        //
        sprintf( fullPathName, "%C:", hookExt->LogicalDrive );
    
        //
        // Now, start at the end and work back to the beginning of the path
        //
        pathOffset = fullPathName + pathLen - fileObject->FileName.Length/2;

        RtlUnicodeStringToAnsiString( &componentName, &fileObject->FileName, TRUE );    

        strncpy( pathOffset, componentName.Buffer, componentName.Length + 1 );

        RtlFreeAnsiString( &componentName );

        relatedFileObject = fileObject->RelatedFileObject;
    
        if( fileObject->FileName.Buffer[0] != L'\\' )  {

            while( relatedFileObject ) {

                *(pathOffset - 1) = '\\';

                pathOffset -= relatedFileObject->FileName.Length/2 + 1;

                //
                // Bail when we've maxed the string.
                //

                if( pathOffset <= fullPathName ) {

                    break;
                }

                RtlUnicodeStringToAnsiString( &componentName, 
                                              &relatedFileObject->FileName, TRUE );

                strncpy( pathOffset, componentName.Buffer,
                         componentName.Length );

                RtlFreeAnsiString( &componentName );

                relatedFileObject = relatedFileObject->RelatedFileObject;
            }
        }  

        //
        // If we added two '\' to the front because there was a relative file object
        // that specified the root directory, remove one
        //
        if( pathLen > 3 && fullPathName[2] == '\\' && fullPathName[3] == '\\' )  {
        
            strcpy( fullPathName + 2, fullPathName + 3 );
        }

        //
        // If its a network drive, take off the first drive letter. This is because
        // network drives share a common device objects and the names are fully formed to 
        // specify the network mounted drive letter already
        //
        if( pathLen > 5 && fullPathName[4] == ':' && fullPathName[5] == '\\') {

            strcpy( fullPathName, fullPathName + 3 );
        }
    } except( EXCEPTION_EXECUTE_HANDLER ) {

        //
        // No name available - go with a dummy name
        //
        sprintf( fullPathName, "%C: ???", hookExt->LogicalDrive );
    }

    // 
    // Now that we have a name associated with the file object, put the
    // association in a hash table
    //
    KeAcquireSpinLock( &HashMutex, &oldirql );

    //
    // Allocate a hash entry
    //
    newEntry = MemAllocateFromNPagedLookasideList( &HashLookaside );

    //
    // If no memory for a new entry, oh well.
    //
    if( newEntry ) {

        //
        // Fill in the new entry and put it in the hash table
        //
        newEntry->FileObject 	= fileObject;

        newEntry->FullPathName	= MemAllocatePool( NonPagedPool, strlen(fullPathName)+1);

        //
        // Make sure there was memory for the name
        //
        if( !newEntry->FullPathName ) {

            MemFreeToNPagedLookasideList( &HashLookaside, newEntry );

        } else {

            newEntry->Next = HashTable[ HASHOBJECT(fileObject) ];
            HashTable[ HASHOBJECT(fileObject) ] = newEntry;	
            strcpy( newEntry->FullPathName, fullPathName );
        }
    }
    KeReleaseSpinLock( &HashMutex, oldirql );
}


//----------------------------------------------------------------------
//
// FilemonGetProcessNameOffset
//
// In an effort to remain version-independent, rather than using a
// hard-coded into the KPEB (Kernel Process Environment Block), we
// scan the KPEB looking for the name, which should match that
// of the system process. This is because we are in the system process'
// context in DriverEntry, where this is called.
//
//----------------------------------------------------------------------
ULONG FilemonGetProcessNameOffset()
{
    PEPROCESS       curproc;
    int             i;

    curproc = PsGetCurrentProcess();

    //
    // Scan for 12KB, hoping the KPEB never grows that big!
    //
    for( i = 0; i < 3*PAGE_SIZE; i++ ) {
     
        if( !strncmp( SYSNAME, (PCHAR) curproc + i, strlen(SYSNAME) )) {

            return i;
        }
    }

    //
    // Name not found - oh, well
    //
    return 0;
}


//----------------------------------------------------------------------
//
// FilemonGetProcess
//
// Uses undocumented data structure offsets to obtain the name of the
// currently executing process.
//
//----------------------------------------------------------------------
PCHAR FilemonGetProcess( PCHAR Name )
{
#if GETPROCESS
    PEPROCESS       curproc;
    char            *nameptr;
    ULONG           i;
    KIRQL           oldirql;

    //
    // We only do this if we determined the process name offset
    //
    if( ProcessNameOffset ) {
      
        //
        // Get a pointer to the current process block
        //
        curproc = PsGetCurrentProcess();

        //
        // Dig into it to extract the name 
        //
        nameptr   = (PCHAR) curproc + ProcessNameOffset;
         
        strncpy( Name, nameptr, NT_PROCNAMELEN );

        //
        // Terminate in case process name overflowed
        //
        Name[NT_PROCNAMELEN] = 0;

    } else {

        strcpy( Name, "???" );
    }

    //
    // Apply process name filters
    //
    KeAcquireSpinLock( &FilterMutex, &oldirql );
    for( i = 0; i < NumProcessExcludeFilters; i++ ) {

        if( MatchWithPattern( ProcessExcludeFilters[i], Name )) {

            KeReleaseSpinLock( &FilterMutex, oldirql );
            return NULL;
        }
    }
    for( i = 0; i < NumProcessFilters; i++ ) {

        if( MatchWithPattern( ProcessFilters[i], Name ) ) {

            KeReleaseSpinLock( &FilterMutex, oldirql );
            return Name;
        }
    }
    KeReleaseSpinLock( &FilterMutex, oldirql );
    return NULL;
#else

    //
    // We're not getting names, so just return something
    //
    strcpy( Name, "??" );

    return Name;
#endif
}


//----------------------------------------------------------------------
//          H O O K / U N H O O K   R O U T I N E S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// HookDrive
//
// Hook the drive specified by determining which device object to 
// attach to. The algorithm used here is similar to the one used
// internally by NT to determine which device object a file system request
// is directed at.
//
//----------------------------------------------------------------------
BOOLEAN HookDrive( IN char Drive, IN PDRIVER_OBJECT DriverObject )
{
    IO_STATUS_BLOCK     ioStatus;
    HANDLE              ntFileHandle;   
    OBJECT_ATTRIBUTES   objectAttributes;
    PDEVICE_OBJECT      fileSysDevice;
    PDEVICE_OBJECT      hookDevice;
    UNICODE_STRING      fileNameUnicodeString;
    WCHAR               filename[] = L"\\DosDevices\\A:\\";
    NTSTATUS            ntStatus;
    ULONG               i;
    PFILE_OBJECT        fileObject;
    PHOOK_EXTENSION     hookExtension;
    
    //
    // Translate the drive letter to a 0-based integer
    //
    if ( Drive >= 'a' && Drive <= 'z' ) {

        Drive -= 'a';

    } else {

        Drive -= 'A';

    }

    //
    // Is it a legal drive letter?
    //
    if ( (unsigned char) Drive >= 26 )  {

        return FALSE;
    }

    //
    // Has this drive already been hooked?
    //
    if ( LDriveDevices[Drive] == NULL )  {

        //
        // Frob the name to make it refer to the drive specified in the input 
        // parameter.
        //
        filename[12] = 'A'+Drive;

        //
        // We have to figure out what device to hook - first open the volume's 
        // root directory
        //
        RtlInitUnicodeString( &fileNameUnicodeString, filename );
        InitializeObjectAttributes( &objectAttributes, &fileNameUnicodeString, 
                                    OBJ_CASE_INSENSITIVE, NULL, NULL );
        ntStatus = ZwCreateFile( &ntFileHandle, SYNCHRONIZE|FILE_ANY_ACCESS, 
                                 &objectAttributes, &ioStatus, NULL, 0, FILE_SHARE_READ|FILE_SHARE_WRITE, 
                                 FILE_OPEN, 
                                 FILE_SYNCHRONOUS_IO_NONALERT|FILE_DIRECTORY_FILE, 
                                 NULL, 0 );
        if( !NT_SUCCESS( ntStatus ) ) {

            DbgPrint(("Filemon: Could not open drive %c: %x\n", 'A'+Drive, ntStatus ));

            return FALSE;
        }

        DbgPrint(("Filemon:  opened the root directory!!! handle: %x\n", ntFileHandle));   

        //
        // Got the file handle, so now look-up the file-object it refers to
        //
        ntStatus = ObReferenceObjectByHandle( ntFileHandle, FILE_READ_DATA, 
                                              NULL, KernelMode, &fileObject, NULL );
        if( !NT_SUCCESS( ntStatus )) {

            DbgPrint(("Filemon: Could not get fileobject from handle: %c\n", 'A'+Drive ));
            ZwClose( ntFileHandle );

            return FALSE;
        }

        //  
        // Next, find out what device is associated with the file object by getting its related
        // device object
        //
        fileSysDevice = IoGetRelatedDeviceObject( fileObject );

        if ( ! fileSysDevice ) {

            DbgPrint(("Filemon: Could not get related device object: %c\n", 'A'+Drive ));

            ObDereferenceObject( fileObject );
            ZwClose( ntFileHandle );

            return FALSE;
        }

        //  
        // Check the device list to see if we've already attached to this particular device. 
        // This can happen when more than one drive letter is being handled by the same network
        // redirecter
        //  
        for( i = 0; i < 26; i++ ) {

            if( LDriveDevices[i] == fileSysDevice ) {

                //
                // If we're already watching it, associate this drive letter
                // with the others that are handled by the same network driver. This
                // enables us to intelligently update the hooking menus when the user
                // specifies that one of the group should not be watched -we mark all
                // of the related drives as unwatched as well
                //
                ObDereferenceObject( fileObject );

                ZwClose( ntFileHandle );

                LDriveMap[ Drive ]     = LDriveMap[i];
                LDriveDevices[ Drive ] = fileSysDevice;

                return TRUE;
            }
        }

        //
        // The file system's device hasn't been hooked already, so make a hooking device
        //  object that will be attached to it.
        //
        ntStatus = IoCreateDevice( DriverObject,
                                   sizeof(HOOK_EXTENSION),
                                   NULL,
                                   fileSysDevice->DeviceType,
                                   0,
                                   FALSE,
                                   &hookDevice );
        if ( !NT_SUCCESS(ntStatus) ) {

            DbgPrint(("Filemon: failed to create associated device: %c\n", 'A'+Drive ));   

            ObDereferenceObject( fileObject );
            ZwClose( ntFileHandle );

            return FALSE;
        }

        //
        // Clear the device's init flag as per NT DDK KB article on creating device 
        // objects from a dispatch routine
        //
        hookDevice->Flags &= ~DO_DEVICE_INITIALIZING;

        //
        // Setup the device extensions. The drive letter and file system object are stored
        // in the extension.
        //
        hookExtension = hookDevice->DeviceExtension;
        hookExtension->LogicalDrive = 'A'+Drive;
        hookExtension->FileSystem   = fileSysDevice;

        //
        // Finally, attach to the device. The second we're successfully attached, we may 
        // start receiving IRPs targetted at the device we've hooked.
        //
        ntStatus = IoAttachDeviceByPointer( hookDevice, fileSysDevice );
        if ( !NT_SUCCESS(ntStatus) )  {

            //
            // Couldn' attach for some reason
            //
            DbgPrint(("Filemon: Connect with Filesystem failed: %c (%x) =>%x\n", 
                      'A'+Drive, fileSysDevice, ntStatus ));

            //
            // Derefence the object and get out
            //
            ObDereferenceObject( fileObject );
            ZwClose( ntFileHandle );

            return FALSE;

        } else {

            // 
            // Make a new drive group for the device,l if it does not have one 
            // already
            // 
            DbgPrint(("Filemon: Successfully connected to Filesystem device %c\n", 'A'+Drive ));
            if( !LDriveMap[ Drive ] ) {

                LDriveMap[ Drive ] = ++LDriveGroup;
            }
        }
    
        //
        // Close the file and update the hooked drive list by entering a
        // pointer to the hook device object in it.
        //
        ObDereferenceObject( fileObject );

        ZwClose( ntFileHandle );

        LDriveDevices[Drive] = hookDevice;
    }

    return TRUE;
}


//----------------------------------------------------------------------
//
// UnhookDrive
//
// Unhook a previously hooked drive.
//
//----------------------------------------------------------------------
BOOLEAN UnhookDrive( IN char Drive )
{
    PHOOK_EXTENSION hookExt;

    //
    // Translate the drive letter to a 0-based integer
    //
    if ( Drive >= 'a' && Drive <= 'z' ) {

        Drive -= 'a';

    } else {

        Drive -= 'A';

    }

    //
    // If the drive has been hooked, unhook it and delete the hook
    // device object
    //
    if ( LDriveDevices[Drive] )  {

        hookExt = LDriveDevices[Drive]->DeviceExtension;

        IoDetachDevice( hookExt->FileSystem );

        IoDeleteDevice( LDriveDevices[Drive] );

        LDriveDevices[Drive] = NULL;
    }
    return TRUE;
}


//----------------------------------------------------------------------
//
// HookDriveSet
//
// Hook/Unhook a set of drives specified by user. Return the set 
// that is currently hooked.
//
//----------------------------------------------------------------------
ULONG HookDriveSet( IN ULONG DriveSet, IN PDRIVER_OBJECT DriverObject )
{
    ULONG drive, i;
    ULONG bit;

    //
    // Scan the drive table, looking for hits on the DriveSet bitmask
    //
    for ( drive = 0; drive < 26; ++drive )  {

        bit = 1 << drive;

        //
        // Are we suppoed to hook this drive?
        //
        if ( bit & DriveSet )  {

            //
            // Try to hook drive 
            //
            if ( ! HookDrive( (char)('A'+drive), DriverObject ) ) {
             
                //
                // Remove from drive set if can't be hooked
                //
                DriveSet &= ~bit;

            } else {

                //
                // hook drives in same drive group      
                //
                for( i = 0; i < 26; i++ ) {

                    if( LDriveMap[i] == LDriveMap[ drive ] &&

                        !LDriveDevices[i] ) {
                        DriveSet |= ( 1<<i );
                        LDriveDevices[i] = LDriveDevices[drive];

                    }
                }
            }

        } else {

            //
            // Try to unhook drive 
            //
            if ( ! UnhookDrive( (char)('A'+drive) ) )  {

                //
                // Unhook failed, leave the drive marked as hooked
                //
                DriveSet |= bit;    

            } else {

                // 
                // Unhook worked. Mark all drives in same group as
                // unhooked
                //
                for( i = 0; i< 26; i++ ) {

                    if( LDriveMap[i] == LDriveMap[ drive ] && 
                        LDriveDevices[i] ) {

                        DriveSet &= ~(1 << i); 
                        LDriveDevices[i] = NULL;
                    }
                }
            }
        }
    }

    //
    // Return set of drives currently hooked
    //
    return DriveSet;
}

//----------------------------------------------------------------------
//
// ControlCodeString
//
// Takes a control code and sees if we know what it is.
//
//----------------------------------------------------------------------
PCHAR ControlCodeString( ULONG ControlCode, PCHAR Buffer )
{
    switch( ControlCode ) {

    case FSCTL_REQUEST_OPLOCK_LEVEL_1:
        sprintf( Buffer, "FSCTL_REQUEST_OPLOCK_LEVEL_1" );
        break;
    case FSCTL_REQUEST_OPLOCK_LEVEL_2:
        sprintf( Buffer, "FSCTL_REQUEST_OPLOCK_LEVEL_2" );
        break;
    case FSCTL_REQUEST_BATCH_OPLOCK:
        sprintf( Buffer, "FSCTL_REQUEST_BATCH_OPLOCK" );
        break;        
    case FSCTL_OPLOCK_BREAK_ACKNOWLEDGE:
        sprintf( Buffer, "FSCTL_OPLOCK_BREAK_ACKNOWLEDGE" );
        break;
    case FSCTL_OPBATCH_ACK_CLOSE_PENDING:
        sprintf( Buffer, "FSCTL_OPBATCH_ACK_CLOSE_PENDING" );
        break;
    case FSCTL_OPLOCK_BREAK_NOTIFY:
        sprintf( Buffer, "FSCTL_OPLOCK_BREAK_NOTIFY" );
        break;
    case FSCTL_LOCK_VOLUME:
        sprintf( Buffer, "FSCTL_LOCK_VOLUME" );
        break;
    case FSCTL_UNLOCK_VOLUME:
        sprintf( Buffer, "FSCTL_UNLOCK_VOLUME" );
        break;
    case FSCTL_DISMOUNT_VOLUME:
        sprintf( Buffer, "FSCTL_DISMOUNT_VOLUME" );
        break;
    case FSCTL_IS_VOLUME_MOUNTED:
        sprintf( Buffer, "FSCTL_IS_VOLUME_MOUNTED" );
        break;
    case FSCTL_IS_PATHNAME_VALID:
        sprintf( Buffer, "FSCTL_IS_PATHNAME_VALID" );
        break;
    case FSCTL_MARK_VOLUME_DIRTY:
        sprintf( Buffer, "FSCTL_MARK_VOLUME_DIRTY" );
        break;
    case FSCTL_QUERY_RETRIEVAL_POINTERS:
        sprintf( Buffer, "FSCTL_QUERY_RETRIEVAL_POINTERS" );
        break;
    case FSCTL_GET_COMPRESSION:
        sprintf( Buffer, "FSCTL_GET_COMPRESSION" );
        break;
    case FSCTL_SET_COMPRESSION:
        sprintf( Buffer, "FSCTL_SET_COMPRESSION" );
        break;
    case FSCTL_OPLOCK_BREAK_ACK_NO_2:
        sprintf( Buffer, "FSCTL_OPLOCK_BREAK_ACK_NO_2" );
        break;
    case FSCTL_QUERY_FAT_BPB:
        sprintf( Buffer, "FSCTL_QUERY_FAT_BPB" );
        break;
    case FSCTL_REQUEST_FILTER_OPLOCK:
        sprintf( Buffer, "FSCTL_REQUEST_FILTER_OPLOCK" );
        break;
    case FSCTL_FILESYSTEM_GET_STATISTICS:
        sprintf( Buffer, "FSCTL_FILESYSTEM_GET_STATISTICS" );
        break;
    case FSCTL_GET_NTFS_VOLUME_DATA:
        sprintf( Buffer, "FSCTL_GET_NTFS_VOLUME_DATA" );
        break;
    case FSCTL_GET_NTFS_FILE_RECORD:
        sprintf( Buffer, "FSCTL_GET_NTFS_FILE_RECORD" );
        break;
    case FSCTL_GET_VOLUME_BITMAP:
        sprintf( Buffer, "FSCTL_GET_VOLUME_BITMAP" );
        break;
    case FSCTL_GET_RETRIEVAL_POINTERS:
        sprintf( Buffer, "FSCTL_GET_RETRIEVAL_POINTERS" );
        break;
    case FSCTL_MOVE_FILE:
        sprintf( Buffer, "FSCTL_MOVE_FILE" );
        break;
    case FSCTL_IS_VOLUME_DIRTY:
        sprintf( Buffer, "FSCTL_IS_VOLUME_DIRTY" );
        break;
    case FSCTL_ALLOW_EXTENDED_DASD_IO:
        sprintf( Buffer, "FSCTL_ALLOW_EXTENDED_DASD_IO" );
        break;
        //
        // *** new to NT 5.0
        //
#if NT5_IOCTLS
    case FSCTL_READ_PROPERTY_DATA:
        sprintf( Buffer, "FSCTL_READ_PROPERTY_DATA" );
        break;
    case FSCTL_WRITE_PROPERTY_DATA:
        sprintf( Buffer, "FSCTL_WRITE_PROPERTY_DATA" );
        break;
    case FSCTL_FIND_FILES_BY_SID:
        sprintf( Buffer, "FSCTL_FIND_FILES_BY_SID" );
        break;
    case FSCTL_DUMP_PROPERTY_DATA:
        sprintf( Buffer, "FSCTL_DUMP_PROPERTY_DATA" );
        break;
    case FSCTL_SET_OBJECT_ID:
        sprintf( Buffer, "FSCTL_SET_OBJECT_ID" );
        break;
    case FSCTL_GET_OBJECT_ID:
        sprintf( Buffer, "FSCTL_GET_OBJECT_ID" );
        break;
    case FSCTL_DELETE_OBJECT_ID:
        sprintf( Buffer, "FSCTL_DELETE_OBJECT_ID" );
        break;
    case FSCTL_SET_REPARSE_POINT:
        sprintf( Buffer, "FSCTL_SET_REPARSE_POINT" );
        break;
    case FSCTL_GET_REPARSE_POINT:
        sprintf( Buffer, "FSCTL_GET_REPARSE_POINT" );
        break;
    case FSCTL_DELETE_REPARSE_POINT:
        sprintf( Buffer, "FSCTL_DELETE_REPARSE_POINT" );
        break;
    case FSCTL_ENUM_USN_DATA:
        sprintf( Buffer, "FSCTL_ENUM_USN_DATA" );
        break;
    case FSCTL_SECURITY_ID_CHECK:
        sprintf( Buffer, "FSCTL_SECURITY_ID_CHECK" );
        break;
    case FSCTL_READ_USN_JOURNAL:
        sprintf( Buffer, "FSCTL_READ_USN_JOURNAL" );
        break;
    case FSCTL_SET_OBJECT_ID_EXTENDED:
        sprintf( Buffer, "FSCTL_SET_OBJECT_ID_EXTENDED" );
        break;
    case FSCTL_CREATE_OR_GET_OBJECT_ID:
        sprintf( Buffer, "FSCTL_CREATE_OR_GET_OBJECT_ID" );
        break;
    case FSCTL_SET_SPARSE:
        sprintf( Buffer, "FSCTL_SET_SPARSE" );
        break;
    case FSCTL_SET_ZERO_DATA:
        sprintf( Buffer, "FSCTL_SET_ZERO_DATA" );
        break;
    case FSCTL_QUERY_ALLOCATED_RANGES:
        sprintf( Buffer, "FSCTL_QUERY_ALLOCATED_RANGES" );
        break;
    case FSCTL_ENABLE_UPGRADE:
        sprintf( Buffer, "FSCTL_ENABLE_UPGRADE" );
        break;
    case FSCTL_SET_ENCRYPTION:
        sprintf( Buffer, "FSCTL_SET_ENCRYPTION" );
        break;
    case FSCTL_ENCRYPTION_FSCTL_IO:
        sprintf( Buffer, "FSCTL_ENCRYPTION_FSCTL_IO" );
        break;
    case FSCTL_WRITE_RAW_ENCRYPTED:
        sprintf( Buffer, "FSCTL_WRITE_RAW_ENCRYPTED" );
        break;
    case FSCTL_READ_RAW_ENCRYPTED:
        sprintf( Buffer, "FSCTL_READ_RAW_ENCRYPTED" );
        break;
    case FSCTL_CREATE_USN_JOURNAL:
        sprintf( Buffer, "FSCTL_CREATE_USN_JOURNAL" );
        break;
    case FSCTL_READ_FILE_USN_DATA:
        sprintf( Buffer, "FSCTL_READ_FILE_USN_DATA" );
        break;
    case FSCTL_WRITE_USN_CLOSE_RECORD:
        sprintf( Buffer, "FSCTL_WRITE_USN_CLOSE_RECORD" );
        break;
    case FSCTL_EXTEND_VOLUME:
        sprintf( Buffer, "FSCTL_EXTEND_VOLUME" );
        break;
#endif
    default:

        sprintf( Buffer, "IOCTL: 0x%X", ControlCode );
        break;
    }
    return Buffer;
}


//----------------------------------------------------------------------
//
// ErrorString
//
// Returns string representing the passed error condition.
//
//----------------------------------------------------------------------
PCHAR ErrorString( NTSTATUS RetStat, PCHAR Buffer ) 
{
    switch( RetStat ) {

    case STATUS_SUCCESS:
        strcpy( Buffer, "SUCCESS" );
        break;
    case STATUS_CRC_ERROR:
        strcpy( Buffer, "CRC ERROR" );
        break;
    case STATUS_NOT_IMPLEMENTED:
        strcpy( Buffer, "NOT IMPLEMENTED" );
        break;
    case STATUS_EAS_NOT_SUPPORTED:
        strcpy( Buffer, "EAS NOT SUPPORTED" );
        break;
    case STATUS_EA_TOO_LARGE:
        strcpy( Buffer, "EA TOO LARGE");
        break;
    case STATUS_NONEXISTENT_EA_ENTRY:
        strcpy( Buffer, "NONEXISTENT EA ENTRY");
        break;
    case STATUS_BAD_NETWORK_NAME:
        strcpy( Buffer, "BAD NETWORK NAME" );
        break;
    case STATUS_NOTIFY_ENUM_DIR:
        strcpy( Buffer, "NOTIFY ENUM DIR" );
        break;
    case STATUS_FILE_CORRUPT_ERROR:
        strcpy( Buffer, "FILE CORRUPT" );
        break;
    case STATUS_DISK_CORRUPT_ERROR:
        strcpy( Buffer, "DISK CORRUPT" );
        break;
    case STATUS_RANGE_NOT_LOCKED:
        strcpy( Buffer, "RANGE NOT LOCKED" );
        break;
    case STATUS_FILE_CLOSED:
        strcpy( Buffer, "FILE CLOSED" );
        break;
    case STATUS_IN_PAGE_ERROR:
        strcpy( Buffer, "IN PAGE ERROR" );
        break;
    case STATUS_CANCELLED:
        strcpy( Buffer, "CANCELLED" );
        break;
    case STATUS_QUOTA_EXCEEDED:
        strcpy( Buffer, "QUOTA EXCEEDED" );
        break;
    case STATUS_NOT_SUPPORTED:
        strcpy( Buffer, "NOT SUPPORTED" );
        break;
    case STATUS_NO_MORE_FILES:
        strcpy( Buffer, "NO MORE FILES" );
        break;
    case STATUS_OBJECT_NAME_INVALID:
        strcpy( Buffer, "NAME INVALID" );
        break;
    case STATUS_OBJECT_NAME_NOT_FOUND:
        strcpy( Buffer, "FILE NOT FOUND" );
        break;
    case STATUS_NOT_A_DIRECTORY:
        strcpy( Buffer, "NOT A DIRECTORY" );
        break;
    case STATUS_NO_SUCH_FILE:
        strcpy( Buffer, "NO SUCH FILE" );
        break;
    case STATUS_OBJECT_NAME_COLLISION:
        strcpy( Buffer, "NAME COLLISION" );
        break;
    case STATUS_NONEXISTENT_SECTOR:
        strcpy( Buffer, "NONEXISTENT SECTOR" );
        break;
    case STATUS_BAD_NETWORK_PATH:
        strcpy( Buffer, "BAD NETWORK PATH" );
        break;
    case STATUS_OBJECT_PATH_NOT_FOUND:
        strcpy( Buffer, "PATH NOT FOUND" );
        break;
    case STATUS_NO_SUCH_DEVICE:
        strcpy( Buffer, "INVALID PARAMETER" );
        break;
    case STATUS_END_OF_FILE:
        strcpy( Buffer, "END OF FILE" );
        break;
    case STATUS_NOTIFY_CLEANUP:
        strcpy( Buffer, "NOTIFY CLEANUP" );
        break;
    case STATUS_BUFFER_OVERFLOW:
        strcpy( Buffer, "BUFFER OVERFLOW" );
        break;
    case STATUS_NO_MORE_ENTRIES:
        strcpy( Buffer, "NO MORE ENTRIES" );
        break;
    case STATUS_ACCESS_DENIED:
        strcpy( Buffer, "ACCESS DENIED" );
        break;
    case STATUS_SHARING_VIOLATION:
        strcpy( Buffer, "SHARING VIOLATION" );
        break;       
    case STATUS_INVALID_PARAMETER:
        strcpy( Buffer, "INVALID PARAMETER" );
        break;       
    case STATUS_OPLOCK_BREAK_IN_PROGRESS:
        strcpy( Buffer, "OPLOCK BREAK" );
        break;        
    case STATUS_OPLOCK_NOT_GRANTED:
        strcpy( Buffer, "OPLOCK NOT GRANTED" );
        break;
    case STATUS_FILE_LOCK_CONFLICT:
        strcpy( Buffer, "FILE LOCK CONFLICT" );
        break;
    case STATUS_PENDING:
        strcpy( Buffer, "PENDING" );
        break;       
    case STATUS_REPARSE:
        strcpy( Buffer, "REPARSE" );
        break;       
    case STATUS_MORE_ENTRIES:
        strcpy( Buffer, "MORE" );
        break;       
    case STATUS_DELETE_PENDING:
        strcpy( Buffer, "DELETE PEND" );
        break;       
    case STATUS_LOCK_NOT_GRANTED:
        strcpy( Buffer, "NOT GRANTED" );
        break;       
    case STATUS_FILE_IS_A_DIRECTORY:
        strcpy( Buffer, "IS DIRECTORY" );
        break;
    case STATUS_ALREADY_COMMITTED:
        strcpy( Buffer, "ALREADY COMMITTED" );
        break;
    case STATUS_INVALID_EA_FLAG:
        strcpy( Buffer, "INVALID EA FLAG" );
        break;
    case STATUS_INVALID_INFO_CLASS:
        strcpy( Buffer, "INVALID INFO CLASS" );
        break;
    case STATUS_INVALID_HANDLE:
        strcpy( Buffer, "INVALID HANDLE" );
        break;
    case STATUS_INVALID_DEVICE_REQUEST:
        strcpy( Buffer, "INVALID DEVICE REQUEST" );
        break;
    case STATUS_WRONG_VOLUME:
        strcpy( Buffer, "WRONG VOLUME" );
        break;
    case STATUS_UNEXPECTED_NETWORK_ERROR:
        strcpy( Buffer, "NETWORK ERROR" );
        break;
    case STATUS_DFS_UNAVAILABLE:
        strcpy( Buffer, "DFS UNAVAILABLE" );
        break;
    case STATUS_LOG_FILE_FULL:
        strcpy( Buffer, "LOG FILE FULL" );
    	break;
    default:
        sprintf( Buffer, "* 0x%X", RetStat );
        break;
    }

    return Buffer;
}

//----------------------------------------------------------------------
//                F A S T I O   R O U T I N E S
//
// NOTE: There is no need for us to worry about accessing fastio 
// parameters within try/except because the I/O manager has either
// probed the validity of the arguments or calls within its own 
// try/except block (it doesn't trust us anyway :-) ).
//
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// FilemonFastIoCheckIfPossible
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoCheckifPossible( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
                                       IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN BOOLEAN CheckForReadOperation,
                                       OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN         retval = FALSE;
    PHOOK_EXTENSION hookExt;
    CHAR            *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER   timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoCheckIfPossible ) ) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }   
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoCheckIfPossible( 
            FileObject, FileOffset, Length,
            Wait, LockKey, CheckForReadOperation, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_CHECK_IF_POSSIBLE\t%s\t%s Offset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         CheckForReadOperation ? "Read:" : "Write:",
                         FileOffset->LowPart, Length, 
                         retval?"SUCCESS":"FAILURE" ); 
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }

    return retval;
}


//----------------------------------------------------------------------
// 
// FilemonFastIoRead
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoRead( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
                            IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, OUT PVOID Buffer,
                            OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoRead ) ) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoRead( 
            FileObject, FileOffset, Length,
            Wait, LockKey, Buffer, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilterDef.logreads && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_READ\t%s\tOffset: %d Length: %ld\t%s", 
                         name, fullPathName, 
                         FileOffset->LowPart, Length, 
                         retval?ErrorString( IoStatus->Status, errorBuf):"FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoWrite
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoWrite( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
                             IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN PVOID Buffer,
                             OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN              retval = FALSE;
    PHOOK_EXTENSION      hookExt;
    CHAR                 *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER        timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoWrite )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoWrite( 
            FileObject, FileOffset, Length, Wait, LockKey, 
            Buffer, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilterDef.logwrites && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_WRITE\t%s\tOffset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         FileOffset->LowPart, Length, 
                         retval?ErrorString( IoStatus->Status, errorBuf ):"FAILURE" ); 
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryBasicinfo
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoQueryBasicInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
                                      OUT PFILE_BASIC_INFORMATION Buffer,
                                      OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoQueryBasicInfo ) ) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryBasicInfo( 
            FileObject, Wait, Buffer, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            if( retval ) {
                UpdateStore( InterlockedIncrement(&Sequence), 
                             &timeResult,
                             "%s\tFASTIO_QUERY_BASIC_INFO\t%s\tAttributes:%04x\t%s", 
                             name, fullPathName, 
                             ((PFILE_BASIC_INFORMATION) Buffer)->FileAttributes, 
                             retval?ErrorString( IoStatus->Status, errorBuf ):"FAILURE" );

            } else { 
                UpdateStore( InterlockedIncrement(&Sequence), 
                             &timeResult,
                             "%s\tFASTIO_QUERY_BASIC_INFO\t%s\t\t%s", 
                             name, fullPathName, retval?"SUCCESS":"FAILURE" );
            }
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryStandardInfo
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoQueryStandardInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
                                         OUT PFILE_STANDARD_INFORMATION Buffer,
                                         OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoQueryStandardInfo ) ) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryStandardInfo( 
            FileObject, Wait, Buffer, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            if( retval ) {
                UpdateStore( InterlockedIncrement(&Sequence), 
                             &timeResult,
                             "%s\tFASTIO_QUERY_STANDARD_INFO\t%s\tSize: %d\t%s", 
                             name, fullPathName,
                             ((PFILE_STANDARD_INFORMATION) Buffer)->EndOfFile.LowPart, 
                             retval?"SUCCESS":"FAILURE" );
            } else {
                UpdateStore( InterlockedIncrement(&Sequence), 
                             &timeResult,
                             "%s\tFASTIO_QUERY_STANDARD_INFO\t%s\t\t%s", 
                             name, fullPathName, 
                             retval?ErrorString( IoStatus->Status, errorBuf ):"FAILURE" );
            }
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoLock
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoLock( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
                            IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
                            BOOLEAN FailImmediately, BOOLEAN ExclusiveLock,
                            OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoLock )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoLock( 
            FileObject, FileOffset, Length, ProcessId, Key, FailImmediately, 
            ExclusiveLock, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_LOCK\t%s\tExcl: %s Offset: %d Length: %d\t%s", 
                         name, fullPathName,
                         ExclusiveLock ? "Yes":"No", FileOffset ? FileOffset->LowPart : 0,
                         Length ? Length->LowPart : 0, retval?ErrorString( IoStatus->Status, errorBuf ):"FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoUnlockSingle
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoUnlockSingle( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
                                    IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
                                    OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;   
    
    if( FASTIOPRESENT( hookExt, FastIoUnlockSingle )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockSingle(
            FileObject, FileOffset, Length, ProcessId, Key, 
            IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_UNLOCK\t%s\tOffset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         FileOffset? FileOffset->LowPart : 0, Length ? Length->LowPart : 0,
                         retval?ErrorString( IoStatus->Status, errorBuf ):"FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoUnlockAll
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoUnlockAll( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId,
                                 OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if ( FASTIOPRESENT(hookExt, FastIoUnlockAll ) ) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockAll( 
            FileObject, ProcessId, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_UNLOCK_ALL\t%s\t\t%s", 
                         name, fullPathName, 
                         retval?ErrorString( IoStatus->Status, errorBuf ):"FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoUnlockAllByKey
//
//----------------------------------------------------------------------    
BOOLEAN  FilemonFastIoUnlockAllByKey( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId, ULONG Key,
                                      OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoUnlockAllByKey )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockAllByKey( 
            FileObject, ProcessId, Key, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_UNLOCK_ALL_BY_KEY\t%s\t\t%s", 
                         name, fullPathName, 
                         retval?ErrorString( IoStatus->Status, errorBuf):"FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryNetworkOpenInfo
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoQueryNetworkOpenInfo(IN PFILE_OBJECT FileObject,
                                          IN BOOLEAN Wait,
                                          OUT struct _FILE_NETWORK_OPEN_INFORMATION *Buffer,
                                          OUT PIO_STATUS_BLOCK IoStatus,
                                          IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER   timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoQueryNetworkOpenInfo )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryNetworkOpenInfo( 
            FileObject, Wait, Buffer, IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_QUERY_NETWORK_OPEN_INFO\t%s\t\t%s", 
                         name, fullPathName, 
                         retval ? ErrorString( IoStatus->Status, errorBuf ): "FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoAcquireForModWrite
//
//----------------------------------------------------------------------    
NTSTATUS FilemonFastIoAcquireForModWrite( IN PFILE_OBJECT FileObject,
                                          IN PLARGE_INTEGER EndingOffset,
                                          OUT struct _ERESOURCE **ResourceToRelease,
                                          IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, errval[ERRORLEN], name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, AcquireForModWrite )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireForModWrite( 
            FileObject, EndingOffset, ResourceToRelease, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_ACQUIRE_FOR_MOD_WRITE\t%s\tEndOffset: %d\t%s", 
                         name, fullPathName, EndingOffset, 
                         ErrorString( retval, errval ) );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlRead
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoMdlRead( IN PFILE_OBJECT FileObject,
                              IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                              IN ULONG LockKey, OUT PMDL *MdlChain,
                              OUT PIO_STATUS_BLOCK IoStatus,
                              IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, MdlRead )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlRead( 
            FileObject, FileOffset, Length, LockKey, MdlChain, 
            IoStatus, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_MDL_READ\t%s\tOffset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         FileOffset->LowPart, Length, 
                         retval ? ErrorString( IoStatus->Status, errorBuf ): "FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlReadComplete
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoMdlReadComplete( IN PFILE_OBJECT FileObject,
                                      IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, MdlReadComplete )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = (BOOLEAN) hookExt->FileSystem->DriverObject->FastIoDispatch->MdlReadComplete( FileObject, 
                                                                                               MdlChain, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_MDL_READ_COMPLETE\t%s\t\t%s", 
                         name, fullPathName, "OK" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoPrepareMdlWrite
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoPrepareMdlWrite( IN PFILE_OBJECT FileObject,
                                      IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                                      IN ULONG LockKey, OUT PMDL *MdlChain,
                                      OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;
    IoStatus->Status      = STATUS_NOT_SUPPORTED;
    IoStatus->Information = 0;

    if( FASTIOPRESENT( hookExt, PrepareMdlWrite )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->PrepareMdlWrite( 
            FileObject, FileOffset, Length, LockKey, MdlChain, IoStatus, 
            hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_PREPARE_MDL_WRITE\t%s\tOffset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         FileOffset->LowPart, Length, 
                         retval ? ErrorString( IoStatus->Status, errorBuf ): "FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    } 
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlWriteComplete
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoMdlWriteComplete( IN PFILE_OBJECT FileObject,
                                       IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain, 
                                       IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, MdlWriteComplete )) { 

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlWriteComplete( 
            FileObject, FileOffset, MdlChain, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_MDL_WRITE_COMPLETE\t%s\tOffset: %d\tOK", 
                         name, fullPathName, FileOffset->LowPart );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoReadCompressed
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoReadCompressed( IN PFILE_OBJECT FileObject,
                                     IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                                     IN ULONG LockKey, OUT PVOID Buffer,
                                     OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
                                     OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
                                     IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoReadCompressed )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoReadCompressed( 
            FileObject, FileOffset, Length, LockKey, Buffer, MdlChain, IoStatus,
            CompressedDataInfo, CompressedDataInfoLength, hookExt->FileSystem );

        if( fullPathName && FilterDef.logreads && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_READ_COMPRESSED\t%s\tOffset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         FileOffset->LowPart, Length,
                         retval ? ErrorString( IoStatus->Status, errorBuf ) : "FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoWriteCompressed
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoWriteCompressed( IN PFILE_OBJECT FileObject,
                                      IN PLARGE_INTEGER FileOffset, IN ULONG Length,
                                      IN ULONG LockKey, OUT PVOID Buffer,
                                      OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
                                      OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
                                      IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN], errorBuf[ERRORLEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoWriteCompressed )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoWriteCompressed( 
            FileObject, FileOffset, Length, LockKey, Buffer, MdlChain, IoStatus,
            CompressedDataInfo, CompressedDataInfoLength, hookExt->FileSystem );

        if( fullPathName && FilterDef.logwrites && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_WRITE_COMPRESSED\t%s\tOffset: %d Length: %d\t%s", 
                         name, fullPathName, 
                         FileOffset->LowPart, Length, 
                         retval ? ErrorString( IoStatus->Status, errorBuf ) : "FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlReadCompleteCompressed
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoMdlReadCompleteCompressed( IN PFILE_OBJECT FileObject,
                                                IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, MdlReadCompleteCompressed )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlReadCompleteCompressed( 
            FileObject, MdlChain, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }

            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_MDL_READ_COMPLETE_COMPRESSED\t%s\t\t%s", 
                         name, fullPathName, "OK" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlWriteCompleteCompressed
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoMdlWriteCompleteCompressed( IN PFILE_OBJECT FileObject,
                                                 IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain, 
                                                 IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension; 

    if( FASTIOPRESENT( hookExt, MdlWriteCompleteCompressed )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlWriteCompleteCompressed( 
            FileObject, FileOffset, MdlChain, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_MDL_WRITE_COMPLETE_COMPRESSED\t%s\tOffset: %d\t%s", 
                         name, fullPathName, FileOffset->LowPart, "OK" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryOpen
//
// This call actually passes an IRP! 
//
//----------------------------------------------------------------------    
BOOLEAN FilemonFastIoQueryOpen( IN PIRP Irp,
                                OUT PFILE_NETWORK_OPEN_INFORMATION NetworkInformation,
                                IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    PFILE_OBJECT        fileobject;
    CHAR                *fullPathName, name[PROCNAMELEN];
    PIO_STACK_LOCATION  currentIrpStack;
    PIO_STACK_LOCATION  nextIrpStack;
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoQueryOpen )) {

        currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
        nextIrpStack    = IoGetNextIrpStackLocation(Irp);
        fileobject      = currentIrpStack->FileObject;

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( fileobject, hookExt, fullPathName );
        }       

        //
        // copy parameters down to next level in the stack
        //
        *nextIrpStack = *currentIrpStack;
        nextIrpStack->DeviceObject = hookExt->FileSystem;
        IoSetNextIrpStackLocation( Irp );
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryOpen( 
            Irp, NetworkInformation, hookExt->FileSystem );

        //
        // Restore the stack location because pre-NT 5.0 checked bbuilds complain
        //
        Irp->CurrentLocation++;
        Irp->Tail.Overlay.CurrentStackLocation++;

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_QUERY_OPEN\t%s\t\t%s", 
                         name, fullPathName, retval ? "SUCCESS" : "FAILURE" );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoReleaseForModWrite
//
//----------------------------------------------------------------------    
NTSTATUS FilemonFastIoReleaseForModWrite( IN PFILE_OBJECT FileObject,
                                          IN struct _ERESOURCE *ResourceToRelease,
                                          IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, errval[ERRORLEN], name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, ReleaseForModWrite )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseForModWrite( 
            FileObject,  ResourceToRelease, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_RELEASE_FOR_MOD_WRITE\t%s\t\t%s", 
                         name, fullPathName, ErrorString( retval, errval ));
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoAcquireForCcFlush
//
//----------------------------------------------------------------------    
NTSTATUS FilemonFastIoAcquireForCcFlush( IN PFILE_OBJECT FileObject,
                                         IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS                retval = STATUS_NOT_SUPPORTED;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, errval[ERRORLEN], name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, AcquireForCcFlush )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireForCcFlush( 
            FileObject, hookExt->FileSystem );
        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_ACQUIRE_FOR_CC_FLUSH\t%s\t\t%s", 
                         name, fullPathName, ErrorString( retval, errval));
        }
        if ( fullPathName) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoReleaseForCcFlush
//
//----------------------------------------------------------------------    
NTSTATUS FilemonFastIoReleaseForCcFlush( IN PFILE_OBJECT FileObject,
                                         IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, errval[ERRORLEN], name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    hookExt = DeviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, ReleaseForCcFlush )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        retval = hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseForCcFlush( 
            FileObject, hookExt->FileSystem );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), 
                         &timeResult,
                         "%s\tFASTIO_RELEASE_FOR_CC_FLUSH\t%s\t\t%s", 
                         name, fullPathName, ErrorString( retval, errval) ); 
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoDeviceControl
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoDeviceControl( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait,
                                     IN PVOID InputBuffer, IN ULONG InputBufferLength, 
                                     OUT PVOID OutputBuffer, IN ULONG OutputBufferLength, IN ULONG IoControlCode,
                                     OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
    BOOLEAN             retval = FALSE;
    PHOOK_EXTENSION     hookExt;
    PSTORE_BUF          old;
    CHAR                fullPathName[2*MAXPATHLEN], name[PROCNAMELEN], errorBuf[ERRORLEN];
    KIRQL               oldirql;
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    if ( DeviceObject == GUIDevice )  {

        //
        // Its a message from our GUI!
        //
        IoStatus->Status      = STATUS_SUCCESS; // Assume success
        IoStatus->Information = 0;      // Assume nothing returned

        switch ( IoControlCode ) {

        case FILEMON_version:

            //
            // Version #
            //
            if ( OutputBufferLength < sizeof(ULONG) ||
                 OutputBuffer == NULL ) {

                IoStatus->Status = STATUS_INVALID_PARAMETER;
                break;
            }
            
            *(ULONG *)OutputBuffer = FILEMONVERSION;
            IoStatus->Information = sizeof(ULONG);
            break;

        case FILEMON_setdrives:

            //
            // Hook and/or unhook drives
            //
            DbgPrint (("Filemon: set drives\n"));

            if ( InputBufferLength != sizeof(ULONG) ||
                 OutputBufferLength < sizeof(ULONG) ||
                 InputBuffer == NULL ||
                 OutputBuffer == NULL ) {

                IoStatus->Status = STATUS_INVALID_PARAMETER;
                break;
            }

            *(ULONG *)OutputBuffer = HookDriveSet( *(ULONG *)InputBuffer, DeviceObject->DriverObject );
            IoStatus->Information = sizeof(ULONG);
            break;

        case FILEMON_stopfilter:
            
            //
            // Turn off logging
            //
            DbgPrint(("Filemon: stop logging\n"));
            FilterOn = FALSE;
            break;

        case FILEMON_startfilter:
          
            //
            // Turn on logging 
            //
            DbgPrint(("Filemon: start logging\n"));
            FilterOn = TRUE;
            break;

        case FILEMON_setfilter:
  
            //
            // Gui is updating the filter functions
            //
            DbgPrint(("Filemon: set filter\n"));
            FilterDef = *(PFILTER) InputBuffer;
            FilemonUpdateFilters();
            break;

        case FILEMON_timetype:

            //	    
            // Set the timing type
            //	    
            TimeIsDuration = *(PBOOLEAN) InputBuffer;
            break;

        case FILEMON_unloadquery:

            //
            // is it possible to unload?
            //
            KeAcquireSpinLock( &CountMutex, &oldirql );
            IoStatus->Information = OutstandingIRPCount;

            //
            // Any outstanding Irps?
            //
            if( !OutstandingIRPCount ) {

                //
                // Nope, so don't process anymore
                //
                UnloadInProgress = TRUE;

                KeReleaseSpinLock( &CountMutex, oldirql );

                //
                // Disconnect from filesystems
                //
                HookDriveSet( 0, DeviceObject->DriverObject );

            } else {

                KeReleaseSpinLock( &CountMutex, oldirql );
            }
            break;

        case FILEMON_zerostats:

            //
            // Reset all output buffers
            //
            DbgPrint (("Filemon: zero stats\n"));

            KeAcquireSpinLock( &StoreMutex, &oldirql );

            while ( Store->Next )  {

                //
                // Free all but the first output buffer
                //
                old = Store->Next;
                Store->Next = old->Next;

                MemFreePool( old );

                NumStore--;
            }
 
            //
            // Set the output pointer to the start of the output buffer
            //
            Store->Len = 0;
            Sequence = 0;

            KeReleaseSpinLock( &StoreMutex, oldirql );
            break;

        case FILEMON_getstats:

            //
            // Copy the oldest output buffer to the caller
            //
            DbgPrint (("Filemon: get stats\n"));

			//
            // If the output buffer is too large to fit into the caller's buffer
            //
            if ( MAX_STORE > OutputBufferLength )  {
                IoStatus->Status = STATUS_INVALID_PARAMETER;
                return FALSE;
            }

            KeAcquireSpinLock( &StoreMutex, &oldirql );

            if ( Store->Len  ||  Store->Next ) {

                //
                // Start output to a new output buffer
                //
                FilemonNewStore();

                //
                // Fetch the oldest to give to user
                //
                old = FilemonOldestStore();

                KeReleaseSpinLock( &StoreMutex, oldirql );

                //
                // Copy it to the caller's buffer
                //
                memcpy( OutputBuffer, old->Data, old->Len );

                //
                // Return length of copied info
                //
                IoStatus->Information = old->Len;

                //
                // Deallocate buffer
                //
                MemFreePool( old );

            } else {

                //
                // There is no unread data
                //
                KeReleaseSpinLock( &StoreMutex, oldirql );
				IoStatus->Information = 0;
            }
            break;
 
        default:

            //
            // Unknown control
            // 
            DbgPrint (("Filemon: unknown IRP_MJ_DEVICE_CONTROL\n"));
            IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
            break;
        }

        retval = TRUE;

    } else {

        //
        // Its a call for a file system, so pass it through
        //
        hookExt = DeviceObject->DeviceExtension;

        if( FASTIOPRESENT( hookExt, FastIoDeviceControl )) {
        
            FilemonGetFullPath( FileObject, hookExt, fullPathName );
            if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
            else                 KeQuerySystemTime( &timeResult );

            retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoDeviceControl( 
                FileObject, Wait, InputBuffer, InputBufferLength, OutputBuffer, 
                OutputBufferLength, IoControlCode, IoStatus, hookExt->FileSystem );

            if( FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

                if( TimeIsDuration ) {
                    timeStampComplete = KeQueryPerformanceCounter(NULL);
                    timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
                }
                UpdateStore( InterlockedIncrement(&Sequence), &timeResult, 
                             "%s\tFASTIO_DEVICE_CONTROL\t%s\tIOCTL: 0x%X\t%s", 
                             name, fullPathName,
                             IoControlCode, 
                             retval ?  ErrorString( IoStatus->Status, errorBuf ) : "FAILURE" );
            }
        }
    }

    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoAcquireFile
//
//----------------------------------------------------------------------
VOID FilemonFastIoAcquireFile( PFILE_OBJECT FileObject ) {
    PDEVICE_OBJECT      deviceObject;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    deviceObject = IoGetRelatedDeviceObject( FileObject );

    hookExt = deviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, AcquireFileForNtCreateSection )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireFileForNtCreateSection( 
            FileObject );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), &timeResult, 
                         "%s\tFASTIO_ACQUIRE_FILE\t%s\t\tOK", name, 
                         fullPathName );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
}


//----------------------------------------------------------------------
//
// FilemonFastIoReleaseFile
//
//----------------------------------------------------------------------
VOID FilemonFastIoReleaseFile( PFILE_OBJECT FileObject ) {
    PDEVICE_OBJECT      deviceObject;
    PHOOK_EXTENSION     hookExt;
    CHAR                *fullPathName, name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    deviceObject = IoGetRelatedDeviceObject( FileObject );

    hookExt = deviceObject->DeviceExtension;

    if( FASTIOPRESENT( hookExt, ReleaseFileForNtCreateSection )) {

        fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );

        if( fullPathName ) {

            FilemonGetFullPath( FileObject, hookExt, fullPathName );
        }       
        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );

        hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseFileForNtCreateSection( FileObject );

        if( fullPathName && FilemonGetProcess( name ) && ApplyNameFilter(fullPathName)) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), &timeResult, 
                         "%s\tFASTIO_RELEASE_FILE\t%s\t\tOK", name,
                         fullPathName );
        }
        if ( fullPathName ) MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }
}


//----------------------------------------------------------------------
//
// FilemonFastIoDetachDevice
//
// We get this call when a device that we have hooked is being deleted.
// This happens when, for example, a floppy is formatted. We have
// to detach from it and delete our device. We should notify the GUI
// that the hook state has changed, but its not worth the trouble.
//
//----------------------------------------------------------------------
VOID FilemonFastIoDetachDevice( PDEVICE_OBJECT SourceDevice, PDEVICE_OBJECT TargetDevice ) {
    PHOOK_EXTENSION     hookExt;
    ULONG               i;
    CHAR                name[PROCNAMELEN];
    LARGE_INTEGER       timeStampStart, timeStampComplete, timeResult;

    //
    // See if a device (like a floppy) is being removed out from under us. If so,
    // we have to detach from it before it disappears  
    //
    for( i = 0; i < 26; i++ ) {

        if( SourceDevice == LDriveDevices[i] ) {

            //
            // We've hooked it, so we must detach
            //
            hookExt = SourceDevice->DeviceExtension;

            DbgPrint(("Filemon: Detaching from drive: %c\n", 
                      hookExt->LogicalDrive ));

            IoDetachDevice( TargetDevice );
            IoDeleteDevice( SourceDevice );

            LDriveDevices[i] = NULL;
            LDriveMap[i] = 0;

            return;
        }
    }

    //
    // Now we can pass the call through
    //
    hookExt = SourceDevice->DeviceExtension;

    if( FASTIOPRESENT( hookExt, FastIoDetachDevice )) {

        if( TimeIsDuration ) timeStampStart = KeQueryPerformanceCounter(NULL);
        else                 KeQuerySystemTime( &timeResult );
        hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoDetachDevice( 
            SourceDevice, TargetDevice );

        if( FilemonGetProcess( name ) ) {

            if( TimeIsDuration ) {
                timeStampComplete = KeQueryPerformanceCounter(NULL);
                timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
            }
            UpdateStore( InterlockedIncrement(&Sequence), &timeResult, 
                         "%s\tFASTIO_DETACH_DEVICE\t\t\tOK", 
                         name);
        }
    }
}


//----------------------------------------------------------------------
//     D I S P A T C H   A N D   H O O K   E N T R Y   P O I N T S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// 
// FilemonHookDone
//
// Gets control after a filesystem operation has completed so that
// we can get return status information about it.
//
//----------------------------------------------------------------------
NTSTATUS FilemonHookDone( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp,
                          IN PVOID Context )
{
    PIO_STACK_LOCATION   IrpSp;
    int                  seq = (int)Context;
    CHAR                 errstring[ERRORLEN], errval[ERRORLEN];
    KIRQL                oldirql;
    LARGE_INTEGER        timeStampStart, timeStampComplete, timeResult;

    //
    // A request completed - look at the result 
    //
    IrpSp = IoGetCurrentIrpStackLocation( Irp );    

    //
    // Store the return status in the output buffer. Tag it with the 
    // sequence number so that the GUI can match it with the IRP input information.
    //
    if( FilterOn ) {

        //
        // Quick, get the completion time
        //
        timeStampStart = IrpSp->Parameters.Read.ByteOffset;
        if( IrpSp->Parameters.Read.Length ) {
            timeStampComplete   = KeQueryPerformanceCounter(NULL);
            timeResult.QuadPart = timeStampComplete.QuadPart - timeStampStart.QuadPart;
        } else {
            timeResult = timeStampStart;
        }
        sprintf( errstring, "\t\t\t\t%s", ErrorString( Irp->IoStatus.Status, errval ));
        UpdateStore( seq, &timeResult, errstring );
    }

    //
    // If this was an error on a create call, we have to free the hash entry we
    // created for this name
    //
    if( IrpSp->MajorFunction == IRP_MJ_CREATE && 
        !NT_SUCCESS( Irp->IoStatus.Status )) {

        FilemonFreeHashEntry( IrpSp->FileObject );
    }
  
    //
    // We have finished processing an IRP so decrement oustanding IRP count
    //
    KeAcquireSpinLock( &CountMutex, &oldirql );
    OutstandingIRPCount--;
    KeReleaseSpinLock( &CountMutex, oldirql );

    //
    // Now we have to mark Irp as pending if necessary
    //
    if( Irp->PendingReturned ) {

        IoMarkIrpPending( Irp );
    }
    return Irp->IoStatus.Status;
}


//----------------------------------------------------------------------
//
// FilemonHookRoutine
//
// This routine is the main hook routine where we figure out what
// calls are being sent to the file system.
//
//----------------------------------------------------------------------
NTSTATUS FilemonHookRoutine( PDEVICE_OBJECT HookDevice, IN PIRP Irp )
{
    PIO_STACK_LOCATION  currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
    PIO_STACK_LOCATION  nextIrpStack    = IoGetNextIrpStackLocation(Irp);
    PFILE_OBJECT        fileObject;
    PHOOK_EXTENSION     hookExt;
    LARGE_INTEGER       timeStampStart;
    PCHAR               fullPathName;
    CHAR                controlCodeBuffer[ERRORLEN];
    CHAR                name[PROCNAMELEN];
    ULONG               seq = InterlockedIncrement(&Sequence);
    KIRQL               oldirql;

    //
    // Allocate a buffer
    //
    fullPathName = MemAllocateFromNPagedLookasideList( &PathLookaside );
    fullPathName[0] = 0;

#if DBG
    MemTrackPrintStats("Filemon");
#endif

    //
    // Extract the file object from the IRP
    //
    fileObject    = currentIrpStack->FileObject;

    //
    // Point at the device extension, which contains information on which
    // file system this IRP is headed for
    //
    hookExt = HookDevice->DeviceExtension;

    //
    // If a GUI is up there, get the canonical pathname
    //
    if( FilterOn ) {

        if( currentIrpStack->MajorFunction == IRP_MJ_CREATE ) {

            //
            // Clear any existing fileobject/name association stored in the
            // hash table
            //
            FilemonFreeHashEntry( fileObject );
        } 
        FilemonGetFullPath( fileObject, hookExt, fullPathName );
    }

    // 
    // If measuring absolute time go and get the timestamp.
    // 
    if( !TimeIsDuration ) KeQuerySystemTime( &timeStampStart );
    else                  timeStampStart.QuadPart = 0;

    //
    // Only log it if it passes the filter
    //
    if( FilemonGetProcess( name ) && ApplyNameFilter(fullPathName) ) {

        //
        // Determine what function we're dealing with
        //
        switch( currentIrpStack->MajorFunction ) {

        case IRP_MJ_CREATE:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_CREATE\t%s\tAttributes:%04x Options:%08x", 
                         name, fullPathName,
                         currentIrpStack->Parameters.Create.FileAttributes,
                         currentIrpStack->Parameters.Create.Options );
            break;

        case IRP_MJ_READ:

            if( FilterDef.logreads ) {
                UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_READ\t%s\tOffset: %d Length: %d", 
                             name, fullPathName, 
                             currentIrpStack->Parameters.Read.ByteOffset.LowPart,
                             currentIrpStack->Parameters.Read.Length );
            }
            break;

        case IRP_MJ_WRITE:

            if( FilterDef.logwrites ) {
                UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_WRITE\t%s\tOffset: %d Length: %d", 
                             name, fullPathName, 
                             currentIrpStack->Parameters.Write.ByteOffset.LowPart,
                             currentIrpStack->Parameters.Write.Length );
            }
            break;

        case IRP_MJ_CLOSE:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_CLOSE\t%s\t", name, fullPathName );

            //
            // This fileobject/name association can be discarded now.
            //
            FilemonFreeHashEntry( fileObject );
            break;

        case IRP_MJ_FLUSH_BUFFERS:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_FLUSH\t%s\t", name, fullPathName );
            break;

        case IRP_MJ_QUERY_INFORMATION:
 
            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_QUERY_INFORMATION\t%s\t%s", 
                         name, fullPathName, 
                         FileInformation[currentIrpStack->Parameters.QueryFile.FileInformationClass] );
            break;

        case IRP_MJ_SET_INFORMATION:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_SET_INFORMATION\t%s\t%s", 
                         name, fullPathName,
                         FileInformation[currentIrpStack->Parameters.SetFile.FileInformationClass] );
            break;

        case IRP_MJ_QUERY_EA:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_QUERY_EA\t%s\t", name, fullPathName );
            break;

        case IRP_MJ_SET_EA:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_SET_EA\t%s\t", name, fullPathName );
            break;

        case IRP_MJ_QUERY_VOLUME_INFORMATION:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_QUERY_VOLUME_INFORMATION\t%s\t%s", 
                         name, fullPathName,
                         VolumeInformation[currentIrpStack->Parameters.QueryVolume.FsInformationClass] );
            break;

        case IRP_MJ_SET_VOLUME_INFORMATION:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_SET_VOLUME_INFORMATION\t%s\t", 
                         name, fullPathName );
            break;

        case IRP_MJ_DIRECTORY_CONTROL:

            switch( currentIrpStack->MinorFunction ) {
            case IRP_MN_NOTIFY_CHANGE_DIRECTORY:
                UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_DIRECTORY_CONTROL\t%s\tChange Notify", 
                             name, fullPathName );
                break;
            case IRP_MN_QUERY_DIRECTORY:
                UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_DIRECTORY_CONTROL\t%s\t%s", 
                             name, fullPathName, 
                             FileInformation[((PQUERY_DIRECTORY)&currentIrpStack->Parameters)->FileInformationClass]);
                break; 
            default:
                UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_DIRECTORY_CONTROL\t%s\t", 
                             name, fullPathName );
                break;
            }
            break;

        case IRP_MJ_FILE_SYSTEM_CONTROL:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_FILE_SYSTEM_CONTROL\t%s\t%s", 
                         name, fullPathName, 
                         ControlCodeString( currentIrpStack->Parameters.DeviceIoControl.IoControlCode,
                                            controlCodeBuffer ));
            break;

        case IRP_MJ_SHUTDOWN:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_SHUTDOWN\t\t", name );
            break;

        case IRP_MJ_LOCK_CONTROL:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_LOCK_CONTROL\t%s\tOffset: %d Length: %d",
                         name, fullPathName,
                         ((PLOCK_CONTROL)&currentIrpStack->Parameters)->ByteOffset.LowPart,
                         ((PLOCK_CONTROL)&currentIrpStack->Parameters)->Length ?
                         ((PLOCK_CONTROL)&currentIrpStack->Parameters)->Length->LowPart : 0 );
            break;

        case IRP_MJ_CLEANUP:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_CLEANUP\t%s\t", name, fullPathName );
            break;

        case IRP_MJ_DEVICE_CONTROL:
 
            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_DEVICE_CONTROL\t%s\tIOCTL: 0x%X", name, 
                         fullPathName, currentIrpStack->Parameters.DeviceIoControl.IoControlCode );
            break;

        case IRP_MJ_CREATE_MAILSLOT:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_CREAT_MAILSLOT\t%s\t", 
                         name, fullPathName );
            break;

        case IRP_MJ_QUERY_SECURITY:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_QUERY_SECURITY\t%s\t", 
                         name, fullPathName );
            break;

        case IRP_MJ_SET_SECURITY:

            UpdateStore( seq, &timeStampStart, "%s\tIRP_MJ_SET_SECURITY\t%s\t", 
                         name, fullPathName );
            break;

        default:

            UpdateStore( seq, &timeStampStart, "%s\t*UNKNOWN* 0x%X\t\t", name, currentIrpStack->MajorFunction );
            break;
        }
    } else {

        //
        // Do name processing for the sake of keeping the hash table current
        //
        switch( currentIrpStack->MajorFunction ) {

        case IRP_MJ_CREATE:

            //
            // Clear any existing fileobject/name association stored in the
            // hash table
            //
            FilemonFreeHashEntry( fileObject );
            break;
 
        case IRP_MJ_CLOSE:

            //
            // This fileobject/name association can be discarded now.
            //
            FilemonFreeHashEntry( fileObject );
            break;
        }        
    }

    //
    // Free the buffer
    //
    if ( fullPathName ) {

        MemFreeToNPagedLookasideList( &PathLookaside, fullPathName );
    }

    //
    // Copy parameters down to next level in the stack for the driver below us
    //
    *nextIrpStack = *currentIrpStack;

    //
    // If an unload isn't in progress, we should register a completion callback
    // so that the IRP's return status can be examined.
    //
    KeAcquireSpinLock( &CountMutex, &oldirql );

    if( !UnloadInProgress && FilterOn ) {

        //
        // Increment the outstanding IRP count since this IRP will be headed
        // for our completion routine
        //
        OutstandingIRPCount++;

        //
        // Grab the time stamp and store it in the current stack location. This
        // is legal since the stack location is ours, and we're done looking at 
        // the parameters. This makes it easy to pass this to the completion routine. The
        // DiskPerf example in the NT DDK uses this trick.
        //
        if( TimeIsDuration ) {
            currentIrpStack->Parameters.Read.ByteOffset = KeQueryPerformanceCounter((PVOID)NULL);
            currentIrpStack->Parameters.Read.Length     = 1;
        } else {
            currentIrpStack->Parameters.Read.ByteOffset = timeStampStart;
            currentIrpStack->Parameters.Read.Length     = 0;
        }

        IoSetCompletionRoutine( Irp, FilemonHookDone, (void *)seq, TRUE, TRUE, TRUE );

    } else {

        //
        // Set no completion routine
        //
        IoSetCompletionRoutine( Irp, FilemonHookDone, NULL, FALSE, FALSE, FALSE );
    }

    KeReleaseSpinLock( &CountMutex, oldirql );

    //
    // Return the results of the call to the caller
    //
    return IoCallDriver( hookExt->FileSystem, Irp );
}


//----------------------------------------------------------------------
//
// FilemonDeviceRoutine
//
// In this routine we handle requests to our own device. The only 
// requests we care about handling explicitely are IOCTL commands that
// we will get from the GUI. We also expect to get Create and Close 
// commands when the GUI opens and closes communications with us.
//
//----------------------------------------------------------------------
NTSTATUS FilemonDeviceRoutine( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp )
{
    PIO_STACK_LOCATION  irpStack;
    PVOID               inputBuffer;
    PVOID               outputBuffer;
    ULONG               inputBufferLength;
    ULONG               outputBufferLength;
    ULONG               ioControlCode;

    //
    // Go ahead and set the request up as successful
    //
    Irp->IoStatus.Status      = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    //
    // Get a pointer to the current location in the Irp. This is where
    // the function codes and parameters are located.
    //
    irpStack = IoGetCurrentIrpStackLocation (Irp);

    //
    // Get the pointer to the input/output buffer and its length
    //
    inputBuffer     = Irp->AssociatedIrp.SystemBuffer;
    inputBufferLength   = irpStack->Parameters.DeviceIoControl.InputBufferLength;
    outputBuffer    = Irp->AssociatedIrp.SystemBuffer;
    outputBufferLength  = irpStack->Parameters.DeviceIoControl.OutputBufferLength;
    ioControlCode   = irpStack->Parameters.DeviceIoControl.IoControlCode;

    switch (irpStack->MajorFunction) {
    case IRP_MJ_CREATE:

        DbgPrint(("Filemon: IRP_MJ_CREATE\n"));

        // 
        // Start the sequence number at 0
        // 
        Sequence = 0;
        break;

    case IRP_MJ_CLOSE:

        DbgPrint(("Filemon: IRP_MJ_CLOSE\n"));

        //
        // A GUI is closing communication
        //
        FilterOn  = FALSE;

        //
        // If the GUI has no more references to us, reset the output
        // buffers and hash table.
        //
        FilemonResetStore();
        FilemonHashCleanup();

        //
        // Disconnect from file systems
        //
        HookDriveSet( 0, DeviceObject->DriverObject );
        break;

    case IRP_MJ_DEVICE_CONTROL:

        //
        // This path will never execute because we have registered a 
        // fast I/O path for device control. That means that the fast I/O entry 
        // point will ALWAYS be called for Device Control operations
        //
        DbgPrint (("Filemon: IRP_MJ_DEVICE_CONTROL\n"));

        //
        // Get output buffer if its passed as an MDL
        //
        if( Irp->MdlAddress ) {

            outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );
        }

        //
        // Its a request from the GUI. Simply call our fast handler.
        //
        FilemonFastIoDeviceControl( irpStack->FileObject, TRUE,
                                    inputBuffer, inputBufferLength, 
                                    outputBuffer, outputBufferLength,
                                    ioControlCode, &Irp->IoStatus, DeviceObject );
        break;
    }

    //
    // Complete the IRP
    //
    IoCompleteRequest( Irp, IO_NO_INCREMENT );
    return STATUS_SUCCESS;   
}


//----------------------------------------------------------------------
//
// FilemonDispatch
//
// Based on which device the Irp is destined for we call either the
// filesystem filter function, or our own device handling routine.
//
//----------------------------------------------------------------------
NTSTATUS FilemonDispatch( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp )
{
    //
    // Determine if its a request from the GUI to us, or one that is
    // directed at a file system driver that we've hooked
    //
    if ( GUIDevice != DeviceObject ) {

        return( FilemonHookRoutine( DeviceObject, Irp ));

    } else {

        return( FilemonDeviceRoutine( DeviceObject, Irp ));

    }
}


//----------------------------------------------------------------------
//
// FilemonUnload
//
// Our job is done - time to leave.
//
//----------------------------------------------------------------------
VOID FilemonUnload( IN PDRIVER_OBJECT DriverObject )
{
    WCHAR                  deviceLinkBuffer[]  = L"\\DosDevices\\Filemon";
    UNICODE_STRING         deviceLinkUnicodeString;

    //
    // Disconnect from file systems
    //
    HookDriveSet( 0, DriverObject );

    //
    // Delete the symbolic link for our GUI device
    //
    RtlInitUnicodeString( &deviceLinkUnicodeString, deviceLinkBuffer );
    IoDeleteSymbolicLink( &deviceLinkUnicodeString );

    DbgPrint(("Filemon.SYS: unloading\n"));

    //
    // Delete the device object, making sure that the GUI device
    // object is always deleted.
    //
    if ( GUIDevice == DriverObject->DeviceObject )  {

        IoDeleteDevice( DriverObject->DeviceObject );

    } else {

        IoDeleteDevice( DriverObject->DeviceObject );
        IoDeleteDevice( GUIDevice ); 
    }

    DbgPrint(("Filemon.SYS: deleted devices\n"));

    //
    // Now we can free any memory that is allocatedp
    //
    FilemonHashCleanup();
    FilemonFreeStore();

    DbgPrint(("Filemon.SYS: freed memory\n"));
}


//----------------------------------------------------------------------
//
// DriverEntry
//
// Installable driver initialization. Here we just set ourselves up.
//
//----------------------------------------------------------------------
NTSTATUS DriverEntry(IN PDRIVER_OBJECT DriverObject, IN PUNICODE_STRING RegistryPath )
{
    NTSTATUS                ntStatus;
    WCHAR                   deviceNameBuffer[]  = L"\\Device\\Filemon";
    UNICODE_STRING          deviceNameUnicodeString;
    WCHAR                   deviceLinkBuffer[]  = L"\\DosDevices\\Filemon";
    UNICODE_STRING          deviceLinkUnicodeString;
    ULONG                   i;

    DbgPrint (("Filemon.SYS: entering DriverEntry\n"));

    //
    // Initialize memory tracker
    //
    MemTrackInit();

    //
    // If not NT 4.0 Final Release, shorten the Fast I/O table so that Filemon 
    // will work on the Betas and Release Candidates
    //
    if( *NtBuildNumber < NT4FINAL ) {

        FastIOHook.SizeOfFastIoDispatch = (ULONG) &FastIOHook.FastIoQueryNetworkOpenInfo - 
            (ULONG) &FastIOHook;
    } 

    //    
    // Setup the device name
    //    
    RtlInitUnicodeString (&deviceNameUnicodeString,
                          deviceNameBuffer );

    //
    // Create the device used for GUI communications
    //
    ntStatus = IoCreateDevice ( DriverObject,
                                0,
                                &deviceNameUnicodeString,
                                FILE_DEVICE_FILEMON,
                                0,
                                TRUE,
                                &GUIDevice );

    //
    // If successful, make a symbolic link that allows for the device
    // object's access from Win32 programs
    //
    if (NT_SUCCESS(ntStatus)) {

        //
        // Create a symbolic link that the GUI can specify to gain access
        // to this driver/device
        //
        RtlInitUnicodeString (&deviceLinkUnicodeString,
                              deviceLinkBuffer );
        ntStatus = IoCreateSymbolicLink (&deviceLinkUnicodeString,
                                         &deviceNameUnicodeString );
        if (!NT_SUCCESS(ntStatus)) {

            DbgPrint (("Filemon.SYS: IoCreateSymbolicLink failed\n"));
        
        }

        //
        // Create dispatch points for all routines that must be handled. 
        // All entry points are registered since we might filter a
        // file system that processes all of them.
        //
        DriverObject->MajorFunction[IRP_MJ_READ]            = 
            DriverObject->MajorFunction[IRP_MJ_CLOSE]           =
            DriverObject->MajorFunction[IRP_MJ_WRITE]           =
            DriverObject->MajorFunction[IRP_MJ_CREATE]          =
            DriverObject->MajorFunction[IRP_MJ_FLUSH_BUFFERS]   =
            DriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION]   =
            DriverObject->MajorFunction[IRP_MJ_SET_INFORMATION] =
            DriverObject->MajorFunction[IRP_MJ_QUERY_EA]        =
            DriverObject->MajorFunction[IRP_MJ_SET_EA]          =
            DriverObject->MajorFunction[IRP_MJ_QUERY_VOLUME_INFORMATION] = 
            DriverObject->MajorFunction[IRP_MJ_SET_VOLUME_INFORMATION] =
            DriverObject->MajorFunction[IRP_MJ_DIRECTORY_CONTROL]   =
            DriverObject->MajorFunction[IRP_MJ_FILE_SYSTEM_CONTROL] =
            DriverObject->MajorFunction[IRP_MJ_QUERY_SECURITY]  =
            DriverObject->MajorFunction[IRP_MJ_SET_SECURITY]    =
            DriverObject->MajorFunction[IRP_MJ_SHUTDOWN]        =
            DriverObject->MajorFunction[IRP_MJ_LOCK_CONTROL]    =
            DriverObject->MajorFunction[IRP_MJ_CLEANUP]         =
            DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL]  = FilemonDispatch;
        DriverObject->DriverUnload                          = FilemonUnload;

        //
        // Set up the Fast I/O dispatch table
        //
        DriverObject->FastIoDispatch = &FastIOHook;
    }

    //
    // If something went wrong, cleanup the device object and don't load
    //
    if (!NT_SUCCESS(ntStatus)) {

        DbgPrint(("Filemon: Failed to create our device!\n"));

        if( GUIDevice ) {

            IoDeleteDevice( GUIDevice );
        }
        return ntStatus;
    }

    //
    // Initialize the drive map 
    //
    for(i = 0; i < 26; i++ )        LDriveMap[i] = 0;

    //
    // Initialize the name hash table
    //
    for(i = 0; i < NUMHASH; i++ )   HashTable[i] = NULL;

    //
    // Find the process name offset
    //
    ProcessNameOffset = FilemonGetProcessNameOffset();

    //
    // Initialize the mutexes
    //
    KeInitializeSpinLock( &StoreMutex );
    KeInitializeSpinLock( &CountMutex );
    KeInitializeSpinLock( &HashMutex );
    KeInitializeSpinLock( &FilterMutex );

    //
    // Initialize our lookaside lists
    //
    MemInitializeNPagedLookasideList( &HashLookaside,
                                     NULL, NULL, 0, 
                                     sizeof(HASH_ENTRY), 'nomF',
                                     256 );
    MemInitializeNPagedLookasideList( &PathLookaside,
                                     NULL, NULL, 0, 
                                     2*MAXPATHLEN, 'nomF',
                                     256 );        
    //
    // Allocate the first output buffer
    //
    Store   = MemAllocatePool( NonPagedPool, sizeof(*Store) );
    if ( !Store ) {

        // 
        // Oops - we can't do anything without at least one buffer
        // 
        IoDeleteDevice( GUIDevice );
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    // 
    // Set the buffer pointer to the start of the buffer just allocated
    // 
    Store->Len  = 0;
    Store->Next = NULL;
    NumStore = 1;

    return ntStatus;
}
