;=====================================================================
;
; UPS0206 : XY LCD-BASED PLOTTER
;
; Target processor : PIC18F252
; Author           : (name deleted for the contest)
; Copyright 2002
;
; This firmware is used with Microchip's boot loader (AN851), and is
; then located from $200
;
;=====================================================================

	TITLE		"XY LCD-BASED PLOTTER / (c) 2002"

	PROCESSOR	18F252

	RADIX		HEX
	ERRORLEVEL	0



;=====================================================================
;
;
;
; Header section
;
;
;
;=====================================================================

;---------------------------------------------------------------------
; Configuration
;---------------------------------------------------------------------

	__CONFIG	_CONFIG1H, _HSPLL_OSC_1H

;---------------------------------------------------------------------
; Includes
;---------------------------------------------------------------------

	#include P18F252.INC

;---------------------------------------------------------------------
; Constants
;---------------------------------------------------------------------

#define	BASE	0x200			; Base program memory

; Port A bits
#define	RA_X_BIT	0
#define RA_Y_BIT	1
#define	RA_AUX1_BIT	2
#define	RA_AUX2_BIT	3
#define	RA_VEEON_BIT	4
#define	RA_DIGAUX4_BIT	5

; Port B bits
#define RB_LCDDATA_MASQ	0x0F
#define RB_DIGAUX1_MASQ	0x0F
#define	RB_DIGAUX2_MASQ	0x70
#define RB_DIGAUX2_SHFT	4
#define RB_LCDXSCL_BIT	7

; Port C bits
#define RC_KEYHOLD_BIT	0
#define RC_KEYMODE_BIT	1
#define RC_KEYACC_BIT	2
#define RC_LCDDIN_BIT	3
#define RC_LCDLP_BIT	4
#define RC_DIGAUX3_BIT	5

; ADC Channels
#define ADC_CHAN_X	0x00
#define ADC_CHAN_Y	0x08
#define ADC_CHAN_AUX1	0x10
#define ADC_CHAN_AUX2	0x18

; Operating modes
#define	MODE_HOLD_OFF	0x00
#define	MODE_HOLD_ON	0x01
#define MODE_DISP_SAMP	0x00
#define MODE_DISP_MEAN	0x01
#define MODE_DISP_MAXI	0x02
#define MODE_DISP_PEAK	0x03
#define MODE_ACC_OFF	0x00
#define MODE_ACC_ON	0x01

; Increments for the TBLDRAW table
#define	START_TBLMIN	.16
#define	START_TBLMAX	.20

;---------------------------------------------------------------------
; Macros
;---------------------------------------------------------------------

SEND_1_PULSE macro
	bsf	PORTB,RB_LCDXSCL_BIT
	bcf	PORTB,RB_LCDXSCL_BIT
	endm

SEND_2_PULSES macro
	SEND_1_PULSE
	SEND_1_PULSE
	endm

SEND_3_PULSES macro
	SEND_1_PULSE
	SEND_1_PULSE
	SEND_1_PULSE
	endm

SEND_4_PULSES macro
	SEND_1_PULSE
	SEND_1_PULSE
	SEND_1_PULSE
	SEND_1_PULSE
	endm

SEND_5_PULSES macro
	SEND_1_PULSE
	SEND_1_PULSE
	SEND_1_PULSE
	SEND_1_PULSE
	SEND_1_PULSE
	endm

;---------------------------------------------------------------------
; Variables
;---------------------------------------------------------------------

; Bank 0 (0x000 to 0x0FF), first half is Access Bank

	cblock 0x000

	  bank0:0			; used as a flag

	  operating_mode_hold		; operating modes
	  operating_mode_acc		; (value from MODE_XXXX_XXX)
	  operating_mode_disp
	
	  adc_val_l, adc_val_h		; ADC output values

	  xcol				; current x index (columns)

	  tmp_send_w_pulses		; temporary for send_w_pulses

	  m, n, cptbytes, v		; temporary for dispsample
	  ystart,ystop
	  ystart_prev,ystop_prev
	  i

	  tmpuart			; temporary for manage_uart

	  TEM0,TEM1,TEM2,TEM3		; temporary for bin to ascii conversion
	  TEMPL,TEMPH
	  convasc4,convasc3,convasc2
	  convasc1, convasc0

	  tmp_adcdelay			; temporary for adc_delay

	  prev_keys			; previous state of the keys	
	  new_keys			; new state of keys

	  disp_item			; selector of text item to display

	  inc_txt, xcol_div8		; exec_text variables
	  adr_txt0l,adr_txt0h,adr_txt1l
	  adr_txt1h,adr_txt2l,adr_txt2h

	  line_batch_step		; current state of the line_batch machine
	  line_batch_x, line_batch_y	; previous x 
	  line_oldx, line_oldy		; tmp for line_batch

	  tmp				; generic tmp

	endc

; Bank 1 (0x100 to 0x1FF)

	cblock	0x100

	  bank1:0			; used as a flag

	  txt_line0:.30			; ASCII chars for text line 0
	  txt_line1:.30			; ASCII chars for text line 1
	  txt_line2:.30			; ASCII chars for text line 2
					; WARNING : exectext assumes that these
					; three lines are 30 char long and 
					; start at 0x100 (hardcoded for speed !)
	endc

; Bank 2 (0x200 to 0x2FF)

	cblock	0x200

 	  bank2:0			; used as a flag

	  minval:.256			; Minimun values for each X

	endc

; Bank 3 (0x300 to 0x3FF)

	cblock	0x300

 	  bank3:0			; used as a flag

	  maxval:.256			; Maximun values for each X

	endc

; Bank 4 (0x400 to 0x4FF)

	cblock	0x400

 	  bank4:0			; used as a flag

	  sampleval:.256		; Sampled values for each X

	endc

; Bank 5 (0x500 to 0x5FF)

	cblock	0x500

 	  bank5:0			; used as a flag

	  tbldraw:.256			; copy of TBLDRAW (segments)

	endc

;---------------------------------------------------------------------
; Vectors area
;---------------------------------------------------------------------

	org 	BASE 
	bra	reset_vector

	org	BASE+0x08
	bra	irq_high

	org	BASE+0x18
	bra	irq_low

;---------------------------------------------------------------------
; Program area
;---------------------------------------------------------------------

	org	BASE+0x20

;=====================================================================
;
;
;
; Strings
;
;
;
;=====================================================================

FIXED0_STR
	; 30 chars, fixed text
	data 	"FREQ=     MHz  REF = -   dBm  "

FIXED1_STR
	; 30 chars, fixed text
	data 	"SCAN=      /d  VERT=     dB/d "

FIXED2_STR
	; 30 chars, fixed text
	data	"RESL=      Hz                 "

DISP_STR
	; 4 chars per display mode, 4 display modes
	data	"SAMPMEANMAXIPEAK"

ACC_STR
	; 3 chars, accumulation off/on
	data	"   ACC"

HOLD_STR
	; 2 chars, hold off/on
	data	"  HD"

DA1_STR
	; 5 chars per scan width, 16 scan widths
	data	"0-2 G100 M 50 M 20 M 10 M  5 M  2 M  1 M500 K200 K100 K 50 K 20 K 10 K  5 KFIXED"

DA2_STR
	; 5 chars per resolution, 8 resolutions
	data	"  3 M  1 M300 K100 K 30 K 10 K  3 K  1 K"

DA3_STR
	; 3 chars, filtering off/on
	data	"   FLT"

DA4_STR
	; 2 chars, vertical extension off/on
	data	"10 2"

;=====================================================================
;
;
;
; Utility routines
;
;
;
;=====================================================================

;---------------------------------------------------------------------
; Send one byte through the uart
;---------------------------------------------------------------------

uart_send_byte
wait_transmit
	btfss	PIR1, TXIF
	bra	wait_transmit
	movwf	TXREG
	retlw	0

;---------------------------------------------------------------------
; Check if one byte is received on the serial port
; W=1 if byte available
; Received char is then in RCREG
;---------------------------------------------------------------------

uart_check_received
	btfss	PIR1, RCIF
	retlw	0
	bcf	PIR1, RCIF
	retlw	1

;---------------------------------------------------------------------
; Select ADC input channel
; Channel selected in W when called (ADC_CHAN_XXX)
;---------------------------------------------------------------------

adc_select_chan
	movwf	ADCON0			; select channel
	movlw	0xC2			; right justified, fosc/64, 5/0
	movwf	ADCON1
	movf	ADCON0,W
	iorlw	0x80			; fosc/64
	movwf	ADCON0
	iorlw	0x01			; ADON
	movwf	ADCON0
	retlw 0

;---------------------------------------------------------------------
; Launch ADC convertion
; Must be called not sooner than 13s after call to adc_select_chan
;---------------------------------------------------------------------

adc_launch_conv
	bsf	ADCON0,2		; set bit GO/DONE
	retlw 0

;---------------------------------------------------------------------
; Read ADC converted value
; Will wait if called sooner than 19.2s after call to adc_launch_conv
; Output value (right justified) is in adc_val_h/adc_val_l
;---------------------------------------------------------------------

adc_read_result
	btfsc	ADCON0,2		; wait for conversion done
	goto	adc_read_result
	movff	ADRESH,adc_val_h
	movff	ADRESL,adc_val_l
	movlw	0xC7			; all inputs back as digital
	movwf	ADCON1
	retlw 0

;---------------------------------------------------------------------
; Safe delay between ADC operations (25s)
;---------------------------------------------------------------------

adc_delay
	movlw	.83			; 83x300ns=25microsec
	movwf	tmp_adcdelay
loop_adc_delay
	decfsz	tmp_adcdelay		; 3 instr per loop, or 300ns
	goto	loop_adc_delay
	retlw	0


;=====================================================================
;
;
;
; System management
;
;
;
;=====================================================================

;---------------------------------------------------------------------
; Vectors management
;---------------------------------------------------------------------

reset_vector
	bra	entry

irq_high
	retfie

irq_low
	retfie

;---------------------------------------------------------------------
; CPU initialisation (registers, ports, I/Os, etc)
;---------------------------------------------------------------------

init_cpu
	; Interrupt initialisation
	movlw	0x00
	movwf	INTCON			; Disable all interrupts
	movwf	INTCON2			; pullups enables on port B
	movwf	INTCON3

	; Port directions & pullups
	movlw	(0xFF-0x10)		; only bit 4 is an output
	movwf	TRISA
	clrf	LATA
	bsf	PORTA,RA_VEEON_BIT

	movlw	0x70			; only bits 4-5-6 are input
	movwf	TRISB
	clrf	LATB

	movlw	B'10100111'		; bits 7,5,2-1-0 are inputs
	movwf	TRISC
	clrf	LATC

	; Timer0 : unused
	; Timer1 : unused
	; Timer2 : unused
	; Timer3 : unused
	; CCP : unused
	; MSSP : unused

	; USART configuration : 19200bps/8/N/1 (SPBRG=129 hs @ 40MHz)
	; transmit & receive

	movlw	.129
	movwf	SPBRG
	movlw	b'10010000'		; Setup rx and tx
	movwf	RCSTA
	movlw	b'00100110'		
	movwf	TXSTA

	; ADC configuration
	;(done as part of ADC routines)

	; watchdog & brownout configuration
	; default

	retlw	0


;=====================================================================
;
;
;
; Applicative program
;
;
;
;=====================================================================


;---------------------------------------------------------------------
; Main program
;---------------------------------------------------------------------

entry

	call	init_cpu		; initializations
	call	check_boot		; go back to boot mode ?
	call	init_variables		; variables initialisation
	call	lcd_on			; switch Vee
	bra	main_loop

;---------------------------------------------------------------------
; Check if request to go back to boot mode
;
; Detected by simultaneous press on the three keys at bootup
; If so, change back the last data byte and reset
;---------------------------------------------------------------------

check_boot
	; Any key released ?
	btfsc	PORTC,RC_KEYHOLD_BIT
	bra	check_boot_return
	btfsc	PORTC,RC_KEYMODE_BIT
	bra	check_boot_return
	btfsc	PORTC,RC_KEYACC_BIT
	bra	check_boot_return

	; if not, bootmode is requested : reset last data location
	setf	EEADR
	setf	EEDATA
	movlw	0x04
	movwf	EECON1
	movlw	0x55
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf	EECON1, WR
datawriteloop
	nop
	btfsc	EECON1,WR
	bra	datawriteloop

	; and reset
	reset

check_boot_return
	retlw	0

;---------------------------------------------------------------------
; Transmit identity message
;---------------------------------------------------------------------

send_hello
	movlw	'X'
	call	uart_send_byte
	movlw	'Y'
	call	uart_send_byte
	movlw	'0'
	call	uart_send_byte
	movlw	'1'
	call	uart_send_byte
	movlw	'\n'
	call	uart_send_byte
	retlw	0

;---------------------------------------------------------------------
; Initialisation of all variables
;---------------------------------------------------------------------

init_variables
	movlw	MODE_DISP_SAMP		; Default mode
	movwf	operating_mode_disp
	movlw	MODE_ACC_OFF
	movwf	operating_mode_acc
	movlw	MODE_HOLD_OFF
	movwf	operating_mode_hold

	lfsr	0,minval		;initialize minval
loopinitmin
	BANKSEL	minval
	movlw	0xFF
	movwf	INDF0
	incfsz	FSR0L
	bra	loopinitmin

	lfsr	0,maxval		;initialize maxval
loopinitmax
	BANKSEL	maxval
	clrf	INDF0
	incfsz	FSR0L
	bra	loopinitmax

	lfsr	0,sampleval		;initialize sampleval
loopinitsample
	BANKSEL	sampleval
	movlw	0x80
	movwf	INDF0
	incfsz	FSR0L
	bra	loopinitsample

	movf	PORTC,W			; Initial status of the keys
	movwf	prev_keys

	clrf	line_batch_step
	clrf	line_batch_x		; Initial line_batch status

	; Initialize text to fixed text
	; "FREQ=     MHz  REF =     dBm  "
	; "SCAN=      /d  VERT=     dB/d "
	; "RESL=      Hz                 "

	movlw	UPPER(FIXED0_STR)	; point to string table, first line
   	movwf	TBLPTRU
   	movlw	HIGH(FIXED0_STR)
  	movwf	TBLPTRH
   	movlw	LOW(FIXED0_STR)
   	movwf	TBLPTRL
	movlw	.30
	movwf	tmp
	lfsr	0,txt_line0
fixed0
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line0
	movwf	INDF0
	BANKSEL	bank0           
	incf	FSR0L
	decfsz	tmp
	bra	fixed0

	movlw	UPPER(FIXED1_STR)	; point to string table, second line
   	movwf	TBLPTRU
   	movlw	HIGH(FIXED1_STR)
  	movwf	TBLPTRH
   	movlw	LOW(FIXED1_STR)
   	movwf	TBLPTRL
	movlw	.30
	movwf	tmp
	lfsr	0,txt_line1
fixed1
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	INDF0
	BANKSEL	bank0           
	incf	FSR0L
	decfsz	tmp
	bra	fixed1

	movlw	UPPER(FIXED2_STR)	; point to string table, third line
   	movwf	TBLPTRU
   	movlw	HIGH(FIXED2_STR)
  	movwf	TBLPTRH
   	movlw	LOW(FIXED2_STR)
   	movwf	TBLPTRL
	movlw	.30
	movwf	tmp
	lfsr	0,txt_line2
fixed2
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	INDF0
	BANKSEL	bank0           
	incf	FSR0L
	decfsz	tmp
	bra	fixed2

	; Copy TBLDRAW (ROM) into tbldraw (RAM)
	movlw	UPPER(TBLDRAW)	; point to table
   	movwf	TBLPTRU
   	movlw	HIGH(TBLDRAW)
  	movwf	TBLPTRH
   	movlw	LOW(TBLDRAW)
   	movwf	TBLPTRL
	lfsr	0,tbldraw
	clrf	tmp
loopcopytbldraw
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	tbldraw
	movwf	INDF0
	BANKSEL	bank0           
	incf	FSR0L
	decfsz	tmp
	bra	loopcopytbldraw

	retlw	0

;---------------------------------------------------------------------
; Turn on the LCD power (Vee)
;---------------------------------------------------------------------

lcd_on
	bcf	PORTA,RA_VEEON_BIT
	retlw	0

;---------------------------------------------------------------------
; main loop
; (never return)
;
; Overall timing is the following :
;   13.4ms = Overall loop time (frame refresh, should be 12 to 15ms)
;	200s = frame batch (auxiliary A/D conversions, push-buttons, etc)
;	13.2ms - display of 240 lines 
;		55s/line x 240 lines
;			6s  - line batch (X and Y A/D and min/max calculations)
;			45s - display of one LCD line (320 pixels, per nibbles)
;				5s  - loop management
;				1s  - display of the lower frame border
;				27s - display of the curve data (256 pixels)
;				1s  - display of the higher frame border
;				11s - display of the 3 lines of text (24 pixels)
;			4s  - loop management
;
; Average time to manage each nibble in the curve data display :
; 27s/64=420ns (4.2 instructions...)
;---------------------------------------------------------------------

main_loop
	clrwdt				; clear watchdog
	call	frame_batch		; execute one-time-per-frame functions (3ms)
	bsf	PORTC,RC_LCDDIN_BIT	; DIN=1 (start of frames signal)
	clrf	xcol			; main loop for x=0 to 239 (42s per loop)

loopx
	call	line_batch		; execute one-time-per-line functions (10s)
	
	bsf	PORTC,RC_LCDLP_BIT	; LP=1 (pulse = start of line)
	bcf	PORTC,RC_LCDLP_BIT	; LP=0 
	nop				; to respect DIN hold time
	bcf	PORTC,RC_LCDDIN_BIT	; used for the first line)

	movlw	.120
	cpfslt	xcol			; two segemnts as calculated jumps on one page
	bra	execpart2

	goto	loopaligned1
loopnotaligned1
	org	(1 + high loopnotaligned1)*.256	; MUST START ON A PAGE BOUNDARY
loopaligned1

	movlw	high loopaligned1
	movwf	PCLATH			; high byte of new PC should be defined			
	
	movf	xcol,W			; unloop columns, instruction N+1 will be
	bcf	STATUS,C
	rlcf	WREG
	addwf	PCL,F			; executed if xcol=0

	bra	execcol000
	bra	execcol001
	bra	execcol002
	bra	execcol003		; Each function should return in
	bra	execcol004		; less than 25s, after having
	bra	execcol005		; pulsed exactly 80-9=71 times the XSCL line
	bra	execcol006
	bra	execcol007
	bra	execcol008
	bra	execcol009
	bra	execcol010
	bra	execcol011
	bra	execcol012
	bra	execcol013
	bra	execcol014
	bra	execcol015
	bra	execcol016
	bra	execcol017
	bra	execcol018
	bra	execcol019
	bra	execcol020
	bra	execcol021
	bra	execcol022
	bra	execcol023
	bra	execcol024
	bra	execcol025
	bra	execcol026
	bra	execcol027
	bra	execcol028
	bra	execcol029
	bra	execcol030
	bra	execcol031
	bra	execcol032
	bra	execcol033
	bra	execcol034
	bra	execcol035
	bra	execcol036
	bra	execcol037
	bra	execcol038
	bra	execcol039
	bra	execcol040
	bra	execcol041
	bra	execcol042
	bra	execcol043
	bra	execcol044
	bra	execcol045
	bra	execcol046
	bra	execcol047
	bra	execcol048
	bra	execcol049
	bra	execcol050
	bra	execcol051
	bra	execcol052
	bra	execcol053
	bra	execcol054
	bra	execcol055
	bra	execcol056
	bra	execcol057
	bra	execcol058
	bra	execcol059
	bra	execcol060
	bra	execcol061
	bra	execcol062
	bra	execcol063
	bra	execcol064
	bra	execcol065
	bra	execcol066
	bra	execcol067
	bra	execcol068
	bra	execcol069
	bra	execcol070
	bra	execcol071
	bra	execcol072
	bra	execcol073
	bra	execcol074
	bra	execcol075
	bra	execcol076
	bra	execcol077
	bra	execcol078
	bra	execcol079
	bra	execcol080
	bra	execcol081
	bra	execcol082
	bra	execcol083
	bra	execcol084
	bra	execcol085
	bra	execcol086
	bra	execcol087
	bra	execcol088
	bra	execcol089
	bra	execcol090
	bra	execcol091
	bra	execcol092
	bra	execcol093
	bra	execcol094
	bra	execcol095
	bra	execcol096
	bra	execcol097
	bra	execcol098
	bra	execcol099
	bra	execcol100
	bra	execcol101
	bra	execcol102
	bra	execcol103
	bra	execcol104
	bra	execcol105
	bra	execcol106
	bra	execcol107
	bra	execcol108
	bra	execcol109
	bra	execcol110
	bra	execcol111
	bra	execcol112
	bra	execcol113
	bra	execcol114
	bra	execcol115
	bra	execcol116
	bra	execcol117
	bra	execcol118
	bra	execcol119		; Must still be in the same page than 001 !

execpart2
	goto	loopaligned2
loopnotaligned2
	org	(1 + high loopnotaligned2)*.256	; MUST START ON A PAGE BOUNDARY
loopaligned2
	movlw	high loopaligned2
	movwf	PCLATH			; high byte of new PC should be defined			
	
	movf	xcol,W			; unloop columns, instruction N+1 will be
	sublw	.120
	negf	WREG
	bcf	STATUS,C
	rlcf	WREG
	addwf	PCL,F			; executed if xcol=0

	bra	execcol120
	bra	execcol121
	bra	execcol122
	bra	execcol123
	bra	execcol124
	bra	execcol125
	bra	execcol126
	bra	execcol127
	bra	execcol128
	bra	execcol129
	bra	execcol130
	bra	execcol131
	bra	execcol132
	bra	execcol133
	bra	execcol134
	bra	execcol135
	bra	execcol136
	bra	execcol137
	bra	execcol138
	bra	execcol139
	bra	execcol140
	bra	execcol141
	bra	execcol142
	bra	execcol143
	bra	execcol144
	bra	execcol145
	bra	execcol146
	bra	execcol147
	bra	execcol148
	bra	execcol149
	bra	execcol150
	bra	execcol151
	bra	execcol152
	bra	execcol153
	bra	execcol154
	bra	execcol155
	bra	execcol156
	bra	execcol157
	bra	execcol158
	bra	execcol159
	bra	execcol160
	bra	execcol161
	bra	execcol162
	bra	execcol163
	bra	execcol164
	bra	execcol165
	bra	execcol166
	bra	execcol167
	bra	execcol168
	bra	execcol169
	bra	execcol170
	bra	execcol171
	bra	execcol172
	bra	execcol173
	bra	execcol174
	bra	execcol175
	bra	execcol176
	bra	execcol177
	bra	execcol178
	bra	execcol179
	bra	execcol180
	bra	execcol181
	bra	execcol182
	bra	execcol183
	bra	execcol184
	bra	execcol185
	bra	execcol186
	bra	execcol187
	bra	execcol188
	bra	execcol189
	bra	execcol190
	bra	execcol191
	bra	execcol192
	bra	execcol193
	bra	execcol194
	bra	execcol195
	bra	execcol196
	bra	execcol197
	bra	execcol198
	bra	execcol199
	bra	execcol200
	bra	execcol201
	bra	execcol202
	bra	execcol203
	bra	execcol204
	bra	execcol205
	bra	execcol206
	bra	execcol207
	bra	execcol208
	bra	execcol209
	bra	execcol210
	bra	execcol211
	bra	execcol212
	bra	execcol213
	bra	execcol214
	bra	execcol215
	bra	execcol216
	bra	execcol217
	bra	execcol218
	bra	execcol219
	bra	execcol220
	bra	execcol221
	bra	execcol222
	bra	execcol223
	bra	execcol224
	bra	execcol225
	bra	execcol226
	bra	execcol227
	bra	execcol228
	bra	execcol229
	bra	execcol230
	bra	execcol231
	bra	execcol232
	bra	execcol233
	bra	execcol234
	bra	execcol235
	bra	execcol236
	bra	execcol237
	bra	execcol238
	bra	execcol239		; Must still be in the same page than 120 !

returnfromexeccol
loopnext
	call	exectext		; generates the "text" part of the bitmap
					; should return in less than 5s, after having
					; pulsed exactly 9 times the XSCL line
					; (3 times per text line, giving 12 pixel heigh)

	incf	xcol,F			; next x (0.5s)
	movlw	.240
	cpfseq	xcol
	goto	loopx
	
	goto	main_loop		; and loop until end of the world


;---------------------------------------------------------------------
; Send W pulses to the XSCK line (W=0 to 80)
; Must be a very time optimized routine !
; Execution duration : 100ns x (2xW + 20) for W<60
; Average with W=20 (worst case) giving 3 instructions/pulse (300ns)
;---------------------------------------------------------------------

send_w_pulses
	movwf	tmp_send_w_pulses
	movlw	.60
	cpfsgt	tmp_send_w_pulses
	bra	send_upto60_pulses	; less than 60 : in one step
	subwf	tmp_send_w_pulses
	call	send_upto60_pulses	; more than 60 : n-60 first
	movlw	.60			; and then 60 more
	movwf	tmp_send_w_pulses	
	
send_upto60_pulses			; Limited to 60 due to page boundary issue !
					; input in tmp_send_w_pulses
	movf	tmp_send_w_pulses,W
	sublw	.60			; calculate 2x(60-w)
	rlncf	WREG
	rlncf	WREG
	movwf	tmp_send_w_pulses

	goto	pulsesaligned
pulsesnotaligned
	org	(1 + high pulsesnotaligned)*.256	; MUST START ON A PAGE BOUNDARY
pulsesaligned

	movlw	high pulsesaligned
	movwf	PCLATH			; high byte of new PC should be defined			
	movf	tmp_send_w_pulses,W
	addwf	PCL,F			; jump to next instr if W=0 (80 pulses)
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 60
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 59
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 58
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 57
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 56
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 55
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 54
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 53
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 52
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 51
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 50
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 49
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 48
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 47
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 46
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 45
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 44
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 43
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 42
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 41
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 40
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 39
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 38
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 37
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 36
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 35
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 34
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 33
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 32
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 31
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 30
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 29
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 28
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 27
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 26
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 25
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 24
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 23
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 22
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 21
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 20
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 19
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 18
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 17
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 16
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 15
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 14
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 13
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 12
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 11
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 10
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 09
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 08
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 07
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 06
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 05
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 04
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 03
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 02
	bcf	PORTB,RB_LCDXSCL_BIT
	bsf	PORTB,RB_LCDXSCL_BIT	; pulse 01
	bcf	PORTB,RB_LCDXSCL_BIT
	retlw	0			; Must be in the same page than the first one !


;---------------------------------------------------------------------
; generates the "graphic" part of the bitmap
; Each function should return in less than 30s, after having
; pulsed exactly 80-9=71 times the XSCL line (leaving 9 nibbles for the text)
;---------------------------------------------------------------------

; Exterme left or exterme right of the display : just 0, 5 and 10 markers
execcol000
execcol001
execcol238
execcol239
	clrf	PORTB			; 2 blanks
	SEND_2_PULSES
	movlw	0x01			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	movlw	.30			; 30 blanks
	call	send_w_pulses
	movlw	0x08			; "5" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	movlw	.29			; 29 blanks
	call	send_w_pulses
	movlw	0x08			; "0" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	movlw	.7			; 7 blanks
	call	send_w_pulses
	goto	returnfromexeccol

; Left and right graduations : 11 markers (0 to 10)
execcol002
execcol003
execcol004
execcol235
execcol236
execcol237
	clrf	PORTB			; 2 blanks
	SEND_2_PULSES
	movlw	0x01			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_1_PULSE			; 6 blanks
	SEND_5_PULSES
	movlw	0x08			; "9" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "8" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "7" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "6" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "5" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "4" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "3" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "2" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "1" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_5_PULSES			; 5 blanks
	movlw	0x08			; "0" graduation
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	movlw	.7			; 7 blanks
	call	send_w_pulses
	goto	returnfromexeccol

; Plain lines for the border
execcol005
execcol234
	clrf	PORTB			; 2 blanks
	SEND_2_PULSES
	movlw	0x01			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	movlw	0xff			; 64 blacks
	movwf	PORTB
	movlw	.64
	call	send_w_pulses
	movlw	0x08			; top frame
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_3_PULSES			; 3 blanks
	goto	returnfromexeccol

; Remaining part of the screen is 240-2-6-4=228 pixels wide
; column 007 correspond to sample 16 (leftmost border)
; Column 119 correspond to sample 128 (middle of the screen)
; Column 232 correspond to sample 241 (rightmost border)

; Normal columns (just low & high borders + curve)
execcol007
execcol008
execcol009
execcol010
execcol011
execcol012
execcol013
execcol014
execcol015
execcol016
execcol017
execcol018
execcol019
execcol020
execcol021
execcol022
execcol023
execcol025
execcol026
execcol027
execcol028
execcol029
execcol030
execcol031
execcol032
execcol033
execcol034
execcol035
execcol036
execcol037
execcol038
execcol039
execcol040
execcol041
execcol042
execcol043
execcol044
execcol045
execcol046
execcol047
execcol049
execcol050
execcol051
execcol052
execcol053
execcol054
execcol055
execcol056
execcol057
execcol058
execcol059
execcol060
execcol061
execcol062
execcol063
execcol064
execcol065
execcol066
execcol067
execcol068
execcol069
execcol070
execcol071
execcol073
execcol074
execcol075
execcol076
execcol077
execcol078
execcol079
execcol080
execcol081
execcol082
execcol083
execcol084
execcol085
execcol086
execcol087
execcol088
execcol089
execcol090
execcol091
execcol092
execcol093
execcol094
execcol095
execcol097
execcol098
execcol099
execcol100
execcol101
execcol102
execcol103
execcol104
execcol105
execcol106
execcol107
execcol108
execcol109
execcol110
execcol111
execcol112
execcol113
execcol114
execcol115
execcol116
execcol117
execcol118
execcol120
execcol121
execcol122
execcol123
execcol124
execcol125
execcol126
execcol127
execcol128
execcol129
execcol130
execcol131
execcol132
execcol133
execcol134
execcol135
execcol136
execcol137
execcol138
execcol139
execcol140
execcol141
execcol142
execcol143
execcol145
execcol146
execcol147
execcol148
execcol149
execcol150
execcol151
execcol152
execcol153
execcol154
execcol155
execcol156
execcol157
execcol158
execcol159
execcol160
execcol161
execcol162
execcol163
execcol164
execcol165
execcol166
execcol167
execcol169
execcol170
execcol171
execcol172
execcol173
execcol174
execcol175
execcol176
execcol177
execcol178
execcol179
execcol180
execcol181
execcol182
execcol183
execcol184
execcol185
execcol186
execcol187
execcol188
execcol189
execcol190
execcol191
execcol193
execcol194
execcol195
execcol196
execcol197
execcol198
execcol199
execcol200
execcol201
execcol202
execcol203
execcol204
execcol205
execcol206
execcol207
execcol208
execcol209
execcol210
execcol211
execcol212
execcol213
execcol214
execcol215
execcol217
execcol218
execcol219
execcol220
execcol221
execcol222
execcol223
execcol224
execcol225
execcol226
execcol227
execcol228
execcol229
execcol230
execcol231
execcol232
	clrf	PORTB			; 2 blanks
	SEND_2_PULSES
	movlw	0x01			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	call	dispsample		; actual curve (64 clocks, 256 pixels)
	movlw	0x08			; top frame
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_3_PULSES			; 3 blanks
	goto	returnfromexeccol

; right-most & left-most graph columns : empty except border
execcol006
execcol233
	clrf	PORTB			; 2 blanks
	SEND_2_PULSES
	movlw	0x01			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB			; 64 white
	movlw	.64
	call	send_w_pulses
	movlw	0x08			; top frame
	movwf	PORTB
	SEND_1_PULSE
	clrf	PORTB
	SEND_3_PULSES			; 3 blanks
	goto	returnfromexeccol

; Columns with big graduations (+ curve)
execcol024
execcol072
execcol168
execcol216
	movlw	0x0F
	movwf	PORTB			; 2 blacks
	SEND_2_PULSES
	movlw	0x0F			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	call	dispsample		; actual curve (64 clocks, 256 pixels)
	movlw	0x0F			; top frame
	movwf	PORTB
	SEND_1_PULSE
	movlw	0xFF
	movwf	PORTB			; 2 blacks
	SEND_2_PULSES	
	clrf	PORTB			; 1 white
	SEND_1_PULSE
	goto	returnfromexeccol

; Columns with small graduations (+ curve)
execcol048
execcol096
execcol144
execcol192
	clrf	PORTB
	SEND_1_PULSE			; 1 white
	movlw	0x07
	movwf	PORTB
	SEND_1_PULSE			; 1 short
	movlw	0x0F
	movwf	PORTB			; 1 blacks
	SEND_1_PULSE
	call	dispsample		; actual curve (64 clocks, 256 pixels)
	movlw	0x0F			; top frame
	movwf	PORTB
	SEND_1_PULSE
	movlw	0x0F
	movwf	PORTB			; 1 black
	SEND_1_PULSE
	movlw	0x0E
	movwf	PORTB			; 1 short
	SEND_1_PULSE
	clrf	PORTB			; 1 white
	SEND_1_PULSE
	goto	returnfromexeccol

; Middle column (dashed line)
execcol119
	movlw	0x0F
	movwf	PORTB			; 2 blacks
	SEND_2_PULSES
	movlw	0x0F			; bottom frame
	movwf	PORTB
	SEND_1_PULSE
	movlw	0x05
	movwf	PORTB			; 64 dotted
	movlw	.64
	call	send_w_pulses
	movlw	0x0F			; top frame
	movwf	PORTB
	SEND_1_PULSE
	movlw	0xFF
	movwf	PORTB			; 2 blacks
	SEND_2_PULSES	
	clrf	PORTB			; 1 white
	SEND_1_PULSE
	goto	returnfromexeccol

;---------------------------------------------------------------------
; Display an actual column of pixels for the data area 
; Send 256 pixels to the LCD (64 nibbles)
;
; selected column is designed by xcol :
; xcol=007 correspond to sample 16 (leftmost border)
; xcol=119 correspond to sample 128 (middle of the screen)
; xcol=232 correspond to sample 241 (rightmost border)
;
; Use the operating_mode_disp to select the required visualisation mode :
;   MODE_DISP_SAMP	Display just a pixel at the sampled value
;   MODE_DISP_MEAN	Display just a pixel at max+min/2 value
;   MODE_DISP_MAXI	Display just a pixel at max value
;   MODE_DISP_PEAK	Display a bar from min to max
;
; Average execution time : 27s (24 to 28s)
;---------------------------------------------------------------------

dispsamplenotaligned
	org	(1 + high dispsamplenotaligned)*.256	; MUST START ON A PAGE BOUNDARY
dispsamplealigned

dispsample
	movlw	high dispsamplealigned
	movwf	PCLATH			; high byte of new PC should be defined			
	
	movf	operating_mode_disp,W	; unloop mode, instruction N+1 will be
	bcf	STATUS,C
	rlcf	WREG
	addwf	PCL,F			; executed if mode=0
	bra	dispsample_SAMP
	bra	dispsample_MEAN
	bra	dispsample_MAXI		; each one initialize ystart/ystop with
	bra	dispsample_PEAK		; the limits of the "black" line to draw

dispsample_SAMP
	movf	xcol,W
	addlw	.9			; align to screen (xcol7=sample16)
	movwf	FSR0L			; get sample value for this X
	movlw	HIGH(sampleval)
	movwf	FSR0H
	BANKSEL	sampleval
	movf	INDF0,W
	BANKSEL	bank0
	movwf	ystart
	movwf	ystop
	bra	dispsample_joint

dispsample_MEAN
	movf	xcol,W
	addlw	.9			; align to screen (xcol7=sample16)
	movwf	FSR0L			; get min value for this X
	movlw	HIGH(minval)
	movwf	FSR0H
	BANKSEL	minval
	movf	INDF0,W
	BANKSEL	bank0
	movwf	ystart
	movf	xcol,W
	addlw	.9			; align to screen (xcol7=sample16)
	movwf	FSR0L			; get max value for this X
	movlw	HIGH(maxval)
	movwf	FSR0H
	BANKSEL	maxval
	movf	INDF0,W
	BANKSEL	bank0
	addwf	ystart,F		; add min and max
	rrcf	ystart			; divide by 2 (getting back MSB)
	movff	ystart,ystop
	bra	dispsample_joint

dispsample_MAXI
	movf	xcol,W
	addlw	.9			; align to screen (xcol7=sample16)
	movwf	FSR0L			; get max value for this X
	movlw	HIGH(maxval)
	movwf	FSR0H
	BANKSEL	maxval
	movf	INDF0,W
	BANKSEL	bank0
	movwf	ystart
	movwf	ystop
	bra	dispsample_joint

dispsample_PEAK
	movf	xcol,W
	addlw	.9			; align to screen (xcol7=sample16)
	movwf	FSR0L			; get min value for this X
	movlw	HIGH(minval)
	movwf	FSR0H
	BANKSEL	minval
	movf	INDF0,W
	BANKSEL	bank0
	movwf	ystart
	movf	xcol,W
	addlw	.9			; align to screen (xcol7=sample16)
	movwf	FSR0L			; get max value for this X
	movlw	HIGH(maxval)
	movwf	FSR0H
	BANKSEL	maxval
	movf	INDF0,W
	BANKSEL	bank0
	movwf	ystop
	bra	dispsample_joint

dispsample_joint

	; check if start/stop in good order
	;----------------------------------
	movf	ystart,W
	cpfslt	ystop			; good order ?
	bra	dispsample_yorderok
	movff	ystop,tmp		; if not permute
	movff	ystart,ystop
	movff	tmp,ystart	
dispsample_yorderok

	; join to previous line if not first row
	;---------------------------------------	
	movlw	.7	
	cpfseq	xcol
	bra	dispsample_joinline
	bra	dispsample_endjoin
dispsample_joinline
	movf	ystart_prev,W
	cpfslt	ystop			; if new segment under previous one
	bra	dispsample_join2
	movff	ystart_prev,ystop	; align
	bra	dispsample_endjoin
dispsample_join2
	movf	ystop_prev,W
	cpfsgt	ystart			; if new segment under previous one
	bra	dispsample_endjoin
	movff	ystop_prev,ystart	; align
dispsample_endjoin

	; Initialize rolling registers
	;-----------------------------
	clrf	cptbytes
	movff	ystart,m
	movff	ystop,n

	; send full "blank" nibbles
	;--------------------------
	movf	m,W			;v=m/4
	rrcf	WREG
	rrcf	WREG
	andlw	B'00111111'
	movwf	v
	bz	skipdispblanks		; if v!=0...
	clrf	PORTB			; send V times 0000
	movf	v,W
	call	send_w_pulses
	movf	v,W			; m=m-4v
	rlcf	WREG
	rlcf	WREG
	andlw	B'11111100'
	subwf	m,F
	subwf	n,F			; n=n-4v
	movf	v,W
	addwf	cptbytes,F		; cptbytes+=v

skipdispblanks

	; send one combined white/back/white if less than 4 blacks to display
	;----------------------------------------------------------------
	movlw	.4
	cpfslt	n			; n<4 ?
	bra	dispcombined
	movf	m,W			; calculate m*4+n
	rlcf	WREG
	rlcf	WREG
	andlw	B'11111100'
	addwf	n,W
	movwf	FSR0L			; send tblcombined[m*4+n]
	movlw	HIGH(tbldraw)
	movwf	FSR0H
	BANKSEL	tbldraw
	movf	INDF0,W
	BANKSEL	bank0
	movwf	PORTB
	SEND_1_PULSE
	incf	cptbytes		; cptbytes++
	bra	dispwhites

	; else send one starting combined white/black...
	;------------------------------------------------
dispcombined
	movf	m,W			; send tblmin[m]
	addlw	START_TBLMIN
	movwf	FSR0L
	movlw	HIGH(tbldraw)
	movwf	FSR0H
	BANKSEL	tbldraw
	movf	INDF0,W
	BANKSEL	bank0
	movwf	PORTB
	SEND_1_PULSE
	incf	cptbytes		; cptbytes++

	; ... and full "black" nibbles as long as needed...
	;------------------------------------------------
	movlw	.4
	subwf	n,F			; N-=4
	movf	n,W			; v=n/4
	rrcf	WREG
	rrcf	WREG
	andlw	B'00111111'
	movwf	v
	bz	skipdispblacks		; if v!=0...
	movlw	0x0F
	movwf	PORTB			; send V times 1111
	movf	v,W
	call	send_w_pulses
	movlw	B'00000011'
	andwf	n,F			; N=N mod 4
	movf	v,W
	addwf	cptbytes,F		; cptbytes+=v

	; ... and one ending combined black/white
	;-----------------------------------------
skipdispblacks
	movf	n,W			; send tblmax[n]
	addlw	START_TBLMAX
	movwf	FSR0L
	movlw	HIGH(tbldraw)
	movwf	FSR0H
	BANKSEL	tbldraw
	movf	INDF0,W
	BANKSEL	bank0
	movwf	PORTB
	SEND_1_PULSE
	incf	cptbytes		; cptbytes++

	; Send remaining "full white" nibbles to finish the display
	;-----------------------------------------------------------
dispwhites
	movlw	.64
	bsf	STATUS,C
	subfwb	cptbytes,W		; caclulate 64-cptbytes
	bz	skipendingwhites
	clrf	PORTB
	call	send_w_pulses		; send send 0000 good number of times
skipendingwhites

	; store ymin/ymax for next call
	;------------------------------
	movff	ystart,ystart_prev
	movff	ystop,ystop_prev

	retlw	0

;---------------------------------------------------------------------
; generates the "text" part of the bitmap
;
; should return in 10s in average, after having
; pulsed exactly 9 times the XSCL line
; (3 times per text line, giving 12 pixel heigh)
;
; Takes as inputs the characters stored in txt_line0/1/2
;
; for each of the three lines the three nibbles to send to the lcd are 
; in the three successive bytes starting at :
;   ascii_table+txt_lineN[xcol/8]*24+(xol%8)*3
;   \---- adr_txtN----------------/ \inc_txt/
;
; For speed optimization adr_txt is calculated only one each 8 calls
; and inc_txt is incremented by 3 each call
;
; number of instructions / execution time
; normal column   : 8.9s
; every 8 columns : 18.6s
; average : 10.1s
;---------------------------------------------------------------------

exectext
	movf	xcol,W
	bnz	text_xcolnotzero	; if xcol=0

	; Start of the frame conditions
	;------------------------------
	movlw	.21
	movwf	inc_txt	
	movlw	0xFF
	movwf	xcol_div8

text_xcolnotzero

	; Next column of pixels
	;----------------------
	incf	inc_txt			; inc_txt+=3
	incf	inc_txt
	incf	inc_txt

	movlw	.24
	cpfseq	inc_txt			; if xcol_div8=24 (ie xcol%8=0)...
	bra	txt_col

	; Start of a new block of characters
	;-----------------------------------
	clrf	inc_txt			; inc_txt=0
	incf	xcol_div8		; next block of characters

	movlw	LOW(txt_line0)		; calculate adr_txt0
	movwf	FSR0L
	movlw	HIGH(txt_line0)
	movwf	FSR0H
	movf	xcol_div8,W
	addwf	FSR0L			; assuming MSB will not change (same page)
	BANKSEL	txt_line0
	movff	INDF0,adr_txt0l		; get ASCII char on that line
	clrf	adr_txt0h
	bcf	STATUS,C
	rlcf	adr_txt0l		; times 2
	rlcf	adr_txt0h
	movf	INDF0,W
	addwf	adr_txt0l,F		; times 3
	clrf	WREG
	addwfc	adr_txt0h,F
	BANKSEL	bank0
	bcf	STATUS,C
	rlcf	adr_txt0l		; times 6
	rlcf	adr_txt0h
	bcf	STATUS,C
	rlcf	adr_txt0l		; times 12
	rlcf	adr_txt0h
	bcf	STATUS,C
	rlcf	adr_txt0l		; times 24
	rlcf	adr_txt0h
	movlw	LOW(ascii_table)
	addwf	adr_txt0l		; plus ascii_table
	movlw	HIGH(ascii_table)
	addwfc	adr_txt0h

	movlw	LOW(txt_line1)		; calculate adr_txt1
	movwf	FSR0L
	movlw	HIGH(txt_line1)
	movwf	FSR0H
	movf	xcol_div8,W
	addwf	FSR0L			; assuming MSB will not change (same page)
	BANKSEL	txt_line1
	movff	INDF0,adr_txt1l		; get ASCII char on that line
	clrf	adr_txt1h
	bcf	STATUS,C
	rlcf	adr_txt1l		; times 2
	rlcf	adr_txt1h
	movf	INDF0,W
	addwf	adr_txt1l,F		; times 3
	clrf	WREG
	addwfc	adr_txt1h,F
	BANKSEL	bank0
	bcf	STATUS,C
	rlcf	adr_txt1l		; times 6
	rlcf	adr_txt1h
	bcf	STATUS,C
	rlcf	adr_txt1l		; times 12
	rlcf	adr_txt1h
	bcf	STATUS,C
	rlcf	adr_txt1l		; times 24
	rlcf	adr_txt1h
	movlw	LOW(ascii_table)
	addwf	adr_txt1l		; plus ascii_table
	movlw	HIGH(ascii_table)
	addwfc	adr_txt1h

	movlw	LOW(txt_line2)		; calculate adr_txt2
	movwf	FSR0L
	movlw	HIGH(txt_line2)
	movwf	FSR0H
	movf	xcol_div8,W
	addwf	FSR0L			; assuming MSB will not change (same page)
	BANKSEL	txt_line2
	movff	INDF0,adr_txt2l		; get ASCII char on that line
	clrf	adr_txt2h
	bcf	STATUS,C
	rlcf	adr_txt2l		; times 2
	rlcf	adr_txt2h
	movf	INDF0,W
	addwf	adr_txt2l,F		; times 3
	clrf	WREG
	addwfc	adr_txt2h,F
	BANKSEL	bank0
	bcf	STATUS,C
	rlcf	adr_txt2l		; times 6
	rlcf	adr_txt2h
	bcf	STATUS,C
	rlcf	adr_txt2l		; times 12
	rlcf	adr_txt2h
	bcf	STATUS,C
	rlcf	adr_txt2l		; times 24
	rlcf	adr_txt2h
	movlw	LOW(ascii_table)
	addwf	adr_txt2l		; plus ascii_table
	movlw	HIGH(ascii_table)
	addwfc	adr_txt2h

txt_col
	; Effective display
	;------------------
	
	; Send *(adr_txt2+inc_txt), *(adr_txt2+inc_txt+1), *(adr_txt2+inc_txt+2)
 	clrf	TBLPTRU
   	movff	adr_txt2h,TBLPTRH  
   	movff	adr_txt2l,TBLPTRL
	movf	inc_txt,W
	addwf	TBLPTRL
	clrf	WREG
	addwfc	TBLPTRH
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	
	; Send *(adr_txt1+inc_txt), *(adr_txt1+inc_txt+1), *(adr_txt1+inc_txt+2)
   	movff	adr_txt1h,TBLPTRH
   	movff	adr_txt1l,TBLPTRL
	movf	inc_txt,W
	addwf	TBLPTRL
	clrf	WREG
	addwfc	TBLPTRH
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE

	; Send *(adr_txt0+inc_txt), *(adr_txt0+inc_txt+1), *(adr_txt02+inc_txt+2)
   	movff	adr_txt0h,TBLPTRH
   	movff	adr_txt0l,TBLPTRL
	movf	inc_txt,W
	addwf	TBLPTRL
	clrf	WREG
	addwfc	TBLPTRH
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE
	tblrd*+
	movff	TABLAT,PORTB
	SEND_1_PULSE

	retlw	0

;---------------------------------------------------------------------
; Frame batch
; Executed every 13ms
;
; Tasks :
;  - Update the flags operating_mode_xxx given the push button status
;  - Check if an order was received from the UART
;  If not in hold mode :
;  - Read auxiliary analog inputs and update the displayed text (txt_linex)
;  - Read auxiliary digital inputs and update the displayed text (txt_linex)
;
; Duration : 150 to 200s
;---------------------------------------------------------------------

frame_batch
	; Update the flags operating_mode_xxx given the push button status
	
	movff	PORTC,new_keys		; get new state

	; Check hold mode
	btfss	prev_keys,RC_KEYHOLD_BIT; was not released : skip
	bra	check_disp
	btfsc	new_keys,RC_KEYHOLD_BIT	; is still released: skip
	bra	check_disp
	incf	operating_mode_hold	; if not, change hold mode
	movlw	MODE_HOLD_ON+1
	cpfslt	operating_mode_hold
	clrf	operating_mode_hold

check_disp
	; Check disp mode
	btfss	prev_keys,RC_KEYMODE_BIT; was not released : skip
	bra	check_acc
	btfsc	new_keys,RC_KEYMODE_BIT	; is still released: skip
	bra	check_acc 
	incf	operating_mode_disp	; if not, change hold mode
	movlw	MODE_DISP_PEAK+1
	cpfslt	operating_mode_disp
	clrf	operating_mode_disp
	
check_acc
	; Check acc mode
	btfss	prev_keys,RC_KEYACC_BIT	; was not released : skip
	bra	end_checks
	btfsc	new_keys,RC_KEYACC_BIT	; is still released: skip
	bra	end_checks 
	incf	operating_mode_acc	; if not, change hold mode
	movlw	MODE_ACC_ON+1
	cpfslt	operating_mode_acc
	clrf	operating_mode_acc
	
end_checks
	movff	new_keys,prev_keys	; store new state

	; Check if an order was received from the uart
	call	manageuart

	; if not in hold mode...
	movlw	MODE_HOLD_OFF
	cpfseq	operating_mode_hold
	goto	end_frame

	; Given the operating disp mode, update the textual info
	; ..............................
	; ..............................
	; ...............PEAK...........
	
	movlw	UPPER(DISP_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(DISP_STR)
  	movwf	TBLPTRH
   	movlw	LOW(DISP_STR)
   	movwf	TBLPTRL
	
	clrf	tmp			; skip until good one
loop_dispdisp
	movf	operating_mode_disp,W
	cpfslt	tmp
	bra	disp_disp
	tblrd*+
	tblrd*+
	tblrd*+
	tblrd*+
	incf	tmp
	bra	loop_dispdisp
disp_disp
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.15
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.16
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.17
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.18
	BANKSEL	bank0           

	; Display the ACC mode if any
	; ..............................
	; ..............................
	; ....................ACC.......
	
	movlw	UPPER(ACC_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(ACC_STR)
  	movwf	TBLPTRH
   	movlw	LOW(ACC_STR)
   	movwf	TBLPTRL
	
	movf	MODE_ACC_ON,W
	cpfseq	operating_mode_acc
	bra	disp_accoff
	tblrd*+
	tblrd*+
	tblrd*+
disp_accoff
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.20
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.21
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.22
	BANKSEL	bank0           

	; Display the hold mode if any
	; ..............................
	; ..............................
	; ............................HD
	
	movlw	UPPER(HOLD_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(HOLD_STR)
  	movwf	TBLPTRH
   	movlw	LOW(HOLD_STR)
   	movwf	TBLPTRL
	
	movf	MODE_HOLD_ON,W
	cpfseq	operating_mode_hold
	bra	disp_holdoff
	tblrd*+
	tblrd*+
disp_holdoff
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.28
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.29
	BANKSEL	bank0           

	; Read auxiliary analog input aux1 and update the displayed text
	; .....1234.....................
	; ..............................
	; ..............................

	movlw	ADC_CHAN_AUX1
	call	adc_select_chan		; get ADC value
	call	adc_delay
	call	adc_launch_conv
	call	adc_delay
	call	adc_read_result
	movff	adc_val_h,TEM2		; store result (0-1023)
	movff	adc_val_l,TEM0
	bcf	STATUS,C
	rlcf	TEM0			; multiply per 2 (0-2048)
	rlcf	TEM2
	call	conv16bin_to_ascii	; convert to ASCII
	movf	convasc3,W
	BANKSEL txt_line0
	movwf	txt_line0+5		; store text to screen map
	BANKSEL	bank0
	movf	convasc2,W
	BANKSEL txt_line0
	movwf	txt_line0+6
	BANKSEL	bank0
	movf	convasc1,W
	BANKSEL txt_line0
	movwf	txt_line0+7
	BANKSEL	bank0
	movf	convasc0,W
	BANKSEL txt_line0
	movwf	txt_line0+8
	BANKSEL	bank0

	; Read auxiliary analog input aux2 and update the displayed text
	; ......................96......
	; ..............................
	; ..............................

	movlw	ADC_CHAN_AUX2
	call	adc_select_chan		; get ADC value
	call	adc_delay
	call	adc_launch_conv
	call	adc_delay
	call	adc_read_result
	movff	adc_val_h,TEM2		; store result (0-1023)
	movff	adc_val_l,TEM0
	bcf	STATUS,C
	rrcf	TEM2			; divide by 8 (0-128)
	rrcf	TEM0
	bcf	STATUS,C
	rrcf	TEM2
	rrcf	TEM0
	bcf	STATUS,C
	rrcf	TEM2
	rrcf	TEM0
	bcf	STATUS,C
	rrcf	TEM2
	rrcf	TEM0
	call	conv16bin_to_ascii	; convert to ASCII
	movf	convasc2,W
	BANKSEL txt_line0
	movwf	txt_line0+.21
	BANKSEL	bank0
	movf	convasc1,W
	BANKSEL txt_line0
	movwf	txt_line0+.22
	BANKSEL	bank0
	movf	convasc0,W
	BANKSEL txt_line0
	movwf	txt_line0+.23
	BANKSEL	bank0

	; Read auxiliary digital input da1 and update the displayed text
	; ..............................
	; ......200 K...................
	; ..............................

	movlw	0xFF			; temporary set portb as input only
	movwf	TRISB
	nop
	nop
	movf	PORTB,W			; read DA1
	andlw	RB_DIGAUX1_MASQ
	movwf	disp_item
	movlw	0x70			; reset portb direction : only bits 4-5-6 are input
	movwf	TRISB

	movlw	UPPER(DA1_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(DA1_STR)
  	movwf	TBLPTRH
   	movlw	LOW(DA1_STR)
   	movwf	TBLPTRL
	
	clrf	tmp			; skip until good one
loop_dispda1
	movf	disp_item,W
	cpfslt	tmp
	bra	disp_da1
	tblrd*+
	tblrd*+
	tblrd*+
	tblrd*+
	tblrd*+
	incf	tmp
	bra	loop_dispda1
disp_da1
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+6
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+7
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+8
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+9
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+.10
	BANKSEL	bank0           

	; Read auxiliary digital input da2 and update the displayed text
	; ..............................
	; ..............................
	; ......  1 K...................

	movf	PORTB,W			; read DA2
	andlw	RB_DIGAUX2_MASQ
	movwf	disp_item
	bcf	STATUS,C		; shift it 4 times to right-justify it
	rrcf	disp_item
	bcf	STATUS,C		
	rrcf	disp_item
	bcf	STATUS,C		
	rrcf	disp_item
	bcf	STATUS,C		
	rrcf	disp_item

	movlw	UPPER(DA2_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(DA2_STR)
  	movwf	TBLPTRH
   	movlw	LOW(DA2_STR)
   	movwf	TBLPTRL
	
	clrf	tmp			; skip until good one
loop_dispda2
	movf	disp_item,W
	cpfslt	tmp
	bra	disp_da2
	tblrd*+
	tblrd*+
	tblrd*+
	tblrd*+
	tblrd*+
	incf	tmp
	bra	loop_dispda2
disp_da2
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+6
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+7
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+8
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+9
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.10
	BANKSEL	bank0           

	; Read auxiliary digital input da3 and update the displayed text
	; ..............................
	; ..............................
	; ........................FLT...

	movlw	UPPER(DA3_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(DA3_STR)
  	movwf	TBLPTRH
   	movlw	LOW(DA3_STR)
   	movwf	TBLPTRL
	
	btfss	PORTC,RC_DIGAUX3_BIT	; test input bit
	bra	disp_da3off
	tblrd*+
	tblrd*+
	tblrd*+
disp_da3off
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.24
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.25
	BANKSEL	bank0    
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line2
	movwf	txt_line2+.26
	BANKSEL	bank0    

	; Read auxiliary digital input da4 and update the displayed text
	; ..............................
	; ......................10......
	; ..............................

	movlw	UPPER(DA4_STR)		; point to string table
   	movwf	TBLPTRU
   	movlw	HIGH(DA4_STR)
  	movwf	TBLPTRH
   	movlw	LOW(DA4_STR)
   	movwf	TBLPTRL
	
	btfss	PORTA,RA_DIGAUX4_BIT	; test input bit
	bra	disp_da4off
	tblrd*+
	tblrd*+
disp_da4off
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+.22
	BANKSEL	bank0           
 	tblrd*+
	movf	TABLAT,W  
	BANKSEL	txt_line1
	movwf	txt_line1+.23
	BANKSEL	bank0    

end_frame
	retlw	0

;---------------------------------------------------------------------
; manage the uart :
;
; check if something was received :
;  '?' : send hello string again
;  'D' : dump the full RAM in binary
;  other : do nothing
;
; Non-critic timing (LCD refresh no longer managed during UART operations)
;---------------------------------------------------------------------

manageuart
	movlw	1
	movwf	tmpuart
	call	uart_check_received
	cpfseq	tmpuart
	bra	enduart			; nothing received...
	movff	RCREG, tmpuart		; else get char
	movlw	'?'
	cpfseq	tmpuart
	bra	notquestion		; "?" -> send hello string
	call	send_hello
	bra	enduart
notquestion
	movlw	'D'
	cpfseq	tmpuart
	bra	notdump			; "D" -> dump memory

	lfsr	0,bank0			; send block 0
loopsendblk0	
	BANKSEL	bank0
	movf	INDF0, W
	BANKSEL bank0
	call	uart_send_byte
	incfsz	FSR0L
	bra	loopsendblk0

	lfsr	0,bank1			; send block 1
loopsendblk1
	BANKSEL	bank1
	movf	INDF0, W
	BANKSEL bank0
	call	uart_send_byte
	incfsz	FSR0L
	bra	loopsendblk1

	lfsr	0,bank2			; send block 2
loopsendblk2
	BANKSEL	bank2
	movf	INDF0, W
	BANKSEL bank0
	call	uart_send_byte
	incfsz	FSR0L
	bra	loopsendblk2

	lfsr	0,bank3			; send block 3
loopsendblk3
	BANKSEL	bank3
	movf	INDF0, W
	BANKSEL bank0
	call	uart_send_byte
	incfsz	FSR0L
	bra	loopsendblk3

	lfsr	0,bank4			; send block 4
loopsendblk4
	BANKSEL	bank4
	movf	INDF0, W
	BANKSEL bank0
	call	uart_send_byte
	incfsz	FSR0L
	bra	loopsendblk4

	lfsr	0,bank5			; send block 5
loopsendblk5
	BANKSEL	bank5
	movf	INDF0, W
	BANKSEL bank0
	call	uart_send_byte
	incfsz	FSR0L
	bra	loopsendblk5

notdump
enduart
	retlw	0


;---------------------------------------------------------------------
; Line batch
; Executed every 55s 
;
; Tasks :
;    - Manage a 5-steps state machine (one step executed per call, ie each 42s)
;    - Step 0 (at initialization or during blanking interval) :
;       configure ADC for channel X
;       next step will be step 2
;    - Step1 : 
;       read ADC value for Y
;	if not in mode hold :
;         add this new Y value to the measurement array indexed with previous X (min/max/sample)
;	configure ADC for channel X
;       next step will be step 2
;    - Step 2:
;       launch ADC convertion for channel X
;       next step will be step 3
;    - Step 3:
;       read ADC value for X
;       store this value as index X for the Y that will be aquired next
;       if higher than previous X (and if not in ACC mode) reset min-max(newX) 
;       if higher or equal than previous X
;         configure ADC for channel Y    
;         next step will be step 4
;       else (lower than previous X, meaning pen lift)
;         next step will be step 0
;    - Step 4:
;       launch ADC convertion for channel Y
;       next step will be step 1
;
; Timing worst case : 7.4s (step1), average 5.3s
;---------------------------------------------------------------------

linebatchnotaligned
	org	(1 + high linebatchnotaligned)*.256	; MUST START ON A PAGE BOUNDARY
linebatchaligned

line_batch
	movlw	high linebatchaligned
	movwf	PCLATH			; high byte of new PC should be defined			
	
	movf	line_batch_step,W	; unloop state, instruction N+1 will be
	bcf	STATUS,C
	rlcf	WREG
	addwf	PCL,F			; executed if xcol=0
	bra	line_batch_step0
	bra	line_batch_step1
	bra	line_batch_step2
	bra	line_batch_step3
	bra	line_batch_step4
	retlw	0

line_batch_step0
	movlw	ADC_CHAN_X		; configure ADC for channel X
	call	adc_select_chan
	movlw	2			; next step will be step 2
	movwf	line_batch_step
	retlw	0

line_batch_step1
	call	adc_read_result		; read ADC value for Y (0-1024)
	rrcf	adc_val_h
	rrcf	adc_val_l		; 0-512
	rrcf	adc_val_h
	rrcf	adc_val_l		; 0-256 (MSB discarded)
	movff	line_batch_y,line_oldy
	movff	adc_val_l,line_batch_y
	movf	operating_mode_hold,W	; if not in mode hold :
	bnz	line_step1_hold
					; add this new Y value to the measurement 
					; array indexed with previous X (min/max/sample)
	lfsr	0,sampleval		
	movff	line_batch_x,FSR0L	; sample...
	movf	line_batch_y,W
	BANKSEL	sampleval
	movwf	INDF0
	BANKSEL	bank0

	lfsr	0,minval		
	movff	line_batch_x,FSR0L	; min...
	BANKSEL	minval
	movf	INDF0,W			; get old min
	BANKSEL	bank0
	cpfslt	line_batch_y
	bra	line_step1_notnewmin
	movf	line_batch_y,W
	BANKSEL	minval
	movwf	INDF0			; store new min
	BANKSEL	bank0
line_step1_notnewmin

	lfsr	0,maxval		
	movff	line_batch_x,FSR0L	; max...
	BANKSEL	maxval
	movf	INDF0,W			; get old max
	BANKSEL	bank0
	cpfsgt	line_batch_y
	bra	line_step1_notnewmax
	movf	line_batch_y,W
	BANKSEL	maxval
	movwf	INDF0			; store new max
	BANKSEL	bank0
line_step1_notnewmax
	
line_step1_hold
	movlw	ADC_CHAN_X		; configure ADC for channel X
	call	adc_select_chan
	movlw	2			; next step will be step 2
	movwf	line_batch_step
	retlw	0

line_batch_step2
	call	adc_launch_conv		; launch ADC convertion for channel X
	movlw	3			; next step will be step 3
	movwf	line_batch_step
	retlw	0

line_batch_step3
	call	adc_read_result		; read ADC value for X (0-1024)
	rrcf	adc_val_h
	rrcf	adc_val_l		; 0-512
	rrcf	adc_val_h
	rrcf	adc_val_l		; 0-256 (MSB discarded)
	movff	line_batch_x,line_oldx
	movff	adc_val_l,line_batch_x	; store this value as index X for the Y that will be aquired next
	movf	operating_mode_acc,W
	bnz	line_step3skip1		; if higher than previous X (and if not in ACC mode)
	movf	line_oldx,W
	cpfsgt	line_batch_x
	bra	line_step3skip1
	lfsr	0,minval		; reset min-max(newX)
	movff	line_batch_x,FSR0L
	BANKSEL	minval
	movlw	0xff
	movwf	INDF0
	BANKSEL	bank0
	lfsr	0,maxval
	movff	line_batch_x,FSR0L
	BANKSEL	maxval
	clrf	INDF0
	BANKSEL	bank0
line_step3skip1
	movf	line_batch_x,W		; if higher or equal than previous X
	cpfsgt	line_oldx
	bra	line_step3_gety
	bra	line_step3_blanking
line_step3_gety
	movlw	ADC_CHAN_Y		; configure ADC for channel Y    
	call	adc_select_chan
	movlw	4			; next step will be step 4
	movwf	line_batch_step
	bra	line_step3_end
line_step3_blanking			; else (lower than previous X, meaning pen lift)
	clrf	line_batch_step		; next step will be step 0
line_step3_end
	retlw	0

line_batch_step4
	call	adc_launch_conv		; launch ADC convertion for channel Y
	movlw	1			; next step will be step 1
	movwf	line_batch_step
	retlw	0

;=====================================================================
;
;
;
; TABLES
;
;
;
;=====================================================================


;---------------------------------------------------------------------
; ASCII TABLE. Each char (from 0 to 127) is encoded as follow :
; 
; Each char is represented in a 12x8 matrix, encoded on 24 bytes 
;
; Each coding byte represent a nibble (4 bits, LSB only, MSB=0), vertically 
; oriented, with D3 at the bottom and D0 at the top :
;
;      B2D0   .      .      .      .      .      .      B23D0
;      B2D1   .      .      .      .      .      .      B23D1
;      B2D2   .      .      .      .      .      .      etc
;      B2D3   .      .      .      .      .      .      .
;      B1D0   .      .      .      .      .      .      .
;      B1D1   .      .      .      .      .      .      .
;      B1D2   .      .      .      .      .      .      .
;      B1D3   .      .      .      .      .      .      .
;      B0D0   .      .      .      .      .      .      .
;      B0D1   etc    .      .      .      .      .      .
;      B0D2   B3D2   .      .      .      .      .      .
;      B0D3   B3D3   .      .      .      .      .      .
; 
;---------------------------------------------------------------------

ascii_table
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0    		; 0 (0)
  db  .1, .15, .12, .2, .0, .2, .2, .4, .10, .2, .12, .2, .2, .12, .2, .2, .4, .10, .2, .0, .2, .1, .15, .12    	; 1 (0)
  db  .1, .15, .12, .3, .15, .14, .3, .11, .6, .3, .3, .14, .3, .3, .14, .3, .11, .6, .3, .15, .14, .1, .15, .12    	; 2 (0)
  db  .0, .7, .8, .0, .15, .12, .1, .15, .12, .3, .15, .0, .1, .15, .12, .0, .15, .12, .0, .7, .8, .0, .0, .0    	; 3 (0)
  db  .0, .2, .0, .0, .7, .0, .0, .15, .8, .1, .15, .12, .0, .15, .8, .0, .7, .0, .0, .2, .0, .0, .0, .0    		; 4 (0)
  db  .0, .4, .0, .0, .14, .0, .2, .14, .8, .3, .1, .12, .2, .14, .8, .0, .14, .0, .0, .4, .0, .0, .0, .0    		; 5 (0)
  db  .0, .6, .0, .0, .15, .8, .2, .15, .12, .3, .7, .14, .2, .15, .12, .0, .15, .8, .0, .6, .0, .0, .0, .0    		; 6 (0)
  db  .0, .0, .0, .0, .0, .0, .0, .6, .0, .0, .15, .0, .0, .15, .0, .0, .6, .0, .0, .0, .0, .0, .0, .0    		; 7 (0)
  db  .15, .15, .15, .15, .15, .15, .15, .9, .15, .15, .0, .15, .15, .0, .15, .15, .9, .15, .15, .15, .15, .15, .15, .15; 8 (0)
  db  .0, .0, .0, .0, .6, .0, .0, .15, .0, .1, .9, .8, .1, .9, .8, .0, .15, .0, .0, .6, .0, .0, .0, .0    		; 9 (0)
  db  .15, .15, .15, .15, .9, .15, .15, .0, .15, .14, .6, .7, .14, .6, .7, .15, .0, .15, .15, .9, .15, .15, .15, .15    ; 10 (0)
  db  .1, .12, .0, .3, .14, .0, .2, .3, .0, .2, .3, .10, .3, .14, .14, .1, .13, .14, .0, .1, .14, .0, .0, .0    	; 11 (0)
  db  .0, .0, .0, .0, .9, .12, .0, .11, .14, .3, .14, .2, .3, .14, .2, .0, .11, .14, .0, .9, .12, .0, .0, .0    	; 12 (0)
  db  .1, .0, .0, .3, .8, .0, .3, .8, .0, .3, .15, .14, .1, .15, .14, .0, .0, .10, .0, .0, .14, .0, .0, .0    		; 13 (0)
  db  .0, .4, .0, .0, .14, .0, .0, .15, .15, .2, .7, .15, .7, .0, .5, .7, .15, .15, .3, .15, .15, .0, .0, .0    	; 14 (0)
  db  .1, .0, .2, .1, .11, .6, .0, .15, .12, .3, .12, .15, .3, .12, .15, .0, .15, .12, .1, .11, .6, .1, .0, .2    	; 15 (0)
  db  .3, .15, .14, .1, .15, .12, .1, .15, .12, .0, .15, .8, .0, .7, .0, .0, .7, .0, .0, .2, .0, .0, .0, .0    		; 16 (0)
  db  .0, .2, .0, .0, .7, .0, .0, .15, .8, .0, .15, .8, .1, .15, .12, .1, .15, .12, .3, .15, .14, .0, .0, .0    	; 17 (0)
  db  .0, .0, .0, .0, .8, .8, .1, .8, .12, .3, .15, .14, .3, .15, .14, .1, .8, .12, .0, .8, .8, .0, .0, .0    		; 18 (0)
  db  .0, .0, .0, .3, .7, .14, .3, .7, .14, .0, .0, .0, .0, .0, .0, .3, .7, .14, .3, .7, .14, .0, .0, .0    		; 19 (0)
  db  .0, .1, .12, .0, .3, .14, .0, .2, .2, .3, .15, .14, .3, .15, .14, .0, .0, .2, .3, .15, .14, .3, .15, .14    	; 20 (0)
  db  .6, .6, .6, .14, .15, .15, .8, .11, .9, .8, .15, .1, .9, .13, .1, .15, .15, .7, .6, .6, .6, .0, .0, .0    	; 21 (0)
  db  .3, .8, .0, .3, .8, .0, .3, .8, .0, .3, .8, .0, .3, .8, .0, .3, .8, .0, .3, .8, .0, .0, .0, .0    		; 22 (0)
  db  .0, .0, .0, .2, .4, .8, .2, .12, .12, .3, .15, .14, .3, .15, .14, .2, .12, .12, .2, .4, .8, .0, .0, .0    	; 23 (0)
  db  .0, .0, .0, .0, .0, .8, .0, .0, .12, .3, .15, .14, .3, .15, .14, .0, .0, .12, .0, .0, .8, .0, .0, .0    		; 24 (0)
  db  .0, .0, .0, .0, .8, .0, .1, .8, .0, .3, .15, .14, .3, .15, .14, .1, .8, .0, .0, .8, .0, .0, .0, .0    		; 25 (0)
  db  .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .15, .8, .0, .15, .8, .0, .7, .0, .0, .2, .0    		; 26 (0)
  db  .0, .2, .0, .0, .7, .0, .0, .15, .8, .0, .15, .8, .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .0, .0    		; 27 (0)
  db  .0, .15, .0, .0, .15, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .0, .0    		; 28 (0)
  db  .0, .2, .0, .0, .7, .0, .0, .15, .8, .0, .2, .0, .0, .2, .0, .0, .15, .8, .0, .7, .0, .0, .2, .0    		; 29 (0)
  db  .1, .8, .0, .1, .14, .0, .1, .15, .12, .1, .15, .14, .1, .15, .12, .1, .14, .0, .1, .8, .0, .0, .0, .0    	; 30 (0)
  db  .0, .0, .6, .0, .3, .14, .0, .15, .14, .1, .15, .14, .0, .15, .14, .0, .3, .14, .0, .0, .6, .0, .0, .0    	; 31 (0)
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0    		; 32 (' ')
  db  .0, .0, .0, .0, .0, .0, .0, .1, .12, .3, .7, .14, .3, .7, .14, .0, .1, .12, .0, .0, .0, .0, .0, .0    		; 33 ('!')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .7, .0, .0, .15, .0, .0, .0, .0, .0, .15, .0, .0, .7, .0, .0, .0    		; 34 ('"')
  db  .0, .9, .0, .3, .15, .14, .3, .15, .14, .0, .9, .0, .3, .15, .14, .3, .15, .14, .0, .9, .0, .0, .0, .0    	; 35 ('#')
  db  .1, .1, .8, .3, .3, .12, .2, .6, .4, .14, .6, .7, .14, .6, .7, .3, .12, .12, .1, .8, .8, .0, .0, .0    		; 36 ('$')
  db  .2, .0, .0, .3, .1, .8, .1, .9, .8, .0, .12, .0, .0, .6, .0, .3, .3, .0, .3, .1, .8, .0, .0, .0    		; 37 ('%')
  db  .1, .12, .0, .3, .14, .4, .2, .7, .14, .2, .7, .10, .1, .9, .14, .3, .14, .4, .2, .14, .0, .0, .0, .0    		; 38 ('&')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .8, .0, .0, .15, .0, .0, .7, .0, .0, .0, .0, .0, .0    		; 39 (''')
  db  .0, .0, .0, .0, .0, .0, .0, .15, .8, .1, .15, .12, .3, .0, .6, .2, .0, .2, .0, .0, .0, .0, .0, .0    		; 40 ('(')
  db  .0, .0, .0, .0, .0, .0, .2, .0, .2, .3, .0, .6, .1, .15, .12, .0, .15, .8, .0, .0, .0, .0, .0, .0    		; 41 (')')
  db  .0, .2, .0, .0, .10, .8, .0, .15, .8, .0, .7, .0, .0, .15, .8, .0, .10, .8, .0, .2, .0, .0, .0, .0    		; 42 ('*')
  db  .0, .0, .0, .0, .2, .0, .0, .2, .0, .0, .15, .8, .0, .15, .8, .0, .2, .0, .0, .2, .0, .0, .0, .0    		; 43 ('+')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .4, .0, .0, .7, .8, .0, .3, .8, .0, .0, .0, .0, .0, .0, .0    		; 44 (')
  db  .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .2, .0, .0, .0, .0    		; 45 ('-')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .3, .0, .0, .3, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0    		; 46 ('.')
  db  .3, .0, .0, .1, .8, .0, .0, .12, .0, .0, .6, .0, .0, .3, .0, .0, .1, .8, .0, .0, .12, .0, .0, .0    		; 47 ('/')
  db  .1, .15, .12, .3, .15, .14, .2, .6, .2, .2, .3, .2, .2, .1, .10, .3, .15, .14, .1, .15, .12, .0, .0, .0    	; 48 ('0')
  db  .0, .0, .0, .2, .0, .4, .2, .0, .4, .3, .15, .14, .3, .15, .14, .2, .0, .0, .2, .0, .0, .0, .0, .0    		; 49 ('1')
  db  .3, .0, .12, .3, .8, .14, .2, .12, .2, .2, .6, .2, .2, .3, .2, .3, .1, .14, .3, .0, .12, .0, .0, .0    		; 50 ('2')
  db  .1, .0, .4, .3, .0, .6, .2, .2, .2, .2, .2, .2, .2, .2, .2, .3, .15, .14, .1, .13, .12, .0, .0, .0    		; 51 ('3')
  db  .0, .6, .0, .0, .7, .0, .0, .5, .8, .0, .4, .12, .3, .15, .14, .3, .15, .14, .0, .4, .0, .0, .0, .0    		; 52 ('4')
  db  .1, .3, .14, .3, .3, .14, .2, .2, .2, .2, .2, .2, .2, .2, .2, .3, .14, .2, .1, .12, .2, .0, .0, .0    		; 53 ('5')
  db  .1, .15, .12, .3, .15, .14, .2, .2, .2, .2, .2, .2, .2, .2, .2, .3, .14, .6, .1, .12, .4, .0, .0, .0    		; 54 ('6')
  db  .0, .0, .6, .0, .0, .6, .3, .14, .2, .3, .15, .2, .0, .1, .10, .0, .0, .14, .0, .0, .6, .0, .0, .0    		; 55 ('7')
  db  .1, .13, .12, .3, .15, .14, .2, .2, .2, .2, .2, .2, .2, .2, .2, .3, .15, .14, .1, .13, .12, .0, .0, .0    	; 56 ('8')
  db  .1, .1, .12, .3, .3, .14, .2, .2, .2, .2, .2, .2, .2, .2, .2, .3, .15, .14, .1, .15, .12, .0, .0, .0    		; 57 ('9')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .1, .9, .8, .1, .9, .8, .0, .0, .0, .0, .0, .0    		; 58 (':')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .4, .0, .0, .7, .9, .8, .3, .9, .8, .0, .0, .0, .0, .0, .0    		; 59 (';')
  db  .0, .2, .0, .0, .7, .0, .0, .13, .8, .1, .8, .12, .3, .0, .6, .2, .0, .2, .0, .0, .0, .0, .0, .0    		; 60 ('<')
  db  .0, .5, .0, .0, .5, .0, .0, .5, .0, .0, .5, .0, .0, .5, .0, .0, .5, .0, .0, .5, .0, .0, .0, .0    		; 61 ('=')
  db  .0, .0, .0, .2, .0, .2, .3, .0, .6, .1, .8, .12, .0, .13, .8, .0, .7, .0, .0, .2, .0, .0, .0, .0    		; 62 ('>')
  db  .0, .0, .12, .0, .0, .14, .0, .0, .2, .3, .6, .2, .3, .7, .2, .0, .1, .14, .0, .0, .12, .0, .0, .0    		; 63 ('?')
  db  .1, .15, .12, .3, .15, .14, .2, .0, .2, .2, .15, .2, .2, .15, .2, .2, .15, .14, .2, .7, .12, .0, .0, .0    	; 64 ('@')
  db  .3, .15, .8, .3, .15, .12, .0, .4, .6, .0, .4, .2, .0, .4, .6, .3, .15, .12, .3, .15, .8, .0, .0, .0    		; 65 ('A')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .2, .2, .2, .2, .2, .3, .15, .14, .1, .13, .12, .0, .0, .0    	; 66 ('B')
  db  .0, .15, .8, .1, .15, .12, .3, .0, .6, .2, .0, .2, .2, .0, .2, .3, .0, .6, .1, .0, .4, .0, .0, .0    		; 67 ('C')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .0, .2, .3, .0, .6, .1, .15, .12, .0, .15, .8, .0, .0, .0    		; 68 ('D')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .2, .2, .2, .2, .2, .3, .2, .6, .3, .0, .6, .0, .0, .0    		; 69 ('E')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .2, .2, .0, .2, .2, .0, .2, .6, .0, .0, .6, .0, .0, .0    		; 70 ('F')
  db  .1, .15, .12, .3, .15, .14, .2, .0, .2, .2, .0, .2, .2, .4, .2, .3, .12, .14, .1, .12, .12, .0, .0, .0    	; 71 ('G')
  db  .3, .15, .14, .3, .15, .14, .0, .2, .0, .0, .2, .0, .0, .2, .0, .3, .15, .14, .3, .15, .14, .0, .0, .0    	; 72 ('H')
  db  .0, .0, .0, .0, .0, .0, .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .0, .2, .0, .0, .0, .0, .0, .0    		; 73 ('I')
  db  .1, .8, .0, .3, .8, .0, .2, .0, .2, .3, .15, .14, .1, .15, .14, .0, .0, .2, .0, .0, .0, .0, .0, .0    		; 74 ('J')
  db  .3, .15, .14, .3, .15, .14, .0, .3, .0, .0, .7, .8, .0, .12, .12, .3, .8, .6, .3, .0, .2, .0, .0, .0    		; 75 ('K')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .0, .2, .2, .0, .0, .3, .0, .0, .3, .8, .0, .0, .0, .0    		; 76 ('L')
  db  .3, .15, .14, .3, .15, .14, .0, .1, .8, .0, .15, .0, .0, .1, .8, .3, .15, .14, .3, .15, .14, .0, .0, .0    	; 77 ('M')
  db  .3, .15, .14, .3, .15, .14, .0, .3, .8, .0, .6, .0, .1, .12, .0, .3, .15, .14, .3, .15, .14, .0, .0, .0    	; 78 ('N')
  db  .1, .15, .12, .3, .15, .14, .2, .0, .2, .2, .0, .2, .2, .0, .2, .3, .15, .14, .1, .15, .12, .0, .0, .0    	; 79 ('O')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .2, .2, .0, .2, .2, .0, .3, .14, .0, .1, .12, .0, .0, .0    		; 80 ('P')
  db  .1, .15, .12, .3, .15, .14, .2, .0, .2, .3, .0, .2, .2, .0, .2, .7, .15, .14, .5, .15, .12, .0, .0, .0    	; 81 ('Q')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .0, .6, .2, .0, .14, .2, .3, .11, .14, .3, .1, .12, .0, .0, .0    	; 82 ('R')
  db  .1, .0, .12, .3, .1, .14, .2, .3, .2, .2, .2, .2, .2, .6, .2, .3, .12, .6, .1, .8, .4, .0, .0, .0    		; 83 ('S')
  db  .0, .0, .0, .0, .0, .6, .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .0, .2, .0, .0, .6, .0, .0, .0    		; 84 ('T')
  db  .1, .15, .14, .3, .15, .14, .2, .0, .0, .2, .0, .0, .2, .0, .0, .3, .15, .14, .1, .15, .14, .0, .0, .0    	; 85 ('U')
  db  .0, .7, .14, .0, .15, .14, .1, .8, .0, .3, .0, .0, .1, .8, .0, .0, .15, .14, .0, .7, .14, .0, .0, .0    		; 86 ('V')
  db  .3, .15, .14, .3, .15, .14, .0, .12, .0, .0, .7, .8, .0, .12, .0, .3, .15, .14, .3, .15, .14, .0, .0, .0    	; 87 ('W')
  db  .3, .0, .6, .3, .8, .14, .0, .15, .8, .0, .7, .0, .0, .15, .8, .3, .8, .14, .3, .0, .6, .0, .0, .0    		; 88 ('X')
  db  .0, .0, .0, .0, .1, .14, .2, .3, .14, .3, .14, .0, .3, .14, .0, .2, .3, .14, .0, .1, .14, .0, .0, .0    		; 89 ('Y')
  db  .3, .8, .14, .3, .12, .6, .2, .6, .2, .2, .3, .2, .2, .1, .10, .3, .0, .14, .3, .8, .6, .0, .0, .0    		; 90 ('Z')
  db  .0, .0, .0, .3, .15, .14, .3, .15, .14, .2, .0, .2, .2, .0, .2, .2, .0, .2, .0, .0, .0, .0, .0, .0    		; 91 ('[')
  db  .0, .0, .12, .0, .1, .8, .0, .3, .0, .0, .6, .0, .0, .12, .0, .1, .8, .0, .3, .0, .0, .0, .0, .0    		; 92 ('\')
  db  .0, .0, .0, .2, .0, .2, .2, .0, .2, .2, .0, .2, .3, .15, .14, .3, .15, .14, .0, .0, .0, .0, .0, .0    		; 93 (']')
  db  .0, .0, .8, .0, .0, .12, .0, .0, .6, .0, .0, .3, .0, .0, .6, .0, .0, .12, .0, .0, .8, .0, .0, .0    		; 94 ('^')
  db  .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0, .8, .0, .0    		; 95 ('_')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .0, .7, .0, .0, .15, .0, .0, .8, .0, .0, .0, .0, .0, .0    		; 96 ('`')
  db  .1, .8, .0, .3, .13, .0, .2, .5, .0, .3, .5, .0, .1, .15, .0, .3, .14, .0, .2, .0, .0, .0, .0, .0    		; 97 ('a')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .1, .0, .2, .1, .0, .3, .15, .0, .1, .14, .0, .0, .0, .0    		; 98 ('b')
  db  .1, .14, .0, .3, .15, .0, .2, .1, .0, .2, .1, .0, .2, .1, .0, .3, .3, .0, .1, .2, .0, .0, .0, .0    		; 99 ('c')
  db  .1, .14, .0, .3, .15, .0, .2, .1, .0, .2, .1, .2, .3, .15, .14, .3, .15, .14, .2, .0, .0, .0, .0, .0    		; 100 ('d')
  db  .1, .14, .0, .3, .15, .0, .2, .5, .0, .2, .5, .0, .2, .5, .0, .3, .7, .0, .1, .6, .0, .0, .0, .0    		; 101 ('e')
  db  .0, .2, .0, .2, .2, .0, .3, .15, .12, .3, .15, .14, .2, .2, .2, .0, .2, .6, .0, .0, .4, .0, .0, .0    		; 102 ('f')
  db  .4, .14, .0, .13, .15, .0, .9, .1, .0, .9, .1, .0, .9, .2, .0, .15, .15, .0, .7, .15, .0, .0, .0, .0    		; 103 ('g')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .0, .2, .0, .0, .1, .0, .3, .15, .0, .3, .14, .0, .0, .0, .0    		; 104 ('h')
  db  .0, .0, .0, .0, .0, .0, .2, .1, .0, .3, .15, .6, .3, .15, .6, .2, .0, .0, .0, .0, .0, .0, .0, .0    		; 105 ('i')
  db  .6, .0, .0, .14, .0, .0, .8, .0, .0, .8, .1, .0, .15, .15, .6, .7, .15, .6, .0, .0, .0, .0, .0, .0    		; 106 ('j')
  db  .2, .0, .2, .3, .15, .14, .3, .15, .14, .0, .4, .0, .0, .14, .0, .3, .11, .0, .3, .1, .0, .0, .0, .0    		; 107 ('k')
  db  .0, .0, .0, .0, .0, .0, .2, .0, .2, .3, .15, .14, .3, .15, .14, .2, .0, .0, .0, .0, .0, .0, .0, .0    		; 108 ('l')
  db  .3, .14, .0, .3, .15, .0, .0, .3, .0, .0, .14, .0, .0, .3, .0, .3, .15, .0, .3, .14, .0, .0, .0, .0    		; 109 ('m')
  db  .0, .1, .0, .3, .15, .0, .3, .14, .0, .0, .1, .0, .0, .1, .0, .3, .15, .0, .3, .14, .0, .0, .0, .0    		; 110 ('n')
  db  .1, .14, .0, .3, .15, .0, .2, .1, .0, .2, .1, .0, .2, .1, .0, .3, .15, .0, .1, .14, .0, .0, .0, .0    		; 111 ('o')
  db  .8, .1, .0, .15, .15, .0, .15, .14, .0, .9, .1, .0, .1, .1, .0, .1, .15, .0, .0, .14, .0, .0, .0, .0    		; 112 ('p')
  db  .0, .14, .0, .1, .15, .0, .1, .1, .0, .9, .1, .0, .15, .14, .0, .15, .15, .0, .8, .1, .0, .0, .0, .0    		; 113 ('q')
  db  .2, .1, .0, .3, .15, .0, .3, .14, .0, .2, .1, .0, .0, .1, .0, .0, .3, .0, .0, .2, .0, .0, .0, .0    		; 114 ('r')
  db  .1, .2, .0, .3, .7, .0, .2, .5, .0, .2, .13, .0, .2, .9, .0, .3, .11, .0, .1, .2, .0, .0, .0, .0    		; 115 ('s')
  db  .0, .1, .0, .0, .1, .0, .1, .15, .14, .3, .15, .14, .2, .1, .0, .3, .1, .0, .1, .0, .0, .0, .0, .0    		; 116 ('t')
  db  .1, .15, .0, .3, .15, .0, .2, .0, .0, .2, .0, .0, .1, .15, .0, .3, .15, .0, .2, .0, .0, .0, .0, .0    		; 117 ('u')
  db  .0, .7, .0, .0, .15, .0, .1, .8, .0, .3, .0, .0, .1, .8, .0, .0, .15, .0, .0, .7, .0, .0, .0, .0    		; 118 ('v')
  db  .1, .15, .0, .3, .15, .0, .3, .0, .0, .1, .12, .0, .3, .0, .0, .3, .15, .0, .1, .15, .0, .0, .0, .0    		; 119 ('w')
  db  .2, .1, .0, .3, .3, .0, .1, .14, .0, .0, .12, .0, .1, .14, .0, .3, .3, .0, .2, .1, .0, .0, .0, .0    		; 120 ('x')
  db  .4, .15, .0, .13, .15, .0, .9, .0, .0, .9, .0, .0, .8, .8, .0, .15, .15, .0, .7, .15, .0, .0, .0, .0    		; 121 ('y')
  db  .2, .3, .0, .3, .1, .0, .3, .9, .0, .2, .13, .0, .2, .7, .0, .2, .3, .0, .3, .1, .0, .0, .0, .0    		; 122 ('z')
  db  .0, .0, .0, .0, .2, .0, .0, .2, .0, .1, .15, .12, .3, .13, .14, .2, .0, .2, .2, .0, .2, .0, .0, .0    		; 123 ('{')
  db  .0, .0, .0, .0, .0, .0, .0, .0, .0, .3, .13, .14, .3, .13, .14, .0, .0, .0, .0, .0, .0, .0, .0, .0    		; 124 ('|')
  db  .0, .0, .0, .2, .0, .2, .2, .0, .2, .3, .13, .14, .1, .15, .12, .0, .2, .0, .0, .2, .0, .0, .0, .0    		; 125 ('}')
  db  .0, .0, .4, .0, .0, .6, .0, .0, .2, .0, .0, .6, .0, .0, .4, .0, .0, .6, .0, .0, .2, .0, .0, .0    		; 126 ('~')
  db  .1, .0, .0, .1, .12, .0, .1, .15, .0, .1, .3, .8, .1, .15, .0, .1, .12, .0, .1, .0, .0, .0, .0, .0    		; 127 (0)


;---------------------------------------------------------------------
; TBLDRAW
; 
; Copyied into RAM at statup (tbldraw area)
;
; First 16 bytes give the pattern for combined white/back/white combinaisons :
;
; tblcombined[]={
;	"1000","1100","1110","1111",
;	"0100","0100","0110","0111",
;	"0010","0010","0010","0011",
;	"0001","0001","0001","0001"}
;
; Next 4 bytes gives the pattern for white/back combinaisons :
;
; tblmin[]={"1111","0111","0011","0001"}
;
; Next 4 bytes gives the pattern for back/white combinaisons :
;
; tblmax[]={"1000","1100","1110","1111"}
;---------------------------------------------------------------------

TBLDRAW
	db	B'00001000',B'00001100',B'00001110',B'00001111'
	db	B'00000100',B'00000100',B'00000110',B'00000111'
	db	B'00000010',B'00000010',B'00000010',B'00000011'
	db	B'00000001',B'00000001',B'00000001',B'00000001'

	db	B'00001111',B'00000111',B'00000011',B'00000001'

	db	B'00001000',B'00001100',B'00001110',B'00001111'

;=====================================================================
;
;
;
; BINARY TO DECIMAL CONVERSION
;
;
;
;=====================================================================


; PIC Code for Binary to Decimal Conversion Part of the Binary to Decimal Conversion 
; Tutorial  <http://www.cs.uiowa.edu/~jones/bcd/decimal.html> 
; by Douglas W. Jones <http://www.cs.uiowa.edu/~jones/>  
; THE UNIVERSITY OF IOWA <http://www.uiowa.edu/> 
; Department of Computer Science <http://www.cs.uiowa.edu/> 
; The following decimal print routine for the 14-bit family of PIC microcontrollers 
; is lifted from production code I developed. The production code comes from a 
; commercial product, and I have verified that no substantial changes have been made 
; in the process of extracting it from that context; nonetheless, the usual 
; disclaimers apply! 

conv16bin_to_ascii
        ; preconditions 
        ;   TEM0 contains least significant 8 bits of 16-bit operand 
        ;   TEM2 contains most significant 8 bits of 16-bit operand 
        ;   the 16-bit operand is positive (if it was negative, it 
        ;     already been negated and the sign printed) 

        ; resource usage 
        ;   TEM0 holds low nibble, is converted to low decimal digit 
        ;   TEM1 holds second nibble, is converted to second digit 
        ;   TEM2 holds third nibble, converted to third digit 
        ;   TEM3 holds 4th nipple, converted to 4th digit 
        ;   TEMPH short-term scratch 
        ;   TEMPL short-term scratch 
        ; 
        ;   calls PUTDIG, to prints each decimal digit 
        ;     when called with an ingeger from 0 to 9 in W 
        ;     PUTDIG may not use TEM0, TEM1, TEM2, TEM3 
        ; 
        ;   all internal labels have the prefix PUTDEC 
   
        ; copyright 2002, Douglas W. Jones 
        ;   this code may be incorporated into any product so long 
        ;   this notice is preserved in the source code and so long 
        ;   as any copyright notice on the final software or firmware 
        ;   product gives credit to the author. 

        ; warranty 
        ;   This is freeware, you get what you pay for.  The author 
        ;   is pretty sure this code works, but can't offer anything 
        ;   better than that.  He who transcribes this code for 
        ;   incorporation into some larger program must take full 
        ;   responsibility for any errors. 

	; Modified by (name deleted for the contest), 2002, in order to
	; - convert result to ASCII chars
	; - store result in convasc4..0

  
; needed for the 18F...
#define RLF RLCF
#define RRF RRCF

	movlw	' '            	; init output to blank
	movwf	convasc4
	movwf	convasc3
	movwf	convasc2
	movwf	convasc1
	movwf	convasc0

        SWAPF   TEM0,W          ; get high nibble of low byte 
        MOVWF   TEM1            ;   into position 
        SWAPF   TEM2,W          ; get high nibble of high byte 
        MOVWF   TEM3            ;   into position 
        MOVLW   0x0F            ; setup to mask nibbles 
        ANDWF   TEM0,F          ;   in each nibble, use only low 4 bits 
        ANDWF   TEM1,F 
        ANDWF   TEM2,F 
        ANDWF   TEM3,F 

        ; TEM3 ... TEM0 hold 4 nibbles of number, TEM0 holds LSB 
        ; TEM0 = (TEM3 + TEM2 + TEM1)*6 + TEM0 
        ; TEM0 <=(   8 +   15 +   15)*6 +   15 = 243 
        MOVF    TEM3,W 
        ADDWF   TEM2,W 
        ADDWF   TEM1,W          ; note, C reset because TEM1+TEM2+TEM3 <= 38 
        MOVWF   TEMPL           ; TEMPL = W = (TEM3+TEM2+TEM1) 
        RLF     TEMPL,F         ; TEMPL =(TEM3+TEM2+TEM1)*2 
        ADDWF   TEMPL,F         ; TEMPL =(TEM3+TEM2+TEM1)*3(C reset,TEMPL<=114) 
        RLF     TEMPL,W         ; W     =(TEM3+TEM2+TEM1)*6(C reset, W <= 228) 
        ADDWF   TEM0,F          ; TEM0 done; TEM0 <= 243, C is still reset 

        ; note TEMPH and TEMPL are no-longer needed 
        ; TEMPH = TEM0 div 10 is carry into next digit 
        ; TEM0  = TEM0 mod 10 is the least significant digit 

        ; approximate TEM0/10 by TEM0*.000110011 (assume C already reset) 
        MOVF    TEM0,W 
        MOVWF   TEMPH           ; TEMPH = TEM0*1. 
        RRF     TEMPH,F         ; TEMPH = TEM0*0.1 (assumes C initially zero) 
        ADDWF   TEMPH,F         ; TEMPH = TEM0*1.1 (result high bit in C) 
        RRF     TEMPH,F         ; TEMPH = TEM0*0.11 (high bit recovered) 
        SWAPF   TEMPH,W         ; W     = TEM0*0.000011 
        ANDLW   0x0F            ; (discard high bits) 
        ADDWF   TEMPH,F         ; TEMPH = TEM0*0.110011 (result high bit in C) 
        RRF     TEMPH,F         ; TEMPH = TEM0*0.0110011 (high bit recovered) 
        RRF     TEMPH,F         ; TEMPH = TEM0*0.00110011 (bit of junk) 
        RRF     TEMPH,F         ; TEMPH = TEM0*0.000110011 (2 bits of junk) 
        MOVLW   0x3F            ; (discard junk bits) 
        ANDWF   TEMPH,F 

        ; TEM0 = TEM0 - 10*TEMPH, the remainder approximation 
        RLF     TEMPH,W         ; high bits of TEMPH are zero, so this clears C 
        ANDLW   0xFE            ; (clear any random carry in) 
        MOVWF   TEMPL           ; TEMPL = W = TEMPH*2 
        RLF     TEMPL,F         ; TEMPL = TEMPH*4 
        RLF     TEMPL,F         ; TEMPL = TEMPL*8 
        ADDWF   TEMPL,W         ; W = TEMPH*8 + TEMPH*2 
        SUBWF   TEM0,F          ; TEM0 = TEM0 - 10*TEMPH 

        ; TEMPH may be off by 1 and TEM0 off by 10 
        MOVLW   -D'10' 
        ADDWF   TEM0,W          ; W = TEM0 - 10 (sets C if TEM0 >= 10) 
        BTFSC   STATUS,C 
        MOVWF   TEM0            ; if W >= 0, TEM0 = TEM0 - 10 
        BTFSC   STATUS,C 
        INCF    TEMPH,F         ; if W >= 0, TEMPH++ 

        ; TEM0 is now the least significant decimal digit! 

        ; TEM1 = TEMPH + TEM3*9 + TEM2*5 + TEM1 
        ; TEM1 <=   24 +    8*9 +   15*5 +   15 = 186 (carry never set!) 
        MOVF    TEMPH,W 
        ADDWF   TEM3,W 
        ADDWF   TEM2,W 
        ADDWF   TEM1,F          ; TEM1 = TEMPH + TEM3*1 + TEM2*1 + TEM1(C reset) 
        RLF     TEM3,W 
        ADDWF   TEM2,W          ; W    =         TEM3*2 + TEM2*1 
        MOVWF   TEMPL 
        RLF     TEMPL,F         ; TEMPL =        TEM3*4 + TEM2*2 
        RLF     TEMPL,W         ; W    =         TEM3*8 + TEM2*4 
        ADDWF   TEM1,F          ; TEM1 = TEMPH + TEM3*9 + TEM2*5 + TEM1 
        BTFSC   STATUS,Z 
        GOTO    PUTDEC1         ; if rest of number is zero, print just 1 digit 

        ; TEMPH = TEM1 div 10 is carry into next digit 
        ; TEM1  = TEM1 mod 10 is the next to the least significant digit 

        ; approximate TEM1/10 by TEM1*.000110011 
        MOVF    TEM1,W 
        MOVWF   TEMPH           ; TEMPH = TEM1*1. 
        RRF     TEMPH,F         ; TEMPH = TEM1*0.1 
        ADDWF   TEMPH,F         ; TEMPH = TEM1*1.1 (high bit is in carry) 
        RRF     TEMPH,F         ; TEMPH = TEM1*0.11 
        SWAPF   TEMPH,W         ; W     = TEM1*0.000011 
        ANDLW   0x0F            ; (discard high bits) 
        ADDWF   TEMPH,F         ; TEMPH = TEM1*0.110011 
        RRF     TEMPH,F         ; TEMPH = TEM1*0.0110011 
        RRF     TEMPH,F         ; TEMPH = TEM1*0.00110011 
        RRF     TEMPH,F         ; TEMPH = TEM1*0.000110011 
        MOVLW   0x3F            ; (discard high bits) 
        ANDWF   TEMPH,F 

        ; TEM1 = TEM1 - 10*TEMPH, the remainder approximation 
        RLF     TEMPH,W 
        ANDLW   0xFE            ; (clear any random carry in) 
        MOVWF   TEMPL           ; TEMPL = W = TEMPH*2 
        RLF     TEMPL,F         ; TEMPL = TEMPH*4 
        RLF     TEMPL,F         ; TEMPL = TEMPL*8 
        ADDWF   TEMPL,W         ; W = TEMPH*8 + TEMPH*2 
        SUBWF   TEM1,F          ; TEM1 = TEM1 - 10*TEMPH 

        ; TEMPH may be off by 1 and TEM1 off by 10 
        MOVLW   -D'10' 
        ADDWF   TEM1,W          ; W = TEM1 - 10 
        BTFSC   STATUS,C 
        MOVWF   TEM1            ; if W >= 0, TEM1 = TEM1 - 10 
        BTFSC   STATUS,C 
        INCF    TEMPH,F         ; if W >= 0, TEMPH++ 

        ; TEM1 is the next to the least significant decimal digit! 

        ; TEM2 = TEMPH + TEM2*2 
        ; TEM2 <=   18 +   15*2 = 48 (no carry, and top 2 bits unused too) 
        MOVF    TEMPH,W 
        ADDWF   TEM2,W 
        ADDWF   TEM2,F          ; TEM2 = TEMPH + TEM2*2 (never carry out!) 
        MOVF    TEM2,W          ; check if need to print high digits 
        IORWF   TEM3,W          ; TEM3 must be rolled into test! 
        BTFSC   STATUS,Z 
        GOTO    PUTDEC2         ; if rest is zero, print just 2 digits 

        ; TEMPH = TEM2 div 10 is carry into high two digits 
        ; TEM2  = TEM2 mod 10 is the middle digit 

        ; get exact TEM2/10 by TEM2*.0001101 (good because TEM2 < 68) 
        MOVF    TEM2,W          ; W     = TEM2*1.0 
        MOVWF   TEMPH           ; TEMPH = TEM2*1.0 
        RRF     TEMPH,F         ; TEMPH = TEM2*0.1 
        RRF     TEMPH,F         ; TEMPH = TEM2*0.01 
        BCF     TEMPH,7         ; (discard high bit; next to high already clear) 
        ADDWF   TEMPH,F         ; TEMPH = TEM2*1.01 
        RRF     TEMPH,F         ; TEMPH = TEM2*0.101 
        ADDWF   TEMPH,F         ; TEMPH = TEM2*1.101 (does not set carry!) 
        SWAPF   TEMPH,W         ; W     = TEM2*0.0001101 
        ANDLW   0x0F            ; (discard high bits) 
        MOVWF   TEMPH           ; W = TEMPH = TEM2*0.0001101 = TEM2/10 

        ; TEM2 = TEM2 - 10*TEMPH, the remainder 
        RLF     TEMPH,W 
        MOVWF   TEMPL           ; TEMPL = W = TEMPH*2 
        RLF     TEMPL,F         ; TEMPL = TEMPH*4 
        RLF     TEMPL,F         ; TEMPL = TEMPL*8 
        ADDWF   TEMPL,W         ; W = TEMPH*8 + TEMPH*2 
        SUBWF   TEM2,F          ; TEM2 = TEM2 - 10*TEMPH 

        ; TEM2 is the middle decimal digit! 

        ; TEM3 = TEMPH + TEM3*4 
        ; TEM3 <=    8 +    8*4 = 40 (no carry, and top 2 bits unused too) 
        BCF     STATUS,C 
        RLF     TEM3,F          ; TEM3 =         TEM3*2 
        RLF     TEM3,W          ; TEM3 =         TEM3*4 
        ADDWF   TEMPH,W         ; W    = TEMPH + TEM3*4 (no carry!) 
        BTFSC   STATUS,Z 
        GOTO    PUTDEC3         ; if rest is zero, just print bottom 3 digits 

        ; TEMPH = W div 10 is the high digit 
        ; TEM3  = W mod 10 is the least significant digit 

        ; get exact W/10 by W*.0001101 (good because W < 68) 
        MOVWF   TEM3            ; TEM3  = W*1.0 
        MOVWF   TEMPH           ; TEMPH = TEM3*1.0 
        RRF     TEMPH,F         ; TEMPH = TEM3*0.1   (carry in is zero!) 
        RRF     TEMPH,F         ; TEMPH = TEM3*0.01 
        BCF     TEMPH,7         ; (discard high bit; next to high already clear) 
        ADDWF   TEMPH,F         ; TEMPH = TEM3*1.01  (carry out is zero!) 
        RRF     TEMPH,F         ; TEMPH = TEM3*0.101 
        ADDWF   TEMPH,F         ; TEMPH = TEM3*1.101 (does not set carry!) 
        SWAPF   TEMPH,W         ; W     = TEM3*0.0001101 
        ANDLW   0x0F            ; (discard high bits) 
        MOVWF   TEMPH           ; TEMPH = TEM3*0.0001101 

        ; TEM3 = TEM3 - 10*TEMPH, the remainder 
        RLF     TEMPH,W         ; W     = TEMPH*2 
        MOVWF   TEMPL           ; TEMPL = TEMPH*2 
        RLF     TEMPL,F         ; TEMPL = TEMPH*4 
        RLF     TEMPL,F         ; TEMPL = TEMPL*8 
        ADDWF   TEMPL,W         ; W = TEMPH*8 + TEMPH*2 
        SUBWF   TEM3,F          ; TEM3 = TEM3 - 10*TEMPH 

        ; TEM3 is the next to most significant decimal digit! 
        ; TEMPH is the most significant decimal digit! 

        ; conversion to decimal is done! now output digits 
        MOVF    TEMPH,W 
        BTFSC   STATUS,Z        ; don't print it if leading digit is zero 
	bra	dontprintlead
        addlw	'0'            	; output char 
	movwf	convasc4
dontprintlead
        MOVF    TEM3,W 
        addlw	'0'             ; output char 
	movwf	convasc3
PUTDEC3: 
        MOVF    TEM2,W 
        addlw	'0'             ; output char 
	movwf	convasc2
PUTDEC2: 
        MOVF    TEM1,W 
        addlw	'0'             ; output char 
	movwf	convasc1
PUTDEC1: 
        MOVF    TEM0,W 
        addlw	'0'             ; output char 
	movwf	convasc0

        ; postconditions 
        ;   the number has been printed by a series of calls to PUTDIG 
        ;   TEM0, TEM1, TEM2, TEM3, TEMPH and TEMPL have been used 
        ;   along with any memory locations used by PUTDIG.

	retlw	0


;---------------------------------------------------------------------
;
; that's all, folks !
;
; (should end before $7FFF)
;
;---------------------------------------------------------------------

	end

