/* WWW Interface Version
 *
 *	web_cw_beacon.c, multiplx at dct.com based on:
 *
 *	cwbeacon              Part of the beaconProject, ea4rj@amsat.org
 *	GW4PTS Morse Tutor    By Alan Cox (morse.c 1.00, 21/09/93)
 *	
 *	Check <metalab.unc.edu/pub/Linux/apps/ham/morse> for these programs
*/

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>

#ifdef __FreeBSD__
#include <machine/cpufunc.h>
#else
#include <asm/io.h>
#endif

#define BASE_ADDRESS	0x3BC	/* lp0 = 0x3BC, lp1 = 0x378, lp2 = 0x278 */
#define MSG_FILE	"/tmp/.cw_beacon"	/* holds user defined message */
#define UserID		65534	/* uid of who should own that file (httpd) */
#define INTERVAL	120	/* seconds to sleep after ID/message */

/* default ID */
char id[] = "v v de CALL/b GRID li qsl via EMAIL";

char usr_message[64], speed[3], delay[2], ant[2], att[2];

int morse_gap = 1;		/* gap between characters */
int morse_speed = 12;		/* morse speed */
unsigned char morse_att = 0;	/* attenuator on/off */
unsigned char morse_ant = 0;	/* antenna select */
long int dot = 1000000;		/* dot length = unit */
long int dash = 3000000;	/* dash length = 3-dot */
long int tone = 2000000;	/* initial tone length * morse_speed */

extern int errno;

typedef struct
{
  char code;
  enum
  {
    NIL,
    DIH,
    DAH,
  }
  data[7];
}
Morse;

Morse MorseTable[] = {
  ' ', NIL, NIL, NIL, NIL, NIL, NIL, NIL,
  'A', DIH, DAH, NIL, NIL, NIL, NIL, NIL,
  'B', DAH, DIH, DIH, DIH, NIL, NIL, NIL,
  'C', DAH, DIH, DAH, DIH, NIL, NIL, NIL,
  'D', DAH, DIH, DIH, NIL, NIL, NIL, NIL,
  'E', DIH, NIL, NIL, NIL, NIL, NIL, NIL,
  'F', DIH, DIH, DAH, DIH, NIL, NIL, NIL,
  'G', DAH, DAH, DIH, NIL, NIL, NIL, NIL,
  'H', DIH, DIH, DIH, DIH, NIL, NIL, NIL,
  'I', DIH, DIH, NIL, NIL, NIL, NIL, NIL,
  'J', DIH, DAH, DAH, DAH, NIL, NIL, NIL,
  'K', DAH, DIH, DAH, NIL, NIL, NIL, NIL,
  'L', DIH, DAH, DIH, DIH, NIL, NIL, NIL,
  'M', DAH, DAH, NIL, NIL, NIL, NIL, NIL,
  'N', DAH, DIH, NIL, NIL, NIL, NIL, NIL,
  'O', DAH, DAH, DAH, NIL, NIL, NIL, NIL,
  'P', DIH, DAH, DAH, DIH, NIL, NIL, NIL,
  'Q', DAH, DAH, DIH, DAH, NIL, NIL, NIL,
  'R', DIH, DAH, DIH, NIL, NIL, NIL, NIL,
  'S', DIH, DIH, DIH, NIL, NIL, NIL, NIL,
  'T', DAH, NIL, NIL, NIL, NIL, NIL, NIL,
  'U', DIH, DIH, DAH, NIL, NIL, NIL, NIL,
  'V', DIH, DIH, DIH, DAH, NIL, NIL, NIL,
  'W', DIH, DAH, DAH, NIL, NIL, NIL, NIL,
  'X', DAH, DIH, DIH, DAH, NIL, NIL, NIL,
  'Y', DAH, DIH, DAH, DAH, NIL, NIL, NIL,
  'Z', DAH, DAH, DIH, DIH, NIL, NIL, NIL,
  '1', DIH, DAH, DAH, DAH, DAH, NIL, NIL,
  '2', DIH, DIH, DAH, DAH, DAH, NIL, NIL,
  '3', DIH, DIH, DIH, DAH, DAH, NIL, NIL,
  '4', DIH, DIH, DIH, DIH, DAH, NIL, NIL,
  '5', DIH, DIH, DIH, DIH, DIH, NIL, NIL,
  '6', DAH, DIH, DIH, DIH, DIH, NIL, NIL,
  '7', DAH, DAH, DIH, DIH, DIH, NIL, NIL,
  '8', DAH, DAH, DAH, DIH, DIH, NIL, NIL,
  '9', DAH, DAH, DAH, DAH, DIH, NIL, NIL,
  '0', DAH, DAH, DAH, DAH, DAH, NIL, NIL,
  '/', DAH, DIH, DIH, DAH, DIH, NIL, NIL,
  '*', DIH, DAH, DIH, DAH, DIH, NIL, NIL,	/* 'AR', end of message */
  '.', DIH, DAH, DIH, DAH, DIH, DAH, NIL,
  '-', DAH, DIH, DIH, DIH, DIH, DAH, NIL,
  ',', DAH, DAH, DIH, DIH, DAH, DAH, NIL,
  '?', DIH, DIH, DAH, DAH, DIH, DIH, NIL,
    0, NIL, NIL, NIL, NIL, NIL, NIL, NIL	/* END MARKER */
};

Morse *
CharToMorse (char c)
{
  int ct = 0;

  while (MorseTable[ct].code)
    {
      if (MorseTable[ct].code == c)
	return (&MorseTable[ct]);
      ct++;
    }
  return (NULL);
}

void
Key (long duration)
{

#ifdef __FreeBSD__
  int fd;
  if (fd = open ("/dev/io", O_RDONLY) < 0)
#else
  if (ioperm (BASE_ADDRESS, 3, 1) < 0)
#endif
    {
      fprintf (stderr, "ERROR: Can't open port 0x%x\n", BASE_ADDRESS);
      perror ("port");
      exit (errno);
    }

#ifdef __FreeBSD__
  outb (BASE_ADDRESS, 1 + morse_att + morse_ant);
#else
  outb (1 + morse_att + morse_ant, BASE_ADDRESS);
#endif

  usleep (duration / morse_speed);

#ifdef __FreeBSD__
  outb (BASE_ADDRESS, 0 + morse_att + morse_ant);
  close (fd);
#else
  outb (0 + morse_att + morse_ant, BASE_ADDRESS);
#endif
}

void
PlayMorse (Morse * m)
{
  int d = 0;

  while (d < 6 && m->data[d] != NIL)
    {
      if (m->data[d] == DIH)
	Key (dot);
      if (m->data[d] == DAH)
	Key (dash);
      d++;
      usleep (dot / morse_speed);	/* dot-dash delay */
    }

  usleep (dash / morse_speed + 100000 * morse_gap);	/* morse signs delay */
}

void
PlayMorseString (char *x)
{

  while (*x)
    {
      char c = islower (*x) ? toupper (*x) : *x;
      Morse *m = CharToMorse (c);

      if (m != NULL)
	PlayMorse (m);
      x++;
    }

  sleep (INTERVAL);
}

int
main (int argc, char **argv)
{
  extern char *optarg;
  struct stat filebuf;
  FILE *fp;

  if (geteuid () && getuid ())

    {
      fprintf (stderr,
	       "%s must be run as root to access hardware ports.\n", argv[0]);
      fprintf (stderr, "Leaving...\n");
      exit (1);
    }

/* Do main loop */

  for (;;)
    {

      if (lstat (MSG_FILE, &filebuf) < 0)
	filebuf.st_uid = 1;	/* file not found */

      if (filebuf.st_uid == UserID)
	{

	  if ((fp = fopen (MSG_FILE, "r")) == NULL)
	    {
	      perror ("fopen");
	      exit (errno);
	    }

	  fscanf (fp, "%s\n%s\n%s\n%s\n%64c", speed, delay, att, ant,
		  usr_message);
	  fclose (fp);

	  unlink (MSG_FILE);	/* reset for next user */

	  morse_speed = atoi (speed);
	  morse_gap = atoi (delay);
	  morse_att = atoi (att);
	  morse_ant = atoi (ant);

	  /* CW key user message */
	  Key (tone * morse_speed);	/* key initial tone */
	  PlayMorseString (usr_message);
	}

      else
	{
	  morse_speed = 12;	/* defaults - copy from above */
	  morse_gap = 1;
	  morse_att = 0;
	  morse_ant = 0;

	  /* CW key normal message (ID) */
	  Key (tone * morse_speed);	/* key initial tone */
	  PlayMorseString (id);
	}
    }
}
