// 
// AMD Expanded Debugging utility for Windows
//   The application is used to control some of the secret MSR registers,
//   internally used by the AMD-series processors for specifying hidden
//   debugging features.
//
//
// Author:  j00ru//vx
// Date:    2010-11-23
// E-mail:  j00ru.vx@gmail.com
// WWW:     http://j00ru.vexillium.org/
//
// Version: 0.0.1
// Module:  User-mode client
//
// Note: Special thanks to Czernobyl (http://www.czerno.tk/) for taking the time
//       to actually discover and reverse engineer the yet unknown processor 
//       internals.
//
// References:
//   1) http://www.woodmann.com/forum/showthread.php?13891
//      The original forum post, where Czarnobyl shared initial information related
//      to the mystery mechanism.
//
//   2) http://www.czerno.tk/
//      Czernobyl's home-page, with a brief description of the current progress state.
//
//   3) http://hardware.slashdot.org/story/10/11/12/047243/Hidden-Debug-Mode-Found-In-AMD-Processors
//      A slashdot article regarding the issue, blocking the woodmann forums for about a day.
//
//   4) http://cbid.softnology.biz/html/undocmsrs.html
//      Undocumented Machine-Specific Registers: internal names of the MSR registers, which
//      are not publicly known or taken advantage of.
//
//
// Known bugs:
//   1) The code is only compatible with the x86 (32-bit) AMD architecture.
//   2) ?
//
//
// License:
//
//   Permission is hereby granted to use, copy, modify, and distribute this source code, 
//   or portions hereof, documentation and executables, for any purpose, without fee, 
//   subject to the following restrictions:
//   
//   1. The origin of this source code must not be misrepresented.
//  
//   2. Altered versions must be plainly marked as such and must not be misrepresented 
//      as being the original source.
//  
//   3. This Copyright notice may not be removed or altered from any source or altered 
//      source distribution.
//   
//   This software is provided "AS IS". The author does not guarantee that this program works 
//   or is bug-free. The author takes no responsibility for any damage caused by this program.
//   Use at your own risk.
//
#define _WIN32_WINNT 0x0600
#include <cstdio>
#include <cstdlib>
#include <string>
#include <vector>
#include <windows.h>
#include <ddk\ntapi.h>
#include <psapi.h>
#include <shlwapi.h>
#include <dbghelp.h>
#include "driver.h"
using namespace std;


/****************************************************************************
 * Helper preprocessor macros ***********************************************
 ****************************************************************************/


#define PRINT_ATTR(text,attr) {\
                                SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE),attr);\
                                printf text;\
                                fflush(stdout);\
                                SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE),orgConsoleInfo.wAttributes);\
                              }

#define INFO(text) {\
                     PRINT_ATTR(("[INFO]  "),orgConsoleInfo.wAttributes);\
                     PRINT_ATTR(text,orgConsoleInfo.wAttributes);\
                   }

#define ERR(text)  {\
                     PRINT_ATTR(("[ERROR] "),FOREGROUND_RED|FOREGROUND_INTENSITY);\
                     PRINT_ATTR(text,FOREGROUND_RED|FOREGROUND_INTENSITY);\
                   }
#define DIE(text)  {\
                     PRINT_ATTR(text,FOREGROUND_RED|FOREGROUND_INTENSITY);\
                     ExitProcess(1);\
                   }
#define SUCCESS    {\
                      PRINT_ATTR(("Success.\r\n"),FOREGROUND_GREEN|FOREGROUND_INTENSITY);\
                      SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE),orgConsoleInfo.wAttributes);\
                   }

#define INPUT(text) {\
                     PRINT_ATTR(("[INPUT] "),orgConsoleInfo.wAttributes);\
                     PRINT_ATTR(text,orgConsoleInfo.wAttributes);\
                   }




/****************************************************************************
 * Constants ****************************************************************
 ****************************************************************************/
const CHAR ServiceName[]    = "AMD_dbg";
const CHAR DeviceName[]     = "\\\\.\\AMD_dbg";




/****************************************************************************
 * Constants ****************************************************************
 ****************************************************************************/
HANDLE hDevice;
CONSOLE_SCREEN_BUFFER_INFO orgConsoleInfo;



/****************************************************************************
 * Internal functions' declarations *****************************************
 ****************************************************************************/

/* Kernel-mode module communication routines 
 */
BOOL QueryDriverVersion(HANDLE hDriver);
BOOL SendFinishSignal(HANDLE hDriver);
BOOL GetSecretMSR(HANDLE hDriver, struct tagMSR* MSR);
BOOL SetSecretMSR(HANDLE hDriver, struct tagMSR* MSR);
BOOL SetSingleMSR(HANDLE hDriver, struct tagSingleMSR* MSR);

/* Kernel-module loading routines
 */
BOOL LoadDriver();
VOID UnloadDriver();




/* LoadDriver routine.
 *
 * Arguments:
 *  None
 * 
 * Return value:
 *  TRUE, if the driver has been successfully loaded and executed
 *  FALSE otherwise
 */
BOOL LoadDriver()
{
  SC_HANDLE hSCManager;
  SC_HANDLE hService;
  SERVICE_STATUS ss;
  CHAR Path[4096];


  /* Before trying to manually load the driver, make sure it
   * is not currently loaded.
   */
  UnloadDriver();

  /* Form the initial version of driver path
   */
  GetCurrentDirectory(sizeof(Path),Path);
  strcat(Path, "\\");
  strcat(Path, DriverFileName);

  /* Open the SC manager
   */
  hSCManager = OpenSCManager(NULL, NULL, SC_MANAGER_CREATE_SERVICE);
  if(!hSCManager)
  {
    return false;
  }

  /* Check if the service isn't already active.
   * If it is, unload it from the system. We do NOT want to use an already running instance 
   * of the driver.
   */
  hService = OpenService(hSCManager, ServiceName, SERVICE_START | DELETE | SERVICE_STOP | SERVICE_QUERY_STATUS);
  if(hService)
  {
    ControlService(hService, SERVICE_CONTROL_STOP, &ss);
    DeleteService(hService);
    CloseServiceHandle(hService);
  }

  /* Try to create our service
   */
  hService = CreateService(hSCManager, ServiceName, ServiceName,
                          	SERVICE_START|DELETE|SERVICE_STOP|SERVICE_QUERY_STATUS, SERVICE_KERNEL_DRIVER,
                            SERVICE_DEMAND_START, SERVICE_ERROR_IGNORE, Path,
                           	NULL, NULL, NULL, NULL, NULL);
  if(!hService)
  {
    CloseServiceHandle(hSCManager);
    return FALSE;
  }

  /* If the service is not already running, try to make it run
   */
  QueryServiceStatus(hService, &ss);
  if(ss.dwCurrentState != SERVICE_RUNNING)
  {
    if(!StartService(hService, 0, NULL))
    {
      ControlService(hService, SERVICE_CONTROL_STOP, &ss);
      DeleteService(hService);
      CloseServiceHandle(hService);
      CloseServiceHandle(hSCManager);
      return FALSE;
    }
  }

  CloseServiceHandle(hService);
  CloseServiceHandle(hSCManager);
  return TRUE;
}


/* UnloadDriver routine
 *
 * Arguments:
 *  None
 * 
 * Return value:
 *  None
 */
VOID UnloadDriver()
{
  SC_HANDLE hSCManager;
  SC_HANDLE hService;
  SERVICE_STATUS ss;

  /* Open the service manager
   */
  hSCManager = OpenSCManager(NULL, NULL, SC_MANAGER_CREATE_SERVICE);

  /* Open the service
   */
  hService = OpenService(hSCManager, ServiceName, SERVICE_START | DELETE | SERVICE_STOP | SERVICE_QUERY_STATUS);

  /* Stop and delete the service
   */
  ControlService(hService, SERVICE_CONTROL_STOP, &ss);
  if(!DeleteService(hService))
  {
    // ???
    // TODO
    // ???
  }

  /* Close handles
   */  
  CloseServiceHandle(hService);
  CloseServiceHandle(hSCManager);
}



/* QueryDriverVersion routine
 *
 * Arguments:
 *  hDriver - handle to the active AMD Debug Service driver
 * 
 * Return value:
 *  TRUE, if the version has been successfully received
 *  FALSE otherwise
 */
BOOL QueryDriverVersion(HANDLE hDriver)
{
  STATIC CHAR VersionBuffer[4096];
  DWORD dwBytesReturned;
  
  /* Send an IOCTL signal to the driver and wait for a response
   */
  if(!DeviceIoControl(hDriver,
                      IOCTL_QUERY_VERSION,
                      NULL,
                      0,
                      VersionBuffer,
                      sizeof(VersionBuffer),
                      &dwBytesReturned,
                      NULL))
  {
    return FALSE;
  }

  /* Print the obtained string
   */
  INFO(("Driver Version: %s\n",VersionBuffer));
  return TRUE;
}


/* SendFinishSignal routine
 *
 * Arguments:
 *  hDriver - handle to the active AMD Debug Service driver
 * 
 * Return value:
 *  TRUE, if all the monitored processes have been succesfully dumped
 *  FALSE otherwise
 */
BOOL SendFinishSignal(HANDLE hDriver)
{
  DWORD dwBytesReturned;

  /* Send an IOCTL signal to the driver and wait for a response
   */
  if(!DeviceIoControl(hDriver,
                      IOCTL_FINISH,
                      NULL,
                      0,
                      NULL,
                      0,
                      &dwBytesReturned,
                      NULL))
  {
    return FALSE;
  }

  return TRUE;  
}


/* GetSecretMSR routine
 *
 * Arguments:
 *  HANDLE hDriver     - handle to the AMD_dbg.sys device driver
 *  struct tagMSR* MSR - pointer into an output structure, storing
 *                       the MSR values.
 * 
 * Return value:
 *  TRUE  on success.
 *  FALSE otherwise. Check GetLastError() for more information.
 */
BOOL GetSecretMSR(HANDLE hDriver, struct tagMSR* MSR)
{
  DWORD dwBytesReturned;

  /* Send an IOCTL signal to the driver and wait for a response
   */
  if(!DeviceIoControl(hDriver,
                      IOCTL_GET_SECRET_MSR,
                      NULL,
                      0,
                      MSR,
                      sizeof(struct tagMSR),
                      &dwBytesReturned,
                      NULL))
  {
    return FALSE;
  }

  return TRUE;  
}


/* SetSecretMSR routine
 *
 * Arguments:
 *  HANDLE hDriver     - handle to the AMD_dbg.sys device driver,
 *  struct tagMSR* MSR - pointer into an input structure, storing
 *                       the MSR values to be set.
 * 
 * Return value:
 *  TRUE  on success.
 *  FALSE otherwise. Check GetLastError() for more information.
 */
BOOL SetSecretMSR(HANDLE hDriver, struct tagMSR* MSR)
{
  DWORD dwBytesReturned;

  /* Send an IOCTL signal to the driver and wait for a response
   */
  if(!DeviceIoControl(hDriver,
                      IOCTL_SET_SECRET_MSR,
                      MSR,
                      sizeof(struct tagMSR),
                      NULL,
                      0,
                      &dwBytesReturned,
                      NULL))
  {
    return FALSE;
  }

  return TRUE;
}


/* SetSingleMSR routine
 *
 * Arguments:
 *  HANDLE hDriver     - handle to the AMD_dbg.sys device driver,
 *  ULONG MsrRegister  - desired MSR identifier,
 *  ULONG MsrValue     - desired MSR value.
 * 
 * Return value:
 *  TRUE  on success.
 *  FALSE otherwise. Check GetLastError() for more information.
 */
BOOL SetSingleMSR(HANDLE hDriver, ULONG MsrRegister, ULONG MsrValue)
{
  DWORD dwBytesReturned;
  struct tagSingleMSR MSR;
 
  MSR.MsrRegister = MsrRegister;
  MSR.MsrValue    = MsrValue;

  /* Send an IOCTL signal to the driver and wait for a response
   */
  if(!DeviceIoControl(hDriver,
                      IOCTL_SET_SINGLE_MSR,
                      &MSR,
                      sizeof(struct tagSingleMSR),
                      NULL,
                      0,
                      &dwBytesReturned,
                      NULL))
  {
    return FALSE;
  }

  return TRUE;
}



/* HandlerRoutine
 *
 * Arguments:
 *  Irrevelant.
 *
 *  Check "HandlerRoutine Callback Function" for more details
 *  (http://msdn.microsoft.com/en-us/library/ms683242%28v=VS.85%29.aspx).
 * 
 * Return value:
 *  Irrevelant.
 */
BOOL STDCALL HandlerRoutine(DWORD dwCtrlType)
{
  switch(dwCtrlType)
  {
    case CTRL_C_EVENT:
    case CTRL_BREAK_EVENT:
    case CTRL_CLOSE_EVENT:
    case CTRL_LOGOFF_EVENT:
    case CTRL_SHUTDOWN_EVENT:
    {
      PRINT_ATTR(("\r\n\r\nCTRL+C console event detected. Cleaning up.\r\n"),FOREGROUND_BLUE|FOREGROUND_INTENSITY);

      // TODO: add some more uninitialization code here      
      UnloadDriver();

      PRINT_ATTR(("The environment has been succesfully cleaned up. Bye bye.\r\n"),FOREGROUND_BLUE|FOREGROUND_INTENSITY);
      ExitProcess(0);
      break;
    }
  }
  return TRUE;
}



/* main routine
 *
 * Arguments:
 *  Irrevelant (information about the console arguments)
 * 
 * Return value:
 *  Irrevelant
 */
int main(int argc, char** argv)
{
  struct tagMSR MSR;
  DWORD dwProcessor = 1;

  puts("-=*( AMD Debug Service " AMDDBG_VERSION " by j00ru//vx )*=-");
  printf("Usage: %s [Processor ID]\n",argv[0]);

  /* Custom CPU chosen by the user?
   */
  if(argc >= 2)
  {
    dwProcessor = atoi(argv[1]);
  }

  /* Retrieve the original screen attributes before using any of the 
   * INFO / ERR / DIE macros.
   */
  GetConsoleScreenBufferInfo(GetStdHandle(STD_OUTPUT_HANDLE),&orgConsoleInfo);

  /* Set the current process affinity to a certain CPU, in order to avoid problems
   * with different MSRs on different cores.
   */
  INFO(("Setting process affinity mask...\r\n"));
  if(SetProcessAffinityMask(GetCurrentProcess(),(1<<dwProcessor)) == FALSE)
  {
    ERR(("Unable to set the process affinity to processor %d.\r\n"
         "Defaulting to processor0.\r\n",(INT)dwProcessor));
    SetProcessAffinityMask(GetCurrentProcess(),(1<<0));
  }

  /* Set-up a console handler, in case the user aims to quit before the
   * application automatically terminates its execution
   */
  INFO(("Installing an console event handler... "));
  if(SetConsoleCtrlHandler(HandlerRoutine,TRUE) == FALSE)
    DIE(("Failed (0x%.8x)",(UINT)GetLastError()));
  SUCCESS;

  /* Load the device driver into kernel-mode
   */
  INFO(("Loading driver... "));
  if(!LoadDriver())
    DIE(("Failed (0x%.8x)",(UINT)GetLastError()));
  SUCCESS;

  /* Open a handle to the device, created by our driver
   */
  INFO(("Opening driver... "));
  hDevice = CreateFileA(DeviceName,GENERIC_READ|GENERIC_WRITE,0,NULL,OPEN_EXISTING,0,NULL);
  if(hDevice==NULL)
  {
    UnloadDriver();
    DIE(("Failed (0x%.8x)",(UINT)GetLastError()));
  }
  SUCCESS;  

  /* Verify that the IOCTL communication works fine,
   * by querying for the driver version.
   */
  INFO(("Querying for the driver version...\r\n"));
  if(!QueryDriverVersion(hDevice))
  {
    UnloadDriver();
    DIE(("Failed (0x%.8x)",(UINT)GetLastError()));
  }

  /* Ask for the current MSR values.
   *
   * Note: when launching the application for the first times,
   *       all of the retrieved numbers should be zero.
   */
  INFO(("Querying the secret MSR register values... "));
  if(!GetSecretMSR(hDevice,&MSR))
  {
    if(GetLastError() == ERROR_ACCESS_DENIED)
    {
      ERR(("\r\nFailed with ERROR_ACCESS_DENIED.\r\n"
           "This means that either:\r\n"
           " 1) a very weird error has been encountered somewhere\r\n"
           "    along the way (less likely),\r\n"
           " 2) the secret AMD Extended Debugging features are not\r\n"
           "    accessible on your processor (very likely).\r\n"));
    }
    else
    {
      ERR(("Failed (0x%.8x)",(UINT)GetLastError()));
    }

    UnloadDriver();
    ExitProcess(0);
  }
  SUCCESS;

  PRINT_ATTR(("The MSRs successfully retrieved: AMD extended debugging features are supported.\r\n"),
             FOREGROUND_BLUE|FOREGROUND_INTENSITY);

  /* Infinite loop: print the current MSRs' contents,
   * and obtain the new ones to be set.
   *
   * In order to quit the program, use CTRL+C, or simply close the console window.
   */
  while(1)
  {
    CHAR chInputBuffer[1024];
    ULONG dwValue;

    /* Try to obtain the secret MSRs
     */
    if(!GetSecretMSR(hDevice,&MSR))
    {
      UnloadDriver();
      DIE(("GetSecretMSR failed (0x%.8x)\r\n",(UINT)GetLastError()));
    }
    else
    {
      INFO(("Debugger- featuring register values:\r\n"));
      INFO(("  MSR_CONTROL:      0x%.8x\r\n",(UINT)MSR.MsrControl));
      INFO(("  MSR_DATA_MASK:    0x%.8x\r\n",(UINT)MSR.MsrDataMask));
      INFO(("  MSR_DATA_MATCH:   0x%.8x\r\n",(UINT)MSR.MsrDataMatch));
      INFO(("  MSR_ADDRESS_MASK: 0x%.8x\r\n\r\n",(UINT)MSR.MsrAddressMask));
    }


    /* 
     * Ask for a new value for each Model Specific Register, or skip.
     */

    INPUT((" MSR_CONTROL [\"s\" for skip]:      "));
    scanf("%1023s",chInputBuffer);
    if(chInputBuffer[0] != 's')
    {
      sscanf(chInputBuffer,"%i",(PINT)&dwValue);
      if(!SetSingleMSR(hDevice,MSR_CONTROL_ID,dwValue))
        ERR(("SetSingleMSR(0x%.8x,MSR_CONTROL_ID,0x%.8x) failed (0x%.8x)\r\n",
              (UINT)hDevice,(UINT)dwValue,
              (UINT)GetLastError()));
    }

    INPUT((" MSR_DATA_MASK [\"s\" for skip]:    "));
    scanf("%1023s",chInputBuffer);
    if(chInputBuffer[0] != 's')
    {
      sscanf(chInputBuffer,"%i",(PINT)&dwValue);
      if(!SetSingleMSR(hDevice,MSR_DATA_MASK_ID,dwValue))
        ERR(("SetSingleMSR(0x%.8x,MSR_DATA_MASK_ID,0x%.8x) failed (0x%.8x)\r\n",
              (UINT)hDevice,(UINT)dwValue,
              (UINT)GetLastError()));
    }

    INPUT((" MSR_DATA_MATCH [\"s\" for skip]:   "));
    scanf("%1023s",chInputBuffer);
    if(chInputBuffer[0] != 's')
    {
      sscanf(chInputBuffer,"%i",(PINT)&dwValue);
      if(!SetSingleMSR(hDevice,MSR_DATA_MATCH_ID,dwValue))
        ERR(("SetSingleMSR(0x%.8x,MSR_DATA_MATCH_ID,0x%.8x) failed (0x%.8x)\r\n",
              (UINT)hDevice,(UINT)dwValue,
              (UINT)GetLastError()));
    }

    INPUT((" MSR_ADDRESS_MASK [\"s\" for skip]: "));
    scanf("%1023s",chInputBuffer);
    if(chInputBuffer[0] != 's')
    {
      sscanf(chInputBuffer,"%i",(PINT)&dwValue);
      if(!SetSingleMSR(hDevice,MSR_ADDRESS_MASK_ID,dwValue))
        ERR(("SetSingleMSR(0x%.8x,MSR_ADDRESS_MASK_ID,0x%.8x) failed (0x%.8x)\r\n",
              (UINT)hDevice,(UINT)dwValue,
              (UINT)GetLastError()));
    }

    INFO(("\r\n"));
  }
  
  /* Eventually, unload the driver and exit.
   *
   * XXX: Due to above infinite loop, we should never get here.
   *      But... who knows?
   */
  INFO(("Unloading the device driver now...\r\n"));
  UnloadDriver();

  return 0;
}

