// 
// AMD Expanded Debugging utility for Windows
//   The application is used to control some of the secret MSR registers,
//   internally used by the AMD-series processors for specifying hidden
//   debugging features.
//
//
// Author:  j00ru//vx
// Date:    2010-11-23
// E-mail:  j00ru.vx@gmail.com
// WWW:     http://j00ru.vexillium.org/
//
// Version: 0.0.1
// Module:  Kernel-mode device driver
//
// Note: Special thanks to Czernobyl (http://www.czerno.tk/) for taking the time
//       to actually discover and reverse engineer the yet unknown processor 
//       internals.
//
// References:
//   1) http://www.woodmann.com/forum/showthread.php?13891
//      The original forum post, where Czarnobyl shared initial information related
//      to the mystery mechanism.
//
//   2) http://www.czerno.tk/
//      Czernobyl's home-page, with a brief description of the current progress state.
//
//   3) http://hardware.slashdot.org/story/10/11/12/047243/Hidden-Debug-Mode-Found-In-AMD-Processors
//      A slashdot article regarding the issue, blocking the woodmann forums for about a day.
//
//   4) http://cbid.softnology.biz/html/undocmsrs.html
//      Undocumented Machine-Specific Registers: internal names of the MSR registers, which
//      are not publicly known or taken advantage of.
//
//
// Known bugs:
//   1) The code is only compatible with the x86 (32-bit) AMD architecture.
//   2) ?
//
//
// License:
//
//   Permission is hereby granted to use, copy, modify, and distribute this source code, 
//   or portions hereof, documentation and executables, for any purpose, without fee, 
//   subject to the following restrictions:
//   
//   1. The origin of this source code must not be misrepresented.
//  
//   2. Altered versions must be plainly marked as such and must not be misrepresented 
//      as being the original source.
//  
//   3. This Copyright notice may not be removed or altered from any source or altered 
//      source distribution.
//   
//   This software is provided "AS IS". The author does not guarantee that this program works 
//   or is bug-free. The author takes no responsibility for any damage caused by this program.
//   Use at your own risk.
//
#include <stdio.h>
#include <ntddk.h>
#include <windef.h>
#include "driver.h"




/****************************************************************************
 * Constants ****************************************************************
 ****************************************************************************/
const  CHAR DriverVersion[] =  "AMD Debug Service driver " AMDDBG_VERSION " by j00ru//vx";
const ULONG SECRET_KEY = 0x9C5A203A;




/****************************************************************************
 * Globals ******************************************************************
 ****************************************************************************/

PDEVICE_OBJECT pDeviceObject = NULL;  // Just a device object associated with the driver




/****************************************************************************
 * Internal functions' declarations *****************************************
 ****************************************************************************/
VOID DriverUnload(PDRIVER_OBJECT pDriverObject);
NTSTATUS DriverCmd(PDEVICE_OBJECT, PIRP);
NTSTATUS DriverUnsupported(PDEVICE_OBJECT, PIRP);




/*
 * READ_MSR macro.
 * Note: if the specified, internal register is not accessible,
 *       a GP# exception is generated. Always call the function from
 *       within a try-except block.
 *
 * Arguments:
 *
 *  ULONG MsrReg - Number of the MSR register to query.
 *
 * Return value:
 *  
 *  The value of the desired MSR.
 *  
 */
ULONG READ_MSR(ULONG MsrReg)
{
  __asm
  {
    mov edi, SECRET_KEY
    mov ecx, MsrReg
    rdmsr
    nop
    nop
    nop
  }
}


/*
 * WRITE_MSR macro.
 * Note: if the specified, internal register is not accessible,
 *       a GP# exception is generated. Always call the function from
 *       within a try-except block.
 *
 * Arguments:
 *
 *  ULONG MsrReg   - Number of the MSR register to write to.
 *  ULONG MsrValue - The actual value to be assigned to an MSR.
 *
 * Return value:
 *  
 *  None.
 *  
 */
VOID WRITE_MSR(ULONG MsrReg, ULONG MsrValue)
{
  __asm
  {
    mov edi, SECRET_KEY
    mov ecx, MsrReg

    xor edx, edx
    mov eax, MsrValue
    wrmsr
    nop
    nop
    nop
  }
}




/* DriverEntry routine
 *
 * Arguments:
 *  Not important (refer to MSDN for details)
 * 
 * Return value:
 *  Not important (refer to MSDN for details)
 */
NTSTATUS
DriverEntry(PDRIVER_OBJECT DriverObject, PUNICODE_STRING RPath)
{  
  UNICODE_STRING u_DriverName, u_DosDeviceName;
  NTSTATUS NtStatus;
  UINT i;

  DbgPrint("[+] AMD_dbg.SYS: DriverEntry() called.\r\n");

  DbgPrint("[+] AMD_dbg.SYS: Registering an Unload routine.\r\n");
	DriverObject->DriverUnload = DriverUnload;

  DbgPrint("[+] AMD_dbg.SYS: Creating a named device for the driver.\r\n");
 	RtlInitUnicodeString(&u_DriverName, DriverName);
 	RtlInitUnicodeString(&u_DosDeviceName, DosDeviceName);
    
  /* Create a named device in order to be able to communicate ring3 - ring0
   */
  NtStatus = IoCreateDevice(
    DriverObject,
    0,
    &u_DriverName,
    FILE_DEVICE_UNKNOWN,
    FILE_DEVICE_SECURE_OPEN,
    FALSE,
    &pDeviceObject
    );

  if(!NT_SUCCESS(NtStatus))
    return NtStatus;

  /* Create a symbolic link between the long and shortened driver name
   */
  IoCreateSymbolicLink(
      &u_DosDeviceName,
      &u_DriverName
      );

  /* Register the appropriate handler functions for the desired IRP events
   * (only interested in IOCTL in this case).
   */
  for(i=0;i<IRP_MJ_MAXIMUM_FUNCTION;i++)
  {
    DriverObject->MajorFunction[i] = DriverUnsupported;
  }
  DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = DriverCmd;

  return STATUS_SUCCESS;
}


/* DriverUnload routine
 *
 * Arguments:
 *  Not important (refer to MSDN for details)
 * 
 * Return value:
 *  None
 *
 *  The function is responsible for removing the old process notify callback
 *  (not doing it would likely lead to a system crash due to an invalid pointer
 *  derefecence).
 */
VOID
DriverUnload(PDRIVER_OBJECT pDriverObject)
{
	UNICODE_STRING u_DosDeviceName;
  DbgPrint("[+] AMD_dbg.SYS: DriverUnload() called.\r\n");

  /* Unmap the symbolic link and delete the named device 
   */
	RtlInitUnicodeString(&u_DosDeviceName, DosDeviceName);
	IoDeleteSymbolicLink(&u_DosDeviceName);
	IoDeleteDevice(pDeviceObject);

  DbgPrint("[+] AMD_dbg.SYS: DriverUnload() succeeded, returning.\r\n");
}



/* DriverUnsupported routine
 *
 * Arguments:
 *  Not important (refer to MSDN for details)
 * 
 * Return value:
 *  Always returns STATUS_NOT_SUPPORTED.
 */
NTSTATUS 
DriverUnsupported(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	return STATUS_NOT_SUPPORTED;
}


/* DriverCmd routine
 *
 * Arguments:
 *  Not important (refer to MSDN for details)
 * 
 * Return value:
 *  The value returned by separate signal-handling functions, indicating
 *  the execution state.
 *
 *  This routine is called everytime an IOCTL message is sent by a ring3 application.
 *  It's purpose is to analyze the signal and perform proper actions (call the right
 *  handler functions here) depending on the signal ID being passed.
 */
NTSTATUS 
DriverCmd(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
  NTSTATUS NtStatus = STATUS_SUCCESS;
	PIO_STACK_LOCATION irpStack;
	PVOID InBuffer, OutBuffer;
	ULONG InBufferSize, OutBufferSize;
	ULONG IOCtlCode;
	IO_STATUS_BLOCK IoStatusBlock;

	irpStack = IoGetCurrentIrpStackLocation(Irp);
	
  /* Get the basic data pointers provided by a user-mode application */
	InBuffer      = irpStack->Parameters.DeviceIoControl.Type3InputBuffer;
	InBufferSize  = irpStack->Parameters.DeviceIoControl.InputBufferLength;
	OutBuffer     = Irp->UserBuffer;
	OutBufferSize = irpStack->Parameters.DeviceIoControl.OutputBufferLength;
	IOCtlCode     = irpStack->Parameters.DeviceIoControl.IoControlCode;

	/* At the begining, assume everything will be ok */
	NtStatus = STATUS_SUCCESS;

	switch(IOCtlCode)
	{
		case IOCTL_QUERY_VERSION:
      NtStatus = HandleQueryVersion(InBuffer,OutBuffer,InBufferSize,OutBufferSize);
      break;
    case IOCTL_FINISH:
      NtStatus = HandleFinish(InBuffer,OutBuffer,InBufferSize,OutBufferSize);
      break;
    case IOCTL_GET_SECRET_MSR:
      NtStatus = HandleGetMSR(InBuffer,OutBuffer,InBufferSize,OutBufferSize);
      break;
    case IOCTL_SET_SECRET_MSR:
      NtStatus = HandleSetMSR(InBuffer,OutBuffer,InBufferSize,OutBufferSize);
      break;
    case IOCTL_SET_SINGLE_MSR:
      NtStatus = HandleSetSingleMSR(InBuffer,OutBuffer,InBufferSize,OutBufferSize);
      break;
    default:
      NtStatus = STATUS_NOT_SUPPORTED;
      break;
	}
	
	Irp->IoStatus.Status = NtStatus;
	IoCompleteRequest(Irp, IO_NO_INCREMENT);

	return NtStatus;
}


/* HandleQueryVersion routine
 *
 * Arguments:
 *  The arguments contain standard IOCTL information (input and output buffer
 *  together with their sizes).
 * 
 * Return value:
 *  STATUS_SUCCESS if the function succeeds
 *  STATUS_ACCESS_VIOLATION or any other error code in case of failure (when reading the
 *  user-specified buffer etc)
 */
NTSTATUS
HandleQueryVersion(
	PVOID InBuffer,     // UNUSED
	PVOID OutBuffer,    
	ULONG InBufferSize, // UNUSED
	ULONG OutBufferSize
)
{
  DbgPrint("[+] AMD_dbg.SYS: HandleQueryVersion() called.\r\n");

  __try
  {
    if(OutBufferSize<sizeof(DriverVersion))
    {
      ExRaiseStatus(STATUS_INVALID_BUFFER_SIZE);
    }

    /* Check if there's enought space in the OutBuffer user-mode memory... */
    ProbeForWrite(OutBuffer,sizeof(DriverVersion),sizeof(CHAR));

    /* Copy the version string to the destination buffer */
    RtlCopyMemory(OutBuffer,DriverVersion,sizeof(DriverVersion));
  }
  except(EXCEPTION_EXECUTE_HANDLER)
  {
    return GetExceptionCode();
  }

  return STATUS_SUCCESS;
}


/* HandleFinish routine
 *
 * Arguments:
 *  None
 * 
 * Return value:
 *  STATUS_SUCCESS on success.
 */
NTSTATUS
HandleFinish(
	PVOID InBuffer,       // UNUSED
	PVOID OutBuffer,      // UNUSED
	ULONG InBufferSize,   // UNUSED
	ULONG OutBufferSize   // UNUSED
)
{
  DbgPrint("[+] AMD_dbg.SYS: HandleFinish() called.\r\n");

  /* TODO: add something here?
   */

  DbgPrint("[+] AMD_dbg.SYS: Finish signal successfully handled.\r\n");
  return STATUS_SUCCESS;
}


/* HandleGetMSR routine
 *
 * Arguments:
 *  OutBuffer - a pointer to the tagMSR structure, placed within user-mode memory
 *  OutBufferSize - length of the output buffer, should be sizeof(struct tagMSR)
 * 
 * Return value:
 *  STATUS_SUCCESS             on success
 *  STATUS_INVALID_BUFFER_SIZE on invalid output buffer size
 *  STATUS_ACCESS_VIOLATION    on invalid output buffer pointer
 *  STATUS_ACCESS_DENIED       on invalid MSR read access (i.e. unsupported mechanism)
 *  
 */
NTSTATUS
HandleGetMSR(
	PVOID InBuffer,       // UNUSED 
	PVOID OutBuffer,
	ULONG InBufferSize,   // UNUSED
	ULONG OutBufferSize
)
{
  struct tagMSR MSR;

  DbgPrint("[+] AMD_dbg.SYS: HandleGetMSR() called.\r\n");
  
  __try
  {
    /* The specified input size must be equal sizeof(HANDLE)
     */
    if(OutBufferSize != sizeof(struct tagMSR))
    {
      ExRaiseStatus(STATUS_INVALID_BUFFER_SIZE);
    }

    /* Check the virtual address validity
     */
    ProbeForRead(OutBuffer,OutBufferSize,sizeof(BYTE));
  }
  __except(EXCEPTION_EXECUTE_HANDLER)
  {
    return GetExceptionCode();
  }

  /* Read the secret MSR registers
   */
  __try
  {
    MSR.MsrControl     = READ_MSR(MSR_CONTROL_ID);
    MSR.MsrDataMask    = READ_MSR(MSR_DATA_MASK_ID);
    MSR.MsrDataMatch   = READ_MSR(MSR_DATA_MATCH_ID);
    MSR.MsrAddressMask = READ_MSR(MSR_ADDRESS_MASK_ID);

    memcpy(OutBuffer,&MSR,sizeof(struct tagMSR));
  }
  __except(EXCEPTION_EXECUTE_HANDLER)
  {
    return (STATUS_ACCESS_DENIED);
  }
  
  return STATUS_SUCCESS;
}


/* HandleSetMSR routine
 *
 * Arguments:
 *  InBuffer - a pointer to the tagMSR structure, placed within user-mode memory
 *  InBufferSize - length of the input buffer, should be sizeof(struct tagMSR)
 * 
 * Return value:
 *  STATUS_SUCCESS             on success
 *  STATUS_INVALID_BUFFER_SIZE on invalid input buffer size
 *  STATUS_ACCESS_VIOLATION    on invalid input buffer pointer
 *  STATUS_ACCESS_DENIED       on invalid MSR write access (i.e. unsupported mechanism)
 *  
 */
NTSTATUS
HandleSetMSR(
	PVOID InBuffer,
	PVOID OutBuffer,      // UNUSED 
	ULONG InBufferSize,
	ULONG OutBufferSize   // UNUSED
)
{
  struct tagMSR MSR;

  DbgPrint("[+] AMD_dbg.SYS: HandleSetMSR() called.\r\n");
  
  __try
  {
    /* The specified input size must be equal sizeof(HANDLE)
     */
    if(InBufferSize != sizeof(struct tagMSR))
    {
      ExRaiseStatus(STATUS_INVALID_BUFFER_SIZE);
    }

    /* Check the virtual address validity
     */
    ProbeForRead(InBuffer,InBufferSize,sizeof(BYTE));
    memcpy(&MSR,InBuffer,sizeof(struct tagMSR));
  }
  __except(EXCEPTION_EXECUTE_HANDLER)
  {
    return GetExceptionCode();
  }

  /* Write the secret MSRs
   */
  __try
  {
    WRITE_MSR(MSR_CONTROL_ID      , MSR.MsrControl);
    WRITE_MSR(MSR_DATA_MASK_ID    , MSR.MsrDataMask);
    WRITE_MSR(MSR_DATA_MATCH_ID   , MSR.MsrDataMatch);
    WRITE_MSR(MSR_ADDRESS_MASK_ID , MSR.MsrAddressMask);
  }
  __except(EXCEPTION_EXECUTE_HANDLER)
  {
    return (STATUS_ACCESS_DENIED);
  }
  
  return STATUS_SUCCESS;
}



/* HandleSetSingleMSR routine
 *
 * Arguments:
 *  InBuffer - a pointer to the tagSingleMSR structure, placed within user-mode memory
 *  InBufferSize - length of the input buffer, should be sizeof(struct tagSingleMSR)
 * 
 * Return value:
 *  STATUS_SUCCESS             on success
 *  STATUS_INVALID_BUFFER_SIZE on invalid input buffer size
 *  STATUS_ACCESS_VIOLATION    on invalid input buffer pointer
 *  STATUS_NOT_SUPPORTED       on invalid MSR identifier
 *  STATUS_ACCESS_DENIED       on invalid MSR write access (i.e. unsupported mechanism)
 *  
 */
NTSTATUS
HandleSetSingleMSR(
	PVOID InBuffer,
	PVOID OutBuffer,      // UNUSED 
	ULONG InBufferSize,
	ULONG OutBufferSize   // UNUSED
)
{
  CONST ULONG SupportedMsrNo[] =
  {
    MSR_CONTROL_ID,
    MSR_DATA_MATCH_ID,
    MSR_DATA_MASK_ID,
    MSR_ADDRESS_MASK_ID,
    MSR_TERMINATOR_ID
  };

  struct tagSingleMSR MSR;
  UINT i;

  DbgPrint("[+] AMD_dbg.SYS: HandleSetSingleMSR() called.\r\n");
  
  __try
  {
    /* The specified input size must be equal sizeof(HANDLE)
     */
    if(InBufferSize != sizeof(struct tagSingleMSR))
    {
      ExRaiseStatus(STATUS_INVALID_BUFFER_SIZE);
    }

    /* Check the virtual address validity
     */
    ProbeForRead(InBuffer,InBufferSize,sizeof(BYTE));
    memcpy(&MSR,InBuffer,sizeof(struct tagSingleMSR));

    /* Verify, that the requested MSR no. is a supported one
     * (i.e. the client application doesn't want to kill the system).
     */
    for( i=0;SupportedMsrNo[i] != MSR_TERMINATOR_ID;i++ )
    {
      if(MSR.MsrRegister == SupportedMsrNo[i])
        break;
    }

    /* Bail out, in case of suspicious input
     */
    if(SupportedMsrNo[i] != MSR.MsrRegister)
    {
      ExRaiseStatus(STATUS_NOT_SUPPORTED);
    }
  }
  __except(EXCEPTION_EXECUTE_HANDLER)
  {
    return GetExceptionCode();
  }

  /* Try to write the specific MSR - bail out, if not possible
   */
  __try
  {
    WRITE_MSR(MSR.MsrRegister,MSR.MsrValue);
  }
  __except(EXCEPTION_EXECUTE_HANDLER)
  {
    return (STATUS_ACCESS_DENIED);
  }
  
  return STATUS_SUCCESS;
}


