﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Reflection;
using System.IO;
using System.Collections;

namespace AdmiralDebilitate
{
    public partial class MainForm : Form
    {
        const String BACKUP_FOLDER = "\\AdmiralDebilitate Backup";
        bool ignore_mscorlib = true;
        bool ignore_system = true;

        public static String GetFilePath(String file)
        {
            return file.Substring(0, file.LastIndexOf('\\'));
        }

        Hashtable assemblies = new Hashtable();
        List<TreeNode> nodes = new List<TreeNode>();

        int total_nodes = 0;
        void ImportAssembly(String filename, TreeNode node, float start_percent, float end_percent)
        {
            ++total_nodes;
            bool recurse = false;
            try
            {
                if (!filename.ToLower().Contains(".dll") &&
                    !filename.ToLower().Contains(".exe"))
                {
                    if (File.Exists(filename + ".dll"))
                    {
                        filename += ".dll";
                        node.ImageIndex = ImageList.Images.IndexOfKey("dll");
                    }
                    else
                    {
                        if (File.Exists(filename + ".exe"))
                        {
                            filename += ".exe";
                            node.ImageIndex = ImageList.Images.IndexOfKey("app");
                        }
                        else
                        {
                            throw new FileNotFoundException();
                        }
                    }
                }
                node.SelectedImageIndex = node.ImageIndex;

                String shortname = Path.GetFileName(filename);
                if (shortname.EndsWith(".dll", StringComparison.OrdinalIgnoreCase) || shortname.EndsWith(".exe", StringComparison.OrdinalIgnoreCase))
                {
                    shortname = shortname.Substring(0, shortname.Length - 4);
                }
                NETAssembly assembly = (NETAssembly)assemblies[shortname];
                if (assembly == null)
                {
                    recurse = true;
                    assembly = new NETAssembly();
                    assembly.Load(filename);
                    assemblies[shortname] = assembly;
                }
                node.ForeColor = Color.Black;
                node.Tag = assembly;

                Environment.CurrentDirectory = GetFilePath(assembly.Location);
                List<AssemblyRef> references = assembly.GetReferencedAssemblies();

                for (int i = 0; i < references.Count(); ++i)
                {
                    if (recurse)
                    {
                        bool import = true;
                        if (ignore_mscorlib)
                        {
                            if (references[i].Name == "mscorlib")
                            {
                                import = false;
                            }
                        }
                        if (ignore_system)
                        {
                            if (references[i].Name.StartsWith("System.") || references[i].Name == "System")
                            {
                                import = false;
                            }
                        }
                        if (import)
                        {
                            TreeNode child = new TreeNode(references[i].Name);
                            node.Nodes.Add(child);
                            float sub_start_percent = start_percent + ((float)i / references.Count()) * (end_percent - start_percent);
                            float sub_end_percent = start_percent + ((float)(i + 1) / references.Count()) * (end_percent - start_percent);
                            ImportAssembly(references[i].Name, child, sub_start_percent, sub_end_percent);
                        }
                    }
                }

                LoadModulesWorker.ReportProgress((int)end_percent);
            }
            catch (FileLoadException)
            {
                node.Text = Path.GetFileName(filename) + " [Not accessible]";
                node.ForeColor = Color.Red;
            }
            catch (FileNotFoundException)
            {
                node.Text = Path.GetFileName(filename) + " [Not found]";
                node.ForeColor = Color.Red;
            }
            catch (DirectoryNotFoundException)
            {
                node.Text = Path.GetFileName(filename) + " [Not found]";
                node.ForeColor = Color.Red;
            }
            catch (BadImageFormatException)
            {
                node.Text = Path.GetFileName(filename) + " [Bad image format]";
                node.ForeColor = Color.Red;
            }
            catch (NotNETAssemblyException)
            {
                node.Text = Path.GetFileName(filename) + " [Not a NET assembly]";
                node.ForeColor = Color.Green;
            }
            catch (IOException)
            {
                node.Text = Path.GetFileName(filename) + " [File is in use]";
                node.ForeColor = Color.Red;
            }
            catch (Exception)
            {
                node.Text = Path.GetFileName(filename) + " [Unknown error]";
                node.ForeColor = Color.Red;
            }
        }

        void ShowOpenFileDialog()
        {
            DialogResult result = OpenFileDialog.ShowDialog();
            if (result != DialogResult.OK) return;

            CleanUp();
            DisableGUI();
            LoadModulesWorker.RunWorkerAsync(OpenFileDialog.FileNames);
        }

        public MainForm()
        {
            InitializeComponent();
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ShowOpenFileDialog();
        }

        bool gui_enabled = true;
        private void DisableGUI()
        {
            gui_enabled = false;
            fileToolStripMenuItem.Enabled = false;
            helpToolStripMenuItem.Enabled = false;
            cmdApply.Enabled = false;
            cmdClear.Enabled = false;
            chkCreateBackups.Enabled = false;
        }

        private void EnableGUI()
        {
            gui_enabled = true;
            fileToolStripMenuItem.Enabled = true;
            helpToolStripMenuItem.Enabled = true;
            cmdApply.Enabled = true;
            cmdClear.Enabled = true;
            chkCreateBackups.Enabled = true;
        }

        string[] seed_files;
        private void LoadModulesWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            MarkedAssemblies = new List<NETAssembly>();
            PrimaryMarkedAssemblies = new List<NETAssembly>();
            SBCancel.Visible = true;
            nodes.Clear();
            total_nodes = 0;
            string[] filenames = (string[]) e.Argument;
            seed_files = filenames;
            for (int i = 0; i < filenames.Count(); ++i)
            {
                if (LoadModulesWorker.CancellationPending)
                {
                    return;
                }

                TreeNode node = new TreeNode(Path.GetFileNameWithoutExtension(filenames[i])); 
                node.Expand();
                nodes.Add(node);
                float start_percent = (float)i / filenames.Count();
                float end_percent = 100.0f * (float)(i + 1) / filenames.Count();
                ImportAssembly(filenames[i], node, start_percent, end_percent);
            }
        }

        private void LoadModulesWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            if (e.Cancelled)
            {
                SBCancel.Visible = false;
                SBProgress.Value = 0;
                SBProgress.Visible = false;
                SBLabel.Text = "Loading was cancelled.";
            }
            else
            {
                SBProgress.Value = 0;
                SBProgress.Visible = false;
                SBCancel.Visible = false;
                SBLabel.Text = assemblies.Count + " modules total.";
                Tree.Nodes.Clear();
                for (int i = 0; i < nodes.Count; ++i)
                {
                    Tree.Nodes.Add((TreeNode)nodes[i]);
                }

                if (Tree.Nodes.Count > 0) Tree.SelectedNode = Tree.Nodes[0];

                foreach (TreeNode node in Tree.Nodes)
                {
                    UpdateMarkedTreeGUI(node);
                }
            }

            VerifyTree(null);
            lstProblems.Items.Clear();
            foreach (String problem in tree_problems)
            {
                lstProblems.Items.Add(problem);
            }

            EnableGUI();
        }

        DateTime last_progress_update = DateTime.Now;
        private void LoadModulesWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            if (!this.Visible) return;
            SBProgress.Visible = true;
            if ((DateTime.Now - last_progress_update).Milliseconds > 200)
            {
                last_progress_update = DateTime.Now;
                SBProgress.Value = e.ProgressPercentage;
                SBLabel.Text = "Loading assemblies - " + e.ProgressPercentage + "% (" + assemblies.Count + " modules)";
            }
        }

        private void SBCancel_Click(object sender, EventArgs e)
        {
            if (LoadModulesWorker.IsBusy)
            {
                LoadModulesWorker.CancelAsync();
            }
            SBCancel.Visible = false;
        }

        private void DisplaySummary(NETAssembly assembly)
        {
            AssemblySummary.Items.Clear();

            if (assembly == null) return;
            if (!assembly.IsLoaded()) return;
            
            AssemblySummary.Items.Add(assembly.Name);
            AssemblySummary.Items.Add("Framework version " + assembly.metadata.Version);
            AssemblySummary.Items.Add("");
            if ((assembly.clr_header.Flags & 8) != 0)
            {
                AssemblySummary.Items.Add("Assembly claims to be signed.");
            }
            else
            {
                AssemblySummary.Items.Add("Assembly claims not to be signed.");
            }
            if ((assembly.AssemblyFlags & 1) != 0)
            {
                AssemblySummary.Items.Add("Public key is present.");
                String pks = "Public key:";
                for (int i = 0; i < assembly.PublicKey.Length; ++i) pks += " " + assembly.PublicKey[i].ToString("X2");
                AssemblySummary.Items.Add(pks);
            }
            else
            {
                AssemblySummary.Items.Add("No public key is present.");
                String pks = "(Residual key:";
                for (int i = 0; i < assembly.PublicKey.Length; ++i) pks += " " + assembly.PublicKey[i].ToString("X2");
                pks += ")";
                AssemblySummary.Items.Add(pks);
            }
        }

        private void Tree_AfterSelect(object sender, TreeViewEventArgs e)
        {
            DisplaySummary((NETAssembly) e.Node.Tag);
        }

        // Ancestor-based fast recursive marking
        // Doesn't work with multiple roots
        /*
        private void MarkAllReferees(NETAssembly assembly)
        {
            Stack<TreeNode> ancestors = new Stack<TreeNode>();
            foreach (TreeNode node in Tree.Nodes)
            {
                ancestors.Push(node);
            }
            MarkAllReferees(assembly, ancestors);
        }

        private void MarkAllReferees(NETAssembly assembly, Stack<TreeNode> ancestors)
        {
            if (assembly == null) return;
            TreeNode top_node = ancestors.Peek();
            NETAssembly top = (NETAssembly) top_node.Tag;
            if (top != null)
            {
                if (top.Name == assembly.Name)
                {
                    foreach (TreeNode current_node in ancestors)
                    {
                        NETAssembly current = (NETAssembly) current_node.Tag;
                        if (current != null)
                        {
                            if (!MarkedAssemblies.Contains(current))
                            {
                                MarkedAssemblies.Add(current);
                                MarkAllReferees(current);
                            }
                        }
                    }
                }
                int i = 0;
                foreach (TreeNode child in top_node.Nodes)
                {
                    ancestors.Push(child);
                    MarkAllReferees(assembly, ancestors);
                    ancestors.Pop();
                    ++i;
                }
            }
        }

        private void UpdateMarkStatus(TreeNode node)
        {
            // Walk tree for dependencies
            MarkedAssemblies.Clear();
            for (int i = 0; i < PrimaryMarkedAssemblies.Count; ++i)
            {
                MarkedAssemblies.Add(PrimaryMarkedAssemblies[i]);
                MarkAllReferees(PrimaryMarkedAssemblies[i]);
            }
        }

        private void UpdateMarkBackgroundWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            foreach (TreeNode node in Tree.Nodes)
            {
                UpdateMarkStatus(node);
            }
        }
        */

        // Brute-force iterative marking
        // Entirely general but runs in O(nd) time
        // (Number of nodes * height of tree)
        bool ContainsByName(List<AssemblyRef> list, NETAssembly assembly)
        {
            foreach (AssemblyRef existing in list)
            {
                if (assembly.Name.Contains(existing.Name))
                {
                    return true;
                }
            }
            return false;
        }

        bool ContainsByLocation(List<NETAssembly> list, NETAssembly assembly)
        {
            foreach (NETAssembly existing in list)
            {
                if (existing.Location == assembly.Location)
                {
                    return true;
                }
            }
            return false;
        }

        bool IterateMarks(TreeNode node)
        {
            bool changes_made = false;
            NETAssembly assembly = (NETAssembly)node.Tag;
            if (assembly != null)
            {
                if (ContainsByLocation(MarkedAssemblies, assembly))
                {
                    TreeNode parent_node = node.Parent;
                    if (parent_node != null)
                    {
                        NETAssembly parent = (NETAssembly)parent_node.Tag;
                        if (parent != null)
                        {
                            if (!ContainsByLocation(MarkedAssemblies, parent))
                            {
                                MarkedAssemblies.Add(parent);
                                changes_made = true;
                            }
                        }
                    }
                }
            }

            // Recurse
            foreach (TreeNode child_node in node.Nodes)
            {
                changes_made |= IterateMarks(child_node);
            }
            return changes_made;
        }

        private void UpdateMarkBackgroundWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            MarkedAssemblies.Clear();
            foreach (NETAssembly assembly in PrimaryMarkedAssemblies)
            {
                if (!MarkedAssemblies.Contains(assembly)) MarkedAssemblies.Add(assembly);
            }

            bool changes_made = true;
            while (changes_made)
            {
                changes_made = false;
                foreach (TreeNode node in Tree.Nodes)
                {
                    changes_made = IterateMarks(node);
                }
            }
        }

        private void UpdateMarkedTreeGUI(TreeNode node)
        {
            if (Tree.Nodes.Count == 0) return;

            if (node.Tag == null) return;
            if (MarkedAssemblies.Contains((NETAssembly)node.Tag))
            {
                node.ForeColor = Color.Blue;
                node.ImageKey = "unlock";
                node.SelectedImageKey = node.ImageKey;
            }
            else
            {
                node.ForeColor = Color.Black;
                NETAssembly assembly = ((NETAssembly)node.Tag);
                if (assembly.IsSigned())
                {
                    node.ImageKey = "lock"; node.SelectedImageIndex = node.ImageIndex;
                    node.SelectedImageKey = node.ImageKey;
                }
                else
                {
                    if (assembly.Location.Contains(".dll"))
                    {
                        node.ImageKey = "dll"; node.SelectedImageIndex = node.ImageIndex;
                        node.SelectedImageKey = node.ImageKey;
                    }
                    else
                    {
                        node.ImageKey = "app";
                        node.SelectedImageKey = node.ImageKey;
                    }
                }
            }

            for (int i = 0; i < node.Nodes.Count; ++i)
            {
                UpdateMarkedTreeGUI(node.Nodes[i]);
            }
        }

        private void UpdateMarkedLists()
        {
            lstMarkedAssemblies.Items.Clear();
            lstIndirectlyMarkedAssemblies.Items.Clear();

            foreach (NETAssembly assembly in PrimaryMarkedAssemblies)
            {
                lstMarkedAssemblies.Items.Add(assembly.Name);
            }

            foreach (NETAssembly assembly in MarkedAssemblies)
            {
                if (!lstMarkedAssemblies.Items.Contains(assembly.Name)) lstIndirectlyMarkedAssemblies.Items.Add(assembly.Name);
            }
        }

        private void UpdateMarks()
        {
            DisableGUI();
            SBLabel.Text = "Cross-referencing dependencies...";
            UpdateMarkBackgroundWorker.RunWorkerAsync();
        }

        private void MarkOrUnmarkToolStripMenuItem_Click(object sender, EventArgs e)
        {
            NETAssembly assembly = null;
            if (Tree.Focused)
            {
                assembly = (NETAssembly) Tree.SelectedNode.Tag;
            }
            else return;

            if (assembly == null)
            {
                return;
            }
            if (!assembly.IsLoaded())
            {
                return;
            }
            if (PrimaryMarkedAssemblies.Contains(assembly))
            {
                PrimaryMarkedAssemblies.Remove(assembly);
            }
            else
            {
                PrimaryMarkedAssemblies.Add(assembly);
            }

            UpdateMarks();
        }

        List<NETAssembly> MarkedAssemblies;
        List<NETAssembly> PrimaryMarkedAssemblies;

        private void Tree_MouseDown(object sender, MouseEventArgs e)
        {
            if ((e.Button & MouseButtons.Right) != 0)
            {
                Tree.SelectedNode = Tree.GetNodeAt(e.X, e.Y);
            }
        }

        private void TreeMenu_Opening(object sender, CancelEventArgs e)
        {
            if (!gui_enabled)
            {
                e.Cancel = true;
                return;
            }

            NETAssembly assembly = null;
            if (Tree.SelectedNode != null)
            {
                assembly = (NETAssembly)Tree.SelectedNode.Tag;
            }
            if (assembly == null)
            {
                TreeMenu.Items[0].Text = "Mark/Unmark";
                TreeMenu.Items[0].Enabled = false;
            }
            else
            {
                if (PrimaryMarkedAssemblies.Contains(assembly))
                {
                    TreeMenu.Items[0].Text = "Unmark";
                }
                else
                {
                    TreeMenu.Items[0].Text = "Mark";
                }
                TreeMenu.Items[0].Enabled = true;
            }
        }

        private void UpdateMarkBackgroundWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            foreach (TreeNode node in Tree.Nodes)
            {
                UpdateMarkedTreeGUI(node);
            }
            UpdateMarkedLists();
            SBLabel.Text = "Dependency tree updated.";
            EnableGUI();
        }

        private bool LocateTreeItemByName(String name, TreeNode node)
        {
            NETAssembly assembly = (NETAssembly)node.Tag;
            if (assembly == null) return false;
            if (assembly.Name == name)
            {
                Tree.SelectedNode = node;
                Tree.Focus();
                return true;
            }
            else
            {
                foreach (TreeNode child in node.Nodes)
                {
                    if (LocateTreeItemByName(name, child)) return true;
                }
            }
            return false;
        }

        private void LocateTreeItemByName(String name)
        {
            foreach (TreeNode node in Tree.Nodes)
            {
                if (LocateTreeItemByName(name, node)) break;
            }
        }

        

        private void lstMarkedAssemblies_SelectedIndexChanged(object sender, EventArgs e)
        {
            String name = (String) lstMarkedAssemblies.SelectedItem;
            lstMarkedAssemblies.SelectedItem = null;
            if (name != null)
            {
                LocateTreeItemByName(name);
            }
        }

        private void lstIndirectlyMarkedAssemblies_SelectedIndexChanged(object sender, EventArgs e)
        {
            String name = (String) lstIndirectlyMarkedAssemblies.SelectedItem;
            lstIndirectlyMarkedAssemblies.SelectedItem = null;
            if (name != null)
            {
                LocateTreeItemByName(name);
            }
        }

        private void cmdClear_Click(object sender, EventArgs e)
        {
            PrimaryMarkedAssemblies.Clear();
            UpdateMarks();
        }

        private void cmdApply_Click(object sender, EventArgs e)
        {
            DisableGUI();
            SBLabel.Text = "Patching files...";
            SBProgress.Value = 0;
            SBProgress.Visible = true;
            ApplyPatchesWorker.RunWorkerAsync();
        }

        private void CreateBackup(NETAssembly assembly)
        {
            String path = Path.GetDirectoryName(assembly.Location) + BACKUP_FOLDER;
            if (!Directory.Exists(path)) Directory.CreateDirectory(path);
            try
            {
                File.Copy(assembly.Location, path + '\\' + Path.GetFileName(assembly.Location), false);
            }
            catch (IOException)
            {
                // Already exists
            }
        }

        private void PatchAssembly(NETAssembly assembly)
        {
            try
            {
                if (chkCreateBackups.Checked) CreateBackup(assembly);
                assembly.RemoveSigning();
                assembly.RemoveSignedReferences(assemblies, MarkedAssemblies);
            }
            catch (IOException)
            {
                MessageBox.Show("Failed to open '" + assembly.Name + "' for writing.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            catch (Exception)
            {
                MessageBox.Show("An unknown error occurred when patching '" + assembly.Name + "'.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ApplyPatchesWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            try
            {
                for (int i = 0; i < MarkedAssemblies.Count; ++i)
                {
                    ApplyPatchesWorker.ReportProgress(100 * i / MarkedAssemblies.Count);
                    PatchAssembly(MarkedAssemblies[i]);
                    ApplyPatchesWorker.ReportProgress(100 * (i + 1) / MarkedAssemblies.Count);
                }
            }
            catch (IOException)
            {
                MessageBox.Show("One or more files could not be written. Patching was unsuccessful.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            catch (Exception)
            {
                MessageBox.Show("An unknown error occurred.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ApplyPatchesWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            SBProgress.Value = e.ProgressPercentage;
        }

        private void CleanUp()
        {
            Tree.Nodes.Clear();

            if (MarkedAssemblies != null) MarkedAssemblies.Clear();
            if (PrimaryMarkedAssemblies != null) PrimaryMarkedAssemblies.Clear();
            if (nodes != null) nodes.Clear();
            if (assemblies != null) assemblies.Clear();
            lstIndirectlyMarkedAssemblies.Items.Clear();
            lstMarkedAssemblies.Items.Clear();
            AssemblySummary.Items.Clear();
        }

        private void Reload()
        {
            GC.Collect();
            DisableGUI();
            LoadModulesWorker.RunWorkerAsync(seed_files);
        }

        private void ApplyPatchesWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            SBLabel.Text = "Finished patching files.";
            SBProgress.Value = 0;
            SBProgress.Visible = false;

            CleanUp();
            Reload();
        }

        private void cmdMarkAll_Click(object sender, EventArgs e)
        {
            MarkedAssemblies.Clear();
            PrimaryMarkedAssemblies.Clear();
            String master_directory = Path.GetDirectoryName(seed_files[0]);
            foreach (DictionaryEntry entry in assemblies)
            {
                NETAssembly assembly = (NETAssembly)entry.Value;
                if (assembly != null)
                {
                    if (assembly.IsLoaded())
                    {
                        if (assembly.Location.Contains(master_directory))
                        {
                            PrimaryMarkedAssemblies.Add(assembly);
                        }
                    }
                }
            }
            UpdateMarkBackgroundWorker.RunWorkerAsync();
        }

        List<String> tree_problems = new List<string>();

        bool PublicKeyCompare(byte[] Key1, byte[] Key2)
        {
            if (Key1.Length != Key2.Length) return false;
            for (int i = 0; i < Key1.Length; ++i)
            {
                if (Key1[i] != Key2[i]) return false;
            }
            return true;
        }

        void VerifyTree(TreeNode node)
        {
            if (node == null)
            {
                tree_problems.Clear();
                foreach (TreeNode root in Tree.Nodes)
                {
                    VerifyTree(root);
                }
                return;
            }
            // Verify this node
            NETAssembly assembly = ((NETAssembly)node.Tag);
            if (assembly != null)
            {
                foreach (AssemblyRef reference in assembly.GetReferencedAssemblies())
                {
                    NETAssembly ref_assembly = (NETAssembly)assemblies[reference.Name];
                    if (ref_assembly != null)
                    {
                        if (reference.PublicKey.Length == 0 && ref_assembly.PublicKey.Length != 0)
                        {
                            tree_problems.Add("'" + assembly.Name + "' demands that that '" + reference.Name + "' is not signed.");
                        }
                        else if (reference.PublicKey.Length != 0 && ref_assembly.PublicKey.Length == 0)
                        {
                            tree_problems.Add("'" + assembly.Name + "' demands that that '" + reference.Name + "' is signed.");
                        }
                        else if (reference.PublicKey.Length != 0 && ref_assembly.PublicKey.Length != 0)
                            if (!PublicKeyCompare(reference.PublicKey, ref_assembly.PublicKey))
                            {
                                // Mismatch
                                tree_problems.Add("'" + assembly.Name + "' AssemblyRef public key mismatch with '" + reference.Name + "'");
                            }
                    }
                }
            }
        }

        void UndoChanges()
        {
            // List all paths concerned
            List<String> paths = new List<string>();
            foreach (DictionaryEntry de_assembly in assemblies)
            {
                NETAssembly assembly = (NETAssembly)de_assembly.Value;
                assembly.Close();
                String path = Path.GetDirectoryName(assembly.Location);
                if (!paths.Contains(path)) paths.Add(path);
            }

            CleanUp();
            GC.Collect();

            // Restore from backup folder
            foreach (String path in paths)
            {
                try
                {
                    String backup_path = path + BACKUP_FOLDER;
                    string[] backup_files = Directory.GetFiles(backup_path);
                    foreach (string backup_file in backup_files)
                    {
                        File.Copy(backup_file, backup_file.Replace(BACKUP_FOLDER, ""), true);
                    }
                }
                catch (IOException)
                {
                    // Do nothing
                }
            }

            Reload();
        }

        private void cmdUndo_Click(object sender, EventArgs e)
        {
            DialogResult dr = MessageBox.Show("This will restore all listed files to their original backups. Do you want to continue?", "Confirm Overwrite", MessageBoxButtons.OKCancel, MessageBoxIcon.Information);
            if (dr == DialogResult.OK)
            {
                UndoChanges();
            }
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            (new About()).Show();
        }
    }
}
