//////////////////////////////////////////////////////////////
//                                                          //
//  D3DLookingGlass.cpp                                     //
//  Version 0.1                                             //
//  Greg Jenkins, April 2008                                //
//  Ring3 Circus (www.ring3circus.com)                      //
//  Creative Commons Attribution 3.0 Unported License       //
//                                                          //
//  Global-hook DLL to force Direct3D 9 programs to run     //
//  in windowed mode.                                       //
//                                                          //
//////////////////////////////////////////////////////////////

#include <fstream>
#include <Windows.h>
#include <d3d9.h>

// Don't link against D3D9 so we can determine if it was already present
// in the process we get injected into.
// #pragma comment(lib, "d3d9.lib")

// Forward declarations
void WriteHook(void* address, unsigned char* patch);
void InstallSecondHook(IDirect3D9* d3d);
void InstallThirdHook(IDirect3DDevice9* device);
void FixPresentationParameters(D3DPRESENT_PARAMETERS *pPresentationParameters);

void* add_Direct3DCreate9 = NULL;
void* add_CreateDevice = NULL;
void* add_Reset = NULL;
void* add_CreateAdditionalSwapChain = NULL;
void* add_CreateWindowExW = NULL;
void* add_ShowWindow = NULL;

unsigned char backup_Direct3DCreate9[5];
unsigned char patch_Direct3DCreate9[5];

unsigned char backup_CreateDevice[5];
unsigned char patch_CreateDevice[5];

unsigned char backup_CreateAdditionalSwapChain[5];
unsigned char patch_CreateAdditionalSwapChain[5];

unsigned char backup_Reset[5];
unsigned char patch_Reset[5];

unsigned char backup_CreateWindowExW[5];
unsigned char patch_CreateWindowExW[5];

unsigned char backup_ShowWindow[5];
unsigned char patch_ShowWindow[5];

bool installed_first_hook = false;
bool installed_second_hook = false;
bool installed_third_hook = false;

int override_width = 0;
int override_height = 0;

IDirect3D9* __stdcall Direct3DCreate9Hook(UINT SDKVersion) {
	WriteHook(add_Direct3DCreate9, backup_Direct3DCreate9);

	IDirect3D9* new_d3d = NULL;
	__asm {
		PUSH SDKVersion
		CALL add_Direct3DCreate9
		MOV new_d3d, EAX
	}

	WriteHook(add_Direct3DCreate9, patch_Direct3DCreate9);
	InstallSecondHook(new_d3d);
	return new_d3d;
}

HRESULT __stdcall CreateDeviceHook(IDirect3D9* d3d9, UINT Adapter, D3DDEVTYPE DeviceType, 
								   HWND hFocusWindow, DWORD BehaviorFlags, 
								   D3DPRESENT_PARAMETERS *pPresentationParameters, 
								   IDirect3DDevice9 **ppReturnedDeviceInterface) {
	WriteHook(add_CreateDevice, backup_CreateDevice);
	FixPresentationParameters(pPresentationParameters);

	// Since we haven't statically linked against d3d9.dll, these functions need
	// to be invoked via their function-pointers as calculated earlier
	HRESULT return_value = S_OK; 

	__asm {
		PUSH ppReturnedDeviceInterface
		PUSH pPresentationParameters
		PUSH BehaviorFlags
		PUSH hFocusWindow
		PUSH DeviceType
		PUSH Adapter
		PUSH d3d9

		CALL add_CreateDevice
		MOV return_value, EAX
	}
	
	WriteHook(add_CreateDevice, patch_CreateDevice);
	if (*ppReturnedDeviceInterface != NULL) InstallThirdHook(*ppReturnedDeviceInterface);
	return return_value;
}

HRESULT __stdcall ResetHook(IDirect3DDevice9* device, D3DPRESENT_PARAMETERS* pPresentationParameters) {
	WriteHook(add_Reset, backup_Reset);
	FixPresentationParameters(pPresentationParameters);

	HRESULT hresult = S_OK;
	__asm {
		PUSH pPresentationParameters
		PUSH device

		CALL add_Reset
		MOV hresult, EAX
	}

	WriteHook(add_Reset, patch_Reset);
	return hresult;
}

HRESULT __stdcall CreateAdditionalSwapChainHook(IDirect3DDevice9* device, D3DPRESENT_PARAMETERS* pPresentationParameters, 
												IDirect3DSwapChain9** ppSwapChain) 
{
	WriteHook(add_CreateAdditionalSwapChain, backup_CreateAdditionalSwapChain);
	FixPresentationParameters(pPresentationParameters);
	
	HRESULT hresult = S_OK;
	__asm {
		PUSH ppSwapChain
		PUSH pPresentationParameters
		PUSH device

		CALL add_CreateAdditionalSwapChain
		MOV hresult, EAX
	}
	
	WriteHook(add_CreateAdditionalSwapChain, patch_CreateAdditionalSwapChain);
	return hresult;
}

// Beware that under the loader, this function probably won't trigger in time.
// Windows hooks are installed by nature at the latest possible moment, and
// most cases this will be after the first call to CreateWindow. So unless
// the target program creates some windows beforehand, we'll have no control
// over the resulting window creation flags. I'm not sure how problematic
// this is yet.
HWND __stdcall CreateWindowExWHook(DWORD dwExStyle, LPCWSTR lpClassName, LPCWSTR lpWindowName, 
								  DWORD dwStyle, int X, int Y, int nWidth, int nHeight, HWND hWndParent,
								  HMENU hMenu, HINSTANCE hInstance, LPVOID lpParam) 
{
	WriteHook(add_CreateWindowExW, backup_CreateWindowExW);
	if (installed_first_hook) {
		nWidth = override_width;
		nHeight = override_height;
		dwStyle = WS_OVERLAPPED | WS_MAXIMIZEBOX | WS_MINIMIZEBOX | WS_SYSMENU | WS_THICKFRAME | WS_VISIBLE;
	}
	HWND hWnd = CreateWindowExW(dwExStyle, lpClassName, lpWindowName, dwStyle, X, Y, nWidth, nHeight, hWndParent, hMenu, hInstance, lpParam);
	WriteHook(add_CreateWindowExW, patch_CreateWindowExW);
	return hWnd;
}

BOOL __stdcall ShowWindowHook(HWND hWnd, int nCmdShow) {
	WriteHook(add_ShowWindow, backup_ShowWindow);

	if (installed_first_hook) {
		// Direct3D is present, so we can go ahead and force the window's show state.
		nCmdShow = SW_RESTORE;
	}

	BOOL result = ShowWindow(hWnd, nCmdShow);
	WriteHook(add_ShowWindow, patch_ShowWindow);
	return result;
}

void FixPresentationParameters(D3DPRESENT_PARAMETERS *pPresentationParameters) {
	// Alter presentation parameters
	pPresentationParameters->Windowed = TRUE;
	pPresentationParameters->FullScreen_RefreshRateInHz = 0;
	pPresentationParameters->BackBufferFormat = D3DFMT_UNKNOWN;
}

__declspec(naked) void* ResolveVirtualFunction(IDirect3D9* pD3D, ...) {
    __asm {
        mov eax, dword ptr ss:[esp+0x08]
        add eax, 0x8
        cmp byte ptr ds:[eax-1], 0xA0
        mov eax, dword ptr ds:[eax]
        je normal_index
        and eax, 0xFF
normal_index:
        mov ecx, eax
        mov eax, dword ptr ss:[esp+0x4]
        mov eax, dword ptr ds:[eax]
        mov eax, dword ptr ds:[eax+ecx]
        retn
    }
}

__declspec(naked) void* ResolveVirtualFunctionDevice(IDirect3DDevice9* pDevice, ...) {
    __asm {
        mov eax, dword ptr ss:[esp+0x08]
        add eax, 0x8
        cmp byte ptr ds:[eax-1], 0xA0
        mov eax, dword ptr ds:[eax]
        je normal_index
        and eax, 0xFF
normal_index:
        mov ecx, eax
        mov eax, dword ptr ss:[esp+0x4]
        mov eax, dword ptr ds:[eax]
        mov eax, dword ptr ds:[eax+ecx]
        retn
    }
}

void WriteHook(void* address, unsigned char* patch) {
	// This will write the five-byte buffer at 'patch' to 'address',
	// after making sure that it's safe to write there

	// Set access
	DWORD old_protect = 0;
	if (VirtualProtect(address, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	if (IsBadWritePtr(address, 5) != FALSE) return;

	memcpy(address, reinterpret_cast<void*> (patch), 5);
}

// Static hook for CreateWindowExW, ShowWindow
void InstallWindowHooks() {
	HMODULE user32_dll = GetModuleHandle("user32.dll");
	if (!user32_dll) return; // This won't happen for any Win32 processes

	// Locate functions
	add_CreateWindowExW = GetProcAddress(user32_dll, "CreateWindowExW");
	if (!add_CreateWindowExW) return;

	add_ShowWindow = GetProcAddress(user32_dll, "ShowWindow");
	if (!add_ShowWindow) return;

	// Create Backups
	DWORD new_protect = PAGE_EXECUTE_READWRITE;
	DWORD old_protect = 0;
	if (VirtualProtect(add_CreateWindowExW, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	memcpy(backup_CreateWindowExW, add_CreateWindowExW, 5);

	if (VirtualProtect(add_ShowWindow, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	memcpy(backup_ShowWindow, add_ShowWindow, 5);

	// Create patches
	DWORD from_int = reinterpret_cast<DWORD> (add_CreateWindowExW);
	DWORD to_int = reinterpret_cast<DWORD> (&CreateWindowExWHook);
	DWORD offset = to_int - from_int - 5;

	patch_CreateWindowExW[0] = 0xE9; // Rel32 JMP
	*(reinterpret_cast<DWORD*> (patch_CreateWindowExW + 1)) = offset;

	from_int = reinterpret_cast<DWORD> (add_ShowWindow);
	to_int = reinterpret_cast<DWORD> (&ShowWindowHook);
	offset = to_int - from_int - 5;

	patch_ShowWindow[0] = 0xE9; // Rel32 JMP
	*(reinterpret_cast<DWORD*> (patch_ShowWindow + 1)) = offset;

	// Install hook
	WriteHook(add_CreateWindowExW, patch_CreateWindowExW);
	WriteHook(add_ShowWindow, patch_ShowWindow);
}

// Static hook for Direct3DCreate
void InstallFirstHook() {
	if (installed_first_hook) return;

	HMODULE d3d9_dll = GetModuleHandle("d3d9.dll");
	if (!d3d9_dll) {
		// This process isn't a Direct3D application, or it hasn't bound
		// to the DLL yet
		return;
	}

	// Locate function
	add_Direct3DCreate9 = GetProcAddress(d3d9_dll, "Direct3DCreate9");
	if (!add_Direct3DCreate9) return;

	// Create backup
	DWORD old_protect = 0;
	
	if (VirtualProtect(add_Direct3DCreate9, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	memcpy(backup_Direct3DCreate9, add_Direct3DCreate9, 5);

	// Create patch
	DWORD from_int, to_int, offset;
	from_int = reinterpret_cast<DWORD> (add_Direct3DCreate9);
	to_int = reinterpret_cast<DWORD> (&Direct3DCreate9Hook);
	offset = to_int - from_int - 5;

	patch_Direct3DCreate9[0] = 0xE9; // Rel32 JMP
	*(reinterpret_cast<DWORD*> (patch_Direct3DCreate9 + 1)) = offset;

	// Install hook
	WriteHook(add_Direct3DCreate9, patch_Direct3DCreate9);
	
	installed_first_hook = true;
}

// Dynamic hook for IDirect3D9::CreateDevice
void InstallSecondHook(IDirect3D9* d3d) {
	if (installed_second_hook) return;
	
	// Locate functions
	add_CreateDevice =	ResolveVirtualFunction(d3d, &IDirect3D9::CreateDevice);
	if (!add_CreateDevice) return;

	// Create backup
	DWORD old_protect = 0;
	
	if (VirtualProtect(add_CreateDevice, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	memcpy(backup_CreateDevice, add_CreateDevice, 5);

	// Create patch
	DWORD from_int, to_int, offset;
	from_int = reinterpret_cast<DWORD> (add_CreateDevice);
	to_int = reinterpret_cast<DWORD> (&CreateDeviceHook);
	offset = to_int - from_int - 5;

	patch_CreateDevice[0] = 0xE9; // Rel32 JMP
	*(reinterpret_cast<DWORD*> (patch_CreateDevice + 1)) = offset;

	// Install hook
	WriteHook(add_CreateDevice, patch_CreateDevice);

	installed_second_hook = true;
}

// Dynamic hooks for IDirect3DDevice9::CreateAdditionalSwapChain, IDirect3DDevice9::Reset
void InstallThirdHook(IDirect3DDevice9* device) {
	if (installed_third_hook) return;
	
	// Locate functions
	add_Reset =	ResolveVirtualFunctionDevice(device, &IDirect3DDevice9::Reset);
	if (!add_Reset) return;

	add_CreateAdditionalSwapChain =	ResolveVirtualFunctionDevice(device, &IDirect3DDevice9::CreateAdditionalSwapChain);
	if (!add_CreateAdditionalSwapChain) return;

	// Create backups
	DWORD old_protect = 0;
	
	if (VirtualProtect(add_Reset, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	memcpy(backup_Reset, add_Reset, 5);

	if (VirtualProtect(add_CreateAdditionalSwapChain, 5, PAGE_EXECUTE_READWRITE, &old_protect) == FALSE) return;
	memcpy(backup_CreateAdditionalSwapChain, add_CreateAdditionalSwapChain, 5);

	// Create patches
	DWORD from_int, to_int, offset;
	from_int = reinterpret_cast<DWORD> (add_Reset);
	to_int = reinterpret_cast<DWORD> (&ResetHook);
	offset = to_int - from_int - 5;

	patch_Reset[0] = 0xE9; // Rel32 JMP
	*(reinterpret_cast<DWORD*> (patch_Reset + 1)) = offset;

	from_int = reinterpret_cast<DWORD> (add_CreateAdditionalSwapChain);
	to_int = reinterpret_cast<DWORD> (&CreateAdditionalSwapChainHook);
	offset = to_int - from_int - 5;

	patch_CreateAdditionalSwapChain[0] = 0xE9; // Rel32 JMP
	*(reinterpret_cast<DWORD*> (patch_CreateAdditionalSwapChain + 1)) = offset;

	// Install hooks
	WriteHook(add_Reset, patch_Reset);
	WriteHook(add_CreateAdditionalSwapChain, patch_CreateAdditionalSwapChain);

	installed_third_hook = true;
}

void CleanUp() {
	if ((add_CreateDevice) && (backup_CreateDevice[0])) {
		WriteHook(add_CreateDevice, backup_CreateDevice);
	}
	if ((add_Direct3DCreate9) && (backup_Direct3DCreate9[0])) {
		WriteHook(add_Direct3DCreate9, backup_Direct3DCreate9);
	}
	if ((add_Reset) && (backup_Reset[0])) {
		WriteHook(add_Reset, backup_Reset);
	}
	if ((add_CreateAdditionalSwapChain) && (backup_CreateAdditionalSwapChain[0])) {
		WriteHook(add_CreateAdditionalSwapChain, backup_CreateAdditionalSwapChain);
	}
	if ((add_CreateWindowExW) && (backup_CreateWindowExW[0])) {
		WriteHook(add_CreateWindowExW, backup_CreateWindowExW);
	}
	if ((add_ShowWindow) && (backup_ShowWindow[0])) {
		WriteHook(add_ShowWindow, backup_ShowWindow);
	}
}

LRESULT CALLBACK CBTProc(int nCode, WPARAM wParam, LPARAM lParam) { 
	if (nCode > 0) {
		switch (nCode) 
		{ 
			case HCBT_ACTIVATE:
			case HCBT_CREATEWND:
				// If the program late-binds to d3d9.dll, then the intial install attempt
				// will fail. This will solve that provided the linkage is made before
				// the device window is created
				InstallFirstHook();
				break;
		}
    } 
    return CallNextHookEx(0, nCode, wParam, lParam);
}

void LoadSettings(HMODULE module) {
	char file_name[MAX_PATH];
	GetModuleFileNameA(module, (LPCH) &file_name, MAX_PATH);
	strncat_s(file_name, MAX_PATH, ".settings", 10);
	std::ifstream settings(file_name, std::ios::binary);
	if (!settings) {
		// Failed to load settings - use defaults
		return;
	}
	settings.read(reinterpret_cast<char*> (&override_width), 4);
	settings.read(reinterpret_cast<char*> (&override_height), 4);
}

BOOL APIENTRY DllMain(HINSTANCE hModule, DWORD dwReason, PVOID lpReserved) {	
	switch (dwReason) 
	{
    case DLL_PROCESS_ATTACH:
		IsDebuggerPresent();
		LoadSettings(hModule);
		InstallWindowHooks();
		InstallFirstHook();
		break;
	case DLL_PROCESS_DETACH:
		CleanUp();
		break;		
	}
	return TRUE;
}

void Binder() {
	// I'm sure there's a better way to force a .NET assembly to statically link
	// against a regular Win32 DLL, but this will do for the moment
}