using System;

namespace Support.ILConverter.SDILReader
{
    #region support class
    public static class Globals
    {
        static Globals()
        {
            LoadOpCodes();
        }

        public static System.Collections.Generic.Dictionary<int, object> Cache = new System.Collections.Generic.Dictionary<int, object>();

        public static System.Reflection.Emit.OpCode[] multiByteOpCodes;
        public static System.Reflection.Emit.OpCode[] singleByteOpCodes;
        public static System.Reflection.Module[] modules = null;

        private static void LoadOpCodes()
        {
            singleByteOpCodes = new System.Reflection.Emit.OpCode[0x100];
            multiByteOpCodes = new System.Reflection.Emit.OpCode[0x100];
            System.Reflection.FieldInfo[] infoArray1 = typeof(System.Reflection.Emit.OpCodes).GetFields();
            for (int num1 = 0; num1 < infoArray1.Length; num1++)
            {
                System.Reflection.FieldInfo info1 = infoArray1[num1];
                if (info1.FieldType == typeof(System.Reflection.Emit.OpCode))
                {
                    System.Reflection.Emit.OpCode code1 = (System.Reflection.Emit.OpCode)info1.GetValue(null);
                    ushort num2 = (ushort)code1.Value;
                    if (num2 < 0x100)
                    {
                        singleByteOpCodes[(int)num2] = code1;
                    }
                    else
                    {
                        if ((num2 & 0xff00) != 0xfe00)
                        {
                            throw new Exception("Invalid OpCode.");
                        }
                        multiByteOpCodes[num2 & 0xff] = code1;
                    }
                }
            }
        }


        /// <summary>
        /// Retrieve the friendly name of a type
        /// </summary>
        /// <param name="typeName">
        /// The complete name to the type
        /// </param>
        /// <returns>
        /// The simplified name of the type (i.e. "int" instead f System.Int32)
        /// </returns>
        public static string ProcessSpecialTypes(string typeName)
        {
            string result = typeName;
            switch (typeName)
            {
                case "System.string":
                case "System.String":
                case "String":
                    result = "string"; break;
                case "System.Int32":
                case "Int":
                case "Int32":
                    result = "int"; break;
            }
            return result;
        }
    }
    public class ILInstruction
    {
        // Fields
        private System.Reflection.Emit.OpCode code;
        private object operand;
        private byte[] operandData;
        private int offset;

        // Properties
        public System.Reflection.Emit.OpCode Code
        {
            get { return code; }
            set { code = value; }
        }

        public object Operand
        {
            get { return operand; }
            set { operand = value; }
        }

        public byte[] OperandData
        {
            get { return operandData; }
            set { operandData = value; }
        }

        public int Offset
        {
            get { return offset; }
            set { offset = value; }
        }

        /// <summary>
        /// Returns a friendly strign representation of this instruction
        /// </summary>
        /// <returns></returns>
        public string GetCode()
        {
            string result = "";
            result += GetExpandedOffset(offset) + " : " + code;
            if (operand != null)
            {
                switch (code.OperandType)
                {
                    case System.Reflection.Emit.OperandType.InlineField:
                        System.Reflection.FieldInfo fOperand = ((System.Reflection.FieldInfo)operand);
                        result += " " + Globals.ProcessSpecialTypes(fOperand.FieldType.ToString()) + " " +
                            Globals.ProcessSpecialTypes(fOperand.ReflectedType.ToString()) +
                            "::" + fOperand.Name + "";
                        break;
                    case System.Reflection.Emit.OperandType.InlineMethod:
                        try
                        {
                            System.Reflection.MethodInfo mOperand = (System.Reflection.MethodInfo)operand;
                            result += " ";
                            if (!mOperand.IsStatic) result += "instance ";
                            result += Globals.ProcessSpecialTypes(mOperand.ReturnType.ToString()) +
                                " " + Globals.ProcessSpecialTypes(mOperand.ReflectedType.ToString()) +
                                "::" + mOperand.Name + "()";
                        }
                        catch
                        {
                            try
                            {
                                System.Reflection.ConstructorInfo mOperand = (System.Reflection.ConstructorInfo)operand;
                                result += " ";
                                if (!mOperand.IsStatic) result += "instance ";
                                result += "void " +
                                    Globals.ProcessSpecialTypes(mOperand.ReflectedType.ToString()) +
                                    "::" + mOperand.Name + "()";
                            }
                            catch
                            {
                            }
                        }
                        break;
                    case System.Reflection.Emit.OperandType.ShortInlineBrTarget:
                    case System.Reflection.Emit.OperandType.InlineBrTarget:
                        result += " " + GetExpandedOffset((int)operand);
                        break;
                    case System.Reflection.Emit.OperandType.InlineType:
                        result += " " + Globals.ProcessSpecialTypes(operand.ToString());
                        break;
                    case System.Reflection.Emit.OperandType.InlineString:
                        if (operand.ToString() == "\r\n") result += " \"\\r\\n\"";
                        else result += " \"" + operand.ToString() + "\"";
                        break;
                    case System.Reflection.Emit.OperandType.ShortInlineVar:
                        result += operand.ToString();
                        break;
                    case System.Reflection.Emit.OperandType.InlineI:
                    case System.Reflection.Emit.OperandType.InlineI8:
                    case System.Reflection.Emit.OperandType.InlineR:
                    case System.Reflection.Emit.OperandType.ShortInlineI:
                    case System.Reflection.Emit.OperandType.ShortInlineR:
                        result += operand.ToString();
                        break;
                    case System.Reflection.Emit.OperandType.InlineTok:
                        if (operand is Type)
                            result += ((Type)operand).FullName;
                        else
                            result += "not supported";
                        break;

                    default: result += "not supported"; break;
                }
            }
            return result;

        }

        /// <summary>
        /// Add enough zeros to a number as to be represented on 4 characters
        /// </summary>
        /// <param name="offset">
        /// The number that must be represented on 4 characters
        /// </param>
        /// <returns>
        /// </returns>
        private string GetExpandedOffset(long offset)
        {
            string result = offset.ToString();
            for (int i = 0; result.Length < 4; i++)
            {
                result = "0" + result;
            }
            return result;
        }

        public ILInstruction()
        {

        }
    }
    #endregion ---support class---

    /// <summary>
    /// Thanks to Sorin Serban for this code
    /// </summary>
    public static class MethodBodyReader
    {
        #region il read methods
        private static int ReadInt16(byte[] il, ref int position)
        {
            return ((il[position++] | (il[position++] << 8)));
        }
        private static ushort ReadUInt16(byte[] il, ref int position)
        {
            return (ushort)((il[position++] | (il[position++] << 8)));
        }
        private static int ReadInt32(byte[] il, ref int position)
        {
            return (((il[position++] | (il[position++] << 8)) | (il[position++] << 0x10)) | (il[position++] << 0x18));
        }
        private static ulong ReadInt64(byte[] il, ref int position)
        {
            return (ulong)(((il[position++] | (il[position++] << 8)) | (il[position++] << 0x10)) | (il[position++] << 0x18) | (il[position++] << 0x20) | (il[position++] << 0x28) | (il[position++] << 0x30) | (il[position++] << 0x38));
        }
        private static double ReadDouble(byte[] il, ref int position)
        {
            return (((il[position++] | (il[position++] << 8)) | (il[position++] << 0x10)) | (il[position++] << 0x18) | (il[position++] << 0x20) | (il[position++] << 0x28) | (il[position++] << 0x30) | (il[position++] << 0x38));
        }
        private static sbyte ReadSByte(byte[] il, ref int position)
        {
            return (sbyte)il[position++];
        }
        private static byte ReadByte(byte[] il, ref int position)
        {
            return (byte)il[position++];
        }
        private static Single ReadSingle(byte[] il, ref int position)
        {
            return (Single)(((il[position++] | (il[position++] << 8)) | (il[position++] << 0x10)) | (il[position++] << 0x18));
        }
        #endregion

        #region getCode
        /// <summary>
        /// Gets the IL code of the method
        /// </summary>
        /// <returns></returns>
        public static string GetBodyCode(System.Reflection.MethodInfo mi)
        {
            System.Text.StringBuilder result = new System.Text.StringBuilder();
            System.Collections.Generic.List<SDILReader.ILInstruction> instructions = ConstructInstructions(mi);
            if (instructions != null)
            {
                for (int i = 0; i < instructions.Count; i++)
                {
                    result.AppendLine(instructions[i].GetCode());
                }
            }
            return result.ToString();
        }

        /// <summary>
        /// open function inside of code
        /// Constructs the array of ILInstructions according to the IL byte code.
        /// </summary>
        /// <param name="module"></param>
        public static object GetRefferencedOperand(System.Reflection.Module module, int metadataToken)
        {
            System.Reflection.AssemblyName[] assemblyNames = module.Assembly.GetReferencedAssemblies();
            for (int i = 0; i < assemblyNames.Length; i++)
            {
                System.Reflection.Module[] modules = System.Reflection.Assembly.Load(assemblyNames[i]).GetModules();
                for (int j = 0; j < modules.Length; j++)
                {
                    try
                    {
                        Type t = modules[j].ResolveType(metadataToken);
                        return t;
                    }
                    catch
                    {

                    }

                }
            }
            return null;
        }
        #endregion ---getCode---

        #region ConstructInstructions
        private static System.Collections.Generic.List<SDILReader.ILInstruction> ConstructInstructions(System.Reflection.MethodInfo metIN)
        {
            byte[] il = metIN.GetMethodBody().GetILAsByteArray();
            System.Reflection.Module module = metIN.Module;
            int position = 0;
            System.Collections.Generic.List<SDILReader.ILInstruction> instructions = null;
            instructions = new System.Collections.Generic.List<ILInstruction>();
            while (position < il.Length)
            {
                ILInstruction instruction = new ILInstruction();

                // get the operation code of the current instruction
                System.Reflection.Emit.OpCode code = System.Reflection.Emit.OpCodes.Nop;
                ushort value = il[position++];
                if (value != 0xfe)
                {
                    code = Globals.singleByteOpCodes[(int)value];
                }
                else
                {
                    value = il[position++];
                    code = Globals.multiByteOpCodes[(int)value];
                    value = (ushort)(value | 0xfe00);
                }
                instruction.Code = code;
                instruction.Offset = position - 1;
                int metadataToken = 0;
                // get the operand of the current operation
                switch (code.OperandType)
                {
                    case System.Reflection.Emit.OperandType.InlineBrTarget:
                        metadataToken = ReadInt32(il, ref position);
                        metadataToken += position;
                        instruction.Operand = metadataToken;
                        break;
                    case System.Reflection.Emit.OperandType.InlineField:
                        metadataToken = ReadInt32(il, ref position);
                        instruction.Operand = module.ResolveField(metadataToken);
                        break;
                    case System.Reflection.Emit.OperandType.InlineMethod:
                        metadataToken = ReadInt32(il, ref position);
                        try
                        {
                            instruction.Operand = module.ResolveMethod(metadataToken);
                        }
                        catch
                        {
                            instruction.Operand = module.ResolveMember(metadataToken);
                        }
                        break;
                    case System.Reflection.Emit.OperandType.InlineSig:
                        metadataToken = ReadInt32(il, ref position);
                        instruction.Operand = module.ResolveSignature(metadataToken);
                        break;
                    case System.Reflection.Emit.OperandType.InlineTok:
                        metadataToken = ReadInt32(il, ref position);
                        try
                        {
                            instruction.Operand = module.ResolveType(metadataToken);
                        }
                        catch
                        {

                        }
                        // SSS : see what to do here
                        break;
                    case System.Reflection.Emit.OperandType.InlineType:
                        metadataToken = ReadInt32(il, ref position);
                        // now we call the ResolveType always using the generic attributes type in order
                        // to support decompilation of generic methods and classes

                        // thanks to the guys from code project who commented on this missing feature

                        instruction.Operand = module.ResolveType(metadataToken, metIN.DeclaringType.GetGenericArguments(), metIN.GetGenericArguments());
                        break;
                    case System.Reflection.Emit.OperandType.InlineI:
                        {
                            instruction.Operand = ReadInt32(il, ref position);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.InlineI8:
                        {
                            instruction.Operand = ReadInt64(il, ref position);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.InlineNone:
                        {
                            instruction.Operand = null;
                            break;
                        }
                    case System.Reflection.Emit.OperandType.InlineR:
                        {
                            instruction.Operand = ReadDouble(il, ref position);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.InlineString:
                        {
                            metadataToken = ReadInt32(il, ref position);
                            instruction.Operand = module.ResolveString(metadataToken);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.InlineSwitch:
                        {
                            int count = ReadInt32(il, ref position);
                            int[] casesAddresses = new int[count];
                            for (int i = 0; i < count; i++)
                            {
                                casesAddresses[i] = ReadInt32(il, ref position);
                            }
                            int[] cases = new int[count];
                            for (int i = 0; i < count; i++)
                            {
                                cases[i] = position + casesAddresses[i];
                            }
                            break;
                        }
                    case System.Reflection.Emit.OperandType.InlineVar:
                        {
                            instruction.Operand = ReadUInt16(il, ref position);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.ShortInlineBrTarget:
                        {
                            instruction.Operand = ReadSByte(il, ref position) + position;
                            break;
                        }
                    case System.Reflection.Emit.OperandType.ShortInlineI:
                        {
                            instruction.Operand = ReadSByte(il, ref position);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.ShortInlineR:
                        {
                            instruction.Operand = ReadSingle(il, ref position);
                            break;
                        }
                    case System.Reflection.Emit.OperandType.ShortInlineVar:
                        {
                            instruction.Operand = ReadByte(il, ref position);
                            break;
                        }
                    default:
                        {
                            throw new Exception("Unknown operand type.");
                        }
                }
                instructions.Add(instruction);
            }
            return instructions;
        }
        #endregion ---ConstructInstructions---
    }
}
