#ifndef __BPR_H__
#define __BPR_H__

#include "defs.h"

#ifdef  __BPR_C__
#define EXPORT
#define INIT    =0
#else
#define EXPORT       extern
#define INIT
#endif

////////////////////////////////////////////////////////////////////////////
//                            CONSTANTS            
////////////////////////////////////////////////////////////////////////////
#define MAX_BPR      32         // 32 BPRs maximum allowed

////////////////////////////////////////////////////////////////////////////
//                           GLOBAL VARS            
////////////////////////////////////////////////////////////////////////////
EXPORT ULONG bpr_Count  INIT;   // Is there any breakpoints on range active?
                                // Just for speedup
                                // Contains:
                                // 0 - no bpr is active
                                // ELSE bpr count

////////////////////////////////////////////////////////////////////////////
//                  BREAK POINT ON RANGE STRUCTURE            
////////////////////////////////////////////////////////////////////////////

typedef struct BPR_TYPE
{
    ULONG Flags;         // ENABLE or DISABLED

#define BPRFLG_DISABLED      1
#define BPRFLG_DEACTIVATED   2
#define BPRFLG_SET           4

    ULONG Type;          // 'R' - ON READ
                         // 'W' - ON WRITE
                         // 'G' - ON READ or WRITE (Generic)
                         // 'X' - ON EXECUTION

    ULONG CR3;           // Identifies address space of the process in which
                         // breakpoint is located

    ULONG VA;            // Virtual address of the breakpoint start

    ULONG Len;           // Length of the breakpoint in bytes
    ULONG *PTEs;         // Saved PTEs
}BPR_TYPE;

////////////////////////////////////////////////////////////////////////////
//                     LOCAL DEFINITIONS            
////////////////////////////////////////////////////////////////////////////
static BPR_TYPE BPR[MAX_BPR];

////////////////////////////////////////////////////////////////////////////
//                              Function prototypes
////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////
//                                                                          
// bpr_Clear 
//  Clears breakpoint on range.
//
////////////////////////////////////////////////////////////////////////////
void bpr_Clear(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
//
// bpr_Set 
//  Set breakpoint on range.
//
////////////////////////////////////////////////////////////////////////////
void bpr_Set(ULONG bpr_Type,
             ULONG bpr_Addr,
             ULONG bpr_Len,
             ULONG bpr_CR3);

////////////////////////////////////////////////////////////////////////////
//
// bpr_Check 
//  Check if any break point on range matches conditions
//
////////////////////////////////////////////////////////////////////////////
ULONG bpr_Check(ULONG _CR3, ULONG _CR2, ULONG ErrCode);

////////////////////////////////////////////////////////////////////////////
//
// bpr_Disable 
//  Disables given breakpoint on range.
//
////////////////////////////////////////////////////////////////////////////
void bpr_Disable(ULONG bpNo);
                                                                            
////////////////////////////////////////////////////////////////////////////
//
// bpr_Enable 
//  Enables given breakpoint on range.
//
////////////////////////////////////////////////////////////////////////////
void bpr_Enable(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
//
// bpr_IsDisabled 
//  Check if given breakpoint is disabled.
//
////////////////////////////////////////////////////////////////////////////
ULONG bpr_IsDisabled(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
//
// bpr_SavePTEsAndMakeInvalid
//  Saves PTEs of the given breakpoint
//
////////////////////////////////////////////////////////////////////////////
ULONG bpr_SavePTEsAndMakeInvalid(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
//
// bpr_RestorePTEs
//  Restores PTEs of the given breakpoint
//
////////////////////////////////////////////////////////////////////////////
ULONG bpr_RestorePTEs(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
//
// bpr_Deactivate 
//  Deactivates given breakpoint on range.
//
////////////////////////////////////////////////////////////////////////////
void bpr_Deactivate(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
//
// bpr_Activate 
//  Activates given breakpoint on range.
//
////////////////////////////////////////////////////////////////////////////
ULONG bpr_Activate(ULONG bpNo);

////////////////////////////////////////////////////////////////////////////
// bpr_ActivateAll 
//
//   Tries to activate all breakpoints on range.
//
////////////////////////////////////////////////////////////////////////////
ULONG bpr_ActivateAll();

////////////////////////////////////////////////////////////////////////////
// bpr_DeactivateAll 
//
//   Tries to deactivate all breakpoints on range.
//
////////////////////////////////////////////////////////////////////////////
void bpr_DeactivateAll();

////////////////////////////////////////////////////////////////////////////
//
// bpr_ClearAll
//  Clear all breakpoints for the current process.
//
////////////////////////////////////////////////////////////////////////////
void bpr_ClearAll();

////////////////////////////////////////////////////////////////////////////
//
// bpr_ListAll
//  Lists all breakpoints on range
//
////////////////////////////////////////////////////////////////////////////
void bpr_ListAll();

////////////////////////////////////////////////////////////////////////////
//
// bpr_CheckPTEs 
//  Check PTEs of given break point for range for validity.
//  Print error messages if asked.
//
////////////////////////////////////////////////////////////////////////////
BOOLEAN bpr_CheckPTEs(ULONG bpNo, BOOLEAN PrintInfo);

#undef EXPORT
#undef INIT

#endif
