/*++
    original: Tetris v1.0 by Joe Wingbermuehle
              19981226
              http://www.usmo.com/~joewing
              joewing@usmo.com
 
Module Name:
    cmd_tetris.cpp

Abstract: Implements !TETRIS extension commands. 

Adaptation to SoftICE 9x: The Owl 2000/07
   
Adaptation to SoftICE NT: Sten    2002/06
 
    keys: left/right arrows to move the block
          up arrow to rotate it
          down arrow to make it fall faster
          esc to end the game
--*/
 
extern "C" {
#pragma warning ( push, 3 )
#include <ntddk.h>
#pragma warning ( pop )
}

#pragma warning ( disable: 4514 ) // unreferenced inline function has been removed

#include <windef.h>
#include <ntverp.h>

#include <stdio.h>

#include "wdbgexts.h"
#include "defs.h"
#include "softice.h"
#include "screen.h"
#include "keyboard.h"

static DWORD    ticks       = 0;

static DWORD    seed        = 0;
static DWORD    a           = 16807;
static DWORD    r           = 2836;
static DWORD    q           = 127773;

static DWORD    delay       = 200;

static DWORD    LINES       = 0;
static DWORD    WIDTH       = 0;
static DWORD    NumOfBlocks = 7;

static struct stats
{
    DWORD    score;
    DWORD    lines; 
    DWORD    blocks;
}stats;


static struct board
{
    DWORD    x;
    DWORD    y;
    DWORD    lines;
    DWORD    width;
}board;

static struct block
{
    DWORD    x;
    DWORD    y;
    DWORD    id;
    DWORD    lines;
}block;

static WORD blocks[] =
{
    0x7F20, 0x0000, 0x0000, 0x0000,        // 00
    0x7F20, 0x0000, 0x0000, 0x0000,
    0x7F20, 0x0000, 0x0000, 0x0000,
    0x7F20, 0x0000, 0x0000, 0x0000,
    4,

    0x7F20, 0x7F20, 0x7F20, 0x7F20,        // 01
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    1,

    0x7F20, 0x0000, 0x0000, 0x0000,        // 02
    0x7F20, 0x0000, 0x0000, 0x0000,
    0x7F20, 0x0000, 0x0000, 0x0000,
    0x7F20, 0x0000, 0x0000, 0x0000,
    4,

    0x7F20, 0x7F20, 0x7F20, 0x7F20,        // 03
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    1,

    0xAF20, 0x0000, 0x0000, 0x0000,        // 10
    0xAF20, 0x0000, 0x0000, 0x0000,
    0xAF20, 0xAF20, 0x0000, 0x0000,    
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x0000, 0x0000, 0xAF20, 0x0000,        // 11
    0xAF20, 0xAF20, 0xAF20, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0xAF20, 0xAF20, 0x0000, 0x0000,        // 12
    0x0000, 0xAF20, 0x0000, 0x0000,
    0x0000, 0xAF20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0xAF20, 0xAF20, 0xAF20, 0x0000,        // 13
    0xAF20, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x5F20, 0x0000, 0x0000, 0x0000,        // 20
    0x5F20, 0x5F20, 0x0000, 0x0000,
    0x5F20, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x0000, 0x5F20, 0x0000, 0x0000,        // 21
    0x5F20, 0x5F20, 0x5F20, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x0000, 0x5F20, 0x0000, 0x0000,        // 22
    0x5F20, 0x5F20, 0x0000, 0x0000,
    0x0000, 0x5F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x5F20, 0x5F20, 0x5F20, 0x0000,        // 23
    0x0000, 0x5F20, 0x0000, 0x0000,  
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x4F20, 0x0000, 0x0000, 0x0000,       // 30
    0x4F20, 0x4F20, 0x0000, 0x0000,
    0x0000, 0x4F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x0000, 0x4F20, 0x4F20, 0x0000,       // 31
    0x4F20, 0x4F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x4F20, 0x0000, 0x0000, 0x0000,       // 32
    0x4F20, 0x4F20, 0x0000, 0x0000,
    0x0000, 0x4F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x0000, 0x4F20, 0x4F20, 0x0000,       // 33
    0x4F20, 0x4F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x0000, 0x3F20, 0x0000, 0x0000,       // 40
    0x0000, 0x3F20, 0x0000, 0x0000,
    0x3F20, 0x3F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x3F20, 0x3F20, 0x3F20, 0x0000,       // 41
    0x0000, 0x0000, 0x3F20, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x3F20, 0x3F20, 0x0000, 0x0000,       // 42
    0x3F20, 0x0000, 0x0000, 0x0000,
    0x3F20, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x3F20, 0x0000, 0x0000, 0x0000,       // 43
    0x3F20, 0x3F20, 0x3F20, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x0000, 0x2F20, 0x0000, 0x0000,       // 50
    0x2F20, 0x2F20, 0x0000, 0x0000,
    0x2F20, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x2F20, 0x2F20, 0x0000, 0x0000,       // 51
    0x0000, 0x2F20, 0x2F20, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,    

    0x0000, 0x2F20, 0x0000, 0x0000,       // 52
    0x2F20, 0x2F20, 0x0000, 0x0000,
    0x2F20, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    3,

    0x2F20, 0x2F20, 0x0000, 0x0000,       // 53
    0x0000, 0x2F20, 0x2F20, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,    

    0x9F20, 0x9F20, 0x0000, 0x0000,       // 60
    0x9F20, 0x9F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x9F20, 0x9F20, 0x0000, 0x0000,       // 61
    0x9F20, 0x9F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x9F20, 0x9F20, 0x0000, 0x0000,       // 62
    0x9F20, 0x9F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2,

    0x9F20, 0x9F20, 0x0000, 0x0000,       // 63
    0x9F20, 0x9F20, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000,
    2
};

//////////////////////////////////////////////////////////////////////////////
//  Prototypes
//////////////////////////////////////////////////////////////////////////////
void DrawBoard   (); 
void DrawNewBlock();
void DrawBlock   ();
void TetrisMain  ();
void Tick        ();

void TetrisScrollLeft  ();
void TetrisScrollRight ();
void TetrisScrollDown  ();
void TetrisFastDown    ();
void TetrisRotateBlock ();

void TetrisEraseBlock  ();
void TetrisCompactLines();


//////////////////////////////////////////////////////////////////////////////
//  !tetris [speed]
//////////////////////////////////////////////////////////////////////////////
DECLARE_API(tetris)
{
	UNREFERENCED_PARAMETER(dwProcessor);
	UNREFERENCED_PARAMETER(dwCurrentPc);
	UNREFERENCED_PARAMETER(hCurrentThread);
	UNREFERENCED_PARAMETER(hCurrentProcess);

    if (args[0]=='!') args += 9; // "! tetris "

    if (args[0] != 0)
    {
        _asm
        {
             mov       esi, args
             call      si_Expression2Integer
             jb        syntax_error
             mov       delay, eax
        }
    }
    else
         delay = 250;

    SaveScreen();
    ClearScreen();

    LINES = GetHeight();
    WIDTH = GetWidth();

    ticks = 0;
    if (seed == 0) seed = 0x3292; // TO DO: get seed from time

    si_EmptyKbdBuffer();

    board.lines = 20;
    board.width = 12;

    board.x = (WIDTH - board.width) / 2;
    board.y = (LINES - board.lines) / 2;

    stats.score  = 0;
    stats.lines  = 0;
    stats.blocks = 0;
  
    DrawBoard();
    DrawNewBlock();
   
    HideCursor();

    char ch; ch = si_GetChar();

    TetrisMain();          // main game loop

    RestoreScreen();

    DbgPrint("lines completed: %d,      score: %d\n", stats.lines, stats.score);    
    return;

syntax_error:
    DbgPrint("Syntax error.\n");
    return;
}

//////////////////////////////////////////////////////////////////////////////
//
//  TetrisMain
//    Main game loop
//
//////////////////////////////////////////////////////////////////////////////

void __declspec(naked) TetrisMain ()
{
    __asm
    {
_main_loop:
          call         Tick
          mov          eax,ticks
          cmp          eax,delay
          jb           _F01

          and          ticks, 0       // ticks = 0

          call         TetrisScrollDown
          jnc          _main_loop 

          call         DrawNewBlock
          jnc          _main_loop

          jmp          short _exit
       
_F01:
          call         si_ReadFromKbdBuffer_char

          cmp          al, KBD_ESC
          jz           _exit

          push         _main_loop     // small trick :)

          cmp          al, KBD_LEFT
          jnz          not_left
          jmp          TetrisScrollLeft
not_left:

          cmp          al, KBD_RIGHT
          jnz          not_right
          jmp          TetrisScrollRight
not_right:

          cmp          al, KBD_DOWN
          jnz          not_down
          jmp          TetrisFastDown
not_down:

          cmp          al, KBD_UP
          jnz          not_up
          jmp          TetrisRotateBlock
not_up:

_exit:
          retn
           
    }
}

void __declspec(naked) TetrisScrollLeft ()
{
    __asm
    {
          call         TetrisEraseBlock

          dec          block.x
          call         DrawBlock
          jnc          exit_loc

          inc          block.x
          call         DrawBlock

exit_loc:
          retn
    }
}

void __declspec(naked) TetrisScrollRight()
{
    __asm
    {
          call         TetrisEraseBlock

          inc          block.x
          call         DrawBlock
          jnc          exit_loc
  
          dec          block.x
          call         DrawBlock

exit_loc:
          retn
    }
}

void __declspec(naked) TetrisScrollDown()
{
    __asm
    {
          call         TetrisEraseBlock

          inc          block.y
          call         DrawBlock
          jnc          exit_loc

          dec          block.y
          call         DrawBlock

          call         TetrisCompactLines

          stc

exit_loc:
          retn
    }
}

void __declspec(naked) TetrisFastDown   ()
{
    __asm
    {
          call          TetrisScrollDown
          jnc           exit_loc

          call          DrawNewBlock
          jnc           exit_loc

          add           esp, 4          // skip _main_loop

exit_loc:
          retn
    }
}

void __declspec(naked) TetrisRotateBlock()
{
    __asm
    {
          call         TetrisEraseBlock

          push         block.id
          push         block.lines

          inc          block.id
          test         byte ptr [block.id], 3
          jnz          _F01

          sub          block.id, 4

_F01:
          imul         eax, block.id, 2*4*4+2
          add          eax, offset blocks+2*4*4
          movzx        eax,word ptr [eax]
          mov          [block.lines],eax

          call         DrawBlock
          jc           _F02

          add          esp, 8
          retn

_F02:
          pop          block.lines
          pop          block.id
          call         DrawBlock
          retn
    }
}

void __declspec(naked) TetrisCompactLines()
{
    __asm
    {
          push         ebx
          push         ecx
          push         edx
          push         esi
          push         edi

          mov          ebx,WIDTH

          call         GetScreenPtr
          mov          esi,eax

          mov          eax,block.y
          imul         eax,ebx
          add          eax,board.x
          lea          esi,[2*eax+esi]        ; esi: left border of blockwindow

          mov          ecx,block.lines

_loop_outer:
          push         ecx
          mov          edx,esi
          mov          ecx,board.width

_loop_inner:
          lodsw
          test         ax,0x0F00
          loopnz       _loop_inner

          mov          esi,edx

          jz           _next_line

; compact a line
          inc          stats.lines
          mov          eax, stats.blocks
          add          stats.score,eax

          test         delay,111b             ; gets faster after 8 completed lines
          jnz          _F01

          cmp          delay, 1               ; no that if anyone ever got this far...
          jbe          _F01

          dec          delay

_F01:
          mov          ecx,block.y
          sub          ecx,board.y
          inc          ecx

_loop_moveline: 
          push         ecx

; move down upper neighbour

          mov          edi,esi
          sub          esi,ebx
          sub          esi,ebx
          push         esi

          mov          ecx,board.width
          rep          movsw

          pop          esi
          pop          ecx
          loop         _loop_moveline

          mov          esi,edx

_next_line:
          lea          esi,[2*ebx+esi]
         
          pop          ecx
          loop         _loop_outer

          call         si_UpdateScreen

          pop          edi
          pop          esi
          pop          edx
          pop          ecx
          pop          ebx
          retn
    }
}


void __declspec(naked) TetrisEraseBlock()
{
    __asm
    {
          push         ebx
          push         ecx
          push         edx
          push         esi
          push         edi

          mov          ebx, WIDTH

          call         GetScreenPtr
          mov          edi, eax

          mov          eax, block.y
          imul         eax, ebx
          add          eax, block.x
          lea          edi, [2*eax+edi]     ; edi: upper left corner of block window

          imul         esi, block.id, 2*4*4+2
          add          esi, offset blocks   ; esi: upper left corner of blockdata
          
          mov          edx, 0x20
          mov          ecx, block.lines

_loop_draw:
          lodsw
          test         ax, ax
          jz           _F01

          mov          [edi], dx

_F01: 

          lodsw
          test         ax, ax
          jz           _F02

          mov          [edi+2], dx

_F02: 

          lodsw
          test         ax, ax
          jz           _F03

          mov          [edi+4], dx

_F03: 

          lodsw
          test         ax, ax
          jz           _F04

          mov          [edi+6], dx

_F04:
          lea          edi, [edi+2*ebx]
          loop         _loop_draw

          call         si_UpdateScreen

          pop          edi
          pop          esi
          pop          edx
          pop          ecx
          pop          ebx 
          retn
    }
}


//////////////////////////////////////////////////////////////////////////////
//
//  Tick
//
//////////////////////////////////////////////////////////////////////////////

void __declspec(naked) Tick ()
{
    __asm
    {
          mov          eax,1
          call         si_DelayMilliSec
          inc          ticks
          retn
    }
}

//////////////////////////////////////////////////////////////////////////////
//  DrawHLine
//     dl:  left.x
//     dh:  left.y
//     bl:  char
//     bh:  color
//     ecx: length
//////////////////////////////////////////////////////////////////////////////

void __declspec(naked) DrawHLine()
{
    __asm
    {
          call         si_PrintChar
          retn     
    }
}

//////////////////////////////////////////////////////////////////////////////
//  DrawVLine
//     dl: top.x
//     dh: top.y
//     bl: char
//     bh: color
//     ecx: length
//////////////////////////////////////////////////////////////////////////////
void __declspec(naked) DrawVLine()
{
    __asm
    {
drawVloop:
          push         ecx
             
          mov          ecx,1
          call         si_PrintChar

          inc          dh
          pop          ecx
          loop         drawVloop

          retn  
    }
}
//////////////////////////////////////////////////////////////////////////////
//  Draw board
//////////////////////////////////////////////////////////////////////////////

void __declspec(naked) DrawBoard()
{
    __asm
    {
          mov          dl,byte ptr [board.x]
          dec          dl
          mov          dh,byte ptr [board.y]
          add          dh,byte ptr [board.lines]
          movzx        ecx,board.width
          add          ecx,2
          mov          ebx,0x7F20
          call         DrawHLine

          mov          dl,byte ptr [board.x]
          dec          dl
          mov          dh,byte ptr [board.y]
          mov          ecx,board.lines
          inc          ecx
          mov          ebx,0x7F20
          call         DrawVLine

          mov          dl, byte ptr [board.x]
          add          dl, byte ptr [board.width]
          mov          dh, byte ptr [board.y]
          mov          ecx, board.lines
          inc          ecx
          mov          ebx, 0x7F20
          call         DrawVLine

          retn
    }
} 

//////////////////////////////////////////////////////////////////////////////
//  GetRandomBlockID
//////////////////////////////////////////////////////////////////////////////
int __declspec(naked) GetRandomBlockID ()
{
    __asm
    {
          push        edx

          rdtsc                  // MUST be supported by CPU
                                 // TO DO: test for rdtsc support

          xor         edx,edx
          idiv        NumOfBlocks
          lea         eax,[4*edx]

          pop         edx
          retn
    }
/*
    __asm
    {
          push        edx
       
          xor         edx,edx
          mov         eax,seed
          idiv        q
          imul        eax,r
          imul        edx,a
          sub         edx,eax
          lea         eax,[edx+0x7FFFFFFF]
          test        edx,edx
          jle         _F01

          mov         eax,edx

_F01:
          mov         seed,eax

          xor         edx,edx
          idiv        NumOfBlocks
          lea         eax,[4*edx]

          pop         edx
          retn
    }
*/
}

//////////////////////////////////////////////////////////////////////////////
//  Draw block
//    stc if block cannot be drawn
//////////////////////////////////////////////////////////////////////////////

void __declspec(naked) DrawBlock()
{
    __asm
    {
          push         ebx
          push         ecx
          push         edx
          push         esi
          push         edi

          mov          ebx, WIDTH

          call         GetScreenPtr
          mov          edi,eax

          mov          eax,block.y
          imul         eax,ebx
          add          eax,block.x
          lea          edi,[2*eax+edi]     ; edi: upper left corner of blockwindow

          imul         esi,[block.id], 2*4*4 + 2
          add          esi,offset blocks   ; esi: upper left corner of blockdata

          mov          edx,esi
          mov          ecx,[block.lines]

loop_check:
          lodsd
          and          eax,[edi]
          and          eax,0x0F000F00
          jz           _F01

          stc
          jmp          _ret

_F01:  
          lodsd
          and          eax,[edi+4]
          and          eax,0x0F000F00
          jz           _F02

          stc
          jmp          short _ret

_F02:
          lea          edi,[edi+2*ebx]
          loop         loop_check

; now let's draw it
          imul         eax,[block.lines],2
          imul         eax,ebx
          sub          edi,eax             ; edi: upper left corner of blocwindow
          mov          esi,edx             ; esi: upper left corner of blockdata

          mov          ecx,[block.lines]

_loop_draw:
          lodsw
          test         ax,ax
          jz           _F03

          mov          [edi],ax

_F03:       
          lodsw
          test         ax,ax
          jz           _F04

          mov          [edi+2],ax

_F04:
          lodsw
          test         ax,ax
          jz           _F05

          mov          [edi+4],ax

_F05:
          lodsw
          test         ax,ax
          jz           _F06

          mov          [edi+6],ax

_F06:
          lea          edi,[edi+2*ebx]
          loop         _loop_draw

          call         si_UpdateScreen

          clc

_ret:
          pop          edi
          pop          esi
          pop          edx
          pop          ecx
          pop          ebx
          retn
    }
}
//////////////////////////////////////////////////////////////////////////////
//  Draw new block
//    stc if board is full
//////////////////////////////////////////////////////////////////////////////

void __declspec(naked) DrawNewBlock()
{
    __asm
    {
          mov          eax,board.width
          sub          eax,2
          shr          eax,1
          add          eax,board.x
          mov          block.x,eax

          mov          eax,board.y
          mov          block.y,eax

          call         GetRandomBlockID
          mov          [block.id],eax

          imul         eax, 4*4*2+2
          add          eax, offset blocks + 2*4*4
          movzx        eax, word ptr [eax]
          mov          block.lines,eax

          call         DrawBlock
          jc           full

          inc          stats.blocks
full:
          retn
    }
}
