
/*++
    Copyright (c) 2005 Godness
    Contact information:
        mail: godness@omen.ru

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Module Name: 
    compiler.cpp

Abstract: This module exported one function - Compile(), that compile given
    condition to executable code for the eXTreme tRaCer Engine

Revision History:

 Godness     01/10/2005
     Initial release

--*/

extern "C" {
#pragma warning ( push, 3 )
#include <ntddk.h>
#pragma warning ( pop )
}

#pragma warning ( disable: 4309 ) // "truncation of constant value"

struct str_compiled_condition
{
    str_compiled_condition *next;
    str_compiled_condition *prev;
    char    flags; // 0x1 - jump opcode was invert; 0x2 - use long jump;
    char    brackets_right;
    char    brackets_left;
    char    short_opcode_jump;
    char    short_address_jump;
    short   long_opcode_jump;
    int     long_address_jump;
    short   size_compiled_code;
    char    mas_compiled_code[1];
}; 

typedef str_compiled_condition *pcompiled_condition;

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

char stub_code[7] = {0x55, 0x89, 0xE5, 0x31, 0xC0, 0xC9, 0xC3};
                    // ^^^^^^^^^
                    // push  ebp
                    // mov   ebp, esp
                    // xor   eax, eax
                    // leave
                    // ret

char compiled_code[0x800] = {0x55, 0x89, 0xE5, 0x31, 0xC0, 0xC9, 0xC3};

char mas_temp_struct[0x800];
char mas_for_cmps[0x100];

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

char *mas_type_conv[] = 
{
    "byte", "word", "dword"
};

char *mas_registers[] = 
{
    "edi", "esi", "ebp", "esp", "ebx", "edx", "ecx", "eax", "eip", "cs", "eflags",
    "di", "si", "bp", "sp", "bx", "dx", "cx", "ax",
    "bl", "dl", "cl", "al",
    "bh", "dh", "ch", "ah"
};

char mas_reg_offset[] =
{
    0x08, 0x0C, 0x10, 0x34, 0x18, 0x1C, 0x20, 0x24, 0x28, 0x2C, 0x30,
    0x08, 0x0C, 0x10, 0x14, 0x18, 0x1C, 0x20, 0x24,
    0x18, 0x1C, 0x20, 0x24,
    0x19, 0x1D, 0x21, 0x25
};

char *mas_conditions_equal[] = 
{
    "==", "!=", ">=", "<=", ">", "<"
};

char byte_for_reverse_eflags = 0;

char mas_conditions_opcode[] =
{
    0x74, 0x75, 0x73, 0x76, 0x77, 0x72
};

char *mas_logic_conditions[] = 
{
    "&&", "||"
};

char mas_exit_command[] =
{
    0x31, 0xC0, 0xEB, 0x02, 0xB0, 0x01, 0xC9, 0xC3

    //^^^^^^^^^^^^^^^^
    //  xor   eax, eax
    //  jmp   exit
    //  mov   al, 0x01
    //exit:
    //  leave
    //  ret
};

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void CorrectJumpLength(pcompiled_condition pelem);
bool CompileConditionInBrackets(char *buffer, pcompiled_condition pmas);
bool CompileOneCondition(char *&buffer, pcompiled_condition mas);
char FindLogicCondition(char *&buffer);
bool FindEqualCondition(char *&buffer, char &condition_opcode);
bool FindRegister(char *&buffer, char &reg_offset, char &reg_size);
bool my_atoi(char *&buffer, ULONG& value);
bool FindPointerCondition(char *&buffer, int &pointer_level, int &type_convertion);
bool FindBinaryCondition(char *&buffer, int &out_type, int &out_size, char *out_massive);
bool si_atoi(PCSTR string, ULONG& value);

extern void* CheckAddressAndMakePageIn(void *address);

int OffsetOfMakePageInFunction = (int)(&CheckAddressAndMakePageIn);

////////////////////////////////////////////////////////////////////////////
//
// Compile() - function that compile given condition to execute code
// 
////////////////////////////////////////////////////////////////////////////

bool Compile(char *buffer)
{
    pcompiled_condition pmas_pointer = (pcompiled_condition)mas_temp_struct;

    char *mas = (char *)compiled_code;
    memset(mas, 0, sizeof(compiled_code));
    memcpy(mas, stub_code, sizeof(stub_code));
    mas += 3;

    if (!CompileConditionInBrackets(buffer, pmas_pointer))
        return false;

    CorrectJumpLength(pmas_pointer);

    do
    {
        memcpy(mas, pmas_pointer->mas_compiled_code, pmas_pointer->size_compiled_code);
        mas += pmas_pointer->size_compiled_code;

        if (pmas_pointer->flags & 0x2)
        {
            memcpy(mas, &pmas_pointer->long_opcode_jump, 6);
            mas += 6;
        }
        else
        {
            memcpy(mas, &pmas_pointer->short_opcode_jump, 2);
            mas += 2;
        }
    }
    while ((pmas_pointer = pmas_pointer->next) != 0);

    memcpy(mas, (char *)&mas_exit_command, sizeof(mas_exit_command));
    return true;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void CorrectJumpLength(pcompiled_condition pelem)
{
    pcompiled_condition ptemp   = NULL;
    ULONG size                  = 0;
    char level                  = 0;

    while (pelem->next)
        pelem = pelem->next;

    do
    {
        ptemp = pelem;
        level = 0;
        size = 0;

        while ((ptemp = ptemp->next) != 0)
        {
            size += ptemp->size_compiled_code + ((ptemp->flags & 0x2) ? 6 : 2);

            if (ptemp->next)
            {
                if (ptemp->brackets_left)
                    level = level + ptemp->brackets_left;

                if (level && ptemp->brackets_right)
                {
                    int i = ptemp->brackets_right;

                    while (level && i)
                    {
                        --level;
                        --i;
                    }
                }
            }
            else
                level = 0;

            if (pelem->flags & 0x1)
            {
                if ((!ptemp->next) || (!level && !(ptemp->flags & 0x1)))
                    break;
            }
            else
            {
                if (!ptemp->next)
                {
                    size += 4;
                    break;
                }

                if (!level && (ptemp->flags & 0x1) && ptemp->brackets_right)
                    break;
            }
        }

        if (size == 0) size = 4;

        if (size > 0x7F)
        {
            pelem->flags |= 0x2;
            pelem->long_opcode_jump = ( ((pelem->short_opcode_jump & 0x0F) | 0x80) << 8 ) + 0x0F;
            pelem->long_address_jump = size;
        }
        else
        {
            pelem->short_address_jump = (char)size;
        }  
    }
    while ((pelem = pelem->prev) != 0);

    return;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool CompileConditionInBrackets(char *pbuffer, pcompiled_condition pmas)
{
    pcompiled_condition prev    = NULL;
    char level                  = 0;
    char count                  = 0;

    memset(pmas, 0, sizeof(str_compiled_condition));

    while (pbuffer[0] == ' ') ++pbuffer;
    if (!pbuffer[0]) return false;

    while (pbuffer[0])
    {
        level = 0;
        while (pbuffer[0] == ' ') ++pbuffer;
        while (pbuffer[0] == '(')
        {
            ++count;
            ++level;
            ++pbuffer;
            while (pbuffer[0] == ' ') ++pbuffer;
        }
        pmas->brackets_left = level;

        if (!CompileOneCondition(pbuffer, pmas))
            return false;

        level = 0;
        while (pbuffer[0] == ' ') ++pbuffer;
        while (pbuffer[0] == ')')
        {
            --count;
            ++level;
            ++pbuffer;
            while (pbuffer[0] == ' ') ++pbuffer;
        }
        pmas->brackets_right = level;

        switch (FindLogicCondition(pbuffer))
        {
        case 1:
            {
                if (pmas->short_opcode_jump & 1)
                    --pmas->short_opcode_jump;
                else
                    ++pmas->short_opcode_jump;

                pmas->flags |= 0x1;
                break;
            }

        case -1:
            return false;
        }

        if (prev)
        {
            pmas->prev = prev;
            pmas->prev->next = pmas;
        }

        prev = pmas;
        pmas = (pcompiled_condition)((char *)pmas + pmas->size_compiled_code + sizeof(str_compiled_condition) - 1);
        memset(pmas, 0, sizeof(str_compiled_condition));
    }

    if (count) 
        return false;

    return true;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool CompileOneCondition(char *&buffer, pcompiled_condition pmas)
{
    char *buf               = buffer;
    ULONG value             = 0;

    char condition_opcode   = 0;
    short count             = 0;
    int i;

    int type_convertion_fir = 0;
    int pointer_level_fir   = 0;
    char reg_offset_fir     = 0;
    char reg_size_fir       = 0;
    int out_type_fir        = 0;
    int out_size_fir        = 0;
    char mas_fir[0x100];

    int type_convertion_sec = 0;
    int pointer_level_sec   = 0;
    char reg_offset_sec     = 0;
    char reg_size_sec       = 0;
    int out_type_sec        = 0;
    int out_size_sec        = 0;
    char mas_sec[0x100];

    for (i = 0; i <= 0x100; i++) 
        mas_fir[i] = mas_sec[i] = 0;

    int n1 = (int)FindPointerCondition(buf, pointer_level_fir, type_convertion_fir);
    int n2 = (int)FindRegister(buf, reg_offset_fir, reg_size_fir);
    int n3 = (int)FindBinaryCondition(buf, out_type_fir, out_size_fir, mas_fir);

    int n4 = (int)FindEqualCondition(buf, condition_opcode);

    int n5 = (int)FindPointerCondition(buf, pointer_level_sec, type_convertion_sec);
    int n6 = (int)FindRegister(buf, reg_offset_sec, reg_size_sec);
    int n7 = (int)FindBinaryCondition(buf, out_type_sec, out_size_sec, mas_sec);

    switch (condition_opcode)
    {
    case 0x74: case 0x72: case 0x76: 
        byte_for_reverse_eflags = 0x00;
        break;

    case 0x75:
        byte_for_reverse_eflags = 0x40;
        break;

    case 0x73: 
        byte_for_reverse_eflags = 0x01;
        break;

    case 0x77:
        byte_for_reverse_eflags = 0x41;
    }

    //DbgPrint("%X, %X, %X\n", condition_opcode, byte_for_reverse_eflags, &byte_for_reverse_eflags);
    
    int res = (n1 << 6*4) + (n2 << 5*4) + (n3 << 4*4) + (n4 << 3*4) + (n5 << 2*4) + (n6 << 1*4) + (n7 << 0*4);

    switch (res)
    {
        /////////////////////////////////////////////////////////////////////////////////
        // in case     eax == edx
        /////////////////////////////////////////////////////////////////////////////////

        case 0x0101010:

            if (reg_size_fir < reg_size_sec)
                return false;

            switch (reg_size_sec)
            {
            case 4:
                {
                    pmas->mas_compiled_code[count++] = 0x8B;
                    pmas->mas_compiled_code[count++] = 0x45;
                    pmas->mas_compiled_code[count++] = reg_offset_fir;
                    pmas->mas_compiled_code[count++] = 0x3B;
                    pmas->mas_compiled_code[count++] = 0x45;
                    pmas->mas_compiled_code[count++] = reg_offset_sec;
                    break;
                }

            case 2:
                {
                    pmas->mas_compiled_code[count++] = 0x66;
                    pmas->mas_compiled_code[count++] = 0x8B;
                    pmas->mas_compiled_code[count++] = 0x45;
                    pmas->mas_compiled_code[count++] = reg_offset_fir;
                    pmas->mas_compiled_code[count++] = 0x66;
                    pmas->mas_compiled_code[count++] = 0x3B;
                    pmas->mas_compiled_code[count++] = 0x45;
                    pmas->mas_compiled_code[count++] = reg_offset_sec;
                    break;
                }

            case 1:
                {
                    pmas->mas_compiled_code[count++] = 0x8A;
                    pmas->mas_compiled_code[count++] = 0x45;
                    pmas->mas_compiled_code[count++] = reg_offset_fir;
                    pmas->mas_compiled_code[count++] = 0x3A;
                    pmas->mas_compiled_code[count++] = 0x45;
                    pmas->mas_compiled_code[count++] = reg_offset_sec;
                }
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     eax == 0x00400000
        /////////////////////////////////////////////////////////////////////////////////

        case 0x0101001:
            
            if (out_type_sec) return false;
            value = *(PULONG)mas_sec;

            switch (reg_size_fir)
            {

            case 4: //if register = dword
                {
                    if (value == 0 || value == -1)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;
                        pmas->mas_compiled_code[count++] = 0x7D;
                        pmas->mas_compiled_code[count++] = reg_offset_fir;
                        pmas->mas_compiled_code[count++] = (char)value;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x81;
                        pmas->mas_compiled_code[count++] = 0x7D;
                        pmas->mas_compiled_code[count++] = reg_offset_fir;
                        *(int *)(pmas->mas_compiled_code + count) = value;
                        count += sizeof(int);
                    }

                    break;
                }

            case 2: //if register = word
                {
                    if ((value & 0xFFFF) == 0 || (value & 0xFFFF) == -1)
                    {
                        pmas->mas_compiled_code[count++] = 0x66;
                        pmas->mas_compiled_code[count++] = 0x83;
                        pmas->mas_compiled_code[count++] = 0x7D;
                        pmas->mas_compiled_code[count++] = reg_offset_fir;
                        pmas->mas_compiled_code[count++] = (char)value;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x66;                    
                        pmas->mas_compiled_code[count++] = 0x81;
                        pmas->mas_compiled_code[count++] = 0x7D;
                        pmas->mas_compiled_code[count++] = reg_offset_fir;
                        *(short *)(pmas->mas_compiled_code + count) = (short)value;
                        count += sizeof(short);
                    }

                    break;
                }

            case 1: //if register = byte
                {
                    pmas->mas_compiled_code[count++] = 0x80;
                    pmas->mas_compiled_code[count++] = 0x7D;
                    pmas->mas_compiled_code[count++] = reg_offset_fir;
                    pmas->mas_compiled_code[count++] = (char)value;
                }
            }
            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     eax == *edx  or  *eax == edx
        /////////////////////////////////////////////////////////////////////////////////

        case 0x0101110: case 0x1101010:
            
            char offset_fir;
            char offset_sec;
            char size_fir;
            char size_sec;
            
            int type_conv;
            int level;
            int cmp_exit;
            int tmp;

            if (res & 0x1000000)
            {
                level = pointer_level_fir;
                type_conv = type_convertion_fir;
                offset_fir = reg_offset_fir;
                offset_sec = reg_offset_sec;
                size_fir = reg_size_fir;
                size_sec = reg_size_sec;
            }
            else
            {
                level = pointer_level_sec;
                type_conv = type_convertion_sec;
                offset_fir = reg_offset_sec;
                offset_sec = reg_offset_fir;
                size_fir = reg_size_sec;
                size_sec = reg_size_fir;
            }

            if ( (size_fir != 4) || (type_conv == 1 && size_sec > 1) || (type_conv == 2 && size_sec > 2) )
                return false;

            if (type_conv == 3 || type_conv == 0) 
                type_conv = 4;

            cmp_exit = 5; // if *byte == byte
            if (type_conv == 4) 
            {
                if (size_sec == 4) cmp_exit = 0; // if *dword == dword
                else if (size_sec == 2) cmp_exit = 1; // if *dword == word
                else if (size_sec == 1) cmp_exit = 2; // if *dword == byte
            }
            else if (type_conv == 2)
            {
                if (size_sec == 2) cmp_exit = 3; // if *word == word
                else if (size_sec == 1) cmp_exit = 4; // if *word == byte
            }

            if (level > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = level;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, [ebp + offset]
            pmas->mas_compiled_code[count++] = 0x45;
            pmas->mas_compiled_code[count++] = offset_fir;

            if (level > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf

            if (level > 1 || cmp_exit != 5)
            {
                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +1E
                if ((type_conv == 2 || type_conv == 1) && level > 1)
                {
                    if (type_conv == 2)
                        tmp = 0xD;
                    else
                        tmp = 0x5;
                }
                else if (level > 1)
                    tmp = 0x3;
                else
                    tmp = 0x0;

                if (cmp_exit == 0)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x05 + (UCHAR)tmp;
                else if (cmp_exit == 1)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x0A + (UCHAR)tmp;
                else if (cmp_exit == 2)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x09 + (UCHAR)tmp;
                else if (cmp_exit == 3)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x07 + (UCHAR)tmp;
                else if (cmp_exit == 4)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x0B + (UCHAR)tmp;
                else
                    pmas->mas_compiled_code[count++] = 0x16 + 0x08 + (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_conv == 2 || type_conv == 1) && level > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;

                    if (type_conv == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_conv == 2 && level == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;            

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (type_conv == 1 && level > 1)
                    tmp = 0xE2;
                else if (type_conv == 2 && level > 1)
                    tmp = 0xDD;
                else
                    tmp = 0xE7;

                pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (level > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x49;    // dec ecx

                    if (type_conv == 1)
                        tmp = 0xD0;
                    else if (type_conv == 2)
                        tmp = 0xCB;
                    else
                        tmp = 0xD5;

                    pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                    pmas->mas_compiled_code[count++] = UCHAR(tmp);
                }
            }
            else
            {
                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +
                if (cmp_exit == 0)
                    pmas->mas_compiled_code[count++] = 0x05;
                else if (cmp_exit == 1)
                    pmas->mas_compiled_code[count++] = 0x0A;
                else if (cmp_exit == 2)
                    pmas->mas_compiled_code[count++] = 0x09;
                else if (cmp_exit == 3)
                    pmas->mas_compiled_code[count++] = 0x07;
                else if (cmp_exit == 4)
                    pmas->mas_compiled_code[count++] = 0x0B;
                else
                    pmas->mas_compiled_code[count++] = 0x05;
            }

            switch (cmp_exit)
            {
            case 0: // if dword == dword
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, dword [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = offset_sec;
                break;

            case 1: // if dword == word
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x33;    // xor edx, edx
                pmas->mas_compiled_code[count++] = 0xD2;
                pmas->mas_compiled_code[count++] = 0x66;    // mov dx, word ptr [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x55;
                pmas->mas_compiled_code[count++] = offset_sec;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, edx
                pmas->mas_compiled_code[count++] = 0xC2;
                break;

            case 2: // if dword == byte
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x33;    // xor edx, edx
                pmas->mas_compiled_code[count++] = 0xD2;
                pmas->mas_compiled_code[count++] = 0x8A;    // mov dl, word ptr [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x55;
                pmas->mas_compiled_code[count++] = offset_sec;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, edx
                pmas->mas_compiled_code[count++] = 0xC2;
                break;

            case 3: // if word == word
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x66;    // cmp ax, word [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x3B;
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = offset_sec;
                break;

            case 4: // if word == byte
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x33;    // xor edx, edx
                pmas->mas_compiled_code[count++] = 0xD2;
                pmas->mas_compiled_code[count++] = 0x8A;    // mov dl, word ptr [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x55;
                pmas->mas_compiled_code[count++] = offset_sec;
                pmas->mas_compiled_code[count++] = 0x66;    // cmp ax, dx
                pmas->mas_compiled_code[count++] = 0x3B;
                pmas->mas_compiled_code[count++] = 0xC2;
                break;

            case 5: // if byte == byte
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x3A;    // cmp al, byte [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = offset_sec;
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     *eax == *edx
        /////////////////////////////////////////////////////////////////////////////////

        case 0x1101110:
            
            if (type_convertion_fir == 0 || type_convertion_fir == 3)
                type_convertion_fir = 4;
            
            if (type_convertion_sec == 0 || type_convertion_sec == 3)
                type_convertion_sec = 4;

            if (type_convertion_fir != type_convertion_sec)
                return false;

            if (reg_size_fir != 4 || reg_size_sec != 4)
                return false;

            if (pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, [ebp + offset]
            pmas->mas_compiled_code[count++] = 0x45;
            pmas->mas_compiled_code[count++] = reg_offset_fir;

            if (pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jne +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf

            tmp = 0;
            switch (type_convertion_fir)
            {
            case 4:
                (pointer_level_fir > 1) ? tmp = 0x2E : tmp = 0x2B;
                break;

            case 2:
                (pointer_level_fir > 1) ? tmp = 0x39 : tmp = 0x2C;
                break;

            case 1:
                (pointer_level_fir > 1) ? tmp = 0x33 : tmp = 0x15;
            }

            if (pointer_level_sec > 1) tmp += 9;

            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
            pmas->mas_compiled_code[count++] = (UCHAR)tmp;

            if (type_convertion_fir != 1 || pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_convertion_fir == 2 || type_convertion_fir == 1) && pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;                

                    if (type_convertion_fir == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_convertion_fir == 2 && pointer_level_fir == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (pointer_level_fir > 1)
                {
                    switch (type_convertion_fir)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                        pmas->mas_compiled_code[count++] = 0xDD;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                        pmas->mas_compiled_code[count++] = 0xE2;
                    }
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                    pmas->mas_compiled_code[count++] = 0xE7;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_fir > 1)
                {
                    switch (type_convertion_fir)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                        pmas->mas_compiled_code[count++] = 0xD5;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                        pmas->mas_compiled_code[count++] = 0xCB;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                        pmas->mas_compiled_code[count++] = 0xD0;
                    }
                }
            }

            switch (type_convertion_fir)
            {
            case 4:
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 1:
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov edi, eax
            pmas->mas_compiled_code[count++] = 0xF8;

            if (pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = pointer_level_sec;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, [ebp + offset]
            pmas->mas_compiled_code[count++] = 0x45;
            pmas->mas_compiled_code[count++] = reg_offset_sec;

            if (pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jne +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf

            tmp = 0;
            switch (type_convertion_sec)
            {
            case 4:
                (pointer_level_sec > 1) ? tmp = 0x1F : tmp = 0x1C;
                break;

            case 2:
                (pointer_level_sec > 1) ? tmp = 0x2B : tmp = 0x1E;
                break;

            case 1:
                (pointer_level_sec > 1) ? tmp = 0x24 : tmp = 0x06;
            }

            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +1E
            pmas->mas_compiled_code[count++] = (UCHAR)tmp;

            if (type_convertion_sec != 1 || pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_convertion_sec == 2 || type_convertion_sec == 1) && pointer_level_sec > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;                

                    if (type_convertion_sec == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_convertion_sec == 2 && pointer_level_sec == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (pointer_level_sec > 1)
                {
                    switch (type_convertion_sec)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                        pmas->mas_compiled_code[count++] = 0xDD;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                        pmas->mas_compiled_code[count++] = 0xE2;
                    }
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                    pmas->mas_compiled_code[count++] = 0xE7;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_sec > 1)
                {
                    switch (type_convertion_sec)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                        pmas->mas_compiled_code[count++] = 0xD5;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                        pmas->mas_compiled_code[count++] = 0xCB;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                        pmas->mas_compiled_code[count++] = 0xD0;
                    }
                }
            }

            switch (type_convertion_sec)
            {
            case 4:
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 1:
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, edi
            pmas->mas_compiled_code[count++] = 0xDF;

            switch (type_convertion_fir)
            {
            case 1:
                pmas->mas_compiled_code[count++] = 0x3A;    // cmp al, bl
                pmas->mas_compiled_code[count++] = 0xC3;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp ax, bx
                pmas->mas_compiled_code[count++] = 0xC3;
                break;

            case 4:
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     *eax == 0x00400000  or  *eax == 'some_string'
        /////////////////////////////////////////////////////////////////////////////////

        case 0x1101001:

            if (out_type_sec)
            {
                if (type_convertion_fir || reg_size_fir != 4)
                    return false;

                if (condition_opcode != 0x74 && condition_opcode != 0x75)
                    return false;

                memset(mas_for_cmps, 0, sizeof(mas_for_cmps));
                memcpy(mas_for_cmps, mas_sec, out_size_sec);

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                    *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                    count += 4;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = reg_offset_fir;
                
                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                    pmas->mas_compiled_code[count++] = 0x02;
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                }

                pmas->mas_compiled_code[count++] = 0x50;    // push eax

                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                pmas->mas_compiled_code[count++] = 0x75;    // jne +5
                pmas->mas_compiled_code[count++] = 0x05;

                pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
                pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
                pmas->mas_compiled_code[count++] = 0x9E;    // sahf

                (pointer_level_fir > 1) ? tmp = 0x34 : tmp = 0x27;
                if ((ULONG)out_size_sec > 0x7F) tmp += 0x3;

                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;                

                    if ((ULONG)out_size_sec < 0x80)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = (UCHAR)out_size_sec - 1;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +8
                        pmas->mas_compiled_code[count++] = 0x08;

                        pmas->mas_compiled_code[count++] = 0x81;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        *(ULONG *)(pmas->mas_compiled_code + count) = (ULONG)out_size_sec - 1;
                        count += 4;
                    }

                    pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                    pmas->mas_compiled_code[count++] = 0x03;

                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }
                else
                {
                    if ((ULONG)out_size_sec < 0x80)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = (UCHAR)out_size_sec - 1;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x81;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        *(ULONG *)(pmas->mas_compiled_code + count) = (ULONG)out_size_sec - 1;
                        count += 4;
                    }
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                (pointer_level_fir > 1) ? tmp = 0xDD : tmp = 0xE7;
                if ((ULONG)out_size_sec > 0x7F) tmp += 0x3;

                pmas->mas_compiled_code[count++] = 0x74;    // jz -tmp
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_fir > 1)
                {
                    tmp = 0xCB;
                    if ((ULONG)out_size_sec > 0x7F) tmp += 0x3;

                    pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                    pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                    pmas->mas_compiled_code[count++] = (UCHAR)tmp;
                }
                
                pmas->mas_compiled_code[count++] = 0xFC;    // cld
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, out_size_fir
                *(int *)(pmas->mas_compiled_code + count) = out_size_sec;
                count += 4;
                pmas->mas_compiled_code[count++] = 0xBE;    // mov esi, offset mas_for_cmps
                *(int *)(pmas->mas_compiled_code + count) = (int)mas_for_cmps;
                count += 4;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov edi, eax
                pmas->mas_compiled_code[count++] = 0xF8;

                pmas->mas_compiled_code[count++] = 0xF3;    // repz cmpsb
                pmas->mas_compiled_code[count++] = 0xA6;

                pmas->mas_compiled_code[count++] = 0x09;    // or ecx, ecx
                pmas->mas_compiled_code[count++] = 0xC9;
            }
            else
            {
                if (reg_size_fir != 4) return false;
                value = *(PULONG)mas_sec;

                if (type_convertion_fir == 0 || type_convertion_fir == 3)
                    type_convertion_fir = 4;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                    *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                    count += 4;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = reg_offset_fir;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                    pmas->mas_compiled_code[count++] = 0x02;
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                }

                pmas->mas_compiled_code[count++] = 0x50;    // push eax

                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                pmas->mas_compiled_code[count++] = 0x75;    // jne +5
                pmas->mas_compiled_code[count++] = 0x05;

                pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
                pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
                pmas->mas_compiled_code[count++] = 0x9E;    // sahf

                tmp = 0;
                switch (type_convertion_fir)
                {
                case 4:
                    (pointer_level_fir > 1) ? tmp = 0x20 : tmp = 0x1D;
                    break;

                case 2:
                    (pointer_level_fir > 1) ? tmp = 0x2A : tmp = 0x1D;
                    break;

                case 1:
                    (pointer_level_fir > 1) ? tmp = 0x22 : tmp = 0x04;
                }

                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                if (type_convertion_fir != 1 || pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                    pmas->mas_compiled_code[count++] = 0xD8;

                    pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                    pmas->mas_compiled_code[count++] = 0xD0;

                    if ((type_convertion_fir == 2 || type_convertion_fir == 1) && pointer_level_fir > 1)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                        pmas->mas_compiled_code[count++] = 0xF9;
                        pmas->mas_compiled_code[count++] = 0x01;                

                        if (type_convertion_fir == 2)
                        {
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                            pmas->mas_compiled_code[count++] = 0x05;

                            pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                            pmas->mas_compiled_code[count++] = 0xC2;
                            pmas->mas_compiled_code[count++] = 0x01;

                            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                            pmas->mas_compiled_code[count++] = 0x03;
                        }
                        else
                        {
                            pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                            pmas->mas_compiled_code[count++] = 0x12;
                        }
                    }

                    if (type_convertion_fir == 2 && pointer_level_fir == 1)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x03;
                    }

                    pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                    pmas->mas_compiled_code[count++] = 0x0D;

                    pmas->mas_compiled_code[count++] = 0x52;    // push edx
                    pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                    pmas->mas_compiled_code[count++] = 0x15;
                    *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                    count += 4;

                    pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                    pmas->mas_compiled_code[count++] = 0xC0;

                    if (pointer_level_fir > 1)
                    {
                        switch (type_convertion_fir)
                        {
                        case 4:
                            pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                            pmas->mas_compiled_code[count++] = 0xE7;
                            break;

                        case 2:
                            pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                            pmas->mas_compiled_code[count++] = 0xDD;
                            break;

                        case 1:
                            pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                            pmas->mas_compiled_code[count++] = 0xE2;
                        }
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                    }

                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                    pmas->mas_compiled_code[count++] = 0xC3;

                    if (pointer_level_fir > 1)
                    {
                        switch (type_convertion_fir)
                        {
                        case 4:
                            pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                            pmas->mas_compiled_code[count++] = 0xD5;
                            break;

                        case 2:
                            pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                            pmas->mas_compiled_code[count++] = 0xCB;
                            break;

                        case 1:
                            pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                            pmas->mas_compiled_code[count++] = 0xD0;
                        }
                    }
                }

                switch (type_convertion_fir)
                {
                case 4:
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                    pmas->mas_compiled_code[count++] = 0x3D;    // cmp eax, value
                    *(int *)(pmas->mas_compiled_code + count) = (int)value;
                    count += 4;
                    break;

                case 2:
                    pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x8B;
                    pmas->mas_compiled_code[count++] = 0x00;
                    pmas->mas_compiled_code[count++] = 0x66;    // cmp ax, value
                    pmas->mas_compiled_code[count++] = 0x3D;
                    *(short *)(pmas->mas_compiled_code + count) = (short)value;
                    count += 2;
                    break;

                case 1:
                    pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                    pmas->mas_compiled_code[count++] = 0x3C;    // cmp al, value
                    pmas->mas_compiled_code[count++] = (char)value;
                }
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     eax == *(0x00400000)  or  *(0x00400000) == eax
        /////////////////////////////////////////////////////////////////////////////////

        case 0x0101101: case 0x1011010:

            int out_type;

            if (res & 0x100000)
            {
                value = *(PULONG)mas_sec;
                type_conv = type_convertion_sec;
                level = pointer_level_sec;
                offset_sec = reg_offset_fir;
                out_type = out_type_sec;
                size_sec = reg_size_fir;
            }
            else
            {
                value = *(PULONG)mas_fir;
                type_conv = type_convertion_fir;
                level = pointer_level_fir;
                offset_sec = reg_offset_sec;
                out_type = out_type_fir;
                size_sec = reg_size_sec;
            }

            if ((type_conv == 1 && size_sec > 1) || (type_conv == 2 && size_sec > 2))
                return false;

            if (type_conv == 3 || type_conv == 0) 
                type_conv = 4;

            cmp_exit = 5; // if *byte == byte
            if (type_conv == 4) 
            {
                if (size_sec == 4) cmp_exit = 0; // if *dword == dword
                else if (size_sec == 2) cmp_exit = 1; // if *dword == word
                else if (size_sec == 1) cmp_exit = 2; // if *dword == byte
            }
            else if (type_conv == 2)
            {
                if (size_sec == 2) cmp_exit = 3; // if *word == word
                else if (size_sec == 1) cmp_exit = 4; // if *word == byte
            }

            if (level > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = level;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0xB8;    // mov eax, value
            *(int *)(pmas->mas_compiled_code + count) = (int)value;
            count += 4;

            if (level > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf

            if (level > 1 || cmp_exit != 5)
            {
                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +1E
                if ((type_conv == 2 || type_conv == 1) && level > 1)
                {
                    if (type_conv == 2)
                        tmp = 0xD;
                    else
                        tmp = 0x5;
                }
                else if (level > 1)
                    tmp = 0x3;
                else
                    tmp = 0x0;

                if (cmp_exit == 0)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x05 + (UCHAR)tmp;
                else if (cmp_exit == 1)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x0A + (UCHAR)tmp;
                else if (cmp_exit == 2)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x09 + (UCHAR)tmp;
                else if (cmp_exit == 3)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x07 + (UCHAR)tmp;
                else if (cmp_exit == 4)
                    pmas->mas_compiled_code[count++] = 0x16 + 0x0B + (UCHAR)tmp;
                else
                    pmas->mas_compiled_code[count++] = 0x16 + 0x08 + (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_conv == 2 || type_conv == 1) && level > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;

                    if (type_conv == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_conv == 2 && level == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;            

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (type_conv == 1 && level > 1)
                    tmp = 0xE2;
                else if (type_conv == 2 && level > 1)
                    tmp = 0xDD;
                else
                    tmp = 0xE7;

                pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (level > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x49;    // dec ecx

                    if (type_conv == 1)
                        tmp = 0xD0;
                    else if (type_conv == 2)
                        tmp = 0xCB;
                    else
                        tmp = 0xD5;

                    pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                    pmas->mas_compiled_code[count++] = UCHAR(tmp);
                }
            }
            else
            {
                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +
                if (cmp_exit == 0)
                    pmas->mas_compiled_code[count++] = 0x05;
                else if (cmp_exit == 1)
                    pmas->mas_compiled_code[count++] = 0x0A;
                else if (cmp_exit == 2)
                    pmas->mas_compiled_code[count++] = 0x09;
                else if (cmp_exit == 3)
                    pmas->mas_compiled_code[count++] = 0x07;
                else if (cmp_exit == 4)
                    pmas->mas_compiled_code[count++] = 0x0B;
                else
                    pmas->mas_compiled_code[count++] = 0x05;
            }

            switch (cmp_exit)
            {
            case 0: // if dword == dword
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, dword [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = offset_sec;
                break;

            case 1: // if dword == word
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x33;    // xor edx, edx
                pmas->mas_compiled_code[count++] = 0xD2;
                pmas->mas_compiled_code[count++] = 0x66;    // mov dx, word ptr [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x55;
                pmas->mas_compiled_code[count++] = offset_sec;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, edx
                pmas->mas_compiled_code[count++] = 0xC2;
                break;

            case 2: // if dword == byte
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x33;    // xor edx, edx
                pmas->mas_compiled_code[count++] = 0xD2;
                pmas->mas_compiled_code[count++] = 0x8A;    // mov dl, word ptr [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x55;
                pmas->mas_compiled_code[count++] = offset_sec;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, edx
                pmas->mas_compiled_code[count++] = 0xC2;
                break;

            case 3: // if word == word
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x66;    // cmp ax, word [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x3B;
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = offset_sec;
                break;

            case 4: // if word == byte
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x33;    // xor edx, edx
                pmas->mas_compiled_code[count++] = 0xD2;
                pmas->mas_compiled_code[count++] = 0x8A;    // mov dl, word ptr [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x55;
                pmas->mas_compiled_code[count++] = offset_sec;
                pmas->mas_compiled_code[count++] = 0x66;    // cmp ax, dx
                pmas->mas_compiled_code[count++] = 0x3B;
                pmas->mas_compiled_code[count++] = 0xC2;
                break;

            case 5: // if byte == byte
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                pmas->mas_compiled_code[count++] = 0x3A;    // cmp al, byte [ebp + offset]
                pmas->mas_compiled_code[count++] = 0x45;
                pmas->mas_compiled_code[count++] = offset_sec;
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     *eax == *(0x00400000)  or  *(0x00400000) == *eax
        /////////////////////////////////////////////////////////////////////////////////

        case 0x1101101: case 0x1011110:

            if (res & 0x100000)
            {
                int i = pointer_level_sec;
                
                pointer_level_sec = pointer_level_fir;
                reg_offset_sec = reg_offset_fir;
                reg_size_sec = reg_size_fir;

                pointer_level_fir = i;
                reg_size_fir = 4;

                value = *(PULONG)mas_sec;
                out_type = out_type_sec;
            }
            else
            {
                reg_size_fir = 4;
                value = *(PULONG)mas_fir;
                out_type = out_type_fir;
            }

            if (reg_size_sec != 4 || out_type)
                return false;

            if (type_convertion_fir == 0 || type_convertion_fir == 3)
                type_convertion_fir = 4;
            
            if (type_convertion_sec == 0 || type_convertion_sec == 3)
                type_convertion_sec = 4;

            if (type_convertion_fir != type_convertion_sec)
                return false;

            if (pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0xB8;    // mov eax, value
            *(int *)(pmas->mas_compiled_code + count) = (int)value;
            count += 4;

            if (pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jne +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf 

            tmp = 0;
            switch (type_convertion_fir)
            {
            case 4:
                (pointer_level_fir > 1) ? tmp = 0x2E : tmp = 0x2B;
                break;

            case 2:
                (pointer_level_fir > 1) ? tmp = 0x39 : tmp = 0x2C;
                break;

            case 1:
                (pointer_level_fir > 1) ? tmp = 0x33 : tmp = 0x15;
            }

            if (pointer_level_sec > 1) tmp += 9;

            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
            pmas->mas_compiled_code[count++] = (UCHAR)tmp;

            if (type_convertion_fir != 1 || pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_convertion_fir == 2 || type_convertion_fir == 1) && pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;

                    if (type_convertion_fir == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_convertion_fir == 2 && pointer_level_fir == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (pointer_level_fir > 1)
                {
                    switch (type_convertion_fir)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                        pmas->mas_compiled_code[count++] = 0xDD;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                        pmas->mas_compiled_code[count++] = 0xE2;
                    }
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                    pmas->mas_compiled_code[count++] = 0xE7;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_fir > 1)
                {
                    switch (type_convertion_fir)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                        pmas->mas_compiled_code[count++] = 0xD5;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                        pmas->mas_compiled_code[count++] = 0xCB;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                        pmas->mas_compiled_code[count++] = 0xD0;
                    }
                }
            }

            switch (type_convertion_fir)
            {
            case 4:
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 1:
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov edi, eax
            pmas->mas_compiled_code[count++] = 0xF8;

            if (pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = pointer_level_sec;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, [ebp + offset]
            pmas->mas_compiled_code[count++] = 0x45;
            pmas->mas_compiled_code[count++] = reg_offset_sec;

            if (pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jne +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf

            tmp = 0;
            switch (type_convertion_sec)
            {
            case 4:
                (pointer_level_sec > 1) ? tmp = 0x1F : tmp = 0x1C;
                break;

            case 2:
                (pointer_level_sec > 1) ? tmp = 0x2B : tmp = 0x1E;
                break;

            case 1:
                (pointer_level_sec > 1) ? tmp = 0x24 : tmp = 0x06;
            }

            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +1E
            pmas->mas_compiled_code[count++] = (UCHAR)tmp;

            if (type_convertion_sec != 1 || pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_convertion_sec == 2 || type_convertion_sec == 1) && pointer_level_sec > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;                

                    if (type_convertion_sec == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_convertion_sec == 2 && pointer_level_sec == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (pointer_level_sec > 1)
                {
                    switch (type_convertion_sec)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                        pmas->mas_compiled_code[count++] = 0xDD;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                        pmas->mas_compiled_code[count++] = 0xE2;
                    }
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                    pmas->mas_compiled_code[count++] = 0xE7;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_sec > 1)
                {
                    switch (type_convertion_sec)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                        pmas->mas_compiled_code[count++] = 0xD5;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                        pmas->mas_compiled_code[count++] = 0xCB;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                        pmas->mas_compiled_code[count++] = 0xD0;
                    }
                }
            }

            switch (type_convertion_sec)
            {
            case 4:
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 1:
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, edi
            pmas->mas_compiled_code[count++] = 0xDF;

            switch (type_convertion_fir)
            {
            case 1:
                pmas->mas_compiled_code[count++] = 0x3A;    // cmp al, bl
                pmas->mas_compiled_code[count++] = 0xC3;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp ax, bx
                pmas->mas_compiled_code[count++] = 0xC3;
                break;

            case 4:
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     *(0x00400000) == 0x00400000  or  *(0x00400000) == 'some_string'
        /////////////////////////////////////////////////////////////////////////////////

        case 0x1011001:

            if (out_type_sec)
            {
                if (type_convertion_fir || out_type_fir)
                    return false;

                if (condition_opcode != 0x74 && condition_opcode != 0x75)
                    return false;

                memset(mas_for_cmps, 0, sizeof(mas_for_cmps));
                memcpy(mas_for_cmps, mas_sec, out_size_sec);

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                    *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                    count += 4;
                }

                pmas->mas_compiled_code[count++] = 0xB8;    // mov eax, value
                *(int *)(pmas->mas_compiled_code + count) = *(int *)mas_fir;
                count += 4;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                    pmas->mas_compiled_code[count++] = 0x02;
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                }

                pmas->mas_compiled_code[count++] = 0x50;    // push eax

                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                pmas->mas_compiled_code[count++] = 0x75;    // jne +5
                pmas->mas_compiled_code[count++] = 0x05;

                pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
                pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
                pmas->mas_compiled_code[count++] = 0x9E;    // sahf

                (pointer_level_fir > 1) ? tmp = 0x34 : tmp = 0x27;
                if ((ULONG)out_size_sec > 0x7F) tmp += 0x3;

                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;                

                    if ((ULONG)out_size_sec < 0x80)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = (UCHAR)out_size_sec - 1;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +8
                        pmas->mas_compiled_code[count++] = 0x08;

                        pmas->mas_compiled_code[count++] = 0x81;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        *(ULONG *)(pmas->mas_compiled_code + count) = (ULONG)out_size_sec - 1;
                        count += 4;
                    }

                    pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                    pmas->mas_compiled_code[count++] = 0x03;

                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }
                else
                {
                    if ((ULONG)out_size_sec < 0x80)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = (UCHAR)out_size_sec - 1;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x81;    // add edx, out_size_sec - 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        *(ULONG *)(pmas->mas_compiled_code + count) = (ULONG)out_size_sec - 1;
                        count += 4;
                    }
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                (pointer_level_fir > 1) ? tmp = 0xDD : tmp = 0xE7;
                if ((ULONG)out_size_sec > 0x7F) tmp += 0x3;

                pmas->mas_compiled_code[count++] = 0x74;    // jz -tmp
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_fir > 1)
                {
                    tmp = 0xCB;
                    if ((ULONG)out_size_sec > 0x7F) tmp += 0x3;

                    pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                    pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                    pmas->mas_compiled_code[count++] = (UCHAR)tmp;
                }
                
                pmas->mas_compiled_code[count++] = 0xFC;    // cld
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, out_size_fir
                *(int *)(pmas->mas_compiled_code + count) = out_size_sec;
                count += 4;
                pmas->mas_compiled_code[count++] = 0xBE;    // mov esi, offset mas_for_cmps
                *(int *)(pmas->mas_compiled_code + count) = (int)mas_for_cmps;
                count += 4;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov edi, eax
                pmas->mas_compiled_code[count++] = 0xF8;

                pmas->mas_compiled_code[count++] = 0xF3;    // repz cmpsb
                pmas->mas_compiled_code[count++] = 0xA6;

                pmas->mas_compiled_code[count++] = 0x09;    // or ecx, ecx
                pmas->mas_compiled_code[count++] = 0xC9; 
            }
            else
            {
                if (out_type_fir) return false;
                value = *(PULONG)mas_sec;

                if (type_convertion_fir == 0 || type_convertion_fir == 3)
                    type_convertion_fir = 4;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                    *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                    count += 4;
                }            

                pmas->mas_compiled_code[count++] = 0xB8;    // mov eax, value
                *(int *)(pmas->mas_compiled_code + count) = *(int *)mas_fir;
                count += 4;

                if (pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                    pmas->mas_compiled_code[count++] = 0x02;
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                }

                pmas->mas_compiled_code[count++] = 0x50;    // push eax

                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                pmas->mas_compiled_code[count++] = 0x75;    // jne +5
                pmas->mas_compiled_code[count++] = 0x05;

                pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
                pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
                pmas->mas_compiled_code[count++] = 0x9E;    // sahf

                tmp = 0;
                switch (type_convertion_fir)
                {
                case 4:
                    (pointer_level_fir > 1) ? tmp = 0x20 : tmp = 0x1D;
                    break;

                case 2:
                    (pointer_level_fir > 1) ? tmp = 0x2A : tmp = 0x1D;
                    break;

                case 1:
                    (pointer_level_fir > 1) ? tmp = 0x22 : tmp = 0x04;
                }

                pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
                pmas->mas_compiled_code[count++] = (UCHAR)tmp;

                if (type_convertion_fir != 1 || pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                    pmas->mas_compiled_code[count++] = 0xD8;

                    pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                    pmas->mas_compiled_code[count++] = 0xD0;

                    if ((type_convertion_fir == 2 || type_convertion_fir == 1) && pointer_level_fir > 1)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                        pmas->mas_compiled_code[count++] = 0xF9;
                        pmas->mas_compiled_code[count++] = 0x01;                

                        if (type_convertion_fir == 2)
                        {
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                            pmas->mas_compiled_code[count++] = 0x05;

                            pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                            pmas->mas_compiled_code[count++] = 0xC2;
                            pmas->mas_compiled_code[count++] = 0x01;

                            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                            pmas->mas_compiled_code[count++] = 0x03;
                        }
                        else
                        {
                            pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                            pmas->mas_compiled_code[count++] = 0x12;
                        }
                    }

                    if (type_convertion_fir == 2 && pointer_level_fir == 1)
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x03;
                    }

                    pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                    pmas->mas_compiled_code[count++] = 0x0D;

                    pmas->mas_compiled_code[count++] = 0x52;    // push edx
                    pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                    pmas->mas_compiled_code[count++] = 0x15;
                    *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                    count += 4;

                    pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                    pmas->mas_compiled_code[count++] = 0xC0;

                    if (pointer_level_fir > 1)
                    {
                        switch (type_convertion_fir)
                        {
                        case 4:
                            pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                            pmas->mas_compiled_code[count++] = 0xE7;
                            break;

                        case 2:
                            pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                            pmas->mas_compiled_code[count++] = 0xDD;
                            break;

                        case 1:
                            pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                            pmas->mas_compiled_code[count++] = 0xE2;
                        }
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                    }

                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                    pmas->mas_compiled_code[count++] = 0xC3;

                    if (pointer_level_fir > 1)
                    {
                        switch (type_convertion_fir)
                        {
                        case 4:
                            pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                            pmas->mas_compiled_code[count++] = 0xD5;
                            break;

                        case 2:
                            pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                            pmas->mas_compiled_code[count++] = 0xCB;
                            break;

                        case 1:
                            pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                            pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                            pmas->mas_compiled_code[count++] = 0xD0;
                        }
                    }
                }

                switch (type_convertion_fir)
                {
                case 4:
                    pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                    pmas->mas_compiled_code[count++] = 0x3D;    // cmp eax, value
                    *(int *)(pmas->mas_compiled_code + count) = (int)value;
                    count += 4;
                    break;

                case 2:
                    pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x8B;
                    pmas->mas_compiled_code[count++] = 0x00;
                    pmas->mas_compiled_code[count++] = 0x66;    // cmp ax, value
                    pmas->mas_compiled_code[count++] = 0x3D;
                    *(short *)(pmas->mas_compiled_code + count) = (short)value;
                    count += 2;
                    break;

                case 1:
                    pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                    pmas->mas_compiled_code[count++] = 0x00;
                    pmas->mas_compiled_code[count++] = 0x3C;    // cmp al, value
                    pmas->mas_compiled_code[count++] = (char)value;
                }
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // in case     *(0x00400000) == *(0x00401000)
        /////////////////////////////////////////////////////////////////////////////////

        case 0x1011101:

            if (out_type_fir || out_type_sec)
                return false;

            if (type_convertion_fir == 0 || type_convertion_fir == 3)
                type_convertion_fir = 4;

            if (type_convertion_sec == 0 || type_convertion_sec == 3)
                type_convertion_sec = 4;

            if (type_convertion_fir != type_convertion_sec)
                return false;

            if (pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = pointer_level_fir;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0xB8;    // mov eax, value
            *(int *)(pmas->mas_compiled_code + count) = *(int *)mas_fir;
            count += 4;

            if (pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jne +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf 

            tmp = 0;
            switch (type_convertion_fir)
            {
            case 4:
                (pointer_level_fir > 1) ? tmp = 0x30 : tmp = 0x2D;
                break;

            case 2:
                (pointer_level_fir > 1) ? tmp = 0x3B : tmp = 0x2E;
                break;

            case 1:
                (pointer_level_fir > 1) ? tmp = 0x35 : tmp = 0x17;
            }

            if (pointer_level_sec > 1) tmp += 9;

            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +tmp
            pmas->mas_compiled_code[count++] = (UCHAR)tmp;

            if (type_convertion_fir != 1 || pointer_level_fir > 1)
            {
                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_convertion_fir == 2 || type_convertion_fir == 1) && pointer_level_fir > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;

                    if (type_convertion_fir == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_convertion_fir == 2 && pointer_level_fir == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (pointer_level_fir > 1)
                {
                    switch (type_convertion_fir)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                        pmas->mas_compiled_code[count++] = 0xDD;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                        pmas->mas_compiled_code[count++] = 0xE2;
                    }
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                    pmas->mas_compiled_code[count++] = 0xE7;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_fir > 1)
                {
                    switch (type_convertion_fir)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                        pmas->mas_compiled_code[count++] = 0xD5;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                        pmas->mas_compiled_code[count++] = 0xCB;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                        pmas->mas_compiled_code[count++] = 0xD0;
                    }
                }
            }

            switch (type_convertion_fir)
            {
            case 4:
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 1:
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov edi, eax
            pmas->mas_compiled_code[count++] = 0xF8;

            if (pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0xB9;    // mov ecx, pointer_level
                *(int *)(pmas->mas_compiled_code + count) = pointer_level_sec;
                count += 4;
            }

            pmas->mas_compiled_code[count++] = 0xB8;    // mov eax, value
            *(int *)(pmas->mas_compiled_code + count) = *(int *)mas_sec;
            count += 4;

            if (pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0xEB;   //  jmp +2
                pmas->mas_compiled_code[count++] = 0x02;
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x50;    // push eax

            pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
            pmas->mas_compiled_code[count++] = 0x15;
            *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
            count += 4;

            pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
            pmas->mas_compiled_code[count++] = 0xC0;

            pmas->mas_compiled_code[count++] = 0x75;    // jne +5
            pmas->mas_compiled_code[count++] = 0x05;

            pmas->mas_compiled_code[count++] = 0xB4;    // mov ah, byte_for_reverse_eflags
            pmas->mas_compiled_code[count++] = byte_for_reverse_eflags;
            pmas->mas_compiled_code[count++] = 0x9E;    // sahf

            tmp = 0;
            switch (type_convertion_sec)
            {
            case 4:
                (pointer_level_sec > 1) ? tmp = 0x1F : tmp = 0x1C;
                break;

            case 2:
                (pointer_level_sec > 1) ? tmp = 0x2B : tmp = 0x1E;
                break;

            case 1:
                (pointer_level_sec > 1) ? tmp = 0x24 : tmp = 0x06;
            }

            pmas->mas_compiled_code[count++] = 0xEB;    // jmp +1E
            pmas->mas_compiled_code[count++] = (UCHAR)tmp;

            if (type_convertion_sec != 1 || pointer_level_sec > 1)
            {
                pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, eax
                pmas->mas_compiled_code[count++] = 0xD8;

                pmas->mas_compiled_code[count++] = 0x8B;    // mov edx, eax
                pmas->mas_compiled_code[count++] = 0xD0;

                if ((type_convertion_sec == 2 || type_convertion_sec == 1) && pointer_level_sec > 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // cmp ecx, 1
                    pmas->mas_compiled_code[count++] = 0xF9;
                    pmas->mas_compiled_code[count++] = 0x01;                

                    if (type_convertion_sec == 2)
                    {
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz +5
                        pmas->mas_compiled_code[count++] = 0x05;

                        pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                        pmas->mas_compiled_code[count++] = 0xC2;
                        pmas->mas_compiled_code[count++] = 0x01;

                        pmas->mas_compiled_code[count++] = 0xEB;    // jmp +3h
                        pmas->mas_compiled_code[count++] = 0x03;
                    }
                    else
                    {
                        pmas->mas_compiled_code[count++] = 0x74;    // jz +0F
                        pmas->mas_compiled_code[count++] = 0x12;
                    }
                }

                if (type_convertion_sec == 2 && pointer_level_sec == 1)
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 1
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x01;
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x83;    // add edx, 3
                    pmas->mas_compiled_code[count++] = 0xC2;
                    pmas->mas_compiled_code[count++] = 0x03;
                }

                pmas->mas_compiled_code[count++] = 0x76;    // jna +0Dh
                pmas->mas_compiled_code[count++] = 0x0D;

                pmas->mas_compiled_code[count++] = 0x52;    // push edx
                pmas->mas_compiled_code[count++] = 0xFF;    // call dword ptr [OffsetOfMakePageInFunction]
                pmas->mas_compiled_code[count++] = 0x15;
                *(int *)(pmas->mas_compiled_code + count) = (int)(&OffsetOfMakePageInFunction);
                count += 4;

                pmas->mas_compiled_code[count++] = 0x0B;    // or eax, eax
                pmas->mas_compiled_code[count++] = 0xC0;

                if (pointer_level_sec > 1)
                {
                    switch (type_convertion_sec)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                        pmas->mas_compiled_code[count++] = 0xE7;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -DD
                        pmas->mas_compiled_code[count++] = 0xDD;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x74;    // jz -E2
                        pmas->mas_compiled_code[count++] = 0xE2;
                    }
                }
                else
                {
                    pmas->mas_compiled_code[count++] = 0x74;    // jz -E7
                    pmas->mas_compiled_code[count++] = 0xE7;
                }

                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;

                if (pointer_level_sec > 1)
                {
                    switch (type_convertion_sec)
                    {
                    case 4:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D5
                        pmas->mas_compiled_code[count++] = 0xD5;
                        break;

                    case 2:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -CB
                        pmas->mas_compiled_code[count++] = 0xCB;
                        break;

                    case 1:
                        pmas->mas_compiled_code[count++] = 0x49;    // dec ecx
                        pmas->mas_compiled_code[count++] = 0x75;    // jnz -D0
                        pmas->mas_compiled_code[count++] = 0xD0;
                    }
                }
            }

            switch (type_convertion_sec)
            {
            case 4:
                pmas->mas_compiled_code[count++] = 0x8B;    // mov eax, dword ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;    // mov ax, word ptr [eax]
                pmas->mas_compiled_code[count++] = 0x8B;
                pmas->mas_compiled_code[count++] = 0x00;
                break;

            case 1:
                pmas->mas_compiled_code[count++] = 0x8A;    // mov al, byte ptr [eax]
                pmas->mas_compiled_code[count++] = 0x00;
            }

            pmas->mas_compiled_code[count++] = 0x8B;    // mov ebx, edi
            pmas->mas_compiled_code[count++] = 0xDF;

            switch (type_convertion_fir)
            {
            case 1:
                pmas->mas_compiled_code[count++] = 0x3A;    // cmp al, bl
                pmas->mas_compiled_code[count++] = 0xC3;
                break;

            case 2:
                pmas->mas_compiled_code[count++] = 0x66;
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp ax, bx
                pmas->mas_compiled_code[count++] = 0xC3;
                break;

            case 4:
                pmas->mas_compiled_code[count++] = 0x3B;    // cmp eax, ebx
                pmas->mas_compiled_code[count++] = 0xC3;
            }

            break;

        /////////////////////////////////////////////////////////////////////////////////
        // default:
        /////////////////////////////////////////////////////////////////////////////////

        default: 
            return false;
    }

    pmas->short_opcode_jump = condition_opcode;
    pmas->size_compiled_code = count;
    buffer = buf;
    return true;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

char FindLogicCondition(char *&buffer)
{
    char *pbuf = buffer;

    while (pbuf[0] == ' ') ++pbuf;
    if ( pbuf[0] == 0 || pbuf[0] == ')' ) return 0;

    for (char i = 0; i <= sizeof(mas_logic_conditions)/4 - 1; i++)
    {
        if (!strncmp(pbuf, mas_logic_conditions[i], strlen(mas_logic_conditions[i])))
        {
            pbuf += strlen(mas_logic_conditions[i]);

            while (pbuf[0] == ' ') ++pbuf;        
            if (!pbuf[0]) 
                return -1;

            buffer = pbuf;
            return ++i;
        }
    }

    return -1;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool FindEqualCondition(char *&buffer, char &condition_opcode)
{
    while (buffer[0] == ' ') ++buffer;
    if (!buffer[0]) return false;    

    for (int i = 0; i <= sizeof(mas_conditions_equal)/4 - 1; i++)
    {
        if (!strncmp(buffer, mas_conditions_equal[i], strlen(mas_conditions_equal[i])))
        {
            condition_opcode = mas_conditions_opcode[i];
            buffer += strlen(mas_conditions_equal[i]);
            return true;
        }
    }

    return false;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool FindRegister(char *&buffer, char &reg_offset, char &reg_size)
{
    char *buf = buffer;

    while (buf[0] == ' ') ++buf;
    if (!buf[0]) return false;

    for (int i = 0; i <= sizeof(mas_registers)/4 - 1; i++)
    {
        if (!_strnicmp(buf, mas_registers[i], strlen(mas_registers[i])))
        {
            reg_offset = mas_reg_offset[i] + 0xC;

            if (i == 9)
                reg_size = 2;
            else if (i < 11) 
                reg_size = 4;
            else if (i < 19)
                reg_size = 2;
            else
                reg_size = 1;

            buffer = buf + strlen(mas_registers[i]);
            return true;
        }
    }

    return false;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool my_atoi(char *&buffer, ULONG& value)
{
    char *str = buffer;
    int j = 0;
    int i = 0;

    while (str[0] == ' ') ++str;
    if (
        str[0] == 0     || 
        str[i] == ')'   || 
        str[i] == '|'   || 
        str[i] == '&'   || 
        str[i] == ','
       ) 
       return false;

    //if (strncmp("0x", str, 2))
    //{
    //    char minus = 0;
    //    if (str[0] == '-') 
    //    { 
    //        minus = 1; 
    //        ++str;
    //        while (str[0] == ' ') ++str;
    //        if (!str[0]) return false;
    //    }

    //    while (
    //        str[i] != 0   && 
    //        str[i] != ' ' && 
    //        str[i] != ')' &&
    //        str[i] != '|' &&
    //        str[i] != '&' &&
    //        str[i] != ','
    //        )
    //    {
    //        if (str[i] >= '0' && str[i] <= '9')
    //            j = 10*j + str[i] - '0';
    //        else
    //            return false;
    //        ++i;
    //    }

    //    if (minus)
    //        value = -j;
    //    else
    //        value = j;

    //    buffer = str + i;
    //    return true;
    //}
    //else
    {
        if (!strncmp("0x", str, 2)) str += 2;
        for (i = 0; i <= 8; i++)
        {
            if ( str[i] >= '0' && str[i] <= '9' )
            {
                j = (j << 4) + str[i] - '0';
            }
            else if ( (str[i] >= 'A' && str[i] <= 'F') || (str[i] >= 'a' && str[i] <= 'f') )
            {
                j = (j << 4) + (str[i] & 0x0F) + 9;
            }
            else if ( 
                (
                str[i] == 0   ||
                str[i] == ' ' ||
                str[i] == ')' ||
                str[i] == '|' ||
                str[i] == '&' ||
                str[i] == ','
                ) && 
                (i != 0) 
                )
            {
                value = j;
                buffer = str + i;
                return true;
            }
            else
                return false;
        }
    }

    return false;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool FindPointerCondition(char *&buffer, int &pointer_level, int &type_convertion)
{
    char *buf = buffer;
    int level = 0;

    do
    {
        while (*buf == ' ') ++buf;
        if (!buf[0]) return false;

        if (buf[0] == '*') 
            ++level;
        else
            break;
    }
    while (++buf);

    if (!level) return false;

    if (buf[0] != '(')
    {
        pointer_level = level;
        type_convertion = 0;
        buffer = buf;
        return true;
    }

    ++buf;
    while (*buf == ' ') ++buf;
    if (!buf[0]) return false;

    for (int i = 0; i <= sizeof(mas_type_conv)/4 - 1; i++)
    {
        if (!_strnicmp(buf, mas_type_conv[i], strlen(mas_type_conv[i])))
        {
            buf += strlen(mas_type_conv[i]);
            type_convertion = ++i;

            i = 0;
            do
            {
                while (*buf == ' ') ++buf;
                if (!buf[0]) return false;

                if (buf[0] == '*') 
                    ++i;
                else
                    break;
            }
            while (++buf);

            if (level == i && buf[0] == ')') 
            {
                pointer_level = level;
                buffer = ++buf;
                return true;
            }
            else
                return false;
        }
    }

    return false;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bool FindBinaryCondition(char *&buffer, int &out_type, int &out_size, char *out_massive)
{
    char *tmp_buf       = buffer;
    char *buf           = buffer;

    int not_first       = 0;
    int count           = 0;
    char chr            = 0;
    int i               = 0;
    ULONG value         = 0;

    while (buf[0] == ' ') ++buf;
    if (!buf[0]) return false;

    if ((buf[0] < '0' || buf[0] > '9') && buf[0] != 0x27)
    {
        while (
            buf[0] != ')' && 
            buf[0] != '|' && 
            buf[0] != '&' && 
            buf[0] != ',' && 
            buf[0] != '=' &&
            buf[0] != '>' &&
            buf[0] != '<' &&
            buf[0] != ' ' &&
            buf[0] != 0
            ) ++buf;

        chr = buf[0];
        buf[0] = 0;

        if (si_atoi((PCSTR)tmp_buf, value))
        {
            *(PULONG)out_massive = value;
            out_type = 0;
            buf[0] = chr;
            buffer = buf;
            return true;
        }
        buf[0] = chr;

        buf = buffer;
    }

    do
    {
        while (*buf == ' ') ++buf;
        if (!buf[0] && !not_first) return false;

        if (buf[0] == 0x27)
        {
            if (buf[1] == 0x27) return false;

            i = 0;
            do
            {
                out_massive[count + i++] = (++buf)[0];
                if (!buf[0]) return false;
            }
            while (buf[1] != 0x27);

            not_first = 1;
            count += i;
            buf += 2;
        }
        else if (my_atoi(buf, value))
        {
            if (!not_first)
            {
                *(PULONG)out_massive = value;
                ++count;
            }
            else if (value <= 0xFF)
                out_massive[count++] = (char)value;
            else
                return false;
        }
        else
            return false;
        
        while (*buf == ' ') ++buf;

        if (buf[0] != ',') break;
        not_first = 1;
    }
    while (++buf);

    out_type = not_first;
    out_size = count;
    buffer = buf;
    return true;
}