/*++
    Copyright  (c) 2002 Sten
    Contact information:
        mail: stenri@mail.ru

    This code is based on mamaich's IceLib.cpp code (see IceX src).

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 
Module Name:
    ntoskernel.cpp

Abstract: Functions for dynamic linking with NTosKernel.EXE and NTDLL.DLL.
    (SoftICE exports very few symbols).

Revision History:

 Sten      02/10/2002
      Initial release

--*/

extern "C" {
#pragma warning ( push, 3 )
#include <ntddk.h>
#pragma warning ( pop )
}

#pragma warning ( disable: 4514 ) // unreferenced inline function has been removed

#define __NTOSKRNL_C__

#include "defs.h"
#include "img.h"

#include "ntoskrnl.h"

#undef  __NTOSKRNL_C__

unsigned int GetAddressofShadowTable();

////////////////////////////////////////////////////////////////////////////
//
// InitNTosKernel
//
//   Find addresses of NTOSKRNL.EXE and NTDLL.DLL in memory
//   Find address and size of NTOSKRNL!CODE section
//
////////////////////////////////////////////////////////////////////////////

NTSTATUS InitNTosKernel(void)
{
     hNTOSKRNL = UtGetModuleBase("ntoskrnl.exe");

     if (hNTOSKRNL == 0)
     {
         DbgPrint("InitNTosKernel: can't find ntoskrnl.exe image.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     DbgPrint("ntoskrnl image:                               %08X\n", hNTOSKRNL);

     hNTDLL = UtGetModuleBase("ntdll.dll");

     if (hNTDLL == 0)
     {
         DbgPrint("InitNTosKernel: can't find ntdll.dll image.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     } 

     DbgPrint("ntdll:                                        %08X\n", hNTDLL);

     // OK. Now we are going to find size and address of CODE section
     //     in NTOSKRNL.EXE

     IMAGE_DOS_HEADER *MZ=(IMAGE_DOS_HEADER*)hNTOSKRNL;
     #include <pshpack1.h>
     const struct HDR
     {
           DWORD signature;
           IMAGE_FILE_HEADER _head;
           IMAGE_OPTIONAL_HEADER opt_head;
           IMAGE_SECTION_HEADER section_header[2];
     } *PE;
     #include <poppack.h>

     PE=(HDR*)(MZ->e_lfanew+hNTOSKRNL);
     DbgPrint("NTOSKRNL Section %4s at:                    %08X\n",PE->section_header[0].Name, PE->section_header[0].VirtualAddress+hNTOSKRNL);

     NTOS_TEXT_ADDR=(BYTE*)(PE->section_header[0].VirtualAddress+hNTOSKRNL);
     NTOS_TEXT_SIZE=PE->section_header[0].SizeOfRawData;

     DbgPrint("NTOS_TEXT_ADDR:                               %08X\n", NTOS_TEXT_ADDR);
     DbgPrint("NTOS_TEXT_SIZE:                               %08X\n", NTOS_TEXT_SIZE);

     // Find address of KeServiceDescriptorTable
     KeServiceDescriptorTable = (ServiceDescriptorTableEntry_t*)UtGetProcAddress(
                                    (HANDLE)hNTOSKRNL, "KeServiceDescriptorTable");

     if (KeServiceDescriptorTable == NULL)
     {
         DbgPrint("InitNTosKernel: can't find KeServiceDescriptorTable address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     DbgPrint("KeServiceDescriptorTable:                     %08X\n", KeServiceDescriptorTable);

     NtDelayExecution = (ZWDELAYEXECUTION)UtGetProcAddress((HANDLE)hNTDLL, "NtDelayExecution");
     if (!NtDelayExecution)
     {
         DbgPrint("InitNTosKernel: can't find NtDelayExecution address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     NtYieldExecution = (ZWYIELDEXECUTION)UtGetProcAddress((HANDLE)hNTDLL, "NtYieldExecution");
     if (!NtYieldExecution)
     {
         DbgPrint("InitNTosKernel: can't find NtYieldExecution address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     NTDLL_NtClose = (DWORD)UtGetProcAddress((HANDLE)hNTDLL, "NtClose");
     if (!NTDLL_NtClose)
     {
         DbgPrint("InitNTosKernel: can't find NTDLL_NtClose address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     NtWaitForSingleObject = (ZWWAITFORSINGLEOBJECT)UtGetProcAddress((HANDLE)hNTDLL, 
                                                              "NtWaitForSingleObject");
     if (!NtWaitForSingleObject)
     {
         DbgPrint("InitNTosKernel: can't find NtWaitForSingleObject address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     ObOpenObjectByPointer = (OBOPENOBJECTBYPOINTER)UtGetProcAddress((HANDLE)hNTOSKRNL, 
                                                              "ObOpenObjectByPointer");
     if (!ObOpenObjectByPointer)
     {
         DbgPrint("InitNTosKernel: can't find ObOpenObjectByPointer address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }

     KeServiceDescriptorTableShadow = (ServiceDescriptorTableEntry_t*)GetAddressofShadowTable();

     if (!KeServiceDescriptorTableShadow)
     {
         DbgPrint("InitNTosKernel: can't find ShadowDescriptorTable address.\n");
         return STATUS_PROCEDURE_NOT_FOUND;
     }
     DbgPrint("KeServiceDescriptorTableShadow:               %08X\n",KeServiceDescriptorTableShadow);

     ZwQueryDirectoryObject = (NTQUERYDIRECTORYOBJECT)UtGetProcAddress((HANDLE)hNTOSKRNL, "ZwQueryDirectoryObject");
     if (!ZwQueryDirectoryObject)
     {
         // NT4 kernel doesn't export ZwQueryDirectoryObject, so I just print
         // error message in case procedure address is not found and continue
         // initialization process
         DbgPrint("InitNTosKernel: can't find ZwQueryDirectoryObject address.\n");
     }

     return STATUS_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////
//
//  GetAddressofShadowTable
//
//     Get Shadow Descriptor Table address      
//     Code is from book "Undocumented Windows NT"
////////////////////////////////////////////////////////////////////////////

unsigned int GetAddressofShadowTable()
{
     int i;
     unsigned char *p;
     unsigned int dwordatbyte;

     p=(unsigned char *)KeAddSystemServiceTable;

     DbgPrint("KeAddSystemServiceTable:                      %08X\n",p);
     
     for (i=0; i<4096; i++, p++)
     {
          __try  // will it work ???
          {
                dwordatbyte = *(unsigned int *)p;
          }
          __except (EXCEPTION_EXECUTE_HANDLER)
          {
                return 0;
          }
 
          if ((PVOID)dwordatbyte==KeServiceDescriptorTable)
          {
              continue;
          }

          if (MmIsAddressValid((PVOID)dwordatbyte))
          {
              if (RtlCompareMemory((PVOID)dwordatbyte, KeServiceDescriptorTable, 16)==16)
              {
                  return dwordatbyte;
              }
          }
     }
     return 0;
}