/*++
    Copyright  (c) 2002 Sten
    Contact information:
        mail: stenri@mail.ru

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 
Module Name: 
    tracer.cpp

Abstract: Very simple tracer engine. It's only purpose is trace the instruction that
    caused break point on range exception so that break point can be restored after 
    instruction execution.
Revision History:

 Godness     01/09/2005
      Adapted for eXTreme tRaCer engine

 Sten        11/12/2002
      Initial release

--*/

extern "C"{
#pragma warning ( push, 3 )
#include <ntddk.h>
#pragma warning ( pop )
}

#pragma warning ( disable: 4514 ) // unreferenced inline function has been removed

#include "defs.h"
#include "ntoskrnl.h"
#include "softice.h"
#include "keyboard.h"
#include "undoc.h"
#include "bpr.h"
#include "tracer.h"

///////////////////////////////////////////////////////////////////////
//                    External definitions
///////////////////////////////////////////////////////////////////////
extern void ExecutePCommand(void);
extern void ExecutePRETCommand(void);

extern ULONG NumberOfTracedCommand;
extern ULONG CountOfCommand;
extern ULONG TracerCallBack;
extern ULONG EipWhenPRET;

extern ULONG ReturnToMakePageIn;
extern ULONG MakePageIn;
extern ULONG SavedCR0;
extern ULONG CR3ForOurProcess;
extern ULONG PreLastEIPValue;
extern ULONG CurrentRing;

void* CheckAddressAndMakePageIn(void *address);
void* GetUserHandler(void);

///////////////////////////////////////////////////////////////////////
//                           Globals
///////////////////////////////////////////////////////////////////////
void *OldInterruptStack;
char NewInterruptStack[0x1000];

int InTraceMode         = 0;
int HideTFflag          = 0;
int EipWait             = 0;

int SystemInt1          = 0;
int DRDebugException    = 0;
int DRSavedHandler      = 0;
char DRSavedByte        = 0;

void* UserHandler       = 0;
void* UserAddress       = 0;
int EmulatePOPFD        = 0;

char Int0Installed = 1;
void *OldInt0      = 0;
void *OldInt1      = 0; // Old int1 handler
void *OldInt6      = 0;
void *OldIntD      = 0;
void *OldInt10     = 0;

char szBad[]          = "Traced through given number of instructions - no trace condition has been met...\n";
char szGood[]         = "One of the trace conditions is met! %d commands has been traced...\n";
char szFPUInst[]      = "FPU exception occured at address %08X\n";
char szInt3Inst[]     = "INT3 instruction has been met at address %08X\n";
char szException[]    = "Exception instruction has been met at address %08X\n";
char szInvalidInst[]  = "Invalid instruction has been met at address %08X\n";
char szTraceStoped[]  = "Tracing has been stopped! %d commands has been traced...\n";
char szZeroDivision[] = "Division on ZERO occured at address %08X\n";

int __cdecl CodeAnalyzer (CLIENT_STATE *pClientState)
{
#define CHANGE_ESP3_VALUE(UserRegister)                 \
    pClientState->esp3 = UserRegister;                  \
    pClientState->eip += 2;                             \
    return 1;

    if (!CheckAddressAndMakePageIn((void *)(pClientState->eip + 0))) return 0;
    if (!CheckAddressAndMakePageIn((void *)(pClientState->eip + 1))) return 0;

    if (*(PUCHAR)(pClientState->eip + 0) == 0x8B)
    {
        switch (*(PUCHAR)(pClientState->eip + 1))
        {
        case 0xE0: CHANGE_ESP3_VALUE(pClientState->eax);  // emulate mov esp, eax
        case 0xE1: CHANGE_ESP3_VALUE(pClientState->ecx);  //  -//-   mov esp, ecx
        case 0xE2: CHANGE_ESP3_VALUE(pClientState->edx);  // ...
        case 0xE3: CHANGE_ESP3_VALUE(pClientState->ebx);
        case 0xE4: CHANGE_ESP3_VALUE(pClientState->esp3); 
        case 0xE5: CHANGE_ESP3_VALUE(pClientState->ebp); 
        case 0xE6: CHANGE_ESP3_VALUE(pClientState->esi); 
        case 0xE7: CHANGE_ESP3_VALUE(pClientState->edi); 
        }
    }

    if (*(PUCHAR)(pClientState->eip + 0) == 0x9C)   // if pushfd command was met
    {                                               // need to remove TF flag from stack
        EipWait = pClientState->eip + 1;
        HideTFflag = 1;
        return 0;
    }

    if (*(PUCHAR)(pClientState->eip + 0) == 0x9D)   // if popfd command was met
    {
        if (!CheckAddressAndMakePageIn((void *)(pClientState->esp3))) return 0;

        if (*(PULONG)pClientState->esp3 & 0x100)
        {
            if ((UserHandler = GetUserHandler()) == 0) return 0;
            UserAddress = (void *)pClientState->eip;

            DWORD cr0 = EnableWrite();
            *(PUCHAR)pClientState->eip = 0xCC;
            SetCR0(cr0);

            EmulatePOPFD = 1;
            return 0;
        }
    }

    return 0;
}

///////////////////////////////////////////////////////////////////////
//
//  Int 00 handler 
//
///////////////////////////////////////////////////////////////////////

void __declspec(naked) Int00Handler(void)
{
    __asm
    {
        cmp     cs:[TracerCallBack], 0
        je      next_handler
        pushad
        push    fs
        push    ds
        push    es

        mov     eax, 30h
        mov     fs, ax
        mov     eax, 23h
        mov     ds, ax
        mov     es, ax

        push    dword ptr [esp + 0x2C]
        push    offset szZeroDivision
        call    DbgPrint
        add     esp, 8

        pop     es
        pop     ds
        pop     fs
        popad

    next_handler:
        jmp     cs:[OldInt0]
    }
}

///////////////////////////////////////////////////////////////////////
//
//  Int 01 handler 
//
///////////////////////////////////////////////////////////////////////

void __declspec(naked) Int01Handler(void)
{
    __asm
    {
        cmp     cs:[MakePageIn], 0
        je      not_make_pagein

        jmp     cs:[ReturnToMakePageIn]   //return to CheckAndMakePageIn function

not_make_pagein:
           
        cmp     cs:[TracerCallBack], 0
        je      do_not_trace

        push    ds
        push    eax
        mov     eax, 23h
        mov     ds, ax

        mov     eax, esp
        add     eax, 8
        mov     OldInterruptStack, eax   // save original stack pointer
        pop     eax

        mov     esp, offset NewInterruptStack + 0x1000 - 0x10

        pushad
        push    fs
        mov     esi, OldInterruptStack
        push    dword ptr [esi - 4]             // save original ds value 
        push    es

        mov     eax, 30h
        mov     fs, ax
        mov     eax, 23h
        mov     es, ax

        mov     al, byte ptr [esi + 4]
        and     al, 3
        cmp     al, byte ptr [CurrentRing]
        je      ring_value_not_change

        pop     es                              // if ring value was suddenly changed
        pop     ds                              // the some other int1 interrupt occured
        pop     fs                              // so we need to pass this way for today...
        popad
        mov     esp, OldInterruptStack
        iretd

ring_value_not_change:

        cld
        mov     ecx, 5
        test    al, 3                           // check in CS value last two bits
        jnz     copy_stack                      // copy interrupt stack to new arrea
        mov     ecx, 3                          // it's need for MakePageIn function
copy_stack:
        mov     edi, offset NewInterruptStack + 0x1000 - 0x10
        repnz   movsd

        cmp     HideTFflag, 0                   // remove TF flag when pushfd command was met
        jz      emulate_popfd

        mov     HideTFflag, 0
        mov     eax, OldInterruptStack          // get original interrupt stack pointer
        mov     edx, dword ptr [eax]
        cmp     edx, EipWait                    // check current eip value
        jne     emulate_popfd
        mov     eax, dword ptr [eax + 0xC]      // get esp ring 3 pointer

        push    eax
        call    CheckAddressAndMakePageIn
        or      eax, eax
        jz      emulate_popfd

        and     dword ptr [eax], not 100h

emulate_popfd:
        cmp     EmulatePOPFD, 0                 // need to correct exception params in stack
        jz      tracer_call_back                // for emulate popfd instruction

        mov     EmulatePOPFD, 0

        mov     eax, dword ptr [esp + 0x2C]     // if user eip value == user's exception handler
        cmp     eax, UserHandler
        jnz     tracer_call_back

        mov     eax, dword ptr [esp + 0x38]
        push    eax
        call    CheckAddressAndMakePageIn
        or      eax, eax
        jz      tracer_call_back

        mov     eax, dword ptr [eax + 4]        // get pointer to ExceptionCode in _EXCEPTION_RECORD *ExceptionRecord
        push    eax
        call    CheckAddressAndMakePageIn
        or      eax, eax
        jz      tracer_call_back

        mov     dword ptr [eax], 0x80000004     // STATUS_SINGLE_STEP exception

        call    EnableWrite
        mov     ebx, eax

        mov     eax, UserAddress
        mov     byte ptr [eax], 0x9D            // patch user code back

        push    ebx
        call    SetCR0

tracer_call_back:
        call    TracerCallBack
        or      eax, eax
        jnz     good_message

        mov     eax, CountOfCommand
        inc     eax
        mov     CountOfCommand, eax
        cmp     eax, NumberOfTracedCommand
        jae     bad_message

        mov     eax, DR6                        // check if DR registers was
        test    al, 0Fh                         // cause of interrupt 
        jz      not_debug_breakpoint

        cmp     CurrentRing, 3                  // ring0 context not support yet...
        jne     not_debug_breakpoint

        call    GetUserHandler                  // in this way we must transfer
        or      eax, eax                        // control to system handler
        jz      dr_exception_bad

        push    eax
        call    CheckAddressAndMakePageIn
        or      eax, eax
        jz      dr_exception_bad

        push    eax
        call    EnableWrite
        mov     ebx, eax
        pop     eax

        mov     dl, byte ptr [eax]
        mov     DRSavedHandler, eax
        mov     DRSavedByte, dl
        mov     byte ptr [eax], 0xCC

        push    ebx
        call    SetCR0

        mov     eax, si_OldIntTable
        mov     eax, [eax + 4]
        mov     SystemInt1, eax

        mov     DRDebugException, 1

dr_exception_bad:
        pop     es
        pop     ds
        pop     fs
        popad
        mov     esp, OldInterruptStack
        or      dword ptr [esp + 8], 100h
        jmp     cs:[SystemInt1]                 // transfer control to system handler

not_debug_breakpoint:

        push    esp
        call    CodeAnalyzer
        add     esp, 4

        or      al, al
        jz      next_check

        mov     eax, CountOfCommand      // increment CountOfCommand
        inc     eax                      // if some instruction was emulated
        mov     CountOfCommand, eax
        cmp     eax, NumberOfTracedCommand
        jae     bad_message

next_check:
        cmp     EipWhenPRET, -1          // if we do not need execute "P" command
        je      trace_next_command

        mov     eax, [esp + 0x2C]
        cmp     byte ptr [eax], 0xE8
        je      direct_call

        cmp     word ptr [eax], 0x15FF
        jne     trace_next_command

        mov     eax, [eax + 2]
        push    eax
        call    CheckAddressAndMakePageIn
        mov     eax, [eax]
        jmp     compare_eip

direct_call:
        add     eax, [eax + 1]
        add     eax, 5

compare_eip:
        cmp     eax, EipWhenPRET
        jae     make_p_command

trace_next_command:
//        cmp     Int0Installed, 1
//        je      int0_installed
//        push    0x8E
//        push    offset Int00Handler
//        push    0x0
//        call    SetInterruptHandler
//        mov     OldInt0, eax
//        cli
//        mov     Int0Installed, 1
//
//int0_installed:
        mov     eax, dword ptr [esp + 0x2C]
        mov     dword ptr [PreLastEIPValue], eax

        pop     es
        pop     ds
        pop     fs
        popad
        mov     esp, OldInterruptStack  // restore original interrupt stack pointer
        or      dword ptr [esp + 8], 100h
        iretd

make_p_command: 
        call    ExecutePCommand

        mov     TracerCallBack, 0

        pop     es
        pop     ds
        pop     fs
        popad
        mov     esp, OldInterruptStack  // restore original interrupt stack pointer
        and     dword ptr [esp + 8], not 100h
        jmp     [OldInt1]

good_message:
        inc     CountOfCommand
        push    CountOfCommand
        push    offset szGood
        call    DbgPrint
        add     esp, 8
        jmp     stop_trace

bad_message:
        push    offset szBad
        call    DbgPrint
        add     esp, 4

stop_trace:
        mov     TracerCallBack, 0
        mov     CountOfCommand, 0
        mov     CR3ForOurProcess, 0
        mov     NumberOfTracedCommand, 0
        mov     EipWhenPRET, -1

        pop     es
        pop     ds
        pop     fs
        popad
        mov     esp, OldInterruptStack  // restore original interrupt stack pointer

do_not_trace:
        pushad
        call    bpr_ActivateAll

        or      eax, eax
        jz      normal_exit

        mov     eax, si_TraceFlag             // in trace mode
        cmp     byte ptr [eax], 0
        jne     normal_exit

        popad
        and     dword ptr [esp + 8], not 100h // turn TF flag off
        iretd                                 // do not pass control to the system handler

normal_exit:

        popad
        jmp     cs:[OldInt1]
     }
}

///////////////////////////////////////////////////////////////////////
//
//  Int 06 handler 
//
///////////////////////////////////////////////////////////////////////

void __declspec(naked) Int06Handler(void)
{
    __asm
    {
        cmp     cs:[TracerCallBack], 0
        je      next_handler
        pushad
        push    fs
        push    ds
        push    es

        mov     eax, 30h
        mov     fs, ax
        mov     eax, 23h
        mov     ds, ax
        mov     es, ax

        push    dword ptr [esp + 0x2C]
        push    offset szInvalidInst
        call    DbgPrint
        add     esp, 8

        pop     es
        pop     ds
        pop     fs
        popad

    next_handler:
        jmp     cs:[OldInt6]
    }
}

///////////////////////////////////////////////////////////////////////
//
//  Int 0D handler 
//
///////////////////////////////////////////////////////////////////////

void __declspec(naked) Int0DHandler(void)
{
    __asm
    {
        cmp     cs:[TracerCallBack], 0
        je      next_handler
        pushad
        push    fs
        push    ds
        push    es

        mov     eax, 30h
        mov     fs, ax
        mov     eax, 23h
        mov     ds, ax
        mov     es, ax

        push    dword ptr [esp + 0x30]
        push    offset szException
        call    DbgPrint
        add     esp, 8

        pop     es
        pop     ds
        pop     fs
        popad

    next_handler:
        jmp     cs:[OldIntD]
    }
}

///////////////////////////////////////////////////////////////////////
//
//  Int 10 handler 
//
///////////////////////////////////////////////////////////////////////

void __declspec(naked) Int10Handler(void)
{
    __asm
    {
        cmp     cs:[TracerCallBack], 0
        je      next_handler
        pushad
        push    fs
        push    ds
        push    es

        mov     eax, 30h
        mov     fs, ax
        mov     eax, 23h
        mov     ds, ax
        mov     es, ax

        push    dword ptr [esp + 0x2C]
        push    offset szFPUInst
        call    DbgPrint
        add     esp, 8

        pop     es
        pop     ds
        pop     fs
        popad

    next_handler:
        jmp     cs:[OldInt10]
    }
}

///////////////////////////////////////////////////////////////////////
//
// InitTracer
//
//   Hooks INT 01 and prepare other stuff for tracer.
//
///////////////////////////////////////////////////////////////////////

NTSTATUS InitTracer()
{
     //-----------------------------------------------------------------------
     // Install tracer's handlers
     //-----------------------------------------------------------------------

     OldInt1 = SetInterruptHandler(0x01, Int01Handler, 0x8E);
     DbgPrint("OldINT 0x1:                                   %08X\n", OldInt1);

     //not need to set on this interrupts ...yet

     //OldInt6 = SetInterruptHandler(0x06, Int06Handler, 0x8E);
     //DbgPrint("OldINT 0x6:                                   %08X\n", OldInt6);

     //OldIntD = SetInterruptHandler(0x0D, Int0DHandler, 0x8E);
     //DbgPrint("OldINT 0xD:                                   %08X\n", OldIntD);

     //OldInt10 = SetInterruptHandler(0x10, Int10Handler, 0x8E);
     //DbgPrint("OldINT 0x10:                                  %08X\n", OldInt10);

     return STATUS_SUCCESS;
}

///////////////////////////////////////////////////////////////////////
//
// TracerDone
//
//   Unhooks INT 01.
//
///////////////////////////////////////////////////////////////////////

void DoneTracer()
{
     // Uninstall tracer's handlers

     if ( (GetInterruptHandler(0x0) == &Int00Handler) && OldInt0 )
     {
         SetInterruptHandler(0x00, OldInt0);
         OldInt0 = 0;
     }

     if (OldInt1)
     {
         SetInterruptHandler(0x01, OldInt1);
         OldInt1 = 0;
     }

     //if (OldInt6)
     //{
     //    SetInterruptHandler(0x06, OldInt6);
     //    OldInt6 = 0;
     //}
     //
     //if (OldIntD)
     //{
     //    SetInterruptHandler(0x0D, OldIntD);
     //    OldIntD = 0;
     //}
     //
     //if (OldInt10)
     //{
     //    SetInterruptHandler(0x10, OldInt10);
     //    OldInt10 = 0;
     //}
}

///////////////////////////////////////////////////////////////////////
//
//  function CheckAddressAndMakePageIn - make "PAGEIN"
//
//  this version of function use other stack arrea, that
//  changed in the beggining of int1 handler
//
///////////////////////////////////////////////////////////////////////

__declspec(naked) void* CheckAddressAndMakePageIn(void *address)
{
    UNREFERENCED_PARAMETER(address);

    static int saved_user_bytes     = 0;
    static int saved_esp_end        = 0;

    __asm
    {
        pushad
        push    fs
        push    ds
        push    es
        push    gs

        cmp     dword ptr [esp + 0x34], 0x1000
        jb      bad_pointer

        cmp     dword ptr [esp + 0x34], NT_HIGHEST_USER_ADDRESS
        jb      make_page_in

        mov     eax, dword ptr [esp + 0x34]          //check pde
        mov     ebx, eax
        push    eax
        call    GetPde
        mov     eax, dword ptr [eax]
        test    al, 1
        jz      bad_pointer
        push    ebx                                  //check pte
        call    GetPte
        mov     eax, dword ptr [eax]
        test    al, 1
        jnz     page_in_memory

        // here need to make PAGEIN for page in range > NT_HIGHEST_USER_ADDRESS
        // not support yet...

bad_pointer:
        pop     gs
        pop     es
        pop     ds
        pop     fs
        popad
        xor     eax, eax
        ret     4

make_page_in:
        mov     eax, dword ptr [esp + 0x34]          //get [address] parameter
        mov     ebx, eax
        push    eax                                  //check pde
        call    GetPde
        mov     eax, dword ptr [eax]
        test    al, 1
        jz      bad_pointer
        push    ebx                                  //check pte
        call    GetPte
        mov     eax, dword ptr [eax]
        test    al, 1
        jnz     page_in_memory

        mov     eax, OldInterruptStack               //if the current code is ring0
        test    byte ptr [eax + 4], 3                //we can't make "page_in"
        jz      bad_pointer                          //yet...

        mov     eax, dword ptr [PreLastEIPValue]

        mov     edx, eax
        and     edx, 0FFFh                           //check if last eip == 0x...FFF
        cmp     edx, 0FFFh                           //and next page may be "page out"
        jne     eip_in_range
        sub     eax, 4
eip_in_range:

        mov     dx, word ptr [eax]                   //save user bytes
        mov     word ptr [saved_user_bytes], dx

        push    eax                                  //patch user code with command mov al, [eax]
        call    EnableWrite
        mov     SavedCR0, eax
        pop     eax
        mov     word ptr [eax], 0x008A

        mov     dword ptr [saved_esp_end], esp
        push    23h                                  //prepare own interrupt stack
        mov     eax, offset NewInterruptStack + 0x1000 - 0x10
        push    dword ptr [eax + 0xC]                //get esp value for ring3
        push    00010306h
        push    1Bh
        push    dword ptr [PreLastEIPValue]
        mov     eax, ebx

        mov     ReturnToMakePageIn, offset return_from_int1
        mov     MakePageIn, 1
        iretd

//=========================================================================================

return_from_int1:

        mov     MakePageIn, 0
        mov     ReturnToMakePageIn, 0

        mov     eax, dword ptr [PreLastEIPValue]     //check if "pagein" success
        add     eax, 2
        cmp     eax, dword ptr [esp]
        mov     esp, dword ptr [saved_esp_end]
        jz      pagein_success
        xor     eax, eax
        mov     dword ptr [esp + 34h], eax

pagein_success:
        mov     eax, dword ptr [PreLastEIPValue]     //restore user bytes
        mov     edx, dword ptr [saved_user_bytes]
        mov     word ptr [eax], dx

        push    SavedCR0
        call    SetCR0

        cld                                          //we must to restore original interrupt stack
        mov     ecx, 5                               //because it was changed by last interrupt
        mov     edi, OldInterruptStack
        mov     esi, offset NewInterruptStack + 0x1000 - 0x10
        repnz   movsd

page_in_memory:
        pop     gs
        pop     es
        pop     ds
        pop     fs
        popad
        mov     eax, dword ptr [esp + 4]
        ret     4
    }
}

///////////////////////////////////////////////////////////////////////
//
//  function CheckAddressAndMakePageIn - make "PAGEIN"
//
//  this version of function using patch TSS is not work good!
//  because in one of 100000 it's not save own stack! ...very bad :(
//  why i leave it? - may be you would have any additional ideas!?...
//
///////////////////////////////////////////////////////////////////////

//__declspec(naked) void* CheckAddressAndMakePageIn(void *address)
//{
//    UNREFERENCED_PARAMETER(address);
//
//    static int original_esp0        = 0;
//    static int saved_esp_beg        = 0;
//    static int saved_esp_end        = 0;
//    static int intterupt_stack[5]   = {0};
//    static int saved_user_bytes[2]  = {0};
//    static __int64 temp_old_descrp  = 0;
//    static __int64 temp_new_descrp  = 0;
//
//    __asm
//    {
//        pushad
//        push    fs
//        push    ds
//        push    es
//        push    gs
//
//        cmp     dword ptr [esp + 0x34], 0x1000
//        jb      bad_pointer
//
//        cmp     dword ptr [esp + 0x34], NT_HIGHEST_USER_ADDRESS
//        jb      make_page_in
//
//        push    dword ptr [esp + 0x34]
//        call    GetPte
//        mov     eax, dword ptr [eax]
//        test    al, 1
//        jnz     page_in_memory
//
//        // here need to make PAGEIN for page in range > NT_HIGHEST_USER_ADDRESS
//        // not support yet...
//
//bad_pointer:
//        pop     gs
//        pop     es
//        pop     ds
//        pop     fs
//        popad
//        xor     eax, eax
//        ret     4
//
//make_page_in:
//        mov     dword ptr [saved_esp_beg], esp
//
//        mov     eax, dword ptr [esp + 0x34]          //get [address] parameter
//        mov     ebx, eax
//        push    eax
//        call    GetPte
//        mov     eax, dword ptr [eax]
//        test    al, 1
//        jnz     page_in_memory
//
//        mov     eax, dword ptr [PreLastEIPValue]
//
//        mov     edx, eax
//        and     edx, 0FFFh                           //check if last eip == 0x...FFF
//        cmp     edx, 0FFBh                           //and next page may be "page out"
//        jbe     eip_in_range
//        sub     eax, 5
//eip_in_range:
//
//        mov     edx, dword ptr [eax]                 //save user bytes
//        mov     dword ptr [saved_user_bytes], edx
//        mov     dl, byte ptr [eax + 4]
//        mov     byte ptr [saved_user_bytes + 4], dl
//
//        push    eax
//        call    EnableWrite
//        mov     SavedCR0, eax
//        pop     eax
//
//        mov     byte ptr [eax], 0xA0                 //patch user code with
//        mov     dword ptr [eax + 1], ebx             //mov al, dword [0xXXX...]
//
//        mov     ecx, esp                             //stack value for saving in TSS
//
//        //===========================================
//
//        push    ds                                   //patch current TSS with new ESP0 value
//        push    gs
//        mov     ax, 23h
//        mov     ds, ax
//        mov     ax, 10h
//        mov     gs, ax
//        sub     esp, 08h
//
//        sgdt    fword ptr [esp]
//
//        str     ax                                   //get TSS selector
//        movzx   eax, ax
//        add     eax, dword ptr [esp + 2]
//
//        mov     edx, dword ptr [eax]
//        mov     dword ptr [temp_new_descrp], edx
//        mov     edx, dword ptr [eax + 4]
//        mov     dword ptr [temp_new_descrp + 4], edx
//
//        mov     ax, 20h                              //means 23h selector
//        movzx   eax, ax
//        add     eax, dword ptr [esp + 2]
//
//        mov     edx, dword ptr [eax]
//        mov     dword ptr [temp_old_descrp], edx
//        mov     edx, dword ptr [eax + 4]
//        mov     dword ptr [temp_old_descrp + 4], edx
//
//        shr     edx, 8
//        mov     byte ptr [temp_new_descrp + 4 + 1], dl
//        shr     edx, 8
//        and     dl, 11000000b
//        mov     byte ptr [temp_new_descrp + 4 + 2], dl
//
//        mov     edx, dword ptr [temp_new_descrp]     //patch GDT with new descriptor
//        mov     dword ptr [eax], edx
//        mov     edx, dword ptr [temp_new_descrp + 4]
//        mov     dword ptr [eax + 4], edx
//        
//        mov     dx, 23h                              //reload shadow part in ds
//        mov     ds, dx
//
//        mov     edx, dword ptr ds:[4]
//        mov     dword ptr gs:[original_esp0], edx
//        mov     dword ptr ds:[4], ecx                //patch ESP0 value at TSS
//
//        mov     edx, dword ptr gs:[temp_old_descrp]  //patch back GDT with old descriptor
//        mov     dword ptr gs:[eax], edx
//        mov     edx, dword ptr gs:[temp_old_descrp + 4]
//        mov     dword ptr gs:[eax + 4], edx
//
//        mov     dx, 23h
//        mov     ds, dx
//
//        add     esp, 08h
//        pop     gs
//        pop     ds
//
//        //===========================================
//
//        mov     eax, dword ptr [original_esp0]
//        push    dword ptr [eax - 4]                  //need to save the old interrupt stack for
//        push    dword ptr [eax - 8]                  //further recollection, because the system
//        push    dword ptr [eax - 0Ch]                //in int0xE(PageFault) handler will change it
//        push    dword ptr [eax - 10h]                //later (don't know why exactly...)
//        push    dword ptr [eax - 14h]
//
//        pop     dword ptr [intterupt_stack + 10h]
//        pop     dword ptr [intterupt_stack + 0Ch]
//        pop     dword ptr [intterupt_stack + 8]
//        pop     dword ptr [intterupt_stack + 4]
//        pop     dword ptr [intterupt_stack + 0]
//        
//        push    23h
//        push    dword ptr [intterupt_stack + 4]      //prepare own interrupt stack
//        push    00010306h
//        push    1Bh
//        push    dword ptr [PreLastEIPValue]
//
//        mov     dword ptr [saved_esp_end], esp
//        mov     ReturnToMakePageIn, offset return_from_int1
//        mov     MakePageIn, 1
//        iretd
//
////=========================================================================================
//
//return_from_int1:
//
//        mov     MakePageIn, 0
//        mov     ReturnToMakePageIn, 0
//
//        mov     eax, dword ptr [PreLastEIPValue]
//        add     eax, 5
//        cmp     eax, dword ptr [esp]
//        jz      pagein_success
//        mov     esp, dword ptr [saved_esp_beg]
//        xor     eax, eax
//        mov     dword ptr [esp + 34h], eax
//pagein_success:
//        mov     esp, dword ptr [saved_esp_end]
//        
//        //===========================================
//
//        push    ds                                   //restore ESP0 in TSS to original value
//        push    gs
//        mov     ax, 23h
//        mov     ds, ax
//        mov     ax, 10h
//        mov     gs, ax
//        sub     esp, 08h
//
//        sgdt    fword ptr [esp]
//
//        mov     ax, 20h                              //means 23h selector
//        movzx   eax, ax
//        add     eax, dword ptr [esp + 2]
//
//        mov     edx, dword ptr [temp_new_descrp]     //patch GDT with new descriptor
//        mov     dword ptr [eax], edx
//        mov     edx, dword ptr [temp_new_descrp + 4]
//        mov     dword ptr [eax + 4], edx
//        
//        mov     dx, 23h
//        mov     ds, dx
//
//        mov     edx, dword ptr gs:[original_esp0]
//        mov     dword ptr ds:[4], edx
//
//        mov     edx, dword ptr gs:[temp_old_descrp]  //patch GDT back
//        mov     dword ptr gs:[eax], edx
//        mov     edx, dword ptr gs:[temp_old_descrp + 4]
//        mov     dword ptr gs:[eax + 4], edx
//
//        mov     dx, 23h                              //reload segment register
//        mov     ds, dx
//
//        add     esp, 08h
//        pop     gs
//        pop     ds
//
//        //===========================================
//
//        mov     eax, dword ptr [PreLastEIPValue]
//        mov     edx, dword ptr [saved_user_bytes]    //restore user bytes
//        mov     dword ptr [eax], edx
//        mov     dl, byte ptr [saved_user_bytes + 4]
//        mov     byte ptr [eax + 4], dl
//
//        push    SavedCR0
//        call    SetCR0
//
//        mov     ebx, dword ptr [original_esp0]       //restore old interrupt stack
//
//        mov     eax, dword ptr [intterupt_stack + 0]
//        mov     dword ptr [ebx - 4], eax
//        mov     eax, dword ptr [intterupt_stack + 4]
//        mov     dword ptr [ebx - 8], eax
//        mov     eax, dword ptr [intterupt_stack + 8]
//        mov     dword ptr [ebx - 0Ch], eax
//        mov     eax, dword ptr [intterupt_stack + 0Ch]
//        mov     dword ptr [ebx - 10h], eax
//        mov     eax, dword ptr [intterupt_stack + 10h]
//        mov     dword ptr [ebx - 14h], eax
//
//        mov     esp, dword ptr [saved_esp_beg]
//
//page_in_memory:
//        pop     gs
//        pop     es
//        pop     ds
//        pop     fs
//        popad
//        mov     eax, dword ptr [esp + 4]
//        ret     4
//    }
//}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void* GetUserHandler(void)
{
    void* handler = 0;
    
    __asm
    {
        push    fs
        mov     ax, 38h
        mov     fs, ax
        mov     dword ptr [handler], 0
        mov     eax, dword ptr fs:[0]
        add     eax, 4

        push    eax
        call    CheckAddressAndMakePageIn
        or      eax, eax
        jz      get_handler_exit

        mov     eax, dword ptr [eax]
        push    eax
        call    CheckAddressAndMakePageIn
        or      eax, eax
        jz      get_handler_exit

        mov     dword ptr [handler], eax

get_handler_exit:
        pop     fs
    }
    
    return handler;
}