//
//  PlugIn name: IDA to SoftIce converter (only for 32bit programs)
//
//
//  This file contains function that deal with types (set/get).
//
//
#include "i2s.h"

//--------------------------------------------------------------------------
//This function updates/sets all variables common to NM32_xxxx functions.
//
// size - size of nm32_type_xxxx struct.
//
// return - type index.
//
unsigned long ida2softice_c::NMTP_UpdateNMTPStruct
(
	unsigned long	size
)
{
	unsigned long	index;

//Set offset to current NM type and update variables.
	*nmtp.pTypesOffsets_current = nmtp.pTypesData_current - nmtp.pTypesData;
	++nmtp.pTypesOffsets_current;
	nmtp.freeOffsets -= sizeof( unsigned long );

//Update data memory.
	nmtp.pTypesData_current += size;
	nmtp.freeData -= size;

//Get current index.
	index = 0x1000 + nmtp.nTypes;
	
//Increase number of types in nmtp structure.
	++nmtp.nTypes;

	return index;
}

//--------------------------------------------------------------------------
//The SeArray function currently sets one dimension nm32 array type.
//
//	elemType - Primitive type. Currently it must be T_CHAR.
//  length - length of one dimension array in bytes.
//
//  return - type index or STATUS_FAILED.
//
unsigned long ida2softice_c::NMTP_SetArray
(
	unsigned long	elemType,
	unsigned long	nElements,
	unsigned long	length
)
{
	STATUS				Status;

	DATAS_LF_ARRAY*		pTypeData;

//Do we have the same array type already in memory?
	//2 because first 2 predefined types are not needed to be scaned
	for ( unsigned long i = 2; i < nmtp.nTypes; i++ )
	{
	//Get current type.
		pTypeData = (DATAS_LF_ARRAY*)( nmtp.pTypesData + nmtp.pTypesOffsets[i] );

	//Is current type an array type?
		if ( pTypeData->lf_index == TYPE_LF_ARRAY )
		{
		//Yes it's an array -> does it have the same length?
			if ( pTypeData->length == length )
			{
			//Yes they are the same length -> but do they have the same element type?
				if ( pTypeData->elemType == elemType )
				{
				//Yes -> current array type is already in memory.
					return ( 0x1000 + i );
				}
			}
		}
	}

//Current array is not in memory yet -> check first if we have enough memory.
	if ( nmtp.freeData <= sizeof( DATAS_LF_ARRAY ) )
	{
		Status = ReallocateMemory(
		  ALLOCATE_BIG,
		  nmtp.hTypesData,
		  (void**)&(nmtp.pTypesData),
		  (void**)&(nmtp.pTypesData_current),
		  &nmtp.freeData,
		  &nmtp.sizeData
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	if ( nmtp.freeOffsets <= sizeof( unsigned long )  )
	{
		Status = ReallocateMemory(
		  ALLOCATE_SMALL,
		  nmtp.hTypesOffsets,
		  (void**)&(nmtp.pTypesOffsets),
		  (void**)&(nmtp.pTypesOffsets_current),
		  &nmtp.freeOffsets,
		  &nmtp.sizeOffsets
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	
//Set new type.
	pTypeData = (DATAS_LF_ARRAY*)nmtp.pTypesData_current;
	pTypeData->lf_index = TYPE_LF_ARRAY;
	pTypeData->elemType = elemType;
	pTypeData->length = length;
	pTypeData->nElements = nElements;
	pTypeData->indexType = TP_ULONG;

//Update NMTP memory.
	return NMTP_UpdateNMTPStruct( sizeof( DATAS_LF_ARRAY ) );
}

//--------------------------------------------------------------------------
//The SetArglist function sets types of arguments passed to the procedure.
//
// pFrame - Frame of the procedure. ( struc_t *get_frame(func_t *pfn) )
// argCount - Number of arguments passed to procedure. (look in NMTP_SetProcedure)
// argOffsetFirst - Offset in frame to first argument.
//
// return - type index or STATUS_FAILED.
//
unsigned long ida2softice_c::NMTP_SetArglist
(
	struc_t*		pFrame,
	unsigned long	argCount,
	unsigned long	argOffset
)
{
	STATUS			Status;

	unsigned long	index_member;
	unsigned long	typeSize;
	member_t*		pMember;
	unsigned long*	pTypeData;

//Get size of current NM type.
	typeSize = sizeof( DATAS_LF_ARGLIST ) + argCount * sizeof( unsigned long );

//Do we have enough memory for current type?
	if ( nmtp.freeData <= typeSize )
	{
		Status = ReallocateMemory(
		  ALLOCATE_BIG,
		  nmtp.hTypesData,
		  (void**)&(nmtp.pTypesData),
		  (void**)&(nmtp.pTypesData_current),
		  &nmtp.freeData,
		  &nmtp.sizeData
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	if ( nmtp.freeOffsets <= sizeof( unsigned long )  )
	{
		Status = ReallocateMemory(
		  ALLOCATE_SMALL,
		  nmtp.hTypesOffsets,
		  (void**)&(nmtp.pTypesOffsets),
		  (void**)&(nmtp.pTypesOffsets_current),
		  &nmtp.freeOffsets,
		  &nmtp.sizeOffsets
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}

//Set new type.
	pTypeData = (unsigned long*)nmtp.pTypesData_current;
	( (DATAS_LF_ARGLIST*)pTypeData )->lf_index = TYPE_LF_ARGLIST;
	( (DATAS_LF_ARGLIST*)pTypeData )->nTypes = argCount;

	//Set argument types.
	if ( argCount != 0 )
	{
	//Go to first argument type position.
		pTypeData += 3;
		while( argOffset != BADADDR )
		{
		//Get argument data.
			pMember = get_member( pFrame, argOffset );
			//Did we get it?
			if ( pMember != NULL )
			{
			//Yes -> set type.
				index_member = GetType_flag( pMember->flag );
				if ( index_member == TP_NOTYPE )
				{
					msg( "I2S: Failed to get argument type.\n" );
					return STATUS_FAILED;
				};
				*pTypeData = index_member;
			
			//Update variables.
				++pTypeData;
				--argCount;
				if ( argCount == 0 )
				{
					break;
				}
			}

		//Go to next argument.
			argOffset = get_struc_next_offset( pFrame, argOffset );
		}

	//Check if have define all the arguments. If not return error.
		if ( argCount > 0 )
		{
			msg( "I2S: Failed to get all arguments.\n" );
			return STATUS_FAILED;
		}
	}

//Update NMTP memory.
	return NMTP_UpdateNMTPStruct( typeSize );
}

//--------------------------------------------------------------------------
//The SetProcedure function sets global function type.
//
// pProcedure - Function struct. (check funcs.hpp)
//
// return - type index.
//
// Notice!
// Local procedures "don't" need this type set(well they have it -> 0x1001 (argCount=0) ).
// Their arguments are taken in account in SYMD section as local variables
// (with EBP+offset).
//
unsigned long ida2softice_c::NMTP_SetProcedure
(
	func_t*	pProcedure
) 
{
	STATUS					Status;

	char					functionName[MAXSTR];
	DATAS_LF_PROCEDURE*		pTypeData;
	unsigned long			index;
	unsigned long			index_argList;

	unsigned long			argCount;
	unsigned long*			pArglistType;
	unsigned long			argOffset;
	unsigned long			argOffsetFirst;
	struc_t*				pFrame;
	member_t*				pMember;
	unsigned long			type;


//Get procedure's number of arguments and the offset to the first argument.
	argCount = SYMD_GetProcedureArgCountAndFirstOffset( pProcedure, &argOffsetFirst );

//Test if we have procedure with same number of arguments already in our buffer.
	argOffset = argOffsetFirst;
	pFrame = get_frame( pProcedure );
		//2 because first 2 predefined types are not needed to be scaned
	for ( unsigned long i = 2; i < nmtp.nTypes; i++ )
	{
	//Get current type.
		pTypeData = (DATAS_LF_PROCEDURE*)( nmtp.pTypesData + nmtp.pTypesOffsets[i] );

	//Is current type a procedure type?
		if ( pTypeData->lf_index == TYPE_LF_PROCEDURE )
		{
		//Yes it's a procedure -> does it have the same number of arguments?
			if ( pTypeData->argCount == argCount )
			{
			//Yes they are the same number of arguments ->
			//but do they have the same argument type?
			//
			//Go to first type.
				pArglistType = (unsigned long*)(
				  (unsigned char*)pTypeData
				  +
				  sizeof( DATAS_LF_PROCEDURE )
				  +
				  sizeof( DATAS_LF_ARGLIST )	//argList is always behind procedure in i2s
				);

			//Scan arguments.
				unsigned long	nArgumentsScaned = 0;
				while ( argOffset != BADADDR )
				{
				//Get argument.
					pMember = get_member( pFrame, argOffset );
					//Did we get it?
					if ( pMember != NULL )
					{
					//Yes we got it -> get it's type.
						type = GetType_flag( pMember->flag );

					//Is it the same as arguments list one?
						if ( *pArglistType != type )
						{
						//Not the same -> stop detection.
							break;
						}

					//It's the same -> go to next argument -> update variables.
						++pArglistType;
						++nArgumentsScaned;
						//Did we check all the arguments?
						if ( nArgumentsScaned == argCount )
						{
						//Yes we did and all the arguments were the same ->
						//return current index.
							return ( 0x1000 + i );
						}
					}

				//Update.
					argOffset = get_struc_next_offset( pFrame, argOffset );
				}
			}
		}
	}

//Current procedure is not in memory yet -> check first if we have enough memory.
	if ( nmtp.freeData <= sizeof( DATAS_LF_PROCEDURE ) )
	{
		Status = ReallocateMemory(
		  ALLOCATE_BIG,
		  nmtp.hTypesData,
		  (void**)&(nmtp.pTypesData),
		  (void**)&(nmtp.pTypesData_current),
		  &nmtp.freeData,
		  &nmtp.sizeData
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	if ( nmtp.freeOffsets <= sizeof( unsigned long )  )
	{
		Status = ReallocateMemory(
		  ALLOCATE_SMALL,
		  nmtp.hTypesOffsets,
		  (void**)&(nmtp.pTypesOffsets),
		  (void**)&(nmtp.pTypesOffsets_current),
		  &nmtp.freeOffsets,
		  &nmtp.sizeOffsets
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}

//Set new type.
	pTypeData = (DATAS_LF_PROCEDURE*) nmtp.pTypesData_current;
	//We need to update the index here as we'll set new arglist also.
	index = NMTP_UpdateNMTPStruct( sizeof(DATAS_LF_PROCEDURE) );

	//Set DATAS_LF_PROCEDURE.
	pTypeData->lf_index = TYPE_LF_PROCEDURE;
	index_argList = NMTP_SetArglist( pFrame, argCount, argOffsetFirst );
	if ( S_FAILED( index_argList ) )
	{
		return STATUS_FAILED;	//It's the same as STATUS_FAILED
	}
	pTypeData->argList = index_argList;
		//It's ok for all functions.
	pTypeData->rType = TP_VOID;
		//Arguments pushed left to right, procedure pops arguments.
	pTypeData->call = CALL_FAR_STDCALL;
	pTypeData->argCount = argCount;

//Set TYTB and HSHT tables.
	//Get function name.
	get_func_name( pProcedure->startEA, (char*)&functionName, MAXSTR );
	//Set tables.
	Status = Set_TYTB_HSHT( index, (char*)&functionName, STATUS_ERROR );
	if ( S_FAILED( Status ) )
	{
		return STATUS_FAILED;
	}

	return index;
}

//--------------------------------------------------------------------------
//The SetMemeber function sets the member index even if member is a
//struct type or union variable.
//
// member - Member structure. (check struct.hpp)
// memberSize - Size of a member.
//
// return - type index or STATUS_FAILED.
//
unsigned long ida2softice_c::NMTP_SetMember
(
	member_t*		pMember,
	unsigned long	memberSize
)
{
	STATUS				Status;

	unsigned long		index;
	unsigned long		indexType;
	unsigned long		iName;
	char				memberName[128];
	DATAS_LF_MEMBER*	pTypeData;

//Do we have enough memory?
	if ( nmtp.freeData <= sizeof( DATAS_LF_MEMBER ) )
	{
		Status = ReallocateMemory(
		  ALLOCATE_BIG,
		  nmtp.hTypesData,
		  (void**)&(nmtp.pTypesData),
		  (void**)&(nmtp.pTypesData_current),
		  &nmtp.freeData,
		  &nmtp.sizeData
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	if ( nmtp.freeOffsets <= sizeof( unsigned long )  )
	{
		Status = ReallocateMemory(
		  ALLOCATE_SMALL,
		  nmtp.hTypesOffsets,
		  (void**)&(nmtp.pTypesOffsets),
		  (void**)&(nmtp.pTypesOffsets_current),
		  &nmtp.freeOffsets,
		  &nmtp.sizeOffsets
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}

//Set/Update
	pTypeData = (DATAS_LF_MEMBER*)nmtp.pTypesData_current;
	//We need to update the index here as member can be a structure too.
	index = NMTP_UpdateNMTPStruct( sizeof(DATAS_LF_MEMBER) );

	//Set DATAS_LF_MEMBER.
	pTypeData->lf_index = TYPE_LF_MEMBER;
	indexType = GetType_member( pMember, memberSize );
	if ( S_FAILED( indexType ) )
	{
		return STATUS_FAILED;
	}
	pTypeData->indexType = indexType;
	get_member_name( pMember->id, memberName, sizeof( memberName ) );
	iName = STTB_SetStringTable( memberName );
	if ( iName == STATUS_ERROR )
	{
		return STATUS_FAILED;
	}
	pTypeData->iName = iName;
	pTypeData->access = ACCESS_PUBLIC;
	//Is current member a member of an union?
	if ( pMember->unimem() )
	{
	//Yes it's a member of an union -> all variables in union have starting
	//offset of 0.
		pTypeData->offset = 0;
	}
	else
	{
	//No it's not an union member -> get it's offset in structure.
		pTypeData->offset = (unsigned short)pMember->soff;
	}

	return index;
}


//--------------------------------------------------------------------------
// The SetDerived function defines all members of the structure.
//
// Notice:
// Undefined bytes will be skipped and will not be defined in the derived
// structure. This means the data wont be defined in SoftIce.
//
// pStructure - The pointer to structure structure. ;) (check struct.hpp)
// structSize - Same variable as supplied to NMTP_SetStructure.
//
// return - type index or STATUS_FAILED.
//
unsigned long ida2softice_c::NMTP_SetDerived
(
	struc_t*		pStructure,
	unsigned long	structSize
)
{
	STATUS			Status;

	unsigned long	index;
	unsigned long	index_member;
	unsigned long*	pTypeData;
	unsigned long	typeSize;

	unsigned long	memberOffset;
	unsigned long	memberSize;
	member_t*		pMember;


//Get size of current NM type.
	typeSize = sizeof(DATAS_LF_DERIVED) + pStructure->memqty * sizeof( unsigned long );

//Do we have enough memory for current type?
	if ( nmtp.freeData <= typeSize )
	{
		Status = ReallocateMemory(
		  ALLOCATE_BIG,
		  nmtp.hTypesData,
		  (void**)&(nmtp.pTypesData),
		  (void**)&(nmtp.pTypesData_current),
		  &nmtp.freeData,
		  &nmtp.sizeData
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	if ( nmtp.freeOffsets <= sizeof( unsigned long )  )
	{
		Status = ReallocateMemory(
		  ALLOCATE_SMALL,
		  nmtp.hTypesOffsets,
		  (void**)&(nmtp.pTypesOffsets),
		  (void**)&(nmtp.pTypesOffsets_current),
		  &nmtp.freeOffsets,
		  &nmtp.sizeOffsets
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}

//Set new type.
	pTypeData = (unsigned long*)nmtp.pTypesData_current;
	//We need to update the index here as members can be a any type of variables.
	index = NMTP_UpdateNMTPStruct( typeSize );

	//Set DATAS_LF_DERIVED.
	( (DATAS_LF_DERIVED*)pTypeData )->lf_index = TYPE_LF_DERIVED;
	( (DATAS_LF_DERIVED*)pTypeData )->nTypes = pStructure->memqty;

	//Set member types.
		//Go to first "member" type.
	pTypeData += 3;
	memberOffset = get_struc_first_offset( pStructure );

	unsigned long memqty = pStructure->memqty;
	while ( memqty > 0 )
	{
	//Don't move this test to lower part aslast member of the
	//variable length structure returns BADADDR.
		if ( memberOffset == BADADDR )
		{
			msg( "I2S: Problems getting structure members.\n" );
			return STATUS_FAILED;
		}

	//Get member.
		pMember = get_member( pStructure, memberOffset );
		//Did we get the member?
		if ( pMember != NULL )
		{
		//We got the member.
		//
		//Get index to the member type.
			memberSize = get_member_size( pMember );
			if ( memberSize == 0 )
			{
			//This SHOULD be the last parameter in the variable length structure.
				if ( structSize < memberOffset )
				{
					msg( "I2S: I2S get negative value for structure memeber.\n" );
					return STATUS_FAILED;
				}
				index_member = NMTP_SetMember( pMember, structSize - memberOffset );
			}
			else
			{
			//This is a normal member.
				index_member = NMTP_SetMember( pMember, memberSize );
			}
			if ( S_FAILED( index_member ) )
			{
				return STATUS_FAILED;
			}
			*pTypeData = index_member;

		//Update.
			++pTypeData;
			--memqty;
		}

	//Update.
		memberOffset = get_struc_next_offset( pStructure, memberOffset );
	}
	return index;
}
		
//--------------------------------------------------------------------------
//The SetStructure(Union) function sets all needed parts
//(DATAS_LF_STRUCT,..DERIVED,..MEMBERs) to define NM32 structure type.
//
// structure - The structure structure. ;) (check struct.hpp)
// size - Size of whole structure.
//
//        You can't get the size of variable length structures with the
//        get_struc_size(..) call. Because of this the size must be supplied by
//        the caller as it has access to other structure data.
//
//        Notice:
//        This value must be equal to get_struc_size(..) for normal structures and
//        bigger for varStructs.
//
// return - type index or STATUS_FAILED.
//
unsigned long ida2softice_c::NMTP_SetStructure
(
	struc_t*		pStructure,
	unsigned long	structSize
)
{
	STATUS				Status;

	unsigned long		index;
	unsigned long		index_derived;
	unsigned long		iName;
	DATAS_LF_STRUCT*	pTypeData;
	char				structName[MAXSTR];

//Test if we have any members.
	if ( pStructure->memqty == 0 )
	{
		get_struc_name( pStructure->id, structName, sizeof( structName ) );
		warning( "Structure(%s) doesn't have any members.", (char*)&structName );
		return STATUS_FAILED;
	}

//Check the struct size.
	if ( ( pStructure->props & SF_VAR ) != 0 )
	{
	//We have a variable length structure -> size must be bigger than
	//that of get_struc_size(..).
		if ( structSize <= get_struc_size( pStructure ) )
		{
			msg( "I2S: Internal error please contact author.\n" );
			return STATUS_FAILED;
		}
	}
	else
	{
	//We have a normaly structure -> size must be equal...
		if ( structSize != get_struc_size( pStructure ) )
		{
			msg( "I2S: Internal error please contact author.\n" );
			return STATUS_FAILED;
		}
	}

//Test if we have the same struct already in our buffer.
//
//Take notice that variable length structures have the length appended to their tail.
	if ( ( pStructure->props & SF_VAR ) != 0 )
	{
	//It's a variable length structure.
		char	structName_temp[128];
		get_struc_name( pStructure->id, structName_temp, sizeof( structName_temp ) );

		qsnprintf( structName, sizeof( structName ), "%s%d", (char*)&structName_temp, structSize );
	}
	else
	{
	//It's a normal structure.
		get_struc_name( pStructure->id, structName, sizeof( structName ) );
	}
	iName = STTB_SetStringTable( (char*)&structName );
	if ( iName == STATUS_ERROR )
	{
		return STATUS_FAILED;
	}
	//Do we have this structure already in memory?
	//
	//Is there a structure with the same name already in our table?
	//(Two structures can't have the same name.)
		//2 because first 2 predefined types are not needed to be scaned
	for ( unsigned long i = 2; i< nmtp.nTypes; i++ )
	{
	//Get type.
		pTypeData = (DATAS_LF_STRUCT*)( nmtp.pTypesData + nmtp.pTypesOffsets[i] );

	//Is current type a structure type?
		if ( pTypeData->lf_index == TYPE_LF_STRUCT )
		{
		//Yes current type is a structure type -> does it have the same name?
			if ( pTypeData->iName == iName )
			{
			//Yes the names are the same -> return index.
				return ( 0x1000 + i );
			}
		}
	}

//Current array is not in memory yet -> check first if we have enough memory.
	if ( nmtp.freeData <= sizeof( DATAS_LF_STRUCT ) )
	{
		Status = ReallocateMemory(
		  ALLOCATE_BIG,
		  nmtp.hTypesData,
		  (void**)&(nmtp.pTypesData),
		  (void**)&(nmtp.pTypesData_current),
		  &nmtp.freeData,
		  &nmtp.sizeData
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	if ( nmtp.freeOffsets <= sizeof( unsigned long )  )
	{
		Status = ReallocateMemory(
		  ALLOCATE_SMALL,
		  nmtp.hTypesOffsets,
		  (void**)&(nmtp.pTypesOffsets),
		  (void**)&(nmtp.pTypesOffsets_current),
		  &nmtp.freeOffsets,
		  &nmtp.sizeOffsets
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}
	
//Set new type.
	pTypeData = (DATAS_LF_STRUCT*)nmtp.pTypesData_current;
	//We need to update the index here as derived structure will get next index.
	index = NMTP_UpdateNMTPStruct( sizeof( DATAS_LF_STRUCT ) );

	//Set DATAS_LF_STRUCTURE.
	pTypeData->lf_index = TYPE_LF_STRUCT;
	index_derived = NMTP_SetDerived( pStructure, structSize );
	if ( S_FAILED( index_derived ) )
	{
		get_struc_name( pStructure->id, structName, sizeof( structName ) );
		warning( "PlugIn failed to set structure (%s).\n", (char*)&structName );
		return STATUS_FAILED;
	}
	pTypeData->derivedType = index_derived;
	pTypeData->iName = iName;
	pTypeData->property = PROPERTY_NULL;
	pTypeData->nFields = pStructure->memqty;
	pTypeData->length = structSize;

//Set TYTB and HSHT tables.
	Status = Set_TYTB_HSHT( index, (char*)&structName, iName );
	if ( S_FAILED( Status ) )
	{
		return Status;
	}

	return index;
}

//--------------------------------------------------------------------------
//Get type.
//
//Notice:
//The procedure doesn't support all types.
//Just next ones:
//byte, word, dword, qword, float, double, packreal, octaword
//
//So this procedure is intended only for requiring argument types.
//
//
// flag - Isn't it obvious.
//
// return - type OR TP_NOTYPE(same as STATUS_FAILED).
//
unsigned long ida2softice_c::GetType_flag(flags_t flag)
{
	flag = flag & DT_TYPE;
	     if ( flag == FF_BYTE     ) return TP_UCHAR;
	else if ( flag == FF_WORD     ) return TP_USHORT;
	else if ( flag == FF_DWRD     ) return TP_ULONG;
	else if ( flag == FF_QWRD     ) return TP_UQUAD;
	else if ( flag == FF_OWRD     ) return TP_REAL128;
	else if ( flag == FF_FLOAT    ) return TP_REAL32;
	else if ( flag == FF_DOUBLE   ) return TP_REAL64;
	else if ( flag == FF_PACKREAL ) return TP_REAL80;		//????
	return TP_NOTYPE;
}

//Supports all types (intended for DATA segment).
type_t		typeBuffer[MAXSTR];
p_list		fnames[MAXSTR];
unsigned long ida2softice_c::GetType_ea
(
	ea_t	address
)
{
	unsigned long	dataSize;
	unsigned long	dataSize_type;
	flags_t			flag;		//just the type flag
	flags_t			flags;		//all flags
	unsigned long	nElements;
	typeinfo_t		typeInfo;

//Get variable type.
	flags = getFlags( address );
	flag = flags & DT_TYPE;

//Is current variable an array? Be careful of variable length structures.
	dataSize = get_item_size( address );

	if ( flag == FF_STRU )
	{
		struc_t*	pStructure;

		get_typeinfo( address, 0, flags, &typeInfo );
		pStructure = get_struc( typeInfo.tid );

		if ( ( pStructure->props & SF_VAR ) != 0 )
		{
		//It's a variable length structure.
			nElements = 1;
		}
		else
		{
		//It's a normal struct.
			goto normalStruct;
		}
	}
	else
	{
	normalStruct:
		dataSize_type = get_data_type_size_by_flags( address, flags );
		nElements = dataSize / dataSize_type;
	}

//Get type.
	if ( nElements > 1 )
	{
	//It's array.
	//
	//Set type.
			 if ( flag == FF_BYTE ) return NMTP_SetArray( TP_UCHAR, nElements, dataSize );
		else if ( flag == FF_WORD ) return NMTP_SetArray( TP_USHORT, nElements, dataSize );
		else if ( flag == FF_DWRD ) return NMTP_SetArray( TP_ULONG, nElements, dataSize );
		else if ( flag == FF_QWRD ) return NMTP_SetArray( TP_UQUAD, nElements, dataSize );
		else if ( flag == FF_TBYT ) return NMTP_SetArray( TP_REAL80, nElements, dataSize );
		else if ( flag == FF_ASCI )
		{
		//Get type of string.
			switch ( dataSize_type )
			{
			case 1 :
			default :
			//ASCSTR_C :
			//ASCSTR_TERMCHR :
			//ASCSTR_PASCAL :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_UCHAR, nElements, dataSize );

			case 2 :
			//ASCSTR_UNICODE :
			//ASCSTR_ULEN2 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_WCHAR, nElements, dataSize );

			case 4 :
			//ASCSTR_LEN4 :
			//ASCSTR_ULEN4 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_ULONG, nElements, dataSize );
			}
		}
		else if ( flag == FF_STRU )
		{
			unsigned long index;

		//Set structure.
			get_typeinfo( address, 0, flags, &typeInfo );
			index = NMTP_SetStructure( get_struc( typeInfo.tid ), dataSize_type );
			if ( S_FAILED( index ) )
			{
				return STATUS_FAILED;
			}

		//Set array of structures.
			return NMTP_SetArray( index, nElements, dataSize );
		}
		else if ( flag == FF_OWRD     ) return NMTP_SetArray( TP_REAL128, nElements, dataSize );
		else if ( flag == FF_FLOAT    ) return NMTP_SetArray( TP_REAL32, nElements, dataSize );
		else if ( flag == FF_DOUBLE   ) return NMTP_SetArray( TP_REAL64, nElements, dataSize );
		else if ( flag == FF_PACKREAL ) return NMTP_SetArray( TP_REAL80, nElements, dataSize );		//????
		return TP_NOTYPE;  //Same as STATUS_FAILED.
	}
	else
	{
	//It's not array.
			 if ( flag == FF_BYTE ) return TP_UCHAR;
		else if ( flag == FF_WORD ) return TP_USHORT;
		else if ( flag == FF_DWRD ) return TP_ULONG;
		else if ( flag == FF_QWRD ) return TP_UQUAD;
		else if ( flag == FF_TBYT ) return TP_REAL80;
		else if ( flag == FF_ASCI )
		{
		//Get type of string.
			switch ( dataSize_type )
			{
			case 1 :
			default :
			//ASCSTR_C :
			//ASCSTR_TERMCHR :
			//ASCSTR_PASCAL :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_UCHAR, nElements, dataSize );

			case 2 :
			//ASCSTR_UNICODE :
			//ASCSTR_ULEN2 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_WCHAR, nElements, dataSize );

			case 4 :
			//ASCSTR_LEN4 :
			//ASCSTR_ULEN4 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_ULONG, nElements, dataSize );
			}
		}
		else if ( flag == FF_STRU )
		{
		//Set structure and return it's index.
			get_typeinfo( address, 0, flags, &typeInfo );
			return NMTP_SetStructure( get_struc( typeInfo.tid ), dataSize );
		}
		else if ( flag == FF_OWRD     ) return TP_REAL128;
		else if ( flag == FF_FLOAT    ) return TP_REAL32;
		else if ( flag == FF_DOUBLE   ) return TP_REAL64;
		else if ( flag == FF_PACKREAL ) return TP_REAL80;		//????
		return TP_NOTYPE;  //Same as STATUS_FAILED.
	}
}

//supports all types (intended for argument/lVariables (procedure frame)).
unsigned long ida2softice_c::GetType_member
(
	member_t*		pMember,
	unsigned long	memberSize
)
{
	flags_t			flag;		//just the type flag
	unsigned long	memberSize_type;
	unsigned long	nElements;
	struc_t*		pStruct;
	typeinfo_t		typeInfo;

//Initialize.
	flag = pMember->flag & DT_TYPE;

//Is current member an array?
	if ( pMember->unimem() )
	{
	//Member is a part of an union -> we'll assume it's not an array.
		nElements = 1;
	}
	else
	{
	//Get item size.
		if ( flag == FF_STRU )
		{
		//Get struct item size.
		//
		//Notice:
		//I really doubt IDA support arrays of variable length structures.
		//But just in case we'll do a check here.
			pStruct = get_sptr( pMember );
			if ( ( pStruct->props & SF_VAR ) != 0 )
			{
				msg( "I2S: I2S doesn't support arrays of variable length structures.\n" );
				return STATUS_FAILED;
			}
			memberSize_type = get_struc_size( pStruct->id );
		}
		else
		{
		//Get variable item size.
			if ( NULL == get_typeinfo( pMember->id, 0, pMember->flag, &typeInfo ) )
			{
			//Failed to receive typeInfo.
				memberSize_type = get_data_type_size( pMember->flag, NULL );
			}
			else
			{
			//Success -> proceed.
				memberSize_type = get_data_type_size( pMember->flag, &typeInfo );
			}
		}
		nElements = memberSize / memberSize_type;
	}

//Get type.
	if ( nElements > 1 )
	{
	//It's an array.
		     if ( flag == FF_BYTE ) return NMTP_SetArray( TP_UCHAR, nElements, memberSize );
		else if ( flag == FF_WORD ) return NMTP_SetArray( TP_USHORT, nElements, memberSize );
		else if ( flag == FF_DWRD ) return NMTP_SetArray( TP_ULONG, nElements, memberSize );
		else if ( flag == FF_QWRD ) return NMTP_SetArray( TP_UQUAD, nElements, memberSize );
		else if ( flag == FF_TBYT ) return NMTP_SetArray( TP_REAL80, nElements, memberSize );
		else if ( flag == FF_ASCI     )
		{
		//Get type of string.
			switch ( memberSize_type )
			{
			case 1 :
			default :
			//ASCSTR_C :
			//ASCSTR_TERMCHR :
			//ASCSTR_PASCAL :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_UCHAR, nElements, memberSize );

			case 2 :
			//ASCSTR_UNICODE :
			//ASCSTR_ULEN2 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_WCHAR, nElements, memberSize );

			case 4 :
			//ASCSTR_LEN4 :
			//ASCSTR_ULEN4 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_ULONG, nElements, memberSize );
			}
		}
		else if ( flag == FF_STRU     )
		{
			unsigned long index;

		//Set structure.
			index = NMTP_SetStructure( get_sptr( pMember ), memberSize_type );
			if ( S_FAILED( index ) )
			{
				return STATUS_FAILED;
			}

		//Set array of structures.
			return NMTP_SetArray( index, nElements, memberSize );
		}
		else if ( flag == FF_OWRD     ) return NMTP_SetArray( TP_REAL128, nElements, memberSize );
		else if ( flag == FF_FLOAT    ) return NMTP_SetArray( TP_REAL32, nElements, memberSize );
		else if ( flag == FF_DOUBLE   ) return NMTP_SetArray( TP_REAL64, nElements, memberSize );
		else if ( flag == FF_PACKREAL ) return NMTP_SetArray( TP_REAL80, nElements, memberSize );		//????
		return TP_NOTYPE;  //Same as STATUS_FAILED.
	}
	else
	{
	//It's not an array.
		     if ( flag == FF_BYTE     ) return TP_UCHAR;
		else if ( flag == FF_WORD     ) return TP_USHORT;
		else if ( flag == FF_DWRD     ) return TP_ULONG;
		else if ( flag == FF_QWRD     ) return TP_UQUAD;
		else if ( flag == FF_TBYT     ) return TP_REAL80;
		else if ( flag == FF_ASCI     )
		{
		//Get type of string.
			switch ( memberSize_type )
			{
			case 1 :
			default :
			//ASCSTR_C :
			//ASCSTR_TERMCHR :
			//ASCSTR_PASCAL :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_UCHAR, nElements, memberSize );

			case 2 :
			//ASCSTR_UNICODE :
			//ASCSTR_ULEN2 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_WCHAR, nElements, memberSize );

			case 4 :
			//ASCSTR_LEN4 :
			//ASCSTR_ULEN4 :
			//
			//Set array and return it's index.
				return NMTP_SetArray( TP_ULONG, nElements, memberSize );
			}
		}
		else if ( flag == FF_STRU     )
		{
			return NMTP_SetStructure( get_sptr( pMember ), memberSize );
		}
		else if ( flag == FF_OWRD     ) return TP_REAL128;		//128bit 
		else if ( flag == FF_FLOAT    ) return TP_REAL32;
		else if ( flag == FF_DOUBLE   ) return TP_REAL64;
		else if ( flag == FF_PACKREAL ) return TP_REAL80;		//???
		return TP_NOTYPE;	//Same as STATUS_FAILED.
	}
}

//Supports only structures/unions (intended for register pointers)
unsigned long ida2softice_c::GetType_structureByName
(
	char*	pStructureName
)
{
	int					i = 0;
	struc_t*			pStructure;
	char*				pStructureName_L = pStructureName;
	tid_t				structID;
	unsigned long		variableSize = 0;


//Check if the structure is a variable length structure ->
//user should also supply the length of the variable length variable.
//ex.: strings [12]
//
	//Go to the end of the string.
	while ( *pStructureName_L != '\x0' ){++pStructureName_L;}
	if ( pStructureName_L == pStructureName )
	{
	//It's zero length string -> fail.
		return STATUS_FAILED;
	}

	//Search for '[' and if you find it -> convert string to number.
	while ( pStructureName_L > pStructureName )
	{
		if ( *pStructureName_L == '[' )
		{
		//Convert string to number.
			variableSize = atol( ++pStructureName_L );
			break;
		}

		--pStructureName_L;
	}

//Get structure.
	if ( variableSize != 0 )
	{
	//Remove the variable length part from the string so that IDA
	//will be able to find the structure.
		pStructureName_L = pStructureName;
		while ( *pStructureName_L != '\x0' )
		{
		//Search for space (' ');
			if ( *pStructureName_L == ' ' )
			{
			//Insert end of string and exit.
				*pStructureName_L = '\x0';
				break;
			}

		//Update.
			++pStructureName_L;
		}
	}

	structID = get_struc_id( pStructureName );	//netnode(name);
	if ( structID == BADNODE )
	{
		return STATUS_FAILED;
	}
	pStructure = get_struc( structID );
	if ( pStructure == NULL )
	{
		return STATUS_FAILED;
	}

//Check if the structure is variable length structure and if the variable length
//is different than 0.
	if ( ( pStructure->props & SF_VAR ) != 0 )
	{
	//It's a variable length structure -> check the variable length.
		if ( variableSize == 0 )
		{
			warning(
			  "I2S: Variable length structure was used for register pointer.\nPlease define the length of the last varible of the structure -> ex.: strings [5] .\n"
			);
			msg( "I2S: Please define the length of the last varible of the structure -> ex.: _strings [5] .\n" );
			return STATUS_FAILED;
		}
	}

//Get the structure type.
	return NMTP_SetStructure( pStructure, get_struc_size( pStructure ) + variableSize );
}




