//
//  PlugIn name: IDA to SoftIce converter (only for 32bit programs)
//
//
//  This file contains function to create SYMD.
//
//
#include "i2s.h"

//This function sets VLP for procedure, label or variable.
//
//Notice:
//Call this procedure only if address has a name.
//
// address - ....
// flags - getFlags( address )
//
// return - STATUS_SUCCESS/STATUS_FAILED
//
STATUS ida2softice_c::SYMD_SetVLP
(
	ea_t		address,
	flags_t		flags
)
{
	unsigned long		aFlags;
	char				name[MAXSTR];
	unsigned long		iName;
	func_t				procedure;
	func_t*				pProcedure;
	SEGDATA_VLPTABLE*	pVLP_current;
	unsigned long*		pVLPOffset_current;
	unsigned long		type;

//Initialize.
	pVLP_current = pSegmentCurrent->pVLP_current;

//Set VLP.
	pVLP_current->size = sizeof( SEGDATA_VLPTABLE );
	if ( isFunc( flags ) )
	{
	//Current address is a start of a procedure.
	//
	//Get procedure data.
		pProcedure = get_func( address );
		if ( ( pProcedure->flags & FUNC_FAR ) != 0 )
		{
		//Global function.
		//
		//We need to preserve procedure data as IDA doesn't do it.
			CopyMemory( &procedure, pProcedure, sizeof( func_t ) );

		//Get type.
			pVLP_current->tableType = TT_GLOBALP;	
			pVLP_current->type = NMTP_SetProcedure( &procedure );
		}
		else
		{
		//Local function.
			pVLP_current->tableType = TT_LOCALP;
			pVLP_current->type = NMTP_DEFAULT_LOCAL_PROCEDURE;
		}
	}
	else
	{
		if ( isCode( flags ) )
		{
		//Current address is a label.
			pVLP_current->tableType = TT_LABEL;
			pVLP_current->type = NMTP_DEFAULT_NEAR_LABEL;
		}
		else
		{
		//Current address is a variable.
		//
		//Is it a local or global variables?
			aFlags = get_aflags( address );
			if ( ( aFlags & AFL_PUBNAM ) != 0 )
			{
			//It's global variable.
				pVLP_current->tableType = TT_GLOBALV;
			}
			else
			{
			//It's local variable.
				pVLP_current->tableType = TT_VARIABLE;
			}
			type = GetType_ea( address );	
			if ( S_FAILED( type ) )
			{
				msg( "I2S: Failed to get type (ea:%x).\n", address );
				return STATUS_FAILED;
			}
			pVLP_current->type = type;
		}
	}
	get_name( address, address, (char*)&name, MAXSTR );
	iName = STTB_SetStringTable( (char*)&name );
	if ( iName == STATUS_ERROR )
	{
		return STATUS_FAILED;
	}
	pVLP_current->iName = iName;
	pVLP_current->offVLP = address - minEA;

//Update.
	++pSegmentCurrent->pVLP_current;

	pVLPOffset_current = pSegmentCurrent->pVLPOffsets_current;
	pVLPOffset_current[1] = pVLPOffset_current[0] + sizeof( SEGDATA_VLPTABLE );
	++pSegmentCurrent->pVLPOffsets_current;

	return STATUS_SUCCESS;
}


//This function sets all the procedure variables
//(local variables, arguments, renamed registers,...).
//
// pProcedure - get_func( address )
//
// return - STATUS_SUCCESS/STATUS_FAILED
//
STATUS ida2softice_c::SYMD_SetProcedureVariables
(
	func_t*		pProcedure
)
{
	STATUS					Status;

	unsigned long			argCount;
	unsigned long			argOffsetFirst;
	unsigned long			localCount;
	unsigned long			lVarCount;
	unsigned long			regCount;

	SEGDATA_VLPTABLE*		pLocalVariables;
	SEGDATA_PROCEDURE_V2*	pProcDefinitions;

	unsigned long			memberOffset;
	member_t*				pMember;
	char					name[MAXSTR];
	unsigned long			iName;

	unsigned long			frameSize;
	struc_t*				pFrame;

	ea_t					currentAddress;
	unsigned long			type;
	unsigned long			type_member;

	SEGDATA_REGISTER*		pRegister_current;
	regvar_t*				pRegisterArray;
	BOOL					regMemoryWasAllocated = FALSE;	//Did we allocate memory for
															//manual build of register array
															//array? (read below)
	regType_s				regType;

	char					regName_canon[256];
	char					regName_comment[256];
	char					regName_user[256];

//Initialize.
	//Get number of arguments and local variables.
	//
	//Does procedure have a frame
	//(are there any arguments or local variables set)?
	if ( ( pProcedure->flags & FUNC_FRAME ) != 0 )
	{
		Status = SYMD_GetNumberOfProcedureVariables(
		  pProcedure,
		  &lVarCount,
		  &argCount,
		  &argOffsetFirst
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
		localCount = argCount + lVarCount;
	}
	else
	{
	//Procedure doesn't have any arguments or local variables.
		argCount = 0;
		lVarCount = 0;
		localCount = 0;
	}

	//Get number of renamed registers.
	regCount = pProcedure->regvarqty;

//Define SEGDATA_PROCEDURE.
	pProcDefinitions = pSegmentCurrent->pProcDefinitions_current;

	//Set
	pProcDefinitions->offStartProc = pProcedure->startEA - minEA;
	pProcDefinitions->offEndProc = pProcedure->endEA - minEA - 1;
	pProcDefinitions->offLocalVariables = pSegmentCurrent->offsetLVarCurrent;
	pProcDefinitions->nLocalVariables = argCount + lVarCount;
	pProcDefinitions->offRegisterVariables = pSegmentCurrent->offsetRVarCurrent;
	pProcDefinitions->sizeRegisterVariables = regCount * sizeof( SEGDATA_REGISTER );
	pProcDefinitions->M1 = 0xFFFFFFFF;
	pProcDefinitions->N1 = 0;

	//Update
	++pSegmentCurrent->pProcDefinitions_current;

//Set local variables and arguments.
	if ( localCount != 0 )
	{
	//Initialize
		pLocalVariables = pSegmentCurrent->pLocalVariables_current;

		pFrame = get_frame( pProcedure );
		frameSize = pProcedure->frsize;
		memberOffset = get_struc_first_offset( pFrame );

	//Set local variables.
		while ( lVarCount != 0 )
		{
		//Get member.
			pMember = get_member( pFrame, memberOffset );
			//Did we get it? There can be nameless bytes in between.
			if ( pMember != NULL )
			{
			//We got the member.
			//
			//Get it's name.
				get_member_name( pMember->id, (char*)&name, sizeof( name ) );

			//Set local variable structure.
				pLocalVariables->size = sizeof( SEGDATA_VLPTABLE );
				pLocalVariables->tableType = TT_LOCALV;
				type = GetType_member( pMember, get_member_size( pMember ) );
				if ( S_FAILED( type ) )
				{
					return STATUS_FAILED;
				}
				pLocalVariables->type = type;
				iName = STTB_SetStringTable( (char*)&name );
				if ( iName == STATUS_ERROR )
				{
					return STATUS_FAILED;
				}
				pLocalVariables->iName = iName;
				pLocalVariables->offVLP = pMember->get_soff() - frameSize;

			//Update.
				++pLocalVariables;

			//Set struct members as separate local variables. So that we'll see them in SIce.
				if ( ( pMember->flag & DT_TYPE ) == FF_STRU )
				{
				//Current member is a struct/union -> define it's members.
					pLocalVariables = SYMD_SetLocalStructMembers(
					  get_sptr( pMember ),
					  (char*)&name,
					  pMember->get_soff() - frameSize,
					  pLocalVariables
					);
					if ( pLocalVariables == STATUS_FAILED )
					{
						return STATUS_FAILED;
					}
					lVarCount -= SYMD_GetStructureMemberCount( get_sptr( pMember ) );
				}

			//Update
				--lVarCount;
			}

		//Go to next member.
			memberOffset = get_struc_next_offset( pFrame, memberOffset );
		}

	//Set arguments.
		memberOffset = argOffsetFirst;
		while ( argCount != 0 )
		{
		//Get member.
			pMember = get_member( pFrame, memberOffset );
			//Did we get it? There can be nameless bytes in between.
			if ( pMember != NULL )
			{
			//We got the member.
			//
			//Get it's name.
				get_member_name( pMember->id, (char*)&name, sizeof( name ) );

			//set
				pLocalVariables->size = sizeof( SEGDATA_VLPTABLE );
				pLocalVariables->tableType = TT_LOCALV;
				type_member = GetType_member( pMember, get_member_size( pMember ) );
				if ( S_FAILED( type_member ) )
				{
					return STATUS_FAILED;
				}
				pLocalVariables->type = type_member;

				iName = STTB_SetStringTable( (char*)&name );
				if ( iName == STATUS_ERROR )
				{
					return STATUS_FAILED;
				}
				pLocalVariables->iName = iName;
				pLocalVariables->offVLP = pMember->get_soff() - frameSize;

			//Update
				++pLocalVariables;
				--argCount;
			}

		//Update.
			memberOffset = get_struc_next_offset( pFrame, memberOffset );
		}

	//Update.
		//Set the current pointer.
		pSegmentCurrent->pLocalVariables_current = pLocalVariables;
		pSegmentCurrent->offsetLVarCurrent += localCount * sizeof( SEGDATA_VLPTABLE );
	}

//Set renamed registers.
//
//struct regvar_t : public area_t
//{
//  char *canon;    // canonical register name (case-insensitive)
//  char *user;     // user-defined register name
//  char *cmt;      // comment to appear near definition
//};
//If comment is not present than plugIn will assume that register is a value.
//If comment is     present than plugIn will assume that register is a pointer to 'structure' type defined in comment.
//ex.:
//
//  canon = ecx
//   user = this
//    cmt = class_ida2softice_c  <- class_ida2softice_c is a structure defined in IDA
//                                  (be cerefull case sensitive)
	if ( regCount != 0 )
	{
	//Initialize.
		currentAddress = pProcedure->startEA;
		pRegister_current = pSegmentCurrent->pRegisters_current;
		pRegisterArray = pProcedure->regvars;

	//Update offset here as we'll use regCount as a downcounter.
		pSegmentCurrent->offsetRVarCurrent += regCount * sizeof( SEGDATA_REGISTER );

	//Check if register array is set -> if not set it manualy.
		if ( pRegisterArray == NULL )
		{
		//The register array was not set by IDA -> i2s will set the array manually.
		//
		//I don't know if this part is needed in version 4.7 and above, but
		//just to be sure I'll leave it here.
			pRegisterArray = (regvar_t*) LocalAlloc( LPTR, regCount * sizeof( regvar_t ) );
			if ( pRegisterArray == NULL )
			{
				msg( "I2S: Failed to allocate memory(SYMD_SetProcedureVariables).\n" );
				return STATUS_FAILED;
			}
			regMemoryWasAllocated = TRUE;

		//Set register array manually.
			Status = SYMD_GetRegisterVariablesData(
			  pProcedure,
			  pRegisterArray
			);
			if ( S_FAILED( Status ) )
			{
				jumpto( pProcedure->startEA, -1 );
				msg(
				  "I2S: Sorry but the plugIn supports only naming of regular registers (from al to edi).\n",
				  0
				);
			//Release resources.
				LocalFree( pRegisterArray );
				return STATUS_FAILED;
			}
		}

		while ( regCount != 0 )
		{
		//Get names.
			qstrncpy( (char*)&regName_user, pRegisterArray->user, sizeof( regName_user ) );
			qstrncpy( (char*)&regName_canon, pRegisterArray->canon, sizeof( regName_canon ) );

		//Get register type.
			Status = SYMD_GetRegisterType( (char*)&regName_canon, &regType );
			if ( S_FAILED( Status ) )
			{
				jumpto( pProcedure->startEA, -1 );
				msg(
				  "I2S: Sorry but the plugIn supports only naming of regular registers (from al to edi).\n",
				  0
				);

			//Free resources.
				if ( regMemoryWasAllocated == TRUE )
				{
					LocalFree( pRegisterArray );
				}
				return STATUS_FAILED;
			}

		//Set SEGDATA_REGISTER.
			pRegister_current->size = sizeof( SEGDATA_REGISTER );
			if ( pRegisterArray->cmt == NULL )
			{
			//Register doesn't have any comment -> i2s will assume the register is a variable.
				pRegister_current->tableType = TT_REGISTER;
				pRegister_current->type = regType.regType;
			}
			else
			{
			//Register has comment -> i2s will assume the register is a pointer.
				//Get comment.
				qstrncpy( (char*)&regName_comment, pRegisterArray->cmt, sizeof( regName_comment ) );

				//Set struct.
				pRegister_current->tableType = TT_REGISTER_POINTER;
				type = GetType_structureByName( (char*)&regName_comment );
				if ( S_FAILED( type ) )
				{
					jumpto( pProcedure->startEA, -1 );
					msg(
					  "I2S: The register( %s<->%s ) structure type(%s) does not exist.\nCheck for spelling(case sensitive).\n(or failed to allocate memory).\n",
					  &regName_canon,
					  &regName_user,
					  &regName_comment
					);
				//Free resources.
					if ( regMemoryWasAllocated == TRUE )
					{
						LocalFree( pRegisterArray );
					}
					return STATUS_FAILED;
				}
				pRegister_current->type = type;
			}
			iName = STTB_SetStringTable( (char*)&regName_user );
			if ( iName == STATUS_ERROR )
			{
			//Failed.
				return STATUS_FAILED;
			}
			pRegister_current->iName = iName;
			pRegister_current->parameterOffset = 0;	//for both parameter and pointer
			pRegister_current->registerNumber = (unsigned short) regType.regNumber;

		//Update
			++pRegisterArray;
			++pRegister_current;
			--regCount;
		}

	//Free resources.
		if ( regMemoryWasAllocated == TRUE )
		{
			LocalFree( pRegisterArray );
		}

	//Update.
		pSegmentCurrent->pRegisters_current = pRegister_current;
	}

	return STATUS_SUCCESS;
}

//--------------------------------------------------------------------------
//Set local structure member names.
//
// pStructure - Pointer to structure.
// pName - Pointer to the structure's name.
//         We'll write to this buffer so the name must be in a 'scratch' buffer.
//
// structOffset - EBP offset in procedure.
// pLocalVariables - Pointer to 1st free VLPTABLE (pLocalVariables_current). 
//
// return - Pointer to 1st free VLPTABLE (localVariablesCurrent) or STATUS_FAILED.
//
SEGDATA_VLPTABLE* ida2softice_c::SYMD_SetLocalStructMembers
(
	struc_t*			pStructure,
	char*				pName,
	unsigned long		structOffset,
	SEGDATA_VLPTABLE*	pLocalVariables
)
{
	char*			pName_afterDot;
	unsigned long	iName;
	member_t*		pMember;
	unsigned long	memberOffset;
	unsigned long	type;
	
//Test if we have any members.
	if ( pStructure->memqty == 0 )
	{
		char	temp_strucName[128];

		get_struc_name( pStructure->id, (char*)&temp_strucName, sizeof( temp_strucName ) );
		warning( "Structure(%s) doesn't have any members.", (char*)&temp_strucName );
		return STATUS_FAILED;
	}

//Update the struct name.
//
//Every member name will made as: structureName.memberName
//So if we have structure in structure: structureName.memberName.memberName
//And on and on and on....
	pName_afterDot = pName;
	//Go to end of name.
	while ( pName_afterDot[0] != '\x0' )
	{
		++pName_afterDot;
	}
	//We are at '\x0' position right now -> add '.' at it's place.
	pName_afterDot[0] = '.';
	pName_afterDot[1] = '\x0';
	++pName_afterDot;

//Get struct first member.
	memberOffset = get_struc_first_offset( pStructure );

//WALDEMAR
//Poglej ali lahko ta procedure prenese nedefinirane byte.
	pMember = get_member( pStructure, memberOffset );
	//We don't have to worry here about nameless bytes as other structure
	//procedures were runed before this and they didn't fail.

//Test if the struct is union. Well set only the first member name.
	if ( pStructure->is_union() )
	{
	//It's union
		pLocalVariables->size = sizeof( SEGDATA_VLPTABLE );
		pLocalVariables->tableType = TT_LOCALV;
		type = GetType_member( pMember, get_member_size( pMember ) );
		if ( S_FAILED( type ) )
		{
			return STATUS_FAILED;
		}
		pLocalVariables->type = type;

		get_member_name( pMember->id, pName_afterDot, MAXSTR - strlen( pName ) - 1 );
		iName = STTB_SetStringTable( pName );
		if ( iName == STATUS_ERROR )
		{
			return STATUS_FAILED;
		}
		pLocalVariables->iName = iName;
		pLocalVariables->offVLP = structOffset;
		++pLocalVariables;
		return pLocalVariables;
	}

//We have a normal structure.
	while ( memberOffset != BADADDR )
	{
	//Get structure member.
		pMember = get_member( pStructure, memberOffset );
		if ( pMember != NULL )
		{
		//Is current member a structure?
			if ( ( pMember->flag & DT_TYPE ) == FF_STRU )
			{
			//Yes it's a structure -> define it's members too.
				get_member_name( pMember->id, pName_afterDot, MAXSTR - strlen( pName ) - 1 );

				pLocalVariables = SYMD_SetLocalStructMembers(
				  get_sptr( pMember ),
				  pName,
				  structOffset,
				  pLocalVariables
				);
			}
			else
			{
			//It's a variable (it can be an array).
				pLocalVariables->size =sizeof( SEGDATA_VLPTABLE );
				pLocalVariables->tableType = TT_LOCALV;
				type = GetType_member( pMember, get_member_size( pMember ) );
				if ( S_FAILED( type ) )
				{
					return STATUS_FAILED;
				}
				pLocalVariables->type = type;

				get_member_name( pMember->id, pName_afterDot, MAXSTR - strlen( pName ) - 1 );
				iName = STTB_SetStringTable( pName );
				if ( iName == STATUS_ERROR )
				{
					return STATUS_FAILED;
				}
				pLocalVariables->iName = iName;
				pLocalVariables->offVLP = structOffset;
				++pLocalVariables;
			}

		//Update.
			structOffset += get_member_size( pMember );
		}
		else
		{
		//Undefined byte.
			structOffset += 1;
		}

	//Update.
		memberOffset = get_struc_next_offset( pStructure, memberOffset );
	}

	return pLocalVariables;
}

//#define	regNames_SegReg		"es\x0cs\x0ss\x0ds\x0fs\x0gs"	//16bit
//#define	regNames_Special	"ip\x0flags\x0eip\x0eflags"		//32bit

//#define	regNames_System		"cr0\x0cr1\x0cr2\x0cr3\x0dr0\x0dr1\x0dr2\x0dr3\x0dr4\x0dr5\x0dr6\x0dr7"
//#define	regNames_extensions	"st0\x0st1\x0st2\x0st3\x0st4\x0st5\x0st6\x0st7\x0control\x0status\x0tag\x0fpid\x0fpcs\x0fpdo\x0fpds\x0isem\x0fpeip\x0fpedo"

//--------------------------------------------------------------------------
//Finds the register number from name.
//
// return - STATUS_SUCCESS/STATUS_FAILED.
//
#pragma pack(push, 4) // pushes the old alignment value (most probably 8) on the compiler stack, and sets the new value to 16
char*	pRegNames_8bit 	= "al\x0\x0" "cl\x0\x0" "dl\x0\x0" "bl\x0\x0" "ah\x0\x0" "ch\x0\x0" "dh\x0\x0" "bh\x0\x0";
char*	pRegNames_16bit	= "ax\x0\x0" "cx\x0\x0" "dx\x0\x0" "bx\x0\x0" "sp\x0\x0" "bp\x0\x0" "si\x0\x0" "di\x0\x0";
char*	pRegNames_32bit	= "eax\x0"   "ecx\x0"   "edx\x0"   "ebx\x0"   "esp\x0"   "ebp\x0"   "esi\x0"   "edi\x0";
#pragma pack(pop) // restores the old value
/*
__declspec(align(4)) char*	pRegNames_8bit 	= "al\x0\x0" "cl\x0\x0" "dl\x0\x0" "bl\x0\x0" "ah\x0\x0" "ch\x0\x0" "dh\x0\x0" "bh\x0\x0";
__declspec(align(4)) char*	pRegNames_16bit	= "ax\x0\x0" "cx\x0\x0" "dx\x0\x0" "bx\x0\x0" "sp\x0\x0" "bp\x0\x0" "si\x0\x0" "di\x0\x0";
__declspec(align(4)) char*	pRegNames_32bit	= "eax\x0"   "ecx\x0"   "edx\x0"   "ebx\x0"   "esp\x0"   "ebp\x0"   "esi\x0"   "edi\x0";
*/
unsigned long	ida2softice_c::SYMD_GetRegisterType
(
	char*		pRegName,
	regType_s*	pRegType
)
{
	unsigned long	regName_caller;
	unsigned long	regName_current;

//Find the register and define it's type.
//
//First check if the required register is one of the types the i2s supports.
	switch ( strlen( pRegName ) )
	{
	case 3:
	//As the register name is 3 chars long it must be one of the extended register ->
	//32bit register.
		regName_caller = *( (unsigned long*)pRegName );

		for ( int i = 0; i < 8; i++ )
		{
		//Get the current register name.
			regName_current = ( (unsigned long*)pRegNames_32bit )[i];

		//Are two register names the same?
			if ( regName_caller == regName_current )
			{
			//We have found the register -> set the register type.
				pRegType->regType = TP_ULONG;
				pRegType->regNumber = REGISTER_EAX + i;
				return STATUS_SUCCESS;
			}
		}

	case 2:
	//As the register names is 2 chars long it must be 8 or 16 bit one.
		//Get the name.
		pRegName[3] = '\x0';
		regName_caller = *( (unsigned long*)pRegName );

	//Scan 16 bit registers.
		for ( int i = 0; i < 8; i++ )
		{
		//Get the current register name.
			regName_current = ( (unsigned long*)pRegNames_16bit )[i];

		//Are two register names the same?
			if ( regName_caller == regName_current )
			{
			//We have found the register -> set the register type.
				pRegType->regType = TP_USHORT;
				pRegType->regNumber = REGISTER_AX + i;
				return STATUS_SUCCESS;
			}
		}

	//Scan 8 bit registers.
		for ( int i = 0; i < 8; i++ )
		{
		//Get the current register name.
			regName_current = ( (unsigned long*)pRegNames_8bit )[i];

		//Are two register names the same?
			if ( regName_caller == regName_current )
			{
			//We have found the register -> set the register type.
				pRegType->regType = TP_UCHAR;
				pRegType->regNumber = REGISTER_AL + i;
				return STATUS_SUCCESS;
			}
		}

	default:
		break;
	}

//Failed to find the register.
	return STATUS_FAILED;

}

//--------------------------------------------------------------------------
//Sets the array of local register variables.
//
// regArray - Pointer to array of regvar_t-s.
//
// return - STATUS_SUCCESS/STATUS_FAILED.
//
unsigned long ida2softice_c::SYMD_GetRegisterVariablesData
(
	func_t*		pProcedure,
	regvar_t*	pRegArray
)
{
	ea_t			procEndEA = pProcedure->endEA;
	ea_t			procStartEA = pProcedure->startEA;
	ea_t			currentAddress;

	long			nRegsToFind = pProcedure->regvarqty;
	regvar_t*		pRegister;
	char*			pRegName_current;

//Find all the renamed register in procedure.
//
//We need to check one register at the time.
	for ( int i = 0; i < 3; i++ )
	{
	//Select the register types.
		switch ( i )
		{
		case 0:
		//32 bit registers
			pRegName_current = pRegNames_32bit;
			break;

		case 1:
		//16 bit registers
			pRegName_current = pRegNames_16bit;
			break;

		default:
		//case 2:
		//8 bit registers
			pRegName_current = pRegNames_8bit;
			break;
		}

	//Search for current register in procedure.
		for ( int j = 0; j < 8; j++ )
		{
		//Initialize.
			currentAddress = procStartEA;

		//Check if current register is renamed.
			pRegister = find_regvar(
			  pProcedure,
			  currentAddress,
			  procEndEA,
			  pRegName_current,
			  NULL
			);
			if ( pRegister != NULL  )
			{
			//This register has at least 1 additional name.
			//
			//Check if this register has other names too.
				while ( currentAddress < procEndEA )
				{
					if ( pRegister != NULL  )
					{
					//Set current renamed register name.
						pRegArray->canon = pRegister->canon;
						pRegArray->user = pRegister->user;
						pRegArray->cmt = pRegister->cmt;
						pRegArray->startEA = pRegister->startEA;
						pRegArray->endEA = pRegister->endEA;

					//Update.
						currentAddress = pRegister->endEA;

						++pRegArray;
						--nRegsToFind;
						if ( nRegsToFind <= 0 )
						{
						//There are no other renamed registers remained.
							return STATUS_SUCCESS;
						}
					}
					else
					{
					//No other name for this register remained.
						break;
					}

				//Check if there is any other
					pRegister = find_regvar(
					  pProcedure,
					  currentAddress,
					  procEndEA,
					  pRegName_current,
					  NULL
					);
				}
			}

		//Finished with current register -> go to next one.
			pRegName_current += 4;
		}
	}

//If we came to here than not all renamed register were supported -> fail.
	return STATUS_FAILED;
}

