//
//  PlugIn name: IDA to SoftIce converter (only for 32bit programs)
//
//
//  This file contains functions that get number of elements in an object.
//
//
#include "i2s.h"

//--------------------------------------------------------------------------
//Get number of local variables and arguments in procedure.
//
// pProcedure - Pointer to procedure structure.
// pLVarCount - Pointer to variable that will receive number of local variables.
// pArgCount -  ......
// pArgFirstOffset - Pointer to variable that will receive the offset in frame
//                   where arguments start.
//
// return - STATUS_SUCCESS/STATUS_FAILED
//
STATUS ida2softice_c::SYMD_GetNumberOfProcedureVariables
(
	func_t*			pProcedure,
	unsigned long*	pLVarCount,
	unsigned long*	pArgCount,
	unsigned long*	pArgFirstOffset
)
{
	struc_t*		pFrame;
	unsigned long	frameSize;

	member_t*		pMember;
	unsigned long	memberOffset;

	unsigned long	argCount = 0;
	unsigned long	lVarCount = 0;
	unsigned long	lVarSize = 0;

//Error in frame.hpp and funcs.hpp!!!
//    +----------------------------------------+        <- SP
//    | local variables                        |
//    +----------------------------------------+        <- BP
//    | saved registers (SI,DI,etc)            |
//    +----------------------------------------+
//    | return address (isn't stored in func_t)|
//    +----------------------------------------+
//    | function arguments                     |
//    +----------------------------------------+

//Get procedure frame.
//
//Frame is a struct_t type, but the members are all 'pushed' + 'locals' + 'pushed regs'.
//But we only need number of 'pushed' variables.
	pFrame = get_frame( pProcedure );
	//Do we have a frame?
	if ( pFrame == NULL )
	{
		if ( pLVarCount != NULL )
		{
			*pLVarCount = 0;
		}
		if ( pArgCount != NULL )
		{
			*pArgCount = 0;
		}
		if ( pArgFirstOffset != NULL )
		{
			*pArgFirstOffset = 0;
		}
		return STATUS_SUCCESS;
	}

//Do we have any EBP based local variables?
//
//If frameSize != 0 then there are EBP based local variables available.
//
//Notice:
//The procedure can have ESP based local variables, but i2s can't handle them.
	frameSize = pProcedure->frsize;
	if ( frameSize != 0 )
	{
	//Procedure has local variables.
	//
	//Just check if current procedure is a real procedure or it's really data
	//converted to code.
		if ( frameSize > 32768 )
		{
			char*	funcName[128];

			get_func_name( pProcedure->startEA, (char*)funcName, sizeof( funcName ) );
			msg(
			  "I2S: Procedure %s(0x%X)has strangely large stack offsets -> fix it as this is a real time penalty for i2s.\n",
			  funcName,
			  pProcedure->startEA
			);
		}

	//Get first member offset.
		memberOffset = get_struc_first_offset( pFrame );
		while ( memberOffset != BADADDR )
		{
		//Get member.
			pMember = get_member( pFrame, memberOffset );

		//Check if current location has a name -> there can be nameless bytes in the frame.
			if ( pMember != NULL )
			{
			//We have a member with a name.
				++lVarCount;

			//Check if this variables is a structure or union.
				if ( ( pMember->flag & DT_TYPE ) == FF_STRU )
				{
					unsigned long	count;

				//Yes it's a structure or union.
				//
				//Get number of elements in structure/union.
					count = SYMD_GetStructureMemberCount( get_sptr( pMember ) );
					if ( count == STATUS_ERROR )
					{
						return STATUS_FAILED;
					}
					lVarCount += count;
				}

			//Update.
				lVarSize += get_member_size( pMember );
			}
			else
			{
			//Current member is a nameless byte.
				++lVarSize;
			}

		//Are we over the local variables frame size?
			if ( lVarSize >= frameSize )
			{
			//Yes we are at the end of local variables frame part.
			//
			//Prepare memberOffset for arguments part.
				memberOffset = get_struc_next_offset( pFrame, memberOffset );
				break;
			}

		//Go to next member.
			memberOffset = get_struc_next_offset( pFrame, memberOffset );
		}
	}
	else
	{
	//Prepare memberOffset for arguments part.
	//
	//The procedure didn't have any local variables.
		memberOffset = get_struc_first_offset( pFrame );
	}
	if ( pLVarCount != NULL )
	{
		*pLVarCount = lVarCount;
	}

//Find " r" -> this is a representative of a return value.
//After this member we have arguments (before this one are local variables
//and pushed registers).
	while ( memberOffset != BADADDR )
	{
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a member with a name.
		//
		//Check if this member is a " r" (return value member).
			char	memberName[4];

			get_member_name( pMember->id, memberName, sizeof( memberName ) );
			memberName[3] = '\x0';

			if ( *(unsigned long*)&memberName == 0x00007220 )
			{
			//Yes it's a return value -> break current loop.
				break;
			}
		}

	//It's not the return value member -> go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}

//Now we are at " r" member. Go to next member -> to first argument and
//count how many arguments the procedure has.
	memberOffset = get_struc_next_offset( pFrame, memberOffset );
	//Do we have an argument?
	if ( pArgFirstOffset != NULL )
	{
		*pArgFirstOffset = memberOffset;
	}
	while( memberOffset != BADADDR )	
	{
	//Yes we have an argument -> get it.
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a name and with this an argument.
			++argCount;
		}

	//Go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}
	if ( pArgCount != NULL )
	{
		*pArgCount = argCount;
	}

	return STATUS_SUCCESS;
}




//--------------------------------------------------------------------------
//Get number of arguments in procedure.
//
// procedure - Procedure structure.
//
// return - Number of arguments in procedure.
//
/*unsigned long ida2softice_c::SYMD_GetProcedureArgCount
(
	func_t*			pProcedure
)
{
	struc_t*		pFrame;

	member_t*		pMember;
	WORD*			pMemberName;
	unsigned long	memberOffset;

	unsigned long	argCount = 0;

//Error in frame.hpp and funcs.hpp!!!
//    +----------------------------------------+        <- SP
//    | local variables                        |
//    +----------------------------------------+        <- BP
//    | saved registers (SI,DI,etc)            |
//    +----------------------------------------+
//    | return address (isn't stored in func_t)|
//    +----------------------------------------+
//    | function arguments                     |
//    +----------------------------------------+
//
//	pArgSize = procedure->argsize;
//	if ( pArgSize == NULL ) return(0); this only means that callee doesn't pop the arguments

//Get procedure frame.
//
//Frame is a struct_t type, but the members are all 'pushed' + 'locals' + 'pushed regs'.
//But we only need number of 'pushed' variables.
	pFrame = get_frame( pProcedure );
	//Do we have a frame?
	if ( pFrame == NULL )
	{
		return 0;
	}
	
//Get first member offset.
//
//Find where is the first argument located in frame.
	memberOffset = get_struc_first_offset( pFrame );

	//Find " r" -> this is a representative of a return value.
	//After this member we have arguments (before this one are local variables
	//and pushed registers).
	while ( memberOffset != BADADDR )
	{
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a member with a name.
		//
		//Check if this member is a " r" (return value member).
			pMemberName = (WORD*)get_member_name( pMember->id );
			if ( pMemberName[0] == 0x7220 )	// 0x7220 <==> " r"
			{
			//Yes it's a return value -> break current loop.
				break;
			}
		}

	//It's not the return value member -> go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}

//Now we are at " r" member. Go to next member -> to first argument and
//count how many arguments the procedure has.
	memberOffset = get_struc_next_offset( pFrame, memberOffset );
	//Do we have an argument?
	while( memberOffset != BADADDR )	
	{
	//Yes we have an argument -> get it.
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a name and with this an argument.
			++argCount;
		}

	//Go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}
	return argCount;
}*/

//--------------------------------------------------------------------------
//Get number of arguments in procedure and the offset to the first one.
//
// pProcedure - Procedure structure. It MUST have a frame! (check funcs.hpp)
// offsetArgFirst - Pointer to variable that will get the offset of 1st argument in frame,
//                  if you don't want it set it to ZERO.
//                  It can be an offset to nameLess byte, but after all nameLess bytes we will came to first argument.
//
// return - Number of arguments in procedure.
//
unsigned long ida2softice_c::SYMD_GetProcedureArgCountAndFirstOffset
(
	func_t*			pProcedure,
	unsigned long*	pOffsetArgFirst
)
{
	struc_t*		pFrame;

	member_t*		pMember;
	unsigned long	memberOffset;

	unsigned long	argCount = 0;

//Error in frame.hpp and funcs.hpp!!!
//    +----------------------------------------+        <- SP
//    | local variables                        |
//    +----------------------------------------+        <- BP
//    | saved registers (SI,DI,etc)            |
//    +----------------------------------------+
//    | return address (isn't stored in func_t)|
//    +----------------------------------------+
//    | function arguments                     |
//    +----------------------------------------+
//
//	pArgSize = procedure->argsize;
//	if ( pArgSize == NULL ) return(0); this only means that callee doesn't pop the arguments

//Get procedure frame.
//
//Frame is a struct_t type, but the members are all 'pushed' + 'locals' + 'pushed regs'.
//But we only need number of 'pushed' variables.
	pFrame = get_frame( pProcedure );
	//Do we have a frame?
	if ( pFrame == NULL )
	{
		return 0;
	}
	
//Get first member offset.
//
//Find where is the first argument located in frame.
	memberOffset = get_struc_first_offset( pFrame );

	//Find " r" -> this is a representative of a return value.
	//After this member we have arguments (before this one are local variables
	//and pushed registers).
	while ( memberOffset != BADADDR )
	{
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a member with a name.
		//
		//Check if this member is a " r" (return value member).
			char	memberName[4];

			get_member_name( pMember->id, memberName, sizeof( memberName ) );
			memberName[3] = '\x0';

			if ( *(unsigned long*)&memberName == 0x00007220 )
			{
			//Yes it's a return value -> break current loop.
				break;
			}
		}

	//It's not the return value member -> go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}

//Now we are at " r" member. Go to next member -> to first argument and
//count how many arguments the procedure has.
	memberOffset = get_struc_next_offset( pFrame, memberOffset );
	//Do we have an argument?
	while( memberOffset != BADADDR )	
	{
	//Yes we have an argument -> get it.
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a name and with this an argument.
			++argCount;
		}

	//Go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}
	return argCount;
}

//--------------------------------------------------------------------------
//Get number of local variables in procedure.
//
// procedure - Procedure structure. It must have frame! (check funcs.hpp)
//
// return - Number of local variables in procedure OR STATUS_ERROR.
//
unsigned long ida2softice_c::SYMD_GetProcedureLVarCount
(
	func_t*		pProcedure
)
{
	struc_t*		pFrame;
	unsigned long	frameSize;

	member_t*		pMember;
	unsigned long	memberOffset;

	unsigned long	lVarCount = 0;
	unsigned long	lVarSize = 0;

//Error in frame.hpp and funcs.hpp!!!
//    +----------------------------------------+        <- SP
//    | local variables                        |
//    +----------------------------------------+        <- BP
//    | saved registers (SI,DI,etc)            |
//    +----------------------------------------+
//    | return address (isn't stored in func_t)|
//    +----------------------------------------+
//    | function arguments                     |
//    +----------------------------------------+

//Do we have any EBP based local variables?
	frameSize = pProcedure->frsize;
	if ( frameSize == 0 )
	{
	//No there aren't any.
	//
	//The procedure can have ESP based local variables, but i2s can't handle them.
		return 0;
	}

//Get procedure frame.
//
//Frame is a struct_t type, but the members are all 'pushed' + 'locals' + 'pushed regs'.
//But we only need number of 'local' variables.
	pFrame = get_frame( pProcedure );
	
//Get first member offset.
	memberOffset = get_struc_first_offset( pFrame );
	while ( memberOffset != BADADDR )
	{
	//Get member.
		pMember = get_member( pFrame, memberOffset );

	//Check if current location has a name -> there can be nameless bytes in the frame.
		if ( pMember != NULL )
		{
		//We have a member with a name.
			++lVarCount;

		//Check if this variables is a structure or union.
			if ( ( pMember->flag & DT_TYPE ) == FF_STRU )
			{
				unsigned long	count;

			//Yes it's a structure or union.
			//
			//Get number of elements in structure/union.
				count = SYMD_GetStructureMemberCount( get_sptr( pMember ) );
				if ( count == STATUS_ERROR )
				{
					return STATUS_ERROR;
				}
				lVarCount += count;
			}

		//Update.
			lVarSize += get_member_size( pMember );
		}
		else
		{
		//Current member is a nameless byte.
			++lVarSize;
		}

	//Are we over the local variables frame size?
		if ( lVarSize >= frameSize )
		{
		//Yes we are at the end of local variables frame part.
			break;
		}

	//Go to next member.
		memberOffset = get_struc_next_offset( pFrame, memberOffset );
	}

	return lVarCount;
}

//--------------------------------------------------------------------------
//Get number of command lines and size of source listing in procedure.
//
// pProcedure - Pointer to a procedure structure.
// pSourceListingSize - Pointer to variable that will receive the size of listing.
//                      This parameter can be NULL.
//
// return - nSourceLines
//
unsigned long ida2softice_c::SYMD_GetProcedureSourceLines
(
	func_t*			pProcedure,
	unsigned long*	pSourceListingSize
)
{
	unsigned long	commandCount = 0;
	ea_t			currentAddress = pProcedure->startEA;
	ea_t			endFunction = pProcedure->endEA;
	unsigned long	sizeOfSourceListing = 0;

//Is function hideen?
	if ( ( pProcedure->flags & FUNC_HIDDEN ) != 0 )
	{
	//Function is minimized.
		//Size of sourceListing.
		if ( pSourceListingSize != NULL )
		{
			*pSourceListingSize = SYMD_GetSizeOfSourceListing( currentAddress );
		}
		//Number of command lines.
		return 1;
	}
	else
	{
	//Function is not minimized -> count the number of command lines in procedure.
		if ( pSourceListingSize != NULL )
		{
		//Size of source listing is needed.
			while ( currentAddress < endFunction )
			{
			//Size of listing.
				sizeOfSourceListing += SYMD_GetSizeOfSourceListing( currentAddress );

			//Get number of command lines.
				++commandCount;

			//Update.
				currentAddress += get_item_size( currentAddress );
			}

			*pSourceListingSize = sizeOfSourceListing;
		}
		else
		{
		//Size of source listing is not needed.
			while ( currentAddress < endFunction )
			{
			//Get number of command lines.
				++commandCount;

			//Update.
				currentAddress += get_item_size( currentAddress );
			}
		}
	}
	return commandCount;
}

//--------------------------------------------------------------------------
//Get the size of text lines at current address.
//
// currentAddress - Address you want the size of text lines for.
// return - Size of text.
//
unsigned long ida2softice_c::SYMD_GetSizeOfSourceListing
(
	ea_t	address
)
{
	unsigned long	i = 0;
	char			lineBuffer[MAXSTR];
	unsigned long	nLines;
	unsigned long	textSize = 0;

//Get the text lines.
	IDAPlace.ea = address;
	lineArray.set_place( &IDAPlace );

//Get size of source listing.
	nLines = lineArray.linecnt;
	while ( i < nLines )
	{
		textSize += tag_remove( lineArray.lines[i], lineBuffer, MAXSTR );
		++i;		
	}
	textSize += nLines;	//add number of 'linefeed' commands

	return textSize;
}
STATUS ida2softice_c::SYMD_GetSizeOfSourceListingAndSaveSource
(
	ea_t			address,
	unsigned long*	pSourceSize
)
{
	STATUS			Status;

	unsigned long	i = 0;
	unsigned long	nLines;
	unsigned long	textSize = 0;
	unsigned long	textSize_current;

//Get the text lines.
	IDAPlace.ea = address;
	lineArray.set_place( &IDAPlace );
	nLines = lineArray.linecnt;

//Check if we have enough memory.
	if ( source.freeSource <= MAXSTR * nLines )
	{
	//Not enough memory left -> reallocate.
		Status = ReallocateMemory(
		  ALLOCATE_REALLYBIG,	//+4MB
		  source.hSource,
		  (void**)&(source.pSource),
		  (void**)&(source.pSource_current),
		  &source.freeSource,
		  &source.sizeSource
		);
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}
	}

//Copy source listing and get it's size.
	while ( i < nLines )
	{
	//Get current text line.
		textSize_current = tag_remove( lineArray.lines[i], source.pSource_current, MAXSTR );

	//Add lineFeed command at the end of this line -> IDA trows it out.
		source.pSource_current[textSize_current] = '\x0D';
		source.pSource_current[textSize_current + 1] = '\x0A';

	//Update.
		textSize_current += 2;	//for the line feed commands
		source.pSource_current += textSize_current;
		textSize += textSize_current;
		++i;		
	}
	if ( pSourceSize != NULL )
	{
		*pSourceSize = textSize;
	}

	return STATUS_SUCCESS;
}
//--------------------------------------------------------------------------
//Get number of variables in structure. If structure has a structure member
//than it will add the count of that struct too.
//
// structure - Pointer to structure.
//
// return - Number of members in structure OR STATUS_ERROR.
//
//ex.
//struct ONE{
//	DWORD onet;
//  DWORD twot;
//};
//
//struct TWO{
//	DWORD twot;
//	ONE   one;
//}
//return = 3
//
//Notice:
//If there is an undefined byte in the structure the byte will be skipped
//as nothing happened.
//
unsigned long ida2softice_c::SYMD_GetStructureMemberCount
(
	struc_t*	pStructure
)
{
	member_t*		pMember;
	unsigned long	memberCount = 0;
	unsigned long	memberOffset;

//Test if we have any members.
	if ( pStructure->memqty == 0 )
	{
		return 0;
	}

//Test if the struct is union. Unions has only one 'real' member.
	if ( pStructure->is_union() )
	{
		return 1;
	}

//Get the number of members.
	memberOffset = get_struc_first_offset( pStructure );

	unsigned long memqty = pStructure->memqty;
	while ( memqty > 0 )
	{
		if ( memberOffset == BADADDR )
		{
			msg( "I2S: Problems getting structure members.\n" );
			return STATUS_ERROR;
		}

	//Get structure member.
		pMember = get_member( pStructure, memberOffset );
		if ( pMember != NULL )
		{
		//Is current member a structure or an union?
			if ( ( pMember->flag & DT_TYPE ) == FF_STRU )
			{
			//Yes it's a structure or union -> get it's member count.
				memberCount += SYMD_GetStructureMemberCount( get_sptr( pMember ) );
			}
			else
			{
			//No it's a common member.
				++memberCount;
			}

		//Update.
			--memqty;
		}

	//Get next member.
		memberOffset = get_struc_next_offset( pStructure, memberOffset );
	}

//Return number of structure members.
	return memberCount;
}
