//
//  PlugIn name: IDA to SoftIce converter (only for 32bit programs)
//
//
//  This file contains functions that are intended for initializing.
//
//
#include "i2s.h"

//--------------------------------------------------------------------------
//One time definitions and test if SoftIce is loaded.
//
//return : STATUS_SUCCESS/STATUS_FAILED
STATUS	ida2softice_c::Initialize_oneTime
(
	void
)
{
//Are plugIn directories/flags set? (dirs, package type, .... )
	if ( i2sNetNode.altval( NVAL_I2S__FLAGS, atag ) == 0 )
	{
	//Not yet -> open setupWindow.
	//
	//DialogBoxParam parameters: - 0 -> enable ALWAYSDEFAULT
	//                           - 1 -> open dialog
	//
		BOOL	Status;

		Status = DialogBoxParam(
		  GetModuleHandle("i2s.plw"),
		  (char*)DIALOG_SETUP,
		  hWnd_pleaseWait,
		  SetupDialogBox,
		  0
		);
		if ( Status == FALSE )
		{
			msg( "I2S: Failed to initialize plugIn.\n" );
			return STATUS_FAILED;
		}
	}

//Check if original file segment data was retrieved.
	if ( sectionTablesNetNode.long_value() == BADNODE )
	{
	//Not defined yet -> get segment info.
		STATUS Status = SetSectionTableNetnode();
		if ( Status != STATUS_SUCCESS )
		{
			msg( "I2S: Failed to get segment data from original file.\n" );
			return STATUS_FAILED;
		}
	}

//Check if source files boundaries are defined and if the definitions are at the right version.
	if (
	  ( sourceNetNode.long_value() == BADNODE )
	  ||
	  ( i2sNetNode.altval( NVAL_I2S__SOURCE_INFO_VER, atag ) < SOURCE_INFO_VERSION )
	)
	{
	//Not defined yet or wrong version -> define them.
		msg( "I2S: Source file boundaries creation (only once).\n" );
		i2s.SetSourceFilesBoundaries();
	}

//Initialize import names variables.
	Imports_initialize();

//Is SoftIce loaded? (try to get a handle to softIce)
	int		i = 0;
	char*	pSicePaths[3];
	char	siceProtectedPath[24];

	//Try to get protected name and intialize pSicePaths.
	if ( S_FAILED( GetSIceProtectedPath( (char*)&siceProtectedPath, sizeof( siceProtectedPath ) ) ) )
	{
		pSicePaths[0] = &"\\\\.\\NTICE";
		pSicePaths[1] = NULL;
	}
	else
	{
		pSicePaths[0] = (char*)&siceProtectedPath;
		pSicePaths[1] = &"\\\\.\\NTICE";
	}
	pSicePaths[2] = NULL;

	//Try which one of the paths works on current OS.
	while ( pSicePaths[i] != NULL )
	{
		HANDLE		hFile;

	//Try to get softIce handle.
		hFile = CreateFile(
		  pSicePaths[i],
		  GENERIC_READ,
		  FILE_SHARE_READ | FILE_SHARE_WRITE,
		  NULL,
		  OPEN_EXISTING,
		  FILE_ATTRIBUTE_NORMAL,
		  0
		);
		if ( hFile != INVALID_HANDLE_VALUE )
		{
		//We got the handle -> current path will be used for softIce access.
			CloseHandle( hFile );
			//Save current path
			qstrncpy( sicePath, pSicePaths[i], sizeof( sicePath ) );
			plugInFlags |= SOFTICE_LOADED;
			break;
		}

	//Update.
		++i;
	}
	return STATUS_SUCCESS;
}

//--------------------------------------------------------------------------
//Get current file segment information.
//(Not all required data is available in IDA so we'll get the info
//directly from the file.)
//
//Procedure will save all the data into sectionTablesNetNode.
//
// return - STATUS_SUCCESS/STATUS_FAILED

//Check if you can get this info out of:
//netnode penode("$ PE header");
//ea_t loaded_base = penode.altval(-2);
unsigned long ida2softice_c::SetSectionTableNetnode
(
	void
)
{
	IMAGE_DOS_HEADER			dosHeader;
	HANDLE						hFile;
	HLOCAL						hMemory;
	char						inputFilePath[MAXSTR];
	unsigned long				nBytesRead;
	unsigned long				nOfSections;
	BYTE*						pHeaders;
	OPENFILENAME				openFile;
	peHeader_sectionTables_s	sectionData;
	unsigned long				size;

//Get input file path.
//
//Copy it as sometimes IDA changes data after calling one of it's procedures.
	get_input_file_path( inputFilePath, sizeof( inputFilePath ) );

//Get a handle to file -> if I2S fails it opens an Open File dialogBox, so that
//user can point to the file we need.
	hFile = CreateFile(
	  (char*)&inputFilePath,
	  GENERIC_READ,
	  FILE_SHARE_READ,
	  NULL,
	  OPEN_EXISTING,
	  FILE_ATTRIBUTE_ARCHIVE,
	  NULL
	);
	if ( hFile == INVALID_HANDLE_VALUE )
	{
		msg( "I2S: Executable could not be found. Hint: change the name in setup dialogBox.\n"	);

	//Zero out openFile structure.
		ZeroMemory( &openFile, sizeof( OPENFILENAME ) );

	//Initialize OPENFILENAME struct.
		openFile.lStructSize = sizeof( OPENFILENAME );
		openFile.nMaxFile = MAXSTR;
		openFile.lpstrFilter = "All Files (*.*)\x0 *.*\x0\x0";
		openFile.nFilterIndex = 1;
		openFile.lpstrTitle = "IDA to Softice - Please locate the executable";
		openFile.Flags = OFN_FILEMUSTEXIST | OFN_LONGNAMES | OFN_PATHMUSTEXIST;
	}
	while ( hFile == INVALID_HANDLE_VALUE )
	{
	//I2S failed to open input file -> user should point to the file required ->
	//open OpenDialogBox.
	//
	//If user will not point to the file the I2S will fail.
		//Initialize.
		get_root_filename( inputFilePath, sizeof( inputFilePath ) );

		openFile.lpstrFile = inputFilePath;

		//Open OpenDialogBox.
		if ( FALSE == GetOpenFileName( &openFile ) )
		{
		//User canceled the I2S'es request -> failed to get the file.
			get_root_filename( inputFilePath, sizeof( inputFilePath ) );
			msg( "I2S: Could not open the input file (%s).\n", (char*)&inputFilePath );
			return STATUS_FAILED;
		}

	//Try to open the new file -> if we came to here the file exists for sure ->
	//but we'll stay paranoid so that everything will be 100% as it should be.
		hFile = CreateFile(
		  (char*)&inputFilePath,
		  GENERIC_READ,
		  FILE_SHARE_READ,
		  NULL,
		  OPEN_EXISTING,
		  FILE_ATTRIBUTE_ARCHIVE,
		  NULL
		);
	}

//Open the file -> get just DOS header for now.
	if ( FALSE == ReadFile( hFile, &dosHeader, sizeof(dosHeader), &nBytesRead, NULL ) )
	{
		CloseHandle( hFile );
		get_root_filename( inputFilePath, sizeof( inputFilePath ) );
		msg( "I2S: Failed to read the input file (%s).\n", (char*)&inputFilePath );
		return STATUS_FAILED;
	}

//Check if we really got the executable file.
	if ( dosHeader.e_magic != IMAGE_DOS_SIGNATURE )
	{
		CloseHandle( hFile );
		msg( "I2S: Wrong file type.\n" );
		return STATUS_FAILED;
	}

//We got the executable -> read the NumberOfSections value.
	//Set the pointer to start of the PE header.
	SetFilePointer( hFile, dosHeader.e_lfanew, NULL, FILE_BEGIN );

	//Get the memory.
	size = sizeof(IMAGE_NT_SIGNATURE) + sizeof(IMAGE_FILE_HEADER) + sizeof(IMAGE_OPTIONAL_HEADER);
	hMemory = LocalAlloc( LMEM_FIXED, size );
	pHeaders = (BYTE*)hMemory;
	if ( pHeaders == NULL )
	{
		CloseHandle( hFile );
		msg( "I2S: Failed to allocate memory(GetSegmentInfo).\n" );
		return STATUS_FAILED;
	}
	//Read the file.
	if ( FALSE == ReadFile( hFile, pHeaders, size, &nBytesRead, NULL ) )
	{
		LocalFree( hMemory );
		CloseHandle( hFile );

		get_root_filename( inputFilePath, sizeof( inputFilePath ) );
		msg( "I2S: Failed to read the input file (%s).\n", (char*)&inputFilePath );
		return STATUS_FAILED;
	}

	//Check if we got a PE file.
	if ( *( (unsigned long*)pHeaders ) != IMAGE_NT_SIGNATURE )
	{
	//No it's not a PE header -> exit with error.
		LocalFree( hMemory );
		msg( "I2S: Wrong file type.\n" );
		return STATUS_FAILED;
	}

	//Get number of sections defined.
	pHeaders += sizeof( IMAGE_NT_SIGNATURE );
	nOfSections = ( (IMAGE_FILE_HEADER*)pHeaders )->NumberOfSections;
	LocalFree( hMemory );

//Get sections data (file pointer is already at IMAGE_SECTION_HEADER section.
	//Get the memory.
	size = sizeof( IMAGE_SECTION_HEADER ) * nOfSections;
	hMemory = LocalAlloc( LMEM_FIXED, size );
	pHeaders = (BYTE*)hMemory;
	if ( pHeaders == NULL )
	{
		CloseHandle( hFile );
		msg( "I2S: Failed to allocate memory(GetSegmentInfo).\n" );
		return STATUS_FAILED;
	}
	//Read the file.
	if ( FALSE == ReadFile( hFile, pHeaders, size, &nBytesRead, NULL ) )
	{
		LocalFree( hMemory );
		CloseHandle( hFile );
		get_root_filename( inputFilePath, sizeof( inputFilePath ) );
		msg( "I2S: Failed to read the input file (%s).\n", (char*)&inputFilePath );
		return STATUS_FAILED;
	}

//Close the file handle, we don't need it anymore.
	CloseHandle( hFile );

//Get the segment info we need.
	sectionTablesNetNode.set_long( nOfSections );

	for ( unsigned short i = 0; i < nOfSections; i++ )
	{
		IMAGE_SECTION_HEADER*		pSection = (IMAGE_SECTION_HEADER*)( pHeaders + sizeof( IMAGE_SECTION_HEADER )*i );

	//Clear sectionData structure.
		ZeroMemory( &sectionData, sizeof( sectionData ) );

	//Set structure values.
		sectionData.nSection = i;
		if ( pSection->SizeOfRawData >= pSection->Misc.VirtualSize )
			sectionData.sizeOfRawData = pSection->SizeOfRawData;
		else
			sectionData.sizeOfRawData = pSection->Misc.VirtualSize;

		*( (BYTE*)&(pSection->Name) + IMAGE_SIZEOF_SHORT_NAME ) = '\x0';
		qstrncpy( sectionData.name, (char*)&pSection->Name, sizeof( sectionData.name ) );

	//Save data to netnode.
		sectionTablesNetNode.supset( i, &sectionData, sizeof( sectionData ), stag );
	}

//Free resources and exit.
	LocalFree( hMemory );
	return STATUS_SUCCESS;
}

