//
//  PlugIn name: IDA to SoftIce converter (only for 32bit programs)
//
//
//  This file contains function that creates and loads nms file.
//
//
#include "i2s.h"

//--------------------------------------------------------------------------
//Get mapped nms file.
//
// set - NM32 creationNumber (look into nm32.h)
//
// returns - pointer or STATUS_FAILED
//
NM32_HEADER* ida2softice_c::CreateNmsDatabase
(
	void
)
{
	STATUS			Status;
	ssize_t			ret;

	int				i;
	unsigned long	iName;
	segment_t*		pSegment;
	ea_t			segmentEnd;
	ea_t			segmentEnd_current;
	int				nSegments;	//How many segments IDA made?
								//(can be bigger than number of selectors ( +iData,... )
								//
								//segment-> current segment in IDA dissasebly
								//selector-> current segment in original file

	unsigned long	currentSelector;
	unsigned long	lastSelector;	//this is also current file selector
	unsigned long	previousSelector;

	ea_t			currentAddress;

	unsigned long	nProcedures;
	unsigned long	proceduresLocalVariablesCount;
	unsigned long	proceduresRenamedRegisterCount;

	unsigned long	nCommandLines;
	unsigned long	nVLP;	//sum of: nVariables, nLabels and nProcedures
							//So we'll know how many names are in dissasembly.
	unsigned long	nameBuffer[2];

	unsigned long	fileSize = 0;
	unsigned long	fileSize_temp;
	unsigned long	indexInSourceNetNodeSupVal;
	unsigned long	nLineTables = 0;
	unsigned long	offInSFDefinition = 0;
	unsigned long	sizeOfSourceListing = 0;
	sourceInfo_s*	pSourceInfo = NULL;
	sourceInfo_s	sourceInfo;

//Allocate STTB memory.
	Status = InitializeMemory_STTB();
	if ( S_FAILED( Status ) )
	{
		return STATUS_FAILED;
	}

//Allocate memory for source listing.
//
//The NVAL_I2S__MAX_SOURCE_SIZE value in main netnode contains the
//biggest value of source listing.
//We'll use this estimate for allocating source listing memory.
//So we wont need to reallocate the memory.
//
//Start size = maxSize + 1MB
	Status = SYMD_AllocateMemory__SourceListing( i2sNetNode.altval( NVAL_I2S__MAX_SOURCE_SIZE, atag ) + 0x100000 );
	if ( S_FAILED( Status ) )
	{
		return STATUS_FAILED;
	}

//Get the number of all resources we have (number of functions, labels,.....).
	if ( includeSource == TRUE )
	{
		msg( "I2S: Gathering segments information and saving source.\n" );
	}
	else
	{
		msg( "I2S: Gathering segments information.\n" );
	}

	//Initialize variables.
	previousSelector = 0;
	lastSelector = 1;
	indexInSourceNetNodeSupVal = 0;

	//Get number of segments that are used in IDA, but not the number of
	//them all in original file.
	nSegments = get_segm_qty();
	for ( i = 0; i < nSegments; i++ )
	{
	//Reset variables.
		nProcedures = 0;
		proceduresLocalVariablesCount = 0;
		proceduresRenamedRegisterCount = 0;

		nCommandLines = 0;
		nVLP = 0;

	//Get segment.
		pSegment = getnseg(i);

	//Check for proper segment.
	//
	//IDA can create a segment when the segment doesn't even exist.
	//i.e. It creates a segment out of the file header -> this should not happen.
	//We will simply exclude this kind of segment -> we wont process it.
		if ( pSegment->sel > sectionTablesNetNode.long_value() )
		{
		//A faulty segment -> skip it.
			continue;
		}

	//-------------------------------------------------------------------------
	//1st part:
	//
	//Get segment selector (the number of the segment in original file).
		currentSelector = pSegment->sel;

	//Do we have to allocate new segment data structure?
	//(Read about this below.)
		if ( previousSelector < currentSelector )
		{
		//Previous real segment number is smaller than current one -> we have
		//a true new segment.
		//
		//Create new segment data.
			if ( S_FAILED( SYMD_AllocateMemory__SEGMENT_DATA() ) )
			{
				return STATUS_FAILED;
			}

		//Check if need to create new segment data structure, because of IDA's
		//segment behaviour.
		//
		//Be carefull at:
		//- IDA can make more than one segment from a single selector(original file segment)
		//- IDA can throw out some selectors (original file segments)
		//  (like RSRC or somethings -> and if this selector is before any CODE or DATA
		//	selector any symbols in those selectors will be wrong -> everything will be
		//	one selector off)
		//
			while ( lastSelector < currentSelector )
			{
				unsigned long				iName_temp;
				peHeader_sectionTables_s	sectionTables;

			//Initialize.
				ret = sectionTablesNetNode.supval( lastSelector, &sectionTables, sizeof( sectionTables ), stag );
				if ( ret == -1 )
				{
					msg( "I2S: Failed to get original segment data. Please contact author about this.\n" );
					return STATUS_FAILED;
				}

			//If we came to here than there is a segment missing in IDA.
			//
			//Define missing segment data.
				pSegmentCurrent->includedInIDA = FALSE;
				pSegmentCurrent->nSeg = lastSelector;
				iName_temp = STTB_SetStringTable( sectionTables.name );
				if ( iName_temp == STATUS_ERROR )
				{
					return STATUS_FAILED;
				}
				pSegmentCurrent->iName = iName_temp;
				pSegmentCurrent->physSize = sectionTables.sizeOfRawData;

			//Update
				++lastSelector;

			//Get new segment data structure.
				if ( S_FAILED( SYMD_AllocateMemory__SEGMENT_DATA() ) )
				{
					return STATUS_FAILED;
				}
			}

		//At this point original file selector and IDA's segments are synchronized.
		//So define segment data.
		//
		//Notice:
		//As IDA can make '.idata' segment out of another segment we should get the
		//real segment name and not the '.idata' name.
			pSegmentCurrent->includedInIDA = TRUE;
			pSegmentCurrent->nOfIDAsSegments = 1;
			pSegmentCurrent->segmentIndexInIDA = i;
			pSegmentCurrent->nSeg = currentSelector;

			//Is current segment '.idata' segment?
			//
			//If it is check if it is made from another segment ->
			//as this is the first one we should check the next IDA segment selector
			//(if it is the same).
			//
			//Notice:
			//If the '.idata' segment will be the second or n-th IDA segment
			//than we don't have to worry as the right name will already be set.
			if ( pSegment->type == SEG_XTRN )
			{
				segment_t*	pSegment_temp;

			//It's '.idata' segment -> check if next segment has the same selector.
				if ( ( i + 1 ) < nSegments )
				{
				//We have at least one more segment available.
					pSegment_temp = getnseg( i+1 );
					if ( pSegment_temp->sel == currentSelector )
					{
					//Yes next segment has the same selector. This means that '.idata'
					//segment was made out of it -> use next segment name.
						char	segmentName[12];

						get_true_segm_name( pSegment_temp, segmentName, sizeof( segmentName ) );
						iName = STTB_SetStringTable( segmentName );
					}
					else
					{
					//No next segment doesn't have the same selector. This means that
					//there is really an '.idata' segment present in file -> use
					//'.idata' segments name.
						goto CND_true_segment_name;
					}
				}
				else
				{
				//This is the last segment.
				//
				//This means that there is really an '.idata' segment present
				//in file -> use '.idata' segments name.
					goto CND_true_segment_name;
				}
			}
			else
			{
			//It's not '.idata' segment -> proceed normaly.
			CND_true_segment_name:
				char	segmentName[12];

				get_true_segm_name( pSegment, segmentName, sizeof( segmentName ) );
				iName = STTB_SetStringTable( segmentName );
			}
			if ( iName == STATUS_ERROR )
			{
				return STATUS_FAILED;
			}
			pSegmentCurrent->iName = iName;

		//Define segment begining.
			minEA = pSegment->startEA;
			pSegmentCurrent->startEA = pSegment->startEA;

		//Update.
			++lastSelector;
		}
		else
		{
		//If we are here than IDA made two or more segments from one.
		//We need to know how many if we want to index sourceNetNode right.
		//That is because sourceNetNode's index runs with IDA's segment numbers
		//and not how much we will create them.
			 ++pSegmentCurrent->nOfIDAsSegments;
		}

	//Define rest of the segment's data.
		segmentEnd = pSegment->endEA;
		pSegmentCurrent->endEA = segmentEnd;

	//Update.
		previousSelector = currentSelector;

	//-------------------------------------------------------------------------
	//2nd part:
	//
	//Get the 1st source file in segment.
	//
	//Notice:
	//Be careful if first segments are data segments -> then don't update the sourceInfo structure and
	//don't reset the currentLineInSource variable.
		if ( ( indexInSourceNetNodeSupVal > 0 ) || ( indexInSourceNetNodeSupVal == 0  ) && ( pSourceInfo == NULL ) )
		{
			ret = sourceNetNode.supval( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo ), stag );
			if ( ret == -1 )
			{
			//Doesn't exist... -> set the flag.
				pSourceInfo = NULL;
			}
			else
			{
			//Exists... -> set the flag.
				pSourceInfo = &sourceInfo;

			//Update runTime variables.
				sourceInfo.sourceSection.offSourceLines = 0;
				sourceInfo.sourceSection.offInSFDef = offInSFDefinition;

			//Is current source file enabled->yes update SFDef offset
				if ( includeSource == TRUE )
				{
					currentSourceFileEnabled = sourceInfo.enable_source;
				}
				else
				{
					currentSourceFileEnabled = FALSE;
				}

			//Save the source file info back to the netnode.
				sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo ), stag	);

			//Reset.
				//Current source file doesn't have any command lines yet.
				nLineTables = 0;
			}
		}


	//-------------------------------------------------------------------------
	//3rd part:
	//
	//Start analysing -> check every "byte" in the segment.
		currentAddress = pSegment->startEA;
		while ( currentAddress < segmentEnd )
		{
		//For debugging
			//msg("currentAddress: 0x%X \n", currentAddress);

		//Check for user break.
			if ( 0 != wasBreak() )
			{
				plugInFlags |= USER_CANCELLED;
				return STATUS_FAILED;
			}

		//Set progressBar position.
			pBar.SetPosition();

		//Do we have to change sourceFile?
		//
		//Some info:
		//If there are DATA segments before first CODE IDA listing will be 'prepended'
		//to 1st source file.
		//
		//Every DATA segments listing will be appended to 'last' source file.
		//
			//Are we still in source file boundaries?
			if ( currentAddress > sourceInfo.endEA )
			{
			//We are over the boundary -> update the current sourceFile info and switch to new one.
			//
			//Check if only DATA segments remained.
			//Notice:
			//DATA segments will be updated at the end of segment loop.
			//
				//Was the source file info defined?
				if ( pSourceInfo != NULL )
				{
				//Yes source file info was defined.
				//
				//Update runTime variables.
					sourceInfo.sourceSection.nLineTables = nLineTables;

					sourceInfo.sourceFile.offSource = sizeOfSourceListing;	//before updating
					sourceInfo.sourceFile.sourceLength = fileSize;

				//Save current sourceFile info.
					sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo_s ), stag );

				//Update variables.
					nLineTables = 0;
					++indexInSourceNetNodeSupVal;
					sizeOfSourceListing += fileSize;
					fileSize = 0;

				//Increase SFDef offset if current source is enabled
					if ( currentSourceFileEnabled == TRUE )
					{
						offInSFDefinition += sizeof( SEGDATA_SOURCEFILE );
					}

				//Get next source file info.
					ret = sourceNetNode.supval( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo ), stag );
					if ( ret == -1 )
					{
					//Doesn't exist... -> set the flag.
						pSourceInfo = NULL;
					}
					else
					{
					//Exist... -> set the flag.
						pSourceInfo = &sourceInfo;

					//Update runTime variables.
						sourceInfo.sourceSection.offSourceLines = nCommandLines * sizeof( SEGDATA_SOURCELINES );
						sourceInfo.sourceSection.offInSFDef = offInSFDefinition;

					//Is current source file enabled->yes update SFDef offset
						if ( includeSource == TRUE )
						{
							currentSourceFileEnabled = sourceInfo.enable_source;
						}
						else
						{
							currentSourceFileEnabled = FALSE;
						}

					//Save the source file info back to the netnode.
						sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo_s ), stag );
					}
				}
			}

		//Are symbols enabled in current file?
			if ( sourceInfo.enable_symbols == FALSE )
			{
			//No symbols are not enabled -> skip file (skip all the addresses).
			//
			//But be careful if current source file is the last one.
				if ( currentAddress < sourceInfo.endEA )
				{
				//We are in current file limits -> update the address.
					currentAddress = sourceInfo.endEA + 1;
					continue;
				}
			}

		//Check every byte.
			flags_t	flags = getFlags( currentAddress );

			//Does current command have a label or a name?
			if ( has_any_name( flags ) )
			{
			//Yes it has a label or a name.
			//
			//Exclude extern definitions.
				if ( pSegment->type != SEG_XTRN )
				{
				//Exclude the unk_xxxxxxxx names too as they are just making a mess
				//in the i2s. It can happen that there is an 'unk_xxxxxxxx' name
				//even if it's not shown in source listing.
				//Now sometimes the name will be detected by the i2s, but sometimes not ->
				//I don't know why, but I'll exclude this from the VLPs ->
				//they are not used anyway so they just use space.
				//
				//Search for 'unk_'.
				//(Exclude the zero length names also.)
					get_name( currentAddress, currentAddress, (char*)&nameBuffer, sizeof( nameBuffer ) );
					if ( ( nameBuffer[0] != '_knu' ) && ( ( (char*)&nameBuffer )[0] != '\x0' ) )
					{
						++nVLP;
					}
				}
			}

			//Dissasembly listing.
			if ( currentSourceFileEnabled == TRUE )
			{
			//Source listing is enabled.
			//
			//Notice:
			//Exclude undefined data lines from source listing as if those
			//are included source listing can soon become more than 100MB in size.
			//
			//But we need to be careful here as align directives can also represent
			//code lines (like int3 or something).
			//But as IDA defines align directive as data type we'll have to
			//trow it out so that nCommandLines will be correct.
				if ( isCode( flags ) )
				{
				//Current line is a code line.
					++nLineTables;
					++nCommandLines;

				//Save source listing and get it's text size.
					Status = SYMD_GetSizeOfSourceListingAndSaveSource( currentAddress, &fileSize_temp );
					if ( S_FAILED( Status ) )
					{
						return STATUS_FAILED;
					}
					fileSize += fileSize_temp;
				}
				else if ( isData( flags ) )
				{
				//Trow out align directives -> they just use memory.
				//
				//Be careful IDA assignes align flag even to some real data.
				//That data must be included.
					if ( ( ( flags & DT_TYPE ) != FF_ALIGN ) || ( has_any_name( flags ) ) )
					{
					//It's not an align directive or it has a name ->
					//save source listing and get it's text size.
						Status = SYMD_GetSizeOfSourceListingAndSaveSource( currentAddress, &fileSize_temp );
						if ( S_FAILED( Status ) )
						{
							return STATUS_FAILED;
						}
						fileSize += fileSize_temp;
					}
				}
			}

			//Is it a function?
			if ( isFunc( flags ) )
			{
				func_t	procedure;

			//Increase number of procedures in segment.
				++nProcedures;

			//Get function.
				CopyMemory( &procedure, get_func( currentAddress ), sizeof( func_t ) );

			//Get procedure variables.
			//
			//Notice:
			//Get only if function is visible (not hidden).
				if ( is_visible_func( &procedure ) )
				{
				//Get number of arguments and local variables.
				//
				//Does procedure have a frame
				//(are there any arguments or local variables set)?
					if ( ( procedure.flags & FUNC_FRAME ) != 0 )
					{
						unsigned long	argCount;
						unsigned long	lVarCount;

						Status = SYMD_GetNumberOfProcedureVariables( &procedure, &lVarCount, &argCount, NULL );
						if ( S_FAILED( Status ) )
						{
							return STATUS_FAILED;
						}

					//Add the number of arguments and local variables to the segment.
						proceduresLocalVariablesCount += argCount + lVarCount;
					}

				//Get number of renamed registers.
					proceduresRenamedRegisterCount += procedure.regvarqty;
				}
				else
				{
				//Function is not visible, so we'll jump over it.
					currentAddress = procedure.endEA;
					continue;
				}
			}

		//Update.
			currentAddress += get_item_size( currentAddress );
		}

	//-------------------------------------------------------------------------
	//4th part:
	//
	//Every segment with code will have it's source file and if after this segment
	//there will be any DATA segments(or similar) than IDA listing will be appended
	//to previous source file.
	//
	//Did this segment have any code in it
		if ( nCommandLines != 0 )
		{
		//Yes current segment had code lines.
		//
		//Update runTime variables
			sourceInfo.sourceSection.nLineTables = nLineTables;

			sourceInfo.sourceFile.offSource = sizeOfSourceListing;	//before updating
			sourceInfo.sourceFile.sourceLength = fileSize;

		//Save current sourceFile info.
			sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo_s ), stag );

		//Update variables.
			nLineTables = 0;
			++indexInSourceNetNodeSupVal;
			sizeOfSourceListing += fileSize;
			fileSize = 0;

		//Increase SFDef offset if current source is enabled
			if ( currentSourceFileEnabled == TRUE )
			{
				offInSFDefinition += sizeof( SEGDATA_SOURCEFILE );
			}

		//Set next sourceInfo.
		//
		//Well this will happen at beginning of segment FOR loop.
		}
		else
		{
		//Current segment didn't have any code in it.
		//Append source to previous source file or if this is the first source file prepend the listing to it.
		//
		//Get last source file info.
			sourceInfo_s	sourceInfo_last;

			ret = sourceNetNode.supval( indexInSourceNetNodeSupVal - 1, &sourceInfo_last, sizeof( sourceInfo_last ), stag );
			if ( ret == -1 )
			{
			//Previous source file is not available -> is it an error or is the current file the first one.
				if ( pSourceInfo == NULL )
				{
				//Error -> we couldn't get the last source file even if we don't have any more left.
					msg( "I2S: Failed to get source file information.\n" );
					return STATUS_FAILED;
				}
				//else
				//{
				//Current source file is the first one -> just leave all variables as they are and
				//source listing will be prepended to it.
				//}
			}
			else
			{
			//We got previous source file -> appened source listing to it.
			//
			//Update runTime variables.
				sourceInfo_last.sourceFile.sourceLength += fileSize;

			//Save sourceFile info back to netnode.
				sourceNetNode.supset( indexInSourceNetNodeSupVal - 1, &sourceInfo_last, sizeof( sourceInfo_last ), stag );

			//Update variables.
				sizeOfSourceListing += fileSize;
				fileSize = 0;
			}
		}

	//-------------------------------------------------------------------------
	//5th part:
	//
	//Alocate memory.
		char	debBuffer[128];
		qsnprintf(
		  debBuffer,
		  sizeof( debBuffer ),
		  "Segment IDA: %d, file: %d, nVLP: %d(0x%X)\n",
		  i,
		  pSegmentCurrent->nSeg,
		  nVLP,
		  nVLP
		);
		OutputDebugString( debBuffer ); 


		//Define count variables.
		pSegmentCurrent->nProcDefinitions += nProcedures;
		pSegmentCurrent->nLocalVariables += proceduresLocalVariablesCount;
		pSegmentCurrent->nRegisters += proceduresRenamedRegisterCount;
		pSegmentCurrent->nSourceLines += nCommandLines;
		pSegmentCurrent->nVLP += nVLP;	//it contains procedure names too
	}

//Update source structure.
	unsigned long	listingSize = i2sNetNode.altval( NVAL_I2S__MAX_SOURCE_SIZE, atag );

	//Was current listing size bigger than the last one?
	if ( sizeOfSourceListing > listingSize )
	{
	//Yes -> update for next time.
		i2sNetNode.altset( NVAL_I2S__MAX_SOURCE_SIZE, sizeOfSourceListing, atag );
	}

//Test if something was defined in the last segments.
//
//If few of the last segments are empty than we will release their memory.
//NOTICE!
//This is true only for the last segments. There can be empty segments between the
//CODE or DATA segments or more if they exist they must be present for later definitions.
	int	somethingInSegment =
	  pSegmentCurrent->nProcDefinitions
	  |
	  pSegmentCurrent->nSourceLines
	  |
	  pSegmentCurrent->nVLP;
	while ( somethingInSegment == 0 )
	{
	//The last segment doesn't have any data defined -> free it's memory.
	//
	//Update.
		if ( pSegmentCurrent->pPrevious != NULL )
		{
		//We have at least 1 more segment in memory.
			pSegmentCurrent = pSegmentCurrent->pPrevious;
			LocalFree( pSegmentCurrent->pNext );
			pSegmentCurrent->pNext = NULL;
			--nSegmentsPresent;

		//Is there any data define in the last segment.
			somethingInSegment =
			  pSegmentCurrent->nProcDefinitions
			  |
			  pSegmentCurrent->nSourceLines
			  |
			  pSegmentCurrent->nVLP;
		}
		else
		{
		//This is the last segment in memory.
			LocalFree( pSegmentCurrent );
			pSegments = NULL;
			pSegmentCurrent = NULL;
			--nSegmentsPresent;

		//There is no point making nms database if there are no data to define -> finish.
			msg( "I2S: Couldn't define the segments or no CODE segment present.\n" );
			return STATUS_FAILED;
		}
	}

//Convert source names to their iName representatives.
	if ( S_FAILED( DefineSourceFilesNames() ) )
	{
		return STATUS_FAILED;
	}

//Initialize other sections memory and intialize NMTP predefined types.
	if ( S_FAILED( i2s.InitializeMemory() ) )
	{
		return STATUS_FAILED;
	}

//Get data.
	unsigned long	currentLineInSource;
	pSourceInfo = NULL;

	msg( "I2S: Creating NMS database.\n" );
	pBar.ResetPosition();

	//Initialize.
	indexInSourceNetNodeSupVal = 0;
	pSegmentCurrent = pSegments;
	while ( pSegmentCurrent != NULL )
	{
		//nProcedures = 0;
		nVLP = 0;

	//Is current segment included in IDA?
		if ( pSegmentCurrent->includedInIDA == FALSE )
		{
		//Current segment is not included in IDA -> go to next one.
			pSegmentCurrent = pSegmentCurrent->pNext;
			continue;
		}

	//Get segment.
	//
	//Note:
	//You need to be careful if current i2s segment is build from many IDA segments.
		pSegment = getnseg( pSegmentCurrent->segmentIndexInIDA );
		segmentEnd_current = pSegment->endEA;

	//Initialize variables.
		currentAddress = pSegmentCurrent->startEA;
		minEA = pSegmentCurrent->startEA;
		segmentEnd = pSegmentCurrent->endEA;

	//Allocate memory for variables.
		Status = SYMD_AllocateMemory__SYMD();
		if ( S_FAILED( Status ) )
		{
			return STATUS_FAILED;
		}

	//-------------------------------------------------------------------------
	//1st part:
	//
	//Get the 1st source file in segment.
	//
	//Notice:
	//Be careful if first segments are data segments -> then don't update the sourceInfo structure and
	//don't reset the currentLineInSource variable.
		if ( ( indexInSourceNetNodeSupVal > 0 ) || ( indexInSourceNetNodeSupVal == 0  ) && ( pSourceInfo == NULL ) )
		{
			ret = sourceNetNode.supval( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo ), stag );
			if ( ret == -1 )
			{
			//Doesn't exist... -> set the flag.
				pSourceInfo = NULL;
			}
			else
			{
			//Exist... -> set the flag.
				pSourceInfo = &sourceInfo;

			//Is current source file enabled?
				if ( includeSource == TRUE )
				{
					currentSourceFileEnabled = sourceInfo.enable_source;
				}
				else
				{
					currentSourceFileEnabled = FALSE;
				}

			//Reset.
				currentLineInSource = 1;
			}
		}

	//-------------------------------------------------------------------------
	//2nd part:
	//
	//Start analysing -> check every "byte" in the segment.
		while ( currentAddress < segmentEnd )
		{
		//For debugging
//			msg("currentAddress: 0x%X \n", currentAddress);

		//Check for user break.
			if ( 0 != wasBreak() )
			{
				plugInFlags |= USER_CANCELLED;
				return STATUS_FAILED;
			}

		//Set progressBar position.
			pBar.SetPosition();

		//Do we have to change sourceFile?
		//
		//Some info:
		//If there are DATA segments before first CODE IDA listing will be 'prepended'
		//to 1st source file.
		//
		//Every DATA segments listing will be appended to 'last' source file.
		//
			//Are we still in source file boundaries?
			if ( currentAddress > sourceInfo.endEA )
			{
			//We are over the boundary -> update the current sourceFile info
			//and switch to new one.

			//Check if only DATA segments remained.
			//
			//Notice:
			//DATA segments will be updated at the end of segment loop.
			//
				//Was the source file info defined?
				if ( pSourceInfo != NULL )
				{
				//Yes source file info was defined.
				//
				//Update runTime variables.

				//Update variables.
					currentLineInSource = 1;
					++indexInSourceNetNodeSupVal;

				//Get next source file info.
					ret = sourceNetNode.supval( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo ), stag );
					if ( ret == -1 )
					{
					//Doesn't exist... -> set the flag.
						pSourceInfo = NULL;
					}
					else
					{
					//Exist... -> set the flag.
						pSourceInfo = &sourceInfo;

					//Is current source file enabled?
						if ( includeSource == TRUE )
						{
							currentSourceFileEnabled = sourceInfo.enable_source;
						}
						else
						{
							currentSourceFileEnabled = FALSE;
						}
					}
				}
			}

		//Are symbols enabled in current file?
			if ( sourceInfo.enable_symbols == FALSE )
			{
			//No symbols are not enabled -> skip file.
			//
			//But be careful if current source file is the last one.
				if ( currentAddress < sourceInfo.endEA )
				{
				//We are in current file limits -> update the address.
					currentAddress = sourceInfo.endEA + 1;
					continue;
				}
			}

		//Get IDA segment.
			if ( currentAddress >= segmentEnd_current )
			{
			//Current i2s segment is build from more than 1 IDA segment.
			//
			//Get next IDA segment;
				pSegment = getseg( currentAddress );
				segmentEnd_current = pSegment->endEA;
			}

		//Check every byte.
			flags_t	flags = getFlags( currentAddress );

			//Does current command have a label or a name?
			if ( has_any_name( flags ) )
			{
			//Yes it has a label or a name.
			//
			//Exclude extern definitions.
				if ( pSegment->type != SEG_XTRN )
				{
				//Exclude the unk_xxxxxxxx names too as they are just making a mess
				//in the i2s. It can happen that there is an 'unk_xxxxxxxx' name
				//even if it's not shown in source listing.
				//Now sometimes the name will be detected by the i2s, but sometimes not ->
				//I don't know why, but I'll exclude this from the VLPs ->
				//they are not used anyway so they just use space.
				//
				//Search for 'unk_'.
				//(Exclude the zero length names also.)
					get_name( currentAddress, currentAddress, (char*)nameBuffer, sizeof( nameBuffer ) );
					if ( ( nameBuffer[0] != '_knu' ) && ( ( (char*)&nameBuffer )[0] != '\x0' ) )
					{
					//Runtime debugger.
						++nVLP;
						if ( nVLP > pSegmentCurrent->nVLP )
						{
							msg( "I2S: VLP detector failure. Please contact author.\n" );
							return STATUS_FAILED;
						}

					//Current address has a name -> set VLP.
						if ( S_FAILED( SYMD_SetVLP( currentAddress, flags ) ) )
						{
							jumpto( currentAddress, -1 );
							return STATUS_FAILED;
						}
					}
				}
			}

			//Dissasembly listing.
			if ( currentSourceFileEnabled == TRUE )
			{
			//Source listing is enabled -> define SEGDATA_SOURCELINES.
				if ( isCode( flags ) )
				{
					SEGDATA_SOURCELINES*	pSourceLines_current;

				//Initialize.
					pSourceLines_current = pSegmentCurrent->pSourceLines_current;

					IDAPlace.ea = currentAddress;
					lineArray.set_place( &IDAPlace );

					if ( lineArray.dlnnum == -1 )
					{
						msg( "I2S: default line number failure -> please contact author." );
					}

				//Set SEGDATA_SOURCELINES.
				//
				//Notice:
				//Be careful as every line can have multiple lines, but only 1 of those is a command line ->
				//the offset to that line is in dlnnum;
					pSourceLines_current->offInCode = currentAddress - minEA;
					pSourceLines_current->lineNumber = currentLineInSource + lineArray.dlnnum;

				//Update.
					currentLineInSource += lineArray.linecnt;
					++pSegmentCurrent->pSourceLines_current;
				}
				else if ( isData( flags ) )
				{
				//Trow out align directives -> they just use memory.
				//
				//Be careful IDA assignes align flag even to some real data.
				//That data must be included.
					if ( ( ( flags & DT_TYPE ) != FF_ALIGN ) || ( has_any_name( flags ) )	)
					{
					//It's not an align directive or it has a name ->
					//save source listing and get it's text size.
						IDAPlace.ea = currentAddress;
						lineArray.set_place( &IDAPlace );

					//Update.
						currentLineInSource += lineArray.linecnt;
					}
				}
			}

			//Is it a function?
			if ( isFunc( flags ) )
			{
				func_t	procedure;

			//Get function.
				CopyMemory( &procedure, get_func( currentAddress ), sizeof( func_t ) );

			//Set procedure variables.
			//
			//Notice:
			//Set only if function is visible (not hidden).
				if ( is_visible_func( &procedure ) )
				{
					if ( S_FAILED( SYMD_SetProcedureVariables( &procedure ) ) )
					{
						jumpto( currentAddress, -1 );
						return STATUS_FAILED;
					}
				}
				else
				{
				//Function is not visible, so we'll jump over it.
				//
				//Define SEGDATA_PROCEDURE.
					SEGDATA_PROCEDURE_V2*	pProcDefinitions = pSegmentCurrent->pProcDefinitions_current;

					pProcDefinitions->offStartProc = currentAddress - minEA;
					pProcDefinitions->offEndProc = procedure.endEA - minEA - 1;
					pProcDefinitions->offLocalVariables = 0;
					pProcDefinitions->nLocalVariables = 0;
					pProcDefinitions->offRegisterVariables = 0;
					pProcDefinitions->sizeRegisterVariables = 0;
					pProcDefinitions->M1 = 0xFFFFFFFF;
					pProcDefinitions->N1 = 0;

					++pSegmentCurrent->pProcDefinitions_current;

				//Jump over.
					currentAddress = procedure.endEA;
					continue;
				}
			}

		//Update.
			currentAddress += get_item_size( currentAddress );
		}

	//Update.
		pSegmentCurrent = pSegmentCurrent->pNext;
	}

//Define sections.
	if ( S_FAILED( SetSection_NMTP() ) ) return STATUS_FAILED;
	if ( S_FAILED( SetSection_STTB() ) ) return STATUS_FAILED;
	if ( S_FAILED( SetSection_TYTB() ) ) return STATUS_FAILED;
	if ( S_FAILED( SetSection_HSHT() ) ) return STATUS_FAILED;
	if ( S_FAILED( SetSection_SYMD() ) ) return STATUS_FAILED;

	//FPOD ( it's just a header so will use stack )
	FPOD_HEADER		fpod;

		//Set section in sections structure.
	sections.pFpod = (char*)&fpod;
		//Set section.
	fpod.FPOD = FPODs;
	fpod.sectionLength = sizeof( FPOD_HEADER );
	fpod.nTables = 0;
	fpod.offTables = 0;
	fpod.N1 = 0;
	fpod.N2 = 0;
	fpod.N3 = 0;
	fpod.N4 = 0;
	fpod.N5 = 0;
	fpod.N6 = 0;
	fpod.N7 = 0;
	fpod.N8 = 0;

	//SRCP ( it's just a header so will use stack )
	SRCP_HEADER		srcp;

		//Set section in sections structure.
	sections.pSrcp = (char*)&srcp;
		//Set section.
	srcp.SRCP = SRCPs;
	srcp.sectionLength = sizeof( SRCP_HEADER );
	srcp.offSourceFiles = 0;  //we'll define this in SetSection_NM32
	srcp.N1 = 0;
	srcp.N2 = 0;

	//GTYP ( it's just a header so will use stack )
	GTYP_HEADER		gtyp;

		//Set section in sections structure.
	sections.pGtyp = (char*)&gtyp;
	gtyp.GTYP = GTYPs;
	gtyp.sectionLength = sizeof( GTYP_HEADER );
	gtyp.offNMTP = 0; //we'll define this in SetSection_NM32
	gtyp.offSTTB = 0; //we'll define this in SetSection_NM32
	gtyp.offTYTB = 0; //we'll define this in SetSection_NM32
	gtyp.offHSHT = 0; //we'll define this in SetSection_NM32
	gtyp.N1 = 0;
	gtyp.N2 = 0;

//Construct the NMS database
	if ( S_FAILED( SetSection_NM32() ) ) return STATUS_FAILED;

	return (NM32_HEADER*)sections.pNms;
}


