//
//  PlugIn name: IDA to SoftIce converter (only for 32bit programs)
//
//
//  This file contains function that are needed for source files management.
//
//
#include "i2s.h"

//--------------------------------------------------------------------------
//Define source files boundaries, dummy name, ...., = initialize netnode.
//This is done only once (in the new database).
//
//Notice:
//This procedure defines just files that contain CPU commands (or CODE) in it.
//It doesn't care about data segments or anything like that.
//For those parts it's up to main loop to include source listings in the
//source files.
void ida2softice_c::SetSourceFilesBoundaries
(
	void
)
{
	ssize_t			ret;

	ea_t			codeStart;
	ea_t			currentAddress;
	ea_t			segmentStart;
	ea_t			segmentEnd;
	unsigned long	offStart;	//offset of the first code line in current segment
	unsigned long	previousSelector = 0;

	unsigned long	indexInSourceNetNodeSupVal = 0;	//also number of files defined

	unsigned long	currentListingSize;
	sourceInfo_s*	pPreviousInfo;
	sourceInfo_s	previousInfo;
	sourceInfo_s	sourceInfo;
	unsigned long	sizeOfSourceListing = 0;

	sourceFiles_s	nSources = {0, 0, 0};

//Check database for code segments.
	//Get number of segments that are used in IDA, but not the number of them all in file.
	int	nSegments = get_segm_qty();
	for ( int i = 0; i < nSegments; i++ )
	{
		segment_t*		pSegment = getnseg(i);

		BOOL			anyCommandLines = FALSE;//so that we will not create source files for data segments
		unsigned long	nCommandLinesInCurrentFile = 0;
		unsigned long	nFilesInSegment = 0;

		unsigned long	nCommandLines;
		func_t			procedure;

	//Check for proper segment.
	//
	//IDA can create a segment when the segment doesn't even exist.
	//i.e. It creates a segment out of the file header -> this should not happen.
	//We will simply exclude this kind of segment -> we wont process it.
		if ( pSegment->sel > sectionTablesNetNode.long_value() )
		{
		//A faulty segment -> skip it.
			continue;
		}

	//Initialize.
		codeStart = pSegment->startEA;
		currentAddress = pSegment->startEA;
		segmentEnd = pSegment->endEA;
		offStart = 0xFFFFFFFF;	//offset of the first code line in current segment

		//Is current segment a new segment in original file too?
		//(don't update the segmentStart if current segment is the same as previous one)
		if ( previousSelector < pSegment->sel )
		{
			segmentStart = pSegment->startEA;
		}
		previousSelector = pSegment->sel;

	//Set index start for current segment source files.
		nSources.indexInSupVal = indexInSourceNetNodeSupVal;

	//Do the honky tonk.
		while( currentAddress < segmentEnd )
		{
		//Check current address.
		//
		//If it's a code address than do the stuff. But be careful if the currentAddress is start
		//of procedure -> the whole procedure must come into 1 file.
		//
			flags_t		flags = getFlags( currentAddress );
			if ( isCode( flags ) )
			{
			//Signal that we have at least 1 line of code in current segment.
				anyCommandLines = TRUE;

			//Set the first code line offset in current segment.
				if ( offStart == 0xFFFFFFFF )
				{
				//The variables was not set yet -> set it.
					offStart = currentAddress - segmentStart;
				}

			//We got a code line -> check if this line is start of the procedure.
			//If not we should just check if there is enough space left in current source file
			//and save the parameters needed for creating of NMS file.
				if ( isFunc( flags ) )
				{
				//It's a start of the procedure.
				//
				//Get the procedure data and copy the values
				//(we need to do this as sometimes IDA overwrites the memory used for holding data, when
				//you call some specific IDA procedures).
					func_t*	pProcedureTemp = get_func( currentAddress );
					CopyMemory( &procedure, pProcedureTemp, sizeof(func_t) );

				//Get the number of the operational code lines.
				//
				//First check if procedure is hidden -> if it is unhide it do the stuff
				//and hide it back. In this way we'll get the number of op's in the function.
					if ( ( procedure.flags & FUNC_HIDDEN ) != 0 )
					{
					//Unhide function -> clear hidden flag.
						set_visible_func( &procedure, true );

					//Get op.codes.
						nCommandLines = SYMD_GetProcedureSourceLines( &procedure, &currentListingSize );

					//Hide function.
						set_visible_func( &procedure, true );
					}
					else
					{
						nCommandLines = SYMD_GetProcedureSourceLines( &procedure, &currentListingSize );
					}
					sizeOfSourceListing += currentListingSize;
				}
				else
				{
				//It's not a function -> just one command line.
					nCommandLines = 1;

					//Size of source listing.
					sizeOfSourceListing += SYMD_GetSizeOfSourceListing( currentAddress );
				}

			//Do we have to create another source file? -> Are we over the boundaries?
				if ( ( nCommandLinesInCurrentFile + nCommandLines ) >= MAX_SOURCE_LINES_PER_FILE )
				{
				//We are over file boundary => create new source file.
				//
				//Next line will fail if previous source file wasn't defined yet.
					ret = sourceNetNode.supval( indexInSourceNetNodeSupVal-1, &previousInfo, sizeof( previousInfo ), stag );
					if ( ret == -1 )
					{
					//Element doesn't exist.
						pPreviousInfo = NULL;
					}
					else
					{
					//Emulate old behaviour.
						pPreviousInfo = &previousInfo;
					}

				//Define sourceInfo_s common info.
					sourceInfo.enable_symbols = TRUE;	//all sources are enabled by default
					sourceInfo.enable_source = TRUE;	//all sources are enabled by default
					qsnprintf(
					  sourceInfo.nameSource,
					  sizeof( sourceInfo.nameSource ),
					  "seg%dfile%d.asm",
					  i+1,
					  nFilesInSegment
					);
					if ( pPreviousInfo == NULL )
					{
					//No source file was set yet -> create the first one.
						//Save current source file code start.
						sourceInfo.startEA = codeStart;
						//Set codeStart for the next source file.
						codeStart = currentAddress;
					}
					else
					{
					//We already have a source file defined -> get stopEA.
						sourceInfo.startEA = pPreviousInfo->endEA + 1;
					}
					sourceInfo.endEA = currentAddress - 1;
					sourceInfo.segmentNumber = i;	//IDA's segment number

				//Set SEGDATA_SOURCESECTION.
				//
				//Notice:
				//Some variables will be updated at runtime.
					if ( nFilesInSegment == 0 )
					{
					//No source file was set yet -> create the first one.
						sourceInfo.sourceSection.offStart = offStart;
					}
					else
					{
					//We already have a source file defined.
						sourceInfo.sourceSection.offStart = pPreviousInfo->sourceSection.offEnd + 1;
					}
					sourceInfo.sourceSection.offEnd = currentAddress - segmentStart - 1;
					sourceInfo.sourceSection.offSourceLines = 0;
					sourceInfo.sourceSection.nLineTables = 0;
					sourceInfo.sourceSection.offInSFDef = 0;

				//Set SEGDATA_SOURCEFILE.
				//
				//Notice:
				//Some variables will be updated at runtime.
					sourceInfo.sourceFile.iName = 0;
					sourceInfo.sourceFile.offSource = 0;
					sourceInfo.sourceFile.sourceLength = 0;
					sourceInfo.sourceFile.N1 = 0;
					sourceInfo.sourceFile.N2 = 0;

				//Save sourceInfo structure.
					sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof(sourceInfo_s), stag );

				//Update.
					++indexInSourceNetNodeSupVal;
					++nFilesInSegment;
					//As we came over the boundary, we already have some source lines.
					nCommandLinesInCurrentFile = nCommandLines;
				}
				else
				{
				//We are still in the boundaries.
					nCommandLinesInCurrentFile += nCommandLines;
				}

			//Get to the next address.
			//
			//Check if it is function or not
			//(we don't want to process the function commands again).
				if ( isFunc( flags ) )
				{
				//It's a function -> go to the end of the function.
					currentAddress = procedure.endEA;
					continue;
				}
//				else
//				{
//				//It's just a command line.
//					currentAddress += get_item_size( currentAddress );
//					continue;
//				}
			}
			else if ( isData( flags ) )
			{
			//Data -> get just source listing size (exclude undefined bytes).
			//
			//Notice:
			//Trow out align directives -> they just use memory.
			//
			//Be careful IDA assignes align flag even to some real data.
			//That data must be included.
				if ( ( ( flags & DT_TYPE ) != FF_ALIGN ) || ( has_any_name( flags ) ) )
				{
					sizeOfSourceListing += SYMD_GetSizeOfSourceListing( currentAddress );
				}
			}

		//Increase currentAddress.
			currentAddress += get_item_size( currentAddress );
		}

	//We are at the end of current segment.
	//
	//Check if this segment was a CODE segment. If it was save the last source file.
		if ( anyCommandLines == TRUE )
		{
		//Do we have any source file defined?
		//
		//The last sourceFile is not saved yet. If something is still in memory
		//we need to save it.
		//
		//Next line will fail if previous source file wasn't defined yet.
			ret = sourceNetNode.supval( indexInSourceNetNodeSupVal-1, &previousInfo, sizeof( previousInfo ), stag );
			if ( ret == -1 )
			{
			//Element doesn't exist.
				pPreviousInfo = NULL;
			}
			else
			{
			//Emulate old behaviour.
				pPreviousInfo = &previousInfo;
			}

		//Common info.
			sourceInfo.enable_symbols = TRUE;	//all sources are enabled by default
			sourceInfo.enable_source = TRUE;	//all sources are enabled by default
			qsnprintf( sourceInfo.nameSource, sizeof( sourceInfo.nameSource ), "seg%dfile%d.asm", i+1, nFilesInSegment );
			if ( pPreviousInfo == NULL )
			{
			//No source file was set yet -> create one.
				sourceInfo.startEA = codeStart;
			}
			else
			{
			//We already have a source file defined -> get stopEA.
				sourceInfo.startEA = pPreviousInfo->endEA + 1;
			}
			sourceInfo.endEA = currentAddress - 1;
			sourceInfo.segmentNumber = i;

		//Set SEGDATA_SOURCESECTION.
		//
		//Notice:
		//Some variables will be updated at runtime.
			if ( nFilesInSegment == 0 )
			{
			//No source file was set yet -> create one.
				sourceInfo.sourceSection.offStart = offStart;
			}
			else
			{
			//We already have a source file defined -> get offEndSourceSection.
				sourceInfo.sourceSection.offStart = pPreviousInfo->sourceSection.offEnd + 1;
			}
			sourceInfo.sourceSection.offEnd = currentAddress - segmentStart - 1;
			sourceInfo.sourceSection.offSourceLines = 0;
			sourceInfo.sourceSection.nLineTables = 0;
			sourceInfo.sourceSection.offInSFDef = 0;

		//Set SEGDATA_SOURCEFILE.
		//
		//Notice:
		//Some variables will be updated at runtime.
			sourceInfo.sourceFile.iName = 0;
			sourceInfo.sourceFile.offSource = 0;
			sourceInfo.sourceFile.sourceLength = 0;
			sourceInfo.sourceFile.N1 = 0;
			sourceInfo.sourceFile.N2 = 0;

		//Save info.
			sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof(sourceInfo_s), stag );

		//Update variables
			++indexInSourceNetNodeSupVal;
			++nFilesInSegment;
		}

	//Save how many source files we have in segment.
		nSources.nEnabled = nFilesInSegment;
		nSources.nAll = nFilesInSegment;
		sourceNetNode.supset( i, &nSources, sizeof( sourceFiles_s ), atag );
	}

//Write number of source files defined to netnode.
	sourceNetNode.set_long( indexInSourceNetNodeSupVal );

//Write version number main netnode.
	i2sNetNode.altset( NVAL_I2S__SOURCE_INFO_VER, SOURCE_INFO_VERSION, atag );

//Write current size of source listing to main netnode.
	i2sNetNode.altset( NVAL_I2S__MAX_SOURCE_SIZE, sizeOfSourceListing, atag );

	return;
}

//Check if all source files really have at least 1 line of code in it.
STATUS ida2softice_c::CheckSourceFilesDatabase( void )
{
	BOOL			codePresent;
	ea_t			currentAddress;
	unsigned long	index = 0;
	unsigned long	nSourceFiles;
	sourceInfo_s	sourceInfo;

//Initialize.
	nSourceFiles = sourceNetNode.long_value();
	for ( index = 0; index < nSourceFiles; index++ )
	{
	//Get current source file info.
		if ( -1 == sourceNetNode.supval( index, &sourceInfo, sizeof( sourceInfo ), stag ) )
		{
			msg( "I2S: ERROR -> failed to retrive source file information.\n" );
			return STATUS_FAILED;
		}

	//Loop through the source files address and check if there is at least 1 line
	//of code in it.
		codePresent = FALSE;

		currentAddress = sourceInfo.startEA;
		while ( currentAddress < sourceInfo.endEA )
		{
		//Get flags and check if if current line is a code.
		//If it's code stop processing current file and start with the next one.
			if ( isCode( getFlags( currentAddress ) ) )
			{
			//We got at least 1 line of code -> go to next file.
				codePresent = TRUE;
				break;
			}

		//Get next line.
			currentAddress += get_item_size( currentAddress );
		}

	//Do we have code?
		if ( codePresent == FALSE )
		{
		//No code in current file -> current database is not proper anymore.
			return STATUS_FAILED;
		}
	}
	return STATUS_SUCCESS;
}
//Gets iName representatives of sourceFile names.
STATUS ida2softice_c::DefineSourceFilesNames( void )
{
	unsigned long	indexInSourceNetNodeSupVal = 0;
	unsigned long	iName;
	unsigned long	nSourceFiles;
	char			pathAndName[64];
	char*			pEndOfPath = (char*)&pathAndName;
	sourceInfo_s	sourceInfo;
	size_t			sizeLeft = 0;

//Initialize.
	nSourceFiles = sourceNetNode.long_value();
	qstrncpy( pathAndName, sourcePath, sizeof( pathAndName ) );
	while( pEndOfPath[0] != '\x0' )
	{
		++pEndOfPath;
	}
	sizeLeft = sizeof( pathAndName ) - ( pEndOfPath - (char*)&pathAndName );

//Loop through all the source files.
	while( indexInSourceNetNodeSupVal < nSourceFiles )
	{
	//Get source file.
		if ( -1 == sourceNetNode.supval( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo ), stag ) )
		{
			msg( "I2S: Failed to retrive source file information.\n" );
			return STATUS_FAILED;
		}

	//Check if current source file was enabled.
		if ( sourceInfo.enable_source == TRUE )
		{
		//Set the relative path.
			qstrncpy( pEndOfPath, sourceInfo.nameSource, sizeLeft );

		//Set iName.
			iName = STTB_SetStringTable( pathAndName );
			if ( iName == STATUS_ERROR )
			{
				return STATUS_FAILED;
			}
			sourceInfo.sourceFile.iName = iName;

		//Save the updated file structure back to netnode.
			sourceNetNode.supset( indexInSourceNetNodeSupVal, &sourceInfo, sizeof( sourceInfo_s ), stag );
		}

	//Update.
		++indexInSourceNetNodeSupVal;
	}

	return STATUS_SUCCESS;
}


//--------------------------------------------------------------------------
//Show defined source files info. (functions group)
//This window uses setup window hot key (Ctrl-F12).
//
//double click on
//line: 1   -> show I2S setup window
//line: 2   -> separator(-----------------------)
//line: 3.. -> shows files info
//             add/delete is possible (delete only if command lines number in previous file is less than MAX_SOURCE_LINES_PER_FILE)
//             source file renameing
//
//
//Local Variables

char*	line0 = "Source files";
char*	line1 = "I2S Setup Window";
char*	line2 = "-------------------------------------------------------------------------------------------------------------------------------------";
char*	line3  = "Enable All (after -> right mouse button -> refresh)";
char*	line4 = "Disable All";
char*	line5 = "Enable All Symbols";
char*	line6 = "Disable All Symbols";
char*	line7 = "Enable All Source Listings on Enabled Symbols";
char*	line8 = "Disable All Source Listings on Enabled Symbols";
char*	line9 = "-------------------------------------------------------------------------------------------------------------------------------------";

char	tempBufferSource[128];
//number of items
ulong idaapi sizerSources(void *obj)
{
	return ( i2s.sourceNetNode.long_value() + 9 );
}

//Description of n-th item (1..n) ;0-th header line
char* idaapi getlSources
(
	void			*obj,
	unsigned long	n,
	char			*buf
)	
{
	char			buffer_temp[64];
	sourceInfo_s	sourceInfo;

//Return required line string.
	switch ( n )
	{
	case 0 : return line0;	//descibe header
	case 1 : return line1;
	case 2 : return line2;
	case 3 : return line3;
	case 4 : return line4;
	case 5 : return line5;
	case 6 : return line6;
	case 7 : return line7;
	case 8 : return line8;
	case 9 : return line9;
	default :
		if ( -1 == i2s.sourceNetNode.supval( n-10, &sourceInfo, sizeof( sourceInfo ), stag ) )
		{
			qstrncpy( tempBufferSource, "Internal error -> please contact author.", sizeof( tempBufferSource ) );
			return (char*)&tempBufferSource;
		}
		if ( sourceInfo.enable_symbols == TRUE )
		{
		//Symbols enabled.
			if ( sourceInfo.enable_source == TRUE )
			{
			//Symbols and source are enabled.
				qstrncpy( buffer_temp, "symbols enabled, source listing enabled", sizeof( buffer_temp ) );
			}
			else
			{
			//Symbols enabled, source disabled.
				qstrncpy( buffer_temp, "symbols enabled, source listing disabled", sizeof( buffer_temp ) );
			}
		}
		else
		{
		//Symbols disabled.
			if ( sourceInfo.enable_source == TRUE )
			{
			//Symbols disabled, source enabled.
				qstrncpy( buffer_temp, "symbols disabled, source listing enabled", sizeof( buffer_temp ) );
			}
			else
			{
			//Symbols and source disabled.
				qstrncpy( buffer_temp, "symbols disabled, source listing disabled", sizeof( buffer_temp ) );
			}
		}
	//Construct return string.
		qsnprintf(
		  tempBufferSource,
		  sizeof( tempBufferSource ),
		  "%s (%X-%X => %s)",
		  sourceInfo.nameSource,
		  sourceInfo.startEA,
		  sourceInfo.endEA,
		  (char*)&buffer_temp
		);

		return (char*)&tempBufferSource;
	}
}

BOOL isSetupWindowOpened = FALSE;
//Double click
void idaapi enterSources
(
	void	*obj,
	unsigned long n		//callback for non-modal "Enter"
)
{
	unsigned long	i;
	unsigned long	nAllFiles;
	unsigned long*	pNEnabled_source;
	unsigned long	nSeg;
	sourceFiles_s	nSources;
	sourceInfo_s	sourceInfo;

	switch (n)
	{
//------------------------------------------------------
	case 1 :
	//Open Setup Window
		if ( isSetupWindowOpened == FALSE )
		{
			isSetupWindowOpened = TRUE;
			DialogBoxParam(
			  GetModuleHandle( "i2s.plw" ),
			  (char*)DIALOG_SETUP,
			  GetForegroundWindow(),
			  SetupDialogBox,
			  1	//1 - open dialog (0-enable ALWAYSDEFAULT)
			);
			isSetupWindowOpened = FALSE;
		}
	break;
//------------------------------------------------------
	case 2 : return;
//------------------------------------------------------
	case 3 :
	//Enable all
		nAllFiles = i2s.sourceNetNode.long_value();
		if ( nAllFiles != 0 )
		{
			for ( i = 0; i < nAllFiles; i++ )
			{
			//Enable symbols and source in all sourceInfo structs.
				if ( -1 == i2s.sourceNetNode.supval( i, &sourceInfo, sizeof( sourceInfo ), stag ) )
				{
					return;
				}
				sourceInfo.enable_symbols = TRUE;
				sourceInfo.enable_source = TRUE;
				i2s.sourceNetNode.supset( i, &sourceInfo, sizeof( sourceInfo ), stag );

				nSeg = sourceInfo.segmentNumber + 1;
			}
			for ( i = 0; i < nSeg; i++ )
			{
			//Enabled source in all segments.
				if ( -1 == i2s.sourceNetNode.supval( i, &nSources, sizeof( nSources ), atag ) )
				{
					return;
				}
				nSources.nEnabled = nSources.nAll;
				i2s.sourceNetNode.supset( i, &nSources, sizeof( sourceFiles_s ), atag );					
			}
		}
	break;
//------------------------------------------------------
	case 4 :
	case 6 :
	case 8 :
	//Disable all
	//Disable all Symbols
	//Disable All Source Listings on Enabled Symbols
		nAllFiles = i2s.sourceNetNode.long_value();
		if ( nAllFiles != 0 )
		{
			for ( i = 0; i < nAllFiles; i++ )
			{
			//Disable symbols and source in all sourceInfo structs.
				if ( -1 == i2s.sourceNetNode.supval( i, &sourceInfo, sizeof( sourceInfo ), stag ) )
				{
					return;
				}
				sourceInfo.enable_symbols = FALSE;
				sourceInfo.enable_source = FALSE;
				i2s.sourceNetNode.supset( i, &sourceInfo, sizeof( sourceInfo ), stag );

				nSeg = sourceInfo.segmentNumber + 1;
			}
			for ( i = 0; i < nSeg; i++ )
			{
			//Disable source in all segments.
				if ( -1 == i2s.sourceNetNode.supval( i, &nSources, sizeof( nSources ), atag ) )
				{
					return;
				}
				nSources.nEnabled = 0;
				i2s.sourceNetNode.supset( i, &nSources, sizeof( nSources ), atag );					
			}
		}
	break;
//------------------------------------------------------
	case 5 :
	//Enable All Symbols
		nAllFiles = i2s.sourceNetNode.long_value();
		if ( nAllFiles != 0 )
		{
			for ( i = 0; i < nAllFiles; i++ )
			{
			//Enable symbols in all sourceInfo structs.
				if ( -1 == i2s.sourceNetNode.supval( i, &sourceInfo, sizeof( sourceInfo ), stag ) )
				{
					return;
				}
				sourceInfo.enable_symbols = TRUE;
				i2s.sourceNetNode.supset( i, &sourceInfo, sizeof( sourceInfo ), stag );
			}
		}
	break;
//------------------------------------------------------
	case 7 :
	//Enable All Source Listings on Enabled Symbols
		nAllFiles = i2s.sourceNetNode.long_value();
		if ( nAllFiles!=NULL )
		{
			nSeg = get_segm_qty();
			pNEnabled_source = (unsigned long*) LocalAlloc( LPTR, nSeg * sizeof( unsigned long ) );
			if ( pNEnabled_source != NULL )
			{
				for ( i = 0; i < nAllFiles; i++ )
				{
				//Enable source in all sourceInfo structs.
					if ( -1 == i2s.sourceNetNode.supval( i, &sourceInfo, sizeof( sourceInfo ), stag ) )
					{
						return;
					}
					if ( sourceInfo.enable_symbols == TRUE )
					{
						if ( sourceInfo.enable_source == FALSE )
						{
						//Enable source.
							sourceInfo.enable_source = TRUE;

						//Increase number of additionaly enabled source listings.
							pNEnabled_source[sourceInfo.segmentNumber] += 1;

						//Save
							i2s.sourceNetNode.supset( i, &sourceInfo, sizeof( sourceInfo ), stag );
						}
					}
				}
				for ( i = 0; i < nSeg; i++ )
				{
				//Enabled source in all segments.
					if ( pNEnabled_source[nSeg] != 0 )
					{
						if ( -1 == i2s.sourceNetNode.supval( i, &nSources, sizeof( nSources ), atag ) )
						{
							return;
						}
						nSources.nEnabled += pNEnabled_source[nSeg];
						i2s.sourceNetNode.supset( i, &nSources, sizeof( nSources ), atag );
					}
				}

				LocalFree( (HLOCAL)pNEnabled_source );
			}
			else
			{
				msg( "I2S: Failed to allocate memory(enterSources).\n" );
			}
		}
	break;
//------------------------------------------------------
	case 9 : return;
//------------------------------------------------------
	default :
	//Open sources definition window.
		DialogBoxParam(
		  GetModuleHandle("i2s.plw"),
		  (char*)DIALOG_SOURCE_INFO,
		  GetForegroundWindow(),
		  SourceFileInfoDialogBox,
		  n - 10
		);
	}
	return;
}

/*
// callback for "Delete"
void idaapi deleteSources(void *obj,ulong n)
{
	MessageBox(GetForegroundWindow(), "Delete", "Delete", MB_OK);
}

// callback for "New"
void idaapi insertSources(void *obj)
{
	MessageBox(GetForegroundWindow(), "Insert", "Insert", MB_OK);
}
*/

//callback for "Update"
//update the whole list
//returns the new location of item 'n'
/*
ulong idaapi updateSources(void *obj,ulong n)
{
}
*/

/*
void idaapi destroySources(void *obj)
{
}
*/

void ida2softice_c::ShowSourceFilesInfo()
{
	choose( FALSE,
			-1,
			-1,
			-1,
			-1,
			NULL,
			150,
			sizerSources,
			getlSources,

			"Source files definition",
			0,
			1,
			NULL,	//deleteSources,
			NULL,	//insertSources,
			NULL,	//updateSources

			NULL,
			enterSources,
			NULL, //destroySources,
#ifndef IDA56UP
			NULL);
#else
			NULL,
			NULL);
#endif

	return;
}

STATUS ida2softice_c::CreateSourceRelativePath( void )
{
	char	fileName[MAX_PATH];
	char	filePath[MAX_PATH];

//Get input path and get the file name out of it.
	if ( -1 == get_input_file_path( filePath, sizeof( filePath ) ) )
	{
		msg( "I2S: Error getting input file path./n" );
		return STATUS_FAILED;
	}
	_splitpath( filePath, NULL, NULL, fileName, NULL );

	//How long is the fileName -> our whole buffer is just 32 chars in size.
	if ( strlen( fileName ) > 23 )
	{
	//The file name is to long -> shorten it.
	//
	//Create source dir name -> append " Source" to the end of the path.
		qstrncpy( (char*)&fileName[23], "_Source\\", sizeof( fileName )-24 );
	}
	else
	{
	//The file isn't too long.
		qstrncat( fileName, "_Source\\", sizeof( fileName ) );
	}

//Scan whole string for spaces and remove them.
	char*	pFullPath = (char*)fileName;
	while ( pFullPath[0] != '\x0' )
	{
		if ( pFullPath[0] == ' ' )
		{
			pFullPath[0] = '_';
		}

	//Update.
		++pFullPath;
	}

//Copy the created string.
	qstrncpy( sourcePath, fileName, sizeof( sourcePath ) );
	return STATUS_SUCCESS;
}

//Save NMS file and source files if they exist.
void ida2softice_c::SaveFiles( void )
{
	char			fileName[MAX_PATH];
	char			fullPath[MAX_PATH];
	char			fullPath_NMS[MAX_PATH];
	OPENFILENAME	openFile;
	
//Initialize.
	if ( 0 < i2sNetNode.supval( NSUP_I2S__NMSPATH, &fullPath_NMS, sizeof( fullPath_NMS ), stag ) )
	{
		if ( strlen( fullPath_NMS ) >= MAX_PATH )
		{
			msg( "I2S: NMS path to long.\n" );
			return;
		}
	}

	//Set OPENFILENAME.
	ZeroMemory( &openFile, sizeof( OPENFILENAME ) );

	openFile.lStructSize = sizeof( OPENFILENAME );
	openFile.hwndOwner = hWnd_pleaseWait;
	openFile.lpstrFilter = ( "Numega symbol files (*.nms)" "\x0" "*.nms" "\x0" "\x0" );
	openFile.nFilterIndex = 1;
	if ( -1 == get_input_file_path( fullPath, sizeof( fullPath ) ) )
	{
		msg( "I2S: Error getting input file path./n" );
		return;
	}
	_splitpath( fullPath, NULL, NULL, fileName, NULL );
	openFile.lpstrFile = fileName;
	openFile.nMaxFile = MAX_PATH;
	openFile.lpstrInitialDir = fullPath_NMS;
	openFile.Flags = OFN_LONGNAMES | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;
	openFile.lpstrDefExt = "nms";

//Open save dialogBox.
	if ( FALSE != GetSaveFileName( &openFile ) )
	{
		HANDLE	hFile;

	//Save nms file
		hFile = CreateFile(
		  fileName,
		  GENERIC_WRITE,
		  FILE_SHARE_READ,
		  NULL,
		  CREATE_ALWAYS,
		  FILE_ATTRIBUTE_ARCHIVE,
		  NULL
		);
		if ( hFile != INVALID_HANDLE_VALUE )
		{
			unsigned long	bytesReadWrite;

		//I2S got the handle to file -> save the NMS file.
			BOOL	retValue;
			retValue = WriteFile(
			  hFile,
			  sections.pNms,
			  sections.fullFileSize,
			  &bytesReadWrite,
			  NULL
			);
			CloseHandle(hFile);
			if ( retValue != FALSE )
			{
			//Find the end of the filePath, then change the path so
			//that the file name will be excluded.
				char* pEndOfPath = (char*)&fileName;
				while ( pEndOfPath[0] != '\x0' )
				{
				//Move to next char.
					++pEndOfPath;
				}
				//We are at the end of the path.
				//Now find the last occurence of '\\' and zero it.
				while ( pEndOfPath[0] != '\\' )
				{
					--pEndOfPath;
				}
				pEndOfPath[0] = '\x0';

			//As I2S will remember the path where to save the NMS file
			//we should check if current path is different than the
			//one already saved. If this is the first time the plugIn
			//was started just save current directory.
				if ( -1 == i2sNetNode.supval( NSUP_I2S__NMSPATH, &fullPath_NMS, sizeof( fullPath_NMS ), stag ) )
				{
				//This is the first time the NMS file was saved ->
				//just save current path as the default one for this NMS file.
					i2sNetNode.supset( NSUP_I2S__NMSPATH, &fileName, 0, stag );
				}
				else
				{
				//Check if current path and the one in the database are different.
					if ( 0 != strcmp( fullPath_NMS, fileName ) )
					{
					//Yes they are different -> ask if I2S should update
					//the path.
						int	askRetValue = askyn_c(
						  1,
						  "Should  IDA2SoftIce  set current path as default path for\n\nsaving the nms file (only for this database)?"
						);
						if ( askRetValue == 1 )
						{
							i2sNetNode.supset( NSUP_I2S__NMSPATH, fileName, 0, stag );
						}
					}
				}

			//Next we need to save all the source files too (if they exist).
			//
			//Create source directory.
			//The source files will be saved to one dir below the NMS file was saved.
			//The name of the dir will be: 'fileName Source'.
			//
			//Should I2S save the source files?
				if ( i2s.includeSource != 0 )
				{
				//Yes save the source files.
					if ( ( creationFlags & CN_T_INCLUDESOURCE ) == 0 )
					{
					//Set '\' back into the path and append the source dir path to current path.
						pEndOfPath[0] = '\\';
						qstrncpy( (char*)&pEndOfPath[1], sourcePath, MAX_PATH-1 );

					//Move the pEndOfPath to the end of path.
						while ( pEndOfPath[0] != '\x0' )
						{
							//Move to next char.
							++pEndOfPath;
						}

					//One back and remove '\' from path.
						--pEndOfPath;
						pEndOfPath[0] = '\x0';

					//Create directory -> check if source directory exist if not create it.
						HANDLE				hDir;
						WIN32_FIND_DATA		findStruct;

						hDir = FindFirstFile( fileName, &findStruct );
						if ( hDir == INVALID_HANDLE_VALUE )
						{
							if (
							  ( ERROR_PATH_NOT_FOUND == GetLastError() ) 
							  ||
							  ( ERROR_FILE_NOT_FOUND == GetLastError() )
							)
							{
							//Failed to get the handle to the dir, so the dir
							//doesn't exist -> create it.
								if ( FALSE == CreateDirectory( fileName, NULL ) )
								{
									warning( "Internal error. Please contact author about this." );
									return;
								}
							}
							else
							{
								warning( "Internal error. Please contact author about this." );
								return;
							}
						}
						else
						{
						//Directory already exist.
							FindClose( hDir );
						}

					//Set '\' back to the path.
						pEndOfPath[0] = '\\';

					//Save files.
						i2s.SaveSourceFiles( fileName );
					}
				}
			}
			else
			{
				msg( "I2S: Failed to write the NMS file!\n" );
			}
		}
		else
		{
		//I2S failed to open/create the nms file.
			msg( "Couldn't create %s\n", fileName );
		}
	}

	return;
}

//--------------------------------------------------------------------------
//Save source files to directory.
// path - 'x:\....\' <- it must have '\' at the end
//
//return - STATUS_SUCCESS/STATUS_FAILED
unsigned long ida2softice_c::SaveSourceFiles( char* pPath )
{
	unsigned long	index;
	size_t			pathLength = strlen( pPath );
	unsigned long	nSourceFiles = sourceNetNode.long_value();
	char*			pEndOfPath = pPath + pathLength;
	sourceInfo_s	sourceInfo;

//Are there any source files to save?
	if ( nSourceFiles == BADNODE )
	{
		return STATUS_FAILED;
	}

//Save files.
	for ( index = 0; index < nSourceFiles; index++ )
	{
	//Get the sourceInfo structure.
		if ( -1 == sourceNetNode.supval( index, &sourceInfo, sizeof( sourceInfo ), stag ) )
		{
			msg( "I2S: Internal error -> please contact author about this.\n" );
			return STATUS_FAILED;
		}

	//If source file is enabled then save it.
		if ( sourceInfo.enable_source == TRUE )
		{
			HANDLE			hFile;
			unsigned long	bytesReadWrite;

		//Create a full path -> append the source file name at the end of the path.
			qstrncpy( pEndOfPath, sourceInfo.nameSource, MAX_PATH - pathLength );

		//Get handle to file and save it.
			hFile = CreateFile(
			  pPath,
			  GENERIC_WRITE,
			  FILE_SHARE_READ,
			  NULL,
			  CREATE_ALWAYS,
			  FILE_ATTRIBUTE_ARCHIVE,
			  NULL
			);
			if ( hFile != INVALID_HANDLE_VALUE )
			{
			//I2S got the file handle -> save the file.
				BOOL	retValue = WriteFile(
				  hFile,
				  source.pSource + sourceInfo.sourceFile.offSource,
				  sourceInfo.sourceFile.sourceLength,
				  &bytesReadWrite,
				  NULL
				);
				if ( retValue == FALSE )
				{
					msg( "I2S: Failed to save source file: %s\n", pPath );
				}
				CloseHandle( hFile );
			}
		}
	}

	return STATUS_SUCCESS;
}
