#include "stdafx.h"
#include <stdlib.h>
#include "distorm.h"

// 10000 hooks should be enough
#define MAX_HOOKS 10000

typedef struct _HOOK_INFO
{
	ULONG_PTR Function;	// Address of the original function

	ULONG_PTR Hook;		// Address of the function to call 
						// instead of the original

	ULONG_PTR Bridge;   // Address of the instruction bridge
						// necessary because of the hook jmp
						// which overwrites instructions

} HOOK_INFO, *PHOOK_INFO;

HOOK_INFO HookInfo[MAX_HOOKS];

UINT NumberOfHooks = 0;

BYTE *pBridgeBuffer = NULL; // Here are going to be stored all the bridges

UINT CurrentBridgeBufferSize = 0; // This number is incremented as
								  // the bridge buffer is growing

#ifdef _M_IX86

#define JUMP_SIZE		5

#else ifdef _M_AMD64

#define JUMP_SIZE		13		// Worst case scenario
								// only the case when there's more than
								// 2GB from the api to the bridge

#endif

#define SMALL_JUMP		5


#define KB_SIZE		1024
#define MB_SIZE		(1024 * KB_SIZE)
#define GB_SIZE		(1024 * MB_SIZE)

BOOL APIENTRY DllMain(HMODULE hModule, DWORD dwReason, LPVOID lpReserved)
{
	switch (dwReason)
	{
	case DLL_PROCESS_ATTACH:
		{
			pBridgeBuffer = (BYTE *) malloc(MAX_HOOKS * (JUMP_SIZE * 3));

			if (pBridgeBuffer)
			{
				DWORD dwOldProtect = 0;

				VirtualProtect(pBridgeBuffer, MAX_HOOKS * (JUMP_SIZE * 3), 
					PAGE_EXECUTE_READWRITE, &dwOldProtect);
			}
		}

	case DLL_THREAD_ATTACH:
	case DLL_THREAD_DETACH:
	case DLL_PROCESS_DETACH:
		break;
	}

	return TRUE;
}

HOOK_INFO *GetHookInfoFromFunction(ULONG_PTR OriginalFunction)
{
	if (NumberOfHooks == 0)
		return NULL;

	for (UINT x = 0; x < NumberOfHooks; x++)
	{
		if (HookInfo[x].Function == OriginalFunction)
			return &HookInfo[x];
	}

	return NULL;
}

#ifdef _M_AMD64

//
// This function evaluates if a 64bit jump is necessary
//

BOOL Is64bitJumpNeeded(ULONG_PTR PosA, ULONG_PTR PosB)
{
	return TRUE; // Always return TRUE

	ULONG_PTR res = max(PosA, PosB) - min(PosA, PosB);

	return res < (GB_SIZE * 2) ?  FALSE : TRUE;
}

#endif

//
// This function writes unconditional jumps
// both for x86 and x64
//

VOID WriteJump(VOID *pAddress, ULONG_PTR JumpTo)
{
	DWORD dwOldProtect = 0;

	VirtualProtect(pAddress, JUMP_SIZE, PAGE_EXECUTE_READWRITE, &dwOldProtect);

	BYTE *pCur = (BYTE *) pAddress;

#ifdef _M_IX86

	*pCur = 0xE9; // jmp

	DWORD RelAddr = (DWORD) (JumpTo - (ULONG_PTR) pAddress) - JUMP_SIZE;

	memcpy(++pCur, &RelAddr, sizeof (DWORD));

#else ifdef _M_AMD64

	if (Is64bitJumpNeeded((ULONG_PTR) pAddress, JumpTo) == FALSE)
	{
		// TODO: Write small jump if a < 2GB jmp is enough
	}
	else
	{
		*pCur = 0x49;		// mov r15, ...
		*(++pCur) = 0xBF;
		memcpy(++pCur, &JumpTo, sizeof (ULONG_PTR));
		pCur += sizeof (ULONG_PTR);
		*pCur = 0x41;		// jmp r15
		*(++pCur) = 0xFF;
		*(++pCur) = 0xE7;
	}

#endif

	DWORD dwBuf = 0;	// nessary othewrise the function fails

	VirtualProtect(pAddress, JUMP_SIZE, dwOldProtect, &dwBuf);
}


//
// This function creates a bridge of the original function
//

VOID *CreateBridge(ULONG_PTR Function, const UINT JumpSize = JUMP_SIZE)
{
	if (pBridgeBuffer == NULL) return NULL;

#define MAX_INSTRUCTIONS 100

	_DecodeResult res;
	_DecodedInst decodedInstructions[MAX_INSTRUCTIONS];
	unsigned int decodedInstructionsCount = 0;

#ifdef _M_IX86

	_DecodeType dt = Decode32Bits;

#else ifdef _M_AMD64

	_DecodeType dt = Decode64Bits;

#endif

	_OffsetType offset = 0;

	res = distorm_decode(offset,	// offset for buffer
		(const BYTE *) Function,	// buffer to disassemble
		50,							// function size (code size to disasm) 
									// 50 instr should be _quite_ enough
		dt,							// x86 or x64?
		decodedInstructions,		// decoded instr
		MAX_INSTRUCTIONS,			// array size
		&decodedInstructionsCount	// how many instr were disassembled?
		);

	if (res == DECRES_INPUTERR)
		return NULL;

	DWORD InstrSize = 0;

	VOID *pBridge = (VOID *) &pBridgeBuffer[CurrentBridgeBufferSize];

	for (UINT x = 0; x < decodedInstructionsCount; x++)
	{
		if (InstrSize >= JumpSize)
			break;

		BYTE *pCurInstr = (BYTE *) (InstrSize + (ULONG_PTR) Function);

		//
		// This is an sample attempt of handling a jump
		// It works, but it converts the jz to jmp
		// since I didn't write the code for writing
		// conditional jumps
		//
		/*
		if (*pCurInstr == 0x74) // jz near
		{
			ULONG_PTR Dest = (InstrSize + (ULONG_PTR) Function)
				+ (char) pCurInstr[1];

			WriteJump(&pBridgeBuffer[CurrentBridgeBufferSize], Dest);

#ifdef _M_IX86

			CurrentBridgeBufferSize += JUMP_SIZE;

#else ifdef _M_AMD64

			if (Is64bitJumpNeeded((ULONG_PTR) &pBridgeBuffer[CurrentBridgeBufferSize], Dest))
				CurrentBridgeBufferSize += JUMP_SIZE;
			else
				CurrentBridgeBufferSize += SMALL_JUMP;

#endif

		}
		else
		{*/
			memcpy(&pBridgeBuffer[CurrentBridgeBufferSize], 
				(VOID *) pCurInstr, decodedInstructions[x].size);

			CurrentBridgeBufferSize += decodedInstructions[x].size;
		//}

		InstrSize += decodedInstructions[x].size;
	}

	WriteJump(&pBridgeBuffer[CurrentBridgeBufferSize], Function + InstrSize);
	CurrentBridgeBufferSize += JUMP_SIZE;

	return pBridge;
}

//
// Hooks a function
//

extern "C" __declspec(dllexport)
BOOL __cdecl HookFunction(ULONG_PTR OriginalFunction, ULONG_PTR NewFunction)
{
	//
	// Check if the function has already been hooked
	// If so, no disassembling is necessary since we already
	// have our bridge
	//

	HOOK_INFO *hinfo = GetHookInfoFromFunction(OriginalFunction);

	if (hinfo)
	{
		WriteJump((VOID *) OriginalFunction, NewFunction);
	}
	else
	{
		if (NumberOfHooks == (MAX_HOOKS - 1))
			return FALSE;

#ifdef _M_IX86

		VOID *pBridge = CreateBridge(OriginalFunction);

#else ifdef _M_AMD64

		UINT JumpSize = Is64bitJumpNeeded(OriginalFunction, NewFunction) 
			== TRUE ? JUMP_SIZE : SMALL_JUMP;
		VOID *pBridge = CreateBridge(OriginalFunction, JumpSize);

#endif

		if (pBridge == NULL)
			return FALSE;

		HookInfo[NumberOfHooks].Function = OriginalFunction;
		HookInfo[NumberOfHooks].Bridge = (ULONG_PTR) pBridge;
		HookInfo[NumberOfHooks].Hook = NewFunction;

		NumberOfHooks++;

		WriteJump((VOID *) OriginalFunction, NewFunction);
	}

	return TRUE;
}


//
// Unhooks a function
//

extern "C" __declspec(dllexport)
VOID __cdecl UnhookFunction(ULONG_PTR Function)
{
	//
	// Check if the function has already been hooked
	// If not, I can't unhook it
	//

	HOOK_INFO *hinfo = GetHookInfoFromFunction(Function);

	if (hinfo)
	{
		//
		// Replaces the hook jump with a jump to the bridge
		// I'm not completely unhooking since I'm not
		// restoring the original bytes
		//

		WriteJump((VOID *) hinfo->Function, hinfo->Bridge);
	}
}

//
// Get the bridge to call instead of the original function from hook
//

extern "C" __declspec(dllexport)
ULONG_PTR __cdecl GetOriginalFunction(ULONG_PTR Hook)
{
	if (NumberOfHooks == 0)
		return NULL;

	for (UINT x = 0; x < NumberOfHooks; x++)
	{
		if (HookInfo[x].Hook == Hook)
			return HookInfo[x].Bridge;
	}

	return NULL;
}
