#define _WIN32_WINNT 0x0400

#include <windows.h>
#include <odbg\plugin.h>        // Please change for your environment

#define MAX_API_NAME_LEN    256
#define MAX_DLL_NAME_LEN    50
#define FILE_ALIGNMENT      0x200
#define SECT_ALIGNMENT      0x1000

typedef  PIMAGE_DOS_HEADER         PIDH;
typedef  PIMAGE_NT_HEADERS         PINH;
typedef  PIMAGE_SECTION_HEADER     PISH;
typedef  PIMAGE_DATA_DIRECTORY     PIDD;
typedef  PIMAGE_IMPORT_DESCRIPTOR  PIID;
typedef  PIMAGE_IMPORT_BY_NAME     PIBN;

typedef struct _IMPORT_API_DATA {
  DWORD                     ThunkRVA;
  DWORD                     ApiAddress;
  WORD                      Ordinal;
  char                      ApiName[MAX_API_NAME_LEN];
  char                      DllName[MAX_DLL_NAME_LEN];
  struct _IMPORT_API_DATA   *next;
} IMPORT_API_DATA, *PIMPORT_API_DATA;

typedef struct _IMPORT_DLL_DATA {
  char                     DllName[MAX_DLL_NAME_LEN];
  DWORD                    FirstThunkRVA;
  IMPORT_API_DATA          ApiHead;
  struct _IMPORT_DLL_DATA  *next;
} IMPORT_DLL_DATA, *PIMPORT_DLL_DATA;

DWORD calc_crc(BYTE *p, DWORD s);
int bm_search(const BYTE *b, const int blen, const BYTE *p, const int plen);
static DWORD rva2offset(DWORD dwRva, PISH pISH, int NumberOfSections);
static DWORD offset2rva(DWORD dwOffset, PISH pISH, int NumberOfSections);
static PISH  rva2section(DWORD dwRva, PISH pISH, int NumberOfSections);
static DWORD GetRealApiAddress(DWORD ApiAddress, char *DllName, char *ApiName, WORD *Ordinal);
static WORD  GetApiNameOrdinal(const DWORD ApiAddress, const char *DllName, char *ApiName);
static WORD  ForwarderSearch(const char *DllName, const char *Forwarded, char *Forwarder);
static BOOL  SearchImportData(PIMPORT_DLL_DATA pDllEntry,PINH pINH,PISH pISH);
static void  MakeIID(BYTE *pMemBase, DWORD dwNewSectSize,PIMPORT_DLL_DATA pDllEntry);

BOOL  RebuildImport(char *szTargetFile);

static const char  *szNewSecName = ".idata2";
static DWORD DllNum;

extern BOOL  SearchAnimation;
extern BOOL  SearchLog;
extern DWORD AnimationWait;
extern int   commandcount;


DWORD calc_crc(BYTE *p, DWORD s)
  // p : pointer to data
  // s : size of data
{
  DWORD x,i;

  x = 0xFFFFFFFF;
  while(s--) {
    x ^= *p++;
    i = 8;
    while(i--) {
      x = (x >> 1) ^ ((x & 1) * 0xEDB88320);
    }
  }
  return(~x);
}

#ifndef max
#define max(a, b)   ((a) > (b) ? a : b)
#endif

// Boyer-Moore ip^[T[`pj
int bm_search(const BYTE *b, const int blen, const BYTE *ptn, const int plen)
{
  int skip[256];
  int i,j;

  for(i=0; i<256; i++) {
    skip[i] = plen;
  }
  for(i=0; i<plen-1; i++) {
    skip[*(ptn+i)] = plen - i - 1;
  }

  i = plen - 1;
  while(i < blen) {
    j = plen - 1;
    while(*(b+i) == *(ptn+j)) {
      if(j==0) {
        return i;
      }
      i--; j--;
    }
    i += max(skip[*(b+i)], plen - j);
  }
  return -1;
}

DWORD rva2offset(DWORD dwRva, PISH pISH, int NumberOfSections)
{
  int i;
  PISH pISH2;

  if(dwRva == 0) {
    return(dwRva);
  }

  pISH2 = pISH;
  // in which section is the import table ?
  for(i=0; i<NumberOfSections; i++) {
    if(dwRva >= pISH2->VirtualAddress && dwRva < pISH2->VirtualAddress+pISH2->Misc.VirtualSize) {
      break;
    }
    pISH2++;
  }
  return(dwRva - pISH2->VirtualAddress + pISH2->PointerToRawData);
}

DWORD offset2rva(DWORD dwOffset, PISH pISH, int NumberOfSections)
{
  int i;
  PISH pISH2;

  if(dwOffset == 0) {
    return(dwOffset);
  }
  pISH2 = pISH;
  // in which section is the import table ?
  for(i=0; i<NumberOfSections; i++) {
    if(dwOffset >= pISH2->PointerToRawData && dwOffset < pISH2->PointerToRawData+pISH2->SizeOfRawData) {
      break;
    }
    pISH2++;
  }
  return(dwOffset + pISH2->VirtualAddress - pISH2->PointerToRawData);
}

PISH rva2section(DWORD dwRva, PISH pISH, int NumberOfSections)
{
  PISH pISH2;
  int i;

  pISH2 = pISH;
  for(i=0; i<NumberOfSections; i++) {
    if(dwRva >= pISH2->VirtualAddress && dwRva < pISH2->VirtualAddress+pISH2->Misc.VirtualSize) {
      break;
    }
    pISH2++;
  }
  return(pISH2);
}

#define MAX_BYTE_COUNT 50

///////////////////////////////////////////////////////////////////////////////////////
//
// Chack the given value is API address. If not, try to find real API address
//
// args
//  DWORD ApiAddress : API address
//  char  *DllName   : buffer for DLL name includes API
//  char  *ApiName   : buffer for API name
//  WORD  *Ordinal   : buffer for API ordinal
//
// return value : API address or 0 (DWORD)
////////////////////////////////////////////////////////////////////////////////////////
DWORD GetRealApiAddress(DWORD ApiAddress, char *DllName, char *ApiName, WORD *Ordinal)
{
  t_memory *pmem;
  t_module *pmod;
  t_disasm dasm;
  char     *exename,*pdest;
  BYTE     cmd[MAXCMDSIZE],r;
  DWORD    Address,AddrBuf,bcount,cmdsize,tElock,SearchAddr,ASPR,PushBuf;

  exename = (char*)Plugingetvalue(VAL_EXEFILENAME);
  pdest   = strrchr(exename,'\\');
  pdest++;
  exename = pdest;

  Address = ApiAddress;
  pmod = Findmodule(Address);
  if(pmod) {
    if((pdest = strrchr(pmod->path,'\\')) != NULL) {
      pdest++;
      wsprintf(DllName,"%s",pdest);
    }

    if(stricmp(DllName,exename) != 0) {
      *Ordinal = GetApiNameOrdinal(Address,DllName,ApiName);
      if(*Ordinal != 0xFFFF) {
        if(SearchLog) {
          //Addtolist(Address,0,"Found in %-12s --- Address:%08X  Ordinal:%04X  API name:%-25s",DllName,Address,*Ordinal,ApiName);
          Updatelist();
        }
        goto ADDRESS_FOUND;
      }
    }
    else {
      SearchAddr = Address;
      if(SearchAnimation) {
        Setcpu(0,SearchAddr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
        Sleep(AnimationWait);
      }
      Readmemory(&r,SearchAddr,1,MM_RESTORE|MM_SILENT);
      if(r == 0xE9) {
        Readmemory(&AddrBuf,SearchAddr+1,sizeof(DWORD),MM_RESTORE|MM_SILENT);
        Address += (AddrBuf+5);
        pmod = Findmodule(Address);
        if(pmod) {
          if((pdest = strrchr(pmod->path,'\\')) != NULL) {
            pdest++;
            wsprintf(DllName,"%s",pdest);
            *Ordinal = GetApiNameOrdinal(Address,DllName,ApiName);
            if(*Ordinal != 0xFFFF) {
              if(SearchLog) {
                //Addtolist(Address,0,"Found in %-12s --- Address:%08X  Ordinal:%04X  API name:%-25s",DllName,Address,*Ordinal,ApiName);
                Updatelist();
              }
              goto ADDRESS_FOUND;
            }
          }
        }
      }
    }
  }
  else { // module not found

    // try to find PELock's disguised API address
    __try {
      if(SearchLog) {
        Addtolist(0,0,"PELock and ASProtect search");
        Updatelist();
      }
      Address = ApiAddress;
      pmem = Findmemory(Address);
      bcount = 0;
      SearchAddr = Address;
      while(bcount < MAX_BYTE_COUNT) {
        if(SearchAnimation) {
          Setcpu(0,SearchAddr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
          Sleep(AnimationWait);
        }
        Readmemory(cmd,SearchAddr,MAXCMDSIZE,MM_RESTORE|MM_SILENT);
        cmdsize = Disasm(cmd,MAXCMDSIZE,SearchAddr,NULL,&dasm,DISASM_ALL,NULL);
        if(SearchLog) {
          Addtolist(SearchAddr,0,"  %-20s  %-30s",dasm.dump,dasm.result);
          Updatelist();
        }
        Readmemory(&r,SearchAddr,1,MM_RESTORE|MM_SILENT);
        if(r == 0x68) {
          PushBuf = dasm.immconst;
        }
        if(r == 0xC3) {
          Address = PushBuf;
          bcount -= 5;
          break;
        }
        if(r == 0xE9) {
          Address = dasm.jmpaddr;
          break;
        }
        if(r == 0xEB) {
          SearchAddr = dasm.jmpaddr;
          continue;
        }
        SearchAddr += cmdsize;
        bcount     += cmdsize;
      }
      if(SearchAddr < pmem->base + pmem->size) {
        Address -= bcount;
        if(SearchAnimation) {
          Setcpu(0,Address,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
          Sleep(AnimationWait);
        }
        pmod = Findmodule(Address);
        if(pmod) {
          if((pdest = strrchr(pmod->path,'\\')) != NULL) {
            pdest++;
            wsprintf(DllName,"%s",pdest);
            *Ordinal = GetApiNameOrdinal(Address,DllName,ApiName);
            if(*Ordinal != 0xFFFF) {
              if(SearchLog) {
                Addtolist(Address,0,"Found in %-12s --- Address:%08X  Ordinal:%04X  API name:%-25s",DllName,Address,*Ordinal,ApiName);
                Updatelist();
              }
              goto ADDRESS_FOUND;
            }
          }
        }
      }
    }
    __except(1) {
      Addtolist(SearchAddr,1,"Exception in searching PELock's method");
    }

    // try to find tElock0.96's disguised API address
    __try {
      if(SearchLog) {
        Addtolist(0,0,"tElock 0.96 search");
      }
      Address = ApiAddress;
      bcount = 0;
      SearchAddr = Address;
      while(bcount < MAX_BYTE_COUNT) {
        if(SearchAnimation) {
          Setcpu(0,SearchAddr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
          Sleep(AnimationWait);
        }
        Readmemory(cmd,SearchAddr,MAXCMDSIZE,MM_RESTORE|MM_SILENT);
        cmdsize = Disasm(cmd,MAXCMDSIZE,SearchAddr,NULL,&dasm,DISASM_ALL,NULL);
        if(SearchLog) {
          Addtolist(SearchAddr,0,"  %-20s  %s",dasm.dump,dasm.result);
          Updatelist();
        }
        Readmemory(&r,SearchAddr,1,MM_RESTORE|MM_SILENT);
        if(!stricmp(dasm.result,"stc")) {
          SearchAddr += cmdsize;
          Readmemory(cmd,SearchAddr,MAXCMDSIZE,MM_RESTORE|MM_SILENT);
          cmdsize = Disasm(cmd,MAXCMDSIZE,SearchAddr,NULL,&dasm,DISASM_ALL,NULL);
          if(SearchLog) {
            Addtolist(SearchAddr,0,"  %-20s  %s",dasm.dump,dasm.result);
            Updatelist();
          }
          Readmemory(&r,SearchAddr,1,MM_RESTORE|MM_SILENT);
          if(r == 0x72) {
            SearchAddr = dasm.jmpaddr;
            continue;
          }
        }
        if(r == 0xEB || r == 0xE9 || r == 0x75 || r == 0x79) {
          SearchAddr = dasm.jmpaddr;
          continue;
        }
        if(r == 0xB8) {
          Readmemory(&AddrBuf,SearchAddr+1,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          Readmemory(&Address,AddrBuf,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          pmod = Findmodule(Address);
          if(pmod) {
            if((pdest = strrchr(pmod->path,'\\')) != NULL) {
              pdest++;
              wsprintf(DllName,"%s",pdest);
              *Ordinal = GetApiNameOrdinal(Address,DllName,ApiName);
              if(*Ordinal != 0xFFFF) {
                if(SearchLog) {
                  Addtolist(Address,0,"Found in %-12s --- Address:%08X  Ordinal:%04X  API name:%-25s",DllName,Address,*Ordinal,ApiName);
                  Updatelist();
                }
                goto ADDRESS_FOUND;
              }
            }
          }
        }
        SearchAddr += cmdsize;
        bcount     += cmdsize;
      }
    }
    __except(1) {
      Addtolist(SearchAddr,1,"Exception in searching tElock0.96's method");
    }

    // try to find tElock0.98's disguised API address
    __try {
      if(SearchLog) {
        Addtolist(0,0,"tElock 0.98 search");
        Updatelist();
      }
      Address = ApiAddress;
      bcount = 0;
      SearchAddr = Address;
      while(bcount < MAX_BYTE_COUNT) {
        if(SearchAnimation) {
          Setcpu(0,SearchAddr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
          Sleep(AnimationWait);
        }
        Readmemory(cmd,SearchAddr,MAXCMDSIZE,MM_RESTORE|MM_SILENT);
        cmdsize = Disasm(cmd,MAXCMDSIZE,SearchAddr,NULL,&dasm,DISASM_ALL,NULL);
        if(SearchLog) {
          Addtolist(SearchAddr,0,"  %-20s  %s",dasm.dump,dasm.result);
          Updatelist();
        }
        Readmemory(&tElock,SearchAddr,sizeof(DWORD),MM_RESTORE|MM_SILENT);
        Readmemory(&r,SearchAddr,1,MM_RESTORE|MM_SILENT);
        if(tElock == 0xC330FF40) {
          Readmemory(&AddrBuf,SearchAddr-4,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          AddrBuf++;
          Readmemory(&Address,AddrBuf,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          if(SearchLog) {
            Addtolist(SearchAddr-5,0,"found tElock 0.98 signature.   API address is in %08X",AddrBuf);
            Updatelist();
          }
          pmod = Findmodule(Address);
          if(pmod) {
            if((pdest = strrchr(pmod->path,'\\')) != NULL) {
              pdest++;
              wsprintf(DllName,"%s",pdest);
              *Ordinal = GetApiNameOrdinal(Address,DllName,ApiName);
              if(*Ordinal != 0xFFFF) {
                if(SearchLog) {
                  Addtolist(Address,0,"Found in %-12s --- Address:%08X  Ordinal:%04X  API name:%-25s",DllName,Address,*Ordinal,ApiName);
                  Updatelist();
                }
                goto ADDRESS_FOUND;
              }
            }
          }
        }
        if(r == 0xEB || r == 0xE9) {
          SearchAddr = dasm.jmpaddr;
          continue;
        }
        SearchAddr += cmdsize;
        bcount     += cmdsize;
      }
    }
    __except(1) {
      Addtolist(SearchAddr,1,"Exception in searching tElock0.98's method");
    }

    // ASProtect Special
    __try {
      if(SearchLog) {
        Addtolist(0,0,"ASProtect Special search");
      }
      Address = ApiAddress;
      bcount = 0;
      SearchAddr = Address;
      while(bcount < MAX_BYTE_COUNT) {
        if(SearchAnimation) {
          Setcpu(0,SearchAddr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
          Sleep(AnimationWait);
        }
        Readmemory(&ASPR,SearchAddr,sizeof(DWORD),MM_RESTORE|MM_SILENT);
        if(ASPR == 0x505207EB) {
          if(SearchLog) {
            Addtolist(SearchAddr,0,"found ASProtect Special signature");
            Updatelist();
          }
          Readmemory(&AddrBuf,SearchAddr+5,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          AddrBuf += SearchAddr + 5 + 4;
          Readmemory(&AddrBuf,AddrBuf+2,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          Readmemory(&Address,AddrBuf,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          pmod = Findmodule(Address);
          if(pmod) {
            if((pdest = strrchr(pmod->path,'\\')) != NULL) {
              pdest++;
              wsprintf(DllName,"%s",pdest);
              *Ordinal = GetApiNameOrdinal(Address,DllName,ApiName);
              if(*Ordinal != 0xFFFF) {
                if(SearchLog) {
                  Addtolist(Address,0,"Found in %-12s --- Address:%08X  Ordinal:%04X  API name:%-25s",DllName,Address,*Ordinal,ApiName);
                  Updatelist();
                }
                goto ADDRESS_FOUND;
              }
              *Ordinal = 0xFFFF;
            }
          }
        }
        SearchAddr++;
        bcount++;
      }
    }
    __except(1) {
      Addtolist(SearchAddr,1,"Exception in searching ASProtect's special method");
    }

  }
  //Addtolist(Address,1,"Not Found API in %-12s",DllName);
  DllName[0] = '\0';
  ApiName[0] = '\0';
  *Ordinal   = 0xFFFF;
  Address    = ApiAddress;
  Updatelist();

ADDRESS_FOUND:
  return(Address);
} // End of GetRealApiAddress()


BOOL SearchImportData(PIMPORT_DLL_DATA pDllEntry,PINH pINH,PISH pISH)
{
  const WORD srch[] = {0x25FF,0x15FF};
  const char *command[] = {"jmp [Thunk]","call[Thunk]"};
  DWORD  i,NumOfSect,ImageBase,dwBuf;
  char   *exename,*pdest;
  char   ApiName[MAX_API_NAME_LEN],DllName[MAX_DLL_NAME_LEN],Forwarded[MAX_API_NAME_LEN+MAX_DLL_NAME_LEN];
  WORD   buf,Ordinal;
  DWORD  s,ThunkData,FunctAddress,RangeTop,RangeEnd,Thunk,ThunkBuf;
  PIMPORT_DLL_DATA pDll,qDll,pDllNew;
  PIMPORT_API_DATA pApi,qApi,pApiNew;
  PISH psech;
  t_memory *pmem;
  BOOL AlreadySearched;

  exename = (char*)Plugingetvalue(VAL_EXEFILENAME);
  pdest = strrchr(exename,'\\');
  pdest++;
  exename = pdest;
  NumOfSect = pINH->FileHeader.NumberOfSections;
  ImageBase = pINH->OptionalHeader.ImageBase;
  RangeTop  = pISH[0].VirtualAddress + ImageBase;
  RangeEnd  = pISH[NumOfSect-1].VirtualAddress + pISH[NumOfSect-1].Misc.VirtualSize + ImageBase;

  __try {
    Addtolist(0,0,"OllyDump -- Start \"JMP [Thunk]\"\(0x25FF\) and \"CALL [Thunk]\"\(0x15FF\) search");
    s = RangeTop;
    pmem = Findmemory(s);
    while(s < pmem->base+pmem->size) {
      Readmemory(&buf,s,sizeof(WORD),MM_RESTORE|MM_SILENT);
      if(buf == srch[0] || buf == srch[1]) {
        Readmemory(&Thunk,s+2,sizeof(DWORD),MM_RESTORE|MM_SILENT); // get operand(=jmp/call address)
        if(!Findmemory(Thunk)) {
          s++;
          continue;
        }
        Readmemory(&dwBuf,Thunk,sizeof(DWORD),MM_RESTORE|MM_SILENT);
        if(!Findmemory(dwBuf)) {
          s++;
          continue;
        }
        i = (buf == srch[0]) ? 0 : 1;
        Addtolist(s,1,"%s found on %08X  Thunk:%08X",command[i],s,Thunk);

        // avoid duplication check
        if(i == 1) {
          pDll = pDllEntry->next;
          AlreadySearched = FALSE;
          while(pDll != NULL) {
            if(Thunk-ImageBase < pDll->FirstThunkRVA) {
              pDll = pDll->next;
              continue;
            }
            else if(Thunk-ImageBase == pDll->FirstThunkRVA) {
              AlreadySearched = TRUE;
              break;
            }
            pApi = pDll->ApiHead.next;
            while(pApi != NULL) {
              if(Thunk-ImageBase == pApi->ThunkRVA) {
                AlreadySearched = TRUE;
                break;
              }
              pApi = pApi->next;
            }
            if(AlreadySearched) {
              break;
            }
            pDll = pDll->next;
          }
          if(AlreadySearched) {
            s++;
            continue;
          }
        }

        if(SearchAnimation) {
          Setcpu(0,Thunk,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS|CPU_REDRAW);
          Sleep(AnimationWait);
        }
        if(Thunk >= RangeTop && Thunk <= RangeEnd-4) {
          Readmemory(&FunctAddress,Thunk,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          if(FunctAddress == 0) {
            s++;
            continue;
          }
          FunctAddress = GetRealApiAddress(FunctAddress,DllName,ApiName,&Ordinal);
          if(Ordinal != 0xFFFF) {
            if(stricmp(exename,DllName)) { // not self module

              // Search FirstThunk
              psech = rva2section(Thunk-ImageBase,pISH,NumOfSect);
              ThunkBuf = Thunk;
              while(ThunkBuf != 0 && ThunkBuf-ImageBase >= psech->VirtualAddress) {
                Readmemory(&ThunkData,ThunkBuf,sizeof(DWORD),MM_RESTORE|MM_SILENT);
                if(ThunkData == 0) {
                  break;
                }
                ThunkBuf -= sizeof(DWORD);
              }
              ThunkBuf += sizeof(DWORD);
              Readmemory(&ThunkData,ThunkBuf,sizeof(DWORD),MM_RESTORE|MM_SILENT);

              // Search Same FirstThunk DLL
              pDll = pDllEntry->next;
              qDll = pDllEntry;
              while(pDll != NULL && ThunkBuf-ImageBase > pDll->FirstThunkRVA) {
                qDll = pDll;
                pDll = pDll->next;
              }
              if(pDll == NULL || pDll->FirstThunkRVA != ThunkBuf-ImageBase) {
                if((pDllNew = (PIMPORT_DLL_DATA)calloc(1,sizeof(IMPORT_DLL_DATA))) == NULL) {
                  Addtolist(0,1,"OllyDump -- Error  Memory Allocation for New DLL Entry Failed!!");
                  return(FALSE);
                }
                pDllNew->next = pDll;
                qDll->next = pDllNew;
                pDllNew->FirstThunkRVA = ThunkBuf - ImageBase;
                wsprintf(pDllNew->DllName,"%s",DllName);
                if((pApiNew = (PIMPORT_API_DATA)calloc(1,sizeof(IMPORT_API_DATA))) == NULL) {
                  Addtolist(0,1,"OllyDump -- Error  Memory Allocation for New API Entry Failed!!");
                  return(FALSE);
                }
                pApiNew->ThunkRVA   = Thunk - ImageBase;
                pApiNew->ApiAddress = FunctAddress;
                pApiNew->Ordinal    = Ordinal;
                wsprintf(pApiNew->ApiName,"%s",ApiName);
                wsprintf(pApiNew->DllName,"%s",DllName);
                pApiNew->next       = NULL;
                pDllNew->ApiHead.next = pApiNew;
              }
              else {
                pApi = pDll->ApiHead.next;
                qApi = &(pDll->ApiHead);
                while(pApi != NULL && Thunk-ImageBase > pApi->ThunkRVA) {
                  qApi = pApi;
                  pApi = pApi->next;
                }
                if(pApi == NULL || Thunk-ImageBase != pApi->ThunkRVA) {
                  if((pApiNew = (PIMPORT_API_DATA)calloc(1,sizeof(IMPORT_API_DATA))) == NULL) {
                    Addtolist(0,1,"OllyDump -- Error  Memory Allocation for New API Entry Failed!!");
                    return(FALSE);
                  }
                  pApiNew->next = pApi;
                  qApi->next    = pApiNew;
                  pApiNew->ThunkRVA   = Thunk - ImageBase;
                  pApiNew->ApiAddress = FunctAddress;
                  pApiNew->Ordinal    = Ordinal;
                  wsprintf(pApiNew->ApiName,"%s",ApiName);
                  wsprintf(pApiNew->DllName,"%s",DllName);
                }
              }
            }
          }
        }
      }
      s++;
    }
  } // __try end
  __except(1) {
    Addtolist(0,0,"OllyDump -- Exception  \"JMP [Thunk]\"\(0x25FF\) and \"CALL [Thunk]\"\(0x15FF\) search block");
  }

  __try {
    Addtolist(0,0,"OllyDump --  Check Leaked Thunks in Thunk Blocks");
    pDll = pDllEntry->next;
    while(pDll != NULL) {
      Thunk = pDll->FirstThunkRVA + ImageBase;
      psech = rva2section(pDll->FirstThunkRVA,pISH,NumOfSect);
      Readmemory(&FunctAddress,Thunk,sizeof(DWORD),MM_RESTORE|MM_SILENT);
      while(FunctAddress != 0 && Thunk-ImageBase >= psech->VirtualAddress && Thunk-ImageBase <= psech->VirtualAddress + psech->Misc.VirtualSize) {
        pApi = pDll->ApiHead.next;

        AlreadySearched = FALSE;
        while(pApi != NULL) {
          if(pApi->ThunkRVA == Thunk-ImageBase) {
            AlreadySearched = TRUE;
            break;
          }
          pApi = pApi->next;
        }
        if(AlreadySearched) {
          Thunk += sizeof(DWORD);
          Readmemory(&FunctAddress,Thunk,sizeof(DWORD),MM_RESTORE|MM_SILENT);
          continue;
        }

        FunctAddress = GetRealApiAddress(FunctAddress,DllName,ApiName,&Ordinal);
        pApi = pDll->ApiHead.next;
        qApi = &(pDll->ApiHead);
        while(pApi != NULL && (Thunk-ImageBase) > pApi->ThunkRVA) {
          qApi = pApi;
          pApi = pApi->next;
        }
        if(pApi == NULL || Thunk-ImageBase != pApi->ThunkRVA) {
          if((pApiNew = (PIMPORT_API_DATA)calloc(1,sizeof(IMPORT_API_DATA))) == NULL) {
            Addtolist(0,1,"OllyDump -- Error  Memory Allocation for New API Entry Failed!!");
            return(FALSE);
          }
          pApiNew->next       = pApi;
          qApi->next          = pApiNew;
          pApiNew->ThunkRVA   = Thunk - ImageBase;
          pApiNew->ApiAddress = FunctAddress;
          pApiNew->Ordinal    = Ordinal;
          wsprintf(pApiNew->ApiName,"%s",ApiName);
          wsprintf(pApiNew->DllName,"%s",DllName);
        }
        Thunk += sizeof(DWORD);
        Readmemory(&FunctAddress,Thunk,sizeof(DWORD),MM_RESTORE|MM_SILENT);
      }
      pDll = pDll->next;
    }
  } // __try end
  __except(1) {
    Addtolist(0,1,"OllyDump -- Exception  Checking Leaked Thunks Block!!");
  }

  __try {
    Addtolist(0,0,"OllyDump -- Resolve Forwarder");
    pDll = pDllEntry->next;
    while(pDll != NULL) {
      //wsprintf(pDll->DllName,"%s",pDll->ApiHead.next->DllName);
      if(!stricmp(pDll->DllName,"ntdll.dll")) {
        pApi = pDll->ApiHead.next;
        while(pApi != NULL) {
          if(!stricmp(pApi->DllName,"kernel32.dll")) {
            wsprintf(pDll->DllName,"%s",pApi->DllName);
            break;
          }
        }
      }
      pApi = pDll->ApiHead.next;
      while(pApi != NULL) {
        if(stricmp(pApi->DllName,pDll->DllName)) {
          wsprintf(Forwarded,"%s",pApi->DllName);
          pdest = strrchr(Forwarded,'.');
          pdest++;
          *pdest = '\0';
          strcat(Forwarded,pApi->ApiName);
          Addtolist(0,0,"%s must be forwarded API from %s",Forwarded,pDll->DllName);
          Ordinal = ForwarderSearch(pDll->DllName, Forwarded, pApi->ApiName);
          if(Ordinal != 0xFFFF) {
            pApi->Ordinal = Ordinal;
            wsprintf(pApi->DllName,"%s",pDll->DllName);
          }
        }
        pApi = pApi->next;
      }
      pDll = pDll->next;
    }
  } // __try end
  __except(1) {
    Addtolist(0,1,"OllyDump -- Exception in Resolve Forwarder !!");
  }

  return(TRUE);
}

////////////////////////////////////////////////////
//
// Get API Name and Ordinal
//
// args
//  DWORD ApiAddress : API entry point address
//  char  *ApiName   : buffer for found API name
//  char  *DllName   : DLL name
//
// return value : API Ordinal
//
WORD GetApiNameOrdinal(const DWORD ApiAddress, const char *DllName, char *ApiName)
{
  DWORD functionentry;
  DWORD *pDW;
  WORD  *pWO;
  DWORD i;
  DWORD functposition,nameposition;
  DWORD modulebase;
  // export table values
  DWORD expbase;
  DWORD functnum;
  DWORD functaddr;
  DWORD namenum;
  DWORD nameaddr;
  DWORD ordinaladdr;
  WORD  ordinal;
  // PE structs
  PIDH dosh;
  PINH peh;
  IMAGE_DATA_DIRECTORY IDD;
  //PIMAGE_EXPORT_DIRECTORY pIED;

  functposition = 0xFFFFFFFF;

  __try {
    // load the dll
    modulebase = (DWORD)GetModuleHandle(DllName);
    if(modulebase == 0) {
      modulebase = (DWORD)LoadLibrary(DllName);
      if(modulebase == 0) {
        return((WORD)(functposition&0xFFFF));
      }
    }
    if(ApiAddress <= modulebase) {
      return((WORD)(functposition&0xFFFF));
    }
    // check whether hmodule is a valid PE file
    dosh = (PIDH)modulebase;
    if(dosh->e_magic != IMAGE_DOS_SIGNATURE) {
      return((WORD)(functposition&0xFFFF));
    }
    peh = (PINH)((DWORD)dosh + dosh->e_lfanew);
    if(peh->Signature != IMAGE_NT_SIGNATURE) {
      return((WORD)(functposition&0xFFFF));
    }

    IDD = peh->OptionalHeader.DataDirectory[0];
    modulebase = peh->OptionalHeader.ImageBase;

    functionentry = ApiAddress - modulebase;

    pDW = (DWORD*)(IDD.VirtualAddress + 0x10 + (DWORD)dosh); // go fast to the base

    // get the export values
    expbase     = *pDW; pDW++;
    functnum    = *pDW; pDW++;
    namenum     = *pDW; pDW++;
    functaddr   = *pDW; pDW++;
    nameaddr    = *pDW; pDW++;
    ordinaladdr = *pDW;

    // search the entry in the RVA array of the export table
    pDW = (DWORD*)((DWORD)dosh + functaddr);
    functposition = 0xFFFFFFFF;
    for(i=0; i<functnum; i++) {
      if(functionentry == *pDW) {
        functposition = i;
        break;
      }
      pDW++;
    }
    if(functposition != 0xFFFFFFFF) {
      nameposition = 0xFFFFFFFF;
      pWO = (WORD*)(ordinaladdr + peh->OptionalHeader.ImageBase);
      for(i=0; i<namenum; i++) {
        if(functposition == (DWORD)*pWO) {
          nameposition = i;
          break;
        }
        pWO++;
      }
      if(nameposition != 0xFFFFFFFF) {
        pDW = (DWORD*)(nameaddr + peh->OptionalHeader.ImageBase);
        pDW += nameposition;
        wsprintf(ApiName,"%s",(char*)(*pDW+peh->OptionalHeader.ImageBase));
      }
      else {
        wsprintf(ApiName,"");
      }
    }
    ordinal = (WORD)((functposition + ((functposition == 0xFFFFFFFF) ? 0 : expbase))&0xFFFF);
  }
  __except(1) {
    wsprintf(ApiName,"");
    return(0xFFFF);
  }
  return(ordinal);
}

WORD ForwarderSearch(const char *DllName, const char *Forwarded, char *Forwarder)
{
  DWORD i, *pDW, functposition, nameposition, modulebase;
  WORD  *pWO,ordinal;

  // PE structs
  PIDH pIDH;
  PINH pINH;
  IMAGE_DATA_DIRECTORY IDD;
  PIMAGE_EXPORT_DIRECTORY pIED;

  //Addtolist(0,0,"in ForwarderSearch -- DllName:%s  Forwarded:%s  Forwarder:%s",DllName,Forwarded,Forwarder);
  __try {
    // load the dll
    modulebase = (DWORD)GetModuleHandle(DllName);
    if(modulebase == 0) {
      modulebase = (DWORD)LoadLibrary(DllName);
      if(modulebase == 0) {
        return(0xFFFF);
      }
    }
    // check whether hmodule is a valid PE file
    pIDH = (PIDH)modulebase;
    if(pIDH->e_magic != IMAGE_DOS_SIGNATURE) {
      return(0xFFFF);
    }
    pINH = (PINH)((DWORD)pIDH + pIDH->e_lfanew);
    if(pINH->Signature != IMAGE_NT_SIGNATURE) {
      return(0xFFFF);
    }

    IDD = pINH->OptionalHeader.DataDirectory[0];
    pIED = (PIMAGE_EXPORT_DIRECTORY)(IDD.VirtualAddress + modulebase);
    modulebase = pINH->OptionalHeader.ImageBase;

    // search the entry in the RVA array of the export table
    Addtolist(pIED->AddressOfFunctions+modulebase,0,"Export Address Table RVA:%08X",pIED->AddressOfFunctions);
    pDW = (DWORD*)(pIED->AddressOfFunctions + modulebase);
    functposition = 0xFFFFFFFF;
    for(i=0; i<pIED->NumberOfFunctions; i++) {
      if(!stricmp((char*)(*pDW+modulebase),Forwarded)) {
        functposition = i;
        Addtolist(*pDW+modulebase,0,"Forwarded API %s found on the ForwarderRVA:%08X  pos:%d",Forwarded,*pDW,functposition);
        break;
      }
      pDW++;
    }
    if(functposition != 0xFFFFFFFF) {
      nameposition = 0xFFFFFFFF;
      pWO = (WORD*)(pIED->AddressOfNameOrdinals + modulebase);
      for(i=0; i<pIED->NumberOfNames; i++) {
        if(functposition == (DWORD)*pWO) {
          nameposition = i;
          break;
        }
        pWO++;
      }
      if(nameposition != 0xFFFFFFFF) {
        pDW = (DWORD*)(pIED->AddressOfNames + modulebase);
        pDW += nameposition;
        wsprintf(Forwarder,"%s",(char*)(*pDW+modulebase));
        Addtolist(*pDW+modulebase,0,"*pDW:%08X  Forwarder:%s  Forwarded:%s",*pDW,Forwarder,Forwarded);
      }
      else {
        wsprintf(Forwarder,"");
      }
    }
    ordinal = (WORD)((functposition + ((functposition == 0xFFFFFFFF) ? 0 : pIED->Base))&0xFFFF);
  }
  __except(1) {
    wsprintf(Forwarder,"");
    return(0xFFFF);
  }
  return(ordinal);
}

void MakeIID(BYTE *pMemBase, DWORD dwNewSectSize,PIMPORT_DLL_DATA pDllEntry)
{
  PIDH    dosh;
  PINH    peh;
  PIID    pIID;
  PIBN    pIBN;
  PISH    pSectionh, pNewSectionh,pSectionh2;
  PIMPORT_DLL_DATA pDll;
  PIMPORT_API_DATA pApi;
  DWORD  i,dwTmpNum,dwMemBase,*pThunk;
  char   *pCH;

  __try {
    dwMemBase = (DWORD)pMemBase;
    // make a new section
    dosh = (PIDH)pMemBase;
    peh  = (PINH)((DWORD)dosh + dosh->e_lfanew);
    pSectionh = (PISH)((DWORD)peh + 0xF8);
    pSectionh2 = pSectionh;
    peh->OptionalHeader.FileAlignment = FILE_ALIGNMENT;
    for(i=0; i<(DWORD)(peh->FileHeader.NumberOfSections); i++) {
      pSectionh++;
    }
    peh->FileHeader.NumberOfSections++;
    for(i=0; i<=7; i++) {
      pSectionh->Name[i] = szNewSecName[i];
    }

    pNewSectionh = pSectionh;
    pNewSectionh->Characteristics  = 0xC0000040;
    pNewSectionh->Misc.VirtualSize = ((dwNewSectSize%SECT_ALIGNMENT) != 0) ? (dwNewSectSize/SECT_ALIGNMENT+1)*SECT_ALIGNMENT : dwNewSectSize;
    pSectionh--;

    // set a valid RawOffset
    dwTmpNum = pSectionh->SizeOfRawData + pSectionh->PointerToRawData;
    if(dwTmpNum%FILE_ALIGNMENT != 0) {
      dwTmpNum = ((dwTmpNum / FILE_ALIGNMENT) + 1) * FILE_ALIGNMENT;
    }
    pNewSectionh->PointerToRawData = dwTmpNum;

    // set a valild VirtualAddress
    dwTmpNum = pSectionh->VirtualAddress + pSectionh->Misc.VirtualSize;
    if(dwTmpNum%SECT_ALIGNMENT != 0) {
      dwTmpNum = ((dwTmpNum/SECT_ALIGNMENT) + 1) * SECT_ALIGNMENT;
    }
    pNewSectionh->VirtualAddress = dwTmpNum;

    // set new section RawSize
    pNewSectionh->SizeOfRawData  = dwNewSectSize;

    // correct the SizeOfImage
    peh->OptionalHeader.SizeOfImage += dwNewSectSize;

    // write the new IID into the new section
    peh->OptionalHeader.DataDirectory[1].VirtualAddress = pNewSectionh->VirtualAddress;
    pIID = (PIID)(dwMemBase + rva2offset(pNewSectionh->VirtualAddress,pSectionh2,peh->FileHeader.NumberOfSections));
    pCH  = (BYTE*)((DWORD)pIID+sizeof(IMAGE_IMPORT_DESCRIPTOR)*(DllNum+1));
    memset(pIID,0,dwNewSectSize);

    pDll = pDllEntry->next;
    while(pDll != NULL) {
      pIID->OriginalFirstThunk = 0;
      pIID->TimeDateStamp      = 0;
      pIID->ForwarderChain     = 0;
      pIID->FirstThunk         = pDll->FirstThunkRVA;
      wsprintf(pCH,"%s",pDll->DllName);
      pIID->Name               = offset2rva((DWORD)(pCH-pMemBase),pSectionh2,peh->FileHeader.NumberOfSections);
      pCH += (strlen(pDll->DllName) + 1);
      pApi = pDll->ApiHead.next;
      while(pApi != NULL) {
        pThunk = (DWORD*)(rva2offset(pApi->ThunkRVA,pSectionh2,peh->FileHeader.NumberOfSections)+dwMemBase);
        if(pApi->Ordinal == 0xFFFF && pApi->ApiName[0] == '\0') {
          *pThunk = 0;
        }
        else {
          if(strlen(pApi->ApiName) == 0) {
            *pThunk = (pApi->Ordinal&0x0000FFFF)|IMAGE_ORDINAL_FLAG;
          }
          else {
            *pThunk = offset2rva((DWORD)(pCH-pMemBase),pSectionh2,peh->FileHeader.NumberOfSections);
            pIBN = (PIBN)pCH;
            pIBN->Hint = pApi->Ordinal;
            wsprintf(pIBN->Name,"%s",pApi->ApiName);
            pCH += sizeof(WORD);
            pCH += (strlen(pApi->ApiName) + 1);
          }
        }
        pApi = pApi->next;
      }
      pDll = pDll->next;
      pIID++;
    }
    pIID->OriginalFirstThunk = 0;
    pIID->TimeDateStamp      = 0;
    pIID->ForwarderChain     = 0;
    pIID->FirstThunk         = 0;
    pIID->Name               = 0;
  }
  __except(1) {
    Addtolist(0,1,"OllyDump -- Exception in MakeIID!!");
  }
}

BOOL RebuildImport(char *szTargetFile)
{
  //const  DWORD AlignBreaker = 0x1200;
  DWORD  dwFsize,dwNewSize,dwNewSectSize,dwBuff,ImageBase,i;
  HANDLE hFile;
  LPBYTE pFileMem;
  PIDH   pFileIDH;
  PINH   pFileINH;
  PISH   pFileISH;
  IMPORT_DLL_DATA DllEntry;
  PIMPORT_DLL_DATA pDll,qDll;
  PIMPORT_API_DATA pApi,qApi;
  BOOL   result;

  result = FALSE;

  // map the file
  hFile = CreateFile(szTargetFile,GENERIC_READ|GENERIC_WRITE,FILE_SHARE_READ|FILE_SHARE_WRITE,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(hFile == INVALID_HANDLE_VALUE) {
    Addtolist(0,1,"OllyDump -- Error  CreateFile %s failed!!",szTargetFile);
    goto CLEAN0;
  }
  dwFsize = GetFileSize(hFile,NULL);
  pFileMem = (BYTE*)HeapAlloc(GetProcessHeap(),HEAP_ZERO_MEMORY,dwFsize);
  if(pFileMem == NULL) {
    Addtolist(0,1,"OllyDump -- Error  HeapAlloc failed!!");
    goto CLEAN1;
  }
  ReadFile(hFile,pFileMem,dwFsize,&dwBuff,0);
  if(dwBuff == 0) {
    Addtolist(0,1,"OllyDump -- Error  ReadFile failed!!");
    goto CLEAN2;
  }

  pFileIDH = (PIDH)pFileMem;
  if(pFileIDH->e_magic != IMAGE_DOS_SIGNATURE) {
    Addtolist(0,1,"OllyDump -- Error  Invalid DOS Signature!!");
    goto CLEAN2;
  }
  pFileINH = (PINH)((DWORD)pFileIDH + pFileIDH->e_lfanew);
  if(pFileINH->Signature != IMAGE_NT_SIGNATURE) {
    Addtolist(0,1,"OllyDump -- Error  Invalid NT Signature!!");
    goto CLEAN2;
  }
  ImageBase = pFileINH->OptionalHeader.ImageBase;
  pFileISH = (PISH)((DWORD)pFileINH + 0xF8);

  DllEntry.next = NULL;
  SearchImportData(&DllEntry,pFileINH,pFileISH);

  // Show DLL and API Search Result
  Addtolist(0,0,"OllyDump -- Import Table");
  pDll = DllEntry.next;
  if(pDll == NULL) {
    Addtolist(0,1,"OllyDump -- Error  No Dll Entry!!");
  }

  // Output Result
  while(pDll != NULL) {
    Addtolist(pDll->FirstThunkRVA+ImageBase,0,"DLL:%s  FirstThunkRVA:%X",pDll->DllName,pDll->FirstThunkRVA);
    pApi = pDll->ApiHead.next;
    if(pApi == NULL) {
      Addtolist(0,1,"OllyDump -- Error  No Api Entry!!");
    }
    Addtolist(0,1,"  DLL Name      Address   Ordinal   API Name");
    while(pApi != NULL) {
      Addtolist(pApi->ThunkRVA+ImageBase,0,"  %-12s  %08X   %04X     %-s",pApi->DllName,pApi->ApiAddress,pApi->Ordinal,pApi->ApiName);
      Updatelist();
      pApi = pApi->next;
    }
    pDll = pDll->next;
  }

  dwNewSize = 0;
  for(i=0; i<(DWORD)(pFileINH->FileHeader.NumberOfSections); i++) {
    if(pFileISH->SizeOfRawData + pFileISH->PointerToRawData > dwNewSize) {
      dwNewSize = pFileISH->SizeOfRawData + pFileISH->PointerToRawData;
    }
    pFileISH++;
  }
  // align the last section and add SECT_ALIGNMENT
  if((dwNewSize % FILE_ALIGNMENT) != 0) {
    dwNewSize = ((dwNewSize / FILE_ALIGNMENT) + 1) * FILE_ALIGNMENT;
  }

  Addtolist(0,-1,"OllyDump -- Calculating New File Size...");
  dwNewSectSize = 0;
  DllNum = 0;
  pDll = DllEntry.next;
  while(pDll != NULL) {
    DllNum++;
    dwNewSectSize += sizeof(IMAGE_IMPORT_DESCRIPTOR);
    dwNewSectSize += (strlen(pDll->DllName) + 1);
    pApi = pDll->ApiHead.next;
    while(pApi != NULL) {
      if(strlen(pApi->ApiName) != 0) {
        dwNewSectSize += (strlen(pApi->ApiName) + 1 + 2);
      }
      pApi = pApi->next;
    }
    pDll = pDll->next;
  }
  if(dwNewSectSize%FILE_ALIGNMENT != 0) {
    dwNewSectSize = ((dwNewSectSize/FILE_ALIGNMENT) + 1) * FILE_ALIGNMENT;
  }
  dwNewSectSize += sizeof(IMAGE_IMPORT_DESCRIPTOR);
  if((dwNewSectSize % FILE_ALIGNMENT) != 0) {
    dwNewSectSize = ((dwNewSectSize / FILE_ALIGNMENT) + 1) * FILE_ALIGNMENT;
  }
  dwFsize = dwNewSize + dwNewSectSize;
  Addtolist(0,0,"New Import Section Size:%X  New File Size:%X",dwNewSectSize,dwFsize);
  pFileMem = (BYTE*)HeapReAlloc(GetProcessHeap(),HEAP_ZERO_MEMORY,(LPVOID)pFileMem,dwFsize);
  if(!pFileMem) {
    Addtolist(0,1,"OllyDump -- Error  HeapReAlloc Error!!");
    goto CLEAN3;
  }

  Addtolist(0,-1,"OllyDump -- Making New Import Table...");
  MakeIID(pFileMem,dwNewSectSize,&DllEntry);

  // write the file back
  SetFilePointer(hFile,0,NULL,FILE_BEGIN);
  WriteFile(hFile,pFileMem,dwFsize,&dwBuff,0);
  if(dwBuff != 0) {
    Addtolist(0,-1,"OllyDump -- Dump and Rebuild Finish!!");
    result = TRUE;
  }

  // clean up
CLEAN3:
  pDll = DllEntry.next;
  while(pDll != NULL) {
    pApi = pDll->ApiHead.next;
    while(pApi != NULL) {
      qApi = pApi;
      pApi = pApi->next;
      free(qApi);
    }
    qDll = pDll;
    pDll = pDll->next;
    free(qDll);
  }
CLEAN2:
  if(!HeapFree(GetProcessHeap(),0,pFileMem)) {
    Addtolist(0,1,"OllyDump -- Error  HeapFree Error!!");
  }
CLEAN1:
  if(!CloseHandle(hFile)) {
    Addtolist(0,1,"OllyDump -- Error  CloseHandle Error!!");
  }
CLEAN0:
  return(result);
}
