
// __________________________________________________________
//
//                          SbsMem.h
//            SBS Memory Management Library V1.00
//                07-20-2001 Sven B. Schreiber
//                       sbs@orgon.com
// __________________________________________________________

#ifndef _SBSMEM_H_
#define _SBSMEM_H_

// =================================================================
// DISCLAIMER
// =================================================================

/*

This software is provided "as is" and any express or implied
warranties, including, but not limited to, the implied warranties of
merchantibility and fitness for a particular purpose are disclaimed.
In no event shall the author Sven B. Schreiber be liable for any
direct, indirect, incidental, special, exemplary, or consequential
damages (including, but not limited to, procurement of substitute
goods or services; loss of use, data, or profits; or business
interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software,
even if advised of the possibility of such damage.

*/

// =================================================================
// REVISION HISTORY
// =================================================================

/*

07-20-2001 V1.00 Original version (SBS).

*/

// =================================================================
// STRUCTURES
// =================================================================

typedef struct _SBS_MEMORY
    {
    DWORD dTag;
    DWORD dSize;
    BYTE  abData [];
    }
    SBS_MEMORY, *PSBS_MEMORY, **PPSBS_MEMORY;

#define SBS_MEMORY_ sizeof (SBS_MEMORY)

// -----------------------------------------------------------------

#ifndef SBS_MEMORY_TAG
#define SBS_MEMORY_TAG '>SBS' // SBS>
#endif

// =================================================================
// FUNCTION PROTOTYPES
// =================================================================

#ifdef _INLINE_SBSMEM_

#define DLL_EXPORT __declspec (dllexport)

#else   // #ifdef _INLINE_SBSMEM_

#define DLL_EXPORT

#endif  // #ifdef _INLINE_SBSMEM_

// -----------------------------------------------------------------

DLL_EXPORT VOID WINAPI _minit (PWORD pwTableA2W,
                               PBYTE pbTableW2A);

DLL_EXPORT VOID WINAPI _mexit (VOID);

DLL_EXPORT VOID WINAPI _mreset (VOID);

DLL_EXPORT VOID WINAPI _mtrack (DWORD dSize,
                                BOOL  fAdd);

DLL_EXPORT BOOL WINAPI _mstatus (PDWORD pdMemoryNow,
                                 PDWORD pdMemoryMax);

// -----------------------------------------------------------------

DLL_EXPORT PVOID WINAPI _mnew (DWORD dSize);

DLL_EXPORT PVOID WINAPI _mnewz (DWORD dSize);

DLL_EXPORT PVOID WINAPI _mnewx (DWORD dSize,
                                BOOL  fZero,
                                DWORD dTag);

DLL_EXPORT PSBS_MEMORY WINAPI _mbase (PVOID pData);

DLL_EXPORT PSBS_MEMORY WINAPI _mbasex (PVOID pData,
                                       DWORD dTag);

DLL_EXPORT DWORD WINAPI _msize (PVOID pData);

DLL_EXPORT DWORD WINAPI _msizex (PVOID pData,
                                 DWORD dTag);

DLL_EXPORT PVOID WINAPI _mresize (PVOID pData,
                                  DWORD dSize,
                                  PBOOL pfOk);

DLL_EXPORT PVOID WINAPI _mresizex (PVOID pData,
                                   DWORD dSize,
                                   PBOOL pfOk,
                                   DWORD dTag);

DLL_EXPORT PVOID WINAPI _mfree (PVOID pData);

DLL_EXPORT PVOID WINAPI _mfreex (PVOID pData,
                                 DWORD dTag);

// -----------------------------------------------------------------

DLL_EXPORT PBYTE WINAPI _tnewA (DWORD dSize);

DLL_EXPORT PWORD WINAPI _tnewW (DWORD dSize);

DLL_EXPORT PBYTE WINAPI _tresizeA (PBYTE pbData,
                                   DWORD dSize,
                                   PBOOL pfOk);

DLL_EXPORT PWORD WINAPI _tresizeW (PWORD pwData,
                                   DWORD dSize,
                                   PBOOL pfOk);

DLL_EXPORT PBYTE WINAPI _tnewA2A (PBYTE pbFrom,
                                  DWORD dChars);

DLL_EXPORT PWORD WINAPI _tnewA2W (PBYTE pbFrom,
                                  DWORD dChars);

DLL_EXPORT PBYTE WINAPI _tnewW2A (PWORD pwFrom,
                                  DWORD dChars);

DLL_EXPORT PWORD WINAPI _tnewW2W (PWORD pwFrom,
                                  DWORD dChars);

DLL_EXPORT DWORD WINAPI _treadA (PVOID pData,
                                 DWORD dOffset,
                                 DWORD dSize,
                                 PBYTE pbBuffer,
                                 DWORD dBuffer);

DLL_EXPORT DWORD WINAPI _treadW (PVOID pData,
                                 DWORD dOffset,
                                 DWORD dSize,
                                 PWORD pwBuffer,
                                 DWORD dBuffer);

DLL_EXPORT PBYTE WINAPI _tmessageA (DWORD dStatus);

DLL_EXPORT PWORD WINAPI _tmessageW (DWORD dStatus);

// -----------------------------------------------------------------

DLL_EXPORT PVOID WINAPI _mcopy (PVOID pTo,
                                PVOID pFrom,
                                DWORD dBytes);

DLL_EXPORT PVOID WINAPI _mmove (PVOID pTo,
                                PVOID pFrom,
                                DWORD dBytes);

DLL_EXPORT PVOID WINAPI _mswap (PVOID pTo,
                                PVOID pFrom,
                                DWORD dBytes);

DLL_EXPORT PVOID WINAPI _mfill (PVOID pTo,
                                DWORD dBytes,
                                BYTE  bData);

DLL_EXPORT PVOID WINAPI _mzero (PVOID pTo,
                                DWORD dBytes);

DLL_EXPORT DWORD WINAPI _mread (PVOID pData,
                                DWORD dOffset,
                                DWORD dSize,
                                PVOID pBuffer,
                                DWORD dBuffer);

// -----------------------------------------------------------------

DLL_EXPORT BYTE WINAPI _mload8 (PBYTE pbData);

DLL_EXPORT WORD WINAPI _mload16 (PWORD pwData);

DLL_EXPORT DWORD WINAPI _mload32 (PDWORD pdData);

DLL_EXPORT DWORDLONG WINAPI _mload64 (PDWORDLONG pdlData);

DLL_EXPORT BYTE WINAPI _msave8 (PBYTE pbData,
                                BYTE  bValue);

DLL_EXPORT WORD WINAPI _msave16 (PWORD pwData,
                                 WORD  wValue);

DLL_EXPORT DWORD WINAPI _msave32 (PDWORD pdData,
                                  DWORD  dValue);

DLL_EXPORT DWORDLONG WINAPI _msave64 (PDWORDLONG pdlData,
                                      DWORDLONG  dlValue);

DLL_EXPORT BYTE WINAPI _madd8 (PBYTE pbData,
                               BYTE  bValue);

DLL_EXPORT WORD WINAPI _madd16 (PWORD pwData,
                                WORD  wValue);

DLL_EXPORT DWORD WINAPI _madd32 (PDWORD pdData,
                                 DWORD  dValue);

DLL_EXPORT DWORDLONG WINAPI _madd64 (PDWORDLONG pdlData,
                                     DWORDLONG  dlValue);

DLL_EXPORT BYTE WINAPI _msub8 (PBYTE pbData,
                               BYTE  bValue);

DLL_EXPORT WORD WINAPI _msub16 (PWORD pwData,
                                WORD  wValue);

DLL_EXPORT DWORD WINAPI _msub32 (PDWORD pdData,
                                 DWORD  dValue);

DLL_EXPORT DWORDLONG WINAPI _msub64 (PDWORDLONG pdlData,
                                     DWORDLONG  dlValue);

DLL_EXPORT BYTE WINAPI _minc8 (PBYTE pbData);

DLL_EXPORT WORD WINAPI _minc16 (PWORD pwData);

DLL_EXPORT DWORD WINAPI _minc32 (PDWORD pdData);

DLL_EXPORT DWORDLONG WINAPI _minc64 (PDWORDLONG pdlData);

DLL_EXPORT BYTE WINAPI _mdec8 (PBYTE pbData);

DLL_EXPORT WORD WINAPI _mdec16 (PWORD pwData);

DLL_EXPORT DWORD WINAPI _mdec32 (PDWORD pdData);

DLL_EXPORT DWORDLONG WINAPI _mdec64 (PDWORDLONG pdlData);

DLL_EXPORT BOOL WINAPI _mflag (PBOOL pfData);

DLL_EXPORT BOOL WINAPI _mtrue (PBOOL pfData);

DLL_EXPORT BOOL WINAPI _mfalse (PBOOL pfData);

// -----------------------------------------------------------------

#undef DLL_EXPORT

// =================================================================
// CONDITIONAL ANSI/UNICODE SYMBOLS
// =================================================================

#ifdef  UNICODE

#define _tnew     _tnewW
#define _tresize  _tresizeW
#define _tnewT2A  _tnewW2A
#define _tnewT2W  _tnewW2W
#define _tnewA2T  _tnewA2W
#define _tnewW2T  _tnewW2W
#define _tread    _treadW
#define _tmessage _tmessageW

#else   // #ifdef UNICODE

#define _tnew     _tnewA
#define _tresize  _tresizeA
#define _tnewT2A  _tnewA2A
#define _tnewT2W  _tnewA2W
#define _tnewA2T  _tnewA2A
#define _tnewW2T  _tnewW2A
#define _tread    _treadA
#define _tmessage _tmessageA

#endif  // #ifdef UNICODE

////////////////////////////////////////////////////////////////////
#ifdef _INLINE_SBSMEM_
////////////////////////////////////////////////////////////////////

// =================================================================
// VARIABLES
// =================================================================

PWORD _pwTableA2W     = NULL;
PBYTE _pbTableW2A     = NULL;

LONG  _lSbsMemorySign = 0;
DWORD _dSbsMemoryNow  = 0;
DWORD _dSbsMemoryMax  = 0;

CRITICAL_SECTION _csSbsMemory;

// =================================================================
// STARTUP & CLEANUP
// =================================================================

VOID WINAPI _minit (PWORD pwTableA2W,
                    PBYTE pbTableW2A)
    {
    InitializeCriticalSection (&_csSbsMemory);

    _pbTableW2A = pbTableW2A;
    _pwTableA2W = pwTableA2W;
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI _mexit (VOID)
    {
    DeleteCriticalSection (&_csSbsMemory);
    return;
    }

// =================================================================
// MEMORY TRACKING
// =================================================================

VOID WINAPI _mreset (VOID)
    {
    EnterCriticalSection (&_csSbsMemory);

    _lSbsMemorySign = 0;
    _dSbsMemoryNow  = 0;
    _dSbsMemoryMax  = 0;

    LeaveCriticalSection (&_csSbsMemory);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI _mtrack (DWORD dSize,
                     BOOL  fAdd)
    {
    if (dSize)
        {
        EnterCriticalSection (&_csSbsMemory);

        if (fAdd)
            {
            if (_dSbsMemoryNow + dSize < _dSbsMemoryNow)
                {
                _lSbsMemorySign++;
                }
            _dSbsMemoryNow += dSize;

            if (!_lSbsMemorySign)
                {
                _dSbsMemoryMax = max (_dSbsMemoryMax,
                                      _dSbsMemoryNow);
                }
            }
        else
            {
            if (_dSbsMemoryNow < dSize)
                {
                _lSbsMemorySign--;
                }
            _dSbsMemoryNow -= dSize;
            }
        LeaveCriticalSection (&_csSbsMemory);
        }
    return;
    }

// -----------------------------------------------------------------

BOOL WINAPI _mstatus (PDWORD pdMemoryNow,
                      PDWORD pdMemoryMax)
    {
    BOOL fMemorySign = FALSE;

    EnterCriticalSection (&_csSbsMemory);

    fMemorySign = (_lSbsMemorySign < 0);

    if (pdMemoryNow != NULL) *pdMemoryNow = _dSbsMemoryNow;
    if (pdMemoryMax != NULL) *pdMemoryMax = _dSbsMemoryMax;

    LeaveCriticalSection (&_csSbsMemory);
    return fMemorySign;
    }

// =================================================================
// GENERAL MEMORY ALLOCATION
// =================================================================

PVOID WINAPI _mnew (DWORD dSize)
    {
    return _mnewx (dSize, FALSE, SBS_MEMORY_TAG);
    }

// -----------------------------------------------------------------

PVOID WINAPI _mnewz (DWORD dSize)
    {
    return _mnewx (dSize, TRUE, SBS_MEMORY_TAG);
    }

// -----------------------------------------------------------------

PVOID WINAPI _mnewx (DWORD dSize,
                     BOOL  fZero,
                     DWORD dTag)
    {
    DWORD       dFlags = LMEM_FIXED | (fZero ? LMEM_ZEROINIT : 0);
    PSBS_MEMORY psm    = NULL;

    if ((SBS_MEMORY_ + dSize >= SBS_MEMORY_)
        &&
        ((psm = LocalAlloc (dFlags, SBS_MEMORY_ + dSize))
         != NULL))
        {
        psm->dTag  = dTag;
        psm->dSize = dSize;

        _mtrack (psm->dSize, TRUE);
        }
    return (psm != NULL ? psm->abData : NULL);
    }

// -----------------------------------------------------------------

PSBS_MEMORY WINAPI _mbase (PVOID pData)
    {
    return _mbasex (pData, SBS_MEMORY_TAG);
    }

// -----------------------------------------------------------------

PSBS_MEMORY WINAPI _mbasex (PVOID pData,
                            DWORD dTag)
    {
    PSBS_MEMORY psm = (PSBS_MEMORY) ((PBYTE) pData - SBS_MEMORY_);

    if ((            pData           ==             NULL    ) ||
        (            psm             ==             NULL    ) ||
        ((DWORD_PTR) psm             >= (DWORD_PTR) pData   ) ||
        (            psm->dTag       !=             dTag    ) ||
        (            psm->dSize      ==             MAXDWORD))
        {
        psm = NULL;
        }
    return psm;
    }

// -----------------------------------------------------------------

DWORD WINAPI _msize (PVOID pData)
    {
    return _msizex (pData, SBS_MEMORY_TAG);
    }

// -----------------------------------------------------------------

DWORD WINAPI _msizex (PVOID pData,
                      DWORD dTag)
    {
    PSBS_MEMORY psm = _mbasex (pData, dTag);

    return (psm != NULL ? psm->dSize : 0);
    }

// -----------------------------------------------------------------

PVOID WINAPI _mresize (PVOID pData,
                       DWORD dSize,
                       PBOOL pfOk)
    {
    return _mresizex (pData, dSize, pfOk, SBS_MEMORY_TAG);
    }

// -----------------------------------------------------------------

PVOID WINAPI _mresizex (PVOID pData,
                        DWORD dSize,
                        PBOOL pfOk,
                        DWORD dTag)
    {
    PSBS_MEMORY psmNew;
    BOOL        fOk = FALSE;
    PSBS_MEMORY psm = _mbasex (pData, dTag);

    if (psm != NULL)
        {
        fOk = TRUE;

        if (psm->dSize != dSize)
            {
            if ((psmNew = LocalReAlloc (psm, SBS_MEMORY_ + dSize,
                                        LMEM_MOVEABLE))
                != NULL)
                {
                psm = psmNew;

                if (dSize >= psm->dSize)
                    {
                    _mtrack (dSize - psm->dSize, TRUE);
                    }
                else
                    {
                    _mtrack (psm->dSize - dSize, FALSE);
                    }
                psm->dSize = dSize;
                }
            else
                {
                if (pfOk != NULL)
                    {
                    fOk = FALSE;
                    }
                else
                    {
                    _mtrack (psm->dSize, FALSE);

                    psm->dSize = MAXDWORD;
                    LocalFree (psm);

                    psm = NULL;
                    }
                }
            }
        }
    if (pfOk != NULL) *pfOk = fOk;
    return (psm != NULL ? psm->abData : NULL);
    }

// -----------------------------------------------------------------

PVOID WINAPI _mfree (PVOID pData)
    {
    return _mfreex (pData, SBS_MEMORY_TAG);
    }

// -----------------------------------------------------------------

PVOID WINAPI _mfreex (PVOID pData,
                      DWORD dTag)
    {
    PSBS_MEMORY psm;

    if ((psm = _mbasex (pData, dTag)) != NULL)
        {
        _mtrack (psm->dSize, FALSE);

        psm->dSize = MAXDWORD;
        LocalFree (psm);
        }
    return NULL;
    }

// =================================================================
// TEXT MEMORY ALLOCATION
// =================================================================

PBYTE WINAPI _tnewA (DWORD dSize)
    {
    return _mnew (dSize);
    }

// -----------------------------------------------------------------

PWORD WINAPI _tnewW (DWORD dSize)
    {
    return _mnew (dSize * sizeof (WORD));
    }

// -----------------------------------------------------------------

PBYTE WINAPI _tresizeA (PBYTE pbData,
                        DWORD dSize,
                        PBOOL pfOk)
    {
    return _mresize (pbData, dSize, pfOk);
    }

// -----------------------------------------------------------------

PWORD WINAPI _tresizeW (PWORD pwData,
                        DWORD dSize,
                        PBOOL pfOk)
    {
    return _mresize (pwData, dSize * sizeof (WORD), pfOk);
    }

// -----------------------------------------------------------------

PBYTE WINAPI _tnewA2A (PBYTE pbFrom,
                       DWORD dChars)
    {
    PBYTE pbTo = NULL;

    __asm
        {
                    mov     esi, pbFrom
                    mov     ecx, dChars

                    test    esi, esi
                    jz      _return

                    cmp     ecx, -1
                    jnz     _label01

                    xor     eax, eax
                    mov     edi, esi
                    repnz   scasb
                    not     ecx

        _label01:   mov     edi, ecx
                    push    ecx
                    call    _tnewA
                    mov     ecx, edi

                    test    eax, eax
                    jz      _return

                    mov     pbTo, eax
                    mov     edi, eax

                    mov     edx, ecx
                    shr     ecx, 2
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     movsb
        _return:
        }
    return pbTo;
    }

// -----------------------------------------------------------------

PWORD WINAPI _tnewA2W (PBYTE pbFrom,
                       DWORD dChars)
    {
    PWORD pwTo = NULL;

    __asm
        {
                    mov     esi, pbFrom
                    mov     ecx, dChars

                    test    esi, esi
                    jz      _return

                    cmp     ecx, -1
                    jnz     _label01

                    xor     eax, eax
                    mov     edi, esi
                    repnz   scasb
                    not     ecx

        _label01:   mov     edi, ecx
                    push    ecx
                    call    _tnewW
                    mov     ecx, edi

                    test    eax, eax
                    jz      _return

                    mov     pwTo, eax
                    mov     edi, eax

                    test    ecx, ecx
                    jz      _return

                    mov     ebx, _pwTableA2W
                    xor     eax, eax
                    test    ebx, ebx
                    jnz     _label03

        _label02:   lodsb
                    stosw
                    dec     ecx
                    jnz     _label02
                    jmp     _return

        _label03:   xor     ah, ah
                    lodsb
                    mov     ax, [ebx+eax*2]
                    stosw
                    dec     ecx
                    jnz     _label03
        _return:
        }
    return pwTo;
    }

// -----------------------------------------------------------------

PBYTE WINAPI _tnewW2A (PWORD pwFrom,
                       DWORD dChars)
    {
    PBYTE pbTo = NULL;

    __asm
        {
                    mov     esi, pwFrom
                    mov     ecx, dChars

                    test    esi, esi
                    jz      _return

                    cmp     ecx, -1
                    jnz     _label01

                    xor     eax, eax
                    mov     edi, esi
                    repnz   scasw
                    not     ecx

        _label01:   mov     edi, ecx
                    push    ecx
                    call    _tnewA
                    mov     ecx, edi

                    test    eax, eax
                    jz      _return

                    mov     pbTo, eax
                    mov     edi, eax

                    test    ecx, ecx
                    jz      _return

                    mov     ebx, _pbTableW2A
                    xor     eax, eax
                    test    ebx, ebx
                    jnz     _label04

        _label02:   lodsw
                    test    ah, ah
                    jz      _label03
                    mov     al, '?'
        _label03:   stosb
                    dec     ecx
                    jnz     _label02
                    jmp     _return

        _label04:   lodsw
                    mov     al, [ebx+eax]
                    stosb
                    dec     ecx
                    jnz     _label04
        _return:
        }
    return pbTo;
    }

// -----------------------------------------------------------------

PWORD WINAPI _tnewW2W (PWORD pwFrom,
                       DWORD dChars)
    {
    PWORD pwTo = NULL;

    __asm
        {
                    mov     esi, pwFrom
                    mov     ecx, dChars

                    test    esi, esi
                    jz      _return

                    cmp     ecx, -1
                    jnz     _label01

                    xor     eax, eax
                    mov     edi, esi
                    repnz   scasw
                    not     ecx

        _label01:   mov     edi, ecx
                    push    ecx
                    call    _tnewW
                    mov     ecx, edi

                    test    eax, eax
                    jz      _return

                    mov     pwTo, eax
                    mov     edi, eax

                    mov     edx, ecx
                    shr     ecx, 1
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 1
                    rep     movsw
        _return:
        }
    return pwTo;
    }

// -----------------------------------------------------------------

DWORD WINAPI _treadA (PVOID pData,
                      DWORD dOffset,
                      DWORD dSize,
                      PBYTE pbBuffer,
                      DWORD dBuffer)
    {
    DWORD n = 0;

    __asm
        {
                    mov     edi, pbBuffer
                    mov     ecx, dBuffer

                    test    edi, edi
                    jz      _label01
                    test    ecx, ecx
                    jz      _label01
                    mov     byte ptr [edi], 0

        _label01:   mov     esi, pData
                    mov     eax, dSize

                    test    esi, esi
                    jz      _return

                    test    edi, edi
                    jz      _label03
                    test    ecx, ecx
                    jz      _label03

                    dec     ecx
                    add     esi, dOffset

                    cmp     ecx, eax
                    jna     _label02
                    mov     ecx, eax

        _label02:   mov     edx, ecx
                    shr     ecx, 2
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     movsb

                    mov     byte ptr [edi], 0

        _label03:   inc     eax
                    mov     n, eax
        _return:
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI _treadW (PVOID pData,
                      DWORD dOffset,
                      DWORD dSize,
                      PWORD pwBuffer,
                      DWORD dBuffer)
    {
    DWORD n = 0;

    __asm
        {
                    mov     edi, pwBuffer
                    mov     ecx, dBuffer

                    test    edi, edi
                    jz      _label01
                    test    ecx, ecx
                    jz      _label01
                    mov     word ptr [edi], 0

        _label01:   mov     esi, pData
                    mov     eax, dSize

                    test    esi, esi
                    jz      _return

                    test    edi, edi
                    jz      _label03
                    test    ecx, ecx
                    jz      _label03

                    dec     ecx
                    shr     eax, 1
                    add     esi, dOffset

                    cmp     ecx, eax
                    jna     _label02
                    mov     ecx, eax

        _label02:   mov     edx, ecx
                    shr     ecx, 1
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 1
                    rep     movsw

                    mov     word ptr [edi], 0

        _label03:   inc     eax
                    mov     n, eax
        _return:
        }
    return n;
    }

// -----------------------------------------------------------------

PBYTE WINAPI _tmessageA (DWORD dStatus)
    {
    DWORD i, j, n;
    PBYTE pbError1 = NULL;
    PBYTE pbError  = NULL;

    n = FormatMessageA (FORMAT_MESSAGE_ALLOCATE_BUFFER | 
                        FORMAT_MESSAGE_FROM_SYSTEM     | 
                        FORMAT_MESSAGE_IGNORE_INSERTS,
                        NULL, dStatus,
                        MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT),
                        (PBYTE) &pbError1, 0, NULL);

    if (n && (pbError1 != NULL))
        {
        if ((pbError = _tnewA (n+1)) != NULL)
            {
            i = 0;
            while ((i < n) && (pbError1 [i] == ' ')) i++;

            j = n;
            while ((j > i) && ((pbError1 [j-1] == ' ' ) ||
                               (pbError1 [j-1] == '\n'))) j--;
            n = 0;
            while (i < j) pbError [n++] = pbError1 [i++];
            pbError [n] = 0;
            }
        LocalFree (pbError1);
        }
    return pbError;
    }

// -----------------------------------------------------------------

PWORD WINAPI _tmessageW (DWORD dStatus)
    {
    DWORD i, j, n;
    PWORD pwError1 = NULL;
    PWORD pwError  = NULL;

    n = FormatMessageW (FORMAT_MESSAGE_ALLOCATE_BUFFER | 
                        FORMAT_MESSAGE_FROM_SYSTEM     | 
                        FORMAT_MESSAGE_IGNORE_INSERTS,
                        NULL, dStatus,
                        MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT),
                        (PWORD) &pwError1, 0, NULL);

    if (n && (pwError1 != NULL))
        {
        if ((pwError = _tnewW (n+1)) != NULL)
            {
            i = 0;
            while ((i < n) && (pwError1 [i] == ' ')) i++;

            j = n;
            while ((j > i) && ((pwError1 [j-1] == ' ' ) ||
                               (pwError1 [j-1] == '\n'))) j--;
            n = 0;
            while (i < j) pwError [n++] = pwError1 [i++];
            pwError [n] = 0;
            }
        LocalFree (pwError1);
        }
    return pwError;
    }

// =================================================================
// MEMORY MANIPULATION
// =================================================================

PVOID WINAPI _mcopy (PVOID pTo,
                     PVOID pFrom,
                     DWORD dBytes)
    {
    PVOID pNext = pTo;

    __asm
        {
                    mov     edi, pTo
                    mov     esi, pFrom
                    mov     ecx, dBytes

                    test    edi, edi
                    jz      _return
                    test    esi, esi
                    jz      _return
                    test    ecx, ecx
                    jz      _return
                    add     pNext, ecx

                    cmp     esi, edi
                    jz      _return

                    mov     edx, ecx
                    shr     ecx, 2
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     movsb
        _return:
        }
    return pNext;
    }

// -----------------------------------------------------------------

PVOID WINAPI _mmove (PVOID pTo,
                     PVOID pFrom,
                     DWORD dBytes)
    {
    PVOID pNext = pTo;

    __asm
        {
                    mov     edi, pTo
                    mov     esi, pFrom
                    mov     ecx, dBytes

                    test    edi, edi
                    jz      _return
                    test    esi, esi
                    jz      _return
                    test    ecx, ecx
                    jz      _return
                    add     pNext, ecx

                    mov     edx, esi
                    sub     edx, edi
                    jz      _return

                    cmp     edx, 4
                    jnb     _label01

                    rep     movsb               ; copy BYTEs up
                    jmp     _return

        _label01:   cmp     edx, -4
                    jna     _label02

                    lea     esi, [esi+ecx-1]    ; copy BYTEs down
                    lea     edi, [edi+ecx-1]
                    std

                    rep     movsb

                    cld
                    jmp     _return

        _label02:   cmp     edi, esi
                    jb      _label03
                    lea     edx, [esi+ecx]
                    cmp     edi, edx
                    jnb     _label03

                    lea     esi, [esi+ecx-1]    ; copy DWORDs down
                    lea     edi, [edi+ecx-1]
                    std

                    mov     edx, ecx
                    and     ecx, 3
                    rep     movsb

                    sub     esi, 3
                    sub     edi, 3

                    mov     ecx, edx
                    shr     ecx, 2
                    rep     movsd

                    cld
                    jmp     _return

        _label03:   mov     edx, ecx            ; copy DWORDs up
                    shr     ecx, 2
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     movsb
        _return:
        }
    return pNext;
    }

// -----------------------------------------------------------------

PVOID WINAPI _mswap (PVOID pTo,
                     PVOID pFrom,
                     DWORD dBytes)
    {
    PVOID pNext = pTo;

    __asm
        {
                    mov     edi, pTo
                    mov     esi, pFrom
                    mov     ecx, dBytes

                    test    edi, edi
                    jz      _return
                    test    esi, esi
                    jz      _return
                    test    ecx, ecx
                    jz      _return
                    add     pNext, ecx

                    cmp     esi, edi
                    jz      _return

                    mov     ebx, -4
                    mov     edx, ecx
                    and     edx, ebx
                    jz      _label02
                    sub     ecx, edx

        _label01:   mov     eax, [edi]
                    movsd
                    mov     [esi+ebx], eax
                    add     edx, ebx
                    jnz     _label01

        _label02:   test    ecx, ecx
                    jz      _return

        _label03:   mov     al, [edi]
                    movsb
                    mov     [esi-1], al
                    dec     ecx
                    jnz     _label03
        _return:
        }
    return pNext;
    }

// -----------------------------------------------------------------

PVOID WINAPI _mfill (PVOID pTo,
                     DWORD dBytes,
                     BYTE  bData)
    {
    PVOID pNext = pTo;

    __asm
        {
                    mov     edi, pTo
                    mov     ecx, dBytes

                    test    edi, edi
                    jz      _return
                    test    ecx, ecx
                    jz      _return
                    add     pNext, ecx

                    mov     ah, bData
                    mov     al, ah
                    shl     eax, 8
                    mov     al, ah
                    shl     eax, 8
                    mov     al, ah

                    mov     edx, ecx
                    shr     ecx, 2
                    rep     stosd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     stosb
        _return:
        }
    return pNext;
    }

// -----------------------------------------------------------------

PVOID WINAPI _mzero (PVOID pTo,
                     DWORD dBytes)
    {
    PVOID pNext = pTo;

    __asm
        {
                    mov     edi, pTo
                    mov     ecx, dBytes

                    test    edi, edi
                    jz      _return
                    test    ecx, ecx
                    jz      _return
                    add     pNext, ecx

                    xor     eax, eax

                    mov     edx, ecx
                    shr     ecx, 2
                    rep     stosd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     stosb
        _return:
        }
    return pNext;
    }

// -----------------------------------------------------------------

DWORD WINAPI _mread (PVOID pData,
                     DWORD dOffset,
                     DWORD dSize,
                     PVOID pBuffer,
                     DWORD dBuffer)
    {
    DWORD n = 0;

    __asm
        {
                    mov     edi, pBuffer
                    mov     ecx, dBuffer

                    test    edi, edi
                    jz      _return
                    test    ecx, ecx
                    jz      _return

                    mov     esi, pData
                    mov     eax, dSize

                    sub     ecx, eax
                    xchg    ecx, eax
                    jnb     _label01
                    add     ecx, eax
                    xor     eax, eax

        _label01:   mov     n, ecx

                    test    esi, esi
                    jz      _label02
                    add     esi, dOffset

                    mov     edx, ecx
                    shr     ecx, 2
                    rep     movsd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     movsb

        _label02:   add     ecx, eax
                    jz      _return

                    xor     eax, eax

                    mov     edx, ecx
                    shr     ecx, 2
                    rep     stosd

                    mov     ecx, edx
                    and     ecx, 3
                    rep     stosb
        _return:
        }
    return n;
    }

// =================================================================
// ATOMIC MEMORY I/O
// =================================================================

BYTE WINAPI _mload8 (PBYTE pbData)
    {
    BYTE bData = 0;

    __asm
        {
                    mov     edx, pbData
                    test    edx, edx
                    jz      _return

                    mov     al, [edx]
                    mov     bData, al
        _return:
        }
    return bData;
    }

// -----------------------------------------------------------------

WORD WINAPI _mload16 (PWORD pwData)
    {
    WORD wData = 0;

    __asm
        {
                    mov     edx, pwData
                    test    edx, edx
                    jz      _return

                    mov     ax, [edx]
                    mov     wData, ax
        _return:
        }
    return wData;
    }

// -----------------------------------------------------------------

DWORD WINAPI _mload32 (PDWORD pdData)
    {
    DWORD dData = 0;

    __asm
        {
                    mov     edx, pdData
                    test    edx, edx
                    jz      _return

                    mov     eax, [edx]
                    mov     dData, eax
        _return:
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORDLONG WINAPI _mload64 (PDWORDLONG pdlData)
    {
    DWORDLONG dlData;

    __asm
        {
                    xor         eax, eax
                    xor         edx, edx

                    mov         edi, pdlData
                    test        edi, edi
                    jz          _label01

                    mov         ebx, eax
                    mov         ecx, edx

            lock    cmpxchg8b   [edi]

        _label01:   mov         dword ptr dlData,   eax
                    mov         dword ptr dlData+4, edx
        }
    return dlData;
    }

// -----------------------------------------------------------------

BYTE WINAPI _msave8 (PBYTE pbData,
                     BYTE  bValue)
    {
    BYTE bData = 0;

    __asm
        {
                    mov     edx, pbData
                    test    edx, edx
                    jz      _return

                    mov     al, bValue
            lock    xchg    [edx], al
                    mov     bData, al
        _return:
        }
    return bData;
    }

// -----------------------------------------------------------------

WORD WINAPI _msave16 (PWORD pwData,
                      WORD  wValue)
    {
    WORD wData = 0;

    __asm
        {
                    mov     edx, pwData
                    test    edx, edx
                    jz      _return

                    mov     ax, wValue
            lock    xchg    [edx], ax
                    mov     wData, ax
        _return:
        }
    return wData;
    }

// -----------------------------------------------------------------

DWORD WINAPI _msave32 (PDWORD pdData,
                       DWORD  dValue)
    {
    DWORD dData = 0;

    __asm
        {
                    mov     edx, pdData
                    test    edx, edx
                    jz      _return

                    mov     eax, dValue
            lock    xchg    [edx], eax
                    mov     dData, eax
        _return:
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORDLONG WINAPI _msave64 (PDWORDLONG pdlData,
                           DWORDLONG  dlValue)
    {
    DWORDLONG dlData;

    __asm
        {
                    xor         eax, eax
                    xor         edx, edx

                    mov         edi, pdlData
                    test        edi, edi
                    jz          _label01

                    mov         ebx, dword ptr dlValue
                    mov         ecx, dword ptr dlValue+4

            lock    cmpxchg8b   [edi]

        _label01:   mov         dword ptr dlData,   eax
                    mov         dword ptr dlData+4, edx
                    jz          _return

            lock    cmpxchg8b   [edi]

        _return:
        }
    return dlData;
    }

// -----------------------------------------------------------------

BYTE WINAPI _madd8 (PBYTE pbData,
                    BYTE  bValue)
    {
    BYTE bData = 0;

    __asm
        {
                    mov     edx, pbData
                    test    edx, edx
                    jz      _return

                    mov     al, bValue
            lock    xadd    [edx], al
                    add     al, bValue
                    mov     bData, al
        _return:
        }
    return bData;
    }

// -----------------------------------------------------------------

WORD WINAPI _madd16 (PWORD pwData,
                     WORD  wValue)
    {
    WORD wData = 0;

    __asm
        {
                    mov     edx, pwData
                    test    edx, edx
                    jz      _return

                    mov     ax, wValue
            lock    xadd    [edx], ax
                    add     ax, wValue
                    mov     wData, ax
        _return:
        }
    return wData;
    }

// -----------------------------------------------------------------

DWORD WINAPI _madd32 (PDWORD pdData,
                      DWORD  dValue)
    {
    DWORD dData = 0;

    __asm
        {
                    mov     edx, pdData
                    test    edx, edx
                    jz      _return

                    mov     eax, dValue
            lock    xadd    [edx], eax
                    add     eax, dValue
                    mov     dData, eax
        _return:
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORDLONG WINAPI _madd64 (PDWORDLONG pdlData,
                          DWORDLONG  dlValue)
    {
    DWORDLONG dlData;

    __asm
        {
                    xor         eax, eax
                    xor         edx, edx

                    mov         edi, pdlData
                    test        edi, edi
                    jz          _label02

        _label01:   mov         ebx, eax
                    mov         ecx, edx

                    add         ebx, dword ptr dlValue
                    adc         ecx, dword ptr dlValue+4

            lock    cmpxchg8b   [edi]
                    jnz         _label01

                    mov         eax, ebx
                    mov         edx, ecx

        _label02:   mov         dword ptr dlData,   eax
                    mov         dword ptr dlData+4, edx
        }
    return dlData;
    }

// -----------------------------------------------------------------

BYTE WINAPI _msub8 (PBYTE pbData,
                    BYTE  bValue)
    {
    BYTE bData = 0;

    __asm
        {
                    mov     edx, pbData
                    test    edx, edx
                    jz      _return

                    mov     al, bValue
                    neg     al
            lock    xadd    [edx], al
                    sub     al, bValue
                    mov     bData, al
        _return:
        }
    return bData;
    }

// -----------------------------------------------------------------

WORD WINAPI _msub16 (PWORD pwData,
                     WORD  wValue)
    {
    WORD wData = 0;

    __asm
        {
                    mov     edx, pwData
                    test    edx, edx
                    jz      _return

                    mov     ax, wValue
                    neg     ax
            lock    xadd    [edx], ax
                    sub     ax, wValue
                    mov     wData, ax
        _return:
        }
    return wData;
    }

// -----------------------------------------------------------------

DWORD WINAPI _msub32 (PDWORD pdData,
                      DWORD  dValue)
    {
    DWORD dData = 0;

    __asm
        {
                    mov     edx, pdData
                    test    edx, edx
                    jz      _return

                    mov     eax, dValue
                    neg     eax
            lock    xadd    [edx], eax
                    sub     eax, dValue
                    mov     dData, eax
        _return:
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORDLONG WINAPI _msub64 (PDWORDLONG pdlData,
                          DWORDLONG  dlValue)
    {
    DWORDLONG dlData;

    __asm
        {
                    xor         eax, eax
                    xor         edx, edx

                    mov         edi, pdlData
                    test        edi, edi
                    jz          _label02

        _label01:   mov         ebx, eax
                    mov         ecx, edx

                    sub         ebx, dword ptr dlValue
                    sbb         ecx, dword ptr dlValue+4

            lock    cmpxchg8b   [edi]
                    jnz         _label01

                    mov         eax, ebx
                    mov         edx, ecx

        _label02:   mov         dword ptr dlData,   eax
                    mov         dword ptr dlData+4, edx
        }
    return dlData;
    }

// -----------------------------------------------------------------

BYTE WINAPI _minc8 (PBYTE pbData)
    {
    BYTE bData = 0;

    __asm
        {
                    mov     edx, pbData
                    test    edx, edx
                    jz      _return

                    mov     al, 1
            lock    xadd    [edx], al
                    inc     al
                    mov     bData, al
        _return:
        }
    return bData;
    }

// -----------------------------------------------------------------

WORD WINAPI _minc16 (PWORD pwData)
    {
    WORD wData = 0;

    __asm
        {
                    mov     edx, pwData
                    test    edx, edx
                    jz      _return

                    mov     ax, 1
            lock    xadd    [edx], ax
                    inc     ax
                    mov     wData, ax
        _return:
        }
    return wData;
    }

// -----------------------------------------------------------------

DWORD WINAPI _minc32 (PDWORD pdData)
    {
    DWORD dData = 0;

    __asm
        {
                    mov     edx, pdData
                    test    edx, edx
                    jz      _return

                    mov     eax, 1
            lock    xadd    [edx], eax
                    inc     eax
                    mov     dData, eax
        _return:
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORDLONG WINAPI _minc64 (PDWORDLONG pdlData)
    {
    DWORDLONG dlData;

    __asm
        {
                    xor         eax, eax
                    xor         edx, edx

                    mov         edi, pdlData
                    test        edi, edi
                    jz          _label02

        _label01:   mov         ebx, eax
                    mov         ecx, edx

                    add         ebx, 1
                    adc         ecx, 0

            lock    cmpxchg8b   [edi]
                    jnz         _label01

                    mov         eax, ebx
                    mov         edx, ecx

        _label02:   mov         dword ptr dlData,   eax
                    mov         dword ptr dlData+4, edx
        }
    return dlData;
    }

// -----------------------------------------------------------------

BYTE WINAPI _mdec8 (PBYTE pbData)
    {
    BYTE bData = 0;

    __asm
        {
                    mov     edx, pbData
                    test    edx, edx
                    jz      _return

                    mov     al, -1
            lock    xadd    [edx], al
                    dec     al
                    mov     bData, al
        _return:
        }
    return bData;
    }

// -----------------------------------------------------------------

WORD WINAPI _mdec16 (PWORD pwData)
    {
    WORD wData = 0;

    __asm
        {
                    mov     edx, pwData
                    test    edx, edx
                    jz      _return

                    mov     ax, -1
            lock    xadd    [edx], ax
                    dec     ax
                    mov     wData, ax
        _return:
        }
    return wData;
    }

// -----------------------------------------------------------------

DWORD WINAPI _mdec32 (PDWORD pdData)
    {
    DWORD dData = 0;

    __asm
        {
                    mov     edx, pdData
                    test    edx, edx
                    jz      _return

                    mov     eax, -1
            lock    xadd    [edx], eax
                    dec     eax
                    mov     dData, eax
        _return:
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORDLONG WINAPI _mdec64 (PDWORDLONG pdlData)
    {
    DWORDLONG dlData;

    __asm
        {
                    xor         eax, eax
                    xor         edx, edx

                    mov         edi, pdlData
                    test        edi, edi
                    jz          _label02

        _label01:   mov         ebx, eax
                    mov         ecx, edx

                    sub         ebx, 1
                    sbb         ecx, 0

            lock    cmpxchg8b   [edi]
                    jnz         _label01

                    mov         eax, ebx
                    mov         edx, ecx

        _label02:   mov         dword ptr dlData,   eax
                    mov         dword ptr dlData+4, edx
        }
    return dlData;
    }

// -----------------------------------------------------------------

BOOL WINAPI _mflag (PBOOL pfData)
    {
    BOOL fData = FALSE;

    __asm
        {
                    mov     edx, pfData
                    test    edx, edx
                    jz      _return

                    mov     eax, [edx]
                    mov     fData, eax
        _return:
        }
    return fData;
    }

// -----------------------------------------------------------------

BOOL WINAPI _mtrue (PBOOL pfData)
    {
    BOOL fData = FALSE;

    __asm
        {
                    mov     edx, pfData
                    test    edx, edx
                    jz      _return

                    mov     eax, TRUE
            lock    xchg    [edx], eax
                    mov     fData, eax
        _return:
        }
    return fData;
    }

// -----------------------------------------------------------------

BOOL WINAPI _mfalse (PBOOL pfData)
    {
    BOOL fData = FALSE;

    __asm
        {
                    mov     edx, pfData
                    test    edx, edx
                    jz      _return

                    mov     eax, FALSE
            lock    xchg    [edx], eax
                    mov     fData, eax
        _return:
        }
    return fData;
    }

////////////////////////////////////////////////////////////////////
#endif // #ifdef _INLINE_SBSMEM_
////////////////////////////////////////////////////////////////////

#endif // #ifndef _SBSMEM_H_

// =================================================================
// END OF FILE
// =================================================================
