/*
    Process Stalker IDA Plug-in
    Copyright (C) 2005 Pedram Amini <pamini@idefense.com,pedram.amini@gmail.com>

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the Free
    Software Foundation; either version 2 of the License, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along with
    this program; if not, write to the Free Software Foundation, Inc., 59 Temple
    Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*
    Process Stalker IDA Plugin-in (Added gdl interface)
    Modified by AmesianX in powerhacker.net <amesianx@nate.com, amesianx@gmail.com>
*/

#include <windows.h>
#include <stdlib.h>
#include <math.h>

#include <ida.hpp>
#include <idp.hpp>
#include <auto.hpp>
#include <bytes.hpp>
#include <expr.hpp>
#include <fpro.h>
#include <frame.hpp>
#include <kernwin.hpp>
#include <loader.hpp>
#include <nalt.hpp>
#include <name.hpp>
#include <entry.hpp>
#include <typeinf.hpp>

#include <vector>
using std::vector;

#pragma warning (disable:4273)
#pragma warning (disable:4800)

#include "function_analyzer.h"

//--------------------------------------------------------------------------
struct item_t
{
    int index;
    ea_t address;
    string filename;
};

//---------------------------------------------------------------------------
// column widths
static const int widths[] = { CHCOL_DEC|4, CHCOL_PATH|8, CHCOL_HEX|6, 70 };

// column headers
static const char *header[] =
{
    "Index",
    "Start Address",
    "GDL Files"
};

typedef vector<item_t> entrylist_t;

//-------------------------------------------------------------------------
// function that returns number of lines in the list
static ulong idaapi sizer(void *obj)
{
    entrylist_t &li = *(entrylist_t *)obj;
    return li.size();
}

//-------------------------------------------------------------------------
// function that generates the list line
static void idaapi desc(void *obj,ulong n,char * const *arrptr)
{
    if ( n == 0 ) // generate the column headers
    {
        for ( int i=0; i < qnumber(header); i++ )
            qstrncpy(arrptr[i], header[i], MAXSTR);
        return;
    }
    n--;
    entrylist_t &li = *(entrylist_t *)obj;

    qsnprintf(arrptr[0], MAXSTR, "%d", li[n].index);
    qsnprintf(arrptr[1], MAXSTR, "0x%08x", li[n].address);
    qsnprintf(arrptr[2], MAXSTR, "%s", li[n].filename.c_str());
}

//-------------------------------------------------------------------------
// function that is called when the user hits Enter
static void idaapi enter_cb(void *obj,ulong n)
{
    entrylist_t &li = *(entrylist_t *)obj;
    jumpto(li[n-1].address);

    function_analyzer *fa = NULL;

    fa = new function_analyzer((ea_t)li[n-1].address);

    if (fa->is_ready())
    {
        fa->run_analysis();
        fa->graph();
    }

    if(fa)
        delete fa;
}

//-------------------------------------------------------------------------
// function that is called when the window is closed
static void idaapi destroy_cb(void *obj)
{
    entrylist_t *li = (entrylist_t *)obj;
    delete li;
}
//-------------------------------------------------------------------------



/////////////////////////////////////////////////////////////////////////////////////////
// _ida_init()
//
// IDA will call this function only once.
// If this function returns PLUGIN_SKIP, IDA will never load it again.
// If this function returns PLUGIN_OK, IDA will unload the plugin but
// remember that the plugin agreed to work with the database.
// The plugin will be loaded again if the user invokes it by
// pressing the hot key or by selecting it from the menu.
// After the second load, the plugin will stay in memory.
// If this function returns PLUGIN_KEEP, IDA will keep the plugin
// in memory.
//
// arguments: none.
// returns:   plugin status.
//

int _ida_init (void)
{
    // this plug-in only works with metapc (x86) CPU types.
    if(strcmp(inf.procName, "metapc") != 0)
    {
        msg("[!] Detected an incompatible non-metapc CPU type: %s\n", inf.procName);
        return PLUGIN_SKIP;
    }

    msg("[*] pStalker> Process Stalker - Profiler\n"
        "[*] pStalker> Pedram Amini <pedram.amini@gmail.com>\n"
        "[*] pStalker> Modified by AmesianX in powerhacker.net <amesianx@nate.com>\n"
        "[*] pStalker> Compiled on "__DATE__"\n");

    // stay in the memory since most likely we will define our own window proc
    // and add subcontrols to the window
    return ( callui(ui_get_hwnd).vptr != NULL ) ? PLUGIN_OK : PLUGIN_SKIP;
}

////////////////////////////////////////////////////////////////////////////////////////
// _ida_run()
//
// the run function is called when the user activates the plugin. this is the main
// function of the plugin.
//
// arguments: arg - the input argument. it can be specified in the
//                  plugins.cfg file. the default is zero.
// returns:   none.
//

void _ida_run (int arg)
{
    int   num_functions;
    int   function_id      = 0;
    int   base_node_id     = 0;
    float current_percent  = 0.0;
    float percent_complete = 0.0;
    char  *file_path;
    char  *tmp;
    char  path[1024];
    FILE  *bpl;
    function_analyzer *fa;

    const short COLORS   = 0x0001;
    const short COMMENTS = 0x0002;
    const short LOOPS    = 0x0004;
    short options;

    const char dialog_format [] =
        "STARTITEM 0\n"
        "Process Stalker (GUI)\n"
        "Process Stalker is modified\n"
        "by AmesianX in powerhacker.net\n"
        "<Enable Instruction Colors:C>\n"
        "<Enable Comments:C>\n"
        "<Allow Self Loops:C>>\n\n";

    /*
    // determine the imagebase of the loaded module. this snippet is from Ilfak
    // and allows us to get the "true" module image base (PE files only)

    netnode n("$ PE header");
    ea_t imagebase = n.altval(-2);

    msg("imagebase: %08x", imagebase);
    */

    // we can only run the program once the auto-analysis has been completed.
    if (!autoIsOk())
    {
        warning("Can not run Process Stalker until auto-analysis is complete.");
        msg("[!] pStalker> Can not run profiler until auto-analysis is complete.");
        return;
    }

    options = COLORS | COMMENTS;

    if (!AskUsingForm_c(dialog_format, &options))
        return;

    // ask the user for a file path to output breakpoint list and function graphs to.
    // XXX - we actually only want the output directory from the user but IDA doesn't have
    //       an SDK routine for that.
    file_path = askfile_c(1, "*.bpl", "Select a file/directory to store the generated files.");

    if (file_path == NULL)
    {
        msg("[!] pStalker> File name not provided.\n");
        return;
    }

    ////////////////////////////////////////////////////////////
    // gather information about the entry points
    entrylist_t *li = new entrylist_t;
    ////////////////////////////////////////////////////////////

    // strip off the filename from the file path, this is later passed to fa->gml_export().
    if ((tmp = strrchr(file_path, '\\')) != NULL)
        *tmp = '\0';

    char root_filename[MAXSTR];
    get_root_filename(root_filename, MAXSTR);

    // construct the breakpoint list file name as "file path + get_root_filename() + .bpl".
    _snprintf(path, sizeof(path), "%s\\%s.bpl", file_path, root_filename);

    // open/create the outfile.
    bpl = qfopen(path, "w+");

    num_functions = get_func_qty();

    // enumerate all functions.
    for (int i = 0; i < num_functions; i++, function_id++)
    {
        // print percent complete on intervals of 25.
        current_percent = (float)floor((float)i / num_functions * 100);

        if (((int)current_percent % 25 == 0) && current_percent > percent_complete)
        {
            percent_complete = current_percent;
            msg("[*] pStalker> Profile analysis %d%% complete.\n", (int)percent_complete);
        }

        // create a new analyzer object to process this function.
        fa = new function_analyzer(i);

        if (!fa->is_ready())
            continue;

        fa->run_analysis();

        // disabling instruction level multi-colors saves a lot of space in the output .gml graph.
        if (options & COLORS)
            fa->set_gml_ins_color(TRUE);
        else
            fa->set_gml_ins_color(FALSE);

        // enable / disable comments in output graph.
        if (options & COMMENTS)
                fa->set_strip_comments(FALSE);
        else
            fa->set_strip_comments(TRUE);

        // orthogonal view is not possible if self-loops exist within the graph.
        if (options & LOOPS)
            fa->set_gml_ignore_self_loops(FALSE);
        else
            fa->set_gml_ignore_self_loops(TRUE);

        // Added by AmesianX
        ea_t startaddr = 0;
        char retpath[MAXSTR];
        memset(retpath, 0, sizeof(retpath));
    
        // Modified by AmesianX
        // original gml_export function is modified..
        fa->gml_export(retpath, &startaddr, file_path, base_node_id);

        // Added by AmesianX
        item_t x;
        x.index = i;
        x.address = startaddr;
        x.filename = retpath;
        li->push_back(x);


        // increase the base node id by the number of nodes in the current routine.
        base_node_id += fa->get_num_nodes();

        // enumerate the function's basic blocks.
        for (int n = 1; n <= fa->get_num_nodes(); n++)
        {
            // output the breakpoint entry.
            // format: module name:function offset:offset
            qfprintf(bpl, "%s:%08x:%08x\n", root_filename,
                                            fa->first_ea()  - inf.minEA,
                                            fa->get_node(n) - inf.minEA);
        }

        // done with the current analyzer object.
        delete fa;
    }

    // flush / close the outfile.
    qflush(bpl);
    qfclose(bpl);

    msg("[*] pStalker> Profile analysis 100%% complete.\n");

    //--------------------------- GDL FileView Window ---------------------------
    // Added by AmesianX
    // now open the window
    choose2(false,                // non-modal window
            -1, -1, -1, -1,       // position is determined by Windows
            li,                   // pass the created array
            qnumber(header),      // number of columns
            widths,               // widths of columns
            sizer,                // function that returns number of lines
            desc,                 // function that generates a line
            "ProcessStalker Graph Panel", // window title
            -1,                   // use the default icon for the window
            0,                    // position the cursor on the first line
            NULL,                 // "kill" callback
            NULL,                 // "new" callback
            NULL,                 // "update" callback
            NULL,                 // "edit" callback
            enter_cb,             // function to call when the user pressed Enter
            destroy_cb,           // function to call when the window is closed
            NULL,                 // use default popup menu items
            NULL);                // use the same icon for all lines
}


/////////////////////////////////////////////////////////////////////////////////////////
// _ida_term()
//
// IDA will call this function when the user asks to exit. this function will not be
// called in the case of emergency exists. usually this callback is empty.
//
// arguments: none.
// returns:   none.
//

void _ida_term (void)
{
    return;
}


// include the data structures that describe the plugin to IDA.
#include "plugin_info.h"