/*
 * types.c
 * 
 * Copyright (c) 2000, BindView Corporation.
 *
 * See LICENSE file.
 *
 */


#include "syscalls.h"
#include "types.h"
#include <ntddk.h>
#include "ioctlcmd.h"

#define min(a,b) (((a)<(b))?(a):(b))

#define MAX_BUF_SIZE 500

static unsigned long copy_buff_safe (void *dest, void *src, size_t sz);

void
copy_arg_four_bytes (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;

    ent->args[ent->args_size/4] = args[arg_num];
    ent->args_size += 4;
}


void
copy_arg_buff_4 (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned long val;

    rc = copy_buff_safe (&val, (void *)args[arg_num], 4);
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
    } else {
        ent->args[ent->args_size/4] = val;
    }
    ent->args_size += 4;
}

void
copy_arg_buff_8 (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char buff[8];

    rc = copy_buff_safe (&buff, (void *)args[arg_num], sizeof (buff));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &buff, sizeof (buff));
        ent->args_size += sizeof (buff);
    }
}

void
copy_arg_buff_12 (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char buff[12];

    rc = copy_buff_safe (&buff, (void *)args[arg_num], sizeof (buff));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &buff, sizeof (buff));
        ent->args_size += sizeof (buff);
    }
}


void
copy_arg_buff_20 (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char buff[20];

    rc = copy_buff_safe (&buff, (void *)args[arg_num], sizeof (buff));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &buff, sizeof (buff));
        ent->args_size += sizeof (buff);
    }
}

void
copy_arg_buff_28 (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char buff[28];

    rc = copy_buff_safe (&buff, (void *)args[arg_num], sizeof (buff));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &buff, sizeof (buff));
        ent->args_size += sizeof (buff);
    }
}


void
copy_arg_POBJ_ATTRIB (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned char *buff;
    unsigned long rc;
    DWORD need;
    OBJECT_ATTRIBUTES attr;
    UNICODE_STRING str;
    POBJECT_ATTRIBUTES pObj = (POBJECT_ATTRIBUTES) args[arg_num];

    rc = copy_buff_safe (&attr, (void *)args[arg_num], sizeof (attr));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &attr, sizeof (attr));
        ent->args_size += sizeof (attr);

        rc = copy_buff_safe (&str, (void *)attr.ObjectName, sizeof (*attr.ObjectName));
        ent->args[ent->args_size/4] = rc;
        ent->args_size += 4;
        if (rc == -1) {
            ent->args[ent->args_size/4] = (unsigned long)attr.ObjectName;
            ent->args_size += 4;
        } else {
            memcpy (&ent->args[ent->args_size/4], &str, sizeof (str));
            ent->args_size += sizeof (str);

            rc = copy_buff_safe (&ent->args[ent->args_size/4+1],
                                 str.Buffer, str.Length);
            ent->args[ent->args_size/4] = rc;
            ent->args_size += 4;
            if (rc == -1) {
                ent->args[ent->args_size/4] = (unsigned long)str.Buffer;
                ent->args_size += 4;
            } else {
                ent->args_size += ((str.Length + 3) / 4) * 4;
            }
        }
    }
}


void
copy_arg_PUNICODE_STRING (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned char *buff;
    unsigned long rc;
    DWORD need;
    UNICODE_STRING str;

    rc = copy_buff_safe (&str, (void *)args[arg_num], sizeof (str));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &str, sizeof (str));
        ent->args_size += sizeof (str);

        rc = copy_buff_safe (&ent->args[ent->args_size/4+1],
                             str.Buffer, str.Length);
        ent->args[ent->args_size/4] = rc;
        ent->args_size += 4;
        if (rc == -1) {
            ent->args[ent->args_size/4] = (unsigned long)str.Buffer;
            ent->args_size += 4;
        } else {
            ent->args_size += ((str.Length + 3) / 4) * 4;
        }
    }
}

typedef struct _LARGE_UNI_STRING {
    DWORD Length;
    DWORD MaxLength;
    unsigned short *Buffer;
} LARGE_UNICODE_STRING;

void
copy_arg_PLARGE_UNICODE_STRING (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned char *buff;
    unsigned long rc;
    DWORD need;
    LARGE_UNICODE_STRING str;

    rc = copy_buff_safe (&str, (void *)args[arg_num], sizeof (str));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &str, sizeof (str));
        ent->args_size += sizeof (str);
        rc = copy_buff_safe (&ent->args[ent->args_size/4+1],
                             str.Buffer, str.Length);
        ent->args[ent->args_size/4] = rc;
        ent->args_size += 4;
        if (rc == -1) {
            ent->args[ent->args_size/4] = (unsigned long)str.Buffer;
            ent->args_size += 4;
        } else {
            ent->args_size += ((str.Length + 3) / 4) * 4;
        }
    }
}


void
copy_arg_PLPC_MESSAGE (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned short msg[12];
    unsigned long copy_size;

    rc = copy_buff_safe (&msg, (void *)args[arg_num], sizeof (msg));
    if (rc == -1) {
        ent->args[ent->args_size/4] = rc;
        ent->args_size += 4;
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        copy_size = msg[0];
#if 0
        /* certain users of LPC ports don't follow this */
        if ((copy_size + sizeof (msg)) != msg[1]) {
            /* something is wrong with the message, don't copy more */
            copy_size = 0;
        }
#endif
        if (copy_size + sizeof (msg) > MAX_BUF_SIZE) {
            copy_size = MAX_BUF_SIZE - sizeof (msg);
        }
        rc = copy_buff_safe (&ent->args[ent->args_size/4+2+sizeof(msg)/4],
                             ((char *)args[arg_num]) + sizeof (msg),
                             copy_size);
        ent->args[ent->args_size/4] = rc;
        ent->args_size += 4;
        if (rc == -1) {
            ent->args[ent->args_size/4] = args[arg_num];
            ent->args_size += 4;
        } else {
            ent->args[ent->args_size/4] = copy_size;
            ent->args_size += 4;
            memcpy (&ent->args[ent->args_size/4], &msg, sizeof (msg));
            ent->args_size += sizeof (msg) + copy_size;
        }
    }
}


void
copy_arg_PLPC_SECTIONINFO (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char msg[24];

    rc = copy_buff_safe (&msg, (void *)args[arg_num], sizeof (msg));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &msg, sizeof (msg));
        ent->args_size += sizeof (msg);
    }
}


void
copy_arg_PLPC_SECTIONMAPINFO (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char msg[12]; /* actually may be bigger, but ignore that now */

    rc = copy_buff_safe (&msg, (void *)args[arg_num], sizeof (msg));
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        memcpy (&ent->args[ent->args_size/4], &msg, sizeof (msg));
        ent->args_size += sizeof (msg);
    }
}

void
copy_arg_info_buff_x (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char use_arg;
    unsigned long use_rc;
    unsigned long buf_size = 0;
    unsigned long use_size = 0;
    unsigned long copy_size = 0;

    /* get buf size */
    buf_size = args[all_arg_info[ent->call_num]->args[arg_num].buf_len_arg];

    /* get use size, if present */
    use_arg = all_arg_info[ent->call_num]->args[arg_num].use_len_arg;
    if (use_arg != (unsigned char) -1) {
        use_rc = copy_buff_safe (&use_size,
                                 (void *)args[use_arg],
                                 sizeof (use_size));
    } else {
        use_rc = -1;
    }
    /* find min of (buf size, use size, max_buf_to_copy) */
    copy_size = min (buf_size, MAX_BUF_SIZE);
    if (use_rc != -1) {
        copy_size = min (copy_size, use_size);
    }

    /* put in info class, amount to copy */
    ent->args[ent->args_size/4+1] = args[all_arg_info[ent->call_num]->args[arg_num].info_arg];
    ent->args[ent->args_size/4+2] = copy_size;

    /* try to copy amount */
    rc = copy_buff_safe (&ent->args[ent->args_size/4+3],
                         (void *)args[arg_num], copy_size);
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        /* if failed, set pointer in data */
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        /* otherwise, inc args_size (rounding up) */
        ent->args_size += 8; /* for info class and copy_size */
        ent->args_size += ((copy_size + 3) / 4) * 4;
    }
}

copy_arg_PVOID_WITH_IO_BUFF (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char use_arg;
    unsigned long use_rc;
    unsigned long buf_size = 0;
    unsigned long use_size = 0;
    unsigned long copy_size = 0;

    /* get buf size */
    buf_size = args[all_arg_info[ent->call_num]->args[arg_num].buf_len_arg];

    /* get use size, if present */
    use_arg = all_arg_info[ent->call_num]->args[arg_num].use_len_arg;
    if (use_arg != (unsigned char) -1) {
        IO_STATUS_BLOCK io;
        use_rc = copy_buff_safe (&io,
                                 (void *)args[use_arg],
                                 sizeof (io));
        use_size = io.Information;
    } else {
        use_rc = -1;
    }
    /* find min of (buf size, use size, max_buf_to_copy) */
    copy_size = min (buf_size, MAX_BUF_SIZE);
    if (use_rc != -1) {
        copy_size = min (copy_size, use_size);
    }

    /* put in amount to copy */
    ent->args[ent->args_size/4+1] = copy_size;

    /* try to copy amount */
    rc = copy_buff_safe (&ent->args[ent->args_size/4+2],
                         (void *)args[arg_num], copy_size);
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        /* if failed, set pointer in data */
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        /* otherwise, inc args_size (rounding up) */
        ent->args_size += 4; /* for copy_size */
        ent->args_size += ((copy_size + 3) / 4) * 4;
    }
}


copy_arg_PVOID_BUFF (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char use_arg;
    unsigned long use_rc;
    unsigned long buf_size = 0;
    unsigned long use_size = 0;
    unsigned long copy_size = 0;

    /* get buf size */
    buf_size = args[all_arg_info[ent->call_num]->args[arg_num].buf_len_arg];

    /* get use size, if present */
    use_arg = all_arg_info[ent->call_num]->args[arg_num].use_len_arg;
    if (use_arg != (unsigned char) -1) {
        use_rc = copy_buff_safe (&use_size,
                                 (void *)args[use_arg],
                                 sizeof (use_size));
    } else {
        use_rc = -1;
    }
    /* find min of (buf size, use size, max_buf_to_copy) */
    copy_size = min (buf_size, MAX_BUF_SIZE);
    if (use_rc != -1) {
        copy_size = min (copy_size, use_size);
    }

    /* put in amount to copy */
    ent->args[ent->args_size/4+1] = copy_size;

    /* try to copy amount */
    rc = copy_buff_safe (&ent->args[ent->args_size/4+2],
                         (void *)args[arg_num], copy_size);
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        /* if failed, set pointer in data */
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        /* otherwise, inc args_size (rounding up) */
        ent->args_size += 4; /* for copy_size */
        ent->args_size += ((copy_size + 3) / 4) * 4;
    }
}


copy_arg_PWINDOW_TEXT (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char use_arg;
    unsigned long use_rc;
    unsigned long buf_size = 0;
    unsigned long use_size = 0;
    unsigned long copy_size = 0;

    /* get buf size */
    buf_size = args[all_arg_info[ent->call_num]->args[arg_num].buf_len_arg];

    /* get use size, if present */
    use_arg = all_arg_info[ent->call_num]->args[arg_num].use_len_arg;
    if (use_arg == (unsigned char) -2) {
        use_size = ent->result;
        if ((signed) use_size >= 0) {
            use_size *= 2;
            use_rc = 0;
        } else {
            use_rc = -1;
        }
    } else if (use_arg != (unsigned char) -1) {
        use_rc = copy_buff_safe (&use_size,
                                 (void *)args[use_arg],
                                 sizeof (use_size));
    } else {
        use_rc = -1;
    }
    /* find min of (buf size, use size, max_buf_to_copy) */
    copy_size = min (buf_size, MAX_BUF_SIZE);
    if (use_rc != -1) {
        copy_size = min (copy_size, use_size);
    }

    /* put in amount to copy */
    ent->args[ent->args_size/4+1] = copy_size;

    /* try to copy amount */
    rc = copy_buff_safe (&ent->args[ent->args_size/4+2],
                         (void *)args[arg_num], copy_size);
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        /* if failed, set pointer in data */
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        /* otherwise, inc args_size (rounding up) */
        ent->args_size += 4; /* for copy_size */
        ent->args_size += ((copy_size + 3) / 4) * 4;
    }
}


copy_arg_PDWORD_ARRAY (void *e, DWORD *args, DWORD arg_num)
{
    ENTRY *ent = (ENTRY *)e;
    unsigned long rc;
    unsigned char use_arg;
    unsigned long use_rc;
    unsigned long buf_size = 0;
    unsigned long use_size = 0;
    unsigned long copy_size = 0;

    /* get buf size */
    buf_size = args[all_arg_info[ent->call_num]->args[arg_num].buf_len_arg];
    /* 4 bytes each */
    buf_size *= 4;

    /* get use size, if present */
    use_arg = all_arg_info[ent->call_num]->args[arg_num].use_len_arg;
    if (use_arg != (unsigned char) -1) {
        use_rc = copy_buff_safe (&use_size,
                                 (void *)args[use_arg],
                                 sizeof (use_size));
        use_size *= 4;
    } else {
        use_rc = -1;
    }
    /* find min of (buf size, use size, max_buf_to_copy) */
    copy_size = min (buf_size, MAX_BUF_SIZE);
    if (use_rc != -1) {
        copy_size = min (copy_size, use_size);
    }

    /* put in amount to copy */
    ent->args[ent->args_size/4+1] = copy_size;

    /* try to copy amount */
    rc = copy_buff_safe (&ent->args[ent->args_size/4+2],
                         (void *)args[arg_num], copy_size);
    ent->args[ent->args_size/4] = rc;
    ent->args_size += 4;
    if (rc == -1) {
        /* if failed, set pointer in data */
        ent->args[ent->args_size/4] = args[arg_num];
        ent->args_size += 4;
    } else {
        /* otherwise, inc args_size */
        ent->args_size += 4 + copy_size;
    }
}


static unsigned long probe_addr = 0x7ffef000;

static unsigned long
copy_buff_safe (void *dest, void *src, size_t sz)
{
    unsigned long rc = 0;

    /*
     * Technically, this == 0 check isn't necessary.  The __try block
     * below will catch such cases, and recover fine.  Problem is that
     * when the exception handler is invoked, it ends up calling
     * NtContinue, which is then seen by our hook and then user space.
     * So, to avoid confusing the user, try to avoid that.
     *
     * Note that that may still happen if the pointer passed is non-null
     * but still invalid.  Nothing we can do about that.  The null case
     * is very common though, and was quite distracting.  Took me a
     * while to figure out wtf was going on.  :)
     */
    if (src == 0)
        return -1;

    /*
     * FIXME: only do the following check if the call originates from
     * user-space.  Some kernel-space calls are failing this check,
     * and we lose their information, unnecessarily.
     */

    /* test userprobeaddress */
    if ((ExGetPreviousMode () == 1)
        && (((unsigned long) src > probe_addr)
            || ((unsigned long) src + sz > probe_addr))) {
        return -1;
    }

    __try {
        memcpy (dest, src, sz);
    } __except (EXCEPTION_EXECUTE_HANDLER) {
        rc = -1;
    }
    return rc;
}

/*
 * map the copy functions to the types that need them.
 * used to have one per type, but that makes for lots of code duplication
 * Obviously, this _must_ be in the same order as typesx.h
 */
struct type_info all_types[] = {
    { copy_arg_four_bytes }, /* DWORD */
    { copy_arg_four_bytes }, /* DWORDx */
    { copy_arg_buff_4 }, /* PDWORD */
    { copy_arg_buff_4 }, /* PDWORDx */
    { copy_arg_four_bytes }, /* HANDLE */
    { copy_arg_buff_4 }, /* PHANDLE */
    { copy_arg_POBJ_ATTRIB }, /* POBJ_ATTRIB */
    { copy_arg_PUNICODE_STRING }, /* PUNICODE_STRING */
    { copy_arg_four_bytes }, /* ACCESS_MASK */
    { copy_arg_buff_8 }, /* PCLIENT_ID */
    { copy_arg_PLPC_MESSAGE }, /* PLPC_MESSAGE */
    { copy_arg_PLPC_SECTIONINFO }, /* PLPC_SECTIONINFO */
    { copy_arg_PLPC_SECTIONMAPINFO }, /* PLPC_SECTIONMAPINFO */
    { copy_arg_four_bytes }, /* SYSTEM_INFO_CLASS */
    { copy_arg_four_bytes }, /* OBJECT_INFO_CLASS */
    { copy_arg_four_bytes }, /* MEMORY_INFO_CLASS */
    { copy_arg_four_bytes }, /* SECTION_INFO_CLASS */
    { copy_arg_four_bytes }, /* THREAD_INFO_CLASS */
    { copy_arg_four_bytes }, /* PROCESS_INFO_CLASS */
    { copy_arg_four_bytes }, /* JOB_INFO_CLASS */
    { copy_arg_four_bytes }, /* TOKEN_INFO_CLASS */
    { copy_arg_four_bytes }, /* FS_INFO_CLASS */
    { copy_arg_four_bytes }, /* FILE_INFO_CLASS */
    { copy_arg_four_bytes }, /* KEY_INFO_CLASS */
    { copy_arg_four_bytes }, /* KEY_VALUE_INFO_CLASS */
    { copy_arg_buff_8 }, /* PIO_STATUS_BLOCK */
    { copy_arg_buff_8 }, /* PLARGE_INTEGER */
    { copy_arg_buff_12 }, /* PSECURITY_QOS */
    { copy_arg_buff_8 }, /* PLUID */
    { copy_arg_info_buff_x }, /* PSYSTEM_INFO_BUFF */
    { copy_arg_buff_20 }, /* PSECURITY_DESCRIPTOR */
    { copy_arg_info_buff_x }, /* POBJECT_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PMEMORY_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PSECTION_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PTHREAD_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PPROCESS_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PJOB_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PTOKEN_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PTIMER_INFO_BUFF */
    { copy_arg_four_bytes }, /* TIMER_INFO_CLASS */
    { copy_arg_four_bytes }, /* EVENT_INFO_CLASS */
    { copy_arg_info_buff_x }, /* PEVENT_INFO_BUFF */
    { copy_arg_four_bytes }, /* SEMAPHORE_INFO_CLASS */
    { copy_arg_info_buff_x }, /* PSEMAPHORE_INFO_BUFF */
    { copy_arg_four_bytes }, /* MUTANT_INFO_CLASS */
    { copy_arg_info_buff_x }, /* PMUTANT_INFO_BUFF */
    { copy_arg_PVOID_WITH_IO_BUFF }, /* PVOID_WITH_IO_BUFF */
    { copy_arg_PVOID_BUFF }, /* PVOID_BUFF */
    { copy_arg_info_buff_x }, /* PKEY_INFO_BUFF */
    { copy_arg_info_buff_x }, /* PKEY_VALUE_INFO_BUFF */
    { copy_arg_PLARGE_UNICODE_STRING }, /* PLARGE_UNICODE_STRING */
    { copy_arg_buff_28 }, /* PMSG */
    { copy_arg_four_bytes }, /* HWND */
    { copy_arg_PWINDOW_TEXT }, /* PWINDOW_TEXT */
    { copy_arg_four_bytes }, /* WINMSG */
    { copy_arg_four_bytes }, /* WPARAM */
    { copy_arg_four_bytes }, /* LPARAM */
    { copy_arg_PDWORD_ARRAY }, /* PHWND_LIST */
    { copy_arg_PDWORD_ARRAY }, /* PHANDLES */
};

