using System;
using System.Threading;

namespace vu.ch.argee.BaseClasses
{
	/// <summary>
	/// Provides the abstract (MustInherit in Visual Basic) base class for the implementation of the IDisposible interface
	/// </summary>
	/// <remarks>
	/// <strong>Notes to Implementers:</strong> This base class is provided to make it easier for implementers to create an object that implements the IDisposible interface.
	/// </remarks>
	/// <exception cref="ObjectDisposedException">The exception that is thrown when an operation is performed on a disposed object.</exception>
	public abstract class BaseDisposableObject: IDisposable
	{
		#region ------------ fields ------------

		private bool isDisposed = false;

		#endregion

		#region ------------ destructor ------------

		/// <summary>
		/// Releases unmanaged resources and performs other cleanup operations before the object is reclaimed by garbage collection.
		/// </summary>
		/// <remarks>
		/// Calls <see cref="DisposeUnmanagedResources"/> to free unmanaged resources and <see cref="DisposeManagedResources"/> to free managed resources.<br/>
		/// This method overrides <see cref="Object.Finalize"/>. Application code should not call this method.<br/>
		/// In derrived classes override  <see cref="DisposeUnmanagedResources"/> and <see cref="DisposeManagedResources"/> to free resources.<br/>
		/// <br/>
		/// In C#, finalizers are expressed using destructor syntax.<br/>
		/// ~BaseDisposableObject();
		/// </remarks>
		~BaseDisposableObject()      
		{
			isDisposed = true;
			DisposeUnmanagedResources();
		}

		#endregion

		#region ------------ protected methods ------------

		/// <summary>
		/// When overridden in a derived class, disposes managed resources.
		/// </summary>
		protected abstract void DisposeManagedResources();

		/// <summary>
		/// When overridden in a derived class, disposes unmanaged resources.
		/// </summary>
		protected abstract void DisposeUnmanagedResources();

		/// <summary>
		/// Determines if the object is already disposed.
		/// </summary>
		/// <value><b>true</b> if the object is already disposed.</value>
		protected bool IsDisposed
		{
			get
			{
				return isDisposed;
			}
		}

		#endregion

		#region ------------ public methods ------------

		/// <summary>
		/// Releases all resources used by this object. The destructor will not be called after calling Dispose.
		/// </summary>
		/// <remarks>If the object is already disposed, a call to Dispose throws no exceptions.<br/></remarks>
		public void Dispose()
		{
			if (Monitor.TryEnter(this))  // Only the first thread needs to enter
			{
				try
				{
					if(!this.isDisposed)
					{
						isDisposed = true;
						DisposeManagedResources();
						DisposeUnmanagedResources();
						// Tell the GC to not call the destructor anymore
						GC.SuppressFinalize(this);
					}
				}
				finally
				{
					Monitor.Exit(this);
				}
			}
		}

		/// <summary>
		/// Tests if the object is disposed. If yes, it throws an <see cref="ObjectDisposedException"/> exception.
		/// </summary>
		/// <remarks>
		/// Use this method in derrived classes to check if the access of a method or property is allowed.
		/// </remarks>
		/// <exception cref="ObjectDisposedException">The exception that is thrown when an operation is performed on a disposed object.</exception>
		public void CheckDisposed()
		{
			if(this.isDisposed)
			{
				throw new ObjectDisposedException(this.GetType().FullName);
			}
		}

		#endregion
	}
}
