using System;
using System.IO;
using System.Diagnostics;
using System.ComponentModel;
using vu.ch.argee.BaseClasses;
using vu.ch.argee.WindowsAPI.WrapperClasses;

namespace vu.ch.argee.MemoryMapping
{
	/// <summary>
	/// Maps a specified file or a part of the paging file into a virtual memory address range.
	/// </summary>
	public class FileMapping: BaseDisposableObject 
	{
		#region ------------ fields ------------

		private IntPtr handle;
		private ulong maximumSize;
		private MappingAccess access;

		#endregion

		#region ------------ static methods ------------

		/// <summary>
		/// Creates a new instance of the FileMapping class.
		/// The system object created is named and can be used
		/// to open the FileMapping from a different process.
		/// </summary>
		/// <param name="fileName">The name of the file to map in memory.</param>
		/// <param name="mode">The FileMode used to open the underlying file.</param>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="share">The FileShare used to open the file.</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <remarks>
		/// If you create a file mapping object of 500 Kb, you have access only to the first 500 Kb of the file, 
		/// regardless of the size of the file. Since it costs you no system resources to create a larger file mapping object, 
		/// create one the size of the file even if you do not expect to view the entire file. 
		/// The cost in system resources comes in creating the views and accessing them.
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping CreateNamed(string fileName, FileMode mode, MappingAccess access, FileShare share, ulong maximumSize, string name)
		{
			FileMapping result;
			IntPtr fileHandle;

			fileHandle = ManagedFileMappingAdaptor.CreateFile(fileName, mode, MappingAccessConverter.ToFileAccess(access), share);
			try
			{
				if (maximumSize == 0)
				{
					maximumSize = (ulong)new FileInfo(fileName).Length;
				}
				result = new FileMapping(ManagedFileMappingAdaptor.CreateFileMapping(fileHandle, MappingAccessConverter.ToPageProtection(access), maximumSize, name), access, maximumSize);
			}
			finally
			{
				ManagedFileMappingAdaptor.CloseHandle(fileHandle);
			}

			return result;
		}

		/// <summary>
		/// creates a file mapping object of the specified size backed by the operating-system paging file rather than by a named file in the file system
		/// </summary>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="maximumSize">The maximum size that will be accessed.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <returns>The new instance of the FileMapping class.</returns>
		/// <remarks>
		/// If you create a file mapping object of 500 Kb, you have access only to the first 500 Kb of the file, 
		/// regardless of the size of the file. Since it costs you no system resources to create a larger file mapping object, 
		/// create one the size of the file even if you do not expect to view the entire file. 
		/// The cost in system resources comes in creating the views and accessing them.
		/// </remarks>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping CreateNamed(MappingAccess access, ulong maximumSize, string name)
		{
			return new FileMapping(ManagedFileMappingAdaptor.CreatePagingFileMapping(MappingAccessConverter.ToPageProtection(access), maximumSize, name), access, maximumSize);
		}

		/// <summary>
		/// Creates a new instance of the FileMapping class with the specified path, maximumsize and readwrite access.
		/// The file is opened for exclusive access. If the file doesn't exist, a new file with the specified fileName is created.
		/// </summary>
		/// <param name="fileName">The name of the file to map in memory.</param>
		/// <param name="maximumSize">The maximum size of the file. This value must include the amount of bytes you want to append to the file. A value of 0 (zero) sets the maximumSize to the current file size. Valid values are either zero, equal or bigger than the file size.</param>
		/// <returns>The new instance of the FileMapping class.</returns>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping CreateReadWrite(string fileName, ulong maximumSize)
		{
			return Create(fileName, FileMode.OpenOrCreate, MappingAccess.ReadWrite, FileShare.None, maximumSize);
		}

		/// <summary>
		/// Creates a new instance of the FileMapping class.
		/// The system object created is not named.
		/// </summary>
		/// <param name="fileName">The name of the file to map in memory.</param>
		/// <param name="mode">The FileMode used to open the underlying file.</param>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="share">The FileShare used to open the file.</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <returns>The new instance of the FileMapping class.</returns>
		/// <remarks>
		/// If you create a file mapping object of 500 Kb, you have access only to the first 500 Kb of the file, 
		/// regardless of the size of the file. Since it costs you no system resources to create a larger file mapping object, 
		/// create one the size of the file even if you do not expect to view the entire file. 
		/// The cost in system resources comes in creating the views and accessing them.
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping Create(string fileName, FileMode mode, MappingAccess access, FileShare share, ulong maximumSize)
		{
			return CreateNamed(fileName, mode, access, share, maximumSize, null);
		}

		/// <summary>
		/// Creates a new instance of the FileMapping class with the specified path and readonly access
		/// </summary>
		/// <param name="fileName">The name of the file to map in memory.</param>
		/// <returns>The new instance of the FileMapping class.</returns>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping CreateReadOnly(string fileName)
		{
			return Create(fileName, FileMode.Open, MappingAccess.ReadOnly, FileShare.Read, 0);
		}

		/// <summary>
		/// creates a file mapping object of the specified size backed by the operating-system paging file rather than by a named file in the file system
		/// </summary>
		/// <param name="access">The mapping access</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <returns>The new instance of the FileMapping class.</returns>
		/// <remarks>
		/// If you create a file mapping object of 500 Kb, you have access only to the first 500 Kb of the file, 
		/// regardless of the size of the file. Since it costs you no system resources to create a larger file mapping object, 
		/// create one the size of the file even if you do not expect to view the entire file. 
		/// The cost in system resources comes in creating the views and accessing them.
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping CreateUsingPagingFile(MappingAccess access, ulong maximumSize)
		{
			return new FileMapping(ManagedFileMappingAdaptor.CreatePagingFileMapping(MappingAccessConverter.ToPageProtection(access), maximumSize, null), access, maximumSize);
		}

		/// <summary>
		/// Creates a new instance of the FileMapping class. 
		/// This constructor openes an existing FileMapping (e.g. created by a different process) by name. 
		/// </summary>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <returns>The new instance of the FileMapping class.</returns>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public static FileMapping CreateFromSystemObjectName(MappingAccess access, string name)
		{
			return new FileMapping(ManagedFileMappingAdaptor.OpenFileMapping(MappingAccessConverter.ToFileMapAccess(access), false, name), access, 0);
		}

		/// <summary>
		/// Creates a new instance of the FileMapping class. 
		/// This constructor openes an existing FileMapping by a file mapping handle of a foreign process.
		/// </summary>
		/// <param name="foreignProcessHandle">the handle of the foreign process</param>
		/// <param name="foreignFileMappingHandle">the foreign file mapping handle</param>
		/// <param name="inheritHandle">indicates if the handle will be inherited to child processes</param>
		/// <returns>the new instance of the FileMapping class.</returns>
		/// <remarks>This method is usefull if the foreign process did not duplicate the handle.</remarks>
		public static FileMapping CreateFromForeignHandle(IntPtr foreignProcessHandle, IntPtr foreignFileMappingHandle, bool inheritHandle)
		{
			return new FileMapping(ManagedFileMappingAdaptor.DuplicateHandle(foreignProcessHandle, foreignFileMappingHandle, Process.GetCurrentProcess().Handle, inheritHandle), 0, 0);
		}

		/// <summary>
		/// Creates a new instance of the FileMapping class. 
		/// This constructor creates a FileMapping wrapper object for an owned file mapping handle.
		/// </summary>
		/// <param name="ownedFileMappingHandle">the file mapping handle</param>
		/// <returns>the new instance of the FileMapping class.</returns>
		/// <remarks>You can use this method if the handles was created by API functions or if the handle was duplicated by a foreign process.</remarks>
		public static FileMapping CreateFromOwnedHandle(IntPtr ownedFileMappingHandle)
		{
			return new FileMapping(ownedFileMappingHandle, 0, 0);
		}
		#endregion

		#region ------------ constructors ------------

		/// <summary>
		/// Creates a wrapper object around the Storage API object.
		/// </summary>
		/// <param name="handle">The handle of the system FileMapping object.</param>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		protected FileMapping(IntPtr handle, MappingAccess access, ulong maximumSize)
		{
			this.handle = handle;
			this.access = access;
			this.maximumSize = maximumSize;
		}

		/// <summary>
		/// <b style="color:red">Obsolete.</b>
		/// Creates a new instance of the FileMapping class.
		/// The system object created is not named.
		/// </summary>
		/// <param name="fileName">The name of the file to map in memory.</param>
		/// <param name="mode">The FileMode used to open the underlying file.</param>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="share">The FileShare used to open the file.</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <remarks>
		/// <b style="color:red">NOTE: This member is now obsolete.</b><br/>
		/// </remarks>
		[Obsolete("You should use the static Create method instead.", false)]
		public FileMapping(string fileName, FileMode mode, MappingAccess access, FileShare share, ulong maximumSize):this(fileName, mode, access, share, maximumSize, null)
		{
		}

		/// <summary>
		/// <b style="color:red">Obsolete.</b>
		/// Creates a new instance of the FileMapping class.
		/// The system object created is named and can be used
		/// to open the FileMapping from a different process.
		/// </summary>
		/// <param name="fileName">The name of the file to map in memory.</param>
		/// <param name="mode">The FileMode used to open the underlying file.</param>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="share">The FileShare used to open the file.</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <remarks>
		/// <b style="color:red">NOTE: This member is now obsolete.</b><br/>
		/// </remarks>
		[Obsolete("You should use the static CreateNamed method instead.", false)]
		public FileMapping(string fileName, FileMode mode, MappingAccess access, FileShare share, ulong maximumSize, string name)
		{
			IntPtr fileHandle;

			this.access = access;
			fileHandle = ManagedFileMappingAdaptor.CreateFile(fileName, mode, MappingAccessConverter.ToFileAccess(access), share);
			try
			{
				if (maximumSize == 0)
				{
					this.maximumSize = (ulong)new FileInfo(fileName).Length;
				}
				else
				{
					this.maximumSize = maximumSize;
				}
				this.handle = ManagedFileMappingAdaptor.CreateFileMapping(fileHandle, MappingAccessConverter.ToPageProtection(access), this.maximumSize, name);
			}
			finally
			{
				ManagedFileMappingAdaptor.CloseHandle(fileHandle);
			}
		}

		/// <summary>
		/// <b style="color:red">Obsolete.</b>
		/// Creates a new instance of the FileMapping class.
		/// The system object created is named and can be used
		/// to open the FileMapping from a different process.
		/// </summary>
		/// <param name="fileStream">the FileStream object</param>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <remarks>
		/// <b style="color:red">NOTE: This member is now obsolete.</b><br/>
		/// You should not access the fileStream used in the constructor directly else it could cause an inconsistent result.
		/// </remarks>
		[Obsolete("You have use the StorageAPI's create file function to create a FileMapping.", true)]
		public FileMapping(FileStream fileStream, MappingAccess access, ulong maximumSize, string name)
		{
			IntPtr fileHandle;

			fileHandle = fileStream.Handle;
			try
			{
				this.handle = ManagedFileMappingAdaptor.CreateFileMapping(fileHandle, MappingAccessConverter.ToPageProtection(access), maximumSize, name);
			}
			finally
			{
				ManagedFileMappingAdaptor.CloseHandle(fileHandle);
			}
		}

		/// <summary>
		/// <b style="color:red">Obsolete.</b>
		/// creates a file mapping object of the specified size backed by the operating-system paging file rather than by a named file in the file system
		/// </summary>
		/// <param name="access">The mapping access</param>
		/// <param name="maximumSize">The maximum size the file will grow to.</param>
		/// <remarks>
		/// <b style="color:red">NOTE: This member is now obsolete.</b><br/>
		/// </remarks>
		[Obsolete("You should use the static CreatePagingFileMapping method instead.", false)]
		public FileMapping(MappingAccess access, ulong maximumSize)
		{
			this.handle = ManagedFileMappingAdaptor.CreatePagingFileMapping(MappingAccessConverter.ToPageProtection(access), maximumSize, null);
		}

		/// <summary>
		/// <b style="color:red">Obsolete.</b>
		/// creates a file mapping object of the specified size backed by the operating-system paging file rather than by a named file in the file system
		/// </summary>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="maximumSize">The maximum size that will be accessed.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <remarks>
		/// <b style="color:red">NOTE: This member is now obsolete.</b><br/>
		/// </remarks>
		[Obsolete("You should use the static CreateNamed method instead.", false)]
		public FileMapping(MappingAccess access, ulong maximumSize, string name)
		{
			this.handle = ManagedFileMappingAdaptor.CreatePagingFileMapping(MappingAccessConverter.ToPageProtection(access), maximumSize, name);
		}

		/// <summary>
		/// <b style="color:red">Obsolete.</b>
		/// Creates a new instance of the FileMapping class. 
		/// This constructor openes an existing FileMapping (e.g. created by a different process) by name. 
		/// </summary>
		/// <param name="access">The MappingAccess.</param>
		/// <param name="name">The system wide object name to associate with the FileMapping.</param>
		/// <remarks>
		/// <b style="color:red">NOTE: This member is now obsolete.</b><br/>
		/// </remarks>
		[Obsolete("You should use the static CreateFromSystemObjectName method instead.", false)]
		public FileMapping(MappingAccess access, string name)
		{
			this.handle = ManagedFileMappingAdaptor.OpenFileMapping(MappingAccessConverter.ToFileMapAccess(access), false, name);
		}

		#endregion

		#region ------------ private methods ------------

        #endregion

		#region ------------ protected methods ------------

		/// <summary>
		/// Closes the FileMapping system handle.
		/// </summary>
		protected override void DisposeUnmanagedResources()
		{
			ManagedFileMappingAdaptor.CloseHandle(handle);
		}

		/// <summary>
		/// Disposes managed resources.
		/// </summary>
		/// <remarks>The current implementaion of this method does nothing.</remarks>
		protected override void DisposeManagedResources()
		{
		}

		#endregion

		#region ------------ public methods ------------

		/// <summary>
		/// Closes the underlying system handle and disposes the object.
		/// </summary>
		public void Close()
		{
			Dispose();
		}

		/// <summary>
		/// Duplicates the Handle to use it in a foreign process. 
		/// </summary>
		/// <param name="targetProcessHandle">the handle of the foreign process</param>
		/// <param name="inheritHandle">indicates if the handle will be inherited to child processes</param>
		/// <returns>The duplicated handle which is valid in the target process.</returns>
		/// <remarks>The Target process can use CreateFromOwnedHandle to create a FileMapping object for this handle.</remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public IntPtr DuplicateHandle(IntPtr targetProcessHandle, bool inheritHandle)
		{
			return ManagedFileMappingAdaptor.DuplicateHandle(targetProcessHandle, Handle, Process.GetCurrentProcess().Handle, inheritHandle);
		}
		
		#endregion

		#region ------------ properties ------------

		/// <summary>
		/// Handle of the underlying system object.
		/// </summary>
		/// <value>
		/// The Handle of the underlying system object.
		/// </value>
		/// <remarks>This handle can be copied in a different process with DuplicateHandle. There you can pass it directly to the FileMappingView constructor.</remarks>
		public IntPtr Handle
		{
			get
			{
				CheckDisposed();
				return handle;
			}
		}

		/// <summary>
		/// MaxiumumSize value used to create this object.
		/// </summary>
		/// <value>
		/// The MaxiumumSize value used to create this object.
		/// </value>
		public ulong MaxiumumSize
		{
			get
			{
				CheckDisposed();
				return maximumSize;
			}
		}

		/// <summary>
		/// MappingAccess value used to create this object.
		/// </summary>
		/// <value>
		/// The MappingAccess value used to create this object.
		/// </value>
		public MappingAccess Access 
		{
			get
			{
				CheckDisposed();
				return access;
			}
		}

		#endregion
	}
}
